/*
 *	Recycle bin IO and object recycling, recovering, and purging.
 */

#ifndef EDVRECYCLE_H
#define EDVRECYCLE_H

#include <sys/stat.h>
#include "edvcontext.h"


/*
 *      Returns a statically allocated string describing the last error
 *      that occured when calling EDVRecycle(), EDVRecover(), or
 *      EDVPurge().  Can return NULL if there was no error.
 */
const gchar *EDVRecycleGetError(edv_context_struct *ctx);

/*
 *      Get statistics of a recycled object by its index.
 *
 *      If the path_rtn is not NULL then an allocated string containing
 *      the full path of the original object will be returned.  The
 *      calling function must deallocate this string.
 *
 *      Inputs path_rtn and stat_buf may be NULL, indicating that the
 *      return is not wanted.
 *
 *      Returns 0 on success and -1 on error.
 */
extern gint EDVRecycledObjectStat(
        edv_context_struct *ctx, guint index,
        gchar **path_rtn,
        struct stat *stat_buf
);

/*
 *      Same as EDVRecycledObjectStat() except that a list of all
 *      recycled objects are returned.  The returned pointer arrays
 *      and each pointed to location must be deallocated.by the calling
 *      function.
 *
 *      Returns 0 on success or non-zero on error.
 */
extern gint EDVRecycledObjectStatAll(
        edv_context_struct *ctx,
        gchar ***path_rtn,
	guint **index_rtn,
        struct stat ***stat_buf,
        gint *total
);

/*
 *      Deletes an actual disk object and places it into the recycle
 *      bin.
 *
 *      If notify is set to TRUE then a "object_removed_notify" and
 *      "recycled_object_added_notify" will be queued on the ctx
 *      (only on success).
 *
 *      Returns the recycled object index number or 0 on error.
 */
extern guint EDVRecycle(
	edv_context_struct *ctx, const gchar *path,
	gboolean notify,
	gint (*progress_cb)(
		gpointer client_data, gulong pos, gulong total
	),
	gpointer client_data
);

/*
 *      Recovers an object from the recycled objects directory.
 *
 *      If path is not NULL, then path will be used as the alternate
 *      location to recover the object.
 *
 *      If notify is set to TRUE then a "object_added_notify" and
 *      "recycled_object_removed_notify" will be queued on the ctx
 *      (only on success).
 *
 *      Returns non-zero on error.
 */
extern gint EDVRecover(
        edv_context_struct *ctx, guint index, const gchar *path,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
);

/*
 *      Permanently removes an object from the recycled objects directory.
 *
 *      If notify is set to TRUE then a "recycled_object_removed_notify"
 *      will be queued on the ctx (only on success).
 *
 *      Returns non-zero on error.
 */
extern gint EDVPurge(
        edv_context_struct *ctx, guint index,
        gboolean notify,
        gint (*progress_cb)(
                gpointer client_data, gulong pos, gulong total
        ),
        gpointer client_data
);

#endif	/* EDVRECYCLE_H */
