#include <glib.h>
#include "../../include/string.h"
#include "../edvinterps.h"
#include "edvcontext.h"
#include "edvnotify.h"
#include "config.h"

static void EDVNotifyQueueCommand(
        edv_context_struct *ctx, gchar *cmd
);
void EDVNotifyQueueObjectAdded(
        edv_context_struct *ctx, const gchar *path
);
void EDVNotifyQueueObjectModified(
        edv_context_struct *ctx, const gchar *path, const gchar *new_path
);
void EDVNotifyQueueObjectRemoved(
        edv_context_struct *ctx, const gchar *path
);
void EDVNotifyQueueObjectMounted(
        edv_context_struct *ctx, const gchar *path
);
void EDVNotifyQueueObjectUnmounted(
        edv_context_struct *ctx, const gchar *path
);
void EDVNotifyQueueRecycledObjectAdded(
        edv_context_struct *ctx, guint index
);
void EDVNotifyQueueRecycledObjectRemoved(
        edv_context_struct *ctx, guint index
);
void EDVNotifyFlush(edv_context_struct *ctx);


/*
 *	Appends the given cmd to the list of queued commands on the
 *	ctx.  The given cmd should not be referenced again after
 *	this call since it is transfered to the ctx.
 */
static void EDVNotifyQueueCommand(
	edv_context_struct *ctx, gchar *cmd
)
{
	gint i;

	if(cmd == NULL)
	    return;

	if(ctx == NULL)
	{
	    g_free(cmd);
	    return;
	}

	/* Increment total. */
	if(ctx->total_queued_commands < 0)
	    ctx->total_queued_commands = 0;
	i = ctx->total_queued_commands;
	ctx->total_queued_commands = i + 1;

	/* Reallocate pointer array. */
	ctx->queued_command = (gchar **)g_realloc(
	    ctx->queued_command,
	    ctx->total_queued_commands * sizeof(gchar *)
	);
	if(ctx->queued_command == NULL)
	{
	    ctx->total_queued_commands = 0;
	    g_free(cmd);
	    return;
	}

	/* Append and transfer given command to newly allocated
	 * index. The transfered pointer should not be referenced
	 * again by the calling function.
	 */
	ctx->queued_command[i] = cmd;
}

/*
 *	Queues an "object_added_notify" command.
 */
void EDVNotifyQueueObjectAdded(
        edv_context_struct *ctx, const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "object_added_notify %s",
	    path
	);
	EDVNotifyQueueCommand(ctx, cmd);
}

/*
 *      Queues an "object_modified_notify" command.
 *
 *	If new_path is NULL then only path will be sent (hinting that
 *	the name of the object was not modified).
 */
void EDVNotifyQueueObjectModified(
        edv_context_struct *ctx, const gchar *path, const gchar *new_path
)
{
        gchar *cmd;
        if(ctx == NULL)
            return;
	if(new_path != NULL)
	    cmd = g_strdup_printf(
                "object_modified_notify %s %s",
                path, new_path
            );
	else
	    cmd = g_strdup_printf(
                "object_modified_notify %s",
                path
            );
        EDVNotifyQueueCommand(ctx, cmd);
}

/*
 *      Queues an "object_removed_notify" command.
 */
void EDVNotifyQueueObjectRemoved(
        edv_context_struct *ctx, const gchar *path
)
{
        gchar *cmd;
        if(ctx == NULL)
            return;
        cmd = g_strdup_printf(
            "object_removed_notify %s",
            path
        );
        EDVNotifyQueueCommand(ctx, cmd);
}

/*
 *      Queues an "object_mounted_notify" command.
 */
void EDVNotifyQueueObjectMounted(
        edv_context_struct *ctx, const gchar *path
)
{
        gchar *cmd;
        if(ctx == NULL)
            return;
        cmd = g_strdup_printf(
            "object_mounted_notify %s",
            path
        );
        EDVNotifyQueueCommand(ctx, cmd);
}

/*
 *      Queues an "object_unmounted_notify" command.
 */
void EDVNotifyQueueObjectUnmounted(
        edv_context_struct *ctx, const gchar *path
)
{
        gchar *cmd;
        if(ctx == NULL)
            return;
        cmd = g_strdup_printf(
            "object_unmounted_notify %s",
            path
        );
        EDVNotifyQueueCommand(ctx, cmd);
}

/*
 *      Queues a "recycled_object_added_notify" command.
 */
void EDVNotifyQueueRecycledObjectAdded(
        edv_context_struct *ctx, guint index
)
{
        gchar *cmd;
        if(ctx == NULL)
            return;
        cmd = g_strdup_printf(
            "recycled_object_added_notify %i",
            index
        );
        EDVNotifyQueueCommand(ctx, cmd);
}

/*
 *      Queues a "recycled_object_removed_notify" command.
 */
void EDVNotifyQueueRecycledObjectRemoved(
        edv_context_struct *ctx, guint index
)
{
        gchar *cmd;
        if(ctx == NULL)
            return;
        cmd = g_strdup_printf(
            "recycled_object_removed_notify %i",
            index
        );
        EDVNotifyQueueCommand(ctx, cmd);
}


/*
 *	Sends all queued notifies to Endeavour.
 *
 *	If there is no lock link in the local Endeavour data directory
 *	then the queued commands will not be sent.
 *
 *	All queued commands will be removed from the ctx regardless of
 *	error or success.
 */
void EDVNotifyFlush(edv_context_struct *ctx)
{
	gint p;

	if(ctx == NULL)
	    return;

	/* Get lock link and find out about the pid of the currently
	 * running (if any) Endeavour.
	 */
	p = EDVInterPSGetLock((edv_cfg_item_struct *)ctx->cfg_list);
	if(p > 0)
	{
	    /* Send all the commands. */
	    EDVInterPSSendCommand(
		(edv_cfg_item_struct *)ctx->cfg_list, p,
		ctx->queued_command, ctx->total_queued_commands
	    );
	}
	/* Remove all queued commands. */
	StringFreeArray(ctx->queued_command, ctx->total_queued_commands);
	ctx->queued_command = NULL;
	ctx->total_queued_commands = 0;
}
