#include <glib.h>
#include "../../include/string.h"
#include "../../include/disk.h"
#include "../edvcfg.h"
#include "../edvcfgfio.h"
#include "../edvcfglist.h"
#include "edvcontext.h"
#include "edvnotify.h"
#include "config.h"

edv_context_struct *EDVContextNew(void);
void EDVContextLoadConfigurationFile(
        edv_context_struct *ctx,
        const gchar *path		/* Can be NULL for default file. */
);
gint EDVContextCommandsPending(edv_context_struct *ctx);
void EDVContextFlush(edv_context_struct *ctx);
void EDVContextWait(edv_context_struct *ctx);
void EDVContextDelete(edv_context_struct *ctx);


/*
 *	Allocates a new Endeavour context structure.
 */
edv_context_struct *EDVContextNew(void)
{
	edv_context_struct *ctx = (edv_context_struct *)g_malloc0(
	    sizeof(edv_context_struct)
	);
	if(ctx == NULL)
	    return(NULL);

	return(ctx);
}

/*
 *	Loads the configuration from the specified Endeavour 
 *	configuration file path.  If path is NULL then the default
 *	configuration file found in the user's home directory
 *	will be used.
 *
 *      This should be called right after EDVContextNew() to ensure
 *      that the ctx is set up properly before passing to any other
 *      function.
 */
void EDVContextLoadConfigurationFile(
	edv_context_struct *ctx,
	const gchar *path		/* Can be NULL for default file. */
)
{
	gchar *dpath;
	const edv_cfg_item_struct src_cfg_list[] = EDV_CONFIGURATION_LIST;


	if(ctx == NULL)
	    return;

        /* Copy source configuration list to the target configuration
         * list on the context structure.
         */
        if(ctx->cfg_list == NULL)
            ctx->cfg_list = EDVCFGItemListCopyList(src_cfg_list);
	if(ctx->cfg_list == NULL)
	    return;

	/* Format configuration file location as dpath. Check if we
	 * should use the default configuration file path by checking if
	 * the given path is NULL.
	 */
	if(path == NULL)
	{
	    /* Use default configuration file, located in the user's
	     * home directory
	     * $HOME/EDV_DEF_LOCAL_DATA_DIR/EDV_DEF_CONFIG_FILE.
	     */
	    const gchar *cstrptr = g_getenv("HOME");
	    dpath = g_strdup_printf(
		"%s%c%s%c%s",
		(cstrptr != NULL) ? cstrptr : "/", DIR_DELIMINATOR,
		EDV_DEF_LOCAL_DATA_DIR, DIR_DELIMINATOR,
		EDV_DEF_CONFIG_FILE
	    );
	}
	else
	{
	    /* Configuration file path given, copy it as dpath. */
	    dpath = g_strdup(path);
	}

	/* Load configuration item values from the configuration file. */
	EDVCFGLoadFromFile(dpath, (edv_cfg_item_struct *)ctx->cfg_list);

	/* Update values on the ctx structure based on the newly loaded
	 * configuration item values from the configuration file.
	 */
	if(ctx->cfg_list != NULL)
	{
	    const gchar *cstrptr;


	    /* Recycled objects index file. */
	    cstrptr = EDVCFGItemListGetValueS(
		(edv_cfg_item_struct *)ctx->cfg_list,
		EDV_CFG_PARM_FILE_RECYCLED_INDEX
	    );
	    g_free(ctx->recycled_index_file);
	    ctx->recycled_index_file = (cstrptr != NULL) ?
		g_strdup(cstrptr) : NULL;




/* Add support for additional members that need to be coppied here. */
	}

	g_free(dpath);
	dpath = NULL;
}

/*
 *      Returns the number of queued interprocess commands pending.
 *
 *      Can return 0 on error, so error is not distinguished.
 */
gint EDVContextCommandsPending(edv_context_struct *ctx)
{
	return((ctx != NULL) ? ctx->total_queued_commands : 0);
}

/*
 *      Processes and flushes all pending operations and resources on
 *      the given ctx to Endeavour.
 *
 *      This call will not block/wait for Endeavour to acknowlage
 *      the request, for that use EDVContextWait();
 */
void EDVContextFlush(edv_context_struct *ctx)
{
        if(ctx == NULL)
            return;

	/* Flush all pending interprocess commands. */
	EDVNotifyFlush(ctx);

/* Add additional things to flush here. */
}

/*
 *      Waits for any pending operations that were sent to Endeavour
 *      to be processed. This will cause the operation to block until
 *      the pending operations (if any) are completed.
 */
void EDVContextWait(edv_context_struct *ctx)
{
        if(ctx == NULL)
            return;

/* Add additional things to wait for here. */
}

/*
 *	Deallocates the Endeavour context and all it's resources.
 */
void EDVContextDelete(edv_context_struct *ctx)
{
	if(ctx == NULL)
            return;

	/* Deallocate any queued commands. */
        StringFreeArray(ctx->queued_command, ctx->total_queued_commands);
        ctx->queued_command = NULL;
        ctx->total_queued_commands = 0;

	/* Deallocate other resources. */
	g_free(ctx->recycled_index_file);
	ctx->recycled_index_file = NULL;

        /* Deallocate configuration list. */
        EDVCFGItemListDeleteList((edv_cfg_item_struct *)ctx->cfg_list);
        ctx->cfg_list = NULL;

	/* Deallocate structure itself. */
	g_free(ctx);
}
