#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "../guiutils.h"
#include "../clipboard.h"
#include "hview.h"
#include "hviewcb.h"

static guint8 HViewHexConvert(const gchar *buf);

static gint HViewScrollUpTOCB(gpointer data);
static gint HViewScrollDownTOCB(gpointer data);

gint HViewConfigureCB(
        GtkWidget *widget, GdkEventConfigure *configure, gpointer data
);
gint HViewVisibilityCB(
        GtkWidget *widget, GdkEventVisibility *visibility, gpointer data
);
gint HViewExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
gint HViewKeyCB(
        GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint HViewButtonCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
);
gint HViewMotionCB(
        GtkWidget *widget, GdkEventMotion *motion, gpointer data
);
void HViewValueChangedCB(GtkAdjustment *adj, gpointer data);

void HViewInsertCB(GtkWidget *w, gpointer data);
void HViewDeleteCB(GtkWidget *w, gpointer data);
void HViewCutCB(GtkWidget *w, gpointer data);
void HViewCopyCB(GtkWidget *w, gpointer data);
void HViewPasteCB(GtkWidget *w, gpointer data);
void HViewEditModeHexCB(GtkWidget *w, gpointer data);
void HViewEditModeASCIICB(GtkWidget *w, gpointer data);


/*
 *	Converts the given hex string (without the 0x prefix) into
 *	an guint8 value.
 */
static guint8 HViewHexConvert(const gchar *buf)
{
	gchar a = buf[0];
	guint8 v = 0x00;

	if((a >= 'A') && (a <= 'F'))
	    v += (guint8)(a - 'A' + 10) * 16;
	else
	    v += (guint8)(a - '0') * 16;

	a = buf[1];

	if((a >= 'A') && (a <= 'F'))
            v += (guint8)(a - 'A' + 10);
        else
            v += (guint8)(a - '0');

	return(v);
}

/*
 *	Scroll up timeout callback.
 */
static gint HViewScrollUpTOCB(gpointer data)
{
	gint x = 0, y = 0;
	GdkModifierType mask = 0;
	GdkWindow *window;
	GdkEventMotion ev;
	GtkAdjustment *adj;
	GtkWidget *w;
        hview_struct *hv = (hview_struct *)data;
        if(hv == NULL)
            return(FALSE);

	adj = hv->vadj;
	w = hv->view_da;
        if((adj == NULL) || (w == NULL))
	    return(FALSE);

	window = w->window;
	if(window == NULL)
	    return(FALSE);

	gdk_window_get_pointer(window, &x, &y, &mask);

	memset(&ev, 0x00, sizeof(GdkEventMotion));
	ev.type = GDK_MOTION_NOTIFY;
	ev.window = window;
	ev.send_event = TRUE;
	ev.time = GDK_CURRENT_TIME;
	ev.x = (gdouble)x;
	ev.y = (gdouble)y;
	ev.state = mask;
	ev.is_hint = FALSE;

	return(HViewMotionCB(w, &ev, data));
}

/*
 *	Scroll down timeout callback.
 */
static gint HViewScrollDownTOCB(gpointer data)
{
	return(HViewScrollUpTOCB(data));
}

/*
 *	"configure_event" callback.
 */
gint HViewConfigureCB(
        GtkWidget *widget, GdkEventConfigure *configure, gpointer data
)
{
	gint width, height;
	GtkAdjustment *adj;
	GtkWidget *w;
        hview_struct *hv = (hview_struct *)data;
        if((configure == NULL) || (hv == NULL))
            return(FALSE);

	w = hv->view_da;
	if(w == NULL)
	    return(FALSE);

	width = configure->width;
	height = configure->height;

	/* Recreate view pixmap. */
	if(hv->view_pm != NULL)
	    gdk_pixmap_unref(hv->view_pm);
	if((width > 0) && (height > 0))
	    hv->view_pm = gdk_pixmap_new(w->window, width, height, -1);
	else
	    hv->view_pm = NULL;

	/* Update adjustments. */
	adj = hv->vadj;
	if(adj != NULL)
	{
	    adj->page_size = (gfloat)height;
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	    if((adj->value + adj->page_size) > adj->upper)
	    {
		adj->value = adj->upper - adj->page_size;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
		gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
	    }
	}

	return(TRUE);
}

/*
 *	"visibility_notify_event" signal callback.
 */
gint HViewVisibilityCB(
        GtkWidget *widget, GdkEventVisibility *visibility, gpointer data
)
{
        hview_struct *hv = (hview_struct *)data;
        if((visibility == NULL) || (hv == NULL))
            return(FALSE);

	hv->visibility = visibility->state;

        /* Check if the drawing area is now fully obscured, suggesting
         * that we no longer need the pixmap buffer.
         *
         * The pixmap buffer will be recreated on resize or the next
         * draw.
         */
        if(visibility->state == GDK_VISIBILITY_FULLY_OBSCURED)
        {
            if(hv->view_pm != NULL)
            {
                gdk_pixmap_unref(hv->view_pm);
                hv->view_pm = NULL;
            }
        }

        return(TRUE);
}

/*
 *	"expose_event" callback.
 */
gint HViewExposeCB(
        GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	const guint8 *buf;
	gint buf_pos, buf_len;
	gint width, height;
	gint cell_width, cell_height, cells_per_row;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkGC *gc;
	GtkAdjustment *adj;
	gint state;
	GtkStyle *style;
	GtkWidget *w;
	hview_struct *hv = (hview_struct *)data;
	if(hv == NULL)
	    return(FALSE);

	buf = hv->buf;
	buf_pos = hv->buf_pos;
	buf_len = hv->buf_len;

	cell_width = hv->cell_width;
	cell_height = hv->cell_height;
	cells_per_row = hv->cells_per_row;

	if((hv->visibility == GDK_VISIBILITY_FULLY_OBSCURED) ||
	   !hv->map_state
	)
	    return(FALSE);

	if((cell_width <= 0) || (cell_height <= 0) || (cells_per_row <= 0))
	    return(FALSE);

	gc = hv->gc;
	adj = hv->vadj;
	w = hv->view_da;
	if((gc == NULL) || (adj == NULL) || (w == NULL))
	    return(FALSE);

	gdk_gc_set_function(gc, GDK_COPY);
	gdk_gc_set_fill(gc, GDK_SOLID);

	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	window = w->window;
	if((style == NULL) || (window == NULL))
	    return(FALSE);

	/* Create view pixmap as needed. */
	if(hv->view_pm == NULL)
	    hv->view_pm = pixmap = gdk_pixmap_new(
		window, w->allocation.width, w->allocation.height, -1
	    );
	else
	    pixmap = hv->view_pm;
	if(pixmap == NULL)
	    return(FALSE);

	gdk_window_get_size(pixmap, &width, &height);
	if((width <= 0) || (height <= 0))
	    return(FALSE);

	/* Draw base. */
	gdk_draw_rectangle(
	    (GdkDrawable *)pixmap, style->base_gc[state],
	    TRUE,
	    0, 0, width, height
	);

	/* Draw text. */
	if((buf != NULL) && (hv->font != NULL))
	{
	    gint border = 0;
	    gint i, j, n, x, y;
	    GdkFont *font = hv->font;
	    gint buf_start = (gint)adj->value / cell_height * cells_per_row;
	    gint y_start = border - ((gint)adj->value % cell_height);
	    gint sel_start = hv->buf_sel_start;
	    gint sel_end = hv->buf_sel_end;
	    gchar tmp_buf[80];


	    /* Set up gc for drawing text. */
	    gdk_gc_set_font(gc, font);

	    /* Sanitize selection bounds. */
	    if((sel_start > -1) && (sel_end > -1))
	    {
		if(sel_end < sel_start)
		{
		    gint t = sel_start;
		    sel_start = sel_end;
		    sel_end = t;
		}
	    }

	    /* Set starting positions. */
	    i = buf_start;
	    if(i < 0)
		i = 0;
	    y = y_start;
	    while((i < buf_len) && (y < height))
	    {
		/* Draw address. */
		x = hv->address_x;
		sprintf(tmp_buf, "0x%.8X", i);
                gdk_gc_set_foreground(gc, &style->text[state]);
                gdk_draw_string(
                    (GdkDrawable *)pixmap, font, gc,
                    x, y + font->ascent, tmp_buf
                );

		/* Draw hex values. */
		for(n = 0, j = i, x = hv->hex_x;
		    n < cells_per_row;
		    n++, j++, x += (3 * cell_width)
		)
		{
		    if(j >= buf_len)
			break;

		    /* Draw selected color background if this cell falls
		     * within the selection bounds?  Also set text color
		     * for upcomming string drawing.
		     */
		    if((j >= sel_start) && (j <= sel_end))
		    {
                        gdk_draw_rectangle(
                            (GdkDrawable *)pixmap,
                            style->bg_gc[GTK_STATE_SELECTED],
                            TRUE, x - 1, y,
			    (2 * cell_width) + 2, cell_height
                        );
			gdk_gc_set_foreground(
                            gc, &style->text[GTK_STATE_SELECTED]
                        );
		    }
		    else
		    {
			gdk_gc_set_foreground(gc, &style->text[state]);
		    }
		    sprintf(tmp_buf, "%.2X", buf[j]);
		    gdk_draw_string(
			(GdkDrawable *)pixmap, font, gc,
			x, y + font->ascent, tmp_buf
		    );

                    /* Cursor rectangle. */
                    if(j == hv->buf_pos)
                    {
                        GdkGCValues gcv;
                        gdk_gc_get_values(
                            style->text_gc[GTK_STATE_SELECTED], &gcv
                        );
                        gdk_gc_set_function(
			    style->text_gc[GTK_STATE_SELECTED], GDK_INVERT
			);
                        gdk_draw_rectangle(
                            (GdkDrawable *)pixmap,
                            style->text_gc[GTK_STATE_SELECTED],
                            FALSE, x - 1, y,
			    (2 * cell_width) + 1, cell_height - 1
                        );
                        gdk_gc_set_function(
                            style->text_gc[GTK_STATE_SELECTED],
                            gcv.function
                        );
                    }
		}

		/* Draw ascii values. */
                for(n = 0, j = i, x = hv->ascii_x;
		    n < cells_per_row;
		    n++, j++, x += cell_width
		)
                {
                    if(j >= buf_len)
                        break;

                    /* Draw selected color background if this cell falls
                     * within the selection bounds?  Also set text color
                     * for upcomming string drawing.
                     */
                    if((j >= sel_start) && (j <= sel_end))
                    {
                        gdk_draw_rectangle(
                            (GdkDrawable *)pixmap,
                            style->bg_gc[GTK_STATE_SELECTED],
                            TRUE, x, y, cell_width, cell_height
                        );
                        gdk_gc_set_foreground(
                            gc, &style->text[GTK_STATE_SELECTED]
                        );
                    }
		    else
                    {
                        gdk_gc_set_foreground(gc, &style->text[state]);
                    }
                    sprintf(tmp_buf, "%c", (gchar)buf[j]);
                    gdk_draw_string(
                        (GdkDrawable *)pixmap, font, gc,
                        x, y + font->ascent, tmp_buf
                    );

		    /* Cursor rectangle. */
		    if(j == hv->buf_pos)
		    {
			GdkGCValues gcv;
			gdk_gc_get_values(
			    style->text_gc[GTK_STATE_SELECTED], &gcv
			);
			gdk_gc_set_function(
			    style->text_gc[GTK_STATE_SELECTED], GDK_INVERT
			);
                        gdk_draw_rectangle(
                            (GdkDrawable *)pixmap,
			    style->text_gc[GTK_STATE_SELECTED],
                            FALSE, x, y, cell_width - 1, cell_height - 1
                        );
			gdk_gc_set_function(
			    style->text_gc[GTK_STATE_SELECTED],
			    gcv.function
			);
		    }
                }

		i += cells_per_row;
		y += cell_height;
	    }

	}

	/* Put the drawn pixmap to the window. */
	gdk_draw_pixmap(
	    (GdkDrawable *)window, gc,
	    (GdkDrawable *)pixmap,
	    0, 0, 0, 0, width, height
	);

	return(TRUE);
}

/*
 *	"key_press_event" or "key_release_event" callback.
 */
gint HViewKeyCB(
        GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
        gint etype;
	gboolean key_press;
	GtkAdjustment *adj;
        hview_struct *hv = (hview_struct *)data;
        if((key == NULL) || (hv == NULL))
            return(FALSE);

/* Stops the "key_press_event" or "key_release_event" signal for cursor
 * keys so GTK+ does not later handle it and change the focus widget.
 *
 * Note that the GDK_Tab key should be ignored so tab can still be
 * used to change focus.
 */
#define STOP_SIGNAL_EMIT	\
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  (etype == GDK_KEY_PRESS) ? "key_press_event" : "key_release_event" \
 ); \
}
	adj = hv->vadj;
        etype = key->type;
	key_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	switch(key->keyval)
	{
	  case GDK_Escape:
	    if(key_press)
	    {
		/* Cancel user editing of current hex value (if the user
		 * was doing that).
		 */
		hv->buf_hex_edit = 0;
		*hv->buf_hex_edit_buf = '\0';
                HViewDraw(hv);
                HViewSetStatusMessage(hv, NULL, FALSE);
	    }
	    status = TRUE;
	    break;

	  case GDK_BackSpace:
	    if(key_press)
	    {
		if(hv->buf_hex_edit > 0)
		    hv->buf_hex_edit--;
		if(hv->buf_hex_edit < 3)
		    hv->buf_hex_edit_buf[hv->buf_hex_edit] = '\0';
                HViewSetStatusMessage(hv, NULL, FALSE);
            }
            status = TRUE;
            break;

	  case GDK_space:
	    if(key_press)
            {
		if((key->state & GDK_CONTROL_MASK) ||
		   (key->state & GDK_SHIFT_MASK)
		)
		{
		    hv->buf_sel_end = hv->buf_pos;
		}
		else
		{
		    if(HVIEW_IS_BUF_SELECTED(hv))
			hv->buf_sel_start = hv->buf_sel_end = -1;
		    else
			hv->buf_sel_start = hv->buf_sel_end = hv->buf_pos;
		}
		HViewDraw(hv);
		HViewUpdateMenus(hv);
                HViewSetStatusMessage(hv, NULL, FALSE);
            }
            status = TRUE;
            break;

	  case GDK_Return:
	  case GDK_KP_Enter:
	  case GDK_ISO_Enter:
	    if(key_press)
	    {
		/* In hex edit mode? */
		if(hv->edit_mode == HVIEW_EDIT_MODE_HEX)
		{
		    guint8 v;

		    /* If the user was in the middle of editing the hex
		     * value, then finish it by assuming the second
		     * character is a '0'.
		     */
		    if(hv->buf_hex_edit == 1)
		    {
			hv->buf_hex_edit_buf[1] = '0';
			hv->buf_hex_edit_buf[2] = '\0';
		    }
		    /* Process new value only if one or more characters
		     * in hex edit buffer, this is so that if this key
		     * is pressed while nothing is edit then nothing is
		     * edited.
		     */
		    if(hv->buf_hex_edit > 0)
		    {
			v = HViewHexConvert(hv->buf_hex_edit_buf);

			/* If selected, then set selection.  Otherwise
			 * just set the cursor position.
			 */
			if(HVIEW_IS_BUF_SELECTED(hv))
			    HViewSetSelected(hv, v);
			else if(HVIEW_IS_BUF_POS_VALID(hv, hv->buf_pos))
			    hv->buf[hv->buf_pos] = v;
		    }

		    /* Clear hex edit buffer. */
		    hv->buf_hex_edit = 0;
		    *hv->buf_hex_edit_buf = '\0';

                    hv->modified = TRUE;
		}

		/* Seek cursor past selection and update selection to
		 * cursor's position.
		 */
		if(hv->buf_sel_end > -1)
		{
		    if(hv->buf_sel_end > hv->buf_sel_start)
			hv->buf_pos = hv->buf_sel_end + 1;
		    else
			hv->buf_pos = hv->buf_sel_start + 1;
		}
		else
		{
		    hv->buf_pos++;
		}
		if(hv->buf_pos >= hv->buf_len)
		    hv->buf_pos = hv->buf_len - 1;
		if(hv->buf_pos < 0)
		    hv->buf_pos = 0;

		/* Unselect. */
		hv->buf_sel_start = hv->buf_sel_end = -1;

                HViewDraw(hv);
                HViewUpdateMenus(hv);
                HViewSetStatusMessage(hv, NULL, FALSE);
                if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
                    GTK_VISIBILITY_FULL
                )
                    HViewScrollTo(hv, hv->buf_pos, 0.5);
	    }
            status = TRUE;
	    break;

          case GDK_Insert:
            if(key_press)
            {
		if(key->state & GDK_CONTROL_MASK)
		    HViewCopyCB(widget, hv);
		else if(key->state & GDK_SHIFT_MASK)
		    HViewPasteCB(widget, hv);
		else
		    HViewInsertCB(widget, hv);
            }
            status = TRUE;
            break;

          case GDK_Delete:
            if(key_press)
            {
		if((key->state & GDK_CONTROL_MASK) ||
                   (key->state & GDK_SHIFT_MASK)
		)
		    HViewCutCB(widget, hv);
		else
		    HViewDeleteCB(widget, hv);
            }
            status = TRUE;
            break;


	  case GDK_Up:
	    if(key_press)
	    {
	        if((key->state & GDK_CONTROL_MASK) &&
	           (adj != NULL)
		)
		{
		    adj->value -= adj->step_increment;
		    if(adj->value < adj->lower)
			adj->value = adj->lower;
		    gtk_signal_emit_by_name(
			GTK_OBJECT(adj), "value_changed"
		    );
		}
		else
		{
		    gint last_pos = hv->buf_pos;
		    hv->buf_pos -= hv->cells_per_row;
		    if(hv->buf_pos < 0)
		        hv->buf_pos = 0;
		    if(key->state & GDK_SHIFT_MASK)
		    {
			if(HVIEW_IS_BUF_SELECTED(hv))
			    hv->buf_sel_end = hv->buf_pos;
			else
			{
			    hv->buf_sel_start = last_pos;
			    hv->buf_sel_end = hv->buf_pos;
			}
			HViewUpdateMenus(hv);
		    }
		    HViewDraw(hv);
                    HViewSetStatusMessage(hv, NULL, FALSE);
                    if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
                        GTK_VISIBILITY_FULL
                    )
                        HViewScrollTo(hv, hv->buf_pos, 0.5);
		}
	    }
	    STOP_SIGNAL_EMIT
            status = TRUE;
	    break;

	  case GDK_Down:
	    if(key_press)
	    {
                if((key->state & GDK_CONTROL_MASK) &&
                   (adj != NULL)
                )
                {
                    adj->value += adj->step_increment;
                    if((adj->value + adj->page_size) > adj->upper)
                        adj->value = adj->upper - adj->page_size;
                    gtk_signal_emit_by_name(
                        GTK_OBJECT(adj), "value_changed"
                    );
                }
                else
                {
                    gint last_pos = hv->buf_pos;
                    hv->buf_pos += hv->cells_per_row;
                    if(hv->buf_pos >= hv->buf_len)
                        hv->buf_pos = hv->buf_len - 1;
                    if(key->state & GDK_SHIFT_MASK)
                    {
                        if(HVIEW_IS_BUF_SELECTED(hv))
                            hv->buf_sel_end = hv->buf_pos;
                        else
                        {
                            hv->buf_sel_start = last_pos;
                            hv->buf_sel_end = hv->buf_pos;
                        }
                        HViewUpdateMenus(hv);
                    }
                    HViewDraw(hv);
		    HViewSetStatusMessage(hv, NULL, FALSE);
                    if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
                        GTK_VISIBILITY_FULL
                    )
                        HViewScrollTo(hv, hv->buf_pos, 0.5);
		}
	    }
	    STOP_SIGNAL_EMIT
            status = TRUE;
	    break;

	  case GDK_Left:
	    if(key_press)
	    {
	        if((key->state & GDK_CONTROL_MASK) &&
	           (adj != NULL)
		)
		{

		}
		else
		{
                    gint last_pos = hv->buf_pos;
                    hv->buf_pos--;
                    if(hv->buf_pos < 0)
                        hv->buf_pos = 0;
                    if(key->state & GDK_SHIFT_MASK)
                    {
                        if(HVIEW_IS_BUF_SELECTED(hv))
                            hv->buf_sel_end = hv->buf_pos;
                        else
                        {
                            hv->buf_sel_start = last_pos;
                            hv->buf_sel_end = hv->buf_pos;
                        }
                        HViewUpdateMenus(hv);
                    }
                    HViewDraw(hv);
                    HViewSetStatusMessage(hv, NULL, FALSE);
                    if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
                        GTK_VISIBILITY_FULL
                    )
                        HViewScrollTo(hv, hv->buf_pos, 0.5);
		}
	    }
	    STOP_SIGNAL_EMIT
            status = TRUE;
	    break;

	  case GDK_Right:
	    if(key_press)
	    {
                if((key->state & GDK_CONTROL_MASK) &&
                   (adj != NULL)
                )
                {

                }
                else
                {
                    gint last_pos = hv->buf_pos;
                    hv->buf_pos++;
                    if(hv->buf_pos >= hv->buf_len)
                        hv->buf_pos = hv->buf_len - 1;
                    if(key->state & GDK_SHIFT_MASK)
                    {
                        if(HVIEW_IS_BUF_SELECTED(hv))
                            hv->buf_sel_end = hv->buf_pos;
                        else
			{
			    hv->buf_sel_start = last_pos;
                            hv->buf_sel_end = hv->buf_pos;
			}
                        HViewUpdateMenus(hv);
                    }
                    HViewDraw(hv);
                    HViewSetStatusMessage(hv, NULL, FALSE);
                    if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
                        GTK_VISIBILITY_FULL
                    )
                        HViewScrollTo(hv, hv->buf_pos, 0.5);
		}
	    }
	    STOP_SIGNAL_EMIT
            status = TRUE;
	    break;

	  case GDK_Page_Up:
          case GDK_KP_Page_Up:
	    if(key_press)
	    {
		if(adj != NULL)
		{
                    adj->value -= adj->page_increment;
                    if(adj->value < adj->lower)
                        adj->value = adj->lower;
                    gtk_signal_emit_by_name(
                        GTK_OBJECT(adj), "value_changed"
                    );
		}
	    }
	    STOP_SIGNAL_EMIT
            status = TRUE;
	    break;

	  case GDK_Page_Down:
          case GDK_KP_Page_Down:
	    if(key_press)
	    {
		if(adj != NULL)
		{
                    adj->value += adj->page_increment;
                    if((adj->value + adj->page_size) > adj->upper)
                        adj->value = adj->upper - adj->page_size;
                    gtk_signal_emit_by_name(
                        GTK_OBJECT(adj), "value_changed"
                    );
		}
	    }
	    STOP_SIGNAL_EMIT
            status = TRUE;
	    break;

          case GDK_Home:
          case GDK_KP_Home:
            if(key_press)
            {
                if(adj != NULL)
                {
                    adj->value = adj->lower;
                    gtk_signal_emit_by_name(
                        GTK_OBJECT(adj), "value_changed"
                    );
                }
            }
            STOP_SIGNAL_EMIT
            status = TRUE;
            break;

          case GDK_End:
          case GDK_KP_End:
            if(key_press)
            {
                if(adj != NULL)
                {
                    adj->value = adj->upper - adj->page_size;
		    if(adj->value < adj->lower)
			adj->value = adj->lower;
                    gtk_signal_emit_by_name(
                        GTK_OBJECT(adj), "value_changed"
                    );
                }
            }
            STOP_SIGNAL_EMIT
            status = TRUE;
            break;

	  default:
	    /* Hex edit mode? check if number or alphabet character */
	    if(isxdigit(key->keyval) &&
	       (hv->edit_mode == HVIEW_EDIT_MODE_HEX) &&
	       ((key->state == 0x00000000) ||
                (key->state == GDK_SHIFT_MASK)
	       )
	    )
	    {
		if(key_press)
		{
		    gint i = hv->buf_hex_edit;

		    /* Add new character to the tempory hex edit
		     * buffer and increment hex edit buffer position.
		     * Do this only if the current hex edit buffer
		     * position is in bounds.
		     */
		    if((i >= 0) && (i < 2))
		    {
			hv->buf_hex_edit_buf[i] =
			    toupper((gchar)key->keyval);
			hv->buf_hex_edit_buf[i + 1] = '\0';
			hv->buf_hex_edit++;
		    }

		    HViewSetStatusMessage(hv, NULL, FALSE);
		}
		status = TRUE;
	    }
	    /* aSCII edit mode? */
	    else if(isascii(key->keyval) &&
                    (hv->edit_mode == HVIEW_EDIT_MODE_ASCII) &&
                    ((key->state == 0x00000000) ||
                     (key->state == GDK_SHIFT_MASK)
		    )
	    )
            {
                if(key_press)
                {
		    if(HVIEW_IS_BUF_SELECTED(hv))
			HViewSetSelected(hv, (guint8 )key->keyval);
		    else if(HVIEW_IS_BUF_POS_VALID(hv, hv->buf_pos))
			hv->buf[hv->buf_pos] = (guint8 )key->keyval;

		    /* Seek cursor past selection and update selection
		     * to cursor's position.
		     */
		    if(HVIEW_IS_BUF_SELECTED(hv))
		    {
			if(hv->buf_sel_end > hv->buf_sel_start)
			    hv->buf_pos = hv->buf_sel_end + 1;
			else
			    hv->buf_pos = hv->buf_sel_start + 1;
		    }
		    else
		    {
			hv->buf_pos++;
		    }
		    if(hv->buf_pos >= hv->buf_len)
			hv->buf_pos = hv->buf_len - 1;
		    if(hv->buf_pos < 0)
			hv->buf_pos = 0;

		    /* Unselect. */
		    hv->buf_sel_start = hv->buf_sel_end = -1;

		    hv->modified = TRUE;

		    HViewDraw(hv);
		    HViewUpdateMenus(hv);
		    HViewSetStatusMessage(hv, NULL, FALSE);
		    if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
			GTK_VISIBILITY_FULL
		    )
			HViewScrollTo(hv, hv->buf_pos, 0.5);
		}
                status = TRUE;
	    }
            /* Cut? */
            else if((key->keyval == 'x') &&  (key->state == GDK_CONTROL_MASK))
	    {
		if(key_press)
		    HViewCutCB(NULL, hv);
                status = TRUE;
	    }
            /* Copy? */
            else if((key->keyval == 'c') &&  (key->state == GDK_CONTROL_MASK))
            {
		if(key_press)
		    HViewCopyCB(NULL, hv);
                status = TRUE;
            }
            /* Paste? */
            else if((key->keyval == 'v') &&  (key->state == GDK_CONTROL_MASK))
            {
		if(key_press)
		    HViewPasteCB(NULL, hv);
                status = TRUE;
            }
	    /* Toggle edit mode? */
            else if((key->keyval == 'm') &&  (key->state == GDK_CONTROL_MASK))
            {
		if(key_press)
		    HViewSetEditMode(
			hv,
			(hv->edit_mode == HVIEW_EDIT_MODE_HEX) ?
			    HVIEW_EDIT_MODE_ASCII : HVIEW_EDIT_MODE_HEX
		    );
		status = TRUE;
            }
	    break;
	}

	return(status);
#undef STOP_SIGNAL_EMIT
}

/*
 *	"button_press_event" or "button_release_event" callback.
 */
gint HViewButtonCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint etype;
	gboolean button_press;
        hview_struct *hv = (hview_struct *)data;
        if((button == NULL) || (hv == NULL))
            return(FALSE);

	etype = button->type;
	button_press = (etype == GDK_BUTTON_PRESS) ? TRUE : FALSE;
	if(button_press)
	    gtk_widget_grab_focus(widget);

	switch(button->button)
	{
	  case 3:
	    if(button_press)
	    {
		/* Map right click menu. */
		GtkMenu *menu = (GtkMenu *)hv->menu;
		if(menu != NULL)
		    gtk_menu_popup(
			menu, NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
	    }
	    break;

	  case 2:
	    if(button_press)
	    {
		GtkWidget *w = hv->view_da;

		hv->dragging_scroll = TRUE;
		hv->last_motion_x = button->x;
		hv->last_motion_y = button->y;
		if(w != NULL)
		    gdk_window_set_cursor(w->window, hv->translate_cur);
	    }
	    else
	    {
                GtkWidget *w = hv->view_da;

                hv->dragging_scroll = FALSE;
                if(w != NULL)
                    gdk_window_set_cursor(w->window, NULL);
	    }
	    break;

	  case 1:
	    if(button_press)
	    {
		/* Focus cell, set selection, and mark dragging. */
		hv->dragging_selection = TRUE;

		hv->buf_sel_start = hv->buf_pos =
		    HViewPositionXYToBuffer(
		    hv, button->x, button->y
		);
		hv->buf_sel_end = -1;

		HViewDraw(hv);
		HViewSetStatusMessage(hv, NULL, FALSE);
		if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
		    GTK_VISIBILITY_FULL
		)
		    HViewScrollTo(hv, hv->buf_pos, 0.5);
	    }
	    else
	    {
		/* End dragging. */
		hv->dragging_selection = FALSE;

		hv->buf_pos = HViewPositionXYToBuffer(
                    hv, button->x, button->y
                );
		if(hv->buf_sel_end < 0)
		    hv->buf_sel_start = -1;

                if(hv->scroll_toid != (guint)-1)
                {
                    gtk_timeout_remove(hv->scroll_toid);
                    hv->scroll_toid = (guint)-1;
                }

                HViewUpdateMenus(hv);
                HViewSetStatusMessage(hv, NULL, FALSE);
	    }
	    break;

	}

	return(TRUE);
}

/*
 *      "pointer_motion_event" callback.
 */
gint HViewMotionCB(
        GtkWidget *widget, GdkEventMotion *motion, gpointer data
)
{
	GtkAdjustment *adj;
        hview_struct *hv = (hview_struct *)data;
        if((motion == NULL) || (hv == NULL))
            return(FALSE);

	adj = hv->vadj;
	if(adj == NULL)
	    return(FALSE);

	/* Request for further motion events if this is a hint and
	 * the pointer button 1 is pressed.
	 */
	if(motion->is_hint &&
           (hv->dragging_selection || hv->dragging_scroll)
	)
	{
	    gint x, y;
	    GdkModifierType mask;

	    /* Request further motion events. */
	    gdk_window_get_pointer(
		motion->window, &x, &y, &mask
	    );
	}

	/* Dragging selection? */
	if(hv->dragging_selection)
	{
	    gint	x = (gint)motion->x,
			y = (gint)motion->y;
	    gint width, height;
	    GdkWindow *window = motion->window;

	    gdk_window_get_size(window, &width, &height);

	    hv->buf_sel_end = hv->buf_pos = HViewPositionXYToBuffer(
		hv, x, y
	    );
	    HViewDraw(hv);
	    HViewSetStatusMessage(hv, NULL, FALSE);
	    if(HViewIsBufferPositionVisible(hv, hv->buf_pos) !=
	        GTK_VISIBILITY_FULL
	    )
		HViewScrollTo(
		    hv, hv->buf_pos,
		    (y < (height / 2)) ? 0.0 : 1.0
		);


	    /* Check if motion y has moved up or down. */

	    /* Above window? */
	    if(y < 0)
	    {
		if(hv->scroll_toid == (guint)-1)
		    hv->scroll_toid = gtk_timeout_add(
			180,
			(GtkFunction)HViewScrollUpTOCB,
			hv
		    );
	    }
	    /* Below window? */
	    else if(y >= height)
	    {
		if(hv->scroll_toid == (guint)-1)
		    hv->scroll_toid = gtk_timeout_add(
                        180,
                        (GtkFunction)HViewScrollDownTOCB,
                        hv
                    );
	    }
	    /* Inside window. */
	    else
	    {
                if(hv->scroll_toid != (guint)-1)
		{
		    gtk_timeout_remove(hv->scroll_toid);
		    hv->scroll_toid = (guint)-1;
		}
	    }
	}
	/* Dragging scroll? */
	else if(hv->dragging_scroll)
	{
	    gint dy = (gint)motion->y - hv->last_motion_y;
	    GtkAdjustment *adj = hv->vadj;

	    if(adj != NULL)
	    {
		adj->value -= (gfloat)dy;
		if((adj->value + adj->page_size) > adj->upper)
		    adj->value = adj->upper - adj->page_size;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
		gtk_signal_emit_by_name(
		    GTK_OBJECT(adj), "value_changed"
		);
	    }
	}

	/* Update last coordinates. */
	hv->last_motion_x = (gint)motion->x;
	hv->last_motion_y = (gint)motion->y;

	return(TRUE);
}

/*
 *	"value_changed" callback.
 */
void HViewValueChangedCB(GtkAdjustment *adj, gpointer data)
{
	HViewDraw((hview_struct *)data);
}

/*
 *	Insert callback.
 */
void HViewInsertCB(GtkWidget *w, gpointer data)
{
	gint i, j, n;
	GtkAdjustment *adj;
	hview_struct *hv = (hview_struct *)data;
	if(hv == NULL)
	    return;

	/* Get insert position (where cursor is). */
	i = hv->buf_pos;
	if(i >= hv->buf_len)
	    i = hv->buf_len - 1;
	if(i < 0)
	    i = 0;

	n = 1;		/* Bytes to insert. */

	/* Increase buffer allocation. */
	if(hv->buf_len < 0)
	    hv->buf_len = 0;
	hv->buf_len += n;
	hv->buf = (guint8 *)g_realloc(
	    hv->buf, hv->buf_len * sizeof(guint8)
	);
	if(hv->buf == NULL)
	{
	    hv->buf_len = 0;
	    hv->buf_pos = 0;
	    return;
	}

	/* Shift buffer. */
	for(j = hv->buf_len - 1; j >= (i + n); j--)
	    hv->buf[j] = hv->buf[j - n];

	/* Reset new inserted bytes. */
	for(j = i; j < (i + n); j++)
	    hv->buf[j] = 0x00;

	if(hv->buf_pos < 0)
	    hv->buf_pos = 0;

        hv->modified = TRUE;

        /* Update adjustment. */
	w = hv->view_da;
        adj = hv->vadj;
        if((adj != NULL) && (hv->cells_per_row > 0) && (hv->cell_height > 0))
        {
            gint ww, wh;

            gdk_window_get_size(w->window, &ww, &wh);

            adj->lower = 0.0f;
            adj->upper = (gfloat)(
                (hv->buf_len / hv->cells_per_row * hv->cell_height) +
                hv->cell_height
            );
            adj->page_size = (gfloat)wh;
            adj->step_increment = hv->cell_height;
            adj->page_increment = adj->page_size / 2.0f;

            gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
            gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
        }
	HViewUpdateMenus(hv);
	HViewSetStatusMessage(hv, NULL, FALSE);
}

/*
 *      Delete callback.
 */
void HViewDeleteCB(GtkWidget *w, gpointer data)
{
        gint i, j, n;
        GtkAdjustment *adj;
        hview_struct *hv = (hview_struct *)data;
        if(hv == NULL)
            return;

        if(hv->buf == NULL)
            return;

	if(HVIEW_IS_BUF_SELECTED(hv))
	{
	    /* Get delete position and how many characters to
	     * delete.
	     */
	    if(hv->buf_sel_start < hv->buf_sel_end)
	    {
		i = hv->buf_sel_start;
		n = hv->buf_sel_end - hv->buf_sel_start + 1;
	    }
	    else
	    {
		i = hv->buf_sel_end;
		n = hv->buf_sel_start - hv->buf_sel_end + 1;
	    }
	}
	else
	{
	    i = hv->buf_pos;
	    n = 1;
	}

        /* Make sure length n at index i does not exceed the length of
         * the buffer.
         */
	if((i + n) > hv->buf_len)
	    n = hv->buf_len - i;
	if(n <= 0)
	    return;

        /* At this point i is the starting index of the buffer and
         * n is the length.  All bounds have been checked.
         */

        /* Shift buffer. */
        for(j = i; (j + n) < hv->buf_len; j++)
            hv->buf[j] = hv->buf[j + n];

	/* Reallocate buffer. */
        hv->buf_len -= n;
	if(hv->buf_len <= 0)
	{
	    g_free(hv->buf);
	    hv->buf = NULL;
	    hv->buf_len = 0;
	    hv->buf_pos = 0;
	}
	else
	{
	    hv->buf = (guint8 *)g_realloc(
		hv->buf, hv->buf_len * sizeof(guint8)
	    );
	    if(hv->buf == NULL)
	    {
		hv->buf_len = 0;
		hv->buf_pos = 0;
		return;
	    }
        }

	/* Unselect. */
	hv->buf_sel_start = -1;
	hv->buf_sel_end = -1;

	if(hv->buf_pos > i)
	{
	    if(hv->buf_pos >= (i + n))
		hv->buf_pos -= n;
	    else
		hv->buf_pos -= (hv->buf_pos - i);
	}
	if(hv->buf_pos >= hv->buf_len)
	    hv->buf_pos = hv->buf_len - 1;
	if(hv->buf_pos < 0)
	    hv->buf_pos = 0;

        hv->modified = TRUE;

	/* Update adjustment. */
	adj = hv->vadj;
	w = hv->view_da;
        if((adj != NULL) && (hv->cells_per_row > 0) && (hv->cell_height > 0))
	{
            gint ww, wh;

            gdk_window_get_size(w->window, &ww, &wh);

            adj->lower = 0.0f;
            adj->upper = (gfloat)(
                (hv->buf_len / hv->cells_per_row * hv->cell_height) +
                hv->cell_height
            );
            adj->page_size = (gfloat)wh;
            adj->step_increment = hv->cell_height;
            adj->page_increment = adj->page_size / 2.0f;

	    if((adj->value + adj->page_size) > adj->upper)
	    {
		adj->value = adj->upper - adj->page_size;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
	    }

	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
	}
	HViewUpdateMenus(hv);
        HViewSetStatusMessage(hv, NULL, FALSE);
}

/*
 *	Cut callback.
 */
void HViewCutCB(GtkWidget *w, gpointer data)
{
	HViewCopyCB(w, data);
	HViewDeleteCB(w, data);
}

/*
 *	Copy callback.
 */
void HViewCopyCB(GtkWidget *w, gpointer data)
{
        gint i, n;
        hview_struct *hv = (hview_struct *)data;
        if(hv == NULL)
            return;

        if((hv->buf_sel_start < 0) || (hv->buf_sel_end < 0))
            return;

        /* Get delete position and how many characters to
         * delete.
         */
        if(hv->buf_sel_start < hv->buf_sel_end)
        {
            i = hv->buf_sel_start;
            n = hv->buf_sel_end - hv->buf_sel_start + 1;
        }
        else
        {
            i = hv->buf_sel_end;
            n = hv->buf_sel_start - hv->buf_sel_end + 1;
        }

	/* Make sure length n at index i does not exceed the length of
	 * the buffer.
	 */
        if((i + n) > hv->buf_len)
            n = hv->buf_len - i;
        if(n <= 0)
            return;

	/* At this point i is the starting index of the buffer and
	 * n is the length.  All bounds have been checked.
	 */
	ClipboardPutBinary(&hv->buf[i], n);
}

/*
 *	Paste callback.
 */
void HViewPasteCB(GtkWidget *w, gpointer data)
{
        gint i, j, n, dde_len;
	guint8 *dde;
        GtkAdjustment *adj;
        hview_struct *hv = (hview_struct *)data;
        if(hv == NULL)
            return;

	/* Get value from clipboard. */
	dde = ClipboardFetchBinary(&dde_len);
	if((dde == NULL) || (dde_len <= 0))
	{
	    g_free(dde);
	    return;
	}

        /* Get insert position (where cursor is). */
        i = hv->buf_pos;
        if(i >= hv->buf_len)
            i = hv->buf_len - 1;
        if(i < 0)
            i = 0;

        n = dde_len;	/* Bytes to insert. */

        /* Increase buffer allocation. */
        if(hv->buf_len < 0)
            hv->buf_len = 0;
        hv->buf_len += n;
        hv->buf = (guint8 *)g_realloc(
            hv->buf, hv->buf_len * sizeof(guint8)
        );
        if(hv->buf == NULL)
        {
            hv->buf_len = 0;
            hv->buf_pos = 0;
            return;
        }

        /* Shift buffer. */
        for(j = hv->buf_len - 1; j >= (i + n); j--)
            hv->buf[j] = hv->buf[j - n];

        /* Copy dde buffer to insert point. */
	memcpy(&hv->buf[i], dde, n * sizeof(guint8));

	g_free(dde);
	dde = NULL;
	dde_len = 0;

        if(hv->buf_pos < 0)
            hv->buf_pos = 0;

	hv->modified = TRUE;

        /* Update adjustment. */
        w = hv->view_da;
        adj = hv->vadj;
        if((adj != NULL) && (hv->cells_per_row > 0) && (hv->cell_height > 0))
        {
            gint ww, wh;

            gdk_window_get_size(w->window, &ww, &wh);

            adj->lower = 0.0f;
            adj->upper = (gfloat)(
                (hv->buf_len / hv->cells_per_row * hv->cell_height) +
                hv->cell_height
            );
            adj->page_size = (gfloat)wh;
            adj->step_increment = hv->cell_height;
            adj->page_increment = adj->page_size / 2.0f;

            gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
            gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
        }
	HViewUpdateMenus(hv);
        HViewSetStatusMessage(hv, NULL, FALSE);
}

/*
 *	Set edit mode to hex callback.
 */
void HViewEditModeHexCB(GtkWidget *w, gpointer data)
{
	HViewSetEditMode((hview_struct *)data, HVIEW_EDIT_MODE_HEX);
}

/*
 *	Set edit mode to ACSII callback.
 */
void HViewEditModeASCIICB(GtkWidget *w, gpointer data)
{
        HViewSetEditMode((hview_struct *)data, HVIEW_EDIT_MODE_ASCII);	
}

