#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvstatusbar.h"
#include "edvcfg.h"
#include "findwin.h"
#include "findwincb.h"
#include "endeavour.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/icon_search_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_stop_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_goto_20x20.xpm"

#include "images/icon_search_48x48.xpm"


gchar *EDVFindWinCurrentSearch(edv_findwin_struct *fw);
void EDVFindWinSetSearch(
        edv_findwin_struct *fw, const gchar *s,
        gbool record_history
);

gchar *EDVFindWinCurrentLocation(edv_findwin_struct *fw);
void EDVFindWinSetLocation(
        edv_findwin_struct *fw, const gchar *path,
        gbool record_history
);

gint EDVFindWinCurrentFindOP(edv_findwin_struct *fw);

void EDVFindWinSetReferenceWindow(
        edv_findwin_struct *fw,
        gint browser_num, gint imbr_num, gint recbin_num,
	gint archiver_num
);

void EDVFindWinListResetColumns(edv_findwin_struct *fw, gint op_code);
void EDVFindWinListAppend(
        edv_findwin_struct *fw,
        const gchar *path, const struct stat *lstat_buf,
	const gchar *excerpt
);
void EDVFindWinListClear(edv_findwin_struct *fw);

void EDVFindWinSyncConfiguration(edv_findwin_struct *fw);

edv_findwin_struct *EDVFindWinNew(gpointer core_ptr);
void EDVFindWinUpdateMenus(edv_findwin_struct *fw);
void EDVFindWinSetBusy(edv_findwin_struct *fw, gbool is_busy);
void EDVFindWinMap(edv_findwin_struct *fw);
void EDVFindWinUnmap(edv_findwin_struct *fw);
void EDVFindWinDelete(edv_findwin_struct *fw);


#define FINDWIN_BTN_WIDTH	(100 + (2 * 3))
#define FINDWIN_BTN_HEIGHT	(30 + (2 * 3))

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))



/*
 *      Returns the string value of the current search string.
 *
 *      Can return NULL on error.
 */
gchar *EDVFindWinCurrentSearch(edv_findwin_struct *fw)
{
        GtkCombo *combo;


        if(fw == NULL)
            return(NULL);

        combo = (GtkCombo *)fw->search_combo;
        if(combo == NULL)
            return(NULL);

        return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *      Sets the search combo of the given find window to the value of
 *      the string s.
 *
 *      If record_history is TRUE then the current value will be recorded
 *      on the combo's list before setting of the new value.
 *      Duplicate values will not be recorded.
 */
void EDVFindWinSetSearch(
        edv_findwin_struct *fw, const gchar *s,
        gbool record_history
)
{
        GtkCombo *combo;
        gchar *new_s;


        if((fw == NULL) || (s == NULL))
            return;

        combo = (GtkCombo *)fw->search_combo;
        if(combo == NULL)
            return;

        /* Copy given path as the new path. */
        new_s = g_strdup(s);

        /* Check for no change in value. */
        if(new_s != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            if(cstrptr != NULL)
            {
                /* No change in value? */
                if(!strcmp(cstrptr, new_s))
                {
                    g_free(new_s);
                    new_s = NULL;
                    return;
                }
            }
        }

        /* Get old value from combo's entry and record it on the list. */
        if(record_history)
        {
            gchar *old_s;
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            old_s = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
            if(old_s != NULL)
            {
                GUIComboAddItem(combo, old_s);
                g_free(old_s);
                old_s = NULL;
            }
        }

        /* Set new value on entry and deallocate coppied new_s. */
        if(new_s != NULL)
        {
            gtk_entry_set_text(GTK_ENTRY(combo->entry), new_s);
            g_free(new_s);
            new_s = NULL;
        }
}

/*
 *      Returns the path to the current location of the find window.
 *
 *      Can return NULL on error.
 */
gchar *EDVFindWinCurrentLocation(edv_findwin_struct *fw)
{
        GtkCombo *combo;


        if(fw == NULL)
            return(NULL);

        combo = (GtkCombo *)fw->location_combo;
        if(combo == NULL)
            return(NULL);

        return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *      Sets the location combo of the given find window to the value of
 *      path.
 *
 *      If record_history is TRUE then the current value will be recorded
 *      on the combo's list before setting of the new value.
 *      Duplicate values will not be recorded.
 */
void EDVFindWinSetLocation(
        edv_findwin_struct *fw, const gchar *path,
        gbool record_history
)
{
        GtkCombo *combo;
        gchar *new_path;


        if((fw == NULL) || (path == NULL))
            return;

        combo = (GtkCombo *)fw->location_combo;
        if(combo == NULL)
            return;

        /* Copy given path as the new path. */
        new_path = g_strdup(path);

        /* Simplify new path, reducing occurances of /.. */
        SimplifyPath(new_path);


        /* Check for no change in value. */
        if(new_path != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            if(cstrptr != NULL)
            {
                /* No change in value? */
                if(!strcmp(cstrptr, new_path))
                {
                    g_free(new_path);
                    new_path = NULL;
                    return;
                }
            }
        }

        /* Get old value from combo's entry and record it on the list. */
        if(record_history)
        {
            gchar *old_path;
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            old_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
            if(old_path != NULL)
            {
                GUIComboAddItem(combo, old_path);
                g_free(old_path);
                old_path = NULL;
            }
        }

        /* Set new value on entry and deallocate coppied new_path. */
        if(new_path != NULL)
        {
            gtk_entry_set_text(GTK_ENTRY(combo->entry), new_path);
            g_free(new_path);
            new_path = NULL;
        }
}

/*
 *	Returns the current find operation code, one of EDV_FINDWIN_FIND_*.
 */
gint EDVFindWinCurrentFindOP(edv_findwin_struct *fw)
{
	gint i;
	const gchar *value, *value2;
	GList *glist;
	GtkCombo *combo;


	if(fw == NULL)
	    return(EDV_FINDWIN_FIND_OBJECT_NAME);

	combo = (GtkCombo *)fw->find_op_combo;
	if(combo == NULL)
	    return(EDV_FINDWIN_FIND_OBJECT_NAME);

	value = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	glist = (GList *)GUIComboGetList(combo);
	if((value == NULL) || (glist == NULL))
	    return(EDV_FINDWIN_FIND_OBJECT_NAME);

	i = 0;
	while(glist != NULL)
	{
	    value2 = (const gchar *)glist->data;
	    if((value2 != NULL) ? !strcasecmp(value2, value) : FALSE)
		return(i);

	    i++;
	    glist = glist->next;
	}

	return(EDV_FINDWIN_FIND_OBJECT_NAME);
}

/*
 *	Sets the reference window for the given find window by the given
 *	index numbers of the corresponding window structures on the core
 *	structure. Only one may be non-negative.
 *
 *	The role of the find window will also be set based on which
 *	reference window was specified.
 */
void EDVFindWinSetReferenceWindow(
        edv_findwin_struct *fw,
        gint browser_num, gint imbr_num, gint recbin_num,
	gint archiver_num
)
{
	if(fw == NULL)
	    return;

	fw->browser_num = browser_num;
	fw->imbr_num = imbr_num;
	fw->recbin_num = recbin_num;
	fw->archiver_num = archiver_num;

	/* Set the role depending on which reference window was specified. */
	if(browser_num > -1)
	    fw->role = EDV_FINDWIN_ROLE_DISK_OBJECT;
        else if(imbr_num > -1)
            fw->role = EDV_FINDWIN_ROLE_DISK_OBJECT;
        else if(recbin_num > -1)
            fw->role = EDV_FINDWIN_ROLE_RECYCLED_OBJECT;
        else if(archiver_num > -1)
            fw->role = EDV_FINDWIN_ROLE_ARCHIVE_OBJECT;
}


/*
 *	Resets the column widths and titles based on the given find
 *	operation code.
 */
void EDVFindWinListResetColumns(edv_findwin_struct *fw, gint op_code)
{
	gint i;
        GtkCList *clist;
	gint width[DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX];
	gint justify[DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX];
	gchar *title[DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX];
	edv_core_struct *core_ptr;


        if(fw == NULL)
            return;

	core_ptr = (edv_core_struct *)fw->core_ptr;
	if(core_ptr == NULL)
	    return;

        clist = (GtkCList *)fw->results_clist;
        if(clist == NULL)
            return;

	/* Make sure we have enough columns as expected, check compile time
	 * defination for number of columns with expected number of
	 * columns.
	 */
	if(DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX < 3)
	    return;

        gtk_clist_freeze(clist);

	/* Reset all column values. */
	for(i = 0; i < DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX; i++)
	{
	    title[i] = "";
	    width[i] = 10;
	    justify[i] = GTK_JUSTIFY_LEFT;
	}

        /* Update clist column settings. */
/*	gtk_clist_column_titles_active(clist); */
        gtk_clist_column_titles_show(clist);

	/* Set column values by find operation code. */
	switch(op_code)
	{
	  case EDV_FINDWIN_FIND_OBJECT_NAME:
	    title[0] = "Name";
	    width[0] = 150;
            title[1] = "Last Modified";
            width[1] = 120;
	    title[2] = "Location";
	    width[2] = 200;
	    break;

          case EDV_FINDWIN_FIND_OBJECT_CONTENT:
            title[0] = "Name";	/* And the location. */
            width[0] = 150;
            title[1] = "Excerpt";
            width[1] = 450;
            title[2] = "Last Modified";
            width[2] = 120;
            title[3] = "Location";
            width[3] = 200;
            break;
	}

	/* Update each column heading. */
        for(i = 0; i < DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX; i++)
        {
            gtk_clist_set_column_auto_resize(
                clist, i, FALSE
            );
            gtk_clist_set_column_title(clist, i, title[i]);
            gtk_clist_set_column_width(clist, i, width[i]);
            gtk_clist_set_column_justification(clist, i, justify[i]);
        }

        gtk_clist_thaw(clist);
}

/*
 *	Appends an item to the results list specified by the given disk
 *	object path and local stats.
 */
void EDVFindWinListAppend(
	edv_findwin_struct *fw,
	const gchar *path, const struct stat *lstat_buf,
	const gchar *excerpt
)
{
	gint op_code, relativity;
        const gchar *format, *date_str;
	gchar **text;
	const gchar *name;
	gint i, row, column, columns;
        GdkPixmap *pixmap;
        GdkBitmap *mask;
	GtkCList *clist;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;


	if((fw == NULL) || (path == NULL))
            return;

	op_code = EDVFindWinCurrentFindOP(fw);

	core_ptr = (edv_core_struct *)fw->core_ptr;
	if(core_ptr == NULL)
	    return;

	clist = (GtkCList *)fw->results_clist;
	if(clist == NULL)
	    return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );


	gtk_clist_freeze(clist);

	/* Allocate row values. */
	columns = MAX(clist->columns, 1);
	text = (gchar **)g_malloc0(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)
	    text[i] = g_strdup("");

	/* Append new row. */
	row = gtk_clist_append(clist, text);

	/* Deallocate row data. */
	for(i = 0; i < columns; i++)
	    g_free(text[i]);
	g_free(text);
	text = NULL;

	/* Error appending row? */
	if(row < 0)
	{
            gtk_clist_thaw(clist);
            return;
        }


        /* Create a new disk object structure, this will be used as row
         * data since the row data always has a disk object structure as
         * its data regardless of the role.
	 *
	 * Here we will create the disk object structure by the role.
         */
	switch(fw->role)
	{
	  case EDV_FINDWIN_ROLE_DISK_OBJECT:
	    obj = EDVObjectNew();
	    if(obj != NULL)
	    {
		EDVObjectSetPath(obj, path);
		if(lstat_buf != NULL)
		    EDVObjectSetStat(obj, lstat_buf);
		EDVObjectValidateLink(obj);
	    }
	    break;

          case EDV_FINDWIN_ROLE_RECYCLED_OBJECT:
            obj = EDVObjectNew();
            if(obj != NULL)
            {
		edv_recbin_object_struct *recobj = EDVRecBinObjectStat(
		    EDVCFGItemListGetValueS(
			core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
		    ),
		    (guint)atoi(path)
		);
		if(recobj != NULL)
		{
		    /* Disk object's name will actually be the recycled
		     * object index number.
		     */
		    g_free(obj->name);
		    obj->name = g_strdup(path);

                    g_free(obj->full_path);
                    obj->full_path = g_strdup_printf(
			"%s%c%s",
			recobj->original_path, DIR_DELIMINATOR,
			recobj->name
		    );
                    obj->type = recobj->type;
                    obj->permissions = recobj->permissions;
                    obj->access_time = recobj->access_time;
                    obj->modify_time = recobj->modify_time;
                    obj->change_time = recobj->change_time;
                    obj->owner_id = recobj->owner_id;
                    obj->group_id = recobj->group_id;
		    obj->size = recobj->size;

		    EDVRecBinObjectDelete(recobj);
		}
            }
	    break;

	  case EDV_FINDWIN_ROLE_ARCHIVE_OBJECT:
	    obj = EDVObjectNew();
            if(obj != NULL)
            {
                g_free(obj->full_path);
                obj->full_path = g_strdup(path);
	    }
	    break;

	  default:
	    obj = NULL;
	    break;
	}
	if(obj == NULL)
	{
	    gtk_clist_thaw(clist);
	    return;
	}

	/* Get pointer to object's name, always dirive it from the full
	 * path since the obj->name may contain some other value
	 * depending on the role.
	 */
	if(obj->full_path != NULL)
	{
	    name = strrchr(obj->full_path, DIR_DELIMINATOR);
	    if(name != NULL)
		name++;
	    else
		name = obj->full_path;
	}
	else
	{
	    name = NULL;
	}


        /* Get pixmap and mask for the given object. */
        EDVMatchObjectIcon(
            core_ptr->device, core_ptr->total_devices,
            core_ptr->mimetype, core_ptr->total_mimetypes,
            obj->type,
            obj->full_path,
            obj->link_valid, obj->permissions,
            0,                  /* Small icons. */
            &pixmap, &mask,
            NULL, NULL, NULL, NULL
        );

	/* Set name. */
	column = 0;
        if(pixmap != NULL)
            gtk_clist_set_pixtext(
                clist, row, column,
                (name != NULL) ? name : "(null)",
                EDV_LIST_PIXMAP_TEXT_SPACING,
                pixmap, mask
            );
        else
            gtk_clist_set_text(
                clist, row, column,
                (name != NULL) ? name : "(null)"
            );

	/* Set subsequent cells by find operation code. */
	switch(op_code)
	{
	  case EDV_FINDWIN_FIND_OBJECT_NAME:
	    /* Last modified. */
	    column = 1;
	    date_str = EDVDateFormatString(
		obj->modify_time, format, relativity
	    );
            gtk_clist_set_text(
                clist, row, column, date_str
            );
            /* Location. */
            column = 2;
            gtk_clist_set_text(
                clist, row, column,
                GetParentDir(obj->full_path)
            );
	    break;

          case EDV_FINDWIN_FIND_OBJECT_CONTENT:
            /* Excerpt. */
            column = 1;
	    if(excerpt != NULL)
                gtk_clist_set_text(
                    clist, row, column, excerpt
                );
            /* Last modified. */
            column = 2;
            date_str = EDVDateFormatString(
                obj->modify_time, format, relativity
            );
            gtk_clist_set_text(
                clist, row, column, date_str
            );
            /* Location. */
            column = 3;
            gtk_clist_set_text(
                clist, row, column,
                GetParentDir(obj->full_path)
            );
	    break;
	}

	/* Set the disk object structure as the row data for this new row.
	 * The disk object structure should not be referenced again after
	 * this point.
	 */
        gtk_clist_set_row_data_full(
            clist, row,
            obj, (GtkDestroyNotify)EDVFindWinListItemDestroyCB
        );
        obj = NULL;

	gtk_clist_thaw(clist);
}

/*
 *	Clears the results clist.
 */
void EDVFindWinListClear(edv_findwin_struct *fw)
{
	GtkCList *clist;


	if(fw == NULL)
	    return;

	clist = (GtkCList *)fw->results_clist;
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}



/*
 *	Sets values on the given find window to the configuration list on
 *	the core structure of the given find window.
 */
void EDVFindWinSyncConfiguration(edv_findwin_struct *fw)
{
        GtkWidget *w;
        edv_core_struct *core_ptr;


        if(fw == NULL)
            return;

        core_ptr = (edv_core_struct *)fw->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get size of toplevel. */
        w = fw->toplevel;
        if(w != NULL)
        {
            GdkWindow *window = w->window;
            gint x = 0, y = 0, width, height;


            if(window != NULL)
                gdk_window_get_root_origin(window, &x, &y);
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_X,
                x, FALSE
            );
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_Y,
                y, FALSE
            );
	    gtk_widget_set_uposition(w, x, y);

            width = w->allocation.width;
            height = w->allocation.height;
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_WIDTH,
                width, FALSE
            );
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_HEIGHT,
                height, FALSE
            );
            gtk_widget_set_usize(w, width, height);
        }

	/* Find operation. */
        EDVCFGItemListSetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_FIND_OP,
            EDVFindWinCurrentFindOP(fw), FALSE
        );

        /* Search string. */
        w = fw->search_combo;
        if(w != NULL)
        {
	    GtkEntry *entry = (GtkEntry *)GTK_COMBO(w)->entry;

            EDVCFGItemListSetValueS(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_SEARCH_STRING,
                gtk_entry_get_text(entry), FALSE
            );
	}


	/* Case sensitive. */
	w = fw->case_sensitive_check;
	if(w != NULL)
	{
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_CASE_SENSITIVE,
                GTK_TOGGLE_BUTTON(w)->active, FALSE
            );
	}

        /* Recursive. */
        w = fw->recursive_check;
        if(w != NULL)
        {
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_RECURSIVE,
                GTK_TOGGLE_BUTTON(w)->active, FALSE
            );
        }

}



/*
 *	Creates a new find window.
 */
edv_findwin_struct *EDVFindWinNew(gpointer core_ptr)
{
	gint i;
	gint border_major = 5, border_minor = 2;
	gint bw = FINDWIN_BTN_WIDTH, bh = FINDWIN_BTN_HEIGHT;
        gint toplevel_x = 0, toplevel_y = 0;
        gint	toplevel_width = EDV_DEF_FINDWIN_WIDTH,
		toplevel_height = EDV_DEF_FINDWIN_HEIGHT;
	gbool status_bar_map_state = TRUE;
	gint find_op = EDV_FINDWIN_FIND_OBJECT_NAME;
	const gchar *search_string = NULL;
	gbool case_sensitive = FALSE, recursive = FALSE;
        edv_status_bar_struct *status_bar;
        GdkColormap *colormap = NULL;
        GList *glist;
        GtkWidget *w, *parent, *parent2, *parent3, *parent4;
        gpointer combo_rtn;
        GtkCombo *combo;
        GtkCList *clist;
        GdkWindow *window;
        GtkAccelGroup *accelgrp;
        edv_findwin_struct *fw = (edv_findwin_struct *)g_malloc0(
            sizeof(edv_findwin_struct)
        );
        if(fw == NULL)
            return(fw);


        /* Get values from core structure. */
        if(core_ptr != NULL)
        {
            edv_core_struct *c_ptr = (edv_core_struct *)core_ptr;

            toplevel_x = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_X
            );
            toplevel_y = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_Y
            );
            toplevel_width = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_WIDTH
            );
            toplevel_height = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_HEIGHT
            );
	    find_op = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_FIND_OP
            );
	    search_string = EDVCFGItemListGetValueS(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_SEARCH_STRING
	    );
	    case_sensitive = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_CASE_SENSITIVE
            );
	    recursive = EDVCFGItemListGetValueI(
                c_ptr->cfg_list, EDV_CFG_PARM_FINDWIN_RECURSIVE
            );

        }

        /* Begin resetting values. */
        fw->initialized = TRUE;
        fw->map_state = FALSE;
        fw->processing = FALSE;
        fw->busy_count = 0;
	fw->core_ptr = core_ptr;
        fw->stop_count = 0;

	fw->status_bar_map_state = status_bar_map_state;

	fw->browser_num = -1;
	fw->imbr_num = -1;
	fw->recbin_num = -1;
	fw->archiver_num = -1;

	fw->role = EDV_FINDWIN_ROLE_DISK_OBJECT;
	fw->find_op = EDV_FINDWIN_FIND_OBJECT_NAME;
	fw->last_write_protect_state = -1;


        /* Keyboard accelerator group. */
        fw->accelgrp = accelgrp = gtk_accel_group_new();

        /* Begin creating widgets. */
        /* Toplevel. */
        fw->toplevel = w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        gtk_widget_set_uposition(
            w, toplevel_x, toplevel_y
        );
        gtk_widget_set_usize(
            w, toplevel_width, toplevel_height
        );
        gtk_window_set_policy(
            GTK_WINDOW(w),
            TRUE, TRUE, TRUE
        );
        gtk_window_set_title(GTK_WINDOW(w), "Find Objects");
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
            GdkGeometry geometry;

            geometry.min_width = 100;
            geometry.min_height = 70;

            geometry.base_width = 0;
            geometry.base_height = 0;

            geometry.width_inc = 1;
            geometry.height_inc = 1;
/*
            geometry.min_aspect = 1.3;
            geometry.max_aspect = 1.3;
 */
            gdk_window_set_geometry_hints(
                window, &geometry,
                GDK_HINT_MIN_SIZE |
                GDK_HINT_BASE_SIZE |
                /* GDK_HINT_ASPECT | */
                GDK_HINT_RESIZE_INC
            );

            colormap = gdk_window_get_colormap(window);

            GUISetWMIcon(window, (u_int8_t **)icon_search_48x48_xpm);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVFindWinDeleteEventCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "destroy",
            GTK_SIGNAL_FUNC(EDVFindWinDestroyCB), fw
        );
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        gtk_container_set_border_width(GTK_CONTAINER(w), 0);
        parent = w;


        /* Main vbox. */
        fw->main_vbox = w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


	/* Upper section hbox to separate two columns. */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_set_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
        parent2 = w;


	/* Left panel vbox. */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;

	/* Begin creating search criteria widgets. */
	w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

	/* Hbox for search string combo and find operation combo. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

	/* Create glist containing a list of find operation names, where
	 * the index of each name corresponds to one of EDV_FINDWIN_FIND_*.
	 */
	glist = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        glist = g_list_append(glist, "Object Name");
        glist = g_list_append(glist, "Object Content");
#endif
#ifdef PROG_LANGUAGE_SPANISH
        glist = g_list_append(glist, "El Nombre Objetivo");
        glist = g_list_append(glist, "El Contenido Objetivo");
#endif
#ifdef PROG_LANGUAGE_FRENCH
        glist = g_list_append(glist, "Opposer le Nom");
        glist = g_list_append(glist, "Opposer le Contenu");
#endif

        /* Find operation combo. */
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Find:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Hallazgo:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Dcouverte:",
#endif
            "",
            glist,
            2,		/* Maximum items. */
            &combo_rtn,
            fw,
            EDVFindWinComboActivateCB,
            NULL
        );
	fw->find_op_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

	w = fw->find_op_combo;
	combo = GTK_COMBO(w);
	gtk_widget_set_usize(w, 150, -1);

	g_list_free(glist);

	w = combo->entry;
	gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	glist = GUIComboGetList(combo);
	glist = g_list_nth(glist, find_op);
	if((glist != NULL) ? (glist->data != NULL) : FALSE)
	    gtk_entry_set_text(GTK_ENTRY(w), (const gchar *)glist->data);


        /* Search string combo. */
        glist = NULL;
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Matching:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Parejo:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Egaler:",
#endif
            "",
            glist,
            20,         /* Maximum items. */
            &combo_rtn,
            fw,
            EDVFindWinComboActivateCB,
            NULL
        );
        fw->search_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
        gtk_widget_show(w);

        w = fw->search_combo;
        combo = GTK_COMBO(w);
        gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	EDVEntrySetDND((edv_core_struct *)core_ptr, w);
	if(search_string != NULL)
	    gtk_entry_set_text(GTK_ENTRY(w), search_string);

        /* Hbox for location combo. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Location combo. */
        glist = NULL;
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Starting At:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Comenzar En:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Commencer A:",
#endif
            "",
            glist,
            20,		/* Maximum items. */
            &combo_rtn,
            fw,
            EDVFindWinComboActivateCB,
            NULL
        );
        fw->location_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
        gtk_widget_show(w);

        w = fw->location_combo;
        combo = GTK_COMBO(w);
        gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
        EDVEntrySetDND((edv_core_struct *)core_ptr, w);



        /* Hbox for options. */
        w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

	/* Case sensitive check. */
	fw->case_sensitive_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Case Sensitive"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Caso Sensible"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Le Cas Sensible"
#endif
	);
        GTK_TOGGLE_BUTTON(w)->active = case_sensitive;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Specifies if string matches should be case sensitive or not"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Especifica si los iguales de cuerda no deben ser el casos sensibles ni"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Spcifie si les allumettes de ficelle devraient tre des cas sensibles ou pas"
#endif
        );
        gtk_widget_show(w);

        /* Recursive check. */
        fw->recursive_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Recursive"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Recursive"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Recursive"
#endif
        );
	GTK_TOGGLE_BUTTON(w)->active = recursive;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Specifies if the search process should recurse into subdirectories"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Especifica si el proceso de la bsqueda debe recurse en el subdirectories"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Spcifie si le procd de recherche doit recurse dans subdirectories"
#endif
        );
        gtk_widget_show(w);



        /* Separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Scrolled window for results clist. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC,
            GTK_POLICY_AUTOMATIC
        );
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Contents clist. */
        fw->results_clist = w = gtk_clist_new(
	    DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX
	);
        clist = GTK_CLIST(w);
        gtk_widget_add_events(
            w,
            GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
            GDK_BUTTON_PRESS_MASK
        );
/*
        gtk_signal_connect(
            GTK_OBJECT(w), "key_press_event",
            GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_release_event",
            GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
        );
 */
        gtk_signal_connect_after(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(EDVFindWinButtonCB), fw
        );
        gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
        for(i = 0; i < clist->columns; i++)
            gtk_clist_set_column_width(clist, i, 10);
        gtk_clist_column_titles_passive(clist);
        gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
        gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_signal_connect(
            GTK_OBJECT(w), "click_column",
            GTK_SIGNAL_FUNC(EDVFindWinClickColumnCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "select_row",
            GTK_SIGNAL_FUNC(EDVFindWinSelectRowCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "unselect_row",
            GTK_SIGNAL_FUNC(EDVFindWinUnselectRowCB), fw
        );
        gtk_widget_show(w);







        /* Right panel vbox. */
        w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	/* Buttons set vbox. */
        w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Search button. */
        fw->search_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_search_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Search",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "La Bsqueda",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Recherche",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinSearchCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Start the search"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Comience la bsqueda"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Commencer la recherche"
#endif
        );
        gtk_widget_show(w);

        /* Stop button. */
        fw->stop_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_stop_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Stop",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "La Parada",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Arrt",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinStopCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Stop the current find procedure"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Pare el procedimiento actual del hallazgo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Arrter la procdure de dcouverte actuelle"
#endif
        );
        gtk_widget_show(w);

        /* Clear button. */
        fw->clear_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_clear_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Clear",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Claro",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Clair",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinClearCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Clear search string and results list"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "La cuerda clara de la bsqueda y resultados listan"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Clair recherche et rsultats numre"
#endif
        );
        gtk_widget_show(w);


        /* Buttons set vbox. */
        w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Close button. */
        fw->close_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_close_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Close",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cierre",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Proche",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinCloseCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Close this window"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cierre esta ventana"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Fermer cette fentre"
#endif
        );
        gtk_widget_show(w);




        /* Separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Buttons hbox. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

	/* Open button. */
	fw->open_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_open_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Open",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Abierto",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ouvert",
#endif
            NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinOpenCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Open selected object"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Abra escogido se opone"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ouvrir l'objet choisi"
#endif
        );
        gtk_widget_show(w);

        /* Open with button. */
        fw->open_with_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_open_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Open With",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Abra Con",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ouvrir Avec",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinOpenWithCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Open selected object using a specific method"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Abra escogido se opone usar un method especfico"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Ouvrir l'objet choisi pour utiliser un method spcifique"
#endif
        );
        gtk_widget_show(w);

        /* Goto button. */
        fw->goto_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_goto_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "GoTo",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "VayaA",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Aller A",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, bw, bh);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindWinGotoCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Go to selected object"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Vaya a escogido se opone"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Aller  l'objet choisi"
#endif
        );
	gtk_widget_show(w);



        /* Create status bar. */
        fw->status_bar = status_bar = EDVStatusBarNew(
            core_ptr, fw->main_vbox
        );
        if(status_bar != NULL)
        {
            if(fw->status_bar_map_state)
                EDVStatusBarMap(status_bar);
            else
                EDVStatusBarUnmap(status_bar);
        }


	EDVFindWinUpdateMenus(fw);

	return(fw);
}

/*
 *      Updates all widgets on the given find window to reflect the current
 *      data.
 */
void EDVFindWinUpdateMenus(edv_findwin_struct *fw)
{
        gbool write_protect, write_protect_changed = FALSE;
        gbool sensitive;
	gint sel_row;
	GList *glist;
        GtkWidget *w;
	GtkCList *clist;
        edv_core_struct *core_ptr;


        if(fw == NULL)
            return;

        if(!fw->initialized)
            return;

        core_ptr = (edv_core_struct *)fw->core_ptr;
        if(core_ptr == NULL)
            return;

	clist = (GtkCList *)fw->results_clist;
	if(clist == NULL)
	    return;

	/* Get last selected row on the results clist. */
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;


#define DO_SET_SENSITIVE        \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}
#define DO_SHOW \
{ \
 if(w != NULL) \
  gtk_widget_show(w); \
}
#define DO_HIDE \
{ \
 if(w != NULL) \
  gtk_widget_hide(w); \
}
#define DO_SET_CHECK_STATE      \
{ \
 if((w != NULL) ? GTK_IS_CHECK_MENU_ITEM(w) : FALSE) \
  GTK_CHECK_MENU_ITEM(w)->active = state; \
}
        /* Get global write protect state. */
        write_protect = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
        );
        if(fw->last_write_protect_state < 0)
        {
            write_protect_changed = TRUE;
        }
        else
        {
            if((write_protect && !fw->last_write_protect_state) ||
               (!write_protect && fw->last_write_protect_state)
            )
                write_protect_changed = TRUE;
        }
        fw->last_write_protect_state = write_protect ? 1 : 0;


	/* Set location combo sensitive or not depending on the role. */
	if(fw->processing ||
	  (fw->role == EDV_FINDWIN_ROLE_RECYCLED_OBJECT)
	)
	    sensitive = FALSE;
	else
	    sensitive = TRUE;
	w = fw->location_combo;
	DO_SET_SENSITIVE


        /* Update buttons. */
        sensitive = fw->processing;
        w = fw->stop_btn;
        DO_SET_SENSITIVE

        sensitive = !fw->processing;
	w = fw->search_combo;
        DO_SET_SENSITIVE
        w = fw->case_sensitive_check;
        DO_SET_SENSITIVE
        w = fw->recursive_check;
        DO_SET_SENSITIVE
        w = fw->find_op_combo;
        DO_SET_SENSITIVE
        w = fw->search_btn;
        DO_SET_SENSITIVE
        w = fw->clear_btn;
        DO_SET_SENSITIVE
        w = fw->close_btn;
        DO_SET_SENSITIVE

	sensitive = (!fw->processing && (sel_row > -1)) ? TRUE : FALSE;
        w = fw->goto_btn;
        DO_SET_SENSITIVE

	sensitive = (!fw->processing && (sel_row > -1)) ? TRUE : FALSE;
        w = fw->open_btn;
        DO_SET_SENSITIVE
        w = fw->open_with_btn;
        DO_SET_SENSITIVE


        /* Update status bar. */
        EDVStatusBarUpdateMenus(fw->status_bar);

#undef DO_SHOW
#undef DO_HIDE
#undef DO_SET_SENSITIVE
#undef DO_SET_CHECK_STATE
}

/*
 *      Marks the find window as busy or ready.
 */
void EDVFindWinSetBusy(edv_findwin_struct *fw, gbool is_busy)
{
        GdkCursor *cursor;
        GtkWidget *w;
        edv_core_struct *core_ptr;


        if(fw == NULL)
            return;

        if(!fw->initialized)
            return;

        core_ptr = (edv_core_struct *)fw->core_ptr;
        if(core_ptr == NULL)
            return;

        w = fw->toplevel;
        if(w != NULL)
        {
            if(is_busy)
            {
                /* Increase busy count. */
                fw->busy_count++;

                /* If already busy then don't change anything. */
                if(fw->busy_count > 1)
                    return;

                cursor = EDVGetCursor(core_ptr, EDV_CURSOR_CODE_BUSY);
            }
            else
            {
                /* Reduce busy count. */
                fw->busy_count--;
                if(fw->busy_count < 0)
                    fw->busy_count = 0;

                /* If still busy do not change anything. */
                if(fw->busy_count > 0)
                    return;

                cursor = NULL;  /* Use default cursor. */
            }

            /* Update toplevel window's cursor. */
            if(w->window != NULL)
            {
                gdk_window_set_cursor(w->window, cursor);
                gdk_flush();
            }
        }
}

/*
 *      Maps the given find window.
 */
void EDVFindWinMap(edv_findwin_struct *fw)
{
        GtkWidget *w;
	GtkCombo *combo;


        if(fw == NULL)
            return;

        if(!fw->initialized)
            return;

        if(!fw->map_state)
        {
	    combo = (GtkCombo *)fw->search_combo;
	    if(combo != NULL)
	    {
		w = combo->entry;
		gtk_widget_grab_focus(w);
                gtk_widget_grab_default(w);
	    }

            w = fw->toplevel;
	    gtk_widget_show_raise(w);
	    fw->map_state = TRUE;
        }
}

/*
 *      Unmaps the given find window.
 */
void EDVFindWinUnmap(edv_findwin_struct *fw)
{
        GtkWidget *w;


        if(fw == NULL)
            return;

        if(!fw->initialized)
            return;

        if(fw->map_state)
        {
            w = fw->toplevel;
            if(w != NULL)
                gtk_widget_hide(w);

            fw->map_state = FALSE;
        }
}

/*
 *	Destroys all resources on the given find window and deallocates
 *	the structure itself.
 */
void EDVFindWinDelete(edv_findwin_struct *fw)
{
        GtkWidget **w;


        if(fw == NULL)
            return;

        if(fw->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

            /* Delete status bar. */
            EDVStatusBarDelete(fw->status_bar);
            fw->status_bar = NULL;

	    /* Destroy right-click menu. */
	    w = &fw->results_clist_menu;
	    fw->results_clist_open_mi = NULL;
	    fw->results_clist_open_with_mi = NULL;
	    fw->results_clist_goto_mi = NULL;
	    DO_DESTROY_WIDGET;

            /* Destroy toplevel and the rest of the widgets. */
            w = &fw->find_op_combo;
            DO_DESTROY_WIDGET
            w = &fw->search_combo;
            DO_DESTROY_WIDGET
            w = &fw->location_combo;
            DO_DESTROY_WIDGET
            w = &fw->case_sensitive_check;
            DO_DESTROY_WIDGET
            w = &fw->recursive_check;
            DO_DESTROY_WIDGET
            w = &fw->search_btn;
            DO_DESTROY_WIDGET
            w = &fw->stop_btn;
            DO_DESTROY_WIDGET
            w = &fw->clear_btn;
            DO_DESTROY_WIDGET
            w = &fw->close_btn;
            DO_DESTROY_WIDGET
            w = &fw->results_clist;
            DO_DESTROY_WIDGET
	    w = &fw->open_btn;
            DO_DESTROY_WIDGET
	    w = &fw->open_with_btn;
            DO_DESTROY_WIDGET
            w = &fw->goto_btn;
	    DO_DESTROY_WIDGET
	    w = &fw->main_vbox;
	    DO_DESTROY_WIDGET
	    w = &fw->toplevel;
            DO_DESTROY_WIDGET

            if(fw->accelgrp != NULL)
            {
                gtk_accel_group_unref(fw->accelgrp);
                fw->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }

        /* Deallocate other memory on find window structure. */

        /* Deallocate structure itself. */
        g_free(fw);
}
