#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/fio.h"

#include "edvtypes.h"
#include "edvutils.h"


void EDVSimplifyPath(gchar *path);
gboolean EDVIsParentPath(const gchar *parent, const gchar *child);
static gboolean EDVIsExtensionIterate(
        const gchar *name, gint name_len,
        const gchar *ext, gint ext_len
);
gboolean EDVIsExtension(const gchar *path, const gchar *ext);

gchar *EDVCopyShortenPath(const gchar *path, gint max);
gchar *EDVCopyEvaluateInputPath(
	const gchar *parent, const gchar *input_path
);
gchar *EDVTmpName(const gchar *dir);

void EDVGetDeviceNumbers(dev_t rdev, gint *major, gint *minor);
dev_t EDVFormatDeviceNumbers(gint major, gint minor);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Simplifies the path by removing any tailing deliminator characters
 *	and reducing any occurances of "/.." in the given path.
 */
void EDVSimplifyPath(gchar *path)
{
	gboolean is_absolute = FALSE;


	if(path == NULL)
	    return;

	if(*path == '\0')
	    return;

	if(*path == DIR_DELIMINATOR)
	    is_absolute = TRUE;

	/* Remove tailing deliminator characters if any. */
	StripPath(path);

	/* Reduce all occurances of "/.." in the path. */
	SimplifyPath(path);

	/* If the path was originally an absolute path and now its
	 * become an empty string then set path as toplevel.
	 */
	if(is_absolute && (*path == '\0'))
	    strcpy(path, "/");
}


/*
 *	Returns TRUE if the given parent path is a parent or grand
 *	parent of the given child path.
 *
 *	Will also return TRUE if parent path and child paths are
 *	identical.
 *
 *	Both paths must be absolute and simplified.
 */
gboolean EDVIsParentPath(const gchar *parent, const gchar *child)
{
	gboolean status;
	gint len;
	gchar *lparent, *lchild;
	gchar delim_str[2];

	
	if((parent == NULL) || (child == NULL))
	    return(FALSE);

	/* Both paths must be absolute. */
	if(!ISPATHABSOLUTE(parent) || !ISPATHABSOLUTE(child))
	    return(FALSE);

#define DO_FREE_LOCALS	\
{ \
 g_free(lparent); \
 lparent = NULL; \
\
 g_free(lchild); \
 lchild = NULL; \
}
	/* Make coppies of the parent and child paths. */
	lparent = g_strdup(parent);
	lchild = g_strdup(child);
	if((lparent == NULL) || (lchild == NULL))
	{
	    DO_FREE_LOCALS
	    return(FALSE);
	}


	/* Need to tack on a tailing DIR_DELIMINATOR character to the
	 * parent path for proper prefix matching below.
	 */
	delim_str[0] = DIR_DELIMINATOR;
	delim_str[1] = '\0';

	len = strlen(lparent);
	if(len > 0)
	{
	    if(lparent[len - 1] != DIR_DELIMINATOR)
		lparent = strcatalloc(lparent, delim_str);
	    if(lparent == NULL)
	    {
		DO_FREE_LOCALS
		return(FALSE);
	    }
        }

        len = strlen(lchild);
        if(len > 0)
        {
            if(lchild[len - 1] != DIR_DELIMINATOR)
                lchild = strcatalloc(lchild, delim_str);
            if(lchild == NULL)
            {
		DO_FREE_LOCALS
                return(FALSE);
            }
        }

	/* Check if the lparent is a prefix of lchild, if it is then
	 * it suggest that it is a parent or grand parent of the child
	 * path.
	 */
	if(strpfx(lchild, lparent))
	    status = TRUE;
	else
	    status = FALSE;

	/* Deallocate coppies of child and parent paths. */
	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

	return(status);
}

/*
 *	Called by EDVIsExtension() to check if ext matches the path.
 *
 *	The ext must only contain one word (no spaces).
 *
 *	Both path and ext may not be NULL.
 */
static gboolean EDVIsExtensionIterate(
	const gchar *name, gint name_len,
	const gchar *ext, gint ext_len
)
{
	/* Extension starts with a '.' deliminator? */
	if(*ext == '.')
	{
	    const gchar *name_ext = name + (name_len - ext_len);

	    /* Check if ext is a suffix of name. */
	    if(name_len < ext_len)
		return(FALSE);

	    return((g_strcasecmp(name_ext, ext)) ? FALSE : TRUE);
	}
	/* Not an extension, so use fnmatch(). */
	else
	{
	    return((fnmatch(ext, name, 0) == 0) ? TRUE : FALSE);
	}
}

/*
 *	Checks if at least one of the given list of space separated
 *	extensions ext is the given path's extensions such as
 *	".txt .doc .nfo *rc Makefile*".
 */
gboolean EDVIsExtension(const gchar *path, const gchar *ext)
{
	const gchar *ss, *name, *ext_ptr;
	gint name_len, ext_len;
	gchar *st;
	gchar cur_ext[NAME_MAX];


	if((path == NULL) || (ext == NULL))
	    return(FALSE);

	/* Given path and extensions list cannot be empty strings. */
	if((*path == '\0') || (*ext == '\0'))
	    return(FALSE);

	/* Seek to name portion of path. */
	name = strrchr(path, DIR_DELIMINATOR);
	if(name != NULL)
            name++;
        else
            name = path;

	name_len = strlen(name);

	/* Set ext_ptr to start of extensions list string, seeking past
	 * any initial spaces.
	 */
	ext_ptr = ext;
	while(ISBLANK(*ext_ptr))
	    ext_ptr++;

	/* Iterate through each word in the extensions list. */
	while(*ext_ptr != '\0')
	{
	    /* Copy this word in the extensions list string to
	     * cur_ext and calculate this extension's length as
	     * ext_len.
	     */
	    for(ext_len = 0,
		st = cur_ext,
                ss = ext_ptr;
                (ext_len < (NAME_MAX - 1)) &&
		!ISBLANK(*ss) && (*ss != '\0');
	        ext_len++
	    )
		*st++ = *ss++;
	    *st = '\0';

	    /* Check this extension word matches. */
	    if(EDVIsExtensionIterate(name, name_len, cur_ext, ext_len))
		return(TRUE);

	    /* At this point ss should be at the end of the word (at
	     * the first space) or at the end of the extensions list
	     * string.
	     */
	    ext_ptr = ss;
	    while(ISBLANK(*ext_ptr))
		ext_ptr++;
	}

	return(FALSE);
}


/*
 *	Returns a dynamically allocated path that contains no more
 *	than max characters.
 *
 *	If the given path is longer than max then the shortened
 *	return stirng will contain "..." prefixed to it.
 */
gchar *EDVCopyShortenPath(const gchar *path, gint max)
{
	gint len;


	if(path == NULL)
	    return(NULL);

	len = strlen(path);
	if((len > max) && (max > 3))
	{
	    /* Need to shorten string. */
	    gint i = len - max + 3;

	    return(g_strdup_printf(
		"...%s", &path[i]
	    ));
	}
	else
	{
	    return(g_strdup(path));   
	}
}

/*
 *	Returns a dynamically allocated path evaluated (all variables
 *	and special notations substituted) based on the given input_path.
 *
 *	If input_path (after evaluation) is not an absolute path then the
 *	given parent path (if it is not NULL) will be prefixed to it.
 *
 *	The final path will be simplified.
 *
 *	The calling function must deallocate the returned pointer.
 */
gchar *EDVCopyEvaluateInputPath(
        const gchar *parent, const gchar *input_path
)
{
	gchar *new_path;


	if(input_path == NULL)
	    return(NULL);

	/* Check for special prefix substitutions for the input path. */
	if(strcasepfx(input_path, "~"))
	{
	    /* Prefix the value from the HOME environment variable to
	     * the input path and generate the new path.
	     */
	    const gchar *cstrptr = getenv("HOME");
	    const gchar *cstrptr2 = input_path + strlen("~");

	    /* If home directory environment variable value was not set
	     * then assume toplevel.
	     */
	    if(cstrptr == NULL)
		cstrptr = "/";

	    /* Seek input path after the special prefix and past any
	     * deliminator characters (deliminator character will be added
	     * when the new path is generated).
	     */
	    while(*cstrptr2 == DIR_DELIMINATOR)
		cstrptr2++;

	    /* Generate new path. */
	    new_path = g_strdup_printf(
		"%s%c%s",
		cstrptr, DIR_DELIMINATOR, cstrptr2
	    );
	}
	else
	{
	    /* If the given input path is an absolute path then do not
	     * prefix the given parent to it.
	     */
	    if(ISPATHABSOLUTE(input_path))
	    {
		new_path = g_strdup(input_path);
	    }
	    else
	    {
		new_path = g_strdup_printf(
		    "%s%c%s",
		    (parent != NULL) ? parent : "/",
		    DIR_DELIMINATOR,
		    input_path
		);
	    }
	}

	/* Begin simplifying the generated path. */

        /* Simplify path, reducing occurances of "../". */
        SimplifyPath(new_path);

        /* Remove tailing deliminators. */
        StripPath(new_path);

	return(new_path);
}

/*
 *	Generates a unique file name in the directory specified by dir
 *	that is gauranteed to not exist.
 *
 *	The calling function must deallocate the returned pointer.
 */
gchar *EDVTmpName(const gchar *dir)
{
#if 0
	guint i;
	gchar *path;
	struct stat lstat_buf;
#endif


	if(dir == NULL)
	    dir = "/tmp";

#if 0
	/* Try and find a non-existant path by using a counter as the
	 * object name postfixed to the given directory.
	 */
	for(i = 0; i < (guint)-1; i++)
	{
	    path = g_strdup_printf("%s/%i", dir, i);

	    /* Does not exist? */
	    if(lstat(path, &lstat_buf))
		return(path);

	    /* Path exists, deallocate it and try again. */
	    g_free(path);
	}

	/* No possible path found, give up. */
	return(NULL);
#endif
	return(tempnam(dir, NULL));
}


/*
 *      Parses and returns the major and minor numbers for the device
 *      given by rdev which should be obtained from a prior call to
 *      *stat() in structure struct stat stat_buf.st_rdev.
 */
void EDVGetDeviceNumbers(dev_t rdev, gint *major, gint *minor)
{
        if(major != NULL)
            *major = ((rdev >> 8) & 0xff);
        if(minor != NULL)
            *minor = ((rdev >> 0) & 0xff);
}

/*
 *      Returns a formatted rdev value formatted from the given device
 *      major and minor numbers.
 */
dev_t EDVFormatDeviceNumbers(gint major, gint minor)
{
        return(((dev_t)major << 8) | ((dev_t)minor << 0));
}

