#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <gtk/gtk.h>

#include "guiutils.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvstatusbar.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvcfglist.h"
#include "config.h"


#include "images/icon_secure_16x16.xpm"
#include "images/icon_insecure_16x16.xpm"


static gint EDVStatusBarButtonEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static gint EDVStatusBarCrossingEventCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

edv_status_bar_struct *EDVStatusBarNew(
        gpointer core_ptr, GtkWidget *parent
);
void EDVStatusBarUpdateMenus(edv_status_bar_struct *sb);
void EDVStatusBarMap(edv_status_bar_struct *sb);
void EDVStatusBarUnmap(edv_status_bar_struct *sb);
void EDVStatusBarDelete(edv_status_bar_struct *sb);

void EDVStatusBarMessage(
        edv_status_bar_struct *sb, const gchar *message,
        gbool allow_gtk_iteration
);
void EDVStatusBarProgress(
        edv_status_bar_struct *sb, gfloat percent,
        gbool allow_gtk_iteration
);



#define EDV_STATUS_BAR_HEIGHT	26


/*
 *	Status bar "button_press_event" signal callback.
 *
 *	For checking of clicks on the event boxes.
 */
static gint EDVStatusBarButtonEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	edv_core_struct *core_ptr;
        edv_status_bar_struct *sb = (edv_status_bar_struct *)data;
	if((widget == NULL) || (sb == NULL))
	    return(FALSE);

	core_ptr = (edv_core_struct *)sb->core_ptr;
	if(core_ptr == NULL)
	    return(TRUE);

	/* Check which widget this event is for. */
	if(widget == sb->write_protect_enabled_eb)
	{
	    EDVCFGItemListSetValueI(
		core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
		FALSE, FALSE
	    );
	    EDVWriteProtectChangedEmit(core_ptr, FALSE);
	}
	else if(widget == sb->write_protect_disabled_eb)
	{
            EDVCFGItemListSetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
                TRUE, FALSE
            );
            EDVWriteProtectChangedEmit(core_ptr, TRUE);
	}

	return(TRUE);
}

/*
 *      Status bar "enter_notify_event" signal callback.
 *
 *      For checking of pointer crossing on the event boxes.
 */
static gint EDVStatusBarCrossingEventCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint etype;
        edv_status_bar_struct *sb = (edv_status_bar_struct *)data;
        if((widget == NULL) || (sb == NULL))
            return(FALSE);

	etype = crossing->type;

        /* Check which widget this event is for. */
        if(widget == sb->write_protect_enabled_eb)
        {
	    const gchar *mesg = NULL;

	    switch(etype)
	    {
	      case GDK_ENTER_NOTIFY:
#ifdef PROG_LANGUAGE_ENGLISH
		mesg = "Write protect is enabled, click to disable it";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                mesg = "Escriba proteja es habilitado, el chasquido a incapacitarlo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                mesg = "Protger en Ecriture est rendu capable, le dclic pour le rendre infirme";
#endif
		break;
/*
	      case GDK_LEAVE_NOTIFY:
		mesg = "";
		break;
 */
	    }
	    EDVStatusBarMessage(sb, mesg, FALSE);
        }
        else if(widget == sb->write_protect_disabled_eb)
        {
            const gchar *mesg = NULL;

            switch(etype)
            {
              case GDK_ENTER_NOTIFY:
#ifdef PROG_LANGUAGE_ENGLISH
                mesg = "Write protect is disabled, click to enable it";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                mesg = "Escriba proteja es incapacitado, el chasquido para habilitarlo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                mesg = "Protger en Ecriture est rendu infirme, le dclic pour le rendre capable";
#endif
                break;
/*
              case GDK_LEAVE_NOTIFY:
                mesg = "";
                break;
 */
            }
	    EDVStatusBarMessage(sb, mesg, FALSE);
        }

	return(TRUE);
}


/*
 *	Creates a new status bar.
 *
 *	Given parent is assumed to be a GtkBox.
 */
edv_status_bar_struct *EDVStatusBarNew(
        gpointer core_ptr, GtkWidget *parent
)
{
        GtkWidget *w, *parent2, *parent3;
        GtkAdjustment *adj;
	edv_status_bar_struct *sb = (edv_status_bar_struct *)g_malloc0(
	    sizeof(edv_status_bar_struct)
	);
	if(sb == NULL)
	    return(NULL);

	/* Reset values. */
	sb->initialized = TRUE;
	sb->map_state = FALSE;
	sb->core_ptr = core_ptr;

        /* Toplevel frame. */
        sb->toplevel = w = gtk_frame_new(NULL);
        gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
/*        gtk_widget_show(w); */
        parent2 = w;

        /* Table in main frame. */
        w = gtk_table_new(1, 3, FALSE);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
        parent2 = w;


	/* Write protect enabled/disabled. */
	/* Write protect enabled. */
	sb->write_protect_enabled_eb = w = gtk_event_box_new();
        gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(EDVStatusBarButtonEventCB),
            (gpointer)sb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB),
            (gpointer)sb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB),
            (gpointer)sb
        );
	gtk_table_attach(
            GTK_TABLE(parent2), w,
            0, 1, 0, 1,
            0, 0,
            2, 0
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
	    "Write protect is enabled"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Escriba proteja es habilitado"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Protger en Ecriture est rendu capable"
#endif
        );
	parent3 = w;
	/* Do not show write protect enabled. */

	w = GUICreateMenuItemIcon(
	    (u_int8_t **)icon_secure_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);


	/* Write protect disabled. */
        sb->write_protect_disabled_eb = w = gtk_event_box_new();
        gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
        gtk_signal_connect(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(EDVStatusBarButtonEventCB),
            (gpointer)sb
        );
	gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB),
            (gpointer)sb
	);
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVStatusBarCrossingEventCB),
            (gpointer)sb
        );
        gtk_table_attach(
            GTK_TABLE(parent2), w,
            0, 1, 0, 1,
            0, 0,
            2, 0
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Write protect is disabled"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Escriba proteja es incapacitado"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Protger en Ecriture est rendu infirme"
#endif
        );
        gtk_widget_show(w);
	parent3 = w;

	w = GUICreateMenuItemIcon(
            (u_int8_t **)icon_insecure_16x16_xpm
        );
	gtk_widget_set_usize(w, 16, 16);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_show(w);


        /* Progress bar. */
        adj = (GtkAdjustment *)gtk_adjustment_new(0, 1, 100, 0, 0, 0);
        sb->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
        gtk_widget_set_usize(w, 100, 20);
        gtk_progress_bar_set_orientation(
            GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
        );
        gtk_progress_bar_set_bar_style(
            GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
        );
        gtk_progress_set_activity_mode(
            GTK_PROGRESS(w), FALSE
        );
        gtk_table_attach(
            GTK_TABLE(parent2), w,
            1, 2, 0, 1,
            0,
            GTK_SHRINK,
            0, 0
        );
        gtk_widget_show(w);

        /* Label. */
        w = gtk_frame_new(NULL);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
        gtk_container_border_width(GTK_CONTAINER(w), 1);
        gtk_table_attach(
            GTK_TABLE(parent2), w,
            2, 3, 0, 1,
            GTK_SHRINK | GTK_EXPAND | GTK_FILL,
            GTK_FILL,
            0, 0
        );
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_hbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_show(w);
        parent3 = w;

        sb->label = w = gtk_label_new("");
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 2);
        gtk_widget_show(w);

	return(sb);
}

/*
 *	Updates menus and related resources on the status bar.
 */
void EDVStatusBarUpdateMenus(edv_status_bar_struct *sb)
{
	gbool write_protect_state;
	GtkWidget *w;
	edv_core_struct *core_ptr;


	if(sb == NULL)
	    return;

        if(!sb->initialized)
            return;

	core_ptr = (edv_core_struct *)sb->core_ptr;
	if(core_ptr == NULL)
	    return;


	/* Get current write protect state. */
	write_protect_state = (gbool)EDVCFGItemListGetValueI(
	    core_ptr->cfg_list, EDV_CFG_PARM_WRITE_PROTECT
	);

	/* Update write protect display. */
	w = sb->write_protect_enabled_eb;
	if(w != NULL)
	{
	    if(write_protect_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}
	w = sb->write_protect_disabled_eb;
        if(w != NULL)
        {
	    if(write_protect_state)
		gtk_widget_hide(w);
	    else
		gtk_widget_show(w);
	}
}

/*
 *	Maps the status bar.
 */
void EDVStatusBarMap(edv_status_bar_struct *sb)
{
        GtkWidget *w;

        if(sb == NULL)
            return;

        if(!sb->initialized)
            return;

        w = sb->toplevel;
        if(w == NULL)
            return;

        if(!sb->map_state)
        {
            gtk_widget_show(w);
            sb->map_state = TRUE;
        }
}

/*
 *	Unmaps the status bar.
 */
void EDVStatusBarUnmap(edv_status_bar_struct *sb)
{
	GtkWidget *w;

	if(sb == NULL)
	    return;

        if(!sb->initialized)
            return;

	w = sb->toplevel;
	if(w == NULL)
	    return;

	if(sb->map_state)
	{
	    gtk_widget_hide(w);
	    sb->map_state = FALSE;
	}
}

/*
 *	Destroys the given status bar.
 */
void EDVStatusBarDelete(edv_status_bar_struct *sb)
{
	GtkWidget **w;


	if(sb == NULL)
	    return;

	if(sb->initialized)
	{
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    /* Begin destroying widgets. */
	    w = &sb->progress_bar;
	    DO_DESTROY_WIDGET

	    w = &sb->label;
	    DO_DESTROY_WIDGET

	    w = &sb->toplevel;
	    DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET
	}

	/* Deallocate structure itself. */
	g_free(sb);
}


/*
 *	Updates the message on the status bar.
 */
void EDVStatusBarMessage(
        edv_status_bar_struct *sb, const gchar *message,
        gbool allow_gtk_iteration
)
{
        GtkWidget *w;


	if(sb == NULL)
	    return;

	if(!sb->initialized || !sb->map_state)
	    return;

        w = sb->label;
        if(w == NULL)
            return;

        gtk_label_set_text(
            GTK_LABEL(w),
            (message != NULL) ? message : ""
        );

        /* Iterate through GTK main function until all events are handled
         * so that we ensure this label text is updated.
         */
        while((gtk_events_pending() > 0) && allow_gtk_iteration)
            gtk_main_iteration();
}

/*
 *	Updates the progress on the status bar.
 *
 *	If value is between 0.0 to 1.0 then normal finite progress
 *	is performed, otherwise if value is negative then unknown time
 *	progress is performed.
 */
void EDVStatusBarProgress(
        edv_status_bar_struct *sb, gfloat value,
        gbool allow_gtk_iteration
)
{
        GtkWidget *w;
	GtkProgress *progress;


        if(sb == NULL)
            return;

        if(!sb->initialized || !sb->map_state)
            return;

        w = sb->progress_bar;
        if(w == NULL)
            return;
	progress = GTK_PROGRESS(w);


        /* Negative? Implies just do activity. */
        if(value < 0.0)
        {
            GtkAdjustment *adj = progress->adjustment;


	    /* Get new value based on unknown position. */
            value = gtk_progress_get_value(progress) + 1.0;
            if(value > adj->upper)
                value = adj->lower;

	    /* Update progress widget for `activity mode' (unknown limit)
	     * and set new value.
	     */
            gtk_progress_set_activity_mode(progress, TRUE);
            gtk_progress_set_show_text(progress, FALSE);
            gtk_progress_set_value(progress, value);

	    /* Reset last progress position to -1.0, indicating no last
	     * position since in unknown time limit we don't keep track of
	     * the last position.
	     */
	    sb->progress_pos_last = -1.0;

            /* Iterate through GTK main function until all events are
             * handled so that we ensure this progress bar setting is
             * updated.
             */
            while((gtk_events_pending() > 0) && allow_gtk_iteration)
                gtk_main_iteration();

            return;
        }

	/* Clip value to 0.0, 1.0. */
        if(value > 1.0)
            value = 1.0;

	/* Reset last progress position if it is greater than the given
	 * value, implying the progress wraped back to the beginning.
	 */
	if(sb->progress_pos_last > value)
	    sb->progress_pos_last = -1.0;

	/* Check if percent did not sufficiently change (use 0.01
	 * increments).
	 */
	if((sb->progress_pos_last > 0.0) &&
           ((value - sb->progress_pos_last) < 0.01)
	)
	    return;

	/* Update progress. */
        gtk_progress_set_activity_mode(progress, FALSE);
        gtk_progress_set_format_string(
	    progress, "%p%%"
        );
        gtk_progress_set_show_text(
            progress, (value > 0.0) ? TRUE : FALSE
        );
	gtk_progress_bar_update(GTK_PROGRESS_BAR(w), value);

        /* Record last position. */
        sb->progress_pos_last = value;

	/* Iterate through GTK main function until all events are
	 * handled so that we ensure this progress bar setting is
	 * updated.
	 */
	while((gtk_events_pending() > 0) && allow_gtk_iteration)
	    gtk_main_iteration();
}
