#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"

#include "edvtypes.h"
#include "edvmimetypes.h"
#include "endeavour.h"
#include "edvopen.h"
#include "edvop.h"
#include "edvutils.h"
#include "config.h"


gint EDVOpenObjectPath(
        edv_core_struct *core_ptr, const gchar *path,
	const gchar *command_name,
        GtkWidget *toplevel, gbool verbose,
        gchar **stdout_path_rtn, gchar **stderr_path_rtn
);

gint EDVOpenWithObjectPath(
        edv_core_struct *core_ptr, const gchar *path,
        const gchar *command_name,
        GtkWidget *toplevel, gbool verbose,
        gchar **stdout_path_rtn, gchar **stderr_path_rtn
);



/*
 *	Procedure to execute an application specified by a matched
 *	MIME Type of the given disk object specified by path.
 *
 *	If and only if stdout_path_rtn or stderr_path_rtn are not NULL,
 *	then they will be set with a dynamically allocated string 
 *	containing the stdout and stderr output file paths. Which the
 *	calling function must deallocate.
 *
 *	Returns:
 *
 *	0	Success
 *	-1	General error
 *	-2	No MIME Type, no application, or ambiguous
 *	-3	System error
 *	-6	Call is a re-enterent to an existing call to this function
 */
gint EDVOpenObjectPath(
        edv_core_struct *core_ptr, const gchar *path,
        const gchar *command_name,
        GtkWidget *toplevel, gbool verbose,
        gchar **stdout_path_rtn, gchar **stderr_path_rtn
)
{
	static gbool reenterent = FALSE;
	pid_t p;
	gbool got_match;
	const gchar *cstrptr;
	gint i, status;
	edv_mimetype_struct *mt_ptr;
	gchar *command;
	const gchar *name, *command_base;
        gchar prev_working_dir[PATH_MAX];


	/* Reset returns. */
	if(stdout_path_rtn != NULL)
	    *stdout_path_rtn = NULL;
	if(stderr_path_rtn != NULL)
	    *stderr_path_rtn = NULL;

	if(reenterent)
	    return(-6);
	else
	    reenterent = TRUE;

	if((core_ptr == NULL) || (path == NULL))
	{
	    reenterent = FALSE;
	    return(-1);
	}

	/* Get pointer to name portion of given path. */
	name = strrchr(path, DIR_DELIMINATOR);
	if(name == NULL)
	    name = path;
	else
	    name++;

	/* Iterate through MIME Types list and check if one matches the
	 * given path.
	 */
	mt_ptr = NULL;
	got_match = FALSE;
	for(i = 0; i < core_ptr->total_mimetypes; i++)
	{
	    mt_ptr = core_ptr->mimetype[i];
	    if(mt_ptr == NULL)
		continue;

	    /* Handle by MIME Type class. */
	    got_match = FALSE;
	    switch(mt_ptr->mt_class)
	    {
	      case EDV_MIMETYPE_CLASS_FORMAT:
		if(EDVIsExtension(path, mt_ptr->value))
		{
		    got_match = TRUE;
		}
		break;

	      case EDV_MIMETYPE_CLASS_UNIQUE:
		if(mt_ptr->value != NULL)
		{
		    if(!strcmp(path, mt_ptr->value))
			got_match = TRUE;
		}
		break;
	    }
	    /* Stop itterating for other MIME Types if a match was made. */
	    if(got_match)
		break;
	}
	/* Unable to match MIME Type? */
	if(!got_match || (mt_ptr == NULL))
	{
	    /* Since this object's name does not match any MIME Type, then
	     * check if the object itself is executable. In which case
	     * execute it.
	     */
	    if(!access(path, X_OK))
	    {
		/* Record previous working dir and set new working dir as
		 * the parent of the given path.
		 */
		*prev_working_dir = '\0';
		getcwd(prev_working_dir, PATH_MAX);
		cstrptr = GetParentDir(path);
		if(cstrptr != NULL)
		    chdir(cstrptr);

		/* Execute the given object. */
		p = Exec(path);
		if(p == 0)
		{
		    status = -1;
		}
		else
		{
		    status = 0;
		}

		/* Restore previous working dir. */
		chdir(prev_working_dir);


		/* Failed to execute the given object? */
                if(status && verbose)
                {
                    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute object:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar objeto:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'objet:\n\
\n\
    %s\n",
#endif
			path
		    );

		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Open Failed",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Abra Fallado",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ouvrir Echou",
#endif
buf,
#ifdef PROG_LANGUAGE_ENGLISH
"If you wish to open this executable object with\n\
a specific MIME Type, try File->Open With.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Si usted desea abrir este objeto de executable\n\
con un MIME Type especfico, la prueba El Archivo->Abre Con.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Si vous souhaitez ouvrir cet objet de executable\n\
avec un MIME Type spcifique, essayez Le Fichier->Ouvrir Avec.",
#endif
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);

		    g_free(buf);
		}

                reenterent = FALSE;
                return(status);
	    }

	    /* No MIME Type associated for the given path. */
	    if(verbose)
	    {
		gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to find MIME Type associated with object:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de encontrar que MIME Type se asoci con objeto:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour trouver MIME Type a associ  l'objet:\n\
\n\
    %s\n",
#endif
		    path
		);

		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Open Failed",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Abra Fallado",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ouvrir Echou",
#endif
buf,
#ifdef PROG_LANGUAGE_ENGLISH
"To define a MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Para definir un MIME Type para este objeto, va a\n\
El Panorama->MIME Types...\n\
\n\
Para la ayuda adicional a establecer MIME Types, ve\n\
La Ayuda->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Pour dfinir un MIME Type pour cet objet, aller \n\
La Vue->MIME Types...\n\
\n\
Pour l'aide supplmentaire sur montant MIME Types, voir\n\
L'Aide->MIME Types...",
#endif
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);

		g_free(buf);
	    }
	    reenterent = FALSE;
	    return(-2);
	}

	/* At this point a MIME Type has been matched, find out how we
	 * should open it by checking its handler code.
	 *
	 * Any handler code specifying an internal Endeavour resource
	 * will return within this switch() statement.
	 */
	switch(mt_ptr->handler)
	{
	  case EDV_MIMETYPE_HANDLER_COMMAND:
	    /* Open this object with a command, so go on with the
	     * rest of this function which deals with opening by the
	     * MIME Type's command.
	     */
	    break;

          case EDV_MIMETYPE_HANDLER_EDV_ARCHIVER:
	    EDVDoNewArchiverOpen(core_ptr, path);
	    reenterent = FALSE;
	    return(0);
            break;

          case EDV_MIMETYPE_HANDLER_EDV_IMAGE_BROWSER:
/* Need to work on this. */
            reenterent = FALSE;
            return(0);
            break;

          case EDV_MIMETYPE_HANDLER_EDV_RECYCLE_BIN:
/* NEed to work on this. */
            reenterent = FALSE;
            return(0);
            break;

	  default:
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Handler Not Supported",
"The handler defined for the MIME Type associated\n\
with this object is not supported.",
                NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El Tratante No Sostuvo",
"El tratante definido para el MIME Type se asoci\n\
con este objeto no es sostenido.",
		NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'Agent N'A Pas Soutenu",
"L'agent dfini pour le MIME Type a associ  cet\n\
objet n'est passoutenu.",
                NULL,
#endif
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	    reenterent = FALSE;
            return(-2);
	    break;
	}

	/* If this point has been reached, it means that an external
	 * command should be executed to open the object.
	 */

	/* Get the base command command_base that matches the given
	 * command name or the first defined command on the MIME Type if
	 * no command name is given.
	 */
	command_base = NULL;
	if(command_name != NULL)
	{
	    /* Command name is given, iterate through commands and get
	     * command as the base command who matches the given command
	     * name.
	     */
	    for(i = 0; i < mt_ptr->total_commands; i++)
	    {
		if(mt_ptr->command_name[i] != NULL)
		{
		    if(!strcasecmp(mt_ptr->command_name[i], command_name))
		    {
			command_base = mt_ptr->command[i];
			break;
		    }
		}
	    }
	}
	else
	{
	    /* No specific command given, use default (first) command on
	     * the MIME Type (if any).
	     */
	    if(mt_ptr->total_commands > 0)
		command_base = mt_ptr->command[0];
	}

	/* Unable to find base command from the matched MIME Type? */
	if((command_base != NULL) ? (*command_base == '\0') : TRUE)
	{
            /* MIME Type has no such command. */
            if(verbose)
            {
                gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"MIME Type `%s' does not have any defined command(s).",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"MIME Type `%s' no tenga ninguna orden(s) definida.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"MIME Type `%s' ne pas avoir l'ordre dfini(s).",
#endif
		    mt_ptr->type
		);

                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Open Failed",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Abra Fallado",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ouvrir Echou",
#endif
buf,
#ifdef PROG_LANGUAGE_ENGLISH
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Para redactar el MIME Type para este objeto, va a\n\
El Panorama->MIME Types...\n\
\n\
Para la ayuda adicional a establecer MIME Types, ve\n\
La Ayuda->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Pour diter le MIME Type pour cet objet, aller \n\
La Vue->MIME Types...\n\
\n\
Pour l'aide supplmentaire sur montant MIME Types, voir\n\
L'Aide->MIME Types...",
#endif
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);

		g_free(buf);
            }
            reenterent = FALSE;
            return(-2);
	}


	/* If the matched base command is not an absolute path then it
	 * implies that it reffers to another MIME Type, so try to find
	 * another MIME Type who's class is EDV_MIMETYPE_CLASS_PROGRAM and
	 * type matches the base command.
	 */
	if(*command_base != DIR_DELIMINATOR)
	{
	    /* Find another MIME Type who's type matches the matched base
	     * command.
	     */
	    mt_ptr = NULL;
	    got_match = FALSE;
	    for(i = 0; i < core_ptr->total_mimetypes; i++)
	    {
		mt_ptr = core_ptr->mimetype[i];
		if(mt_ptr == NULL)
		    continue;

		/* Check only MIME Types who's class is
		 * EDV_MIMETYPE_CLASS_PROGRAM.
		 */
		if(mt_ptr->mt_class == EDV_MIMETYPE_CLASS_PROGRAM)
		{
		    if((mt_ptr->type != NULL) ?
			!strcmp(mt_ptr->type, command_base) : FALSE
		    )
		    {
			/* Found a referenced MIME Type, now get its first
			 * command. Note that only the first command will
			 * be obtained from a MIME Type of class
			 * EDV_MIMETYPE_CLASS_PROGRAM.
			 */
			if(mt_ptr->total_commands > 0)
			    command_base = mt_ptr->command[0];
			else
			    command_base = NULL;
			got_match = TRUE;
		    }
		}
		if(got_match)
		    break;
	    }
	    /* Unable to match MIME Type? */
	    if(!got_match || (mt_ptr == NULL))
	    {
		/* No MIME Type associated for the given path. */
		if(verbose)
		{
		    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to find Application class MIME Type `%s'.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de encontrar la clase de la Aplicacin MIME Type `%s'.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour trouver la classe d'Application MIME Type `%s'.",
#endif
			command_base
		    );

		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Open Failed",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Abra Fallado",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ouvrir Echou",
#endif
buf,
#ifdef PROG_LANGUAGE_ENGLISH
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Para redactar el MIME Type para este objeto, va a\n\
El Panorama->MIME Types...\n\
\n\
Para la ayuda adicional a establecer MIME Types, ve\n\
La Ayuda->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Pour diter le MIME Type pour cet objet, aller \n\
La Vue->MIME Types...\n\
\n\
Pour l'aide supplmentaire sur montant MIME Types, voir\n\
L'Aide->MIME Types...",
#endif
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);

		    g_free(buf);
		}
		reenterent = FALSE;
		return(-2);
	    }
	}


	/* mt_ptr at this point should be considered invalid. */

	/* The command_base may be modified at this point, check if
	 * the MIME Type has no defined command.
	 */
	if(command_base == NULL)
	{
            if(verbose)
            {
                gchar *buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"MIME Type does not have a defined command."
#endif
#ifdef PROG_LANGUAGE_SPANISH
"MIME Type no tenga una orden definida."
#endif
#ifdef PROG_LANGUAGE_FRENCH
"MIME Type ne pas avoir un ordre dfin."
#endif
		);

                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Open Failed",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Abra Fallado",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ouvrir Echou",
#endif
buf,
#ifdef PROG_LANGUAGE_ENGLISH
"To edit the MIME Type for this object, go to\n\
View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Para redactar el MIME Type para este objeto, va a\n\
El Panorama->MIME Types...\n\
\n\
Para la ayuda adicional a establecer MIME Types, ve\n\
La Ayuda->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Pour diter le MIME Type pour cet objet, aller \n\
La Vue->MIME Types...\n\
\n\
Pour l'aide supplmentaire sur montant MIME Types, voir\n\
L'Aide->MIME Types...",
#endif
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);

                g_free(buf);
            }

            reenterent = FALSE;
            return(-2);
	}


	/* We now have sufficient information to run the program. */

	/* Generate command from matched command_base and the given
	 * object path. If the command base has a "%s" string within
	 * then replace it with the path, otherwise append path to the
	 * command string.
	 */
	if(strstr(command_base, "\"%s\"") != NULL)
        {
            command = g_strdup_printf(
                command_base, path
            );
        }
	else if(strstr(command_base, "%s") != NULL)
        {
            /* Modify the given path and add quotes around it, since it
             * should not have it already and command_base (obtained
             * from the MIME Type is not expected to have it either.
             */
            gchar *tmp_path = g_strdup_printf("\"%s\"", path);

            /* Now format the command. */
            command = g_strdup_printf(
                command_base, tmp_path
            );
            g_free(tmp_path);
        }
	else
	{
	    command = g_strdup_printf(
		"%s \"%s\"",
		command_base, path
	    );
	}


	/* Record previous working dir and set new working dir as the
	 * parent of the given path.
	 */
	*prev_working_dir = '\0';
	getcwd(prev_working_dir, PATH_MAX);
	cstrptr = GetParentDir(path);
	if(cstrptr != NULL)
	    chdir(cstrptr);

        /* Execute the generated command. */
        p = Exec(command);
        if(p == 0)
        {
            status = -1;
        }
        else
        {
            status = 0;
        }

        /* Restore previous working dir. */
        chdir(prev_working_dir);

	/* Deallocate command string, it is no longer needed. */
	g_free(command);
	command = NULL;

	reenterent = FALSE;
	return(status);
}

/*
 *      Procedure to execute an application selected on the `open with'
 *	popup list widget mapped at the current pointer position.
 *
 *      If and only if stdout_path_rtn or stderr_path_rtn are not NULL,
 *      then they will be set with a dynamically allocated string
 *      containing the stdout and stderr output file paths. Which the
 *      calling function must deallocate.
 *
 *      Returns:
 *
 *      0       Success
 *      -1      General error
 *      -2      No MIME Type, no application, or ambiguous
 *      -3      System error
 *	-4	User responded with `Cancel'
 *      -6      Call is a renterent to an existing call to this function
 */
gint EDVOpenWithObjectPath(
        edv_core_struct *core_ptr, const gchar *path,
        const gchar *command_name,
        GtkWidget *toplevel, gbool verbose,
        gchar **stdout_path_rtn, gchar **stderr_path_rtn
)
{
        static gbool reenterent = FALSE;
	pulist_struct *pulist;
	const gchar *cstrptr;
        pid_t p;
        gint mt_num, status;
        edv_mimetype_struct *mt_ptr;
        gchar *command;
        const gchar *value, *command_base;
	gchar prev_working_dir[PATH_MAX];


        /* Reset returns. */
        if(stdout_path_rtn != NULL)
            *stdout_path_rtn = NULL;
        if(stderr_path_rtn != NULL)
            *stderr_path_rtn = NULL;

        if(reenterent)
            return(-6);
        else
            reenterent = TRUE;

        if((core_ptr == NULL) || (path == NULL))
        {
            reenterent = FALSE;
            return(-1);
        }

	/* Get pointer to `open with' popup list. */
	pulist = core_ptr->openwith_pulist;
	if(pulist == NULL)
	{
            reenterent = FALSE;
            return(-1);
	}

	/* Map `open with' popup list. */
	value = PUListMapQuery(
	    pulist,
	    NULL,		/* No initial value, use last value. */
	    -1,			/* Default number of visible lines. */
	    PULIST_RELATIVE_DOWN,
	    NULL,		/* No relative widget, meaning use pointer coordinates. */
	    NULL		/* No map widget. */
	);
	/* User canceled? */
	if(value == NULL)
	{
            reenterent = FALSE;
            return(-4);
        }

	/* Get MIME Type index number from the matched value. */
	mt_num = (gint)PUListGetDataFromValue(pulist, value);
	if((mt_num >= 0) && (mt_num < core_ptr->total_mimetypes))
	    mt_ptr = core_ptr->mimetype[mt_num];
	else
	    mt_ptr = NULL;

        /* No such MIME Type? */
        if(mt_ptr == NULL)
        {
            reenterent = FALSE;
            return(-2);
        }

	/* MIME Type must be of class EDV_MIMETYPE_CLASS_PROGRAM. */
	if(mt_ptr->mt_class != EDV_MIMETYPE_CLASS_PROGRAM)
	{
            reenterent = FALSE;
            return(-2);
        }

	/* Get first command on the MIME Type as the base command. */
	if(mt_ptr->total_commands > 0)
	    command_base = mt_ptr->command[0];
	else
	    command_base = NULL;

        /* MIME Type has no defined command? */
        if(command_base == NULL)
        {
            if(verbose)
            {
                gchar *buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"MIME Type does not have a defined command."
#endif
#ifdef PROG_LANGUAGE_SPANISH
"MIME Type no tenga una orden definida."
#endif
#ifdef PROG_LANGUAGE_FRENCH
"MIME Type ne pas avoir un ordre dfin."
#endif
                );

                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Open Failed",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Abra Fallado",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ouvrir Echou",
#endif
buf,
#ifdef PROG_LANGUAGE_ENGLISH
"To edit the MIME Type, go to View->MIME Types...\n\
\n\
For additional help on setting up MIME Types, see\n\
Help->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Para redactar el MIME Type, va a El Panorama->MIME Types...\n\
\n\
Para la ayuda adicional a establecer MIME Types, ve\n\
La Ayuda->MIME Types...",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Pour diter le MIME Type, aller  La Vue->MIME Types...\n\
\n\
Pour l'aide supplmentaire sur montant MIME Types, voir\n\
L'Aide->MIME Types...",
#endif
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);

                g_free(buf);
            }

            reenterent = FALSE;
            return(-2);
        }


        /* We now have sufficient information to run the program. */

        /* Generate command based on matched command_base and the given
         * object path.
         */
        cstrptr = strstr(command_base, "%s");
        if(cstrptr != NULL)
        {
            command = g_strdup_printf(
                command_base, path
            );
        }
        else
        {
            command = g_strdup_printf(
                "%s \"%s\"",
                command_base, path
            );
        }


        /* Record previous working dir and set new working dir as the
	 * parent of the given path.
	 */
        *prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
	cstrptr = GetParentDir(path);
	if(cstrptr != NULL)
	    chdir(cstrptr);

        /* Execute the selected command. */
        p = Exec(command);
        if(p == 0)
        {
            status = -1;
        }
        else
        {
            status = 0;
        }

	/* Restore previous working dir. */
	chdir(prev_working_dir);

        /* Deallocate command string, it is no longer needed. */
        g_free(command);
        command = NULL;

        reenterent = FALSE;
        return(status);
}
