#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvcfgfio.h"
#include "edvhistory.h"
#include "edvobj.h"
#include "edvdevices.h"
#include "edvdevicesfio.h"
#include "edvmimetypes.h"
#include "edvmimetypesfio.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "browser.h"
#include "browserdirtree.h"
#include "imbr.h"
#include "archiver.h"
#include "propwin.h"
#include "recbin.h"
#include "recbincontents.h"
#include "findwin.h"
#include "historywin.h"
#include "optwin.h"
#include "optwinop.h"
#include "edvoptions.h"
#include "edvcustomize.h"
#include "rundlg.h"
#include "deviceswin.h"
#include "mimetypeswin.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvcfglist.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "config.h"

#include "images/icon_trash_32x32.xpm"

gint EDVDoWindowNexus(
        edv_core_struct *core_ptr,
        const gchar *name,              /* Name of window. */
        const gchar *path,              /* Initial object path. */
	const gchar *extra,
        GtkWidget *ref_toplevel
);
gint EDVDoNewBrowser(edv_core_struct *core_ptr);
gint EDVDoNewImbr(edv_core_struct *core_ptr);
gint EDVDoNewArchiverOpen(edv_core_struct *core_ptr, const gchar *path);
gint EDVDoNewArchiver(edv_core_struct *core_ptr);
gint EDVDoPropWinAndSwitchPage(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	GtkWidget *ref_toplevel, const gchar *tab_name
);
gint EDVDoPropWin(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	GtkWidget *ref_toplevel
);

void EDVDoRunDeviceCheck(
        edv_core_struct *core_ptr, edv_device_struct *dev_ptr,
        GtkWidget *ref_toplevel
);
void EDVDoRunDeviceTools(
        edv_core_struct *core_ptr, edv_device_struct *dev_ptr,
        GtkWidget *ref_toplevel
);
void EDVDoRunDeviceFormat(
        edv_core_struct *core_ptr, edv_device_struct *dev_ptr,
        GtkWidget *ref_toplevel
);

void EDVDoRunTerminal(edv_core_struct *core_ptr, const gchar *cmd);

void EDVDoMapRecBinDeskIcon(edv_core_struct *core_ptr);
void EDVDoMapRecBin(edv_core_struct *core_ptr);

void EDVDoMapHistoryListWin(
	edv_core_struct *core_ptr, GtkWidget *rel_toplevel
);

void EDVDoMapOptionsWin(
        edv_core_struct *core_ptr, GtkWidget *rel_toplevel
);
void EDVDoMapCustomizeWin(
        edv_core_struct *core_ptr, GtkWidget *rel_toplevel
);
void EDVDoMapDevicesListWin(
	edv_core_struct *core_ptr, GtkWidget *rel_toplevel
);
void EDVDoMapMIMETypesListWin(
	edv_core_struct *core_ptr, GtkWidget *rel_toplevel
);
void EDVDoMapRunDlg(edv_core_struct *core_ptr);

void EDVDoMapBrowserFindWin(
        edv_core_struct *core_ptr, edv_browser_struct *browser
);
void EDVDoMapImbrFindWin(
        edv_core_struct *core_ptr, edv_imbr_struct *imbr
);
void EDVDoMapArchiverFindWin(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver
);
void EDVDoMapRecBinFindWin(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin
);

void EDVDoUpdateDevicesPUList(edv_core_struct *core_ptr);
void EDVDoUpdateMIMETypeHintIndexes(edv_core_struct *core_ptr);
void EDVDoUpdateOpenWithPUList(edv_core_struct *core_ptr);

void EDVDoSyncDisks(edv_core_struct *core_ptr);

void EDVDoInternetDownloadObject(
	edv_core_struct *core_ptr,
	edv_dnd_object_struct *dnd_obj,		/* Source. */
	const gchar *target_path,		/* Target. */
	GtkWidget *toplevel
);

void EDVDoHelp(
	edv_core_struct *core_ptr, const gchar *topic, GtkWidget *toplevel
);
void EDVDoHelpAbout(edv_core_struct *core_ptr, GtkWidget *toplevel);

void EDVDoRecycleBinSizeCheck(
	edv_core_struct *core_ptr, GtkWidget *toplevel
);


/* Help program argument types, specifies the type of argument it takes
 * (usually a file path or url).
 */
#define HELP_PROG_INPUT_TYPE_PATH       0
#define HELP_PROG_INPUT_TYPE_URL        1

/* Location of help programs, their respective arguments, and argument
 * type in sets of three pointers. The last set of three pointers in the
 * list should be all NULL's.
 */
#define EDV_HELP_PROG_LOCATIONS	{\
{ "/usr/bin/gnome-help-browser", "", HELP_PROG_INPUT_TYPE_URL }, \
{ "/usr/local/bin/gnome-help-browser", "", HELP_PROG_INPUT_TYPE_URL }, \
{ "/bin/gnome-help-browser", "", HELP_PROG_INPUT_TYPE_URL }, \
\
{ "/usr/bin/netscape", "", HELP_PROG_INPUT_TYPE_URL }, \
{ "/usr/local/bin/netscape", "", HELP_PROG_INPUT_TYPE_URL }, \
{ "/bin/netscape", "", HELP_PROG_INPUT_TYPE_URL }, \
\
{ "/usr/X11R6/bin/nxterm", "-e lynx", HELP_PROG_INPUT_TYPE_URL }, \
{ "/usr/X11R6/bin/xterm", "-e lynx", HELP_PROG_INPUT_TYPE_URL }, \
\
{ NULL, NULL, 0 } \
}

/* Structure for help program entries in the EDV_HELP_PROG_LOCATIONS list. */
typedef struct {
	gchar *prog;
	gchar *arg;
	gint arg_type;
} help_prog_struct;


/* Help topics and the location of their help files. The last set of
 * three pointers in the list should be all NULL's.
 */
#define EDV_HELP_FILE_LOCATIONS {\
{ "Contents",		"help/index.html", 0 }, \
{ "File Browser",	"help/file_browser.html", 0 }, \
{ "Image Browser",	"help/image_browser.html", 0 }, \
{ "Archiver",		"help/archiver.html", 0 }, \
{ "Recycle Bin",	"help/recycle_bin.html", 0 }, \
{ "Keys List",		"help/keys_list.html", 0 }, \
{ "Disk Objects",	"help/disk_objects.html", 0 }, \
{ "MIME Types",		"help/mime_types.html", 0 }, \
{ "Devices",		"help/devices.html", 0 }, \
{ "Common Operations",	"help/common_operations.html", 0 }, \
{ "Front Ends",		"help/front_ends.html", 0 }, \
{ "FAQ",		"help/faq.html", 0 }, \
{ "Contacts",		"help/contacts.html", 0 }, \
{ NULL, NULL, 0 } \
}

/* Structure for help file entries in the EDV_HELP_FILE_LOCATIONS list. */
typedef struct {
        gchar *topic;
        gchar *file;
        gint reserved;
} help_file_struct;



/*
 *	Nexus for creating a new endeavour window.
 */
gint EDVDoWindowNexus(
        edv_core_struct *core_ptr,
	const gchar *name,		/* Name of window. */
	const gchar *path,		/* Initial object path. */
	const gchar *extra,
        GtkWidget *ref_toplevel
)
{
	if((core_ptr == NULL) || (name == NULL))
	    return(-1);


	/* Begin checking which window we should create. */

	/* File browser? */
	if(!strcasecmp(name, "file_browser") ||
           !strcasecmp(name, "file-browser") ||
	   !strcasecmp(name, "filebrowser") ||
           !strcasecmp(name, "browser")
	)
	{
	    EDVDoNewBrowser(core_ptr);
	}
	/* Image browser? */
	else if(!strcasecmp(name, "image_browser") ||
                !strcasecmp(name, "image-browser") ||
	        !strcasecmp(name, "imagebrowser") ||
	        !strcasecmp(name, "imbr")
	)
	{
	    EDVDoNewImbr(core_ptr);
	}
	/* Archiver? */
	else if(!strcasecmp(name, "archiver"))
	{
	    EDVDoNewArchiverOpen(core_ptr, path);
	}
	/* Recycle bin? */
	else if(!strcasecmp(name, "recycle_bin") ||
                !strcasecmp(name, "recycle-bin") ||
	        !strcasecmp(name, "recyclebin") ||
                !strcasecmp(name, "rec_bin") ||
                !strcasecmp(name, "rec-bin") ||
	        !strcasecmp(name, "recbin")
	)
	{
	    EDVDoMapRecBin(core_ptr);
	}
	/* Properties window? */
	else if(!strcasecmp(name, "properties") ||
	        !strcasecmp(name, "propwin")
	)
	{
	    struct stat lstat_buf;
	    edv_object_struct *obj = EDVObjectNew();
	    if(path != NULL)
	    {
		EDVObjectSetPath(obj, path);
		if(!lstat(path, &lstat_buf))
		    EDVObjectSetStat(obj, &lstat_buf);
	    }
	    EDVDoPropWinAndSwitchPage(core_ptr, obj, NULL, extra);
	    EDVObjectDelete(obj);
	}
	/* Options window? */
	else if(!strcasecmp(name, "options"))
	{
	    EDVDoMapOptionsWin(core_ptr, NULL);
	}
	/* Customize window? */
	else if(!strcasecmp(name, "customize"))
	{
	    EDVDoMapCustomizeWin(core_ptr, NULL);
	}
	/* Devices list? */
	else if(!strcasecmp(name, "devices"))
	{
	    EDVDoMapDevicesListWin(core_ptr, NULL);
	}
	/* MIMETypes list? */
	else if(!strcasecmp(name, "mimetypes"))
	{
	    EDVDoMapMIMETypesListWin(core_ptr, NULL);
	}
	/* Run dialog? */
	else if(!strcasecmp(name, "run_dialog") ||
	        !strcasecmp(name, "run-dialog") ||
                !strcasecmp(name, "run_dlg") ||
                !strcasecmp(name, "run-dlg") ||
                !strcasecmp(name, "rundlg")
	)
	{
	    EDVDoMapRunDlg(core_ptr);
	}
        /* About dialog? */
        else if(!strcasecmp(name, "about_dialog") ||
                !strcasecmp(name, "about-dialog") ||
                !strcasecmp(name, "about_dlg") ||
                !strcasecmp(name, "about-dlg") ||
                !strcasecmp(name, "aboutdlg")
        )
        {
            EDVDoHelpAbout(core_ptr, NULL);
        }
	/* Help. */
	else if(!strcasecmp(name, "help"))
	{
	    /* Note that the path becomes the help topic, see
	     * EDVDoHelp() for which topics are supported.
	     */
	    EDVDoHelp(core_ptr, path, NULL);
	}


	return(0);
}

/*
 *	Procedure to create a new browser and map it. The new browser
 *	will be appended to the core_ptr or an existing (unmapped)
 *	browser will be used.
 *
 *	Returns the index of the browser on the core structure or -1
 *	on error.
 */
gint EDVDoNewBrowser(edv_core_struct *core_ptr)
{
	gint i;
	edv_browser_struct *browser;


	if(core_ptr == NULL)
	    return(-1);

	/* Sanitize total. */
	if(core_ptr->total_browsers < 0)
	    core_ptr->total_browsers = 0;

	/* Get highest index and increase total. */
	i = core_ptr->total_browsers;
	core_ptr->total_browsers = i + 1;

	/* Allocate more pointers. */
	core_ptr->browser = (edv_browser_struct **)g_realloc(
	    core_ptr->browser,
	    core_ptr->total_browsers * sizeof(edv_browser_struct *)
	);
	if(core_ptr->browser == NULL)
	{
	    core_ptr->total_browsers = 0;
	    return(-1);
	}
	else
	{
	    core_ptr->browser[i] = NULL;
	}

	/* Create new browser. */
	core_ptr->browser[i] = browser = EDVBrowserNew(core_ptr);
	if(browser != NULL)
	{
	    gchar *startup_dir;
	    GtkCList *clist;


            EDVBrowserSetBusy(browser, TRUE);
	    GUIBlockInput(browser->toplevel, TRUE);

	    /* Need to map it immediatly after creation, before any
	     * timeouts check on it.
	     */
	    EDVBrowserMap(browser);

	    /* Get copy of start up directory. */
	    startup_dir = EDVCopyEvaluateInputPath(
		NULL,
		EDVCFGItemListGetValueS(
		    core_ptr->cfg_list, EDV_CFG_PARM_DIR_START_UP
		)
	    );

	    /* Get directory ctree and create initial toplevel nodes. */
	    clist = (GtkCList *)browser->directory_ctree;
	    if(clist != NULL)
	    {
		gtk_clist_freeze(clist);

		/* Create toplevel directories. */
		EDVBrowserDirTreeDoCreateToplevels(browser);

		gtk_clist_thaw(clist);


		/* Select startup directory, do not freeze. */
		EDVBrowserDirTreeDoSelectPath(browser, startup_dir);
	    }

            GUIBlockInput(browser->toplevel, FALSE);
	    EDVBrowserSetBusy(browser, FALSE);

	    /* Deallocate copy of start up directory. */
	    g_free(startup_dir);
	    startup_dir = NULL;

	    return(i);
	}
	else
	{
	    return(-1);
	}
}

/*
 *      Procedure to create a new image browser and map it. The new
 *	image browser will be appended to the core_ptr or an existing
 *	(unmapped)image browser will be used.
 *
 *      Returns the index of the browser on the core structure or -1
 *      on error.
 */
gint EDVDoNewImbr(edv_core_struct *core_ptr)
{
        gint i;
        edv_imbr_struct *imbr;


        if(core_ptr == NULL)
            return(-1);

        /* Sanitize total. */
        if(core_ptr->total_imbrs < 0)
            core_ptr->total_imbrs = 0;

        /* Get highest index and increase total. */
        i = core_ptr->total_imbrs;
        core_ptr->total_imbrs = i + 1;

        /* Allocate more pointers. */
        core_ptr->imbr = (edv_imbr_struct **)g_realloc(
            core_ptr->imbr,
            core_ptr->total_imbrs * sizeof(edv_imbr_struct *)
        );
        if(core_ptr->imbr == NULL)
        {
            core_ptr->total_imbrs = 0;
            return(-1);
        }
        else
        {
            core_ptr->imbr[i] = NULL;
        }

        /* Create new image browser. */
        core_ptr->imbr[i] = imbr = EDVImbrNew(core_ptr);
        if(imbr != NULL)
        {
            gchar *startup_dir;


	    EDVImbrSetBusy(imbr, TRUE);
            GUIBlockInput(imbr->toplevel, TRUE);

            /* Need to map it immediatly after creation, before any
             * timeouts check on it.
             */
            EDVImbrMap(imbr);

            /* Get copy of start up directory. */
            startup_dir = EDVCopyEvaluateInputPath(
                NULL,
                EDVCFGItemListGetValueS(
                    core_ptr->cfg_list, EDV_CFG_PARM_DIR_START_UP
                )
            );

            /* Switch to startup directory. */
	    EDVImbrSelectPath(imbr, startup_dir);

	    EDVImbrUpdateMenus(imbr);

            GUIBlockInput(imbr->toplevel, FALSE);
            EDVImbrSetBusy(imbr, FALSE);

            /* Deallocate copy of start up directory. */
            g_free(startup_dir);
            startup_dir = NULL;

            return(i);
        }
        else
        {
            return(-1);
        }
}

/*
 *      Procedure to create a new archiver and map it. The new
 *	archiver will be appended to the core_ptr or an existing
 *      (unmapped) archiver will be used.
 *
 *      Returns the index of the archiver on the core structure or -1
 *      on error.
 */
gint EDVDoNewArchiverOpen(edv_core_struct *core_ptr, const gchar *path)
{
        gint i;
        edv_archiver_struct *archiver;


        if(core_ptr == NULL)
            return(-1);

        /* Sanitize total. */
        if(core_ptr->total_archivers < 0)
            core_ptr->total_archivers = 0;

        /* Get highest index and increase total. */
        i = core_ptr->total_archivers;
        core_ptr->total_archivers = i + 1;

        /* Allocate more pointers. */
        core_ptr->archiver = (edv_archiver_struct **)g_realloc(
            core_ptr->archiver,
            core_ptr->total_archivers * sizeof(edv_archiver_struct *)
        );
        if(core_ptr->archiver == NULL)
        {
            core_ptr->total_archivers = 0;
            return(-1);
        }
        else
        {
            core_ptr->archiver[i] = NULL;
        }

        /* Create new archiver. */
        core_ptr->archiver[i] = archiver = EDVArchiverNew(core_ptr);
        if(archiver != NULL)
        {
            EDVArchiverSetBusy(archiver, TRUE);
            GUIBlockInput(archiver->toplevel, TRUE);

            /* Need to map it immediatly after creation, before any
             * timeouts check on it.
             */
            EDVArchiverMap(archiver);


	    if(path != NULL)
	    {
		/* Clear the contents clist and load the listing of the
		 * new archive specified by path.
                 */
                EDVArchiverSelectArchive(archiver, path);
	    }


            EDVArchiverUpdateMenus(archiver);

            GUIBlockInput(archiver->toplevel, FALSE);
            EDVArchiverSetBusy(archiver, FALSE);

            return(i);
        }
        else
        {
            return(-1);
        }
}

gint EDVDoNewArchiver(edv_core_struct *core_ptr)
{
	return(EDVDoNewArchiverOpen(core_ptr, NULL));
}


/*
 *	Procedure to create a property window and map it. The new property
 *	window will be appended to the core_ptr or an existing (unmapped)
 *	browser will be used.
 *
 *	Returns the index of the property window on the core structure or
 *	-1 on error.
 */
gint EDVDoPropWinAndSwitchPage(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	GtkWidget *ref_toplevel, const gchar *tab_name
)
{
        gint i;
        edv_propwin_struct *propwin;


        if(core_ptr == NULL)
            return(-1);

        /* Sanitize total. */
        if(core_ptr->total_propwins < 0)
            core_ptr->total_propwins = 0;

        /* Get highest index and increase total. */
        i = core_ptr->total_propwins;
        core_ptr->total_propwins = i + 1;

        /* Allocate more pointers. */
        core_ptr->propwin = (edv_propwin_struct **)g_realloc(
            core_ptr->propwin,
            core_ptr->total_propwins * sizeof(edv_propwin_struct *)
        );
        if(core_ptr->propwin == NULL)
        {
            core_ptr->total_propwins = 0;
            return(-1);
        }
        else
        {
            core_ptr->propwin[i] = NULL;
        }

        /* Create new property window. */
        core_ptr->propwin[i] = propwin = EDVPropWinNew(core_ptr, obj);
        if(propwin != NULL)
        {
	    GtkNotebook *notebook = GTK_NOTEBOOK(propwin->notebook);

	    /* Select tab? */
	    if((tab_name != NULL) && (notebook != NULL))
	    {
		gint page_num = -1, total_pages = 0;
		GList *glist = notebook->first_tab;


		/* Count total pages. */
		while(glist != NULL)
		{
		    total_pages++;
		    glist = glist->next;
		}

		/* Check which tab is to be switched to by matching its
		 * name.
		 */
		if(!strcasecmp(tab_name, "General"))
		    page_num = 0;
		else if(!strcasecmp(tab_name, "Link"))
                    page_num = 1;
                else if(!strcasecmp(tab_name, "Device"))
                    page_num = 1;
                else if(!strcasecmp(tab_name, "Device Node"))
                    page_num = 1;

		/* Matched page to be switched to? */
		if(page_num > -1)
		    gtk_notebook_set_page(notebook, page_num);
	    }


	    EDVCenterWindowToWindow(ref_toplevel, propwin->toplevel);

            /* Need to map it immediatly after creation, before any
             * timeouts check on it.
             */
            EDVPropWinMap(propwin);

            return(i);
        }
        else
        {
            return(-1);
        }
}

gint EDVDoPropWin(
	edv_core_struct *core_ptr, edv_object_struct *obj,
	GtkWidget *ref_toplevel
)
{
	return(
	    EDVDoPropWinAndSwitchPage(core_ptr, obj, ref_toplevel, NULL)
	);
}


/*
 *	Runs the device check command on the given device reference.
 */
void EDVDoRunDeviceCheck(
        edv_core_struct *core_ptr, edv_device_struct *dev_ptr,
	GtkWidget *ref_toplevel
)
{
        pid_t p;
        const gchar *cmd;


	if((core_ptr == NULL) || (dev_ptr == NULL))
	    return;

        cmd = dev_ptr->command_check;
        if(cmd == NULL)
            return;

        /* Seek past spaces in command string and make sure it is not
         * empty.
         */
        while(ISBLANK(*cmd))
            cmd++;
        if(*cmd == '\0')
            return;

        p = Exec(cmd);
        if(p == 0)
        {
            gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute command:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n",
#endif
                cmd
            );

            CDialogSetTransientFor(ref_toplevel);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Cannot Run Program",
                    buf,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
                    buf,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
                    buf,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );

            g_free(buf);
        }
}

/*
 *      Runs the device tools command on the given device reference.
 */
void EDVDoRunDeviceTools(
        edv_core_struct *core_ptr, edv_device_struct *dev_ptr,
        GtkWidget *ref_toplevel
)
{
        pid_t p;
        const gchar *cmd;


        if((core_ptr == NULL) || (dev_ptr == NULL))
            return;

        cmd = dev_ptr->command_tools;
        if(cmd == NULL)
            return;

        /* Seek past spaces in command string and make sure it is not
         * empty.
         */
        while(ISBLANK(*cmd))
            cmd++;
        if(*cmd == '\0')
            return;

        p = Exec(cmd);
        if(p == 0)
        {
            gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute command:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n",
#endif
                cmd
            );

            CDialogSetTransientFor(ref_toplevel);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Cannot Run Program",
                    buf,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
                    buf,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
                    buf,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );

            g_free(buf);
        }
}

/*
 *      Runs the device format command on the given device reference.
 */
void EDVDoRunDeviceFormat(
        edv_core_struct *core_ptr, edv_device_struct *dev_ptr,
        GtkWidget *ref_toplevel
)
{
        pid_t p;
        const gchar *cmd;


        if((core_ptr == NULL) || (dev_ptr == NULL))
            return;

        cmd = dev_ptr->command_format;
        if(cmd == NULL)
            return;

        /* Seek past spaces in command string and make sure it is not
         * empty.
         */
        while(ISBLANK(*cmd))
            cmd++;
        if(*cmd == '\0')
            return;

        p = Exec(cmd);
        if(p == 0)
        {
            gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to execute command:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de ejecutar la orden:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour excuter l'ordre:\n\
\n\
    %s\n",
#endif
                cmd
            );

            CDialogSetTransientFor(ref_toplevel);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Cannot Run Program",
                    buf,
"An error was encountered while attempting to run the\n\
specified program. Please verify that the command\n\
is specified properly by going to Device->Devices...\n\
and edit the device reference.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"No Puede Correr El Programa",
                    buf,
"Un error se encontr al procurar para correr el\n\
el programa especificado. Verifique por favor que la\n\
orden es especificado apropiadamente yendo\n\
Artefacto->Aartefactos... y redacta la referencia de\n\
artefacto.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ne Peut Pas Courir Le Programme",
                    buf,
"Une erreur a t rencontre en tentant pour courir\n\
le programme spcifi. S'il vous plat vrifier que\n\
l'ordre est convenablement spcifi en allant\n\
Appareil->Appareils... et dite la rfrence d'appareil.",
#endif
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );

            g_free(buf);
        }
}

/*
 *	Runs the terminal defined on the configuration list on the
 *	given core structure. If cmd is not NULL, then the command
 *	specified by cmd will be appended to the terminal run command
 *	defined on the core structure.
 */
void EDVDoRunTerminal(edv_core_struct *core_ptr, const gchar *cmd)
{
        if(core_ptr == NULL)
            return;

	if(cmd != NULL)
	{
            /* Get terminal command for running a program with a
	     * terminal, format the command, then run the command.
	     */
            const gchar *cmd_src = EDVCFGItemListGetValueS(
                core_ptr->cfg_list, EDV_CFG_PARM_PROG_TERMINAL_RUN
            );
            if(cmd_src != NULL)
	    {
		gchar *cmd_tar = g_strdup_printf(
		    "%s %s",
		    cmd_src,
		    cmd
		);
		Exec(cmd_tar);
		g_free(cmd_tar);
	    }
	}
	else
	{
	    /* No command given, just run terminal. */
	    const gchar *cmd_src = EDVCFGItemListGetValueS(
		core_ptr->cfg_list, EDV_CFG_PARM_PROG_TERMINAL
	    );
	    if(cmd_src != NULL)
		Exec(cmd_src);
	}
}

/*
 *      Creates the recycle bin desktop icon as needed and maps it.
 */
void EDVDoMapRecBinDeskIcon(edv_core_struct *core_ptr)
{
	if(core_ptr == NULL)
	    return;

	if(core_ptr->recbin_deskicon == NULL)
	    core_ptr->recbin_deskicon = EDVRecBinDeskIconNew(core_ptr);
	EDVRecBinDeskIconMap(core_ptr->recbin_deskicon);
	EDVRecBinDeskIconUpdate(core_ptr->recbin_deskicon);
}

/*
 *	Creates the recycle bin window as needed and maps it.
 */
void EDVDoMapRecBin(edv_core_struct *core_ptr)
{
	GtkCList *clist;
        edv_recbin_struct *recbin;


        if(core_ptr == NULL)
            return;

        /* Get pointer to recycle bin window, create it as needed. */
	recbin = core_ptr->recbin;
        if(recbin == NULL)
            core_ptr->recbin = recbin = EDVRecBinNew(
                core_ptr
            );
        if(recbin == NULL)
            return;

	/* Need to map it immediatly after it is created so that it does
	 * not get deleted in the timeout callback when the recycle bin is
	 * updated further below.
	 */
        EDVRecBinMap(recbin);

	EDVRecBinSetBusy(recbin, TRUE);
	GUIBlockInput(recbin->toplevel, TRUE);

	clist = (GtkCList *)recbin->contents_clist;
	if(clist != NULL)
	{
	    gtk_clist_freeze(clist);
	    EDVRecBinContentsDoUpdate(recbin, TRUE);
	    gtk_clist_thaw(clist);
	}

	GUIBlockInput(recbin->toplevel, FALSE);
	EDVRecBinSetBusy(recbin, FALSE);

	EDVRecBinUpdateMenus(recbin);
}

/*
 *      Creates the history list window as needed and maps it.
 */
void EDVDoMapHistoryListWin(
	edv_core_struct *core_ptr, GtkWidget *rel_toplevel
)
{
        edv_history_listwin_struct *lw;


        if(core_ptr == NULL)
            return;

        /* Get pointer to history list window, create it as needed. */
        lw = core_ptr->history_listwin;
        if(lw == NULL)
            core_ptr->history_listwin = lw = EDVHistoryListWinNew(
                core_ptr
            );
        if(lw == NULL)
            return;

	EDVCenterWindowToWindow(rel_toplevel, lw->toplevel);
        EDVHistoryListWinSetBusy(lw, TRUE);
        EDVHistoryListWinMap(lw);
        EDVHistoryListWinFetchValues(lw);
        EDVHistoryListWinSetBusy(lw, FALSE);
}


/*
 *	Creates the options window as needed and maps it.
 */
void EDVDoMapOptionsWin(
        edv_core_struct *core_ptr, GtkWidget *rel_toplevel
)
{
        if(core_ptr == NULL)
            return;

        if(core_ptr->options_window == NULL)
            core_ptr->options_window = EDVCreateOptionsWindow(
                core_ptr
            );
        if(core_ptr->options_window == NULL)
            return;

        EDVCenterWindowToWindow(
	    rel_toplevel, core_ptr->options_window->toplevel
	);
        OptWinDoFetch(core_ptr->options_window);
        OptWinMap(core_ptr->options_window);
}

/*
 *      Creates the options window as needed and maps it.
 */
void EDVDoMapCustomizeWin(
        edv_core_struct *core_ptr, GtkWidget *rel_toplevel
)
{
        gint i;
        const gchar *tab_name = NULL;
        GtkWidget *w;
        if(core_ptr == NULL)
            return;


        /* Get pointer to customize window, create it as needed. */
        if(core_ptr->customize_window == NULL)
            core_ptr->customize_window = EDVCreateCustomizeWindow(
                core_ptr
            );
        if(core_ptr->customize_window == NULL)
            return;

        EDVCenterWindowToWindow(
	    rel_toplevel,
	    core_ptr->customize_window->toplevel
	);


        /* Fetch values from global configuration. */
        OptWinDoFetch(core_ptr->customize_window);


        /* Get toplevel widget of the given widget if any and see which
         * window called this function.
         */
        w = rel_toplevel;
        /* Check all file browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
        {
            edv_browser_struct *browser = core_ptr->browser[i];
            if(browser == NULL)
                continue;

            if((browser->toplevel != NULL) &&
               (browser->toplevel == w)
            )
                tab_name = "File Browser";
        }
        /* Check all image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
        {
            edv_imbr_struct *imbr = core_ptr->imbr[i];
            if(imbr == NULL)
                continue;

            if((imbr->toplevel != NULL) &&
               (imbr->toplevel == w)
            )
                tab_name = "Image Browser";
        }
        /* Check all archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
        {
            edv_archiver_struct *archiver = core_ptr->archiver[i];
            if(archiver == NULL)
                continue;

            if((archiver->toplevel != NULL) &&
               (archiver->toplevel == w)
            )
                tab_name = "Archiver";
        }
        /* Check recycle bin. */
        if(core_ptr->recbin != NULL)
        {
            edv_recbin_struct *recbin = core_ptr->recbin;
            if((recbin->toplevel != NULL) &&
               (recbin->toplevel == w)
            )
                tab_name = "Recycle Bin";
        }
        /* Select tab based on tab name that represents a matched
         * window (if any).
         */
        EDVCustomizeWindowSelectTab(core_ptr->customize_window, tab_name);

        /* Map customize window. */
        OptWinMap(core_ptr->customize_window);
}

/*
 *      Creates the devices list window as needed and maps it.
 */
void EDVDoMapDevicesListWin(
	edv_core_struct *core_ptr, GtkWidget *rel_toplevel
)
{
        edv_device_listwin_struct *lw;


        if(core_ptr == NULL)
            return;

        /* Get pointer to devices list window, create it as needed. */
        lw = core_ptr->device_listwin;
        if(lw == NULL)
            core_ptr->device_listwin = lw = EDVDevicesListWinNew(
                core_ptr
            );
        if(lw == NULL)
            return;

        EDVCenterWindowToWindow(rel_toplevel, lw->toplevel);
	EDVDevicesListSetBusy(lw, TRUE);
        EDVDevicesListWinMap(lw);
        EDVDevicesListWinFetchValues(lw);
	EDVDevicesListSetBusy(lw, FALSE);
}

/*
 *	Creates the MIME Types list window as needed and maps it.
 */
void EDVDoMapMIMETypesListWin(
	edv_core_struct *core_ptr, GtkWidget *rel_toplevel
)
{
	edv_mimetype_listwin_struct *lw;


	if(core_ptr == NULL)
	    return;

	/* Get pointer to MIME Types list window, create it as needed. */
	lw = core_ptr->mimetype_listwin;
	if(lw == NULL)
	    core_ptr->mimetype_listwin = lw = EDVMimeTypesListWinNew(
		core_ptr
	    );
	if(lw == NULL)
	    return;

        EDVCenterWindowToWindow(rel_toplevel, lw->toplevel);
        EDVMimeTypesListWinSetBusy(lw, TRUE);
	EDVMimeTypesListWinMap(lw);
	EDVMimeTypesListWinFetchValues(lw);
        EDVMimeTypesListWinSetBusy(lw, FALSE);

}

/*
 *	Creates the run dialog as needed and maps it.
 */
void EDVDoMapRunDlg(edv_core_struct *core_ptr)
{
	edv_run_dlg_struct *dlg;


	if(core_ptr == NULL)
            return;

        /* Get pointer to run dialog, create it as needed. */
	dlg = core_ptr->run_dlg;
        if(dlg == NULL)
	{
            core_ptr->run_dlg = dlg = EDVRunDlgNew(core_ptr);
	    EDVRunDlgFetchValues(dlg);
	}
        if(dlg == NULL)
            return;

        EDVRunDlgMap(dlg);
}



/*
 *	Maps the find window relative to the given browser.
 */
void EDVDoMapBrowserFindWin(
        edv_core_struct *core_ptr, edv_browser_struct *browser
)
{
	gint i, browser_num = -1;
	edv_findwin_struct *fw;


	if((core_ptr == NULL) || (browser == NULL))
	    return;

	/* Get pointer to find window and create the find window if it has
	 * not been created yet.
	 */
	fw = core_ptr->findwin;
	if(fw == NULL)
	    core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
	if(fw == NULL)
	    return;

	/* Find browser index on core structure that matches the pointer
	 * of the given browser and record as browser_num.
	 */
	for(i = 0; i < core_ptr->total_browsers; i++)
	{
	    if(core_ptr->browser[i] == browser)
	    {
		browser_num = i;
		break;
	    }
	}

	/* Set reference window for find window, only one input may be
	 * non-negative.
	 */
	EDVFindWinSetReferenceWindow(
	    fw,
	    browser_num,
	    -1,
	    -1,
	    -1
	);

	/* Set initial location of find window to match the current
	 * location of the given browser.
	 */
	EDVFindWinSetLocation(
	    fw,
	    EDVBrowserCurrentLocation(browser),
	    FALSE
	);

        EDVCenterWindowToWindow(browser->toplevel, fw->toplevel);
	EDVFindWinUpdateMenus(fw);
	EDVFindWinMap(fw);
}

/*
 *      Maps the find window relative to the given image browser.
 */
void EDVDoMapImbrFindWin(
        edv_core_struct *core_ptr, edv_imbr_struct *imbr
)
{
        gint i, imbr_num = -1;
        edv_findwin_struct *fw;


        if((core_ptr == NULL) || (imbr == NULL))
            return;

        /* Get pointer to find window and create the find window if it has
         * not been created yet.
         */
        fw = core_ptr->findwin;
        if(fw == NULL)
            core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
        if(fw == NULL)
            return;

        /* Find image browser index on core structure that matches the
	 * pointer of the given image browser and record as imbr_num.
         */
        for(i = 0; i < core_ptr->total_imbrs; i++)
        {
            if(core_ptr->imbr[i] == imbr)
            {
                imbr_num = i;
                break;
            }
        }

        /* Set reference window for find window, only one input may be
         * non-negative.
         */
        EDVFindWinSetReferenceWindow(
            fw,
            -1,
	    imbr_num,
            -1,
	    -1
        );

        /* Set initial location of find window to match the current
         * location of the given image browser.
         */
        EDVFindWinSetLocation(
            fw,
            EDVImbrCurrentLocation(imbr),
            FALSE
        );

        EDVCenterWindowToWindow(imbr->toplevel, fw->toplevel);
        EDVFindWinUpdateMenus(fw);
        EDVFindWinMap(fw);
}

/*
 *	Maps the find window relative to the given archiver.
 */
void EDVDoMapArchiverFindWin(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver
)
{
	gint i, archiver_num = -1;
        edv_findwin_struct *fw;


        if((core_ptr == NULL) || (archiver == NULL))
            return;

        /* Get pointer to find window and create the find window if it has
         * not been created yet.
         */
        fw = core_ptr->findwin;
        if(fw == NULL)
            core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
        if(fw == NULL)
            return;

        /* Find archiver index on core structure that matches the
         * pointer of the given archiver and record as archiver_num.
         */
        for(i = 0; i < core_ptr->total_archivers; i++)
        {
            if(core_ptr->archiver[i] == archiver)
            {
                archiver_num = i;
                break;
            }
        }

        /* Set reference window for find window, only one input may be
         * non-negative.
         */
        EDVFindWinSetReferenceWindow(
            fw,
            -1,
            -1,
            -1,
            archiver_num
        );

        /* Set initial location of find window to match the current
         * location of the given archiver.
         */
        EDVFindWinSetLocation(
            fw,
            EDVArchiverCurrentLocation(archiver),
            FALSE
        );

        EDVCenterWindowToWindow(archiver->toplevel, fw->toplevel);
        EDVFindWinUpdateMenus(fw);
        EDVFindWinMap(fw);
}

/*
 *      Maps the find window relative to the given recycle bin.
 */
void EDVDoMapRecBinFindWin(
        edv_core_struct *core_ptr, edv_recbin_struct *recbin
)
{
	const gchar *recycled_index_file;
        edv_findwin_struct *fw;


        if((core_ptr == NULL) || (recbin == NULL))
            return;

        /* Get pointer to find window and create the find window if it has
         * not been created yet.
         */
        fw = core_ptr->findwin;
        if(fw == NULL)
            core_ptr->findwin = fw = EDVFindWinNew(core_ptr);
        if(fw == NULL)
            return;

        /* Set reference window for find window, only one input may be
         * non-negative.
         */
        EDVFindWinSetReferenceWindow(
            fw,
            -1,
            -1,
	    0,		/* First and only recycle bin. */
	    -1
        );


        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );

        /* Set initial location of find window to be the recycled
	 * objects directory.
         */
        EDVFindWinSetLocation(
            fw,
            GetParentDir(recycled_index_file),
            FALSE
        );

        EDVCenterWindowToWindow(recbin->toplevel, fw->toplevel);
        EDVFindWinUpdateMenus(fw);
        EDVFindWinMap(fw);
}

/*
 *	Deletes all items in the devices popup list widget (if any) and
 *	adds the current devices on the core structure to the devices
 *	popup list.
 *
 *	If the devices popup list does not exist on the core structure
 *	then it will be created. So this function can also be used to
 *	initialize it.
 */
void EDVDoUpdateDevicesPUList(edv_core_struct *core_ptr)
{
	gint i;
	const gchar *dev_name;
	edv_device_struct *dev_ptr;
	pulist_struct *pulist;


	if(core_ptr == NULL)
	    return;

	/* Get devices popup list from core structure, create it as 
	 * needed.
	 */
	pulist = core_ptr->devices_pulist;
	if(pulist == NULL)
	{
	    /* Popup list does not exist, create it. */
	    core_ptr->devices_pulist = pulist = PUListNew();
	}
	if(pulist == NULL)
	    return;

	/* Delete all existing items from the popup list. */
	PUListClear(pulist);

	/* Add current devices list to popup list. */
	for(i = 0; i < core_ptr->total_devices; i++)
	{
	    dev_ptr = core_ptr->device[i];
	    if(dev_ptr == NULL)
		continue;

	    /* Skip devices who are marked as unlisted, in which case 
	     * they should not be listed on the devices popup list.
	     */
	    if(dev_ptr->unlisted)
		continue;


	    /* Realize device, loading icons as needed. */
	    EDVDeviceRealize(dev_ptr, FALSE);

	    /* Get device name. */
	    dev_name = dev_ptr->name;
	    if((dev_name == NULL) && (dev_ptr->device_path != NULL))
	    {
		dev_name = strrchr(dev_ptr->device_path, DIR_DELIMINATOR);
		if(dev_name == NULL)
		    dev_name = dev_ptr->device_path;
		else
		    dev_name++;
	    }
	    if(dev_name == NULL)
		dev_name = "(null)";

	    /* Add new item to devices popup list. */
	    PUListAddItemPixText(
		pulist, dev_name,
		dev_ptr->small_pixmap[0], dev_ptr->small_mask[0],
                (gpointer)i, NULL
	    );
	}
}

/*
 *	Updates the MIME Type class hint index values on the given core
 *	structure to the MIME Types list.
 */
void EDVDoUpdateMIMETypeHintIndexes(edv_core_struct *core_ptr)
{
	gint i;
	edv_mimetype_struct *mt_ptr;


	if(core_ptr == NULL)
	    return;

	/* Reset all indexes for all MIME Type classes. */
	core_ptr->mimetype_system_index_first = -1;
	core_ptr->mimetype_system_index_last = -1;
	core_ptr->mimetype_format_index_first = -1;
	core_ptr->mimetype_format_index_last = -1;
	core_ptr->mimetype_program_index_first = -1;
	core_ptr->mimetype_program_index_last = -1;
	core_ptr->mimetype_unique_index_first = -1;
	core_ptr->mimetype_unique_index_last = -1;

	/* Iterate through all MIME Types and update each class hint
	 * index.
	 */
	for(i = 0; i < core_ptr->total_mimetypes; i++)
	{
	    mt_ptr = core_ptr->mimetype[i];
	    if(mt_ptr == NULL)
		continue;

	    switch(mt_ptr->mt_class)
	    {
	      case EDV_MIMETYPE_CLASS_SYSTEM:
		if(core_ptr->mimetype_system_index_first < 0)
		    core_ptr->mimetype_system_index_first = i;
		core_ptr->mimetype_system_index_last = i;
		break;

              case EDV_MIMETYPE_CLASS_FORMAT:
                if(core_ptr->mimetype_format_index_first < 0)
                    core_ptr->mimetype_format_index_first = i;
                core_ptr->mimetype_format_index_last = i;
                break;

              case EDV_MIMETYPE_CLASS_PROGRAM:
                if(core_ptr->mimetype_program_index_first < 0)
                    core_ptr->mimetype_program_index_first = i;
                core_ptr->mimetype_program_index_last = i;
                break;

              case EDV_MIMETYPE_CLASS_UNIQUE:
                if(core_ptr->mimetype_unique_index_first < 0)
                    core_ptr->mimetype_unique_index_first = i;
                core_ptr->mimetype_unique_index_last = i;
                break;
	    }
	}

}

/*
 *      Deletes all items in the `open with' popup list widget (if any) and
 *      adds the current MIME Types of class application on the core
 *	structure to the `open with' popup list.
 *
 *      If the `open with' popup list does not exist on the core structure
 *      then it will be created. So this function can also be used to
 *      initialize it.
 */
void EDVDoUpdateOpenWithPUList(edv_core_struct *core_ptr)
{
        gint i;
        edv_mimetype_struct *mt_ptr;
        pulist_struct *pulist;


        if(core_ptr == NULL)
            return;

        /* Get `open with' popup list from core structure, create it as
         * needed.
         */
        pulist = core_ptr->openwith_pulist;
        if(pulist == NULL)
        {
            /* Popup list does not exist, create it. */
            core_ptr->openwith_pulist = pulist = PUListNew();
        }
        if(pulist == NULL)
            return;

        /* Delete all existing items from the popup list. */
        PUListClear(pulist);

        /* Add current MIME Types of class application to the popup list. */
        for(i = 0; i < core_ptr->total_mimetypes; i++)
        {
            mt_ptr = core_ptr->mimetype[i];
            if(mt_ptr == NULL)
                continue;

	    if(mt_ptr->type == NULL)
		continue;

	    /* Skip MIME Type if it is not of class application. */
	    if(mt_ptr->mt_class != EDV_MIMETYPE_CLASS_PROGRAM)
		continue;

            /* Realize MIME Type, loading icons as needed. */
            EDVMimeTypeRealize(mt_ptr, FALSE);

            /* Add new item to `open with' popup list. */
            PUListAddItemPixText(
                pulist,
		(mt_ptr->description != NULL) ?
		    mt_ptr->description : mt_ptr->type,
                mt_ptr->small_pixmap[0], mt_ptr->small_mask[0],
		(gpointer)i, NULL
            );
	}
}


/*
 *	Procedure to sync by saving all of Endeavour's configuration in
 *	memory to file and then flush all disk operations by calling
 *	sync().
 */
void EDVDoSyncDisks(edv_core_struct *core_ptr)
{
	static gboolean reenterent = FALSE;
	gboolean object_existed;
	const gchar *cstrptr;
	struct stat lstat_buf;


	if(core_ptr == NULL)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;

	/* Begin saving all of Endeavour's configuration in memory to
	 * file.
	 */

	/* MIME Types. */
        cstrptr = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_MIME_TYPES
        );
        if(cstrptr != NULL)
        {
            gchar *path = g_strdup(cstrptr);
	    object_existed = access(path, F_OK) ? FALSE : TRUE;
            EDVMimeTypeListSaveToFile(
                path, core_ptr->mimetype, core_ptr->total_mimetypes,
                FALSE,		/* Do not write MIME Types marked read_only. */
                NULL, core_ptr
            );
	    if(!lstat(path, &lstat_buf))
            {
                if(object_existed)
                    EDVObjectModifiedEmit(core_ptr, path, NULL, &lstat_buf);
                else
                    EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
            }
            g_free(path);
        }

        /* Device references. */
        cstrptr = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_DEVICES
        );
        if(cstrptr != NULL)
        {
            gchar *path = g_strdup(cstrptr);
            object_existed = access(path, F_OK) ? FALSE : TRUE;
            EDVDeviceListSaveToFile(
                path, core_ptr->device, core_ptr->total_devices,
                NULL, core_ptr
            );
            if(!lstat(path, &lstat_buf))
            {
                if(object_existed)
                    EDVObjectModifiedEmit(core_ptr, path, NULL, &lstat_buf);
                else
                    EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
            }
            g_free(path);
        }

	/* Configuration. */
	if(core_ptr->cfg_file != NULL)
	{
            gchar *path = g_strdup(core_ptr->cfg_file);
	    object_existed = access(path, F_OK) ? FALSE : TRUE;
	    EDVCFGSaveToFile(path, core_ptr->cfg_list);
            if(!lstat(path, &lstat_buf))
            {
                if(object_existed)
                    EDVObjectModifiedEmit(core_ptr, path, NULL, &lstat_buf);
                else
                    EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
            }
            g_free(path);
	}

	/* Flush all disk operations. */
	sync();

	reenterent = FALSE;
}


/*
 *	Calls the Endeavour Internet download program to download the
 *	given URL described in dnd_obj to the directory target_path.
 *
 *	If the dnd_obj is not given then the download program will still
 *	be runned but used to query where the user wants to download
 *	to.
 */
void EDVDoInternetDownloadObject(
        edv_core_struct *core_ptr,
        edv_dnd_object_struct *dnd_obj,         /* Source. */
        const gchar *target_path,               /* Target. */
	GtkWidget *toplevel
)
{
        static gbool reenterent = FALSE;
	gboolean confirm;
	const gchar *prog;


        if((core_ptr == NULL) || (target_path == NULL))
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


	confirm = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_CONFIRM_DOWNLOAD
        );


	/* Get network download program. */
	prog = EDVCFGItemListGetValueS(
	    core_ptr->cfg_list, EDV_CFG_PARM_PROG_NET_DOWNLOAD
	);
	if((prog != NULL) ? (*prog == '\0') : TRUE)
	{
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
                "No Download Program",
"Network download program not set",
"To set the network download program, please go\n\
to Settings->Options->Programs.",
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	}
	else
	{
	    gint p;
	    gchar *url, *cmd;
	    gulong time_start = time(NULL);

	    /* Format url. */
	    if(dnd_obj == NULL)
		url = g_strdup("");
	    else if(dnd_obj->port > 0)
		url = g_strdup_printf(
		    "%s://%s%s%s%s%s:%i%s",
		    (dnd_obj->protocol != NULL) ? dnd_obj->protocol : "file",
		    (dnd_obj->user != NULL) ? dnd_obj->user : "",
		    (dnd_obj->password != NULL) ? ":" : "",
		    (dnd_obj->password != NULL) ? dnd_obj->password : "",
		    (dnd_obj->user != NULL) ? "@" : "",
		    (dnd_obj->host != NULL) ? dnd_obj->host : "localhost",
		    dnd_obj->port,
		    dnd_obj->full_path
		);
	    else
                url = g_strdup_printf(
                    "%s://%s%s%s%s%s%s",
                    (dnd_obj->protocol != NULL) ? dnd_obj->protocol : "file",
                    (dnd_obj->user != NULL) ? dnd_obj->user : "",
                    (dnd_obj->password != NULL) ? ":" : "",
                    (dnd_obj->password != NULL) ? dnd_obj->password : "",
                    (dnd_obj->user != NULL) ? "@" : "",
                    (dnd_obj->host != NULL) ? dnd_obj->host : "localhost",
                    dnd_obj->full_path
                );

            /* Format command, the program should accept arguments being
             * the first argument(s) are the URLs to download and the
             * last argument is the destination path.
             */
	    cmd = g_strdup_printf(
		"%s %s %s \"%s\" \"%s\"",
		prog,
		"-b",
		confirm ? "-c" : "",
		url, target_path
	    );

	    /* Execute download command. */
	    p = Exec(cmd);
	    if(p <= 0)
	    {
		gchar *buf = g_strdup_printf(
"Unable to execute command:\n\n    %s",
		    cmd
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Download Failed",
		    buf,
"Make sure that the network download program is\n\
defined correctly, please go to\n\
Settings->Options->Programs.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
	    }

	    /* Record download history. */
	    EDVAppendHistory(
		core_ptr, EDV_HISTORY_DISK_OBJECT_DOWNLOAD,
		time_start, time(NULL), (p <= 0) ? -1 : 0,
		url, target_path,
		NULL
	    );

	    g_free(cmd);
	    g_free(url);
	}

        reenterent = FALSE;
}


/*
 *	Help calling nexus.
 *
 *	If topic is NULL then it is assumed to display the help contents
 *	"Contents".  Supported values for topic are:
 *
 *	"Contents"
 *      "File Browser"
 *      "Image Browser"
 *      "Archiver"
 *      "Recycle Bin"
 *      "Keys List"
 *	"Disk Objects"
 *	"MIME Types"
 *	"Devices"
 *	"Common Operations"
 *	"Front Ends"
 *	"FAQ"
 */
void EDVDoHelp(
	edv_core_struct *core_ptr, const gchar *topic, GtkWidget *toplevel
)
{
	gint i;
	const gchar *data_dir, *help_file, *help_path;
	help_prog_struct *hp_ptr, help_prog_list[] = EDV_HELP_PROG_LOCATIONS;
	help_file_struct *hf_ptr, help_file_list[] = EDV_HELP_FILE_LOCATIONS;

	if(core_ptr == NULL)
	    return;

	/* If no topic is given, then assume it means the help contents are
	 * to be displayed.
	 */
	if(topic == NULL)
	    topic = "Contents";

	/* Get global data directory in which the help files are assumed
	 * to be located in.
	 */
	data_dir = EDVCFGItemListGetValueS(
	    core_ptr->cfg_list, EDV_CFG_PARM_DIR_GLOBAL
	);
	if(data_dir == NULL)
	    return;

	/* See which help file to load based on the given topic name. */
        for(i = 0, hf_ptr = NULL;
            help_file_list[i].topic != NULL;
            i++
        )
        {
	    if(!strcasecmp(help_file_list[i].topic, topic))
	    {
                hf_ptr = &help_file_list[i];
                break;
            }
        }
	help_file = (hf_ptr != NULL) ? hf_ptr->file : NULL;
	if(help_file == NULL)
	    help_path = NULL;
	else
	    help_path = PrefixPaths(data_dir, help_file);

	/* No help file matched for the given topic? */
	if(help_path == NULL)
	{
	    gchar *buf = g_strdup_printf(
		"Unable to find help topic \"%s\".",
		topic
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"No Help Available", buf, NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    return;
	}


	/* Search for a help program to use. */
	for(i = 0, hp_ptr = NULL;
	    help_prog_list[i].prog != NULL;
	    i++
	)
	{
	    if(!access(help_prog_list[i].prog, X_OK))
	    {
		hp_ptr = &help_prog_list[i];
		break;
	    }
	}

	/* Found a help program? */
	if(hp_ptr != NULL)
	{
            pid_t p;
            gchar *cmd = NULL;

	    /* Format command string. */
	    cmd = strcatalloc(cmd, hp_ptr->prog);
	    if(hp_ptr->arg != NULL)
	    {
		cmd = strcatalloc(cmd, " ");
		cmd = strcatalloc(cmd, hp_ptr->arg);
	    }
	    switch(hp_ptr->arg_type)
	    {
	      case HELP_PROG_INPUT_TYPE_URL:
		cmd = strcatalloc(cmd, " file://");
		cmd = strcatalloc(cmd, help_path);
		break;

	      default:	/* HELP_PROG_INPUT_TYPE_PATH */
		cmd = strcatalloc(cmd, " ");
		cmd = strcatalloc(cmd, help_path);
		break;
	    }

	    /* Execute command. */
	    p = Exec(cmd);

	    /* Execution of help viewer failed? */
	    if(p == 0)
	    {
		gchar *buf = g_strdup_printf(
"Unable to execute help viewer command:\n\
\n\
    %s\n",
		    cmd
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
"Help Viewer Execution Failed",
		    buf,
"An error was encountered while attempting to run the help\n\
viewer to display the requested document. Please verify that\n\
the help viewer and the requested document are both\n\
installed properly.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		g_free(buf);
	    }

	    /* Deallocate command string. */
	    g_free(cmd);
            cmd = NULL;
	}
	else
	{
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
		"No Help Viewer",
"Unable to find a help viewer to display help documentation",
		NULL,
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	}
}

/*
 *	Maps the about window.
 */
void EDVDoHelpAbout(edv_core_struct *core_ptr, GtkWidget *toplevel)
{
        if(core_ptr == NULL)
            return;

        if(core_ptr->about_dialog == NULL)
            core_ptr->about_dialog = AboutDialogNew(core_ptr);
        AboutDialogMap(core_ptr->about_dialog);
}


/*
 *	Tabulates the total size of all objects in the recycled objects
 *	directory and checks if it has exceeded the configuration
 *	specified recycle bin size warn size. If it has exceeded then
 *	the user will be warned.
 */
void EDVDoRecycleBinSizeCheck(
        edv_core_struct *core_ptr, GtkWidget *toplevel
)
{
	const gchar *recycled_index_file;
        edv_recbin_index_struct *rbi_ptr;
        edv_recbin_object_struct *obj;
	gulong recbin_size_warn = 0, total_size = 0;


	if(core_ptr == NULL)
	    return;

	/* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
            return;

	/* Get recycle bin size warn. */
	recbin_size_warn = EDVCFGItemListGetValueUL(
	    core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SIZE_WARN
	);
	/* If the warn size is 0, then this implies no warning should
	 * be given (in which case do not bother to check anything).
	 */
	if(recbin_size_warn == 0)
	    return;

        /* Get recycled objects directory index pointer used to iterate
         * through all recycled objects.
         */
        rbi_ptr = EDVRecBinIndexOpen(recycled_index_file);
        while(!EDVRecBinIndexNext(rbi_ptr))
        {
            /* Get recycled object structure obtained from the recycled
             * objects directory index structure. This structure is shared
             * so do not modify or deallocate it.
             */
            obj = rbi_ptr->obj;
            if(obj == NULL)
                continue;

	    /* Increase total size of recycled objects by this recycled
	     * object.
	     */
	    total_size += obj->size;
	}
        EDVRecBinIndexClose(rbi_ptr);
	rbi_ptr = NULL;


	/* At this point the total size of all recycled objects has been
	 * obtained. Now check if total_size reaches or exceeds the
	 * configuration set size to warn.
	 */
	if(total_size >= recbin_size_warn)
	{
	    gchar total_size_s[256], recbin_size_warn_s[256];
            gchar *buf;

	    strcpy(
		total_size_s,
		EDVGetObjectSizeStr(core_ptr, total_size)
	    );
            strcpy(
                recbin_size_warn_s,
                EDVGetObjectSizeStr(core_ptr, recbin_size_warn)
            );

	    buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"The size of the recycle bin is now %s bytes,\n\
it has exceeded the user specified size of %s bytes\n\
at which this warning is to be shown.\n\
\n\
To purge all the existing recycled objects, first\n\
go to the Recycle Bin and then go to File->Purge All.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El tamao del cajn de la recirculacin es ahora %s bytes,\n\
ha excedido al usuario el tamao especificado de\n\
%s bytes en que este advertir deber ser mostrado.\n\
\n\
Al purge todos los objetos existentes de recycled, van\n\
primero al Cajn de la Recirculacin y entonces van a\n\
File->Purge All.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"La taille du recycle l'huche est maintenant %s bytes,\n\
il a dpass l'utilisateur la taille spcifie de\n\
%s bytes  qui cet avertissement devrait tre montr.\n\
\n\
Purger tout l'existant recycl objets, premirement va\n\
au Recycle Huche et alors va File->Purge All.",
#endif
                total_size_s, recbin_size_warn_s
            );

            CDialogSetTransientFor(toplevel);
            CDialogGetResponseIconData(
#ifdef PROG_LANGUAGE_ENGLISH
                "Recycle Bin Size Warning",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Advertir De Cajn De Recirculacin",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Recycler l'Avertissement d'Huche",
#endif
		buf, NULL,
                (guint8 **)icon_trash_32x32_xpm,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

            g_free(buf);
	}
}
