#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "edvtypes.h"
#include "edvobj.h"
#include "edvutils.h"


gint EDVObjectGetTypeFromStatMode(mode_t m);
mode_t EDVObjectGetTypeFromEDVType(gint type);

guint EDVObjectGetPermissionsFromStatMode(mode_t m);
mode_t EDVObjectGetPermissionsFromEDVPermissions(guint permissions);

edv_object_struct *EDVObjectNew(void);
edv_object_struct *EDVObjectCopy(edv_object_struct *obj);
void EDVObjectSetPath(
        edv_object_struct *obj, const gchar *path
);
void EDVObjectSetPath2(
        edv_object_struct *obj,
        const gchar *parent, const gchar *name
);
void EDVObjectSetStat(
        edv_object_struct *obj, const struct stat *lstat_buf
);
void EDVObjectValidateLink(edv_object_struct *obj);
void EDVObjectDelete(edv_object_struct *obj);



/*
 *	Returns one of EDV_OBJECT_TYPE_* based on the type of object
 *	specified by the mode mask obtained from *stat().
 */
gint EDVObjectGetTypeFromStatMode(mode_t m)
{
	if(S_ISLNK(m))
	    return(EDV_OBJECT_TYPE_LINK);
	else if(S_ISREG(m))
	    return(EDV_OBJECT_TYPE_FILE);
	else if(S_ISDIR(m))
	    return(EDV_OBJECT_TYPE_DIRECTORY);
	else if(S_ISCHR(m))
	    return(EDV_OBJECT_TYPE_DEVICE_CHARACTER);
	else if(S_ISBLK(m))
	    return(EDV_OBJECT_TYPE_DEVICE_BLOCK);
	else if(S_ISFIFO(m))
	    return(EDV_OBJECT_TYPE_FIFO);
	else if(S_ISSOCK(m))
	    return(EDV_OBJECT_TYPE_SOCKET);
	else
	    return(EDV_OBJECT_TYPE_FILE);
}

/*
 *	Returns a stat() mode mask containing one of the type flags.
 */
mode_t EDVObjectGetTypeFromEDVType(gint type)
{
	mode_t m = 0;

	switch(type)
	{
	  case EDV_OBJECT_TYPE_FILE:
	    m |= S_IFREG;
	    break;

          case EDV_OBJECT_TYPE_DIRECTORY:
            m |= S_IFDIR;
            break;

          case EDV_OBJECT_TYPE_LINK:
            m |= S_IFLNK;
            break;

          case EDV_OBJECT_TYPE_FIFO:
            m |= S_IFIFO;
            break;

          case EDV_OBJECT_TYPE_DEVICE_BLOCK:
            m |= S_IFBLK;
            break;

          case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
            m |= S_IFCHR;
            break;

          case EDV_OBJECT_TYPE_SOCKET:
            m |= S_IFSOCK;
            break;
	}

	return(m);
}

/*
 *	Returns an Endeavour permissions value or'ed by EDV_PERMISSION_*
 *	flags based on the permissions of the object specified by the
 *	mode mask obtained from *stat().
 */
guint EDVObjectGetPermissionsFromStatMode(mode_t m)
{
	guint p = 0x00000000;

        if(m & S_IXUSR)
            p |= EDV_PERMISSION_UEXECUTE;
        if(m & S_IRUSR)
            p |= EDV_PERMISSION_UREAD;
        if(m & S_IWUSR)
            p |= EDV_PERMISSION_UWRITE;

        if(m & S_IXGRP)
            p |= EDV_PERMISSION_GEXECUTE;
        if(m & S_IRGRP)
            p |= EDV_PERMISSION_GREAD;
        if(m & S_IWGRP)
            p |= EDV_PERMISSION_GWRITE;

        if(m & S_IXOTH)
            p |= EDV_PERMISSION_AEXECUTE;
        if(m & S_IROTH)
            p |= EDV_PERMISSION_AREAD;
        if(m & S_IWOTH)
            p |= EDV_PERMISSION_AWRITE;

        if(m & S_ISUID)
            p |= EDV_PERMISSION_SETUID;
        if(m & S_ISGID)
            p |= EDV_PERMISSION_SETGID;
        if(m & S_ISVTX)
            p |= EDV_PERMISSION_STICKY;

	return(p);
}

/*
 *      Returns a permissions value or'ed by the stat flags based
 *      on the permissions of the object specified by the Endeavour
 *	permissions mask.
 */
mode_t EDVObjectGetPermissionsFromEDVPermissions(guint permissions)
{
	mode_t m = 0;


        if(permissions & EDV_PERMISSION_UEXECUTE)
            m |= S_IXUSR;
        if(permissions & EDV_PERMISSION_UREAD)
            m |= S_IRUSR;
        if(permissions & EDV_PERMISSION_UWRITE)
            m |= S_IWUSR;

        if(permissions & EDV_PERMISSION_GEXECUTE)
            m |= S_IXGRP;
        if(permissions & EDV_PERMISSION_GREAD)
            m |= S_IRGRP;
        if(permissions & EDV_PERMISSION_GWRITE)
            m |= S_IWGRP;

        if(permissions & EDV_PERMISSION_AEXECUTE)
            m |= S_IXOTH;
        if(permissions & EDV_PERMISSION_AREAD)
            m |= S_IROTH;
        if(permissions & EDV_PERMISSION_AWRITE)
            m |= S_IWOTH;

        if(permissions & EDV_PERMISSION_SETUID)
            m |= S_ISUID;
        if(permissions & EDV_PERMISSION_SETGID)
            m |= S_ISGID;
        if(permissions & EDV_PERMISSION_STICKY)
            m |= S_ISVTX;

	return(m);
}


/*
 *	Allocates a new edv_object_struct.
 */
edv_object_struct *EDVObjectNew(void)
{
	edv_object_struct *obj = (edv_object_struct *)g_malloc0(
	    sizeof(edv_object_struct)
	);
	if(obj == NULL)
	    return(obj);

	obj->type = EDV_OBJECT_TYPE_FILE;
	obj->name = NULL;
	obj->full_path = NULL;
	obj->linked_to = NULL;
	obj->link_valid = FALSE;
	obj->permissions = 0;
	obj->access_time = 0;
	obj->modify_time = 0;
	obj->change_time = 0;
	obj->owner_id = -1;
	obj->group_id = -1;
	obj->hard_links = 0;
	obj->size = 0;
	obj->device = 0;
        obj->inode = 0;
        obj->device_type = 0;
        obj->block_size = 0;
        obj->blocks = 0;

	return(obj);
}


/*
 *	Creates a copy of the given disk object structure.
 */
edv_object_struct *EDVObjectCopy(edv_object_struct *obj)
{
	edv_object_struct *obj2;


	if(obj == NULL)
	    return(NULL);

	/* Create a new disk object structure. */
	obj2 = EDVObjectNew();
	if(obj2 == NULL)
	    return(NULL);

	/* Begin copying values from given disk object structure to the
	 * newly created disk object structure.
	 */
	obj2->type = obj->type;

	if(obj->name != NULL)
	    obj2->name = g_strdup(obj->name);

	if(obj->full_path != NULL)
	    obj2->full_path = g_strdup(obj->full_path);

	if(obj->linked_to != NULL)
	    obj2->linked_to = g_strdup(obj->linked_to);

	obj2->link_valid = obj->link_valid;
	obj2->permissions = obj->permissions;
	obj2->access_time = obj->access_time;
	obj2->change_time = obj->change_time;
	obj2->owner_id = obj->owner_id;
	obj2->group_id = obj->group_id;
	obj2->hard_links = obj->hard_links;
	obj2->size = obj->size;
        obj2->device = obj->device;
        obj2->inode = obj->inode;
        obj2->device_type = obj->device_type;
        obj2->block_size = obj->block_size;
        obj2->blocks = obj->blocks;

	return(obj2);
}


/*
 *      Sets the disk object structure's name and full path from the given
 *	full path.
 */
void EDVObjectSetPath(
        edv_object_struct *obj, const gchar *path
)
{
        if(obj == NULL)
            return;


        /* Reset names first. */
        g_free(obj->name);
        obj->name = NULL;
        g_free(obj->full_path);
        obj->full_path = NULL;


	/* Enough information to set path? */
	if(path != NULL)
	{
	    const gchar *cstrptr;


	    /* Set full path. */
	    obj->full_path = g_strdup(path);

	    /* Remove tailing deliminators if any. */
	    StripPath(obj->full_path);

	    /* Find child and set name. */
	    cstrptr = strrchr(
		(obj->full_path != NULL) ? obj->full_path : path,
		DIR_DELIMINATOR
	    );
	    if(cstrptr != NULL)
	    {
		while(*cstrptr == DIR_DELIMINATOR)
		    cstrptr++;

		/* Last character was a deliminator? */
		if(*cstrptr == '\0')
		    obj->name = g_strdup("/");
		else
		    obj->name = g_strdup(cstrptr);
	    }
	    else
	    {
		/* Unable to find child, so just set full path as
		 * the name.
		 */
		obj->name = g_strdup(path);
	    }
	}
}

/*
 *	Sets the object disk object's name and full path given a parent and
 *	name.
 */
void EDVObjectSetPath2(
	edv_object_struct *obj,
	const gchar *parent, const gchar *name
)
{
        if(obj == NULL)
            return;


	/* Reset names first. */
	g_free(obj->name);
	obj->name = NULL;
	g_free(obj->full_path);
	obj->full_path = NULL;


	/* Enough info to set full path? */
	if((parent == NULL) && (name != NULL))
	{
	    const gchar *cstrptr = PrefixPaths(parent, name);
	    if(cstrptr != NULL)
		obj->full_path = g_strdup(cstrptr);
	}

	/* Enough info to set name? */
	if(name != NULL)
	{
	    obj->name = g_strdup(name);
	}
}

/*
 *	Sets new values for the disk object from the given local stats.
 *
 *	When updating a disk object structure, the path to the disk object
 *	structure must be updated first with a call to EDVObjectSetPath*()
 *	before calling this function.
 */
void EDVObjectSetStat(
        edv_object_struct *obj, const struct stat *lstat_buf
)
{
	mode_t m;


        if((obj == NULL) || (lstat_buf == NULL))
            return;

	m = lstat_buf->st_mode;

	/* Begin setting type and permissions. */
	obj->type = EDVObjectGetTypeFromStatMode(m);
	obj->permissions = EDVObjectGetPermissionsFromStatMode(m);

	/* Begin setting time stamps. */
	obj->access_time = (gulong)lstat_buf->st_atime;
	obj->modify_time = (gulong)lstat_buf->st_mtime;
	obj->change_time = (gulong)lstat_buf->st_ctime;

	/* Begin setting ownership id. */
	obj->owner_id = lstat_buf->st_uid;
	obj->group_id = lstat_buf->st_gid;

	/* Set number of hard links. */
	obj->hard_links = lstat_buf->st_nlink;

	/* Assume link is valid for now. */
	obj->link_valid = TRUE;

	/* Set size in bytes. */
	obj->size = (gulong)lstat_buf->st_size;


	/* Set device and inod. */
        obj->device = lstat_buf->st_dev;
        obj->inode = lstat_buf->st_ino;
        obj->device_type = lstat_buf->st_rdev;
        obj->block_size = lstat_buf->st_blksize;
        obj->blocks = lstat_buf->st_blocks;


	/* If this is a link then update the linked_to destination. */
        if(S_ISLNK(m))
	{
#define buf_len		(PATH_MAX + NAME_MAX)
	    gchar buf[buf_len + 1];
	    gint bytes_read;


	    bytes_read = readlink(
		obj->full_path, buf, buf_len
	    );
	    if((bytes_read >= 0) && (bytes_read < buf_len))
	    {
		buf[bytes_read] = '\0';

		g_free(obj->linked_to);
		obj->linked_to = g_strdup(buf);
	    }
#undef buf_len
	}
}

/*
 *	Updates the disk object structure's link_valid member based on if
 *	the full path specified on the disk object structure can be
 *	stat()'ed.
 *
 *	Note that the object type is not checked, nor does it need to be.
 */
void EDVObjectValidateLink(edv_object_struct *obj)
{
	struct stat stat_buf;


	if(obj == NULL)
	    return;

	if(obj->full_path == NULL)
	    return;

	/* Check if destination can be stat()'ed. */
	if(stat(obj->full_path, &stat_buf))
	    obj->link_valid = FALSE;
	else
	    obj->link_valid = TRUE;
}

/*
 *	Deallocates the given edv_object_struct and all its resources.
 */
void EDVObjectDelete(edv_object_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->full_path);

	g_free(obj->linked_to);

	g_free(obj);
}
