#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edvtypes.h"
#include "edvmimetypes.h"
#include "edvmimetypesfio.h"
#include "endeavour.h"
#include "config.h"


void EDVMimeTypeListImportMailCap(
	const gchar *filename,
	edv_mimetype_struct ***list, gint *total,
        gint insert_index,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer progress_client_data,
	void (*add_emit)(edv_core_struct *, gint, edv_mimetype_struct *),
 	gpointer add_emit_core
);
void EDVMimeTypeListExportMailCap(
        const gchar *filename,
        edv_mimetype_struct **list, gint total,
	gbool include_read_only,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);

void EDVMimeTypeListImportMediaTypes(
        const gchar *filename,
        edv_mimetype_struct ***list, gint *total,
        gint insert_index,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer progress_client_data,
        void (*add_emit)(edv_core_struct *, gint, edv_mimetype_struct *),
        gpointer add_emit_core
);
void EDVMimeTypeListExportMediaTypes(
        const gchar *filename,
        edv_mimetype_struct **list, gint total,
	gbool include_read_only,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);

void EDVMimeTypeListLoadFromFile(
        const gchar *filename,
        edv_mimetype_struct ***list, gint *total,
        gint insert_index,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer progress_client_data,
        void (*add_emit)(edv_core_struct *, gint, edv_mimetype_struct *),
        gpointer add_emit_core,
        gbool mark_all_loaded_read_only
);
void EDVMimeTypeListSaveToFile(
        const gchar *filename,
        edv_mimetype_struct **list, gint total,
	gbool include_read_only,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);


#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define BLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Imports the MIME Types obtained from the MUTT mailcap format file
 *	specified by filename.
 *
 *	The insert_index specifies the MIME Type index in the given list
 *	of MIME Types at which to insert imported MIME Types at. If
 *	insert_index is -1 then all imported MIME Types will be appended.
 */
void EDVMimeTypeListImportMailCap(
        const gchar *filename,
        edv_mimetype_struct ***list, gint *total,
	gint insert_index,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer progress_client_data,
        void (*add_emit)(edv_core_struct *, gint, edv_mimetype_struct *),
        gpointer add_emit_core
)
{
	gbool c_literal;
	gint i, c;
	FILE *fp;
	gulong file_size;
	gchar *strptr;
#define type_str_len	1024
	gchar type_str[type_str_len];
#define cmd_str_len	2048
	gchar cmd_str1[cmd_str_len];
        gchar cmd_str2[cmd_str_len];

	gint mt_num = -1;
	edv_mimetype_struct *mt_ptr = NULL;

	struct stat stat_buf;


	if((list == NULL) || (total == NULL) || (filename == NULL))
            return;

        /* Open MailCap file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
            return;

        /* Get file statistics. */
        if(fstat(fileno(fp), &stat_buf))
            file_size = 0;
        else
            file_size = stat_buf.st_size;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                progress_client_data, 0, file_size
            );

	/* Begin parsing mailcap file, the mailcap file format consists
	 * of newline terminated strings. Each `line' contains ';'
	 * characters used to deliminate fields as follows:
	 * name;command1;command2;command3;command4...
	 */

	while(1)
	{
#define GET_NEXT_CHAR_LITERAL	\
{ \
 c = fgetc(fp); \
 if(c == '\\') \
 { \
  c = fgetc(fp); \
  c_literal = TRUE; \
  if(ISCR(c)) \
  { \
   c = fgetc(fp); \
   c_literal = FALSE; \
  } \
 } \
 else \
 { \
  c_literal = FALSE; \
 } \
}

	    /* Get first character of current `line'. */
	    GET_NEXT_CHAR_LITERAL
	    if(c == EOF)
		break;

	    /* Seek past initial spaces. */
	    while(ISBLANK(c))
                GET_NEXT_CHAR_LITERAL
	    if(c == EOF)
                break;

	    /* Empty line? */
	    if(ISCR(c))
		continue;

	    /* Comment? */
	    if(!c_literal && (c == '#'))
	    {
		/* Seek to start of next line. */
		while(!ISCR(c) && (c != EOF))
		    GET_NEXT_CHAR_LITERAL

		continue;
	    }

	    /* Call progress callback. */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		     progress_client_data, (gulong)ftell(fp), file_size
		))
		    break;
	    }


	    /* Begin fetching type string, this is always the first
	     * field before the first ';' deliminator character.
	     */
	    strptr = type_str;
	    for(i = 0; i < type_str_len; i++)
	    {
		if(!c_literal && (c == ';'))
		{
		    GET_NEXT_CHAR_LITERAL
		    break;
		}
		else if(ISCR(c) || (c == EOF))
		{
		    break;
		}
		else
		{
		    *strptr++ = (gchar)c;
		    GET_NEXT_CHAR_LITERAL
		}
	    }
	    /* Null terminate the string. */
	    if(i < type_str_len)
		type_str[i] = '\0';
	    else
		type_str[type_str_len - 1] = '\0';

	    if(c == EOF)
		break;


	    /* Begin fetching up to two subsequent commands. */
/* Macro to fetch command string pointed to by strptr. */
#define DO_FETCH_COMMAND	\
{ \
 /* Seek past spaces. */ \
 while(ISBLANK(c)) \
  GET_NEXT_CHAR_LITERAL \
\
 for(i = 0; i < cmd_str_len; i++) \
 { \
  if(!c_literal && (c == ';')) \
  { \
   GET_NEXT_CHAR_LITERAL \
   *strptr = '\0'; \
   break; \
  } \
  else if(ISCR(c) || (c == EOF)) \
  { \
   *strptr = '\0'; \
   break; \
  } \
  else \
  { \
   *strptr++ = (gchar)c; \
   GET_NEXT_CHAR_LITERAL \
  } \
 } \
}
	    strptr = cmd_str1;
	    DO_FETCH_COMMAND
	    cmd_str1[cmd_str_len - 1] = '\0';

	    strptr = cmd_str2;
            DO_FETCH_COMMAND
            cmd_str2[cmd_str_len - 1] = '\0';

#undef DO_FETCH_COMMAND

	    /* In case end of line has not been reached, seek untill the
	     * next newline character is encountered.
	     */
	    while(!ISCR(c) && (c != EOF))
		GET_NEXT_CHAR_LITERAL


	    /* Begin creating new imported MIME Type. */

	    /* Reset contexts. */
	    mt_num = -1;
	    mt_ptr = NULL;

	    /* Sanitize total. */
	    if(*total < 0)
		*total = 0;

	    /* Append? */
	    if(insert_index < 0)
	    {
		/* Append. */

                /* Increase total and allocate more pointers. */
                mt_num = *total;
                *total = mt_num + 1;

                *list = (edv_mimetype_struct **)g_realloc(
                    *list,
                    (*total) * sizeof(edv_mimetype_struct *)
                );
                if(*list == NULL)
                {
                    *total = 0;
                    break;
                }

                /* Allocate the new MIME Type structure and append it at
                 * the append index.
                 */
                (*list)[mt_num] = mt_ptr = EDVMimeTypeNew(
                    EDV_MIMETYPE_CLASS_FORMAT,
                    NULL,               /* Value. */
                    type_str,
                    NULL                /* Description. */
                );
	    }
	    else
	    {
		/* Insert. */

		/* Increase total and allocate more pointers. */
		i = *total;
		*total = i + 1;

		*list = (edv_mimetype_struct **)g_realloc(
		    *list,
		    (*total) * sizeof(edv_mimetype_struct *)
		);
		if(*list == NULL)
		{
		    *total = 0;
		    break;
		}

		/* Get insert index and make sure it is in bounds. */
		mt_num = insert_index;
		if(mt_num >= *total)
		    mt_num = *total - 1;

		/* Shift pointers. */
		for(i = *total - 1; i > mt_num; i--)
		    (*list)[i] = (*list)[i - 1];

		/* Allocate the new MIME Type structure and insert it at
		 * the insert index.
		 */
		(*list)[mt_num] = mt_ptr = EDVMimeTypeNew(
		    EDV_MIMETYPE_CLASS_FORMAT,
		    NULL,		/* Value. */
		    type_str,
		    NULL		/* Description. */
		);

		/* Increment insert index for next MIME Type. */
		insert_index++;
	    }

	    /* New MIME Type added successfully? */
	    if((mt_ptr != NULL) && (mt_num >= 0) && (mt_num < *total))
	    {
		/* Get extension if a "x-" string is found after the
		 * last deliminator character of the mime type's name.
		 * For example "image/x-png" would have an extension of
		 * ".png" since "png" is the extension (without the '.'
		 * character) after the "x-" string.
		 */
		strptr = strchr(type_str, '/');
		if(strptr != NULL)
		{
		    strptr++;
		    if(strpfx(strptr, "x-"))
		    {
			strptr += strlen("x-");

			g_free(mt_ptr->value);
			mt_ptr->value = g_strdup_printf(".%s", strptr);
		    }
		}


		/* Append commands. */
#define APPEND_COMMAND	\
if(*strptr != '\0') \
{ \
 i = mt_ptr->total_commands; \
 mt_ptr->total_commands = i + 1; \
\
 mt_ptr->command_name = (gchar **)g_realloc( \
  mt_ptr->command_name, mt_ptr->total_commands * sizeof(gchar *) \
 ); \
 mt_ptr->command = (gchar **)g_realloc( \
  mt_ptr->command, mt_ptr->total_commands * sizeof(gchar *) \
 ); \
 if((mt_ptr->command_name == NULL) || (mt_ptr->command == NULL)) \
 { \
  mt_ptr->total_commands = 0; \
 } \
 else \
 { \
  gchar *strptr2 = strchr(strptr, '='); \
  if(strptr2 != NULL) \
  { \
   mt_ptr->command[i] = g_strdup(strptr2 + 1); \
\
   mt_ptr->command_name[i] = g_strdup(strptr); \
   strptr2 = strchr(mt_ptr->command_name[i], '='); \
   if(strptr2 != NULL) \
    *strptr2 = '\0'; \
  } \
  else \
  { \
   mt_ptr->command_name[i] = g_strdup(""); \
   mt_ptr->command[i] = g_strdup(strptr); \
  } \
 } \
}

		strptr = cmd_str1;
		APPEND_COMMAND

                strptr = cmd_str2;
                APPEND_COMMAND

#undef APPEND_COMMAND

		/* Call add emit. */
		if(add_emit != NULL)
		    add_emit(
			(edv_core_struct *)add_emit_core,
			mt_num,
			mt_ptr
		    );
	    }

#undef GET_NEXT_CHAR_LITERAL
	}

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                progress_client_data, file_size, file_size
            );

        /* Close MailCap file. */
        FClose(fp);
        fp = NULL;

#undef cmd_str_len
#undef type_str_len
}

/*
 *      Writes the given list of MIME Types to the file specified by
 *      filename in Media Types format.
 *
 *      If include_read_only is TRUE then MIME Types that are marked
 *      as read_only will also be written.
 */
void EDVMimeTypeListExportMailCap(
        const gchar *filename,
        edv_mimetype_struct **list, gint total,
	gbool include_read_only,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
        gint mt_num;
        edv_mimetype_struct *mt_ptr;
        gchar *parent_path;
        FILE *fp;


        if((list == NULL) || (filename == NULL))
            return;


        /* Get parent directory and create it as needed. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
        if(parent_path != NULL)
        {
            rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
            g_free(parent_path);
            parent_path = NULL;
        }

        /* Create or truncate MailCap file. */
        fp = FOpen(filename, "wb");
        if(fp == NULL)
            return;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, 0, (gulong)total
            );


        /* Write header. */
        fprintf(
            fp,
"# MailCap\n\
#\n\
# Generated by %s Version %s\n\
\n",
            PROG_NAME, PROG_VERSION
        );


        /* Iterate through all MIME Type structures. */
        for(mt_num = 0; mt_num < total; mt_num++)
        {
            mt_ptr = list[mt_num];

            /* Skip MIME Types that are marked read only, meaning they
             * should not be saved to file since they are created
             * internally or are loaded from a global configuration.
             */
            if(!include_read_only && mt_ptr->read_only)
                continue;

            /* MIME Type string must be defined. */
            if((mt_ptr->type != NULL) ?
                (*mt_ptr->type == '\0') : TRUE
            )
                continue;

            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
                    client_data, (gulong)mt_num, (gulong)total
                ))
                    break;
            }


            /* Begin writing MailCap line. */

	    /* First field is the MIME Type. */
	    fputs(mt_ptr->type, fp);

	    /* Begin writing commands. */
	    if(mt_ptr->total_commands > 0)
	    {
		gint n;

		for(n = 0; n < mt_ptr->total_commands; n++)
		{
		    const gchar *name = mt_ptr->command_name[n];
		    const gchar *cmd = mt_ptr->command[n];

		    if(cmd == NULL)
			continue;

		    /* If this command references another MIME Type,
		     * then find that other MIME Type.
		     */
		    if(*cmd != DIR_DELIMINATOR)
		    {
			edv_mimetype_struct *ref_mt_ptr = EDVMimeTypeMatchListByType(
			    list, total,
			    NULL, cmd, TRUE
			);
			if(ref_mt_ptr == NULL)
			    continue;

			if(ref_mt_ptr->total_commands < 1)
			    continue;

			cmd = ref_mt_ptr->command[0];
			if(cmd == NULL)
			    continue;
		    }

		    /* We now have the command, if the command's name is
		     * also available we should include that.
		     */
		    if(name != NULL)
			fprintf(fp, ";%s=%s", name, cmd);
		    else
			fprintf(fp, ";%s", cmd);

		}
	    }

            /* Write line deliminator (newline character). */
            fputc('\n', fp);
	}

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, (gulong)total, (gulong)total
            );

        /* Close MailCap file. */
        FClose(fp);
        fp = NULL;
}


/*
 *      Imports the MIME Types obtained from the Media Types format file
 *      specified by filename.
 *
 *      The insert_index specifies the MIME Type index in the given list
 *      of MIME Types at which to insert imported MIME Types at. If
 *      insert_index is -1 then all imported MIME Types will be appended.
 */
void EDVMimeTypeListImportMediaTypes(
        const gchar *filename,
        edv_mimetype_struct ***list, gint *total,
        gint insert_index,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer progress_client_data,
        void (*add_emit)(edv_core_struct *, gint, edv_mimetype_struct *),
        gpointer add_emit_core
)
{
        gbool c_literal;
        gint i, c;
        FILE *fp;
        gulong file_size;
        gchar *strptr;
#define type_str_len    1024
        gchar type_str[type_str_len];
#define ext_str_len     2048
        gchar ext_str[ext_str_len];

        gint mt_num = -1;
        edv_mimetype_struct *mt_ptr = NULL;

        struct stat stat_buf;


        if((list == NULL) || (total == NULL) || (filename == NULL))
            return;

        /* Open Media Types file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
            return;

        /* Get file statistics. */
        if(fstat(fileno(fp), &stat_buf))
            file_size = 0;
        else
            file_size = stat_buf.st_size;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                progress_client_data, 0, file_size
            );

        /* Begin parsing media type file, the media type file format consists
         * of newline terminated strings. Each `line' contains fields
         * deliminated by blank characters as follows:
	 *
	 * mime_type ext1 ext2 ext3...
         */

        while(1)
        {
#define GET_NEXT_CHAR_LITERAL   \
{ \
 c = fgetc(fp); \
 if(c == '\\') \
 { \
  c = fgetc(fp); \
  c_literal = TRUE; \
  if(ISCR(c)) \
  { \
   c = fgetc(fp); \
   c_literal = FALSE; \
  } \
 } \
 else \
 { \
  c_literal = FALSE; \
 } \
}
            /* Get first character of current `line'. */
            GET_NEXT_CHAR_LITERAL
            if(c == EOF)
                break;

            /* Seek past initial spaces. */
            while(ISBLANK(c))
                GET_NEXT_CHAR_LITERAL
            if(c == EOF)
                break;

            /* Empty line? */
            if(ISCR(c))
                continue;

            /* Comment? */
            if(!c_literal && (c == '#'))
            {
                /* Seek to start of next line. */
                while(!ISCR(c) && (c != EOF))
                    GET_NEXT_CHAR_LITERAL

                continue;
            }

            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
                     progress_client_data, (gulong)ftell(fp), file_size
                ))
                    break;
            }

            /* Begin fetching type string, this is always the first
             * field before the first blank deliminator character.
             */
            strptr = type_str;
            for(i = 0; i < type_str_len; i++)
            {
                if(!c_literal && ISBLANK(c))
                {
                    GET_NEXT_CHAR_LITERAL
                    break;
                }
                else if(ISCR(c) || (c == EOF))
                {
                    break;
                }
                else
                {
                    *strptr++ = (gchar)c;
                    GET_NEXT_CHAR_LITERAL
                }
            }
            /* Null terminate the string. */
            if(i < type_str_len)
                type_str[i] = '\0';
            else
                type_str[type_str_len - 1] = '\0';

            if(c == EOF)
                break;


            /* Begin fetching extensions up untill the next newline
	     * character.
	     */
            strptr = ext_str;
            for(i = 0; i < ext_str_len; i++)
            {
                if(ISCR(c) || (c == EOF))
                {
                    break;
                }
                else
                {
                    *strptr++ = (gchar)c;
                    GET_NEXT_CHAR_LITERAL
                }
            }
            /* Null terminate the string. */
            if(i < ext_str_len)
                ext_str[i] = '\0';
            else
                ext_str[ext_str_len - 1] = '\0';


            /* In case end of line has not been reached, seek untill the
             * next newline character is encountered.
             */
            while(!ISCR(c) && (c != EOF))
                GET_NEXT_CHAR_LITERAL


            /* Begin creating new imported MIME Type. */

            /* Reset contexts. */
            mt_num = -1;
            mt_ptr = NULL;

            /* Sanitize total. */
            if(*total < 0)
                *total = 0;

            /* Append? */
            if(insert_index < 0)
	    {
                /* Append. */

                /* Increase total and allocate more pointers. */
                mt_num = *total;
                *total = mt_num + 1;

                *list = (edv_mimetype_struct **)g_realloc(
                    *list,
                    (*total) * sizeof(edv_mimetype_struct *)
                );
                if(*list == NULL)
                {
                    *total = 0;
                    break;
                }

                /* Allocate the new MIME Type structure and append it at
                 * the append index.
                 */
                (*list)[mt_num] = mt_ptr = EDVMimeTypeNew(
                    EDV_MIMETYPE_CLASS_FORMAT,
                    NULL,               /* Value. */
                    type_str,
                    NULL                /* Description. */
                );
            }
            else
            {
                /* Insert. */

                /* Increase total and allocate more pointers. */
                i = *total;
                *total = i + 1;

                *list = (edv_mimetype_struct **)g_realloc(
                    *list,
                    (*total) * sizeof(edv_mimetype_struct *)
                );
                if(*list == NULL)
                {
                    *total = 0;
                    break;
                }

                /* Get insert index and make sure it is in bounds. */
                mt_num = insert_index;
                if(mt_num >= *total)
                    mt_num = *total - 1;

                /* Shift pointers. */
                for(i = *total - 1; i > mt_num; i--)
                    (*list)[i] = (*list)[i - 1];

                /* Allocate the new MIME Type structure and insert it at
                 * the insert index.
                 */
                (*list)[mt_num] = mt_ptr = EDVMimeTypeNew(
                    EDV_MIMETYPE_CLASS_FORMAT,
                    NULL,               /* Value. */
                    type_str,
                    NULL                /* Description. */
                );

                /* Increment insert index for next MIME Type. */
                insert_index++;
            }

            /* New MIME Type added successfully? */
            if((mt_ptr != NULL) && (mt_num >= 0) && (mt_num < *total))
            {
		gchar **strv;
		gint strc = 0;


                /* Explode extension string. */
		if(*ext_str != '\0')
		    strv = strexp(ext_str, &strc);
		else
		    strv = NULL;
		if(strv != NULL)
		{
		    gint n;

		    /* Deallocate previous value (if any). */
		    g_free(mt_ptr->value);
		    mt_ptr->value = NULL;

		    /* Iterate through exploded strings, converting them
		     * to a sequence of properly formatted extensions in
		     * the MIME Type structure's value string.
		     */
		    for(n = 0; n < strc; n++)
		    {
			if(strv[n] != NULL)
			{
			    /* Append this exploded string to the value
			     * as an extension with a '.' prefix. Add a
			     * ' ' infront if this is not the first
			     * exploded string.
			     */
			    mt_ptr->value = strcatalloc(
				mt_ptr->value,
				(n > 0) ? " ." : "."
			    );
			    mt_ptr->value = strcatalloc(
                                mt_ptr->value,
				strv[n]
			    );
			}

			/* Deallocate this exploded string. */
			g_free(strv[n]);
			strv[n] = NULL;
                    }

		    /* Deallocate pointer array, each string has already
		     * been deallocated above.
		     */
		    g_free(strv);
		    strv = NULL;
                }


                /* Call add emit. */
                if(add_emit != NULL)
                    add_emit(
                        (edv_core_struct *)add_emit_core,
                        mt_num,
                        mt_ptr
                    );
            }

#undef GET_NEXT_CHAR_LITERAL
        }

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                progress_client_data, file_size, file_size
            );

        /* Close Media Types file. */
        FClose(fp);
        fp = NULL;

#undef ext_str_len
#undef type_str_len
}

/*
 *	Writes the given list of MIME Types to the file specified by
 *	filename in Media Types format.
 *
 *	If include_read_only is TRUE then MIME Types that are marked
 *	as read_only will also be written.
 */
void EDVMimeTypeListExportMediaTypes(
        const gchar *filename,
        edv_mimetype_struct **list, gint total,
	gbool include_read_only,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
        gint mt_num;
        edv_mimetype_struct *mt_ptr;
        gchar *parent_path;
        FILE *fp;


        if((list == NULL) || (filename == NULL))
            return;


        /* Get parent directory and create it as needed. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
        if(parent_path != NULL)
        {
            rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
            g_free(parent_path);
            parent_path = NULL;
        }

        /* Create or truncate Media Types file. */
        fp = FOpen(filename, "wb");
        if(fp == NULL)
            return;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, 0, (gulong)total
            );


	/* Write header. */
	fprintf(
	    fp,
"# Media Types\n\
#\n\
# Generated by %s Version %s\n\
\n",
	    PROG_NAME, PROG_VERSION
	);


        /* Iterate through all MIME Type structures. */
        for(mt_num = 0; mt_num < total; mt_num++)
        {
            mt_ptr = list[mt_num];

            /* Skip MIME Types that are marked read only, meaning they
             * should not be saved to file since they are created
             * internally or are loaded from a global configuration.
             */
            if(!include_read_only && mt_ptr->read_only)
                continue;

	    /* MIME Type string must be defined. */
            if((mt_ptr->type != NULL) ?
		(*mt_ptr->type == '\0') : TRUE
	    )
		continue;

            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
                    client_data, (gulong)mt_num, (gulong)total
                ))
                    break;
            }


            /* Begin writing Media Type line. */

	    /* First value is the type value. */
            fputs(mt_ptr->type, fp);

	    /* If this MIME Type's class is set to file format then write
	     * subsequent values are extensions (without the '.' prefix).
	     */
	    if(mt_ptr->mt_class == EDV_MIMETYPE_CLASS_FORMAT)
	    {
		gchar **strv;
		gint n, strc;


		/* Explode value string which should contain a space separated
		 * list of extensions.
		 */
		strv = strexp(mt_ptr->value, &strc);
		if(strv != NULL)
		{
		    /* Iterate throughe exploded extension strings. */
		    for(n = 0; n < strc; n++)
		    {
			const gchar *cstrptr = strv[n];
			if(cstrptr == NULL)
			    continue;

			/* Write deliminator, a tab character if it is the
			 * first extension or space if it is a subsequent
			 * extension.
			 */
 			fputc((n > 0) ? ' ' : '\t', fp);

			/* Write extension, seek past the first '.' deliminator
			 * so that it is not written.
			 */
			while(*cstrptr == '.')
			    cstrptr++;
			fputs(cstrptr, fp);

			/* Deallocate exploded string. */
			g_free(strv[n]);
			strv[n] = NULL;
		    }

                    /* Deallocate pointer array, each string has already
                     * been deallocated above.
                     */
		    g_free(strv);
		    strv = NULL;
		    strc = 0;
		}
	    }

	    /* Write line deliminator (newline character). */
	    fputc('\n', fp);
	}

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, (gulong)total, (gulong)total
            );

        /* Close Media Types file. */
        FClose(fp);
        fp = NULL;
}


/*
 *	Appends the MIME Types found in the specified file to the given
 *	list.
 *
 *      The insert_index specifies the MIME Type index in the given list
 *      of MIME Types at which to insert imported MIME Types at. If
 *      insert_index is -1 then all imported MIME Types will be appended.
 *
 *	If mark_all_loaded_read_only is TRUE then all MIME Types loaded
 *	from this file will be marked read_only.
 */
void EDVMimeTypeListLoadFromFile(
	const gchar *filename,
        edv_mimetype_struct ***list, gint *total,
	gint insert_index,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer progress_client_data,
        void (*add_emit)(edv_core_struct *, gint, edv_mimetype_struct *),
        gpointer add_emit_core,
        gbool mark_all_loaded_read_only
)
{
	gchar *parm_str;
	FILE *fp;
	gulong file_size;
	gint mt_num = -1;
	edv_mimetype_struct *mt_ptr = NULL;

	struct stat stat_buf;


	if((list == NULL) || (total == NULL) || (filename == NULL))
	    return;

        /* Open MIME Types file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
            return;

	/* Get file statistics. */
	if(fstat(fileno(fp), &stat_buf))
	    file_size = 0;
	else
	    file_size = stat_buf.st_size;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                progress_client_data, 0, file_size
            );

        /* Begin file. */
	parm_str = NULL;
        while(1)
        {
            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
		    progress_client_data, (gulong)ftell(fp), file_size
		))
                    break;
            }

	    /* Read next parameter. */
	    parm_str = FSeekNextParm(
		fp, parm_str,
		EDV_CFG_COMMENT_CHAR,
		EDV_CFG_DELIMINATOR_CHAR
	    );
	    if(parm_str == NULL)
		break;

	    /* Begin handling by parameter. */

	    /* Start of a MIME Type block? */
	    if(!strcasecmp(parm_str, "BeginMIMEType"))
	    {
		gint value[1];
		gint mt_class;


		/* Get class code. */
		FGetValuesI(fp, value, 1);
		mt_class = value[0];

		/* Append? */
		if(insert_index < 0)
		{
		    /* Append a new MIME Type structure. */
		    mt_num = *total;
		    *total = mt_num + 1;
		    *list = (edv_mimetype_struct **)g_realloc(
			*list, (*total) * sizeof(edv_mimetype_struct *)
		    );
		    if(*list == NULL)
		    {
			*total = 0;
			mt_num = -1;
			mt_ptr = NULL;
		    }
		    else
		    {
			(*list)[mt_num] = mt_ptr = EDVMimeTypeNew(
			    mt_class, NULL, NULL, NULL
			);
			if(mt_ptr == NULL)
			{
			    mt_num = -1;
			}
		    }
		}
		else
		{
		    gint i;

                    /* Increase total and allocate more pointers. */
                    i = *total;
                    *total = i + 1;

                    *list = (edv_mimetype_struct **)g_realloc(
                        *list,
                        (*total) * sizeof(edv_mimetype_struct *)
                    );
                    if(*list == NULL)
                    {
                        *total = 0;
                        mt_num = -1;
                        mt_ptr = NULL;
                    }
		    else
		    {
			/* Get insert index and make sure it is in bounds.*/
			mt_num = insert_index;
			if(mt_num >= *total)
			    mt_num = *total - 1;

			/* Shift pointers. */
			for(i = *total - 1; i > mt_num; i--)
			    (*list)[i] = (*list)[i - 1];

			(*list)[mt_num] = mt_ptr = EDVMimeTypeNew(
			    mt_class, NULL, NULL, NULL
			);
                        if(mt_ptr == NULL)
                        {
                            mt_num = -1;
                        }
                    }

                    /* Increment insert index for next MIME Type. */
                    insert_index++;
		}

		if(mt_ptr != NULL)
		{
		    /* Mark all loaded MIME Types as read only? */
		    if(mark_all_loaded_read_only)
			mt_ptr->read_only = TRUE;
		}

	    }

            /* MIME Type type? */
            else if(!strcasecmp(parm_str, "Type") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *type = FGetString(fp);

                g_free(mt_ptr->type);
                mt_ptr->type = (type != NULL) ?
                    g_strdup(type) : NULL;

                g_free(type);
                type = NULL;
            }
            /* MIME Type value? */
            else if(!strcasecmp(parm_str, "Value") &&
                    (mt_ptr != NULL)
	    )
            {
                gchar *value = FGetString(fp);

		g_free(mt_ptr->value);
		mt_ptr->value = (value != NULL) ?
		    g_strdup(value) : NULL;

		g_free(value);
		value = NULL;
	    }
            /* MIME Type verbose description? */
            else if(!strcasecmp(parm_str, "Description") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *description = FGetString(fp);

                g_free(mt_ptr->description);
                mt_ptr->description = (description != NULL) ?
                    g_strdup(description) : NULL;

                g_free(description);
                description = NULL;
            }

            /* MIME Type small icon standard? */
            else if(!strcasecmp(parm_str, "IconSmallStandard") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
		gchar **storage = &mt_ptr->small_icon_file[
		    EDV_MIMETYPE_ICON_STATE_STANDARD
		];

		g_free(*storage);
		*storage = (path != NULL) ? g_strdup(path) : NULL;

		g_free(path);
                path = NULL;
            }
            /* MIME Type small icon selected? */
            else if(!strcasecmp(parm_str, "IconSmallSelected") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->small_icon_file[
                    EDV_MIMETYPE_ICON_STATE_SELECTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }
            /* MIME Type small icon extended? */
            else if(!strcasecmp(parm_str, "IconSmallExtended") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->small_icon_file[
                    EDV_MIMETYPE_ICON_STATE_EXTENDED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }

            /* MIME Type medium icon standard? */
            else if(!strcasecmp(parm_str, "IconMediumStandard") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->medium_icon_file[
                    EDV_MIMETYPE_ICON_STATE_STANDARD
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }
            /* MIME Type medium icon selected? */
            else if(!strcasecmp(parm_str, "IconMediumSelected") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->medium_icon_file[
                    EDV_MIMETYPE_ICON_STATE_SELECTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }
            /* MIME Type medium icon extended? */
            else if(!strcasecmp(parm_str, "IconMediumExtended") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->medium_icon_file[
                    EDV_MIMETYPE_ICON_STATE_EXTENDED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }

            /* MIME Type large icon standard? */
            else if(!strcasecmp(parm_str, "IconLargeStandard") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->large_icon_file[
                    EDV_MIMETYPE_ICON_STATE_STANDARD
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }
            /* MIME Type large icon selected? */
            else if(!strcasecmp(parm_str, "IconLargeSelected") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->large_icon_file[
                    EDV_MIMETYPE_ICON_STATE_SELECTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }
            /* MIME Type large icon extended? */
            else if(!strcasecmp(parm_str, "IconLargeExtended") &&
                    (mt_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &mt_ptr->large_icon_file[
                    EDV_MIMETYPE_ICON_STATE_EXTENDED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
                path = NULL;
            }

            /* MIME Type handler. */
            else if(!strcasecmp(parm_str, "Handler"))
            {
                gchar *s = FGetString(fp);

		if(s != NULL)
		{
		    if(!strcasecmp(s, "Archiver"))
			mt_ptr->handler = EDV_MIMETYPE_HANDLER_EDV_ARCHIVER;
                    else if(!strcasecmp(s, "ImageBrowser"))
                        mt_ptr->handler = EDV_MIMETYPE_HANDLER_EDV_IMAGE_BROWSER;
                    else if(!strcasecmp(s, "RecycleBin"))
                        mt_ptr->handler = EDV_MIMETYPE_HANDLER_EDV_RECYCLE_BIN;
                    else
                        mt_ptr->handler = EDV_MIMETYPE_HANDLER_COMMAND;

		    g_free(s);
		}
	    }

            /* MIME Type command? */
            else if(!strcasecmp(parm_str, "Command") &&
                    (mt_ptr != NULL)
            )
            {
		gint n;
                gchar *command = FGetString(fp);


		/* Allocate more pointers for this new command. */
		n = mt_ptr->total_commands;
		mt_ptr->total_commands = n + 1;

		mt_ptr->command = (gchar **)g_realloc(
		    mt_ptr->command,
		    mt_ptr->total_commands * sizeof(gchar *)
		);
                mt_ptr->command_name = (gchar **)g_realloc(
                    mt_ptr->command_name,
                    mt_ptr->total_commands * sizeof(gchar *)
                );
		if((mt_ptr->command == NULL) ||
                   (mt_ptr->command_name == NULL)
		)
		{
		    g_free(mt_ptr->command);
		    mt_ptr->command = NULL;
		    g_free(mt_ptr->command_name);
		    mt_ptr->command_name = NULL;
		    mt_ptr->total_commands = 0;
		}
		else
		{
		    gchar *strptr;

                    mt_ptr->command[n] = NULL;
		    mt_ptr->command_name[n] = NULL;

		    if(command != NULL)
		    {
			mt_ptr->command_name[n] = g_strdup(command);
			strptr = strchr(
			    mt_ptr->command_name[n],
			    EDV_CFG_DELIMINATOR_CHAR
			);
			if(strptr != NULL)
			    *strptr = '\0';

			strptr = strchr(command, EDV_CFG_DELIMINATOR_CHAR);
			if(strptr == NULL)
			    strptr = command;
			else
			    strptr++;
			mt_ptr->command[n] = g_strdup(strptr);
		    }
		}

		g_free(command);
            }


	    /* End of a MIME Type block? */
	    else if(!strcasecmp(parm_str, "EndMIMEType"))
            {
		FSeekNextLine(fp);

                /* Call add emit. */
                if(add_emit != NULL)
                    add_emit(
                        (edv_core_struct *)add_emit_core,
                        mt_num,
                        mt_ptr
                    );

		/* Reset contexts. */
		mt_num = -1;
		mt_ptr = NULL;
	    }
	    /* All else unsupported parameter or wrong context. */
	    else
	    {
		FSeekNextLine(fp);
	    }
	}

	/* Deallocate parameter just in case. */
	g_free(parm_str);
	parm_str = NULL;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                progress_client_data, file_size, file_size
            );

	/* Close MIME Types file. */
	FClose(fp);
	fp = NULL;
}

/*
 *	Writes the given list of MIME Types to the file specified by
 *	filename.
 *
 *      If include_read_only is TRUE then MIME Types that are marked
 *      as read_only will also be written.
 */
void EDVMimeTypeListSaveToFile(
        const gchar *filename,
        edv_mimetype_struct **list, gint total,
	gbool include_read_only,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
	gint mt_num;
	edv_mimetype_struct *mt_ptr;
	const gchar *cstrptr;
	gchar *parent_path;
	FILE *fp;
	const gchar *path;


        if((list == NULL) || (filename == NULL))
            return;


        /* Get parent directory and create it as needed. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
        if(parent_path != NULL)
        {
            rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
            g_free(parent_path);
            parent_path = NULL;
        }

        /* Create or truncate MIME Types file. */
        fp = FOpen(filename, "wb");
        if(fp == NULL)
            return;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, 0, (gulong)total
            );

	/* Iterate through all MIME Type structures. */
	for(mt_num = 0; mt_num < total; mt_num++)
	{
	    mt_ptr = list[mt_num];
	    if(mt_ptr == NULL)
		continue;

	    /* Skip MIME Types that are marked read only, meaning they
	     * should not be saved to file since they are created
	     * internally or are loaded from a global configuration.
	     */
	    if(!include_read_only && mt_ptr->read_only)
		continue;

            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
                    client_data, (gulong)mt_num, (gulong)total
                ))
                    break;
            }


	    /* Begin writing MIME Type block. */
	    fprintf(
		fp,
		"BeginMIMEType = %i\n",
		mt_ptr->mt_class
	    );

            /* Type list. */
            if(mt_ptr->type != NULL)
                fprintf(
                    fp,
                    "    Type = %s\n",
                    mt_ptr->type
                );

	    /* Value. */
	    if(mt_ptr->value != NULL)
                fprintf(
                    fp,
                    "    Value = %s\n",
                    mt_ptr->value
                );

            /* Verbose description. */
            if(mt_ptr->description != NULL)
                fprintf(
                    fp,
                    "    Description = %s\n",
                    mt_ptr->description
                );

            /* Icon small standard file. */
	    path = mt_ptr->small_icon_file[EDV_MIMETYPE_ICON_STATE_STANDARD];
	    if(path != NULL)
                fprintf(
                    fp,
                    "    IconSmallStandard = %s\n",
		    path
                );
            /* Icon small selected file. */
            path = mt_ptr->small_icon_file[EDV_MIMETYPE_ICON_STATE_SELECTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconSmallSelected = %s\n",
                    path
                );
            /* Icon small extended file. */
            path = mt_ptr->small_icon_file[EDV_MIMETYPE_ICON_STATE_EXTENDED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconSmallExtended = %s\n",
                    path
                );

            /* Icon medium standard file. */
            path = mt_ptr->medium_icon_file[EDV_MIMETYPE_ICON_STATE_STANDARD];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconMediumStandard = %s\n",
                    path
                );
            /* Icon medium selected file. */
            path = mt_ptr->medium_icon_file[EDV_MIMETYPE_ICON_STATE_SELECTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconMediumSelected = %s\n",
                    path
                );
            /* Icon medium extended file. */
            path = mt_ptr->medium_icon_file[EDV_MIMETYPE_ICON_STATE_EXTENDED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconMediumExtended = %s\n",
                    path
                );

            /* Icon large standard file. */
            path = mt_ptr->large_icon_file[EDV_MIMETYPE_ICON_STATE_STANDARD];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconLargeStandard = %s\n",
                    path
                );
            /* Icon large selected file. */
            path = mt_ptr->large_icon_file[EDV_MIMETYPE_ICON_STATE_SELECTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconLargeSelected = %s\n",
                    path
                );
            /* Icon large extended file. */
            path = mt_ptr->large_icon_file[EDV_MIMETYPE_ICON_STATE_EXTENDED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconLargeExtended = %s\n",
                    path
                );

	    /* Handler. */
	    cstrptr = NULL;
	    switch(mt_ptr->handler)
	    {
	      case EDV_MIMETYPE_HANDLER_EDV_ARCHIVER:
		cstrptr = "Archiver";
		break;
              case EDV_MIMETYPE_HANDLER_EDV_IMAGE_BROWSER:
                cstrptr = "ImageBrowser";
                break;
              case EDV_MIMETYPE_HANDLER_EDV_RECYCLE_BIN:
                cstrptr = "RecycleBin";
                break;
              default:	/* EDV_MIMETYPE_HANDLER_COMMAND */
                cstrptr = "Command";
                break;
	    }
	    fprintf(
		fp,
		"    Handler = %s\n",
		cstrptr
	    );

	    /* Commands. */
	    if(mt_ptr->total_commands > 0)
	    {
		gint n;

		for(n = 0; n < mt_ptr->total_commands; n++)
		{
		    fprintf(
                        fp,
                        "    Command = %s=%s\n",
                        mt_ptr->command_name[n],
			mt_ptr->command[n]
                    );
		}
	    }

	    /* End MIME Type block. */
            fprintf(
                fp,
                "EndMIMEType\n"
            );
	}

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, (gulong)total, (gulong)total
            );

        /* Close MIME Types file. */
        FClose(fp);
	fp = NULL;
}
