#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "edvtypes.h"
#include "edvmimetypes.h"


edv_mimetype_struct *EDVMimeTypeMatchListByType(
	edv_mimetype_struct **list, gint total,
	gint *n,
	const gchar *type, gbool case_sensitive
);

edv_mimetype_struct *EDVMimeTypeNew(
	gint mt_class,			/* One of EDV_MIMETYPE_CLASS_*. */
        const gchar *value,
        const gchar *type,              /* MIME Type name. */
        const gchar *description	/* Short verbose description. */
);

static void EDVMimeTypeLoadIconsNexus(
        GdkPixmap **pixmap, GdkBitmap **mask,
        guint8 ***data, const gchar **file
);
void EDVMimeTypeLoadSmallIconsData(
        edv_mimetype_struct *m, guint8 ***data
);
void EDVMimeTypeLoadMediumIconsData(
        edv_mimetype_struct *m, guint8 ***data
);
void EDVMimeTypeLoadLargeIconsData(
        edv_mimetype_struct *m, guint8 ***data
);

void EDVMimeTypeRealize(
        edv_mimetype_struct *m, gbool force_rerealize
);

void EDVMimeTypeDelete(edv_mimetype_struct *m);



/*
 *	Matches a mimetype structure in the given list that matches the
 *	given type.
 */
edv_mimetype_struct *EDVMimeTypeMatchListByType(
        edv_mimetype_struct **list, gint total,
	gint *n,
        const gchar *type, gbool case_sensitive
)
{
	gint i;
	edv_mimetype_struct *m;


	if(n != NULL)
	    *n = -1;

	if(type == NULL)
	    return(NULL);

	/* Iterate through mimetypes list. */
	for(i = 0; i < total; i++)
	{
	    m = list[i];
	    if(m == NULL)
		continue;

	    if(m->type == NULL)
		continue;

	    if(case_sensitive)
	    {
		if(!strcmp(m->type, type))
		{
                    if(n != NULL)
                       *n = i;
		    return(m);
		}
	    }
	    else
	    {
		if(!strcasecmp(m->type, type))
                {
                    if(n != NULL)
                       *n = i;
		    return(m);
                }
	    }
	}

	return(NULL);
}


/*
 *	Allocates a new edv_mimetype_struct.
 */
edv_mimetype_struct *EDVMimeTypeNew(
	gint mt_class,			/* One of EDV_MIMETYPE_CLASS_*. */
	const gchar *value,
	const gchar *type,		/* MIME Type name. */
	const gchar *description	/* Short verbose description. */
)
{
	edv_mimetype_struct *m = (edv_mimetype_struct *)g_malloc0(
	    sizeof(edv_mimetype_struct)
	);
	if(m == NULL)
	    return(m);

	m->mt_class = mt_class;

	m->value = (value != NULL) ?
	    g_strdup(value) : NULL;
	m->type = (type != NULL) ?
	    g_strdup(type) : NULL;
	m->description = (description != NULL) ?
	    g_strdup(description) : NULL;

	m->realized = FALSE;

	m->read_only = FALSE;

	memset(
	    m->small_pixmap, 0x00,
	    EDV_MIMETYPE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
	);
        memset(
            m->small_mask, 0x00,
            EDV_MIMETYPE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
        );

	memset(
            m->medium_pixmap, 0x00,
            EDV_MIMETYPE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
        );
        memset(
            m->medium_mask, 0x00,
            EDV_MIMETYPE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
        );

	memset(
            m->large_pixmap, 0x00,
            EDV_MIMETYPE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
        );
        memset(
            m->large_mask, 0x00,
            EDV_MIMETYPE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
        );

	m->handler = EDV_MIMETYPE_HANDLER_COMMAND;

	m->command = NULL;
	m->command_name = NULL;
	m->total_commands = 0;

	return(m);
}

/*
 *      Loads the pixmap and mask pair from either xpm data or xpm file
 *      depending if data or file is NULL.
 *
 *      Passing both data and file as NULL will only unref the existing icons
 *      (if any) and not load new ones. So this function can also be used
 *      to just deallocate the icons.
 */
static void EDVMimeTypeLoadIconsNexus(
	GdkPixmap **pixmap, GdkBitmap **mask,
	guint8 ***data, const gchar **file
)
{
	gint i;
	GdkWindow *window = (GdkWindow *)GDK_ROOT_PARENT();
	GtkStyle *style = gtk_widget_get_default_style();


	/* Unload each given icon set. */
	for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
	{
	    if(pixmap[i] != NULL)
	    {
/*
GdkWindowPrivate *private = (GdkWindowPrivate *)pixmap[i];
printf("Pixmap refcount = %i\n", private->ref_count - 1);
 */
		gdk_pixmap_unref(pixmap[i]);
		pixmap[i] = NULL;
	    }
            if(mask[i] != NULL)
            {
                gdk_bitmap_unref(mask[i]);
                mask[i] = NULL;
            }
	}

	/* Check if xpm data is given, if it is then the icon image data
	 * will be loaded.
	 */
	if((data != NULL) && (window != NULL) && (style != NULL))
	{
	    guint8 **data_ptr;
	    GdkPixmap *pixmap_ptr;
	    GdkBitmap *mask_ptr;


	    /* Load each icon set from data. */
	    for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
	    {
		data_ptr = data[i];
		if(data_ptr == NULL)
		    continue;

		mask_ptr = NULL;
		pixmap_ptr = gdk_pixmap_create_from_xpm_d(
		    window, &mask_ptr,
		    &style->bg[GTK_STATE_NORMAL],
		    (gchar **)data_ptr
		);

		pixmap[i] = pixmap_ptr;
		mask[i] = mask_ptr;
	    }
	}
        /* Check if xpm file is given, if it is then the icon image data
         * will be loaded.
         */
        else if((file != NULL) && (window != NULL) && (style != NULL))
        {
            const gchar *path_ptr;
            GdkPixmap *pixmap_ptr;
            GdkBitmap *mask_ptr;


            /* Load each icon set from file. */
            for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
            {
                path_ptr = file[i];
                if(path_ptr == NULL)
                    continue;

		mask_ptr = NULL;
                pixmap_ptr = gdk_pixmap_create_from_xpm(
                    window, &mask_ptr,
                    &style->bg[GTK_STATE_NORMAL],
                    path_ptr
                );

                pixmap[i] = pixmap_ptr;
                mask[i] = mask_ptr;
            }
	}

}

/*
 *      Loads the small icons from xpm data. The given data must be an
 *      array of 3 guint8 ** pointers.
 *
 *      Passing data as NULL will only unref the existing icons (if any)
 *      and not load new ones.
 */
void EDVMimeTypeLoadSmallIconsData(
        edv_mimetype_struct *m, guint8 ***data
)
{
        if(m == NULL)
            return;

	EDVMimeTypeLoadIconsNexus(
	    m->small_pixmap, m->small_mask,
	    data, NULL
	);
}

/*
 *      Loads the small icons from xpm data. The given data must be an
 *      array of 3 guint8 ** pointers.
 *
 *      Passing data as NULL will only unref the existing icons (if any)
 *      and not load new ones.
 */
void EDVMimeTypeLoadMediumIconsData(
        edv_mimetype_struct *m, guint8 ***data
)
{
        if(m == NULL)
            return;

        EDVMimeTypeLoadIconsNexus(
            m->medium_pixmap, m->medium_mask,
            data, NULL
        );
}

/*
 *      Loads the small icons from xpm data. The given data must be an
 *      array of 3 guint8 ** pointers.
 *
 *      Passing data as NULL will only unref the existing icons (if any)
 *      and not load new ones.
 */
void EDVMimeTypeLoadLargeIconsData(
        edv_mimetype_struct *m, guint8 ***data
)
{
        if(m == NULL)
            return;

        EDVMimeTypeLoadIconsNexus(
            m->large_pixmap, m->large_mask,
            data, NULL
        );
}


/*
 *	Realizes the given MIME Type structure, loading icons and
 *	related resources.
 *
 *	If the MIME Type is already realized (member realized is TRUE)
 *	then nothing will be done. However if force_rerealize is TRUE
 *	then the above check will be ignored.
 */
void EDVMimeTypeRealize(
        edv_mimetype_struct *m, gbool force_rerealize
)
{
	const gchar **path;


        if(m == NULL)
            return;


	/* Not forcing a re-realize? */
	if(!force_rerealize)
	{
	    /* MIME Type already realized? */
	    if(m->realized)
		return;
	}

	/* Begin realizing this MIME Type structure, loading icons
	 * from the specified file names.
	 */

        /* Get path array for small icon file. */
        path = (const gchar **)m->small_icon_file;
        if(path != NULL)
            EDVMimeTypeLoadIconsNexus(
                m->small_pixmap, m->small_mask,
                NULL, path
            );

        /* Get path array for medium icon file. */
        path = (const gchar **)m->medium_icon_file;
        if(path != NULL)
            EDVMimeTypeLoadIconsNexus(
                m->medium_pixmap, m->medium_mask,
                NULL, path
            );

	/* Get path for large icon file. */
	path = (const gchar **)m->large_icon_file;
	if(path != NULL)
	    EDVMimeTypeLoadIconsNexus(
		m->large_pixmap, m->large_mask,
		NULL, path
	    );

	/* Mark this MIME Type structure as being realized. */
	m->realized = TRUE;
}

/*
 *	Deallocates the given edv_mimetype_struct and all its resources.
 */
void EDVMimeTypeDelete(edv_mimetype_struct *m)
{
	gint i;


	if(m == NULL)
	    return;

	/* Call the icon loaders with NULL as the argument, this will only
	 * unload the icons.
	 */
	EDVMimeTypeLoadSmallIconsData(m, NULL);
        EDVMimeTypeLoadMediumIconsData(m, NULL);
        EDVMimeTypeLoadLargeIconsData(m, NULL);

	for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
	{
	    g_free(m->small_icon_file[i]);
	    m->small_icon_file[i] = NULL;
	}
        for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
	{
            g_free(m->medium_icon_file[i]);
	    m->medium_icon_file[i] = NULL;
	}
        for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
	{
            g_free(m->large_icon_file[i]);
	    m->large_icon_file[i] = NULL;
	}


	for(i = 0; i < m->total_commands; i++)
	{
	    g_free(m->command[i]);
            g_free(m->command_name[i]);
	}
	g_free(m->command);
	m->command = NULL;
	g_free(m->command_name);
	m->command_name = NULL;
	m->total_commands = 0;


	g_free(m->value);
	m->value = NULL;

	g_free(m->type);
	m->type = NULL;

	g_free(m->description);
	m->description = NULL;

	g_free(m);
}

