#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef __SOLARIS__
# include <sys/mnttab.h>
# include <sys/vfstab.h>
#else
# include <mntent.h>
#endif
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edvtypes.h"
#include "edvdevices.h"
#include "edvdevicesfio.h"

#include "config.h"


void EDVDevicesListLoadFromSystem(
        edv_device_struct ***list, gint *total,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);

void EDVDevicesListUpdateMountStatesFromSystem(
	edv_device_struct **list, gint total
);

void EDVDevicesListLoadFromFile(
        const gchar *filename,
        edv_device_struct ***list, gint *total,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);
void EDVDeviceListSaveToFile(
        const gchar *filename,
        edv_device_struct **list, gint total,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
);



/*
 *	Loads devices found in the system (/etc/fstab) list of devices
 *	and appends them to the list.
 */
void EDVDevicesListLoadFromSystem(
        edv_device_struct ***list, gint *total,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
        FILE *fp;
        gulong file_size;
        gint dev_num = -1;
        edv_device_struct *dev_ptr = NULL;
#ifdef __SOLARIS__
	struct vfstab *vfs_ptr = NULL;
	int mtback;
#else
	struct mntent *mt_ptr;
#endif
	struct stat stat_buf;


        if((list == NULL) || (total == NULL))
            return;


        /* Open system devices list file. */
#ifdef __SOLARIS__
	fp = FOpen("/etc/vfstab", "rb");
#else
        fp = setmntent("/etc/fstab", "rb");
#endif
        if(fp == NULL)
            return;

        /* Get file statistics. */
        if(fstat(fileno(fp), &stat_buf))
            file_size = 0;
        else
            file_size = stat_buf.st_size;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, 0, file_size
            );

	/* Begin reading system devices list file. */
#ifdef __SOLARIS__
	vfs_ptr = (struct vfstab *)g_malloc(sizeof(struct vfstab));
	mtback = getvfsent(fp, vfs_ptr);
	while(mtback != 0)
#else
	mt_ptr = getmntent(fp);
	while(mt_ptr != NULL)
#endif
	{
            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
                    client_data, (gulong)ftell(fp), file_size
                ))
                    break;
            }


	    /* Check if an existing device already exists in
	     * the list that matches the device_path.
	     */
	    dev_ptr = EDVDeviceMatchListByDevicePath(
		*list, *total,
		&dev_num, mt_ptr->mnt_fsname
	    );
	    if(dev_ptr == NULL)
	    {
		/* No existing device structure exists that matches the
		 * device path, so create a new one.
		 */
		dev_num = *total;
		*total = dev_num + 1;
		*list = (edv_device_struct **)g_realloc(
		    *list, (*total) * sizeof(edv_device_struct *)
		);
		if(*list == NULL)
		{
		    *total = 0;
		    dev_num = -1;
		    dev_ptr = NULL;
		    break;
		}
		else
		{
		    (*list)[dev_num] = dev_ptr = EDVDeviceNew(
#ifdef __SOLARIS__
			EDVDeviceGetFSFromString(vfs_ptr->vfs_fstype),
			NULL, vfs_ptr->vfs_special, vfs_ptr->vfs_mountp
#else
			EDVDeviceGetFSFromString(mt_ptr->mnt_type),
			NULL, mt_ptr->mnt_fsname, mt_ptr->mnt_dir
#endif
		    );
		    if(dev_ptr == NULL)
		    {
			dev_num = -1;
			break;
		    }

		    /* Set other default values for this new device. */

                    /* Default icons. */
                    if(EDV_DEVICE_TOTAL_ICON_STATES >= 1)
                    {
                        /* Small icon? */
                        if(dev_ptr->small_icon_file[0] == NULL)
                        {
                            dev_ptr->small_icon_file[0] = g_strdup(
                                EDV_DEF_DEVICE_ICON_SMALL_FILE
                            );
                        }
                        /* Medium icon? */
                        if(dev_ptr->medium_icon_file[0] == NULL)
                        {
                            dev_ptr->medium_icon_file[0] = g_strdup(
                                EDV_DEF_DEVICE_ICON_MEDIUM_FILE
                            );
                        }
                        /* Large icon? */
                        if(dev_ptr->large_icon_file[0] == NULL)
                        {
                            dev_ptr->large_icon_file[0] = g_strdup(
                                EDV_DEF_DEVICE_ICON_LARGE_FILE
                            );
                        }
                    }

                    /* Default fsck program. */
                    g_free(dev_ptr->command_check);
                    dev_ptr->command_check = g_strdup_printf(
                        "%s %s",
                        EDV_DEF_FSCK_FRONT_PROGRAM,
#ifdef __SOLARIS__
			vfs_ptr->vfs_special
#else
			mt_ptr->mnt_fsname
#endif
                    );

		    /* Default format program. */
		    g_free(dev_ptr->command_format);
		    dev_ptr->command_format = g_strdup_printf(
			"%s %s",
			EDV_DEF_FORMAT_FRONT_PROGRAM,
#ifdef __SOLARIS__
			vfs_ptr->vfs_special
#else
			mt_ptr->mnt_fsname
#endif
		    );

		}
	    }
	    else
	    {
		/* Existing device structure exists that matches the 
		 * device path, so update its values.
		 */

		/* File system type. */
#ifdef __SOLARIS__
		dev_ptr->fs_type = EDVDeviceGetFSFromString(vfs_ptr->vfs_fstype);
#else
		dev_ptr->fs_type = EDVDeviceGetFSFromString(mt_ptr->mnt_type);
#endif

		/* Mount path. */
#ifdef __SOLARIS__
		if(vfs_ptr->vfs_mountp != NULL)
#else
		if(mt_ptr->mnt_dir != NULL)
#endif
		{
		    g_free(dev_ptr->mount_path);
#ifdef __SOLARIS__
		    dev_ptr->mount_path = g_strdup(vfs_ptr->vfs_mountp);
#else
		    dev_ptr->mount_path = g_strdup(mt_ptr->mnt_dir);
#endif
		}

	    }

	    /* Read next mount entry. */
#ifdef __SOLARIS__
	    mtback = getmntent(fp, vfs_ptr);
#else
	    mt_ptr = getmntent(fp);
#endif
	}


	/* Close system devices list file. */
#ifdef __SOLARIS__
	FClose(fp);
	fp = NULL;
	vfs_ptr = NULL;
#else
	endmntent(fp);
	fp = NULL;
	mt_ptr = NULL;
#endif

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, file_size, file_size
            );
}


/*
 *	Updates all device structures in the given list by checking if
 *	they are actually mounted on the system.
 */
void EDVDevicesListUpdateMountStatesFromSystem(
        edv_device_struct **list, gint total
)
{
        FILE *fp;
        gint dev_num = -1;
        edv_device_struct *dev_ptr = NULL;
#ifdef __SOLARIS__
	struct mnttab *mt_ptr = NULL;
	int mtback;
#else
        struct mntent *mt_ptr;
#endif

        if((list == NULL) || (total <= 0))
            return;


        /* Reset mount states on all device structures. */
        for(dev_num = 0; dev_num < total; dev_num++)
        {
            dev_ptr = list[dev_num];
            if(dev_ptr == NULL)
                continue;

            dev_ptr->is_mounted = FALSE;
        }


        /* Open system mounted devices list file. */
#ifdef __SOLARIS__
	fp = FOpen("/etc/mnttab", "rb");
#else
        fp = setmntent("/etc/mtab", "rb");
#endif
        if(fp == NULL)
            return;

        /* Begin reading system devices list file. */
#ifdef __SOLARIS__
	mt_ptr = (struct mnttab *)g_malloc(sizeof(struct mnttab));
	mtback = getmntent(fp, mt_ptr);
	while(mtback != 0)
#else
        mt_ptr = getmntent(fp);
        while(mt_ptr != NULL)
#endif
        {
            /* Check if this device is in the list. */
            dev_ptr = EDVDeviceMatchListByDevicePath(
                list, total,
#ifdef __SOLARIS__
		&dev_num, mt_ptr->mnt_special
#else
                &dev_num, mt_ptr->mnt_fsname
#endif
            );
            if(dev_ptr != NULL)
            {
		/* Device path of entry from system matches one of the
		 * device structures, so mark the device structure as
		 * being mounteded.
		 */
		dev_ptr->is_mounted = TRUE;
	    }

            /* Read next mount entry. */
#ifdef __SOLARIS__
	    mtback = getmntent(fp, mt_ptr);
#else
            mt_ptr = getmntent(fp);
#endif
        }


        /* Close system mounted devices list file. */
#ifdef __SOLARIS__
	FClose(fp);
	fp = NULL;
	mt_ptr = NULL;
#else
        endmntent(fp);
        fp = NULL;
        mt_ptr = NULL;
#endif
}


/*
 *	Appends the devices found in the specified file to the given
 *	list, updates any existing devices in the list if their 
 *	device_path match.
 */
void EDVDevicesListLoadFromFile(
        const gchar *filename,
        edv_device_struct ***list, gint *total,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
	gchar *parm_str;
	FILE *fp;
	gulong file_size;
	gint dev_num = -1;
	edv_device_struct *dev_ptr = NULL;

	struct stat stat_buf;


	if((list == NULL) || (total == NULL) || (filename == NULL))
	    return;

        /* Open devices file. */
        fp = FOpen(filename, "rb");
        if(fp == NULL)
            return;

	/* Get file statistics. */
	if(fstat(fileno(fp), &stat_buf))
	    file_size = 0;
	else
	    file_size = stat_buf.st_size;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, 0, file_size
            );

        /* Begin file. */
	parm_str = NULL;
        while(1)
        {
            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
		    client_data, (gulong)ftell(fp), file_size
		))
                    break;
            }

	    /* Read next parameter. */
	    parm_str = FSeekNextParm(
		fp, parm_str,
		EDV_CFG_COMMENT_CHAR,
		EDV_CFG_DELIMINATOR_CHAR
	    );
	    if(parm_str == NULL)
		break;

	    /* Begin handling by parameter. */

	    /* Start of a device block? */
	    if(!strcasecmp(parm_str, "BeginDevice"))
	    {
                gchar *device_path = FGetString(fp);

		/* Check if an existing device already exists in
		 * the list that matches the device_path.
		 */
		dev_ptr = EDVDeviceMatchListByDevicePath(
		    *list, *total,
		    &dev_num, device_path
		);
		if(dev_ptr == NULL)
		{
		    /* No existing device structure exists that matches
		     * the given device_path, so create a new one.
		     */
		    dev_num = *total;
		    *total = dev_num + 1;
		    *list = (edv_device_struct **)g_realloc(
			*list, (*total) * sizeof(edv_device_struct *)
		    );
		    if(*list == NULL)
		    {
			*total = 0;
			dev_num = -1;
			dev_ptr = NULL;
		    }
		    else
		    {
			(*list)[dev_num] = dev_ptr = EDVDeviceNew(
			    EDV_FS_TYPE_EMPTY, NULL, device_path, NULL
			);
			if(dev_ptr == NULL)
			{
			    dev_num = -1;
			}
		    }
		}
	    }

            /* File system type? */
            else if(!strcasecmp(parm_str, "FSType") &&
                    (dev_ptr != NULL)
            )
            {
                gint value[1];
                FGetValuesI(fp, value, 1);
		dev_ptr->fs_type = value[0];
            }
            /* No unmount? */
            else if(!strcasecmp(parm_str, "NoUnmount") &&
                    (dev_ptr != NULL)
            )
            {
                gint value[1];
                FGetValuesI(fp, value, 1);
                dev_ptr->no_unmount = (gbool)value[0];
            }
            /* Always mount read only? */
            else if(!strcasecmp(parm_str, "ReadOnly") &&
                    (dev_ptr != NULL)
            )
            {
                gint value[1];
                FGetValuesI(fp, value, 1);
                dev_ptr->read_only = (gbool)value[0];
            }
            /* Unlisted? */
            else if(!strcasecmp(parm_str, "Unlisted") &&
                    (dev_ptr != NULL)
            )
            {
                gint value[1];
                FGetValuesI(fp, value, 1);
                dev_ptr->unlisted = (gbool)value[0];
            }
            /* Descriptive name? */
            else if(!strcasecmp(parm_str, "Name") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->name);
                dev_ptr->name = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }
            /* Device mount path? */
            else if(!strcasecmp(parm_str, "MountPath") &&
                    (dev_ptr != NULL)
	    )
            {
                gchar *value = FGetString(fp);

		g_free(dev_ptr->mount_path);
		dev_ptr->mount_path = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
	    }
            /* Device mount command? */
            else if(!strcasecmp(parm_str, "CommandMount") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->command_mount);
                dev_ptr->command_mount = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }
            /* Device unmount command? */
            else if(!strcasecmp(parm_str, "CommandUnmount") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->command_unmount);
                dev_ptr->command_unmount = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }
            /* Device eject command? */
            else if(!strcasecmp(parm_str, "CommandEject") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->command_eject);
                dev_ptr->command_eject = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }
            /* Device check command? */
            else if(!strcasecmp(parm_str, "CommandCheck") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->command_check);
                dev_ptr->command_check = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }
	    /* Device tools command? */
            else if(!strcasecmp(parm_str, "CommandTools") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->command_tools);
                dev_ptr->command_tools = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }
            /* Device format command? */
            else if(!strcasecmp(parm_str, "CommandFormat") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *value = FGetString(fp);

                g_free(dev_ptr->command_format);
                dev_ptr->command_format = (value != NULL) ?
		    g_strdup(value) : NULL;
                g_free(value);
            }

            /* Device small icon standard? */
            else if(!strcasecmp(parm_str, "IconSmallStandard") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
		gchar **storage = &dev_ptr->small_icon_file[
		    EDV_DEVICE_ICON_STATE_STANDARD
		];

		g_free(*storage);
		*storage = (path != NULL) ? g_strdup(path) : NULL;

		g_free(path);
            }
            /* Device small icon selected? */
            else if(!strcasecmp(parm_str, "IconSmallSelected") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->small_icon_file[
                    EDV_DEVICE_ICON_STATE_SELECTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }
            /* Device small icon unmounted? */
            else if(!strcasecmp(parm_str, "IconSmallUnmounted") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->small_icon_file[
                    EDV_DEVICE_ICON_STATE_UNMOUNTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }

            /* Device medium icon standard? */
            else if(!strcasecmp(parm_str, "IconMediumStandard") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->medium_icon_file[
                    EDV_DEVICE_ICON_STATE_STANDARD
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }
            /* Device medium icon selected? */
            else if(!strcasecmp(parm_str, "IconMediumSelected") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->medium_icon_file[
                    EDV_DEVICE_ICON_STATE_SELECTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }
            /* Device medium icon unmounted? */
            else if(!strcasecmp(parm_str, "IconMediumUnmounted") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->medium_icon_file[
                    EDV_DEVICE_ICON_STATE_UNMOUNTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }

            /* Device large icon standard? */
            else if(!strcasecmp(parm_str, "IconLargeStandard") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->large_icon_file[
                    EDV_DEVICE_ICON_STATE_STANDARD
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }
            /* Device large icon selected? */
            else if(!strcasecmp(parm_str, "IconLargeSelected") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->large_icon_file[
                    EDV_DEVICE_ICON_STATE_SELECTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }
            /* Device large icon unmounted? */
            else if(!strcasecmp(parm_str, "IconLargeUnmounted") &&
                    (dev_ptr != NULL)
            )
            {
                gchar *path = FGetString(fp);
                gchar **storage = &dev_ptr->large_icon_file[
                    EDV_DEVICE_ICON_STATE_UNMOUNTED
                ];

                g_free(*storage);
                *storage = (path != NULL) ? g_strdup(path) : NULL;

                g_free(path);
            }


	    /* End of a device block? */
	    else if(!strcasecmp(parm_str, "EndDevice"))
            {
		FSeekNextLine(fp);

		/* Reset contexts. */
		dev_num = -1;
		dev_ptr = NULL;
	    }
	    /* All else unsupported parameter or wrong context. */
	    else
	    {
		FSeekNextLine(fp);
	    }
	}

	/* Deallocate parameter just in case. */
	g_free(parm_str);
	parm_str = NULL;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, file_size, file_size
            );

	/* Close devices file. */
	FClose(fp);
	fp = NULL;
}

/*
 *	Saves the given list of devices to file.
 */
void EDVDeviceListSaveToFile(
        const gchar *filename,
        edv_device_struct **list, gint total,
        gint (*progress_cb)(gpointer, gulong, gulong),
        gpointer client_data
)
{
	gint dev_num;
	edv_device_struct *dev_ptr;
	gchar *parent_path;
	FILE *fp;
	const gchar *path;


        if((list == NULL) || (filename == NULL))
            return;


        /* Get parent directory and create it as needed. */
        parent_path = GetParentDir(filename);
        parent_path = (parent_path != NULL) ? g_strdup(parent_path) : NULL;
        if(parent_path != NULL)
        {
            rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
            g_free(parent_path);
            parent_path = NULL;
        }

        /* Create or truncate devices file. */
        fp = FOpen(filename, "wb");
        if(fp == NULL)
            return;

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, 0, (gulong)total
            );

	/* Iterate through all device structures. */
	for(dev_num = 0; dev_num < total; dev_num++)
	{
	    dev_ptr = list[dev_num];
	    if(dev_ptr == NULL)
		continue;

	    /* Skip devices that are marked internal, meaning they
	     * should not be saved to file.
	     */
	    if(dev_ptr->internal)
		continue;

            /* Call progress callback. */
            if(progress_cb != NULL)
            {
                if(progress_cb(
                    client_data, (gulong)dev_num, (gulong)total
                ))
                    break;
            }


	    /* Begin writing device block. */
	    fprintf(
		fp,
		"BeginDevice = %s\n",
		dev_ptr->device_path
	    );

            /* File system type. */
            fprintf(
                fp,
                "    FSType = %i\n",
                dev_ptr->fs_type
            );
            /* No unmount. */
            fprintf(
                fp,
                "    NoUnmount = %i\n",
                dev_ptr->no_unmount
            );
            /* Always mount read only. */
            fprintf(
                fp,
                "    ReadOnly = %i\n",
                dev_ptr->read_only
            );
            /* Unlisted. */
            fprintf(
                fp,
                "    Unlisted = %i\n",
                dev_ptr->unlisted
            );
            /* Descriptive name. */
            if(dev_ptr->name != NULL)
                fprintf(
                    fp,
                    "    Name = %s\n",
                    dev_ptr->name
                );
	    /* Mount path. */
	    if(dev_ptr->mount_path != NULL)
                fprintf(
                    fp,
                    "    MountPath = %s\n",
                    dev_ptr->mount_path
                );

            /* Mount command. */
            if(dev_ptr->command_mount != NULL)
                fprintf(
                    fp,
                    "    CommandMount = %s\n",
                    dev_ptr->command_mount
                );
            /* Unmount command. */
            if(dev_ptr->command_unmount != NULL)
                fprintf(
                    fp,
                    "    CommandUnmount = %s\n",
                    dev_ptr->command_unmount
                );
            /* Eject command. */
            if(dev_ptr->command_eject != NULL)
                fprintf(
                    fp,
                    "    CommandEject = %s\n",
                    dev_ptr->command_eject
                );
            /* Device check command. */
            if(dev_ptr->command_check != NULL)
                fprintf(
                    fp,
                    "    CommandCheck = %s\n",
                    dev_ptr->command_check
                );
            /* Device tools command. */
            if(dev_ptr->command_tools != NULL)
                fprintf(
                    fp,
                    "    CommandTools = %s\n",
                    dev_ptr->command_tools
                );
            /* Device format command. */
            if(dev_ptr->command_format != NULL)
                fprintf(
                    fp,
                    "    CommandFormat = %s\n",
                    dev_ptr->command_format
                );

            /* Icon small standard file. */
	    path = dev_ptr->small_icon_file[EDV_DEVICE_ICON_STATE_STANDARD];
	    if(path != NULL)
                fprintf(
                    fp,
                    "    IconSmallStandard = %s\n",
		    path
                );
            /* Icon small selected file. */
            path = dev_ptr->small_icon_file[EDV_DEVICE_ICON_STATE_SELECTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconSmallSelected = %s\n",
                    path
                );
            /* Icon small unmounted file. */
            path = dev_ptr->small_icon_file[EDV_DEVICE_ICON_STATE_UNMOUNTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconSmallUnmounted = %s\n",
                    path
                );

            /* Icon medium standard file. */
            path = dev_ptr->medium_icon_file[EDV_DEVICE_ICON_STATE_STANDARD];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconMediumStandard = %s\n",
                    path
                );
            /* Icon medium selected file. */
            path = dev_ptr->medium_icon_file[EDV_DEVICE_ICON_STATE_SELECTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconMediumSelected = %s\n",
                    path
                );
            /* Icon medium unmounted file. */
            path = dev_ptr->medium_icon_file[EDV_DEVICE_ICON_STATE_UNMOUNTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconMediumUnmounted = %s\n",
                    path
                );

            /* Icon large standard file. */
            path = dev_ptr->large_icon_file[EDV_DEVICE_ICON_STATE_STANDARD];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconLargeStandard = %s\n",
                    path
                );
            /* Icon large selected file. */
            path = dev_ptr->large_icon_file[EDV_DEVICE_ICON_STATE_SELECTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconLargeSelected = %s\n",
                    path
                );
            /* Icon large unmounted file. */
            path = dev_ptr->large_icon_file[EDV_DEVICE_ICON_STATE_UNMOUNTED];
            if(path != NULL)
                fprintf(
                    fp,
                    "    IconLargeUnmounted = %s\n",
                    path
                );

	    /* End device block. */
            fprintf(
                fp,
                "EndDevice\n"
            );
	}

        /* Call progress callback. */
        if(progress_cb != NULL)
            progress_cb(
                client_data, (gulong)total, (gulong)total
            );

        /* Close device file. */
        FClose(fp);
	fp = NULL;
}
