#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#ifdef __SOLARIS__
# include <sys/statvfs.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "edvtypes.h"
#include "edvdevices.h"
#include "edvdevicesfio.h"


edv_device_struct *EDVeviceMatchListByMountPath(
        edv_device_struct **list, gint total,
        gint *n,
        const gchar *mount_path
);
edv_device_struct *EDVDeviceMatchListByDevicePath(
        edv_device_struct **list, gint total,
        gint *n,
        const gchar *device_path
);
void EDVDevicesListUpdateMountStates(
        edv_device_struct **list, gint total
);
void EDVDevicesListUpdateStats(
        edv_device_struct **list, gint total
);

gint EDVDeviceGetFSFromString(const gchar *fs_type_str);
const gchar *EDVDeviceGetFSStringFromNumber(gint fs_type);

edv_device_struct *EDVDeviceNew(
        gint fs_type,                   /* One oF EDV_FS_TYPE_*. */
	const gchar *name,		/* Descriptive name. */
        const gchar *device_path,       /* Path to device object. */
        const gchar *mount_path         /* Path to mount point directory. */
);
static void EDVDeviceLoadIconsNexus(
        GdkPixmap **pixmap, GdkBitmap **mask,
        guint8 ***data, const gchar **file
);
void EDVDeviceLoadSmallIconsData(
        edv_device_struct *d, guint8 ***data
);
void EDVDeviceMediumIconsData(
        edv_device_struct *d, guint8 ***data
);
void EDVDeviceLoadLargeIconsData(
        edv_device_struct *d, guint8 ***data
);

void EDVDeviceRealize(
        edv_device_struct *d, gbool force_rerealize
);

void EDVDeviceDelete(edv_device_struct *d);



/*
 *	Matches a device structure in the given list that matches the
 *	given mount path.
 */
edv_device_struct *EDVeviceMatchListByMountPath(
        edv_device_struct **list, gint total,
        gint *n,
        const gchar *mount_path
)
{
	gint i;
	edv_device_struct *d;


	if(n != NULL)
	    *n = -1;

	if(mount_path == NULL)
	    return(NULL);

	/* Iterate through devices list. */
	for(i = 0; i < total; i++)
	{
	    d = list[i];
	    if(d == NULL)
		continue;

	    if(d->mount_path == NULL)
		continue;

	    if(!strcmp(d->mount_path, mount_path))
	    {
		if(n != NULL)
		    *n = i;
		return(d);
	    }
	}

	return(NULL);
}

/*
 *      Matches a device structure in the given list that matches the
 *      given device path.
 */
edv_device_struct *EDVDeviceMatchListByDevicePath(
        edv_device_struct **list, gint total,
        gint *n,
        const gchar *device_path
)
{
        gint i;
        edv_device_struct *d;


        if(n != NULL)
            *n = -1;

        if(device_path == NULL)
            return(NULL);

        /* Iterate through devices list. */
        for(i = 0; i < total; i++)
        {
            d = list[i];
            if(d == NULL)
                continue;

            if(d->device_path == NULL)
                continue;

            if(!strcmp(d->device_path, device_path))
            {
                if(n != NULL)
                    *n = i;
                return(d);
            }
        }

        return(NULL);
}

/*
 *	Updates the is_mounted state on all the device structures in the
 *	given list.
 */
void EDVDevicesListUpdateMountStates(
        edv_device_struct **list, gint total
)
{
	EDVDevicesListUpdateMountStatesFromSystem(
	    list, total
	);
}

/*
 *	Updates statistics for all device structures in the given list.
 *
 *	If a device is not marked as mounted then it will be skipped and
 *	its stats will be garbage. EDVDevicesListUpdateMountStates()
 *	should be called prior to this function.
 */
void EDVDevicesListUpdateStats(
        edv_device_struct **list, gint total
)
{
	gint i;
	const gchar *mount_path;
	edv_device_struct *dev_ptr;
#ifdef __SOLARIS__
        struct statvfs buf;
#else
	struct statfs buf;
#endif

	/* Iterate through devices. */
	for(i = 0; i < total; i++)
	{
	    dev_ptr = list[i];
	    if(dev_ptr == NULL)
		continue;

	    /* Reset stats. */
	    dev_ptr->blocks_total = 0;
	    dev_ptr->blocks_available = 0;
	    dev_ptr->blocks_free = 0;

	    /* Get stats only if mounted. */
	    if(dev_ptr->is_mounted)
	    {
		mount_path = dev_ptr->mount_path;
#ifdef __SOLARIS__
                if((mount_path != NULL) ? !statvfs(mount_path, &buf) : FALSE)
#else
		if((mount_path != NULL) ? !statfs(mount_path, &buf) : FALSE)
#endif
		{
		    /* Check if the block size transfer rate (which is
		     * really just the block size), is larger than or
		     * equal to the base of 1024 bytes per block.
		     */
		    if(buf.f_bsize >= 1024)
		    {
			gulong block_coeff = buf.f_bsize / 1024;

			dev_ptr->blocks_total = buf.f_blocks * block_coeff;
			dev_ptr->blocks_available = buf.f_bavail * block_coeff;
			dev_ptr->blocks_free = buf.f_bfree * block_coeff;
		    }
		    else if(buf.f_bsize > 0)
		    {
			/* Block size is less than 1024 bytes but positive. */
			gulong block_coeff = 1024 / buf.f_bsize;

			/* Note, block_coeff is in range of 1 to 1024. */
                        dev_ptr->blocks_total = buf.f_blocks / block_coeff;
                        dev_ptr->blocks_available = buf.f_bavail / block_coeff;
                        dev_ptr->blocks_free = buf.f_bfree / block_coeff;
		    }
		}
	    }
	}
}


/*
 *	Returns one of EDV_FS_TYPE_* based on the given fs_type_str.
 */
gint EDVDeviceGetFSFromString(const gchar *fs_type_str)
{
	gint i;
	const edv_fs_type_struct *fs_type_ptr;
	const edv_fs_type_struct fs_type_list[] = EDV_FS_TYPE_LIST;


	if(fs_type_str == NULL)
	    return(EDV_FS_TYPE_EMPTY);

	for(i = 0; fs_type_list[i].fs_type; i++)
	{
	    fs_type_ptr = &fs_type_list[i];
	    if(fs_type_ptr->name != NULL)
	    {
		if(!strcasecmp(fs_type_ptr->name, fs_type_str))
		    return(fs_type_ptr->fs_type);
	    }
	}

	return(EDV_FS_TYPE_EMPTY);
}

/*
 *      Returns a statically allocated string containing the name of
 *	the file system specified by fs_type.
 */
const gchar *EDVDeviceGetFSStringFromNumber(gint fs_type)
{
        gint i;
        const edv_fs_type_struct *fs_type_ptr;
        const edv_fs_type_struct fs_type_list[] = EDV_FS_TYPE_LIST;


        for(i = 0; fs_type_list[i].fs_type; i++)
        {
            fs_type_ptr = &fs_type_list[i];
            if(fs_type_ptr->name != NULL)
            {
                if(fs_type_ptr->fs_type == fs_type)
                    return(fs_type_ptr->name);
            }
        }

        return("");
}


/*
 *	Allocates a new device structure.
 */
edv_device_struct *EDVDeviceNew(
        gint fs_type,                   /* One of EDV_FS_TYPE_*. */
	const gchar *name,		/* Descriptive name. */
        const gchar *device_path,       /* Path to device object. */
        const gchar *mount_path         /* Path to mount point directory. */
)
{
	edv_device_struct *d = (edv_device_struct *)g_malloc0(
	    sizeof(edv_device_struct)
	);
	if(d == NULL)
	    return(d);

	d->fs_type = fs_type;

        d->name = (name != NULL) ?
            g_strdup(name) : NULL;
	d->device_path = (device_path != NULL) ?
	    g_strdup(device_path) : NULL;
        d->mount_path = (mount_path != NULL) ?
            g_strdup(mount_path) : NULL;

	d->is_mounted = FALSE;

	d->no_unmount = FALSE;
	d->read_only = FALSE;
	d->unlisted = FALSE;

	d->realized = FALSE;

	d->internal = FALSE;

	memset(
	    d->small_pixmap, 0x00,
	    EDV_DEVICE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
	);
        memset(
            d->small_mask, 0x00,
            EDV_DEVICE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
        );

	memset(
            d->medium_pixmap, 0x00,
            EDV_DEVICE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
        );
        memset(
            d->medium_mask, 0x00,
            EDV_DEVICE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
        );

	memset(
            d->large_pixmap, 0x00,
            EDV_DEVICE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
        );
        memset(
            d->large_mask, 0x00,
            EDV_DEVICE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
        );

	d->command_mount = NULL;
	d->command_unmount = NULL;
	d->command_eject = NULL;

        d->command_check = NULL;
	d->command_tools = NULL;
	d->command_format = NULL;

	d->blocks_total = 0;
	d->blocks_available = 0;
	d->blocks_free = 0;

	return(d);
}

/*
 *	Loads the pixmap and mask pair from either xpm data or xpm file
 *	depending if data or file is NULL.
 *
 *	Passing both data and file as NULL will only unref the existing icons
 *	(if any) and not load new ones. So this function can also be used 
 *	to just deallocate the icons.
 */
static void EDVDeviceLoadIconsNexus(
	GdkPixmap **pixmap, GdkBitmap **mask,
	guint8 ***data, const gchar **file
)
{
	gint i;
	GdkWindow *window = (GdkWindow *)GDK_ROOT_PARENT();
	GtkStyle *style = gtk_widget_get_default_style();


	/* Unload each given icon set. */
	for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	{
	    if(pixmap[i] != NULL)
	    {
/*
GdkWindowPrivate *private = (GdkWindowPrivate *)pixmap[i];
printf("Pixmap refcount = %i\n", private->ref_count - 1);
 */
		gdk_pixmap_unref(pixmap[i]);
		pixmap[i] = NULL;
	    }
            if(mask[i] != NULL)
            {
                gdk_bitmap_unref(mask[i]);
                mask[i] = NULL;
            }
	}

	/* Check if xpm data is given, if it is then the icon image data
	 * will be loaded.
	 */
	if((data != NULL) && (window != NULL) && (style != NULL))
	{
	    guint8 **data_ptr;
	    GdkPixmap *pixmap_ptr;
	    GdkBitmap *mask_ptr;


	    /* Load each icon set from data. */
	    for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	    {
		data_ptr = data[i];
		if(data_ptr == NULL)
		    continue;

		mask_ptr = NULL;
		pixmap_ptr = gdk_pixmap_create_from_xpm_d(
		    window, &mask_ptr,
		    &style->bg[GTK_STATE_NORMAL],
		    (gchar **)data_ptr
		);

		pixmap[i] = pixmap_ptr;
		mask[i] = mask_ptr;
	    }
	}
        /* Check if xpm file is given, if it is then the icon image data
         * will be loaded.
         */
        else if((file != NULL) && (window != NULL) && (style != NULL))
        {
            const gchar *path_ptr;
            GdkPixmap *pixmap_ptr;
            GdkBitmap *mask_ptr;


            /* Load each icon set from file. */
            for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
            {
                path_ptr = file[i];
                if(path_ptr == NULL)
                    continue;

                mask_ptr = NULL;
                pixmap_ptr = gdk_pixmap_create_from_xpm(
                    window, &mask_ptr,
                    &style->bg[GTK_STATE_NORMAL],
                    path_ptr
                );

                pixmap[i] = pixmap_ptr;
                mask[i] = mask_ptr;
            }
	}

}

/*
 *      Loads the small icons from xpm data. The given data must be an
 *      array of 3 guint8 ** pointers.
 *
 *      Passing data as NULL will only unref the existing icons (if any)
 *      and not load new ones.
 */
void EDVDeviceLoadSmallIconsData(
        edv_device_struct *d, guint8 ***data
)
{
        if(d == NULL)
            return;

	EDVDeviceLoadIconsNexus(
	    d->small_pixmap, d->small_mask,
	    data, NULL
	);
}

/*
 *      Loads the small icons from xpm data. The given data must be an
 *      array of 3 guint8 ** pointers.
 *
 *      Passing data as NULL will only unref the existing icons (if any)
 *      and not load new ones.
 */
void EDVDeviceLoadMediumIconsData(
        edv_device_struct *d, guint8 ***data
)
{
        if(d == NULL)
            return;

        EDVDeviceLoadIconsNexus(
            d->medium_pixmap, d->medium_mask,
            data, NULL
        );
}

/*
 *      Loads the small icons from xpm data. The given data must be an
 *      array of 3 guint8 ** pointers.
 *
 *      Passing data as NULL will only unref the existing icons (if any)
 *      and not load new ones.
 */
void EDVDeviceLoadLargeIconsData(
        edv_device_struct *d, guint8 ***data
)
{
        if(d == NULL)
            return;

        EDVDeviceLoadIconsNexus(
            d->large_pixmap, d->large_mask,
            data, NULL
        );
}


/*
 *	Realizes the given device structure, loading icons and
 *	related resources.
 *
 *	If the device is already realized (member realized is TRUE)
 *	then nothing will be done. However if force_rerealize is TRUE
 *	then the above check will be ignored.
 */
void EDVDeviceRealize(
        edv_device_struct *d, gbool force_rerealize
)
{
	const gchar **path;


        if(d == NULL)
            return;


	/* Not forcing a re-realize? */
	if(!force_rerealize)
	{
	    /* Device already realized? */
	    if(d->realized)
		return;
	}

	/* Begin realizing this device structure, loading icons
	 * from the specified file names.
	 */

        /* Get path array for small icon file. */
        path = (const gchar **)d->small_icon_file;
        if(path != NULL)
            EDVDeviceLoadIconsNexus(
                d->small_pixmap, d->small_mask,
                NULL, path
            );

        /* Get path array for medium icon file. */
        path = (const gchar **)d->medium_icon_file;
        if(path != NULL)
            EDVDeviceLoadIconsNexus(
                d->medium_pixmap, d->medium_mask,
                NULL, path
            );

	/* Get path for large icon file. */
	path = (const gchar **)d->large_icon_file;
	if(path != NULL)
	    EDVDeviceLoadIconsNexus(
		d->large_pixmap, d->large_mask,
		NULL, path
	    );

	/* Mark this device structure as being realized. */
	d->realized = TRUE;
}

/*
 *	Deallocates the device structure and all its resources.
 */
void EDVDeviceDelete(edv_device_struct *d)
{
	gint i;


	if(d == NULL)
	    return;

	/* Call the icon loaders with NULL as the argument, this will only
	 * unload the icons.
	 */
	EDVDeviceLoadSmallIconsData(d, NULL);
        EDVDeviceLoadMediumIconsData(d, NULL);
        EDVDeviceLoadLargeIconsData(d, NULL);

	for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	{
	    g_free(d->small_icon_file[i]);
	    d->small_icon_file[i] = NULL;
	}
        for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	{
            g_free(d->medium_icon_file[i]);
	    d->medium_icon_file[i] = NULL;
	}
        for(i = 0; i < EDV_DEVICE_TOTAL_ICON_STATES; i++)
	{
            g_free(d->large_icon_file[i]);
	    d->large_icon_file[i] = NULL;
	}

	g_free(d->command_mount);
	d->command_mount = NULL;
	g_free(d->command_unmount);
	d->command_unmount = NULL;
        g_free(d->command_eject);
        d->command_eject = NULL;

	g_free(d->command_check);
	d->command_check = NULL;
        g_free(d->command_tools);
        d->command_tools = NULL;
        g_free(d->command_format);
        d->command_format = NULL;

	g_free(d->name);
        d->name = NULL;
	g_free(d->device_path);
	d->device_path = NULL;
        g_free(d->mount_path);
        d->mount_path = NULL;

	g_free(d);
}
