#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "csd.h"
#include "fsd.h"
#include "progressdialog.h"
#include "pdialog.h"
#include "fprompt.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvdde.h"
#include "edvobj.h"
#include "edvarch.h"
#include "edvrecbin.h"
#include "browser.h"
#include "browsercb.h"
#include "imbr.h"
#include "imbrcb.h"
#include "archiver.h"
#include "archivercb.h"
#include "recbin.h"
#include "recbincb.h"
#include "recbinopcb.h"
#include "recbindeskicon.h"
#include "findwin.h"
#include "findwincb.h"
#include "propwin.h"
#include "endeavour.h"
#include "edvcfglist.h"
#include "edvop.h"
#include "edvopen.h"
#include "edvinterps.h"
#include "edvutils.h"
#include "config.h"


/* Callbacks. */
void EDVSignalCB(int s);

static void EDVManageTimeoutDoInterprocessCommand(
	edv_core_struct *core_ptr
);
gint EDVManageTimeoutCB(gpointer data);

void EDVNewBrowserCB(GtkWidget *widget, gpointer data);
void EDVNewImbrCB(GtkWidget *widget, gpointer data);
void EDVNewArchiverCB(GtkWidget *widget, gpointer data);
void EDVMapRecBinCB(GtkWidget *widget, gpointer data);

void EDVMapDevicesListCB(GtkWidget *widget, gpointer data);
void EDVMapMIMETypesListCB(GtkWidget *widget, gpointer data);

void EDVOptionsCB(GtkWidget *widget, gpointer data);
void EDVCustomizeCB(GtkWidget *widget, gpointer data);

static gint EDVCListColumnSortNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
        gint sort_code
);
gint EDVCListColumnSortStringCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
gint EDVCListColumnSortNumericCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

void EDVPurgeAllRecycledObjectsCB(GtkWidget *widget, gpointer data);

gboolean EDVEntryDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc, gint x, gint y, guint t,
        gpointer data
);
void EDVEntryDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);

void EDVLocBarIconDragDataGet(
	GtkWidget *widget, GdkDragContext *context,
	GtkSelectionData *selection_data, guint info, guint time,
	gpointer data
);

/* Emitters. */
void EDVWriteProtectChangedEmit(edv_core_struct *core_ptr, gbool state);

void EDVObjectAddedEmit(
        edv_core_struct *core_ptr, const gchar *path,
        const struct stat *lstat_buf
);
void EDVObjectModifiedEmit(
        edv_core_struct *core_ptr, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
);
void EDVObjectRemovedEmit(
        edv_core_struct *core_ptr, const gchar *path
);

void EDVObjectMountEmit(
        edv_core_struct *core_ptr,
	gint dev_num, edv_device_struct *dev_ptr,
        gbool is_mounted
);

extern void EDVArchiveObjectAddedNotifyCB(
        edv_core_struct *core_ptr, const gchar *arch_path,
        const gchar *path, edv_archive_object_struct *obj
);
extern void EDVArchiveObjectModifiedNotifyCB(
        edv_core_struct *core_ptr, const gchar *arch_path,
        const gchar *path, const gchar *new_path,
        edv_archive_object_struct *obj
);
extern void EDVArchiveObjectRemovedNotifyCB(
        edv_core_struct *core_ptr, const gchar *arch_path,
        const gchar *path
);

void EDVRecycledObjectAddedEmit(
        edv_core_struct *core_ptr, guint index
);
void EDVRecycledObjectRemovedEmit(
        edv_core_struct *core_ptr, guint index
);

void EDVReconfiguredEmit(edv_core_struct *core_ptr);

void EDVMimeTypeAddedEmit(
        edv_core_struct *core_ptr,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVMimeTypeModifiedEmit(
        edv_core_struct *core_ptr,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVMimeTypeRemovedEmit(
        edv_core_struct *core_ptr, gint mt_num
);

void EDVDeviceAddedEmit(
        edv_core_struct *core_ptr,
        gint dev_num, edv_device_struct *dev_ptr
);
void EDVDeviceModifiedEmit(
        edv_core_struct *core_ptr,
        gint dev_num, edv_device_struct *dev_ptr
);
void EDVDeviceRemovedEmit(
        edv_core_struct *core_ptr, gint dev_num
);



/*
 *	Signal callback.
 *
 *	No gtk/gdk calls can be made here, since this is actually
 *	called from a separate thread.
 */
void EDVSignalCB(int s)
{
	static int segfault_count = 0;
	edv_core_struct *core_ptr = edv_core;

	if(core_ptr == NULL)
	    return;

	switch(s)
	{
	  case SIGINT:
	  case SIGTERM:
	  case SIGKILL:
	    core_ptr->need_close_all_windows = TRUE;
	    break;

	  case SIGSEGV:
	    segfault_count++;	/* Count this segfault. */
	    fprintf(
		stderr,
 "%s triggered a segmentation fault (%i times)\n",
		PROG_NAME,
		segfault_count
	    );
	    /* 3 or more segfaults? */
	    if(segfault_count >= 3)
	    {
		fprintf(
		    stderr,
 "%s attempting immediate process exit().\n",
		    PROG_NAME
		);
		exit(1);	/* Explicit exit. */
	    }
	    else
	    {
		core_ptr->need_close_all_windows = TRUE;
	    }
	    break;

#ifdef SIGHUP
	  case SIGHUP:
	    break;
#endif

#ifdef SIGUSR1
          case SIGUSR1:
	    if(core_ptr->pid != 0)
	    {
		/* Record new interprocess command if there is no
		 * existing interprocess command queued.
		 */
		if(core_ptr->interprocess_command == NULL)
		{
		    /* Get new interprocess command. */
		    core_ptr->interprocess_command = EDVInterPSGetCommand(
			core_ptr->cfg_list,
			&core_ptr->total_interprocess_commands
		    );
		    /* Do not remove interprocess command on file, wait
		     * until we process it in the management timeout.
		     */
		}
	    }
            break;
#endif


	}
}

/*
 *	Called by EDVManageTimeoutCB() to handle an interprocess
 *	command.
 *
 *	The interprocess command recorded on the given core structure
 *	will be used and then deallocated.
 *
 *	All inputs assumed valid.
 */
static void EDVManageTimeoutDoInterprocessCommand(
        edv_core_struct *core_ptr
)
{
	const gchar *cmd_ptr;
	gint cmd_num;
	gchar **cmdv = core_ptr->interprocess_command;
	gint cmdc = core_ptr->total_interprocess_commands;
	gint argc;
	gchar **argv;


    /* Iterate through each interprocess commands. */
    for(cmd_num = 0; cmd_num < cmdc; cmd_num++)
    {
	cmd_ptr = cmdv[cmd_num];
	if((cmd_ptr != NULL) ? (*cmd_ptr == '\0') : TRUE)
	    continue;

	/* Explode command into arguments. */
	argv = strexp(cmdv[cmd_num], &argc);

	/* Got any arguments? */
	if((argv != NULL) && (argc > 0))
	{
	    const gchar *op = argv[0];

	    /* Begin parsing command by checking the operation op
	     * (the first argument).
	     */
	    /* Create new window? */
	    if((!strcasecmp(op, "new_window") ||
                !strcasecmp(op, "new")
	       ) && (argc >= 2)
	    )
	    {
		const gchar *window_name = argv[1];
		const gchar *startup_path = (argc >= 3) ? argv[2] : NULL;

		/* Record new startup path if specified from the
		 * arguments, this is so that when a new window is
		 * created it will have the given startup path.
		 */
		if((startup_path != NULL) ? (*startup_path != '\0') : FALSE)
		    EDVCFGItemListSetValueS(
			core_ptr->cfg_list, EDV_CFG_PARM_DIR_START_UP,
			startup_path, FALSE
		    );

		/* Create new window. */
		EDVDoWindowNexus(
		    core_ptr, window_name, startup_path,
		    (argc >= 4) ? argv[3] : NULL,
		    NULL
		);
	    }
	    /* Set configuration int? */
            else if((!strcasecmp(op, "set_configuration_i") ||
                     !strcasecmp(op, "set_cfg_i") ||
                     !strcasecmp(op, "set_configuration") ||
                     !strcasecmp(op, "set_cfg")
                    ) && (argc >= 3)
            )
            {
		const gchar *param = argv[1];
		gint value = (argv[2] != NULL) ? atoi(argv[2]) : 0;
		EDVCFGItemListSetValueI(
		    core_ptr->cfg_list, param, value, FALSE
		);
	    }
            /* Set configuration long? */
            else if((!strcasecmp(op, "set_configuration_l") ||
                     !strcasecmp(op, "set_cfg_l")
                    ) && (argc >= 3)
            )
            {
                const gchar *param = argv[1];
                glong value = (argv[2] != NULL) ? atol(argv[2]) : 0;
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, param, value, FALSE
                );
            }
            /* Set configuration unsigned long? */
            else if((!strcasecmp(op, "set_configuration_ul") ||
                     !strcasecmp(op, "set_cfg_ul")
                    ) && (argc >= 3)
            )
            {
                const gchar *param = argv[1];
                gulong value = (argv[2] != NULL) ? atol(argv[2]) : 0;
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, param, value, FALSE
                );
            }
            /* Set configuration float or double? */
            else if((!strcasecmp(op, "set_configuration_f") ||
                     !strcasecmp(op, "set_cfg_f") ||
                     !strcasecmp(op, "set_configuration_d") ||
                     !strcasecmp(op, "set_cfg_d")
                    ) && (argc >= 3)
            )
            {
                const gchar *param = argv[1];
                gdouble value = (argv[2] != NULL) ? atof(argv[2]) : 0;
                EDVCFGItemListSetValueF(
                    core_ptr->cfg_list, param, value, FALSE
                );
            }
            /* Set configuration string? */
            else if((!strcasecmp(op, "set_configuration_s") ||
                     !strcasecmp(op, "set_cfg_s")
                    ) && (argc >= 3)
            )
            {
		gint i;
                const gchar *param = argv[1];
                gchar *value = g_strdup("");
		for(i = 2; i < argc; i++)
		{
		    value = strcatalloc(value, argv[i]);
		    if(i < (argc - 1))
			value = strcatalloc(value, " ");
		}
                EDVCFGItemListSetValueS(
                    core_ptr->cfg_list, param, value, FALSE
                );
		g_free(value);
            }
	    /* Open object? */
	    else if((!strcasecmp(op, "open_object") ||
                     !strcasecmp(op, "open")
                    ) && (argc >= 2)
            )
	    {
                const gchar *path = argv[1];
		gchar	*stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;

		EDVOpenObjectPath(
		    core_ptr, path,
		    (argc >= 3) ? argv[2] : NULL,	/* Command name. */
		    NULL, TRUE,
		    &stdout_path_rtn, &stderr_path_rtn
		);

		g_free(stdout_path_rtn);
		g_free(stderr_path_rtn);
	    }
            /* Object added notify? */
            else if((!strcasecmp(op, "object_added_notify") ||
                     !strcasecmp(op, "object_add_notify")
                    ) &&
                    (argc >= 2)
            )
	    {
                const gchar *path = argv[1];
		struct stat lstat_buf;

		if(path != NULL)
		{
		    if(!lstat(path, &lstat_buf))
			EDVObjectAddedEmit(core_ptr, path, &lstat_buf);
		}
	    }
            /* Object modified notify? */
            else if((!strcasecmp(op, "object_modified_notify") ||
                     !strcasecmp(op, "object_modify_notify")
                    ) &&
                    (argc >= 2)
            )
            {
                const gchar *path = argv[1];
		const gchar *new_path = (argc >= 3) ? argv[2] : NULL;
                struct stat lstat_buf;

                if(path != NULL)
                {
                    if(!lstat(
			(new_path != NULL) ? new_path : path,
			&lstat_buf
		    ))
                        EDVObjectModifiedEmit(core_ptr, path, new_path, &lstat_buf);
                }
            }
            /* Object removed notify? */
            else if((!strcasecmp(op, "object_removed_notify") ||
                     !strcasecmp(op, "object_remove_notify")
                    ) &&
                    (argc >= 2)
            )
            {
                const gchar *path = argv[1];

                if(path != NULL)
		    EDVObjectRemovedEmit(core_ptr, path);
            }
            /* Object mounted or unmounted notify? */
            else if((!strcasecmp(op, "object_mounted_notify") ||
                     !strcasecmp(op, "object_mount_notify") ||
                     !strcasecmp(op, "object_unmounted_notify") ||
                     !strcasecmp(op, "object_unmount_notify")
                    ) &&
                    (argc >= 2)
            )
	    {
                const gchar *path = argv[1];
		gint dev_num;
		edv_device_struct *dev_ptr;

		/* The path could be a mount path or device path, see
		 * which one it is by matching it in our list of device
		 * references.
		 */
		for(dev_num = 0; dev_num < core_ptr->total_devices; dev_num++)
		{
		    dev_ptr = core_ptr->device[dev_num];
		    if(dev_ptr == NULL)
			continue;

		    if((dev_ptr->mount_path != NULL) && (path != NULL))
		    {
			if(!strcmp(dev_ptr->mount_path, path))
			    break;
		    }
                    if((dev_ptr->device_path != NULL) && (path != NULL))
                    {
                        if(!strcmp(dev_ptr->device_path, path))
                            break;
                    }
		}
		/* Got match? */
		if((dev_num < core_ptr->total_devices) &&
		   (dev_ptr != NULL)
		)
		{
		    /* Need to update mount states and device capacities. */
		    EDVDevicesListUpdateMountStates(
			core_ptr->device, core_ptr->total_devices
		    );
		    EDVDevicesListUpdateStats(
			core_ptr->device, core_ptr->total_devices
                    );
		    /* Send mount signal. */
		    EDVObjectMountEmit(
			core_ptr, dev_num, dev_ptr, dev_ptr->is_mounted
		    );
		}
	    }
            /* Recycled object added notify? */
            else if((!strcasecmp(op, "recycled_object_added_notify") ||
                     !strcasecmp(op, "recycled_object_add_notify")
                    ) &&
                    (argc >= 2)
            )
	    {
		guint index = (argv[1] != NULL) ? atoi(argv[1]) : 0;
		EDVRecycledObjectAddedEmit(core_ptr, index);
	    }
            /* Recycled object removed notify? */
            else if((!strcasecmp(op, "recycled_object_removed_notify") ||
                     !strcasecmp(op, "recycled_object_remove_notify")
                    ) &&
                    (argc >= 2)
            )
            {
		guint index = (argv[1] != NULL) ? atoi(argv[1]) : 0;
		EDVRecycledObjectRemovedEmit(core_ptr, index);
	    }
            /* Reconfigured notify? (when configuration has changed). */
            else if(!strcasecmp(op, "reconfigured_notify") ||
                    !strcasecmp(op, "reconfigure_notify")
            )
            {
                EDVReconfiguredEmit(core_ptr);
            }

            /* Help? */
            else if(!strcasecmp(op, "help"))
            {
		EDVDoHelp(
		    core_ptr,
		    (argc >= 2) ? argv[1] : NULL,
		    NULL
		);
            }
            /* About? */
            else if(!strcasecmp(op, "about"))
            {
                EDVDoHelpAbout(core_ptr, NULL);
            }

	    /* All else display error message about interprocess
	     * commands (if possible).
	     */
	    else if(!CDialogIsQuery())
	    {
		gchar *buf = g_strdup_printf(
"Unsupported interprocess command:\n\
\n\
    %s\n\
\n\
Was received by this application.",
		    op
		);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
"Unsupported Interprocess Command",
buf,
"Interprocess commands are sent between different\n\
processes of this same application to ensure that\n\
only one copy of this application's process exists\n\
per user. If you are writing a program to send\n\
interprocess commands to this application, then\n\
you should contact the authors for assistance.\n\
Otherwise you should regard this message as an\n\
internal programming error.",
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
	    }
	}

	StringFreeArray(argv, argc);
    }

    /* Deallocate queued interprocess commands. */
    StringFreeArray(cmdv, cmdc);
    core_ptr->interprocess_command = NULL;
    core_ptr->total_interprocess_commands = 0;

    /* Remove last interprocess command on file. */
    EDVInterPSRemoveCommand(core_ptr->cfg_list);
}

/*
 *	Core timeout callback.
 */
gint EDVManageTimeoutCB(gpointer data)
{
        static gbool reenterant = FALSE;
        gbool has_changes, still_processing, all_windows_unmapped;
        gint i;
        guint *toid;
/*	GtkWidget *transient_for_w = NULL; */
	edv_browser_struct *browser;
	edv_imbr_struct *imbr;
	edv_archiver_struct *archiver;
	edv_recbin_struct *recbin;
	edv_findwin_struct *findwin;
	edv_propwin_struct *propwin;
	edv_history_listwin_struct *history_listwin;
	edv_run_dlg_struct *run_dlg;
	edv_device_listwin_struct *device_listwin;
	edv_mimetype_listwin_struct *mimetype_listwin;
	edv_optwin_struct *optwin;
	about_dialog_struct *about_dialog;
	edv_core_struct *core_ptr = (edv_core_struct *)data;
	if(core_ptr == NULL)
	    return(FALSE);

        if(reenterant)
            return(TRUE);
        else
            reenterant = TRUE;

/* Macro to remove the timeout id pointed to by toid. */
#define DO_REMOVE_TOID		\
{ \
 if((*toid) != (guint)-1) \
 { \
  gtk_timeout_remove(*toid); \
  (*toid) = (guint)-1; \
 } \
}

/* Macro to remove all timeout callbacks and break from the top most
 * gtk_main loop.
 */
#define DO_GTK_MAIN_QUIT	\
{ \
 /* Remove GTK+ timeout callbacks. */ \
 toid = &core_ptr->manage_toid; \
 DO_REMOVE_TOID \
 \
 /* Flush GTK+ events. */ \
 while(gtk_events_pending() > 0) \
  gtk_main_iteration(); \
 \
 /* Break out of the top most gtk_main loop. */ \
 gtk_main_quit(); \
}

/* Macro to reschedual timeout callback to this function and return FALSE. */
#define DO_RETURN_RESCHEDUAL	\
{ \
 /* Reschedual call to this function. */ \
 core_ptr->manage_toid = gtk_timeout_add( \
  1000, /* 1 second interval. */ \
  (GtkFunction)EDVManageTimeoutCB, \
  core_ptr \
 ); \
\
 reenterant = FALSE; \
 /* Its okay to return FALSE because we just reschedualed another call to \
  * this function above. \
  */ \
 return(FALSE); \
}

/* Macro to return FALSE, thus removing the timeout callback to this
 * function.
 */
#define DO_RETURN_NO_RESCHEDUAL		\
{ \
 core_ptr->manage_toid = (guint)-1; \
 reenterant = FALSE; \
 return(FALSE); \
}

/* Macro to return TRUE, keeping the current timeout callback to this
 * function.
 */
#define DO_RETURN_CURRENT_SCHEDUAL	\
{ \
 reenterant = FALSE; \
 return(TRUE); \
}


	/* Need to close all windows? */
	if(core_ptr->need_close_all_windows)
	{
            /* Reset close all windows marker, which initially
             * triggered this case.
             */
            core_ptr->need_close_all_windows = FALSE;


            /* Check if any dialogs are in query, break query as needed
             * and return. Let next call to this function handle rest of
             * closing all windows.
             */
            if(CDialogIsQuery())
            {
                CDialogBreakQuery();
                DO_RETURN_CURRENT_SCHEDUAL
            }
            if(FileBrowserIsQuery())
            {
                FileBrowserBreakQuery();
                DO_RETURN_CURRENT_SCHEDUAL
            }
            if(ProgressDialogIsQuery())
            {
                ProgressDialogBreakQuery(FALSE);
                DO_RETURN_CURRENT_SCHEDUAL
            }
            if(FSDIsQuery())
            {
                FSDBreakQuery();
                DO_RETURN_CURRENT_SCHEDUAL
            }
            if(CSDIsQuery())
            {
                CSDBreakQuery();
                DO_RETURN_CURRENT_SCHEDUAL
            }
/*
            if(PDialogIsQuery())
            {
                PDialogBreakQuery();
                DO_RETURN_CURRENT_SCHEDUAL
            }
 */
            if(FPromptIsQuery())
            {
                FPromptBreakQuery();
                DO_RETURN_CURRENT_SCHEDUAL
            }
            /* All dialogs not in query. */

            /* Reset still processing marker, which indicates that there
             * are still windows that are processing and cannot be closed.
             */
            still_processing = FALSE;

            /* Reset has changes marker, which indicates there is atleast
             * one window with unsaved modified data.
             */
            has_changes = FALSE;

            /* Begin checking through all windows to see if any are
             * still processing or have changes.
             */

            /* Browser windows. */
            for(i = 0; i < core_ptr->total_browsers; i++)
            {
                browser = core_ptr->browser[i];
                if(browser == NULL)
                    continue;

                if(browser->initialized)
                {
                    if(browser->processing)
                        still_processing = TRUE;

#if 0
                    if(browser->has_changes)
                    {
                        /* Use this editor's toplevel window as the
                         * transient_for_w for the cdialog used in
                         * confirmation later.
                         */
                        transient_for_w = browser->toplevel;
                        has_changes = TRUE;
			break;
                    }
#endif
		}
            }
            /* Image browser windows. */
            for(i = 0; i < core_ptr->total_imbrs; i++)
            {
                imbr = core_ptr->imbr[i];
                if(imbr == NULL)
                    continue;

                if(imbr->initialized)
                {
                    if(imbr->processing)
                        still_processing = TRUE;
		}
	    }
            /* Archiver windows. */
            for(i = 0; i < core_ptr->total_archivers; i++)
            {
                archiver = core_ptr->archiver[i];
                if(archiver == NULL)
                    continue;

                if(archiver->initialized)
                {
                    if(archiver->processing)
                        still_processing = TRUE;
                }
            }
	    /* Recycle bin. */
	    recbin = core_ptr->recbin;
	    if(recbin != NULL)
	    {
                if(recbin->initialized)
                {
/* Nothing to check. */
                }
	    }
	    /* Find window. */
	    findwin = core_ptr->findwin;
	    if(findwin != NULL)
            {
                if(findwin->initialized)
                {
		    if(findwin->processing)
                        still_processing = TRUE;
                }
            }
            /* Property windows. */
            for(i = 0; i < core_ptr->total_propwins; i++)
            {
                propwin = core_ptr->propwin[i];
                if(propwin == NULL)
                    continue;

                if(propwin->initialized)
                {
/* Nothing to check. */
                }
            }

            /* History list window. */
            history_listwin = core_ptr->history_listwin;
            if(history_listwin != NULL)
            {
                /* No processing to check. */
            }
	    /* Run dialog. */
	    run_dlg = core_ptr->run_dlg;
            if(run_dlg != NULL)
            {
                /* No processing to check. */
            }
	    /* Device list window. */
	    device_listwin = core_ptr->device_listwin;
	    if(device_listwin != NULL)
	    {
		/* No processing to check. */
	    }
	    /* MIME Types list window. */
	    mimetype_listwin = core_ptr->mimetype_listwin;
	    if(mimetype_listwin != NULL)
            {
		/* No processing to check. */
            }
	    /* Options window. */
	    optwin = core_ptr->options_window;
	    if(optwin != NULL)
	    {
                if(optwin->initialized)
                {
                    if(optwin->processing)
                        still_processing = TRUE;
                }
	    }
            /* Customize window. */
            optwin = core_ptr->customize_window;
            if(optwin != NULL)
            {
		if(optwin->initialized)
		{
		    if(optwin->processing)
			still_processing = TRUE;
		}
            }
	    /* About dialog. */
	    about_dialog = core_ptr->about_dialog;
            if(about_dialog != NULL)
            {
                if(about_dialog->initialized)
                {

                }
            }

            /* Begin checking results, to see if any windows are still
             * processing or have unsaved changed data.
             */

            /* Any still processing? */
            if(still_processing)
            {
                /* One or more window is still processing, cannot close
                 * now so we need to return. The next call to this
                 * function won't close all windows since we set the
                 * global marker for that to FALSE.
                 */
                DO_RETURN_CURRENT_SCHEDUAL
            }


            /* Any have changed data? */
            if(has_changes)
            {
/* This should never happen since currently there is no editable
 * data support.
 */
	    }

            /* Break out of the top most gtk_main loop and remove the
             * timeout callback to this function.
             */
	    DO_GTK_MAIN_QUIT

	    /* Return and do not keep schedual callback to this function. */
	    DO_RETURN_NO_RESCHEDUAL
	}





        /* Begin checking if any window is still mapped, if they are
         * then all_unmapped will be set to FALSE.
         */
        all_windows_unmapped = TRUE;

        /* Iterate through all browsers, checking if any are not mapped.
	 * All unmapped browsers will be deleted.
	 */
        for(i = 0; i < core_ptr->total_browsers; i++)
        {
            browser = core_ptr->browser[i];
            if(browser == NULL)
                continue;

	    /* Initialized and mapped? */
            if(browser->initialized && browser->map_state)
            {
                all_windows_unmapped = FALSE;
            }
	    else
	    {
		/* Not initialized or unmapped, delete this browser. */
                core_ptr->browser[i] = NULL;
                EDVBrowserDelete(browser);
                browser = NULL;
	    }
        }
        /* Iterate through all image browsers, checking if any are not
	 * mapped. All unmapped image browsers will be deleted.
         */
        for(i = 0; i < core_ptr->total_imbrs; i++)
        {
            imbr = core_ptr->imbr[i];
            if(imbr == NULL)
                continue;

            /* Initialized and mapped? */
            if(imbr->initialized && imbr->map_state)
            {
                all_windows_unmapped = FALSE;
            }
            else
            {
                /* Not initialized or unmapped, delete this image browser. */
                core_ptr->imbr[i] = NULL;
                EDVImbrDelete(imbr);
                imbr = NULL;
            }
        }
        /* Iterate through all archivers, checking if any are not mapped.
	 * All unmapped archivers will be deleted.
         */
        for(i = 0; i < core_ptr->total_archivers; i++)
        {
            archiver = core_ptr->archiver[i];
            if(archiver == NULL)
                continue;

            /* Initialized and mapped? */
            if(archiver->initialized && archiver->map_state)
            {
                all_windows_unmapped = FALSE;
            }
            else
            {
                /* Not initialized or unmapped, delete this archiver. */
                core_ptr->archiver[i] = NULL;
                EDVArchiverDelete(archiver);
                archiver = NULL;
            }
        }
	/* Check if recycle bin is not mapped. */
	recbin = core_ptr->recbin;
	if(recbin != NULL)
	{
	    /* Initialized and mapped? */
            if(recbin->initialized && recbin->map_state)
            {
                all_windows_unmapped = FALSE;
	    }
	    else
	    {
		/* Do not delete recycle bin, even if it is not mapped
		 * since we only have one and it dosen't use much
		 * resource.
		 */
	    }
	}
        /* Check if find window is not mapped. */
        findwin = core_ptr->findwin;
        if(findwin != NULL)
        {
            /* Initialized and mapped? */
            if(findwin->initialized && findwin->map_state)
            {
                all_windows_unmapped = FALSE;
            }
            /* Do not delete find window if not mapped. */
        }
        /* Iterate through all property windows, checking if any are not
	 * mapped. All unmapped property windows will be deleted.
         */
        for(i = 0; i < core_ptr->total_propwins; i++)
        {
            propwin = core_ptr->propwin[i];
            if(propwin == NULL)
                continue;

            /* Initialized and mapped? */
            if(propwin->initialized && propwin->map_state)
            {
                all_windows_unmapped = FALSE;
            }
            else
            {
                /* Not initialized or unmapped, delete this property window. */
                core_ptr->propwin[i] = NULL;
                EDVPropWinDelete(propwin);
                propwin = NULL;
            }
        }



        /* History list window. */
        history_listwin = core_ptr->history_listwin;
        if(history_listwin != NULL)
        {
            if(history_listwin->initialized && history_listwin->map_state)
                all_windows_unmapped = FALSE;
        }

	/* Run dialog. */
	run_dlg = core_ptr->run_dlg;
	if(run_dlg != NULL)
	{
            if(run_dlg->initialized && run_dlg->map_state)
                all_windows_unmapped = FALSE;
        }

	/* Device list window. */
	device_listwin = core_ptr->device_listwin;
        if(device_listwin != NULL)
        {
            if(device_listwin->initialized && device_listwin->map_state)
                all_windows_unmapped = FALSE;
        }

        /* MIME Types list window. */
        mimetype_listwin = core_ptr->mimetype_listwin;
        if(mimetype_listwin != NULL)
        {
            if(mimetype_listwin->initialized && mimetype_listwin->map_state)
                all_windows_unmapped = FALSE;
        }

	/* Options window. */
	optwin = core_ptr->options_window;
	if(optwin != NULL)
	{
	    if(optwin->initialized && optwin->map_state)
		all_windows_unmapped = FALSE;
	}

        /* Customize window. */
        optwin = core_ptr->customize_window;
        if(optwin != NULL)
        {
            if(optwin->initialized && optwin->map_state)
                all_windows_unmapped = FALSE;
        }

	/* About dialog. */
	about_dialog = core_ptr->about_dialog;
	if(about_dialog != NULL)
	{
	    if(about_dialog->initialized && about_dialog->map_state)
		all_windows_unmapped = FALSE;
	}


        /* Were all windows unmapped? */
        if(all_windows_unmapped)
        {
            /* All windows are unmapped, this implies that the application
	     * must begin exiting.
             */

            /* Break out of the top most gtk_main loop and remove the
	     * timeout callback to this function.
             */
            DO_GTK_MAIN_QUIT

            /* Return and do not keep schedual callback to this function. */
            DO_RETURN_NO_RESCHEDUAL
        }



	/* Check for interprocess command. */
	if(core_ptr->interprocess_command != NULL)
	    EDVManageTimeoutDoInterprocessCommand(core_ptr);



        DO_RETURN_RESCHEDUAL

#undef DO_RETURN_RESCHEDUAL
#undef DO_RETURN_NO_RESCHEDUAL
#undef DO_RETURN_CURRENT_SCHEDUAL

#undef DO_GTK_MAIN_QUIT
#undef DO_REMOVE_TOID
}


/*
 *	Creates a new file browser.
 */
void EDVNewBrowserCB(GtkWidget *widget, gpointer data)
{
	EDVDoNewBrowser(
            (edv_core_struct *)data
	);
}

/*
 *      Creates a new image browser.
 */
void EDVNewImbrCB(GtkWidget *widget, gpointer data)
{
        EDVDoNewImbr(
            (edv_core_struct *)data
	);
}

/*
 *      Create new archiver callback.
 */
void EDVNewArchiverCB(GtkWidget *widget, gpointer data)
{
        EDVDoNewArchiver(
            (edv_core_struct *)data
	);
}

/*
 *      Map recycle bin callback.
 */
void EDVMapRecBinCB(GtkWidget *widget, gpointer data)
{
        EDVDoMapRecBin(
            (edv_core_struct *)data
	);
}

/*
 *      Map devices list callback.
 */
void EDVMapDevicesListCB(GtkWidget *widget, gpointer data)
{
        EDVDoMapDevicesListWin(
            (edv_core_struct *)data,
            (widget != NULL) ?
                gtk_widget_get_toplevel(widget) : NULL
	);
}

/*
 *      Map MIME Types list callback.
 */
void EDVMapMIMETypesListCB(GtkWidget *widget, gpointer data)
{
	EDVDoMapMIMETypesListWin(
	    (edv_core_struct *)data,
            (widget != NULL) ?
                gtk_widget_get_toplevel(widget) : NULL
	);
}

/*
 *	Maps the options window, creating it for the first time as needed.
 */
void EDVOptionsCB(GtkWidget *widget, gpointer data)
{
	EDVDoMapOptionsWin(
	    (edv_core_struct *)data,
	    (widget != NULL) ?
		gtk_widget_get_toplevel(widget) : NULL
	);
}

/*
 *	Maps the customize window, creating it for the first time as needed.
 */
void EDVCustomizeCB(GtkWidget *widget, gpointer data)
{
        EDVDoMapCustomizeWin(
            (edv_core_struct *)data,
            (widget != NULL) ?
                gtk_widget_get_toplevel(widget) : NULL
        );
}


/*
 *	GtkCList column sort callback nexus.
 *
 *	These only take the text value from the cell that corresesponds
 *	to the sort column of the given rows. The row data is ignored.
 *
 *	Only cells of type text and pixtext are supported, all other 
 *	cell types will return -1.
 */
static gint EDVCListColumnSortNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
)
{
	gint sort_column;
	GtkSortType sort_type;
	const gchar *text1 = NULL;
	const gchar *text2 = NULL;
	const GtkCListRow *row1 = (const GtkCListRow *)ptr1;
	const GtkCListRow *row2 = (const GtkCListRow *)ptr2;

	if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
	    return(-1);

	/* Get the column number that we are to sort by and what
	 * type of sorting (ascending or descending).
	 */
	sort_column = clist->sort_column;
	if((sort_column < 0) || (sort_column >= clist->columns))
	    return(-1);
	sort_type = clist->sort_type;

	/* Get the pointer to the text of the first cell. */
	switch(row1->cell[sort_column].type)
	{
	  case GTK_CELL_TEXT:
	    text1 = GTK_CELL_TEXT(row1->cell[sort_column])->text;
	    break;
	  case GTK_CELL_PIXTEXT:
	    text1 = GTK_CELL_PIXTEXT(row1->cell[sort_column])->text;
	    break;
	  default:
	    break;
	}

	/* Get the pointer to the text of the second cell. */
	switch(row2->cell[sort_column].type)
	{
	  case GTK_CELL_TEXT:
	    text2 = GTK_CELL_TEXT(row2->cell[sort_column])->text;
	    break;
	  case GTK_CELL_PIXTEXT:
	    text2 = GTK_CELL_PIXTEXT(row2->cell[sort_column])->text;
	    break;
	  default:
	    break;
	}

	if(text2 == NULL)
	    return((text1 != NULL) ? 1 : -1);
	if(text1 == NULL)
	    return(-1);

	/* Sort by the given sort_code. */
	switch(sort_code)
	{
	  case 0:	/* By text. */
	    return(strcmp(text1, text2));
	    break;

	  case 1:	/* By numeric. */
	    {
		/* Number strings may contain ',' characters, need
		 * to copy them to tempory strings and strip them.
		 */
		gchar	ns1[80], ns2[80];
		gint	n1, n2;

		/* Copy strings to tempory and modifyable number
		 * strings.
		 */
		strncpy(ns1, text1, 80);
		ns1[80 - 1] = '\0';
		strncpy(ns2, text2, 80);
		ns2[80 - 1] = '\0';

		/* Strip ',' from coppied tempory number strings. */
		substr(ns1, ",", "");
                substr(ns2, ",", "");

		n1 = strtod(ns1, NULL),
		n2 = strtod(ns2, NULL);

		if(n1 >= n2)
		    return((gint)(n1 > n2));
		else
		    return(-1);
	    }
	    break;

	  default:
	    return(-1);
	    break;
	}
}

/*
 *      GtkCList column sort by string callback.
 */
gint EDVCListColumnSortStringCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVCListColumnSortNexus(clist, ptr1, ptr2, 0));
}

/*
 *      GtkCList column sort by numeric callback.
 */
gint EDVCListColumnSortNumericCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVCListColumnSortNexus(clist, ptr1, ptr2, 1));
}

/*
 *	Maps the recycle bin and purges all of its contents.
 */
void EDVPurgeAllRecycledObjectsCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterent = FALSE;
	edv_core_struct *core_ptr = (edv_core_struct *)data;
	if(core_ptr == NULL)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;

	/* Create recycle bin as needed and map it. */
        EDVDoMapRecBin(core_ptr);
	EDVRecBinOPPurgeAll(core_ptr->recbin);

	reenterent = FALSE;
}


/*
 *      GtkEntry DND "drag_motion" signal callback.
 *
 *      This is used to constrain all drags (regardless of its type
 *      or source data type) to be a drag action of copy.
 */
gboolean EDVEntryDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc, gint x, gint y, guint t,
        gpointer data
)
{
        if(dc == NULL)
            return(FALSE);

        if(dc->actions & GDK_ACTION_COPY)
            gdk_drag_status(dc, GDK_ACTION_COPY, t);
        else
            gdk_drag_status(dc, 0, t);

        return(TRUE);
}

/*
 *	GtkEntry DND "drag_data_received" signal callback.
 *
 *	This is used as an opaque callback to support the dragging of any
 *	disk object to any GtkEntry widget. The value in the GtkEntry
 *	will be replaced with a space separated list of absolute disk
 *	object paths.
 *
 *	Note that only the text is updated, no "activate" signal will
 *	be emitted.
 */
void EDVEntryDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
	GtkEntry *entry;
        edv_dnd_object_struct **dnd_object = NULL, *dnd_obj;
        gint total_dnd_objects = 0;
	edv_core_struct *core_ptr = (edv_core_struct *)data;
	if((widget == NULL) || (dc == NULL) || (selection_data == NULL) ||
           (core_ptr == NULL)
	)
	    return;

#define DO_FREE_LOCALS	\
{ \
 gint i; \
\
 /* Deallocate dnd recycled object reference list. */ \
 for(i = 0; i < total_dnd_objects; i++) \
  EDVDNDObjectDelete(dnd_object[i]); \
 g_free(dnd_object); \
 dnd_object = NULL; \
 total_dnd_objects = 0; \
}

	/* Given widget must have or be a GtkEntry. */
        if(GTK_IS_COMBO(widget))
        {
	    GtkCombo *combo = GTK_COMBO(widget);
            entry = GTK_ENTRY(combo->entry);
        }
	else if(GTK_IS_ENTRY(widget))
	{
	    entry = GTK_ENTRY(widget);
	}
	else
	{
	    DO_FREE_LOCALS
	    return;
	}

        /* Parse the given selection buffer into an array of dnd object
         * structures.
         */
        dnd_object = EDVDNDBufferToObject(
            (const guint8 *)selection_data->data,
            selection_data->length,
            &total_dnd_objects
        );
	if(dnd_object == NULL)
	{
            DO_FREE_LOCALS
            return;
        }

	/* Drag data is a disk object path? */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
        )
	{
	    gint i;
	    gchar *text = g_strdup("");


	    /* Begin generating new text value as a space separated list
	     * of disk object paths.
	     */
	    for(i = 0; i < total_dnd_objects; i++)
	    {
		dnd_obj = dnd_object[i];
		if(dnd_obj == NULL)
		    continue;

		if(dnd_obj->full_path == NULL)
		    continue;

		/* Text value has become invalid (allocation error)? */
		if(text == NULL)
		    break;

		/* Prefix a space if this is not the first path. */
		if(*text != '\0')
		    text = strcatalloc(text, " ");

		/* Append this path to the string. */
		text = strcatalloc(text, dnd_obj->full_path);
	    }

	    /* Generated new text value? */
	    if(text != NULL)
	    {
		gtk_entry_set_text(entry, text);
                gtk_entry_set_position(entry, -1);
	    }

	    /* Deallocate text value, it is no longer needed. */
	    g_free(text);
	}

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}


/*
 *	Location bar icon "drag_data_get" signal callback.
 *
 *	The data is a GtkCombo.
 */
void EDVLocBarIconDragDataGet(
        GtkWidget *widget, GdkDragContext *context,
        GtkSelectionData *selection_data, guint info, guint time,
        gpointer data
)
{
	gboolean data_sent = FALSE;
	edv_dnd_object_struct *dnd_object[1], *dnd_obj;
        GtkEntry *entry;
        GtkCombo *combo = (GtkCombo *)data;
        if((selection_data == NULL) || (combo == NULL))
            return;

        entry = GTK_ENTRY(combo->entry);
        if(entry == NULL)
            return;

	dnd_object[0] = dnd_obj = EDVDNDObjectNew();
	if(dnd_obj != NULL)
	{
            guint8 *buf;
            gint buf_len;
	    const gchar *cstrptr = gtk_entry_get_text(entry);

	    dnd_obj->full_path = g_strdup(
		(cstrptr != NULL) ? cstrptr : ""
	    );

            buf = EDVDNDObjectToBuffer(
                (const edv_dnd_object_struct **)dnd_object, 1,
                &buf_len
            );
            if(buf != NULL)
            {
                /* Send out DND data buffer. */
                gtk_selection_data_set(
                    selection_data,
                    GDK_SELECTION_TYPE_STRING,
                    8,          /* 8 bits per character. */
                    buf, buf_len
                );
                data_sent = TRUE;
                g_free(buf);
            }

	    EDVDNDObjectDelete(dnd_obj);
	}

        /* Failed to send out data? */
        if(!data_sent)
        {
            /* Then send out an error response. */
            const gchar *strptr = "Error";

            gtk_selection_data_set(
                selection_data,
                GDK_SELECTION_TYPE_STRING,
                8,      /* 8 bits per character. */
                strptr, strlen(strptr)
            );
            data_sent = TRUE;
        }
}


/*
 *	Emits a write protect state change signal to all of endeavour's
 *	windows who are interested.
 */
void EDVWriteProtectChangedEmit(edv_core_struct *core_ptr, gbool state)
{
        static gbool reenterent = FALSE;
	gint i;


	if(core_ptr == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


	/* Begin notifying all resources of write protect change. */

	/* Browsers. */
	for(i = 0; i < core_ptr->total_browsers; i++)
	    EDVBrowserWriteProtectChangedCB(
		core_ptr->browser[i], state
	    );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrWriteProtectChangedCB(
                core_ptr->imbr[i], state
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverWriteProtectChangedCB(
                core_ptr->archiver[i], state
            );

	/* Recycle bin desktop icon. */
	EDVRecBinDeskIconUpdate(core_ptr->recbin_deskicon);

	/* Recycle bin. */
	EDVRecBinWriteProtectChangedCB(core_ptr->recbin, state);

	/* Find window. */
	EDVFindWinWriteProtectChangedCB(core_ptr->findwin, state);

        /* Property windows. */
        for(i = 0; i < core_ptr->total_propwins; i++)
            EDVWriteProtectChangedCB(
                core_ptr->propwin[i], state
            );




	reenterent = FALSE;
}

/*
 *      Emits a disk object added signal to all of endeavour's windows who
 *	are interested.
 */
void EDVObjectAddedEmit(
	edv_core_struct *core_ptr, const gchar *path,
        const struct stat *lstat_buf
)
{
        static gbool reenterent = FALSE;
        gint i;
        gchar *dpath;


        if((core_ptr == NULL) || (path == NULL))
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Create a local copy of the path. */
        dpath = g_strdup(path);


        /* Begin notifying all resources of a disk object added. */

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
	    EDVBrowserObjectAddedNotifyCB(
		core_ptr->browser[i], dpath, lstat_buf
	    );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrObjectAddedNotifyCB(
                core_ptr->imbr[i], dpath, lstat_buf
            );

	/* Do not report disk object added to archivers. */




        /* Deallocate coppied path. */
        g_free(dpath);
        dpath = NULL;

	reenterent = FALSE;
}

/*
 *      Emits a disk object modified signal to all of endeavour's windows
 *	who are interested.
 */
void EDVObjectModifiedEmit(
        edv_core_struct *core_ptr, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
)
{
        static gbool reenterent = FALSE;
        gint i;
        gchar *dpath;


        if((core_ptr == NULL) || (path == NULL))
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Create a local copy of the path. */
        dpath = g_strdup(path);


        /* Begin notifying all resources of a disk object modified. */

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
	    EDVBrowserObjectModifiedNotifyCB(
		core_ptr->browser[i], dpath, new_path, lstat_buf
	    );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrObjectModifiedNotifyCB(
                core_ptr->imbr[i], dpath, new_path, lstat_buf
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverObjectModifiedNotifyCB(
                core_ptr->archiver[i], dpath, new_path, lstat_buf
            );

        /* Do not report disk object modified to archivers. */




        /* Deallocate coppied path. */
        g_free(dpath);
        dpath = NULL;

	reenterent = FALSE;
}

/*
 *      Called whenever a disk object has been removed in some way.
 */
void EDVObjectRemovedEmit(edv_core_struct *core_ptr, const gchar *path)
{
        static gbool reenterent = FALSE;
        gint i;
	gchar *dpath;


        if((core_ptr == NULL) || (path == NULL))
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Create a local copy of the path. */
	dpath = g_strdup(path);


        /* Begin notifying all resources of a disk object removed. */

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
	    EDVBrowserObjectRemovedNotifyCB(
		core_ptr->browser[i], dpath
	    );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrObjectRemovedNotifyCB(
                core_ptr->imbr[i], dpath
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverObjectRemovedNotifyCB(
                core_ptr->archiver[i], dpath
            );



	/* Deallocate coppied path. */
	g_free(dpath);
	dpath = NULL;

	reenterent = FALSE;
}

/*
 *	Emits a mount signal to all of endeavour's windows who are
 *	interested.
 */
void EDVObjectMountEmit(
        edv_core_struct *core_ptr,
	gint dev_num, edv_device_struct *dev_ptr,
	gbool is_mounted
)
{
        static gbool reenterent = FALSE;
	gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Begin notifying resources about an object being mounted. */

        /* Devices list window. */
	EDVDevicesListWinMountNotifyCB(
            core_ptr->device_listwin, dev_num, dev_ptr, is_mounted
        );

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserMountNotifyCB(
                core_ptr->browser[i], dev_num, dev_ptr, is_mounted
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrMountNotifyCB(
                core_ptr->imbr[i], dev_num, dev_ptr, is_mounted
            );





        reenterent = FALSE;
}


/*
 *      Emits an archive object added signal to all of endeavour's
 *      windows who are interested.
 *
 *	The arch_path specifies the location of the archive that the
 *	archive object specified by path was added to. The obj contains
 *	the newly added archive object's statistics.
 */
void EDVArchiveObjectAddedNotifyCB(
        edv_core_struct *core_ptr, const gchar *arch_path,
        const gchar *path, edv_archive_object_struct *obj
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverArchiveObjectAddedNotifyCB(
                core_ptr->archiver[i], arch_path, path, obj
            );



        reenterent = FALSE;
}

/*
 *      Emits an archive object modified signal to all of endeavour's
 *      windows who are interested.
 *
 *      The arch_path specifies the location of the archive that the
 *      archive object specified by path was modified. The obj contains
 *      the newly modified archive object's statistics and path reffers
 *	to the archive object's path.
 */
void EDVArchiveObjectModifiedNotifyCB(
        edv_core_struct *core_ptr, const gchar *arch_path,
        const gchar *path, const gchar *new_path,
        edv_archive_object_struct *obj
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverArchiveObjectModifiedNotifyCB(
                core_ptr->archiver[i], arch_path, path, new_path, obj
            );



        reenterent = FALSE;
}

/*
 *      Emits an archive object removed signal to all of endeavour's
 *      windows who are interested.
 *
 *      The arch_path specifies the location of the archive that the
 *      archive object specified by path was removed.
 */
void EDVArchiveObjectRemovedNotifyCB(
        edv_core_struct *core_ptr, const gchar *arch_path,
        const gchar *path
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverArchiveObjectRemovedNotifyCB(
                core_ptr->archiver[i], arch_path, path
            );



        reenterent = FALSE;
}



/*
 *	Emits a recycled object added signal to all of endeavour's
 *	windows who are interested.
 *
 *	The given index is the recycled object index added to the recycle
 *	bin.
 */
void EDVRecycledObjectAddedEmit(
        edv_core_struct *core_ptr, guint index
)
{
        static gbool reenterent = FALSE;
	gint i;
	const gchar *recycled_index_file;
	struct stat stat_buf;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


	/* Get path of recycled object index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );

	/* Update number of recycled objects on the core structure,
	 * for now we will just alternate between core_ptr->last_recbin_items
	 * being 0 or 1 since we really only need to know if there is
	 * atleast one or no recycled objects.
	 */
	if(recycled_index_file != NULL)
	{
	    if(stat(recycled_index_file, &stat_buf))
		core_ptr->last_recbin_items = 0;
	    else
		core_ptr->last_recbin_items = (stat_buf.st_size > 0) ?
		    1 : 0;
	}
	else
	{
	    core_ptr->last_recbin_items = 0;
	}


        /* Begin notifying resources about an object added to the
         * recycle bin.
         */

	/* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserRecycledObjectAddedNotifyCB(
                core_ptr->browser[i], index
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrRecycledObjectAddedNotifyCB(
                core_ptr->imbr[i], index
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverRecycledObjectAddedNotifyCB(
                core_ptr->archiver[i], index
            );

        /* Recycle bin desktop icon. */
        EDVRecBinDeskIconUpdate(core_ptr->recbin_deskicon);

	/* Recycle bin. */
	EDVRecBinRecycledObjectAddedNotifyCB(
	    core_ptr->recbin, index
	);




        reenterent = FALSE;
}

/*
 *      Emits a recycled object removed signal to all of endeavour's
 *      windows who are interested.
 *
 *      The given index is the recycled object index removed from the
 *	recycle bin.
 */
void EDVRecycledObjectRemovedEmit(
        edv_core_struct *core_ptr, guint index
)
{
        static gbool reenterent = FALSE;
	gint i;
        const gchar *recycled_index_file;
        struct stat stat_buf;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Get path of recycled object index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );

        /* Update number of recycled objects on the core structure,
         * for now we will just alternate between core_ptr->last_recbin_items
         * being 0 or 1 since we really only need to know if there is
         * atleast one or no recycled objects.
         */
        if(recycled_index_file != NULL)
        {
            if(stat(recycled_index_file, &stat_buf))
                core_ptr->last_recbin_items = 0;
            else
                core_ptr->last_recbin_items = (stat_buf.st_size > 0) ?
                    1 : 0;
        }
        else
        {
            core_ptr->last_recbin_items = 0;
        }


        /* Begin notifying resources about an object removed from the
	 * recycle bin.
	 */

	/* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserRecycledObjectRemovedNotifyCB(
                core_ptr->browser[i], index
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrRecycledObjectRemovedNotifyCB(
                core_ptr->imbr[i], index
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverRecycledObjectRemovedNotifyCB(
                core_ptr->archiver[i], index
            );

	/* Recycle bin desktop icon. */
	EDVRecBinDeskIconUpdate(core_ptr->recbin_deskicon);

        /* Recycle bin. */
        EDVRecBinRecycledObjectRemovedNotifyCB(
            core_ptr->recbin, index
        );




        reenterent = FALSE;
}


/*
 *      Emits a reconfigure signal to all of endeavour's windows who are
 *      interested.
 */
void EDVReconfiguredEmit(edv_core_struct *core_ptr)
{
	static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;

	/* Update global tooltips state. */
	GUISetGlobalTipsState(
	    (EDVCFGItemListGetValueI(
		core_ptr->cfg_list, EDV_CFG_PARM_SHOW_TOOLTIPS
	    )) ? TRUE : FALSE
	);


        /* Begin notifying resources about the reconfigure. */

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserReconfiguredNotifyCB(
                core_ptr->browser[i]
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrReconfiguredNotifyCB(
                core_ptr->imbr[i]
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverReconfiguredNotifyCB(
                core_ptr->archiver[i]
            );

	/* Recycle bin desktop icon. */
        if(EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_SHOW_DESKTOP_ICON
        ))
	    EDVDoMapRecBinDeskIcon(core_ptr);
        else
            EDVRecBinDeskIconUnmap(core_ptr->recbin_deskicon);

	/* Recycle bin. */
	EDVRecBinReconfiguredNotifyCB(core_ptr->recbin);

	/* Property windows. */
        for(i = 0; i < core_ptr->total_propwins; i++)
            EDVPropWinReconfiguredNotifyCB(
                core_ptr->propwin[i]
            );





	reenterent = FALSE;
}


/*
 *      Emits a MIME Type added signal to all of endeavour's windows who
 *      are interested.
 */
void EDVMimeTypeAddedEmit(
        edv_core_struct *core_ptr,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Update MIME Types class hint indexes on the core structure. */
        EDVDoUpdateMIMETypeHintIndexes(core_ptr);

        /* Reget list of application MIME Types on the `open with' popup
         * list.
         */
        EDVDoUpdateOpenWithPUList(core_ptr);


        /* Begin notifying resources about MIME Type added. */

        /* Mime Types list window. */
        EDVMimeTypesListWinMimeTypeAddedCB(
            core_ptr->mimetype_listwin, mt_num, mt_ptr
        );

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserMimeTypeAddedCB(
                core_ptr->browser[i], mt_num, mt_ptr
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrMimeTypeAddedCB(
                core_ptr->imbr[i], mt_num, mt_ptr
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverMimeTypeAddedCB(
                core_ptr->archiver[i], mt_num, mt_ptr
            );

	/* Recycle bin. */
	EDVRecBinMimeTypeAddedCB(
	    core_ptr->recbin, mt_num, mt_ptr
	);

        reenterent = FALSE;
}

/*
 *      Emits a MIME Type modified signal to all of endeavour's windows who
 *      are interested.
 */
void EDVMimeTypeModifiedEmit(
        edv_core_struct *core_ptr,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Update MIME Types class hint indexes on the core structure. */
        EDVDoUpdateMIMETypeHintIndexes(core_ptr);

        /* Reget list of application MIME Types on the `open with' popup
         * list.
         */
        EDVDoUpdateOpenWithPUList(core_ptr);


        /* Begin notifying resources about MIME Type modified. */

        /* MIME Types list window. */
        EDVMimeTypesListWinMimeTypeModifiedCB(
            core_ptr->mimetype_listwin, mt_num, mt_ptr
        );

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserMimeTypeModifiedCB(
                core_ptr->browser[i], mt_num, mt_ptr
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrMimeTypeModifiedCB(
                core_ptr->imbr[i], mt_num, mt_ptr
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverMimeTypeModifiedCB(
                core_ptr->archiver[i], mt_num, mt_ptr
            );

        /* Recycle bin. */
        EDVRecBinMimeTypeModifiedCB(
            core_ptr->recbin, mt_num, mt_ptr
        );

        reenterent = FALSE;
}

/*
 *      Emits a MIME Type removed signal to all of endeavour's windows who
 *      are interested.
 */
void EDVMimeTypeRemovedEmit(
        edv_core_struct *core_ptr, gint mt_num
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Update MIME Types class hint indexes on the core structure. */
        EDVDoUpdateMIMETypeHintIndexes(core_ptr);

        /* Reget list of application MIME Types on the `open with' popup
	 * list.
	 */
	EDVDoUpdateOpenWithPUList(core_ptr);


        /* Begin notifying resources about MIME Type removed. */

        /* MIME Types list window. */
        EDVMimeTypesListWinMimeTypeRemovedCB(
            core_ptr->mimetype_listwin, mt_num
        );

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserMimeTypeRemovedCB(
                core_ptr->browser[i], mt_num
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrMimeTypeRemovedCB(
                core_ptr->imbr[i], mt_num
            );

        /* Archivers. */
        for(i = 0; i < core_ptr->total_archivers; i++)
            EDVArchiverMimeTypeRemovedCB(
                core_ptr->archiver[i], mt_num
            );

        /* Recycle bin. */
        EDVRecBinMimeTypeRemovedCB(
            core_ptr->recbin, mt_num
        );

        reenterent = FALSE;
}


/*
 *      Emits a device added signal to all of endeavour's windows who
 *      are interested.
 */
void EDVDeviceAddedEmit(
        edv_core_struct *core_ptr,
        gint dev_num, edv_device_struct *dev_ptr
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Reget devices list on devices popup list widget. */
        EDVDoUpdateDevicesPUList(core_ptr);


        /* Begin notifying resources about device added. */

        /* Devices list window. */
        EDVDevicesListWinDeviceAddedCB(
            core_ptr->device_listwin, dev_num, dev_ptr
        );

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserDeviceAddedCB(
                core_ptr->browser[i], dev_num, dev_ptr
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrDeviceAddedCB(
                core_ptr->imbr[i], dev_num, dev_ptr
            );

        reenterent = FALSE;
}

/*
 *      Emits a device modified signal to all of endeavour's windows who
 *	are interested.
 */
void EDVDeviceModifiedEmit(
        edv_core_struct *core_ptr,
        gint dev_num, edv_device_struct *dev_ptr
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


	/* Reget devices list on devices popup list widget. */
	EDVDoUpdateDevicesPUList(core_ptr);


        /* Begin notifying resources about device modified. */

	/* Devices list window. */
	EDVDevicesListWinDeviceModifiedCB(
	    core_ptr->device_listwin, dev_num, dev_ptr
	);

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserDeviceModifiedCB(
                core_ptr->browser[i], dev_num, dev_ptr
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrDeviceModifiedCB(
                core_ptr->imbr[i], dev_num, dev_ptr
            );

        reenterent = FALSE;
}

/*
 *      Emits a device removed signal to all of endeavour's windows who
 *      are interested.
 */
void EDVDeviceRemovedEmit(
        edv_core_struct *core_ptr, gint dev_num
)
{
        static gbool reenterent = FALSE;
        gint i;


        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Reget devices list on devices popup list widget. */
        EDVDoUpdateDevicesPUList(core_ptr);


        /* Begin notifying resources about device removed. */

        /* Devices list window. */
        EDVDevicesListWinDeviceRemovedCB(
            core_ptr->device_listwin, dev_num
        );

        /* Browsers. */
        for(i = 0; i < core_ptr->total_browsers; i++)
            EDVBrowserDeviceRemovedCB(
                core_ptr->browser[i], dev_num
            );

        /* Image browsers. */
        for(i = 0; i < core_ptr->total_imbrs; i++)
            EDVImbrDeviceRemovedCB(
                core_ptr->imbr[i], dev_num
            );

        reenterent = FALSE;
}

