#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvarch.h"
#include "edvarchfio.h"
#include "edvcfg.h"
#include "endeavour.h"
#include "edvarchop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/pdi_file01_20x20.xpm"
#include "images/pdi_file02_20x20.xpm"
#include "images/pdi_file03_20x20.xpm"
#include "images/pdi_file04_20x20.xpm"
#include "images/pdi_file05_20x20.xpm"
#include "images/pdi_file06_20x20.xpm"
#include "images/pdi_trash01_20x20.xpm"
#include "images/pdi_trash02_20x20.xpm"
#include "images/pdi_trash03_20x20.xpm"
#include "images/pdi_trash04_20x20.xpm"
#include "images/pdi_trash05_20x20.xpm"
#include "images/pdi_trash06_20x20.xpm"
#include "images/pdi_folder_32x32.xpm"
#include "images/pdi_folderfile_32x32.xpm"
#include "images/pdi_package_32x32.xpm"
#include "images/pdi_packagefile_32x32.xpm"

#include "images/icon_replace_file_32x32.xpm"


/*
 *      Return values have the following error meanings:
 *
 *      0       Success (no error)
 *      -1      General error
 *      -2      Ambiguous, not found, other error.
 *      -3      Systems error (out of memory/out of disk space)
 *      -4      User responded with "Cancel"
 *      -5      User responded with "No" or response is not available.
 *      -6      Call would cause reentry.
 */

const gchar *EDVArchOPGetError();
static gbool EDVArchOPFPHasData(FILE *fp);
static gint EDVArchOPConfirmOverwriteExtract(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, const gchar *tar_path,
        edv_archive_object_struct *src_obj_stat,
        const struct stat *tar_lstat_buf
);
static void EDVArchOPMapProgressDialogDeleteUnknown(
	const gchar *label, GtkWidget *toplevel, gbool force_remap
);
static void EDVArchOPMapProgressDialogAddUnknown(
        const gchar *label, GtkWidget *toplevel, gbool force_remap
);
static void EDVArchOPMapProgressDialogExtractUnknown(
        const gchar *label, GtkWidget *toplevel, gbool force_remap
);

/* Archive object deleting. */
static gint EDVArchOPDeleteArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        const gchar *src_obj,
        edv_archive_object_struct *src_obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all
);
static gint EDVArchOPDeleteTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        const gchar *src_obj,
        edv_archive_object_struct *src_obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
);
static gint EDVArchOPDeleteZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        const gchar *src_obj,
        edv_archive_object_struct *src_obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all
);
gint EDVArchOPDelete(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct *obj_stat,
	GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all
);

/* Adding disk objects to archive. */
static gint EDVArchOPAddArj(
        edv_core_struct *core_ptr, const gchar *arch_obj,
        gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links
);
static gint EDVArchOPAddTar(
        edv_core_struct *core_ptr, const gchar *arch_obj,
        gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
);
static gint EDVArchOPAddZip(
        edv_core_struct *core_ptr, const gchar *arch_obj,
        gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links
);
gint EDVArchOPAdd(
	edv_core_struct *core_ptr, const gchar *arch_obj,
        gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
	GtkWidget *toplevel,
	gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links
);

/* Extracting archive objects from archive. */
static gint EDVArchOPExtractArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_path,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories
);
static gint EDVArchOPExtractTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
	edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_obj,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
);
static gint EDVArchOPExtractZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_obj,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories
);
gint EDVArchOPExtract(
        edv_core_struct *core_ptr,
	const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_obj,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
	GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories
);



static gchar *last_error = NULL;

#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)
#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Returns the last error message or NULL if there was no error.
 *
c *      The returned pointer must not be deallocated.
 */
const gchar *EDVArchOPGetError()
{
        return(last_error);
}


/*
 *	Checks if the given FILE pointer has data ready to be read.
 */
static gbool EDVArchOPFPHasData(FILE *fp)
{
        gint fd;
        fd_set rfds;
        struct timeval tv;


	if(fp == NULL)
	    return(FALSE);

        /* Get file descriptor number from file pointer. */
        fd = fileno(fp);

        /* Clear file descriptor set return. */
        FD_ZERO(&rfds);
        /* Add our fd to the list of file descriptors to check. */
        FD_SET(fd, &rfds);
        /* Set wait period to 0 seconds. */
        tv.tv_sec = 0;          /* Seconds. */
        tv.tv_usec = 0;         /* Microseconds. */

        switch(select(fd + 1, &rfds, NULL, NULL, &tv))
        {
          case -1:
            /* Error. */
            switch(errno)
            {
              case EBADF:       /* Invalid descriptor. */
		return(FALSE);
                break;

              case EINTR:       /* Non blocked signal was caught. */
		return(FALSE);
		break;

              case ENOMEM:	/* System is out of memory. */
		return(FALSE);
                break;

              default:		/* Unknown error. */
		return(FALSE);
		break;
            }
            break;

          case 0:
            /* No data available. */
            return(FALSE);
            break;

          default:
            /* Data is available. */
	    return(TRUE);
            break;
        }
}


/*
 *      Maps the confirmation dialog and queries user for replacing the
 *      given src_path with the tar_path.
 *
 *      Returns one of CDIALOG_RESPONSE_*.
 */
static gint EDVArchOPConfirmOverwriteExtract(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        const gchar *src_path, const gchar *tar_path,
	edv_archive_object_struct *src_obj_stat,
	const struct stat *tar_lstat_buf
)
{
        gchar *buf, *src_date, *tar_date;
        gint result, relativity;
        const gchar *cstrptr, *format;
        gulong src_size, tar_size;


        if(core_ptr == NULL)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );


        /* Get date strings for source and target objects. */
        cstrptr = EDVDateFormatString(
            (src_obj_stat != NULL) ? src_obj_stat->modify_time : 0,
            format, relativity
        );
        src_date = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;

        cstrptr = EDVDateFormatString(
            (tar_lstat_buf != NULL) ? tar_lstat_buf->st_mtime : 0,
            format, relativity
        );
        tar_date = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        /* Get sizes of source and target objects in units of bytes. */
        src_size = (gulong)((src_obj_stat != NULL) ?
            src_obj_stat->size : 0);

        tar_size = (gulong)((tar_lstat_buf != NULL) ?
            tar_lstat_buf->st_size : 0);


        /* Generate message. */
        buf = g_strdup_printf(
"Replace:\n\
\n\
    %s (%ld byte%s) %s\n\
\n\
With:\n\
\n\
    %s (%ld byte%s) %s\n",

            tar_path, tar_size, (tar_size == 1) ? "" : "s", tar_date,
            src_path, src_size, (src_size == 1) ? "" : "s", src_date
        );

        CDialogSetTransientFor(toplevel);
        result = CDialogGetResponseIconData(
            "Confirm Overwrite", buf, NULL,
            (guint8 **)icon_replace_file_32x32_xpm,
/* Note that we cannot have a "No" option. */
            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
            CDIALOG_BTNFLAG_CANCEL,
            CDIALOG_BTNFLAG_CANCEL
        );
        CDialogSetTransientFor(NULL);

        g_free(buf);
	g_free(tar_date);
	g_free(src_date);

        return(result);
}


/*
 *      Maps the progress dialog as needed in animation mode for deleting
 *	with an unknown progress value (activity mode).
 */
static void EDVArchOPMapProgressDialogDeleteUnknown(
        const gchar *label, GtkWidget *toplevel, gbool force_remap
)
{
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];

        /* Already mapped? */
        if(ProgressDialogIsQuery())
        {
            /* Check if the progress dialog needs to be unmapped and
             * remapped again.
             */
            if(force_remap)
            {
                ProgressDialogBreakQuery(FALSE);
            }
            else
            {
                /* Already mapped and does not need unmapping, so just
                 * update the progress message.
                 */
		ProgressDialogUpdateUnknown(
		    NULL, label, NULL, NULL, TRUE
		);
                return;
            }
        }

        ProgressDialogSetTransientFor(toplevel);

        start_icon_data[0] = (u_int8_t **)pdi_packagefile_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)pdi_package_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)pdi_package_32x32_xpm;
        icon_data[0] = (u_int8_t **)pdi_trash01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_trash02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_trash03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_trash04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_trash05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_trash06_20x20_xpm;
        end_icon_data[0] = (u_int8_t **)NULL;
        end_icon_data[1] = (u_int8_t **)NULL;
        end_icon_data[2] = (u_int8_t **)NULL;

        ProgressDialogMapAnimation(
            "Deleting",
            label,
            "Stop",
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );
        ProgressDialogUpdateUnknown(
	    NULL, NULL, NULL, NULL, TRUE
	);
}

/*
 *	Maps the progress dialog as needed in animation mode for adding
 *	with an unknown progress value (activity mode).
 */
static void EDVArchOPMapProgressDialogAddUnknown(
        const gchar *label, GtkWidget *toplevel, gbool force_remap
)
{
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];

        /* Already mapped? */
        if(ProgressDialogIsQuery())
        {
            /* Check if the progress dialog needs to be unmapped and
             * remapped again.
             */
            if(force_remap)
            {
                ProgressDialogBreakQuery(FALSE);
            }
            else
            {
                /* Already mapped and does not need unmapping, so just
                 * update the progress message.
                 */
                ProgressDialogUpdateUnknown(
                    NULL, label, NULL, NULL, TRUE
                );
                return;
            }
        }

        ProgressDialogSetTransientFor(toplevel);

        start_icon_data[0] = (u_int8_t **)pdi_folderfile_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)pdi_folderfile_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)pdi_folderfile_32x32_xpm;
        icon_data[0] = (u_int8_t **)pdi_file01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_file02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_file03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_file04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_file05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_file06_20x20_xpm;
        end_icon_data[0] = (u_int8_t **)pdi_package_32x32_xpm;
        end_icon_data[1] = (u_int8_t **)pdi_package_32x32_xpm;
        end_icon_data[2] = (u_int8_t **)pdi_packagefile_32x32_xpm;

        ProgressDialogMapAnimation(
            "Adding",
            label,
            "Stop",
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );
        ProgressDialogUpdateUnknown(
            NULL, NULL, NULL, NULL, TRUE
        );
}

/*
 *	Maps the progress dialog as needed in animation mode for
 *	extracting.
 */
static void EDVArchOPMapProgressDialogExtractUnknown(
        const gchar *label, GtkWidget *toplevel, gbool force_remap
)
{
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];

        /* Already mapped? */
        if(ProgressDialogIsQuery())
        {
            /* Check if the progress dialog needs to be unmapped and
             * remapped again.
             */
            if(force_remap)
            {
                ProgressDialogBreakQuery(FALSE);
            }
            else
            {
                /* Already mapped and does not need unmapping, so just
                 * update the progress message.
                 */
                ProgressDialogUpdateUnknown(
                    NULL, label, NULL, NULL, TRUE
                );
                return;
            }
        }

        ProgressDialogSetTransientFor(toplevel);

        start_icon_data[0] = (u_int8_t **)pdi_packagefile_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)pdi_packagefile_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)pdi_packagefile_32x32_xpm;
        icon_data[0] = (u_int8_t **)pdi_file01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_file02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_file03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_file04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_file05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_file06_20x20_xpm;
        end_icon_data[0] = (u_int8_t **)pdi_folder_32x32_xpm;
        end_icon_data[1] = (u_int8_t **)pdi_folder_32x32_xpm;
        end_icon_data[2] = (u_int8_t **)pdi_folderfile_32x32_xpm;

        ProgressDialogMapAnimation(
            "Extracting",
            label,
            "Stop",
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );
        ProgressDialogUpdateUnknown(
            NULL, NULL, NULL, NULL, TRUE
        );
}


/*
 *      Procedure to delete using the arj application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPDeleteArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        const gchar *src_obj,
        edv_archive_object_struct *src_obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all
)
{
        const gchar *prog_arj = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ARJ
        );
        gint status;
        pid_t p;
        gchar *cmd;


#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
}

        status = 0;

        /* Generate command to delete from archive. */
        if(1)
        {
            cmd = g_strdup_printf(
                "%s d -i \"%s\" \"%s\"",
                prog_arj, arch_obj, src_obj
            );
            if(cmd == NULL)
            {
                last_error = "Unable to generate delete command";
                DO_FREE_LOCALS
                return(-2);
            }

            /* Execute delete command. */
            p = Exec(cmd);
            if(p <= 0)
            {
                last_error = "Unable to execute delete command";
                g_free(cmd);
                return(-1);
            }

            /* Deallocate delete command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;

            /* Begin monitoring deleting processing, waiting for it
             * to finish or if user aborts.
             */
            while(ExecProcessExists(p))
            {
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                usleep(8000);
            }
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Procedure to delete using the tar application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPDeleteTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        const gchar *src_obj,
        edv_archive_object_struct *src_obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
)
{
        const gchar *prog_tar = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_TAR
        );
        const gchar *prog_compress = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_COMPRESS
        );
        const gchar *prog_uncompress = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_UNCOMPRESS
        );
        const gchar *prog_gzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_GZIP
        );
        const gchar *prog_gunzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_GUNZIP
        );
        const gchar *prog_bzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BZIP2
        );
        const gchar *prog_bunzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
        );
        gint status;
        pid_t p;
        gchar *strptr, *strptr2;
        gchar *cmd = NULL, *arch_uncompressed_path = NULL;


#define DO_FREE_LOCALS	\
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(arch_uncompressed_path); \
 arch_uncompressed_path = NULL; \
}

        /* Generate tar plain path from the given arch_obj which is
         * gauranteed to have a .tar, .tgz, .tar.gz, or .tar.bz2 postfix
         */
        arch_uncompressed_path = g_strdup(arch_obj);
        strptr = strrchr(arch_uncompressed_path, DIR_DELIMINATOR);
        if(strptr != NULL)
            strptr++;
        else
            strptr = arch_uncompressed_path;
        strptr2 = strstr(strptr, ".tgz");
        if(strptr2 == NULL)
            strptr2 = strstr(strptr, ".tar.gz");
        if(strptr2 == NULL)
            strptr2 = strstr(strptr, ".tar.bz2");
        if(strptr2 == NULL)
            strptr2 = strstr(strptr, ".tar.Z");
        /* Explicitly set new postfix (yes we have enough allocation for
         * 4 bytes, see above match criteria.
         */
        if(strptr2 != NULL)
            strcpy(strptr2, ".tar");


        /* Generate command to decompress original archive. */
	if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s \"%s\"",
                prog_uncompress, arch_obj
            );
        else if(is_gzip_compressed)
            cmd = g_strdup_printf(
                "%s \"%s\"",
                prog_gunzip, arch_obj
            );
        else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s \"%s\"",
                prog_bunzip2, arch_obj
            );
        else
            cmd = NULL;

        status = 0;

        /* Execute decompress original archive command as needed. */
        if(cmd != NULL)
        {
            p = Exec(cmd);
            if(p <= 0)
            {
                last_error = "Unable to execute decompress command";
                DO_FREE_LOCALS
                return(-1);
            }

            /* Deallocate command string, we don't need it anymore. */
            g_free(cmd);
            cmd = NULL;

            /* Wait for decompress process to finish. */
            while(ExecProcessExists(p))
            {
                if(show_progress && ProgressDialogIsQuery())
                {
		    ProgressDialogUpdateUnknown(
			NULL, NULL, NULL, NULL, TRUE
		    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                usleep(8000);
            }
        }
        /* User canceled? */
        if(status)
        {
            DO_FREE_LOCALS
            return(status);
        }


        /* Generate command to delete from archive. */
        if(1)
        {
            cmd = g_strdup_printf(
                "%s --delete -f \"%s\" \"%s\"",
                prog_tar, arch_uncompressed_path, src_obj
            );
            if(cmd == NULL)
            {
                last_error = "Unable to generate delete command";
                DO_FREE_LOCALS
                return(-2);
            }

            /* Execute delete command. */
            p = Exec(cmd);
            if(p <= 0)
            {
                last_error = "Unable to execute delete command";
                DO_FREE_LOCALS
                return(-1);
            }

            /* Deallocate delete command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;

            /* Wait for decompress process to finish. */
            while(ExecProcessExists(p))
            {
                if(show_progress && ProgressDialogIsQuery())
                {
		    ProgressDialogUpdateUnknown(
			NULL, NULL, NULL, NULL, TRUE
		    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
	                }

                usleep(8000);
            }

        }

        /* Generate command to compress original archive. */
        if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s -c \"%s\"",
                prog_compress, arch_uncompressed_path
            );
        else if(is_gzip_compressed)
            cmd = g_strdup_printf(
                "%s -c \"%s\"",
                prog_gzip, arch_uncompressed_path
            );
        else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s -c \"%s\"",
                prog_bzip2, arch_uncompressed_path
            );
        else
            cmd = NULL;

        /* Execute compress command as needed. */
        if(cmd != NULL)
        {
            p = ExecO(cmd, arch_obj);
            if(p <= 0)
            {
                last_error = "Unable to execute compress command";
                DO_FREE_LOCALS
                return(-1);
            }

            /* Deallocate command string, we don't need it anymore. */
            g_free(cmd);
            cmd = NULL;

            /* Wait for compress process to finish. */
            while(ExecProcessExists(p))
            {
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                usleep(8000);
            }

            /* Remove uncompressed version of archive. */
            unlink(arch_uncompressed_path);
        }


        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Procedure to delete using the zip application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPDeleteZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        const gchar *src_obj,
        edv_archive_object_struct *src_obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all
)
{
        const gchar *prog_zip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ZIP
        );
        gint status;
        pid_t p;
        gchar *cmd;


#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
}

	status = 0;

        /* Generate command to delete from archive. */
        if(1)
        {
            cmd = g_strdup_printf(
		"%s -d \"%s\" \"%s\"",
		prog_zip, arch_obj, src_obj
            );
            if(cmd == NULL)
            {
                last_error = "Unable to generate delete command";
                DO_FREE_LOCALS
                return(-2);
            }

            /* Execute delete command. */
            p = Exec(cmd);
            if(p <= 0)
            {
                last_error = "Unable to execute delete command";
                g_free(cmd);
                return(-1);
            }

            /* Deallocate delete command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;

            /* Begin monitoring deleting processing, waiting for it
             * to finish or if user aborts.
             */
            while(ExecProcessExists(p))
            {
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                usleep(8000);
            }
	}

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Deletes the object specified in the obj_stat from the archive
 *	object specified by arch_obj.
 */
gint EDVArchOPDelete(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct *obj_stat,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all
)
{
        static gbool reenterent = FALSE;
        static gint status;
	gulong time_start = (gulong)time(NULL);
        const gchar *src_obj;


        /* Reset returns. */
        /* Leave yes_to_all as whatever value the calling function set it. */

        if(reenterent)
        {
            last_error =
"System is busy, an operation is already in progress";
            return(-6);
        }
        else
        {
            reenterent = TRUE;
        }

        /* Reset last error pointer. */
        last_error = NULL;


        if((core_ptr == NULL) || (arch_obj == NULL) || (obj_stat == NULL) ||
           (yes_to_all == NULL)
	)
        {
            reenterent = FALSE;
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
}

        /* Get full path of source object from source archive object
         * stats.
         */
        src_obj = obj_stat->full_path;
        if(src_obj == NULL)
        {
            last_error =
 "Path not found in source archive object stats";
            DO_FREE_LOCALS
            reenterent = FALSE;
            return(-1);
        }

        /* Update progress? */
        if(show_progress)
        {
            gchar *p1 = EDVCopyShortenPath(
                src_obj, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
            );
            gchar *buf = g_strdup_printf(
"Deleting:\n\
\n\
    %s\n",
                p1
            );

            EDVArchOPMapProgressDialogDeleteUnknown(
		buf, toplevel, FALSE
	    );

            g_free(buf);
            g_free(p1);
	}


        /* Begin deleting the source object from the archive object
	 * arch_obj. The deleting method will be determined by taking
	 * the extension of the archive object's name.
         */

        status = -1;

        if(EDVIsExtension(arch_obj, ".arj"))
        {
            status = EDVArchOPDeleteArj(
                core_ptr, arch_obj, src_obj, obj_stat,
                toplevel, show_progress, interactive, yes_to_all
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.Z"))
        {
            status = EDVArchOPDeleteTar(
                core_ptr, arch_obj, src_obj, obj_stat,
                toplevel, show_progress, interactive, yes_to_all,
		TRUE,		/* Is compress compressed. */
                FALSE,		/* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tgz .tar.gz"))
        {
            status = EDVArchOPDeleteTar(
                core_ptr, arch_obj, src_obj, obj_stat,
                toplevel, show_progress, interactive, yes_to_all,
		FALSE,		/* Not compress compressed. */
                TRUE,           /* Is gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.bz2"))
        {
            status = EDVArchOPDeleteTar(
                core_ptr, arch_obj, src_obj, obj_stat,
                toplevel, show_progress, interactive, yes_to_all,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                TRUE            /* Is bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar"))
        {
            status = EDVArchOPDeleteTar(
                core_ptr, arch_obj, src_obj, obj_stat,
                toplevel, show_progress, interactive, yes_to_all,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".zip"))
        {
            status = EDVArchOPDeleteZip(
                core_ptr, arch_obj, src_obj, obj_stat,
                toplevel, show_progress, interactive, yes_to_all
            );
        }
        else
        {
            last_error = "Unsupported archive format";
        }

        /* Record history. */
        EDVAppendHistory(
            core_ptr,
            EDV_HISTORY_ARCHIVE_OBJECT_DELETE,
            time_start, (gulong)time(NULL),
            status,
	    arch_obj,		/* Source full path is the archive. */
	    src_obj,		/* Target full path is the archive object. */
	    last_error		/* Use last_error as comment if not NULL. */
        );

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        reenterent = FALSE;
        return(status);
}


/*
 *      Procedure to add using the arj application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPAddArj(
        edv_core_struct *core_ptr, const gchar *arch_obj,
	gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links
)
{
        const gchar *prog_arj = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ARJ
        );
        gint status, arch_obj_stat_result;
        pid_t p = 0;
        const gchar *cstrptr;
        gchar *parent_dir = NULL, *stdout_path = NULL;
        gchar *cmd = NULL;
        FILE *fp;
        gchar prev_working_dir[PATH_MAX];
	struct stat arch_obj_stat_buf;


        /* Record previous working dir and set new working dir. */
        *prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
        cstrptr = GetParentDir(arch_obj);
        parent_dir = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(parent_dir == NULL)
            return(-1);
        chdir(parent_dir);

#define DO_FREE_LOCALS			\
{					\
 g_free(parent_dir);			\
 parent_dir = NULL;			\
 g_free(stdout_path);			\
 stdout_path = NULL;			\
 g_free(cmd);				\
 cmd = NULL;				\
					\
 /* Restore previous working dir. */	\
 chdir(prev_working_dir);		\
}


        /* Special arj file check, if the existing arj file is of 0 bytes
         * in size then it must first be removed before new objects can
         * be added to it.
         */
        arch_obj_stat_result = stat(arch_obj, &arch_obj_stat_buf);
        if(!arch_obj_stat_result)
        {
            if(arch_obj_stat_buf.st_size == 0)
            {
                unlink(arch_obj);
            }
        }

        status = 0;

        /* Generate command to add to archive. */
        if(cmd == NULL)
        {
	    gint i;
	    gchar *s;
	    const gchar *tpath;

            cmd = g_strdup_printf(
                "%s a -i -y %s \"%s\"",
                prog_arj,
                recursive ? "-r" : "",
                arch_obj
            );
	    for(i = 0; i < total_tar_paths; i++)
	    {
		tpath = tar_path[i];
		if(tpath == NULL)
		    continue;

		/* Generate relative path to the target path as
		 * needed.
		 */
		if(EDVIsParentPath(parent_dir, tpath))
                {
                    tpath += strlen(parent_dir);
                    while(*tpath == DIR_DELIMINATOR)
                        tpath++;
                }

		s = g_strdup_printf(" \"%s\"", tpath);
		cmd = strcatalloc(cmd, s);
		g_free(s);
	    }
            if(cmd == NULL)
            {
                last_error = "Unable to generate add command";
                DO_FREE_LOCALS
                return(-2);
            }

            /* Generate standard output file. */
            stdout_path = EDVTmpName(NULL);

            /* Execute add command. */
            p = ExecO(cmd, stdout_path);
            if(p <= 0)
            {
                last_error = "Unable to execute add command";
                DO_FREE_LOCALS
                return(-1);
            }

            /* Deallocate add command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;
        }

        /* Open output file on this end for reading. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
#define buf_len         10000
            gint buf_pos = 0;
            gchar buf[buf_len];
            gbool need_break = FALSE;

            /* Begin reading output file. */
            while(1)
            {
                /* Update progress? */
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }


                /* Check if there is new data to be read from the output
                 * file.
                 */
                if(EDVArchOPFPHasData(fp))
                {
                    gint c;
                    gbool got_complete_line = FALSE;

                    /* Copy all available data from the current output
                     * file position to its end to the line buffer buf.
                     */
                    while(1)
                    {
                        c = fgetc(fp);
                        if(c == EOF)
                            break;

                        if((c == '\n') || (c == '\r'))
                        {
                            got_complete_line = TRUE;
                            if(buf_pos < buf_len)
                                buf[buf_pos] = '\0';
                            else
                                buf[buf_len - 1] = '\0';
                            buf_pos = 0;
                            break;
                        }

                        if(buf_pos < buf_len)
                        {
                            buf[buf_pos] = c;
                            buf_pos++;
                        }
                    }
                    /* Got a complete line from the output file and the
                     * progress dialog is mapped?
                     */
                    if(got_complete_line && show_progress &&
		       (strcasepfx(buf, "Adding") ||
		        strcasepfx(buf, "Replacing")
		       )
		    )
                    {
                        /* Update progress dialog label. */
                        gchar *strptr, *strptr2;
                        gchar *p1, *p2, *buf2;

                        /* Set cstrptr to the start of the loaded line buffer
                         * and seek past the first "Adding:" prefix.
                         */
                        strptr = buf;
                        while(ISBLANK(*strptr))
                            strptr++;
                        while(!ISBLANK(*strptr) && (*strptr != '\0'))
                            strptr++;
                        while(ISBLANK(*strptr))
                            strptr++;

                        /* Cap the space character after path. */
                        strptr2 = strchr(strptr, ' ');
                        if(strptr2 != NULL)
                            *strptr2 = '\0';


                        p1 = EDVCopyShortenPath(
                            strptr, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        p2 = EDVCopyShortenPath(
                            arch_obj, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        buf2 = g_strdup_printf(
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
                            p1, p2
                        );
                        EDVArchOPMapProgressDialogAddUnknown(
                            buf2, toplevel, FALSE
                        );
                        g_free(buf2);
                        g_free(p1);
                        g_free(p2);


                        /* Append to return list of new archive object
                         * paths.
                         */
                        if((total_new_obj_paths != NULL) &&
                           (new_obj_path != NULL)
                        )
                        {
                            gint i = *total_new_obj_paths;
                            *total_new_obj_paths = i + 1;
                            *new_obj_path = (gchar **)g_realloc(
                                *new_obj_path,
                                (*total_new_obj_paths) * sizeof(gchar *)
                            );
                            if(*new_obj_path == NULL)
                                *total_new_obj_paths = 0;
                            else
                                (*new_obj_path)[i] = g_strdup(strptr);
                        }

                        continue;
                    }
                }

                if(1)
                {
                    /* No new data to be read from output file, check if
                     * the need_break marker was already set by the previous
                     * loop. If it was set then we actually break out of
                     * this loop.
                     */
                    if(need_break)
                        break;

                    /* Check if add process has exited, if it has then
                     * we set need_break to TRUE. Which will be tested on
                     * the next loop if there is still no more data to be
                     * read.
                     */
                    if(!ExecProcessExists(p))
                        need_break = TRUE;
                }

                usleep(8000);
            }

            FClose(fp);
            fp = NULL;
#undef buf_len
        }

        /* Remove output file. */
        if(stdout_path != NULL)
        {
            unlink(stdout_path);
            g_free(stdout_path);
            stdout_path = NULL;
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Procedure to add using the tar application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPAddTar(
        edv_core_struct *core_ptr, const gchar *arch_obj,
        gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
)
{
        const gchar *prog_tar = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_TAR
        );
        const gchar *prog_compress = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_COMPRESS
        );
        const gchar *prog_uncompress = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_UNCOMPRESS
        );
        const gchar *prog_gzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_GZIP
        );
        const gchar *prog_gunzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_GUNZIP
        );
        const gchar *prog_bzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BZIP2
        );
        const gchar *prog_bunzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
        );
        gint status;
        pid_t p = 0;
	const gchar *cstrptr;
	gchar *strptr, *strptr2;
	gchar *parent_dir = NULL;
        gchar *cmd = NULL, *stdout_path = NULL, *arch_uncompressed_path = NULL;
	FILE *fp;
	gchar prev_working_dir[PATH_MAX];


        /* Record previous working dir and set new working dir. */
        *prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
	cstrptr = GetParentDir(arch_obj);
	parent_dir = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if(parent_dir == NULL)
	    return(-1);
        chdir(parent_dir);

#define DO_FREE_LOCALS			\
{					\
 g_free(parent_dir);			\
 parent_dir = NULL;			\
 g_free(cmd);				\
 cmd = NULL;				\
 g_free(stdout_path);			\
 stdout_path = NULL;			\
 g_free(arch_uncompressed_path);	\
 arch_uncompressed_path = NULL;		\
					\
 /* Restore previous working dir. */	\
 chdir(prev_working_dir);		\
}


	/* Generate tar plain path from the given arch_obj which is
	 * gauranteed to have a .tar, .tgz, .tar.gz, or .tar.bz2 postfix
	 */
	arch_uncompressed_path = g_strdup(arch_obj);
	strptr = strrchr(arch_uncompressed_path, DIR_DELIMINATOR);
	if(strptr != NULL)
	    strptr++;
	else
	    strptr = arch_uncompressed_path;
	strptr2 = strstr(strptr, ".tgz");
	if(strptr2 == NULL)
            strptr2 = strstr(strptr, ".tar.gz");
        if(strptr2 == NULL)
            strptr2 = strstr(strptr, ".tar.bz2");
        if(strptr2 == NULL)
            strptr2 = strstr(strptr, ".tar.Z");
	/* Explicitly set new postfix (yes we have enough allocation for
	 * 4 bytes, see above match criteria.
	 */
	if(strptr2 != NULL)
	    strcpy(strptr2, ".tar");


        /* Generate command to decompress original archive. */
        if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s \"%s\"",
                prog_uncompress, arch_obj
            );
        else if(is_gzip_compressed)
	    cmd = g_strdup_printf(
                "%s \"%s\"",
                prog_gunzip, arch_obj
            );
        else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s -q \"%s\"",
                prog_bunzip2, arch_obj
            );
        else
            cmd = NULL;

	status = 0;

	/* Execute decompress original archive command as needed. */
	if(cmd != NULL)
	{
	    p = Exec(cmd);
	    if(p <= 0)
	    {
                last_error = "Unable to execute decompress command";
		DO_FREE_LOCALS
		return(-1);
	    }

	    /* Deallocate command string, we don't need it anymore. */
	    g_free(cmd);
	    cmd = NULL;

	    /* Wait for decompress process to finish. */
	    while(ExecProcessExists(p))
	    {
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

		usleep(8000);
	    }
	}
	/* User canceled? */
	if(status)
	{
	    DO_FREE_LOCALS
            return(status);
	}


	/* Generate command to add to archive. */
	if(cmd == NULL)
	{
            gint i;
            gchar *s;
            const gchar *tpath;

            cmd = g_strdup_printf(
                "%s -r %s %s -v -f \"%s\"",
                prog_tar,
                recursive ? "" : "--no-recursion",
                dereference_links ? "-h" : "",
                arch_uncompressed_path
            );
            for(i = 0; i < total_tar_paths; i++)
            {
                tpath = tar_path[i];
                if(tpath == NULL)
                    continue;

                /* Generate relative path to the target path as
                 * needed.
                 */
                if(EDVIsParentPath(parent_dir, tpath))
                {
                    tpath += strlen(parent_dir);
                    while(*tpath == DIR_DELIMINATOR)
                        tpath++;
                }

                s = g_strdup_printf(" \"%s\"", tpath);
                cmd = strcatalloc(cmd, s);
                g_free(s);
	    }
	    if(cmd == NULL)
	    {
		last_error = "Unable to generate add command";
		DO_FREE_LOCALS
		return(-2);
	    }

            /* Generate standard output file. */
            stdout_path = EDVTmpName(NULL);

	    /* Execute add command. */
	    p = ExecO(cmd, stdout_path);
            if(p <= 0)
            {
                last_error = "Unable to execute add command";
	        DO_FREE_LOCALS
                return(-1);
            }

	    /* Deallocate add command, it is no longer needed. */
	    g_free(cmd);
	    cmd = NULL;
	}

	/* Open output file on this end for reading. */
	fp = FOpen(stdout_path, "rb");
	if(fp != NULL)
	{
#define buf_len		10000
	    gint buf_pos = 0;
	    gchar buf[buf_len];
	    gbool need_break = FALSE;

            /* Begin reading output file. */
	    while(1)
	    {
		/* Update progress? */
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }


		/* Check if there is new data to be read from the output
		 * file.
		 */
		if(EDVArchOPFPHasData(fp))
		{
		    gint c;
		    gbool got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf.
		     */
		    while(1)
		    {
			c = fgetc(fp);
			if(c == EOF)
			    break;

			if((c == '\n') || (c == '\r'))
			{
			    got_complete_line = TRUE;
			    if(buf_pos < buf_len)
                                buf[buf_pos] = '\0';
			    else
				buf[buf_len - 1] = '\0';
			    buf_pos = 0;
			    break;
			}

			if(buf_pos < buf_len)
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    /* Got a complete line from the output file and the
		     * progress dialog is mapped?
		     */
		    if(got_complete_line && show_progress)
		    {
			/* Update progress dialog label. */
			gchar *p1 = EDVCopyShortenPath(
			    buf, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			);
			gchar *p2 = EDVCopyShortenPath(
			    arch_obj, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			);
			gchar *buf2 = g_strdup_printf(
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
			    p1, p2
			);
			EDVArchOPMapProgressDialogAddUnknown(
			    buf2, toplevel, FALSE
                        );
                        g_free(buf2);
                        g_free(p1);
                        g_free(p2);


			/* Append to return list of new archive object
			 * paths.
			 */
			if((total_new_obj_paths != NULL) &&
                           (new_obj_path != NULL)
			)
			{
			    gint i = *total_new_obj_paths;
			    *total_new_obj_paths = i + 1;
			    *new_obj_path = (gchar **)g_realloc(
				*new_obj_path,
				(*total_new_obj_paths) * sizeof(gchar *)
			    );
			    if(*new_obj_path == NULL)
				*total_new_obj_paths = 0;
			    else
				(*new_obj_path)[i] = g_strdup(buf);
			}

			continue;
		    }
		}

		if(1)
		{
		    /* No new data to be read from output file, check if
		     * the need_break marker was already set by the previous
		     * loop. If it was set then we actually break out of
		     * this loop.
		     */
		    if(need_break)
			break;

		    /* Check if add process has exited, if it has then
		     * we set need_break to TRUE. Which will be tested on
		     * the next loop if there is still no more data to be
		     * read.
		     */
		    if(!ExecProcessExists(p))
			need_break = TRUE;
		}

		usleep(8000);
	    }

	    FClose(fp);
	    fp = NULL;
#undef buf_len
	}

        /* Remove output file. */
        if(stdout_path != NULL)
        {
            unlink(stdout_path);
            g_free(stdout_path);
            stdout_path = NULL;
        }

        /* User canceled? */
        if(status)
        {
            DO_FREE_LOCALS
            return(status);
        }


        /* Generate command to compress original archive. */
        if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s -c \"%s\"",
                prog_compress, arch_uncompressed_path
            );
        else if(is_gzip_compressed)
            cmd = g_strdup_printf(
                "%s -%i -c \"%s\"",
                prog_gzip,
		CLIP(compression * 9 / 100, 1, 9),
		arch_uncompressed_path
            );
        else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s -c \"%s\"",
                prog_bzip2, arch_uncompressed_path
            );
        else
            cmd = NULL;

	/* Execute compress command as needed. */
        if(cmd != NULL)
        {
            p = ExecO(cmd, arch_obj);
            if(p <= 0)
            {
                last_error = "Unable to execute compress command";
		DO_FREE_LOCALS
                return(-1);
            }

            /* Deallocate command string, we don't need it anymore. */
            g_free(cmd);
            cmd = NULL;

            /* Wait for compress process to finish. */
            while(ExecProcessExists(p))
            {
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                usleep(8000);
            }

	    /* Remove uncompressed version of archive. */
	    unlink(arch_uncompressed_path);
	}


	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *      Procedure to add using the zip application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPAddZip(
        edv_core_struct *core_ptr, const gchar *arch_obj,
	gchar **tar_path, gint total_tar_paths,
	gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links
)
{
        const gchar *prog_zip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ZIP
        );
        gint status, arch_obj_stat_result;
        pid_t p = 0;
        const gchar *cstrptr;
        gchar *parent_dir = NULL, *stdout_path = NULL;
        gchar *cmd = NULL;
        FILE *fp;
	struct stat arch_obj_stat_buf;
        gchar prev_working_dir[PATH_MAX];


        /* Record previous working dir and set new working dir. */
        *prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
        cstrptr = GetParentDir(arch_obj);
        parent_dir = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if(parent_dir == NULL)
            return(-1);
        chdir(parent_dir);

#define DO_FREE_LOCALS			\
{					\
 g_free(parent_dir);			\
 parent_dir = NULL;			\
 g_free(stdout_path);			\
 stdout_path = NULL;			\
 g_free(cmd);				\
 cmd = NULL;				\
					\
 /* Restore previous working dir. */	\
 chdir(prev_working_dir);		\
}

	/* Special zip file check, if the existing zip file is of 0 bytes
	 * in size then it must first be removed before new objects can
	 * be added to it.
	 */
	arch_obj_stat_result = stat(arch_obj, &arch_obj_stat_buf);
	if(!arch_obj_stat_result)
	{
	    if(arch_obj_stat_buf.st_size == 0)
	    {
		unlink(arch_obj);
	    }
	}

	status = 0;

        /* Generate command to add to archive. */
        if(cmd == NULL)
        {
            gint i;
            gchar *s;
            const gchar *tpath;

            cmd = g_strdup_printf(
                "%s -%i %s %s \"%s\"",
                prog_zip,
                CLIP(compression * 9 / 100, 0, 9),
                recursive ? "-r" : "",
                dereference_links ? "" : "-y",
                arch_obj
            );
            for(i = 0; i < total_tar_paths; i++)
            {
                tpath = tar_path[i];
                if(tpath == NULL)
                    continue;

                /* Generate relative path to the target path as
                 * needed.
                 */
                if(EDVIsParentPath(parent_dir, tpath))
                {
                    tpath += strlen(parent_dir);
                    while(*tpath == DIR_DELIMINATOR)
                        tpath++;
                }

                s = g_strdup_printf(" \"%s\"", tpath);
                cmd = strcatalloc(cmd, s);
                g_free(s);
            }
            if(cmd == NULL)
            {
                last_error = "Unable to generate add command";
                DO_FREE_LOCALS
                return(-2);
            }

            /* Generate standard output file. */
            stdout_path = EDVTmpName(NULL);

            /* Execute add command. */
            p = ExecO(cmd, stdout_path);
            if(p <= 0)
            {
                last_error = "Unable to execute add command";
                DO_FREE_LOCALS
                return(-1);
            }

            /* Deallocate add command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;
        }

        /* Open output file on this end for reading. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
#define buf_len         10000
            gint buf_pos = 0;
            gchar buf[buf_len];
            gbool need_break = FALSE;

            /* Begin reading output file. */
            while(1)
            {
                /* Update progress? */
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }


                /* Check if there is new data to be read from the output
                 * file.
                 */
                if(EDVArchOPFPHasData(fp))
                {
                    gint c;
                    gbool got_complete_line = FALSE;

                    /* Copy all available data from the current output
                     * file position to its end to the line buffer buf.
                     */
                    while(1)
                    {
                        c = fgetc(fp);
                        if(c == EOF)
                            break;

                        if((c == '\n') || (c == '\r'))
                        {
                            got_complete_line = TRUE;
                            if(buf_pos < buf_len)
                                buf[buf_pos] = '\0';
                            else
                                buf[buf_len - 1] = '\0';
                            buf_pos = 0;
                            break;
                        }

                        if(buf_pos < buf_len)
                        {
                            buf[buf_pos] = c;
                            buf_pos++;
                        }
                    }
                    /* Got a complete line from the output file and the
                     * progress dialog is mapped?
                     */
                    if(got_complete_line && show_progress)
                    {
                        /* Update progress dialog label. */
			gchar *strptr, *strptr2;
                        gchar *p1, *p2, *buf2;

			/* Set cstrptr to the start of the loaded line buffer
			 * and seek past the first "adding:" prefix.
			 */
			strptr = buf;
			while(ISBLANK(*strptr))
			    strptr++;
			while(!ISBLANK(*strptr) && (*strptr != '\0'))
                            strptr++;
			while(ISBLANK(*strptr))
                            strptr++;

			/* Cap the space character after path. */
			strptr2 = strchr(strptr, ' ');
			if(strptr2 != NULL)
			    *strptr2 = '\0';


			p1 = EDVCopyShortenPath(
                            strptr, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        p2 = EDVCopyShortenPath(
                            arch_obj, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        buf2 = g_strdup_printf(
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
                            p1, p2
                        );
                        EDVArchOPMapProgressDialogAddUnknown(
                            buf2, toplevel, FALSE
                        );
                        g_free(buf2);
                        g_free(p1);
                        g_free(p2);


                        /* Append to return list of new archive object
                         * paths.
                         */
                        if((total_new_obj_paths != NULL) &&
                           (new_obj_path != NULL)
                        )
                        {
                            gint i = *total_new_obj_paths;
                            *total_new_obj_paths = i + 1;
                            *new_obj_path = (gchar **)g_realloc(
                                *new_obj_path,
                                (*total_new_obj_paths) * sizeof(gchar *)
                            );
                            if(*new_obj_path == NULL)
                                *total_new_obj_paths = 0;
                            else
                                (*new_obj_path)[i] = g_strdup(strptr);
                        }

                        continue;
                    }
                }

                if(1)
                {
                    /* No new data to be read from output file, check if
                     * the need_break marker was already set by the previous
                     * loop. If it was set then we actually break out of
                     * this loop.
                     */
                    if(need_break)
                        break;

                    /* Check if add process has exited, if it has then
                     * we set need_break to TRUE. Which will be tested on
                     * the next loop if there is still no more data to be
                     * read.
                     */
                    if(!ExecProcessExists(p))
                        need_break = TRUE;
                }

                usleep(8000);
            }

            FClose(fp);
            fp = NULL;
#undef buf_len
        }

        /* Remove output file. */
        if(stdout_path != NULL)
        {
            unlink(stdout_path);
            g_free(stdout_path);
            stdout_path = NULL;
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}


/*
 *	Adds the disk object specified by tar_path to the archive specified
 *	by arch_obj.
 */
gint EDVArchOPAdd(
        edv_core_struct *core_ptr, const gchar *arch_obj,
	gchar **tar_path, gint total_tar_paths,
        gchar ***new_obj_path, gint *total_new_obj_paths,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool recursive, gint compression, gbool dereference_links
)
{
        static gbool reenterent = FALSE;
        static gint status;
	gint i;
	gulong time_start = (gulong)time(NULL);
	gchar **ltar_path = NULL;
	gint total_ltar_paths = 0;

        /* Reset returns. */
	if(new_obj_path != NULL)
	    *new_obj_path = NULL;
	if(total_new_obj_paths != NULL)
	    *total_new_obj_paths = 0;
        /* Leave yes_to_all as whatever value the calling function set it. */

        if(reenterent)
        {
            last_error =
"System is busy, an operation is already in progress";
            return(-6);
        }
        else
        {
            reenterent = TRUE;
        }

        /* Reset last error pointer. */
        last_error = NULL;


        if((core_ptr == NULL) || (arch_obj == NULL) || (tar_path == NULL) ||
           (total_tar_paths <= 0) || (yes_to_all == NULL)
        )
        {
            reenterent = FALSE;
            return(-1);
        }

#define DO_FREE_LOCALS			\
{					\
 for(i = 0; i < total_ltar_paths; i++)	\
  g_free(ltar_path[i]);			\
 g_free(ltar_path);			\
 ltar_path = NULL;			\
 total_ltar_paths = 0;			\
}

	/* Make copy of target paths. */
	total_ltar_paths = total_tar_paths;
	ltar_path = (gchar **)g_realloc(
	    ltar_path, total_ltar_paths * sizeof(gchar *)
	);
	for(i = 0; i < total_ltar_paths; i++)
	{
	    ltar_path[i] = STRDUP(tar_path[i]);
	    /* Simplify target object path notation, stripping
	     * tailing deliminators.
	     */
	    EDVSimplifyPath(ltar_path[i]);
	}

        /* Begin adding the target objects to the archive. The adding
         * method will be determined by taking the extension of the
         * archive's name.
         */

        status = -1;

        if(EDVIsExtension(arch_obj, ".arj"))
        {
            status = EDVArchOPAddArj(
                core_ptr, arch_obj,
		ltar_path, total_ltar_paths,
                new_obj_path, total_new_obj_paths,
                toplevel, show_progress, interactive, yes_to_all,
                recursive, compression, dereference_links
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.Z"))
        {
            status = EDVArchOPAddTar(
                core_ptr, arch_obj,
                ltar_path, total_ltar_paths,
                new_obj_path, total_new_obj_paths,
                toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links,
                TRUE,		/* Is compress compressed. */
                FALSE,		/* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tgz .tar.gz"))
        {
            status = EDVArchOPAddTar(
                core_ptr, arch_obj,
                ltar_path, total_ltar_paths,
		new_obj_path, total_new_obj_paths,
                toplevel, show_progress, interactive, yes_to_all,
                recursive, compression, dereference_links,
                FALSE,          /* Not compress compressed. */
                TRUE,           /* Is gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.bz2"))
        {
            status = EDVArchOPAddTar(
                core_ptr, arch_obj,
                ltar_path, total_ltar_paths,
                new_obj_path, total_new_obj_paths,
                toplevel, show_progress, interactive, yes_to_all,
                recursive, compression, dereference_links,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                TRUE            /* Is bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar"))
        {
            status = EDVArchOPAddTar(
                core_ptr, arch_obj,
                ltar_path, total_ltar_paths,
                new_obj_path, total_new_obj_paths,
                toplevel, show_progress, interactive, yes_to_all,
                recursive, compression, dereference_links,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".zip"))
        {
            status = EDVArchOPAddZip(
                core_ptr, arch_obj,
                ltar_path, total_ltar_paths,
                new_obj_path, total_new_obj_paths,
                toplevel, show_progress, interactive, yes_to_all,
                recursive, compression, dereference_links
            );
        }
        else
        {
            last_error = "Unsupported archive format";
        }

        /* Record history. */
        EDVAppendHistory(
            core_ptr,
            EDV_HISTORY_ARCHIVE_OBJECT_ADD,
            time_start, (gulong)time(NULL),
            status,
            arch_obj,		/* Source full path is the archive. */
            (total_ltar_paths > 0) ?
		ltar_path[0] : NULL,	/* Target full path. */
            last_error          /* Use last_error as comment if not NULL. */
        );

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        reenterent = FALSE;
        return(status);
}


/*
 *      Procedure to extract using the arj application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPExtractArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_path,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories
)
{
        const gchar *prog_arj = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ARJ
        );
        const gchar *cstrptr;
        gint i, status;
        pid_t p;
        gchar *cmd = NULL, *stdout_path = NULL;
        edv_archive_object_struct *src_aobj_ptr;
        FILE *fp;
        gchar prev_working_dir[PATH_MAX];


        /* Record previous working dir and set new working dir. */
        *prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
        chdir(dest_path);

#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(stdout_path); \
 stdout_path = NULL; \
\
 /* Restore previous working dir. */ \
 chdir(prev_working_dir); \
}

        /* Begin generating extract command. */
        cmd = g_strdup_printf(
            "%s e -i -r -y \"%s\"",
            prog_arj, arch_obj
        );
        if(cmd == NULL)
        {
            last_error = "Unable to generate extract command";
            DO_FREE_LOCALS
            return(-3);
        }
        /* Append source objects to extract command string. */
        for(i = 0; i < total_src_aobjs; i++)
        {
            src_aobj_ptr = src_aobj[i];
            if(src_aobj_ptr == NULL)
                continue;
            if(src_aobj_ptr->full_path == NULL)
                continue;

            cmd = strcatalloc(cmd, " ");
            cmd = strcatalloc(cmd, src_aobj_ptr->full_path);
        }
        if(cmd == NULL)
        {
            last_error = "Unable to generate extract command";
            DO_FREE_LOCALS
            return(-3);
        }


        /* Generate tempory standard output file path. */
        stdout_path = EDVTmpName(NULL);


        /* Begin extracting. */

        status = 0;

        /* Execute extract command. */
        p = ExecO(cmd, stdout_path);
        if(p <= 0)
        {
            last_error = "Unable to execute extract command";
            DO_FREE_LOCALS
            return(-1);
        }
        else
        {
            /* Deallocate extract command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;
        }


        /* Open output file on this end for reading. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
#define buf_len         10000
            gint buf_pos = 0;
            gchar buf[buf_len];
            gbool need_break = FALSE;

            /* Begin reading output file. */
            while(1)
            {
                /* Update progress? */
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                /* Check if there is new data to be read from the output
                 * file.
                 */
                if(EDVArchOPFPHasData(fp))
                {
                    gint c;
                    gbool got_complete_line = FALSE;

                    /* Copy all available data from the current output
                     * file position to its end to the line buffer buf.
                     */
                    while(1)
                    {
                        c = fgetc(fp);
                        if(c == EOF)
                            break;

                        if((c == '\n') || (c == '\r'))
                        {
                            got_complete_line = TRUE;
                            if(buf_pos < buf_len)
                                buf[buf_pos] = '\0';
                            else
                                buf[buf_len - 1] = '\0';
                            buf_pos = 0;

                            break;
                        }

                        if(buf_pos < buf_len)
                        {
                            buf[buf_pos] = c;
                            buf_pos++;
                        }
                    }
                    /* Got a complete line from the output file and the
                     * progress dialog is mapped?
                     */
                    if(got_complete_line && show_progress &&
		       (strcasepfx(buf, "Extracting"))
                    )
                    {
                        gchar *strptr, *strptr2;
                        gchar *p1, *p2, *buf2, *new_path = NULL;

                        /* Set strptr to the start of the loaded line buffer
                         * and seek past the first "inflating:" or
                         * "creating:" prefix.
                         */
                        strptr = buf;
                        while(ISBLANK(*strptr))
                            strptr++;
                        while(!ISBLANK(*strptr) && (*strptr != '\0'))
                            strptr++;
                        while(ISBLANK(*strptr))
                            strptr++;

                        /* Cap the space character after path. */
                        strptr2 = strchr(strptr, ' ');
                        if(strptr2 != NULL)
                            *strptr2 = '\0';


                        cstrptr = PrefixPaths(dest_path, strptr);
                        if(cstrptr != NULL)
                        {
                            new_path = g_strdup(cstrptr);
                            StripPath(new_path);
                        }

                        /* Update progress dialog label. */
                        p1 = EDVCopyShortenPath(
                            strptr, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        p2 = EDVCopyShortenPath(
                            new_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        buf2 = g_strdup_printf(
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
                            p1, p2
                        );
                        EDVArchOPMapProgressDialogExtractUnknown(
                            buf2, toplevel, FALSE
                        );
                        g_free(buf2);
                        g_free(p1);
                        g_free(p2);


                        /* Append path to return list of disk object paths. */
                        if((new_obj_rtn != NULL) &&
                           (total_new_obj_rtns != NULL)
                        )
                        {
                            gint i = *total_new_obj_rtns;
                            *total_new_obj_rtns = i + 1;
                            *new_obj_rtn = (gchar **)g_realloc(
                                *new_obj_rtn,
                                (*total_new_obj_rtns) * sizeof(gchar *)
                            );
                            if(*new_obj_rtn == NULL)
                                *total_new_obj_rtns = 0;
                            else
                                (*new_obj_rtn)[i] = g_strdup(new_path);
                        }

                        g_free(new_path);

                        continue;
                    }
                }

                if(1)
                {
                    /* No new data to be read from output file, check if
                     * the need_break marker was already set by the previous
                     * loop. If it was set then we actually break out of
                     * this loop.
                     */
                    if(need_break)
                        break;

                    /* Check if add process has exited, if it has then
                     * we set need_break to TRUE. Which will be tested on
                     * the next loop if there is still no more data to be
                     * read.
                     */
                    if(!ExecProcessExists(p))
                        need_break = TRUE;
                }

                usleep(8000);
            }

            FClose(fp);
            fp = NULL;
#undef buf_len
        }

        /* Remove output file. */
        if(stdout_path != NULL)
        {
            unlink(stdout_path);
            g_free(stdout_path);
            stdout_path = NULL;
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *	Procedure to extract using the tar application.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchOPExtractTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_path,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories,
        gbool is_compress_compressed,
	gbool is_gzip_compressed,
	gbool is_bzip2_compressed
)
{
        const gchar *prog_tar = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_TAR
        );
        const gchar *prog_bunzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
        );
	const gchar *cstrptr;
	gint i, status;
	pid_t p;
	gchar *cmd = NULL, *stdout_path = NULL;
	edv_archive_object_struct *src_aobj_ptr;
        FILE *fp;
        gchar prev_working_dir[PATH_MAX];


        /* Record previous working dir and set new working dir. */
	*prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
        chdir(dest_path);

#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(stdout_path); \
 stdout_path = NULL; \
\
 /* Restore previous working dir. */ \
 chdir(prev_working_dir); \
}

	/* Begin generating extract command. */
        if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s -Z -x -v -f \"%s\"",
                prog_tar, arch_obj
            );
	else if(is_gzip_compressed)
	    cmd = g_strdup_printf(
		"%s -z -x -v -f \"%s\"",
		prog_tar, arch_obj
	    );
	else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s --use-compress-program=%s -x -v -f \"%s\"",
                prog_tar, prog_bunzip2, arch_obj
            );
	else
	    cmd = g_strdup_printf(
                "%s -x -v -f \"%s\"",
                prog_tar, arch_obj
            );
	if(cmd == NULL)
	{
	    last_error = "Unable to generate extract command";
	    DO_FREE_LOCALS
	    return(-3);
	}
	/* Append source objects to extract command string. */
	for(i = 0; i < total_src_aobjs; i++)
	{
	    src_aobj_ptr = src_aobj[i];
	    if(src_aobj_ptr == NULL)
		continue;
	    if(src_aobj_ptr->full_path == NULL)
		continue;

	    cmd = strcatalloc(cmd, " ");
	    cmd = strcatalloc(cmd, src_aobj_ptr->full_path);
	}
        if(cmd == NULL)
        {
            last_error = "Unable to generate extract command";
            DO_FREE_LOCALS
            return(-3);
        }


	/* Generate tempory standard output file path. */
	stdout_path = EDVTmpName(NULL);


        /* Begin extracting. */

	status = 0;

	/* Execute extract command. */
	p = ExecO(cmd, stdout_path);
	if(p <= 0)
	{
	    last_error = "Unable to execute extract command";
	    DO_FREE_LOCALS
	    return(-1);
	}
	else
	{
            /* Deallocate extract command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;
	}


        /* Open output file on this end for reading. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
#define buf_len         10000
            gint buf_pos = 0;
            gchar buf[buf_len];
            gbool need_break = FALSE;

            /* Begin reading output file. */
            while(1)
            {
                /* Update progress? */
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }


                /* Check if there is new data to be read from the output
                 * file.
                 */
                if(EDVArchOPFPHasData(fp))
                {
                    gint c;
                    gbool got_complete_line = FALSE;

                    /* Copy all available data from the current output
                     * file position to its end to the line buffer buf.
                     */
                    while(1)
                    {
                        c = fgetc(fp);
                        if(c == EOF)
                            break;

                        if((c == '\n') || (c == '\r'))
                        {
                            got_complete_line = TRUE;
                            if(buf_pos < buf_len)
                                buf[buf_pos] = '\0';
                            else
                                buf[buf_len - 1] = '\0';
                            buf_pos = 0;
                           break;
                        }

                        if(buf_pos < buf_len)
                        {
                            buf[buf_pos] = c;
                            buf_pos++;
                        }
                    }
                    /* Got a complete line from the output file and the
                     * progress dialog is mapped?
                     */
                    if(got_complete_line && show_progress)
                    {
			gchar *p1, *p2, *buf2, *new_path = NULL;


			cstrptr = PrefixPaths(dest_path, buf);
			if(cstrptr != NULL)
			{
			    new_path = g_strdup(cstrptr);
			    StripPath(new_path);
			}

                        /* Update progress dialog label. */
                        p1 = EDVCopyShortenPath(
                            buf, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        p2 = EDVCopyShortenPath(
                            new_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        buf2 = g_strdup_printf(
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
                            p1, p2
                        );
                        EDVArchOPMapProgressDialogExtractUnknown(
                            buf2, toplevel, FALSE
                        );
                        g_free(buf2);
                        g_free(p1);
                        g_free(p2);


                        /* Append path to return list of disk object paths. */
                        if((new_obj_rtn != NULL) &&
                           (total_new_obj_rtns != NULL)
                        )
                        {
                            gint i = *total_new_obj_rtns;
                            *total_new_obj_rtns = i + 1;
                            *new_obj_rtn = (gchar **)g_realloc(
                                *new_obj_rtn,
                                (*total_new_obj_rtns) * sizeof(gchar *)
                            );
                            if(*new_obj_rtn == NULL)
                                *total_new_obj_rtns = 0;
                            else
                                (*new_obj_rtn)[i] = g_strdup(new_path);
                        }

			g_free(new_path);

                        continue;
                    }
                }

                if(1)
                {
                    /* No new data to be read from output file, check if
                     * the need_break marker was already set by the previous
                     * loop. If it was set then we actually break out of
                     * this loop.
                     */
                    if(need_break)
                        break;

                    /* Check if add process has exited, if it has then
                     * we set need_break to TRUE. Which will be tested on
                     * the next loop if there is still no more data to be
                     * read.
                     */
                    if(!ExecProcessExists(p))
                        need_break = TRUE;
                }

                usleep(8000);
            }

            FClose(fp);
            fp = NULL;
#undef buf_len
        }

        /* Remove output file. */
        if(stdout_path != NULL)
        {
            unlink(stdout_path);
            g_free(stdout_path);
            stdout_path = NULL;
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

	return(status);
}

/*
 *      Procedure to extract using the unzip application.
 *
 *      Inputs assumed valid.
 */
static gint EDVArchOPExtractZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_path,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
        GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories
)
{
        const gchar *prog_unzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_UNZIP
        );
        const gchar *cstrptr;
        gint i, status;
        pid_t p;
        gchar *cmd = NULL, *stdout_path = NULL;
        edv_archive_object_struct *src_aobj_ptr;
        FILE *fp;
        gchar prev_working_dir[PATH_MAX];


        /* Record previous working dir and set new working dir. */
        *prev_working_dir = '\0';
        getcwd(prev_working_dir, PATH_MAX);
        chdir(dest_path);

#define DO_FREE_LOCALS  \
{ \
 g_free(cmd); \
 cmd = NULL; \
 g_free(stdout_path); \
 stdout_path = NULL; \
\
 /* Restore previous working dir. */ \
 chdir(prev_working_dir); \
}

        /* Begin generating extract command. */
	cmd = g_strdup_printf(
	    "%s -o \"%s\"",
            prog_unzip, arch_obj
        );
        if(cmd == NULL)
        {
            last_error = "Unable to generate extract command";
            DO_FREE_LOCALS
            return(-3);
        }
        /* Append source objects to extract command string. */
        for(i = 0; i < total_src_aobjs; i++)
        {
            src_aobj_ptr = src_aobj[i];
            if(src_aobj_ptr == NULL)
                continue;
            if(src_aobj_ptr->full_path == NULL)
                continue;

            cmd = strcatalloc(cmd, " ");
            cmd = strcatalloc(cmd, src_aobj_ptr->full_path);
        }
        if(cmd == NULL)
        {
            last_error = "Unable to generate extract command";
            DO_FREE_LOCALS
            return(-3);
        }


        /* Generate tempory standard output file path. */
        stdout_path = EDVTmpName(NULL);


        /* Begin extracting. */

        status = 0;

        /* Execute extract command. */
        p = ExecO(cmd, stdout_path);
        if(p <= 0)
        {
            last_error = "Unable to execute extract command";
            DO_FREE_LOCALS
            return(-1);
        }
        else
        {
            /* Deallocate extract command, it is no longer needed. */
            g_free(cmd);
            cmd = NULL;
        }


        /* Open output file on this end for reading. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
#define buf_len         10000
            gint buf_pos = 0, line_count = 0;
            gchar buf[buf_len];
            gbool need_break = FALSE;

            /* Begin reading output file. */
            while(1)
            {
                /* Update progress? */
                if(show_progress && ProgressDialogIsQuery())
                {
                    ProgressDialogUpdateUnknown(
                        NULL, NULL, NULL, NULL, TRUE
                    );
                    if(ProgressDialogStopCount() > 0)
                    {
                        kill(p, SIGINT);
                        p = 0;

                        status = -4;
                        break;
                    }
                }

                /* Check if there is new data to be read from the output
                 * file.
                 */
                if(EDVArchOPFPHasData(fp))
                {
                    gint c;
                    gbool got_complete_line = FALSE;

                    /* Copy all available data from the current output
                     * file position to its end to the line buffer buf.
                     */
                    while(1)
                    {
                        c = fgetc(fp);
                        if(c == EOF)
                            break;

                        if((c == '\n') || (c == '\r'))
                        {
                            got_complete_line = TRUE;
                            if(buf_pos < buf_len)
                                buf[buf_pos] = '\0';
                            else
                                buf[buf_len - 1] = '\0';
                            buf_pos = 0;

			    line_count++;	/* Count this line. */

                            break;
                        }

                        if(buf_pos < buf_len)
                        {
                            buf[buf_pos] = c;
                            buf_pos++;
                        }
                    }
                    /* Got a complete line from the output file and the
                     * progress dialog is mapped?
                     */
                    if(got_complete_line && show_progress && 
                       (line_count > 1)
		    )
                    {
                        gchar *strptr, *strptr2;
                        gchar *p1, *p2, *buf2, *new_path = NULL;

                        /* Set strptr to the start of the loaded line buffer
                         * and seek past the first "inflating:" or
			 * "creating:" prefix.
                         */
                        strptr = buf;
                        while(ISBLANK(*strptr))
                            strptr++;
                        while(!ISBLANK(*strptr) && (*strptr != '\0'))
                            strptr++;
                        while(ISBLANK(*strptr))
                            strptr++;

                        /* Cap the space character after path. */
                        strptr2 = strchr(strptr, ' ');
                        if(strptr2 != NULL)
                            *strptr2 = '\0';


                        cstrptr = PrefixPaths(dest_path, strptr);
                        if(cstrptr != NULL)
                        {
                            new_path = g_strdup(cstrptr);
                            StripPath(new_path);
                        }

                        /* Update progress dialog label. */
                        p1 = EDVCopyShortenPath(
                            strptr, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        p2 = EDVCopyShortenPath(
                            new_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        buf2 = g_strdup_printf(
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
                            p1, p2
                        );
                        EDVArchOPMapProgressDialogExtractUnknown(
                            buf2, toplevel, FALSE
                        );
                        g_free(buf2);
                        g_free(p1);
                        g_free(p2);


                        /* Append path to return list of disk object paths. */
                        if((new_obj_rtn != NULL) &&
                           (total_new_obj_rtns != NULL)
                        )
                        {
                            gint i = *total_new_obj_rtns;
                            *total_new_obj_rtns = i + 1;
                            *new_obj_rtn = (gchar **)g_realloc(
                                *new_obj_rtn,
                                (*total_new_obj_rtns) * sizeof(gchar *)
                            );
                            if(*new_obj_rtn == NULL)
                                *total_new_obj_rtns = 0;
                            else
                                (*new_obj_rtn)[i] = g_strdup(new_path);
                        }

                        g_free(new_path);

                        continue;
                    }
                }

                if(1)
                {
                    /* No new data to be read from output file, check if
                     * the need_break marker was already set by the previous
                     * loop. If it was set then we actually break out of
                     * this loop.
                     */
                    if(need_break)
                        break;

                    /* Check if add process has exited, if it has then
                     * we set need_break to TRUE. Which will be tested on
                     * the next loop if there is still no more data to be
                     * read.
                     */
                    if(!ExecProcessExists(p))
                        need_break = TRUE;
                }

                usleep(8000);
            }

            FClose(fp);
            fp = NULL;
#undef buf_len
        }

        /* Remove output file. */
        if(stdout_path != NULL)
        {
            unlink(stdout_path);
            g_free(stdout_path);
            stdout_path = NULL;
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}

/*
 *	Extracts the object specified in the archive object stat
 *	structure src_obj_stat from the archive object specified by arch_obj
 *	to the destination specified as tar_obj.
 *
 *	The tar_obj must specify the exact absolute path name of the object
 *	to extract to.
 *
 *	If new_obj_rtn is not NULL, then on success it's location will
 *	be set to a dynamically allocated string specifying the location
 *	of the extracted object. The calling function must deallocate
 *	this string.
 */
gint EDVArchOPExtract(
        edv_core_struct *core_ptr,
        const gchar *arch_obj,
        edv_archive_object_struct **src_aobj, gint total_src_aobjs,
        const gchar *dest_path,
        gchar ***new_obj_rtn, gint *total_new_obj_rtns,
	GtkWidget *toplevel,
        gbool show_progress, gbool interactive, gbool *yes_to_all,
        gbool preserve_directories
)
{
        static gbool reenterent = FALSE;
        static gint status;
	gulong time_start = (gulong)time(NULL);
	gchar *ldest_path = NULL;


        /* Reset returns. */
        if(new_obj_rtn != NULL)
            *new_obj_rtn = NULL;
        if(total_new_obj_rtns != NULL)
            *total_new_obj_rtns = 0;
        /* Leave yes_to_all as whatever value the calling function set it. */

        if(reenterent)
        {
            last_error =
"System is busy, an operation is already in progress";
            return(-6);
        }
        else
        {
            reenterent = TRUE;
        }

        /* Reset last error pointer. */
        last_error = NULL;


        if((core_ptr == NULL) || (arch_obj == NULL) ||
           (src_aobj == NULL) || (total_src_aobjs <= 0) ||
           (dest_path == NULL) || (yes_to_all == NULL)
        )
        {
            reenterent = FALSE;
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
 g_free(ldest_path); \
 ldest_path = NULL; \
}

	/* Make a copy of the destination path so we can simplify it. */
	ldest_path = g_strdup(dest_path);
	if(ldest_path == NULL)
	{
            last_error = "Unable to generate destination path";
            DO_FREE_LOCALS
            reenterent = FALSE;
            return(-2);
        }

        /* Simplify destination object path notation, stripping tailing
         * deliminators (which would probably exist if it were a
         * directory).
         */
        EDVSimplifyPath(ldest_path);


	/* Do overwrite check? */
	if(interactive && !(*yes_to_all))
	{
	    gint i;
	    const gchar *cstrptr;
	    gchar *tmp_path;
	    edv_archive_object_struct *aobj_ptr;
	    gbool got_cancel = FALSE;
	    struct stat lstat_buf;


	    /* Iterate through the given list of source archive object
	     * structures and check if any of them exists, prompt for
	     * overwrite.
	     */
	    for(i = 0; i < total_src_aobjs; i++)
	    {
		aobj_ptr = src_aobj[i];
		if(aobj_ptr == NULL)
		    continue;
		if(aobj_ptr->full_path == NULL)
		    continue;

		/* Generate full path to what the extracted disk object
		 * path should be.
		 */
		cstrptr = PrefixPaths(ldest_path, aobj_ptr->full_path);
		tmp_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
		if(tmp_path == NULL)
		    continue;

		StripPath(tmp_path);

		status = CDIALOG_RESPONSE_YES;
		/* Check if extracted version of the object exists. */
		if(!lstat(tmp_path, &lstat_buf))
		    status = EDVArchOPConfirmOverwriteExtract(
			core_ptr, toplevel,
			aobj_ptr->full_path,	/* Source archive object path. */
			tmp_path,		/* Target disk object path. */
			aobj_ptr,		/* Source archive object stats. */
			&lstat_buf		/* Target disk object local stats. */
		    );

		/* Deallocate copy of full path to destination disk object,
		 * it is no longer needed.
		 */
		g_free(tmp_path);

		/* Check user response. */
		switch(status)
		{
		  case CDIALOG_RESPONSE_YES_TO_ALL:
		    *yes_to_all = TRUE;
		  case CDIALOG_RESPONSE_YES:
		    break;

		  default:	/* All else assume cancel. */
		    got_cancel = TRUE;
		    break;
		}
		if(*yes_to_all)
		    break;
		if(got_cancel)
		    break;
	    }

	    /* User canceled? */
	    if(got_cancel)
	    {
                DO_FREE_LOCALS
                reenterent = FALSE;
		return(-4);
	    }
	}

	/* Check if the destination path does not exist, if it does not
	 * then we first need to update the list of new path returns
	 * with all non-existent compoent directories of the destination
	 * path and create all compoent directorires of the destination
	 * path.
	 */
	if(access(ldest_path, F_OK))
	{
	    /* Record each compoent of the destination path that does
	     * not exist as a new object in the new_obj_rtn list.
	     */
	    if((new_obj_rtn != NULL) && (total_new_obj_rtns != NULL))
	    {
		gint i;
		gchar *strptr = strchr(ldest_path, DIR_DELIMINATOR);

		/* Seek to first deliminator. */
		if(strptr != NULL)
		    strptr = strchr(strptr + 1, DIR_DELIMINATOR);

		/* Iterate through each path compoent starting from
		 * the first compoent after toplevel to the second to
		 * the last compoent. Note that we know last compoent
		 * does not exist.
		 */
		while(strptr != NULL)
		{
		    /* Tempory set deliminator to a null character then
		     * test if directory does not exist.
		     */
		    *strptr = '\0';
		    if(access(ldest_path, F_OK))
		    {
			/* This directory does not exist, so add it to the
			 * list of directories created.
			 */
			i = *total_new_obj_rtns;
                        *total_new_obj_rtns = i + 1;
                        *new_obj_rtn = (gchar **)g_realloc(
                            *new_obj_rtn,
                            (*total_new_obj_rtns) * sizeof(gchar *)
                        );
                        if(*new_obj_rtn == NULL)
                            *total_new_obj_rtns = 0;
                        else
                            (*new_obj_rtn)[i] = g_strdup(ldest_path);
		    }
		    /* Restore deliminator. */
		    *strptr = DIR_DELIMINATOR;
		    /* Seek to next deliminator (if any). */
		    strptr = strchr(strptr + 1, DIR_DELIMINATOR);
		}

		/* Also add the complete destination as a path added
		 * since we know it does not exist but it would not be
		 * checked in the above loop.
		 */
                i = *total_new_obj_rtns;
                *total_new_obj_rtns = i + 1;
                *new_obj_rtn = (gchar **)g_realloc(
                    *new_obj_rtn,
                    (*total_new_obj_rtns) * sizeof(gchar *)
                );
                if(*new_obj_rtn == NULL)
                    *total_new_obj_rtns = 0;
                else
                    (*new_obj_rtn)[i] = g_strdup(ldest_path);
	    }

	    /* Destination does not exist, need to create it. */
	    if(rmkdir(ldest_path, S_IRUSR | S_IWUSR | S_IXUSR))
	    {
		last_error = "Unable to create destination path";
		DO_FREE_LOCALS
		reenterent = FALSE;
		return(-2);
	    }
	}


	/* Begin extracting the source object to the destination specified
	 * by dest_obj from the archive object arch_obj. The extracting
	 * method will be determined by taking the extension of the
	 * archive object's name.
	 */

	status = -1;

        if(EDVIsExtension(arch_obj, ".arj"))
        {
            status = EDVArchOPExtractArj(
                core_ptr, arch_obj, src_aobj, total_src_aobjs,
                ldest_path, new_obj_rtn, total_new_obj_rtns,
                toplevel, show_progress, interactive, yes_to_all,
                preserve_directories
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.Z"))
        {
            status = EDVArchOPExtractTar(
                core_ptr, arch_obj, src_aobj, total_src_aobjs,
                ldest_path, new_obj_rtn, total_new_obj_rtns,
                toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
		TRUE,		/* Is compress compressed. */
                FALSE,		/* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
	else if(EDVIsExtension(arch_obj, ".tgz .tar.gz"))
	{
	    status = EDVArchOPExtractTar(
		core_ptr, arch_obj, src_aobj, total_src_aobjs,
		ldest_path, new_obj_rtn, total_new_obj_rtns,
		toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
                FALSE,		/* Not compress compressed. */
		TRUE,		/* Is gzip compressed. */
		FALSE		/* Not bzip2 compressed. */
	    );
	}
        else if(EDVIsExtension(arch_obj, ".tar.bz2"))
        {
            status = EDVArchOPExtractTar(
                core_ptr, arch_obj, src_aobj, total_src_aobjs,
                ldest_path, new_obj_rtn, total_new_obj_rtns,
                toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
                FALSE,          /* Not compress compressed. */
                FALSE,		/* Not gzip compressed. */
		TRUE		/* Is bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar"))
        {
            status = EDVArchOPExtractTar(
                core_ptr, arch_obj, src_aobj, total_src_aobjs,
                ldest_path, new_obj_rtn, total_new_obj_rtns,
                toplevel, show_progress, interactive, yes_to_all,
		preserve_directories,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
	else if(EDVIsExtension(arch_obj, ".zip"))
        {
	    status = EDVArchOPExtractZip(
                core_ptr, arch_obj, src_aobj, total_src_aobjs,
                ldest_path, new_obj_rtn, total_new_obj_rtns,
                toplevel, show_progress, interactive, yes_to_all,
		preserve_directories
	    );
	}
	else
	{
	    last_error = "Unsupported archive format";
	}

        /* Record history. */
        EDVAppendHistory(
            core_ptr,
            EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT,
            time_start, (gulong)time(NULL),
            status,
            arch_obj,           /* Source full path is the archive. */
	    ldest_path,		/* Extract destination full path. */
            last_error          /* Use last_error as comment if not NULL. */
        );

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

	reenterent = FALSE;
	return(status);
}
