#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvarch.h"
#include "edvcfg.h"
#include "endeavour.h"
#include "edvarchfio.h"
#include "edvutils.h"
#include "edvcfglist.h"
#include "config.h"


/* Line parsing. */
static void EDVArchFIOParseLineArj(
        edv_archive_object_struct *obj, const gchar *buf
);
static void EDVArchFIOParseLineRPM(
        edv_archive_object_struct *obj, const gchar *buf
);
static void EDVArchFIOParseLineTar(
	edv_archive_object_struct *obj, const gchar *buf
);
static void EDVArchFIOParseLineZip(
        edv_archive_object_struct *obj, const gchar *buf
);

/* Single object stat. */
static edv_archive_object_struct *EDVArchFIOStatArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
);
static edv_archive_object_struct *EDVArchFIOStatRPM(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
);
static edv_archive_object_struct *EDVArchFIOStatTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path,
	gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
);
static edv_archive_object_struct *EDVArchFIOStatZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
);
edv_archive_object_struct *EDVArchFIOStat(
	edv_core_struct *core_ptr,
	const gchar *arch_obj, const gchar *path
);


/* All objects listing stats. */
static edv_archive_object_struct **EDVArchFIOGetListingArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
);
static edv_archive_object_struct **EDVArchFIOGetListingRPM(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
);
static edv_archive_object_struct **EDVArchFIOGetListingTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
);
static edv_archive_object_struct **EDVArchFIOGetListingZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
);
edv_archive_object_struct **EDVArchFIOGetListing(
	edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
	const gchar **path, gint npaths
);



/*
 *	Parses the given null terminated string buf who is Arj archive
 *      format.
 */
static void EDVArchFIOParseLineArj(
        edv_archive_object_struct *obj, const gchar *buf
)
{
        gint c;
        const gchar *buf_ptr;
        struct tm mtime_buf;


        if((obj == NULL) || (buf == NULL))
            return;

        memset(&mtime_buf, 0x00, sizeof(struct tm));

        buf_ptr = buf;


        /* Sample format line:

cookie  63  63 1.000 02-02-09 17:23:27 -rw------- ---  +0

	*/

        /* Full path. */
        if(*buf_ptr != '\0')
        {
            gchar *strptr;
            gchar *tmp_full_path = g_strdup(buf_ptr);


            /* Chop tailing segments. */
            strptr = strchr(tmp_full_path, ' ');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\t');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\n');
            if(strptr != NULL)
                *strptr = '\0';


            /* Set new full path. */
            g_free(obj->full_path);
            obj->full_path = g_strdup(tmp_full_path);


            /* Strip tailing deliminators on the tempory full path and get
             * the name portion of it.
             */
            StripPath(tmp_full_path);

            /* Get name from the simplified full_path. */
            strptr = strrchr(tmp_full_path, DIR_DELIMINATOR);
            if(strptr == NULL)
                strptr = tmp_full_path;
            else
                strptr++;
            g_free(obj->name);
            obj->name = g_strdup(strptr);

            /* Deallocate tempory full path. */
            g_free(tmp_full_path);
            tmp_full_path = NULL;


            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }

        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Size. */
        obj->size = (gulong)atol(buf_ptr);

        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* Compressed size (skip it). */
        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;

        /* Ratio (skip it). */
        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Year-month-date. */
        if(*buf_ptr != '\0')
        {
	    /* Only two digit year. */
            mtime_buf.tm_year = atoi(buf_ptr) + 100;

            while((*buf_ptr != '-') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == '-')
                buf_ptr++;

            mtime_buf.tm_mon = atoi(buf_ptr) - 1;

            while((*buf_ptr != '-') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == '-')
                buf_ptr++;

            mtime_buf.tm_mday = atoi(buf_ptr);

            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;

        }

        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Hour:minute:second. */
        if(*buf_ptr != '\0')
        {
            mtime_buf.tm_hour = atoi(buf_ptr);

            while((*buf_ptr != ':') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == ':')
                buf_ptr++;

            mtime_buf.tm_min = atoi(buf_ptr);

            while((*buf_ptr != ':') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == ':')
                buf_ptr++;

            mtime_buf.tm_sec = atoi(buf_ptr);

            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }

        /* Parse time compoents and generate system time since we now
         * have enough information.
         */
        obj->access_time = (gulong)mktime(&mtime_buf);
        obj->modify_time = (gulong)mktime(&mtime_buf);
        obj->change_time = (gulong)mktime(&mtime_buf);

        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Type. */
        c = *buf_ptr;
        if(c != '\0')
        {
            if(c == 'd')
                obj->type = EDV_OBJECT_TYPE_DIRECTORY;
            else if(c == 'l')
                obj->type = EDV_OBJECT_TYPE_LINK;
            else if(c == 'p')
                obj->type = EDV_OBJECT_TYPE_FIFO;
            else if(c == 'b')
                obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
            else if(c == 'c')
                obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
            else if(c == 's')
                obj->type = EDV_OBJECT_TYPE_SOCKET;
            else
                obj->type = EDV_OBJECT_TYPE_FILE;

            buf_ptr++;
        }

        /* Permissions. */
        obj->permissions = 0x00000000;

        /* Owner read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_UREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_UWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_UEXECUTE : 0;
            buf_ptr++;
        }

        /* Group read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_GREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_GWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_GEXECUTE : 0;
            buf_ptr++;
        }

        /* Anonymous read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_AREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_AWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_AEXECUTE : 0;
            buf_ptr++;
        }


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* Skip GUA and BPMGS fields. */
}

/*
 *      Parses the given null terminated string buf who is RPM archive
 *      format.
 */
static void EDVArchFIOParseLineRPM(
        edv_archive_object_struct *obj, const gchar *buf
)
{
        gint c;
        const gchar *buf_ptr;
        struct tm mtime_buf;


        if((obj == NULL) || (buf == NULL))
            return;

        memset(&mtime_buf, 0x00, sizeof(struct tm));

        buf_ptr = buf;


        /* Sample format line:

-rw-r--r--    1 root    root            27666 Jan  4  2000 /var/log/1.txt

         */


        /* Type and permissions string (this is one string with no
	 * deliminators.
	 */

        /* Type. */
        c = *buf_ptr;
        if(c != '\0')
        {
            if(c == 'd')
                obj->type = EDV_OBJECT_TYPE_DIRECTORY;
            else if(c == 'l')
                obj->type = EDV_OBJECT_TYPE_LINK;
            else if(c == 'p')
                obj->type = EDV_OBJECT_TYPE_FIFO;
            else if(c == 'b')
                obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
            else if(c == 'c')
                obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
            else if(c == 's')
                obj->type = EDV_OBJECT_TYPE_SOCKET;
            else
                obj->type = EDV_OBJECT_TYPE_FILE;

            buf_ptr++;
        }

        /* Permissions. */
        obj->permissions = 0x00000000;

        /* Owner read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_UREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_UWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_UEXECUTE : 0;
            buf_ptr++;
        }

        /* Group read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_GREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_GWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_GEXECUTE : 0;
            buf_ptr++;
        }

        /* Anonymous read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_AREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_AWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_AEXECUTE : 0;
            buf_ptr++;
        }

        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* This is number, not sure what it is but just seek past it. */
        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Owner and group strings, deliminated by a space in between. */
        if(*buf_ptr != '\0')
        {
            gchar *strptr, *strptr2;


            /* Get owner. */
	    g_free(obj->owner_name);
	    obj->owner_name = strptr = g_strdup(buf_ptr);
            strptr2 = strchr(strptr, ' ');
            if(strptr2 != NULL)
                *strptr2 = '\0';
            strptr2 = strchr(strptr, '\t');
            if(strptr2 != NULL)
                *strptr2 = '\0';

	    /* Seek to group. */
            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
            while(ISBLANK(*buf_ptr))
                buf_ptr++;

            /* Get group. */
            g_free(obj->group_name);
            obj->group_name = strptr = g_strdup(buf_ptr);
            strptr2 = strchr(strptr, ' ');
            if(strptr2 != NULL)
                *strptr2 = '\0';
            strptr2 = strchr(strptr, '\t');
            if(strptr2 != NULL)
                *strptr2 = '\0';

	    /* Seek past group to first blank character. */
            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }


        while(ISBLANK(*buf_ptr))
            buf_ptr++;

        /* Size. */
        obj->size = (gulong)atol(buf_ptr);

        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* These next three strings are month, day, and year. Separated by
	 * blank characters but the names are verbose and thus too difficult
	 * to parse.
	 */
	/* Skip month. */
        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;

	/* Skip day. */
        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;

	/* Skip year. */
        while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
            buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;

        /* Full path. */
        if(*buf_ptr != '\0')
        {
            gchar *strptr;
            gchar *tmp_full_path = g_strdup(buf_ptr);


            /* Chop tailing segments. */
            strptr = strchr(tmp_full_path, ' ');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\t');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\n');
            if(strptr != NULL)
                *strptr = '\0';


            /* Set new full path. */
            g_free(obj->full_path);
            obj->full_path = g_strdup(tmp_full_path);


            /* Strip tailing deliminators on the tempory full path and get
             * the name portion of it.
             */
            StripPath(tmp_full_path);

            /* Get name from the simplified full_path. */
            strptr = strrchr(tmp_full_path, DIR_DELIMINATOR);
            if(strptr == NULL)
                strptr = tmp_full_path;
            else
                strptr++;
            g_free(obj->name);
            obj->name = g_strdup(strptr);


            /* Deallocate tempory full path. */
            g_free(tmp_full_path);
            tmp_full_path = NULL;


            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* If this object is a link then we need to parse the link
         * destination.
         */
        if(obj->type == EDV_OBJECT_TYPE_LINK)
        {
            gchar *strptr;


            /* Seek past the "-> " characters. */
            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
            while(ISBLANK(*buf_ptr))
                buf_ptr++;

            /* Get link to destination string and cap end. */
            g_free(obj->linked_to);
            obj->linked_to = g_strdup(buf_ptr);

            strptr = strchr(obj->linked_to, ' ');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(obj->linked_to, '\t');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(obj->linked_to, '\n');
            if(strptr != NULL)
                *strptr = '\0';


            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }
}


/*
 *	Parses the given null terminated string buf who is Tar archive
 *	format.
 */
static void EDVArchFIOParseLineTar(
        edv_archive_object_struct *obj, const gchar *buf
)
{
	gint c;
	const gchar *buf_ptr;
	struct tm mtime_buf;


	if((obj == NULL) || (buf == NULL))
	    return;

	memset(&mtime_buf, 0x00, sizeof(struct tm));

	buf_ptr = buf;


	/* Sample format line:

lrwxrwxrwx root/root   0 2001-11-19 02:57 file.ext -> link.dest

	 */


        /* Type and permissions string (this is one string with no
         * deliminators.
         */

	/* Type. */
	c = *buf_ptr;
	if(c != '\0')
	{
	    if(c == 'd')
		obj->type = EDV_OBJECT_TYPE_DIRECTORY;
            else if(c == 'l')
                obj->type = EDV_OBJECT_TYPE_LINK;
            else if(c == 'p')
                obj->type = EDV_OBJECT_TYPE_FIFO;
            else if(c == 'b')
                obj->type = EDV_OBJECT_TYPE_DEVICE_BLOCK;
            else if(c == 'c')
                obj->type = EDV_OBJECT_TYPE_DEVICE_CHARACTER;
            else if(c == 's')
                obj->type = EDV_OBJECT_TYPE_SOCKET;
	    else
		obj->type = EDV_OBJECT_TYPE_FILE;

	    buf_ptr++;
	}

	/* Permissions. */
	obj->permissions = 0x00000000;

        /* Owner read/write/execute. */
	c = *buf_ptr;
        if(c != '\0')
        {
	    obj->permissions |= (c == 'r') ? EDV_PERMISSION_UREAD : 0;
	    buf_ptr++;
	}
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_UWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_UEXECUTE : 0;
            buf_ptr++;
        }

        /* Group read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_GREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_GWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_GEXECUTE : 0;
            buf_ptr++;
        }

	/* Anonymous read/write/execute. */
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'r') ? EDV_PERMISSION_AREAD : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'w') ? EDV_PERMISSION_AWRITE : 0;
            buf_ptr++;
        }
        c = *buf_ptr;
        if(c != '\0')
        {
            obj->permissions |= (c == 'x') ? EDV_PERMISSION_AEXECUTE : 0;
            buf_ptr++;
        }



	while(ISBLANK(*buf_ptr))
	    buf_ptr++;


	/* Owner and group string, separated by a '/' in between. */
	if(*buf_ptr != '\0')
	{
	    gchar *strptr;
	    gchar s[1024];

	    /* Get owner. */
	    strncpy(s, buf_ptr, 1024);
	    s[1024 - 1] = '\0';

	    strptr = strchr(s, ' ');
	    if(strptr != NULL)
		*strptr = '\0';
            strptr = strchr(s, '\t');
            if(strptr != NULL)
                *strptr = '\0';

	    strptr = strchr(s, '/');
	    if(strptr != NULL)
	    {
		*strptr = '\0';

		g_free(obj->owner_name);
		obj->owner_name = g_strdup(s);

		g_free(obj->group_name);
		obj->group_name = g_strdup(strptr + 1);
	    }

	    while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
		buf_ptr++;
	}


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* Size. */
	obj->size = (gulong)atol(buf_ptr);


	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
	    buf_ptr++;
	while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* Year/month/date. */
	if(*buf_ptr != '\0')
	{
            mtime_buf.tm_year = atoi(buf_ptr) - 1900;

	    while((*buf_ptr != '-') && (*buf_ptr != '\0'))
		buf_ptr++;
            while(*buf_ptr == '-')
		buf_ptr++;

            mtime_buf.tm_mon = atoi(buf_ptr) - 1;

            while((*buf_ptr != '-') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == '-')
                buf_ptr++;

	    mtime_buf.tm_mday = atoi(buf_ptr);

	    while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;

	}


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Hour and minute. */
        if(*buf_ptr != '\0')
        {
            mtime_buf.tm_hour = atoi(buf_ptr);

            while((*buf_ptr != ':') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == ':')
                buf_ptr++;

            mtime_buf.tm_min = atoi(buf_ptr);

            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }

	/* Parse time compoents and generate system time since we now
	 * have enough information.
	 */
	obj->access_time = (gulong)mktime(&mtime_buf);
	obj->modify_time = (gulong)mktime(&mtime_buf);
	obj->change_time = (gulong)mktime(&mtime_buf);


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* Full path. */
	if(*buf_ptr != '\0')
	{
	    gchar *strptr;
	    gchar *tmp_full_path = g_strdup(buf_ptr);


	    /* Chop tailing segments. */
	    strptr = strchr(tmp_full_path, ' ');
	    if(strptr != NULL)
		*strptr = '\0';
            strptr = strchr(tmp_full_path, '\t');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\n');
            if(strptr != NULL)
                *strptr = '\0';


            /* Set new full path. */
            g_free(obj->full_path);
            obj->full_path = g_strdup(tmp_full_path);


	    /* Strip tailing deliminators on the tempory full path and get
	     * the name portion of it.
	     */
	    StripPath(tmp_full_path);

	    /* Get name from the simplified full_path. */
	    strptr = strrchr(tmp_full_path, DIR_DELIMINATOR);
	    if(strptr == NULL)
		strptr = tmp_full_path;
	    else
		strptr++;
	    g_free(obj->name);
	    obj->name = g_strdup(strptr);


	    /* Deallocate tempory full path. */
	    g_free(tmp_full_path);
	    tmp_full_path = NULL;


            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
	}


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* If this object is a link then we need to parse the link 
	 * destination.
	 */
	if(obj->type == EDV_OBJECT_TYPE_LINK)
	{
	    gchar *strptr;

	    /* Seek past the "-> " characters. */
	    while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
	    while(ISBLANK(*buf_ptr))
		buf_ptr++;

            /* Get link to destination string and cap end. */
	    g_free(obj->linked_to);
	    obj->linked_to = g_strdup(buf_ptr);

            strptr = strchr(obj->linked_to, ' ');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(obj->linked_to, '\t');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(obj->linked_to, '\n');
            if(strptr != NULL)
                *strptr = '\0';


	    while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
	}
}

/*
 *      Parses the given null terminated string buf who is PKZip archive
 *      format.
 */
static void EDVArchFIOParseLineZip(
        edv_archive_object_struct *obj, const gchar *buf
)
{
        const gchar *buf_ptr;
        struct tm mtime_buf;


        if((obj == NULL) || (buf == NULL))
            return;

        memset(&mtime_buf, 0x00, sizeof(struct tm));

        buf_ptr = buf;


	/* Sample format line:

396300  01-27-102  01:29   den2/main.o

	 */


	/* Size. */
	obj->size = (gulong)atol(buf_ptr);

	while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
	    buf_ptr++;
        while(ISBLANK(*buf_ptr))
            buf_ptr++;


	/* Date. */
	if(*buf_ptr != '\0')
	{
	    mtime_buf.tm_mon = atoi(buf_ptr) - 1;

            while((*buf_ptr != '-') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == '-')
                buf_ptr++;


            mtime_buf.tm_mday = atoi(buf_ptr);

            while((*buf_ptr != '-') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == '-')
                buf_ptr++;


	    /* The year is relative to 1900, note that there is a Y2K 
	     * problem here but we are compensating for it.
	     */
            mtime_buf.tm_year = atoi(buf_ptr);

            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Hour and minute. */
        if(*buf_ptr != '\0')
        {
            mtime_buf.tm_hour = atoi(buf_ptr);

            while((*buf_ptr != ':') && (*buf_ptr != '\0'))
                buf_ptr++;
            while(*buf_ptr == ':')
                buf_ptr++;

            mtime_buf.tm_min = atoi(buf_ptr);

            while(!ISBLANK(*buf_ptr) && (*buf_ptr != '\0'))
                buf_ptr++;
        }

        /* Parse time compoents and generate system time since we now
         * have enough information.
         */
        obj->access_time = (gulong)mktime(&mtime_buf);
        obj->modify_time = (gulong)mktime(&mtime_buf);
        obj->change_time = (gulong)mktime(&mtime_buf);


        while(ISBLANK(*buf_ptr))
            buf_ptr++;


        /* Name. */
        if(*buf_ptr != '\0')
        {
            gint len;
            gchar *strptr;
            gchar *tmp_full_path = g_strdup(buf_ptr);


            /* Chop tailing segments of the tempory full path. */
            strptr = strchr(tmp_full_path, ' ');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\t');
            if(strptr != NULL)
                *strptr = '\0';
            strptr = strchr(tmp_full_path, '\n');
            if(strptr != NULL)
                *strptr = '\0';

            /* Set new full path. */
            g_free(obj->full_path);
            obj->full_path = g_strdup(tmp_full_path);


            /* Strip tailing deliminators on the tempory full path and get
             * the name portion of it.
             */
            StripPath(tmp_full_path);

            /* Get name from the simplified full_path. */
            strptr = strrchr(tmp_full_path, DIR_DELIMINATOR);
            if(strptr == NULL)
                strptr = tmp_full_path;
            else
                strptr++;
            g_free(obj->name);
            obj->name = g_strdup(strptr);


            /* Deallocate modified tempory full path. */
            g_free(tmp_full_path);
            tmp_full_path = NULL;


            /* Determine object type from the file path, if it has a
             * tailing deliminator then it is a directory. Otherwise it
             * is a file.
             */
            obj->type = EDV_OBJECT_TYPE_FILE;
            len = strlen(obj->full_path);
            if(len > 0)
            {
                if(obj->full_path[len - 1] == DIR_DELIMINATOR)
                    obj->type = EDV_OBJECT_TYPE_DIRECTORY;
            }
        }


	/* Because the PKZip format does not specify permissions, we need
	 * to set the assumed permissions.
	 */
	if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
	    obj->permissions = (
		EDV_PERMISSION_UREAD | EDV_PERMISSION_UWRITE |
		    EDV_PERMISSION_UEXECUTE |
		EDV_PERMISSION_GREAD | EDV_PERMISSION_GEXECUTE |
                EDV_PERMISSION_AREAD | EDV_PERMISSION_AEXECUTE
	    );
	else
            obj->permissions = (
                EDV_PERMISSION_UREAD | EDV_PERMISSION_UWRITE |
                EDV_PERMISSION_GREAD |
		EDV_PERMISSION_AREAD
            );

}


/*
 *      Fetches statistics from the archive object specified by path in
 *      the archive arch_obj in Arj format.
 */
static edv_archive_object_struct *EDVArchFIOStatArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
)
{
        const gchar *prog_arj = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ARJ
        );
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct *obj = NULL;


        /* Begin generating stat command. */
        cmd = g_strdup_printf(
            "%s l -i \"%s\"",
            prog_arj, arch_obj
        );
        if(cmd == NULL)
            return(obj);

        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);


        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
            return(obj);
        }


        /* Read one line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
            gint i;
            gchar *strptr, buf[10000];

            /* Read past the first 9 lines. */
            for(i = 0; i < 9; i++)
            {
                if(fgets(buf, 10000, fp) == NULL)
                    break;
            }

            /* Able to read up to line 9? */
            if((i >= 9) ? (fgets(buf, 10000, fp) != NULL) : FALSE)
            {
                buf[10000 - 1] = '\0';

                strptr = buf;
                while(ISBLANK(*strptr))
                    strptr++;

                obj = EDVArchObjectNew();
                EDVArchFIOParseLineArj(obj, strptr);
            }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

        return(obj);
}

/*
 *      Fetches statistics from the archive object specified by path in
 *      the archive arch_obj in RPM format.
 */
static edv_archive_object_struct *EDVArchFIOStatRPM(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
)
{
	const gchar *prog_rpm = EDVCFGItemListGetValueS(
	    core_ptr->cfg_list, EDV_CFG_PARM_PROG_RPM
	);
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct *obj = NULL;


        /* Begin generating stat command. */
	cmd = g_strdup_printf(
	    "%s -q -l -p -v \"%s\"",
	    prog_rpm, arch_obj
	);
        if(cmd == NULL)
            return(obj);

        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);


        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
            return(obj);
        }


        /* Read one line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
	    gint i;
	    gchar *strptr, *strptr2;
	    gchar buf[10000];

            /* Begin reading each line and get stats of the line that
	     * contains the given path.
	     */
	    while(fgets(buf, 10000, fp) != NULL)
	    {
                buf[10000 - 1] = '\0';

		strptr = buf;

		/* Skip initial spaces. */
		while(ISBLANK(*strptr))
		    strptr++;

		/* Skip no more than 8 fields to position strptr2 at
                 * the start of the full path.
                 */
		strptr2 = strptr;
		for(i = 0; i < 8; i++)
		{
                    while(!ISBLANK(*strptr2) && (*strptr2 != '\0'))
                        strptr2++;
                    while(ISBLANK(*strptr2))
                        strptr2++;
		}

		if(strpfx(strptr2, path))
		{
		    obj = EDVArchObjectNew();
		    EDVArchFIOParseLineRPM(obj, strptr);
		    break;
		}
            }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

        return(obj);
}

/*
 *	Fetches statistics from the archive object specified by path in
 *	the archive arch_obj in Tar format.
 */
static edv_archive_object_struct *EDVArchFIOStatTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
)
{
        const gchar *prog_tar = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_TAR
        );
        const gchar *prog_bunzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
        );
	FILE *fp;
	pid_t p;
	gchar *cmd, *stdout_path;
	edv_archive_object_struct *obj = NULL;


        /* Begin generating stat command. */
	if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s -Z -t -v -f \"%s\" \"%s\"",
                prog_tar, arch_obj, path
            );
        else if(is_gzip_compressed)
            cmd = g_strdup_printf(
                "%s -z -t -v -f \"%s\" \"%s\"",
                prog_tar, arch_obj, path
            );
        else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s --use-compress-program=%s -t -v -f \"%s\" \"%s\"",
                prog_tar, prog_bunzip2, arch_obj, path
            );
        else
            cmd = g_strdup_printf(
                "%s -t -v -f \"%s\" \"%s\"",
                prog_tar, arch_obj, path
            );
        if(cmd == NULL)
	    return(obj);

        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);


        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
	if(p <= 0)
	{
	    g_free(cmd);
	    g_free(stdout_path);
	    return(obj);
	}


	/* Read one line from the output file. */
	fp = FOpen(stdout_path, "rb");
	if(fp != NULL)
	{
	    gchar *strptr, buf[10000];

	    /* Read just the first line. */
	    if(fgets(buf, 10000, fp) != NULL)
	    {
		buf[10000 - 1] = '\0';

		/* Get rid of the newline character if any. */
		strptr = strchr(buf, '\n');
		if(strptr != NULL)
		    *strptr = '\0';

		/* Create a new archive object structure and parse the
		 * loaded line to it.
		 */
		obj = EDVArchObjectNew();
		EDVArchFIOParseLineTar(obj, buf);
	    }

	    FClose(fp);
	}

	/* Remove output file if it exists. */
	if(stdout_path != NULL)
	    unlink(stdout_path);

	g_free(cmd);
	g_free(stdout_path);

	return(obj);
}

/*
 *      Fetches statistics from the archive object specified by path in
 *      the archive arch_obj in PKZip format.
 */
static edv_archive_object_struct *EDVArchFIOStatZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
)
{
        const gchar *prog_unzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_UNZIP
        );
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct *obj = NULL;


        /* Begin generating stat command. */
	cmd = g_strdup_printf(
	    "%s -l \"%s\" \"%s\"",
	    prog_unzip, arch_obj, path
	);
        if(cmd == NULL)
            return(obj);

        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);


        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
            return(obj);
        }


        /* Read one line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
	    gint i;
	    const gchar *hdelim = " --------    ----   ----    ----";
            gchar *strptr, buf[10000];

	    /* Read untill we encounter the last header line. */
	    for(i = 0; 1; i++)
            {
                if(fgets(buf, 10000, fp) == NULL)
                    break;

		buf[10000 - 1] = '\0';
		if(strpfx(buf, hdelim))
		    break;
	    }

	    /* Able to read up to line 3? */
	    if((i >= 3) ? (fgets(buf, 10000, fp) != NULL) : FALSE)
	    {
                buf[10000 - 1] = '\0';

		strptr = buf;
		while(ISBLANK(*strptr))
		    strptr++;

		obj = EDVArchObjectNew();
		EDVArchFIOParseLineZip(obj, strptr);
	    }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

        return(obj);
}

/*
 *      Fetches statistics from the archive object specified by path in
 *      the archive arch_obj.
 *
 *	The returned object must be deallocated by the calling function.
 *	Can return NULL on error.
 */
edv_archive_object_struct *EDVArchFIOStat(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, const gchar *path
)
{
	edv_archive_object_struct *obj = NULL;
	struct stat stat_buf;


	if((core_ptr == NULL) || (arch_obj == NULL) || (path == NULL))
	    return(obj);

        /* Get stats of the archive object, making sure it exists and is
         * not a directory.
         */
        if(stat(arch_obj, &stat_buf))
            return(obj);
        if(S_ISDIR(stat_buf.st_mode))
            return(obj);


        if(EDVIsExtension(arch_obj, ".arj"))
        {
            obj = EDVArchFIOStatArj(
                core_ptr, arch_obj, path
            );
        }
	else if(EDVIsExtension(arch_obj, ".rpm"))
        {
            obj = EDVArchFIOStatRPM(
                core_ptr, arch_obj, path
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.Z"))
        {
            obj = EDVArchFIOStatTar(
                core_ptr, arch_obj, path,
                TRUE,		/* Is compress compressed. */
                FALSE,		/* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tgz .tar.gz"))
        {
	    obj = EDVArchFIOStatTar(
		core_ptr, arch_obj, path,
		FALSE,          /* Not compress compressed. */
		TRUE,		/* Is gzip compressed. */
		FALSE		/* Not bzip2 compressed. */
            );
	}
        else if(EDVIsExtension(arch_obj, ".tar.bz2"))
        {
            obj = EDVArchFIOStatTar(
                core_ptr, arch_obj, path,
                FALSE,          /* Not compress compressed. */
                FALSE,		/* Not gzip compressed. */
                TRUE		/* Is bzip2 compressed. */
            );
        }
	else if(EDVIsExtension(arch_obj, ".tar"))
        {
            obj = EDVArchFIOStatTar(
                core_ptr, arch_obj, path,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".zip"))
        {
            obj = EDVArchFIOStatZip(
                core_ptr, arch_obj, path
            );
        }


	return(obj);
}


/*
 *      Returns a listing of all objects in the Arj archive.
 */
static edv_archive_object_struct **EDVArchFIOGetListingArj(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
)
{
        const gchar *prog_arj = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_ARJ
        );
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct **list = NULL, *obj;


        /* Begin generating listing command. */
        cmd = g_strdup_printf(
            "%s l -i \"%s\"",
            prog_arj, arch_obj
        );
        if(cmd == NULL)
            return(list);


        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);

        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
            return(list);
        }

        /* Read each line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
            gbool got_match;
            gint i;
            gchar *strptr, *strptr2, *strptr3;
            gchar buf[10000];


            /* Read past the first 9 lines. */
            for(i = 0; i < 9; i++)
            {
                if(fgets(buf, 10000, fp) == NULL)
                    break;
            }

            /* Begin reading each line and get stats of the line that
             * contains the given path.
             */
            while(fgets(buf, 10000, fp) != NULL)
            {
                buf[10000 - 1] = '\0';

                strptr = buf;

                /* Skip initial spaces. */
                while(ISBLANK(*strptr))
                    strptr++;

                /* Ending deliminator reached? If so do not parse this line
                 * or any subsequent lines.
                 */
                if(*strptr == '-')
                    break;

		/* Now strptr2 should be positioned at the first field which
		 * is the full path.
		 */
		strptr2 = strptr;

                /* Check if any of the given paths match? In case of no
                 * paths given then match is always true.
                 */
                if((npaths > 0) && (path != NULL))
                {
                    gchar *tmp_path = g_strdup(strptr2);
                    strptr3 = strchr(tmp_path, ' ');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\t');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\n');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';

                    for(i = 0; i < npaths; i++)
                    {
                        if(path[i] != NULL)
                        {
                            if(!strcmp(tmp_path, path[i]))
                                break;
                        }
                    }

                    g_free(tmp_path);

                    got_match = (i < npaths) ? TRUE : FALSE;
                }
                else
                {
                    got_match = TRUE;
                }
                if(got_match)
                {
                    /* Create a new archive object structure and parse
                     * the loaded line to it.
                     */
                    obj = EDVArchObjectNew();
                    EDVArchFIOParseLineArj(obj, strptr);

                    /* Allocate more pointers and append this object. */
                    i = *total;
                    *total = i + 1;

                    list = (edv_archive_object_struct **)g_realloc(
                        list, (*total) * sizeof(edv_archive_object_struct *)
                    );
                    if(list == NULL)
                        *total = 0;
                    else
                        list[i] = obj;
                }
            }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

        return(list);
}

/*
 *      Returns a listing of all objects in the RPM archive.
 */
static edv_archive_object_struct **EDVArchFIOGetListingRPM(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
)
{
        const gchar *prog_rpm = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_RPM
        );
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct **list = NULL, *obj;


        /* Begin generating listing command. */
	cmd = g_strdup_printf(
	    "%s -q -l -p -v \"%s\"",
	    prog_rpm, arch_obj
	);
        if(cmd == NULL)
            return(list);


        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);

        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
            return(list);
        }


        /* Read each line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
	    gbool got_match;
            gint i;
            gchar *strptr, *strptr2, *strptr3;
            gchar buf[10000];

            /* Begin reading each line and get stats of the line that
             * contains the given path.
             */
            while(fgets(buf, 10000, fp) != NULL)
            {
                buf[10000 - 1] = '\0';

                strptr = buf;

                /* Skip initial spaces. */
                while(ISBLANK(*strptr))
                    strptr++;

                /* Skip no more than 8 fields to position strptr2 at
                 * the start of the full path.
                 */
                strptr2 = strptr;
                for(i = 0; i < 8; i++)
                {
                    while(!ISBLANK(*strptr2) && (*strptr2 != '\0'))
                        strptr2++;
                    while(ISBLANK(*strptr2))
                        strptr2++;
                }

		/* Check if any of the given paths match? In case of no
		 * paths given then match is always true.
		 */
                if((npaths > 0) && (path != NULL))
		{
                    gchar *tmp_path = g_strdup(strptr2);
                    strptr3 = strchr(tmp_path, ' ');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\t');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\n');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';

                    for(i = 0; i < npaths; i++)
                    {
                        if(path[i] != NULL)
                        {
                            if(!strcmp(tmp_path, path[i]))
                                break;
                        }
                    }

                    g_free(tmp_path);

		    got_match = (i < npaths) ? TRUE : FALSE;
		}
		else
		{
		    got_match = TRUE;
		}
		if(got_match)
		{
                    /* Create a new archive object structure and parse
                     * the loaded line to it.
                     */
                    obj = EDVArchObjectNew();
                    EDVArchFIOParseLineRPM(obj, strptr);

                    /* Allocate more pointers and append this object. */
                    i = *total;
                    *total = i + 1;

                    list = (edv_archive_object_struct **)g_realloc(
                        list, (*total) * sizeof(edv_archive_object_struct *)
                    );
                    if(list == NULL)
                        *total = 0;
                    else
                        list[i] = obj;
		}
	    }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

        return(list);
}

/*
 *	Returns a listing of all objects in the Tar archive.
 */
static edv_archive_object_struct **EDVArchFIOGetListingTar(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths,
        gbool is_compress_compressed,
        gbool is_gzip_compressed,
        gbool is_bzip2_compressed
)
{
        const gchar *prog_tar = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_TAR
        );
        const gchar *prog_bunzip2 = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
        );
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct **list = NULL, *obj;


        /* Begin generating listing command. */
        if(is_compress_compressed)
            cmd = g_strdup_printf(
                "%s -Z -t -v -f \"%s\"",
                prog_tar, arch_obj
            );
        else if(is_gzip_compressed)
            cmd = g_strdup_printf(
                "%s -z -t -v -f \"%s\"",
                prog_tar, arch_obj
            );
        else if(is_bzip2_compressed)
            cmd = g_strdup_printf(
                "%s --use-compress-program=%s -t -v -f \"%s\"",
                prog_tar, prog_bunzip2, arch_obj
            );
        else
            cmd = g_strdup_printf(
                "%s -t -v -f \"%s\"",
                prog_tar, arch_obj
            );
        if(cmd == NULL)
            return(list);


        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);

        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
	    return(list);
        }


        /* Read each line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
            gbool got_match;
            gint i;
            gchar *strptr, *strptr2, *strptr3;
            gchar buf[10000];

            /* Begin reading each line and get stats of the line that
             * contains the given path.
             */
            while(fgets(buf, 10000, fp) != NULL)
            {
                buf[10000 - 1] = '\0';

                strptr = buf;

                /* Skip initial spaces. */
                while(ISBLANK(*strptr))
                    strptr++;

                /* Skip no more than 5 fields to position strptr2 at
                 * the start of the full path.
                 */
                strptr2 = strptr;
                for(i = 0; i < 5; i++)
                {
                    while(!ISBLANK(*strptr2) && (*strptr2 != '\0'))
                        strptr2++;
                    while(ISBLANK(*strptr2))
                        strptr2++;
                }
                /* Check if any of the given paths match? In case of no
                 * paths given then match is always true.
                 */
                if((npaths > 0) && (path != NULL))
                {
		    gchar *tmp_path = g_strdup(strptr2);
		    strptr3 = strchr(tmp_path, ' ');
		    if(strptr3 != NULL)
			*strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\t');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\n');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';

                    for(i = 0; i < npaths; i++)
                    {
                        if(path[i] != NULL)
                        {
                            if(!strcmp(tmp_path, path[i]))
                                break;
                        }
                    }

		    g_free(tmp_path);

                    got_match = (i < npaths) ? TRUE : FALSE;
                }
                else
                {
                    got_match = TRUE;
                }
                if(got_match)
                {
		    /* Create a new archive object structure and parse
		     * the loaded line to it.
		     */
		    obj = EDVArchObjectNew();
		    EDVArchFIOParseLineTar(obj, strptr);

		    /* Allocate more pointers and append this object. */
		    i = *total;
		    *total = i + 1;

		    list = (edv_archive_object_struct **)g_realloc(
			list, (*total) * sizeof(edv_archive_object_struct *)
		    );
		    if(list == NULL)
			*total = 0;
		    else
			list[i] = obj;
		}
            }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

	return(list);
}

/*
 *      Returns a listing of all objects in the zip archive.
 */
static edv_archive_object_struct **EDVArchFIOGetListingZip(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
)
{
        const gchar *prog_unzip = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_PROG_UNZIP
        );
        FILE *fp;
        pid_t p;
        gchar *cmd, *stdout_path;
        edv_archive_object_struct **list = NULL, *obj;


        /* Begin generating listing command. */
	cmd = g_strdup_printf(
	    "%s -l \"%s\"",
	    prog_unzip, arch_obj
	);
        if(cmd == NULL)
            return(list);


        /* Generate standard output file. */
        stdout_path = EDVTmpName(NULL);

        /* Execute stat command. */
        p = ExecBO(cmd, stdout_path);
        if(p <= 0)
        {
            g_free(cmd);
            g_free(stdout_path);
            return(list);
        }


        /* Read each line from the output file. */
        fp = FOpen(stdout_path, "rb");
        if(fp != NULL)
        {
            gbool got_match;
            gint i;
            gchar *strptr, *strptr2, *strptr3;
	    const gchar *hdelim = " --------    ----   ----    ----";
            gchar buf[10000];


            /* Read untill we encounter the last header line. */
            for(i = 0; 1; i++)
            {
                if(fgets(buf, 10000, fp) == NULL)
                    break;

                buf[10000 - 1] = '\0';
                if(strpfx(buf, hdelim))
                    break;
            }

            /* Begin reading each line and get stats of the line that
             * contains the given path.
             */
            while(fgets(buf, 10000, fp) != NULL)
            {
                buf[10000 - 1] = '\0';

                strptr = buf;

                /* Skip initial spaces. */
                while(ISBLANK(*strptr))
                    strptr++;

                /* Ending deliminator reached? If so do not parse this line
                 * or any subsequent lines.
                 */
                if(*strptr == '-')
                    break;

                /* Skip no more than 3 fields to position strptr2 at
                 * the start of the full path.
                 */
                strptr2 = strptr;
                for(i = 0; i < 3; i++)
                {
                    while(!ISBLANK(*strptr2) && (*strptr2 != '\0'))
                        strptr2++;
                    while(ISBLANK(*strptr2))
                        strptr2++;
                }
                /* Check if any of the given paths match? In case of no
                 * paths given then match is always true.
                 */
                if((npaths > 0) && (path != NULL))
                {
                    gchar *tmp_path = g_strdup(strptr2);
                    strptr3 = strchr(tmp_path, ' ');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\t');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';
                    strptr3 = strchr(tmp_path, '\n');
                    if(strptr3 != NULL)
                        *strptr3 = '\0';

                    for(i = 0; i < npaths; i++)
                    {
                        if(path[i] != NULL)
                        {
                            if(!strcmp(tmp_path, path[i]))
                                break;
                        }
                    }

                    g_free(tmp_path);

                    got_match = (i < npaths) ? TRUE : FALSE;
                }
                else
                {
                    got_match = TRUE;
                }
                if(got_match)
                {
                    /* Create a new archive object structure and parse
                     * the loaded line to it.
                     */
                    obj = EDVArchObjectNew();
                    EDVArchFIOParseLineZip(obj, strptr);

                    /* Allocate more pointers and append this object. */
                    i = *total;
                    *total = i + 1;

                    list = (edv_archive_object_struct **)g_realloc(
                        list, (*total) * sizeof(edv_archive_object_struct *)
		    );
		    if(list == NULL)
			*total = 0;
		    else
			list[i] = obj;
		}
            }

            FClose(fp);
        }

        /* Remove output file if it exists. */
        if(stdout_path != NULL)
            unlink(stdout_path);

        g_free(cmd);
        g_free(stdout_path);

        return(list);
}

/*
 *	Returns a list of archive object structures in the archive
 *	arch_obj filtered by the given path list. Only statistics for
 *	the archive objects matching the given path list will be
 *	returned, however if npath is <= 0 then statistics for all
 *	archive objects within the archive will be returned.
 *
 *	The returned list of objects must be deallocated by the calling
 *	function.
 */
edv_archive_object_struct **EDVArchFIOGetListing(
        edv_core_struct *core_ptr,
        const gchar *arch_obj, gint *total,
        const gchar **path, gint npaths
)
{
	struct stat stat_buf;
	edv_archive_object_struct **list = NULL;


	if(total != NULL)
	    *total = 0;

	if((core_ptr == NULL) || (arch_obj == NULL) || (total == NULL))
	    return(list);

	/* Get stats of the archive object, making sure it exists and is
	 * not a directory.
	 */
	if(stat(arch_obj, &stat_buf))
	    return(list);
	if(S_ISDIR(stat_buf.st_mode))
	    return(list);


        if(EDVIsExtension(arch_obj, ".arj"))
        {
            list = EDVArchFIOGetListingArj(
                core_ptr, arch_obj, total,
                path, npaths
            );
        }
        else if(EDVIsExtension(arch_obj, ".rpm"))
        {
            list = EDVArchFIOGetListingRPM(
                core_ptr, arch_obj, total,
                path, npaths
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.Z"))
        {
            list = EDVArchFIOGetListingTar(
                core_ptr, arch_obj, total,
                path, npaths,
                TRUE,		/* Not compress compressed. */
                FALSE,		/* Not gzip compressed. */
                FALSE		/* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tgz .tar.gz"))
        {
            list = EDVArchFIOGetListingTar(
                core_ptr, arch_obj, total,
                path, npaths,
		FALSE,		/* Not compress compressed. */
                TRUE,           /* Is gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar.bz2"))
        {
            list = EDVArchFIOGetListingTar(
                core_ptr, arch_obj, total,
		path, npaths,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                TRUE            /* Is bzip2 compressed. */
            );
        }
        else if(EDVIsExtension(arch_obj, ".tar"))
        {
            list = EDVArchFIOGetListingTar(
                core_ptr, arch_obj, total,
                path, npaths,
                FALSE,          /* Not compress compressed. */
                FALSE,          /* Not gzip compressed. */
                FALSE           /* Not bzip2 compressed. */
            );
        }
	else if(EDVIsExtension(arch_obj, ".zip"))
        {
            list = EDVArchFIOGetListingZip(
		core_ptr, arch_obj, total,
                path, npaths
	    );
	}


	return(list);
}
