#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvfop.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvrecbinfop.h"
#include "edvarch.h"
#include "edvarchfio.h"
#include "edvarchop.h"
#include "edvcfg.h"
#include "edvdde.h"
#include "edvconfirm.h"
#include "archiveopts.h"
#include "browser.h"
#include "browserdnd.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


static void EDVBrowserDNDUpdateStatusBar(
	edv_browser_struct *browser,
	gint gdk_action, guint info,
	gint total_src_objects,
        gint total_objects_processed, gint status
);

static void EDVBrowserDragDataReceivedNexus(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
        edv_object_struct *object       /* Can be NULL. */
);

gint EDVBrowserLocBarIconCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void EDVBrowserDirTreeDNDSetIcon(
	edv_browser_struct *browser, gint row
);
void EDVBrowserDirTreeDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVBrowserDirTreeDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVBrowserDirTreeDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void EDVBrowserContentsDNDSetIcon(
        edv_browser_struct *browser, gint row, gint column
);
void EDVBrowserContentsDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVBrowserContentsDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVBrowserContentsDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
);


/*
 *	Updates the status bar to indicate the result of the dnd operation
 *	on the target browser.
 */
static void EDVBrowserDNDUpdateStatusBar(
        edv_browser_struct *browser,
	gint gdk_action, guint info,
        gint total_src_objects,
        gint total_objects_processed, gint status
)
{
	gchar *buf = NULL;
	edv_status_bar_struct *sb;


	if(browser == NULL)
	    return;

	/* Get status bar. */
	sb = browser->status_bar;
	if(sb == NULL)
	    return;

	/* Begin formatting status message, handle by dnd target type. */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
        )
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
		switch(status)
		{
                  case 0: case -5:
		    buf = g_strdup_printf(
			"Coppied %i object%s",
			total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
		    );
		    break;

                  case -4:	/* Cancel. */
		    buf = g_strdup_printf(
			"Copy operation canceled"
		    );
                    break;

		  default:	/* Error. */
		    buf = g_strdup_printf(
			"Unable to copy object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
		}
		break;

              case GDK_ACTION_MOVE:
		switch(status)
                {
                  case 0: case -5:
                    buf = g_strdup_printf(
			"Moved %i object%s",
                        total_objects_processed,
                        (total_objects_processed == 1) ? "" : "s"
                    );
                    break;

                  case -4:	/* Cancel. */
		    buf = g_strdup_printf(
			"Move operation canceled"
		    );
		    break;

		  default:	/* Error. */
		    buf = g_strdup_printf(
			"Unable to move object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
                }
                break;

              case GDK_ACTION_LINK:
		switch(status)
		{
		  case 0: case -5:
		    buf = g_strdup_printf(
			"Linked %i object%s",
			total_objects_processed,
                        (total_objects_processed == 1) ? "" : "s"
                    );
                    break;

                  case -4:	/* Cancel. */
		    buf = g_strdup_printf(
			"Link operation canceled"
		    );
		    break;

		  default:	/* Error. */
		    buf = g_strdup_printf(
			"Unable to link object%s",
			(total_src_objects == 1) ? "" : "s"
		    );
		    break;
                }
                break;
	    }
	}
        else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_LINK:
                switch(status)
                {
                  case 0: case -5:
                    buf = g_strdup_printf(
                        "Recovered %i object%s",
                        total_objects_processed,
                        (total_objects_processed == 1) ? "" : "s"
                    );
                    break;

                  case -4:  /* Cancel. */
                    buf = g_strdup_printf(
                        "Recover operation canceled"
                    );
                    break;

                  default:  /* Error. */
                    buf = g_strdup_printf(
                        "Unable to recover object%s",
                        (total_src_objects == 1) ? "" : "s"
                    );
                    break;
                }
		break;
	    }
	}
        else if(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_LINK:
                switch(status)
                {
                  case 0: case -5:
                    buf = g_strdup_printf(
                        "Extracted %i object%s",
                        total_objects_processed,
                        (total_objects_processed == 1) ? "" : "s"
                    );
                    break;

                  case -4:  /* Cancel. */
                    buf = g_strdup_printf(
                        "Extract operation canceled"
                    );
                    break;

                  default:  /* Error. */
                    buf = g_strdup_printf(
                        "Unable to extract object%s",
                        (total_src_objects == 1) ? "" : "s"
                    );
                    break;
                }
                break;
            }
	}
        else
        {
            /* Unsupported target type. */
	}

	/* Set status bar message. */
	EDVStatusBarMessage(sb, buf, FALSE);

	/* Deallocate status bar message buffer. */
	g_free(buf);
	buf = NULL;
}


/*
 *	Data received nexus, all inputs assumed valid.
 */
static void EDVBrowserDragDataReceivedNexus(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	edv_object_struct *object	/* Can be NULL. */
)
{
	gint initial_confirmation_result;
	GtkWidget *toplevel = browser->toplevel;
	const gchar *cstrptr;
	edv_dnd_object_struct **dnd_object, *dnd_obj;
	gint total_dnd_objects;
	gchar *parent_path;


#define DO_FREE_LOCALS	\
{ \
 /* Deallocate path of the parent disk object. */ \
 g_free(parent_path); \
 parent_path = NULL; \
 \
 /* Deallocate dnd disk object reference list. */ \
 if(dnd_object != NULL) \
 { \
  gint i; \
\
  for(i = 0; i < total_dnd_objects; i++) \
   EDVDNDObjectDelete(dnd_object[i]); \
  g_free(dnd_object); \
  dnd_object = NULL; \
  total_dnd_objects = 0; \
 } \
}


	/* Get copy of parent_path as the disk object's full path if the
	 * disk object is not NULL.
	 */
	if(object != NULL)
	    parent_path = (object->full_path != NULL) ?
	        g_strdup(object->full_path) : NULL;
	else
	    parent_path = NULL;

	/* If parent_path is not obtainable from the disk object then
	 * use the image browser's current location as the parent_path.
	 */
	if(parent_path == NULL)
	{
	    cstrptr = EDVBrowserCurrentLocation(browser);
	    parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	}

	/* Check if parent_path does not lead to a directory, if it does
	 * not lead to a directory then we need to make the parent_path's
	 * parent the directory to drop to.
	 */
	if(!ISPATHDIR(parent_path))
	{
	    /* Droped on object does not lead to a directory, so change
	     * the parent_path to reflect its own parent so we know that
	     * it is a directory.
	     */
	    cstrptr = GetParentDir(parent_path);
	    if(cstrptr != NULL)
	    {
		g_free(parent_path);
		parent_path = g_strdup(cstrptr);
	    }
	}

	/* Parse the given selection buffer into an array of dnd object
	 * structures.
	 */
	dnd_object = EDVDNDBufferToObject(
	    (const guint8 *)selection_data->data,
	    selection_data->length,
	    &total_dnd_objects
	);


	/* External drag source checks go here, for example sources
	 * that have a url with a http or ftp protocol.
	 */
	dnd_obj = (total_dnd_objects > 0) ? dnd_object[0] : NULL;
	if((dnd_obj != NULL) ? (dnd_obj->protocol != NULL) : FALSE)
	{
	    const gchar *protocol = dnd_obj->protocol;
	    if(!strcasecmp(protocol, "http") ||
	       !strcasecmp(protocol, "ftp")
	    )
	    {
		EDVDoInternetDownloadObject(
		    core_ptr, dnd_obj, parent_path, toplevel
		);

                /* Do not continue with standard dnd handling. */
                DO_FREE_LOCALS
                return;
	    }
	}


	/* Make initial user confirmation querying to proceed with this
	 * operation.
	 */
	initial_confirmation_result = EDVConfirmDND(
	    core_ptr, (gint)dc->action, info,
            toplevel,
            dnd_object, total_dnd_objects,
            parent_path
	);
	/* User confirmed and dnd target type is a disk object? */
	if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
           ((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
            (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
            (info == EDV_DND_TYPE_INFO_STRING)
	   )
	)
	{
	    gint i, status = -1;
	    gbool yes_to_all = FALSE;
	    gint objects_processed = 0;
	    gchar *new_path;
	    const gchar *error_mesg;
	    struct stat lstat_buf;


	    EDVBrowserSetBusy(browser, TRUE);
	    browser->processing = TRUE;

	    /* Iterate through list of dnd disk object references. */
	    for(i = 0; i < total_dnd_objects; i++)
	    {
		dnd_obj = dnd_object[i];
		if(dnd_obj == NULL)
		    continue;

		if((parent_path == NULL) ||
		   (dnd_obj->full_path == NULL)
		)
		    continue;

		/* Handle by drag action type. */
		status = -1;
		new_path = NULL;
		error_mesg = NULL;
                switch((gint)dc->action)
                {
                  case GDK_ACTION_COPY:
                    status = EDVFOPCopy(
                        core_ptr, dnd_obj->full_path, parent_path,
                        &new_path, toplevel,
                        TRUE, TRUE,
                        &yes_to_all
                    );
                    break;
                  case GDK_ACTION_MOVE:
                    status = EDVFOPMove(
                        core_ptr, dnd_obj->full_path, parent_path,
                        &new_path, toplevel,
                        TRUE, TRUE,
                        &yes_to_all
                    );
                    break;
                  case GDK_ACTION_LINK:
                    status = EDVFOPLink(
                        core_ptr, dnd_obj->full_path, parent_path,
                        &new_path, toplevel,
                        TRUE, TRUE,
                        &yes_to_all
                    );
                    break;
		  default:
		    error_mesg = "Unsupported drag operation";
		    break;
                }

                /* Get error message (if any) that might have occured
                 * in the above operation.
                 */
		if(error_mesg == NULL)
		    error_mesg = EDVFOPGetError();
                if((error_mesg != NULL) && (status != -4))
                {
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

                /* Check if new_path is valid and that no error
                 * occured. This implies the operation was successful
                 * and we must emit a object added signal to all
                 * windows.
                 */
                if(((new_path != NULL) && !status) ?
                    !lstat(new_path, &lstat_buf) : FALSE
                )
                {
		    objects_processed++;

                    browser->processing = FALSE;
                    EDVObjectAddedEmit(
                        core_ptr, new_path, &lstat_buf
                    );
                    browser->processing = TRUE;
                }

                /* Deallocate copy of the new disk object's path. */
                g_free(new_path);
                new_path = NULL;

                /* Error occured (not 0) and it was not a user `no' or
                 * `not available' (-5) response?
                 */
                if(status && (status != -5))
                {
                    /* Skip rest of the DND object references due to error. */
                    break;
                }
            }

            /* Update status bar message. */
            EDVBrowserDNDUpdateStatusBar(
                browser, (gint)dc->action, info,
		total_dnd_objects, objects_processed, status
            );

            /* Unmap progress dialog, it may have been mapped if any
             * operations occured in the above loop.
             */
            ProgressDialogBreakQuery(TRUE);
            ProgressDialogSetTransientFor(NULL);

            browser->processing = FALSE;
            EDVBrowserSetBusy(browser, FALSE);
        }
        /* User confirmed and dnd target type is a recycled object? */
        else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
                (info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        )
        {
            gbool yes_to_all = FALSE;
            gint i, status = -1;
            gint objects_recovered = 0;
            guint index;
            gchar *new_path;
            const gchar *error_mesg;
            struct stat lstat_buf;


            EDVBrowserSetBusy(browser, TRUE);
            browser->processing = TRUE;

            /* Iterate through list of dnd recycled object references. */
            for(i = 0; i < total_dnd_objects; i++)
            {
                dnd_obj = dnd_object[i];
                if(dnd_obj == NULL)
                    continue;

                if(dnd_obj->full_path == NULL)
                    continue;

                /* Handle by drag action type. */
                status = -1;
                new_path = NULL;
		error_mesg = NULL;
                switch((gint)dc->action)
                {
                  case GDK_ACTION_COPY:
		    error_mesg = "Cannot copy a recycled object";
		    break;
                  case GDK_ACTION_MOVE:
                    index = (guint)atoi(dnd_obj->full_path);
                    status = EDVRecBinFOPRecover(
                        core_ptr, index, parent_path,
                        &new_path,
                        toplevel,
                        TRUE, TRUE, &yes_to_all
                    );
                    break;
		  case GDK_ACTION_LINK:
		    error_mesg = "Cannot link a recycled object";
                    break;
                  default:
                    error_mesg = "Unsupported drag operation";
                    break;
                }

                /* Get error message (if any) that might have occured
                 * in the above operation.
                 */
		if(error_mesg == NULL)
		    error_mesg = EDVRecBinFOPGetError();
                if((error_mesg != NULL) && (status != -4))
                {
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

                /* Check if new_path is valid and that no error
                 * occured. This implies the operation was successful
                 * and we must emit a object added signal to all
                 * windows.
                 */
                if(((new_path != NULL) && !status) ?
                    !lstat(new_path, &lstat_buf) : FALSE
                )
                {
                    objects_recovered++;

                    browser->processing = FALSE;
                    EDVObjectAddedEmit(
                        core_ptr, new_path, &lstat_buf
                    );
                    browser->processing = TRUE;
                }

                /* Deallocate copy of the new disk object's path. */
                g_free(new_path);
                new_path = NULL;

                /* Error occured (not 0) and it was not a user `no' or
                 * or `not available' (-5) response?
                 */
                if(status && (status != -5))
                {
                    /* Skip rest of the DND object references due to error. */
                    break;
                }
            }

            /* Update status bar message. */
            EDVBrowserDNDUpdateStatusBar(
                browser, (gint)dc->action, info,
                total_dnd_objects, objects_recovered, status
            );

            /* Unmap progress dialog, it may have been mapped if any
             * operations occured in the above loop.
             */
            ProgressDialogBreakQuery(TRUE);
            ProgressDialogSetTransientFor(NULL);

            browser->processing = FALSE;
            EDVBrowserSetBusy(browser, FALSE);
        }
        /* User confirmed and dnd target type is an archive object? */
        else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
                (info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        )
        {
            gbool yes_to_all = FALSE;
	    gbool preserve_directories = TRUE;
            gint i, status;
            gint objects_extracted = 0;
	    const gchar *arch_obj = NULL;
            const gchar *error_mesg;
	    edv_archive_object_struct **aobj = NULL;
	    gint total_aobjs = 0;
	    gchar **new_path = NULL;
	    gint total_new_paths = 0;


            EDVBrowserSetBusy(browser, TRUE);
            browser->processing = TRUE;

	    /* Get location of archive itself which is the first dnd object
	     * structure in the list.
	     */
	    if(total_dnd_objects >= 1)
	    {
		dnd_obj = dnd_object[0];
		arch_obj = (dnd_obj != NULL) ? dnd_obj->full_path : NULL;
	    }

	    /* Get extract from archive options. */
            if(!EDVArchiveExtractQueryOptions(
                core_ptr, toplevel, arch_obj, &preserve_directories
	    ))
	    {
		/* User canceled. */
		browser->processing = FALSE;
		EDVBrowserSetBusy(browser, FALSE);
		DO_FREE_LOCALS
		return;
	    }

	    /* Get statistics for all archive objects in the archive. */
	    if(total_dnd_objects >= 2)
	    {
		gchar **strv;
		gint j, strc = total_dnd_objects - 1;

		strv = (gchar **)g_malloc0(strc * sizeof(gchar *));
		for(i = 0, j = 1; i < strc; i++, j++)
		{
		    if(j >= total_dnd_objects)
			break;

		    dnd_obj = dnd_object[j];
		    strv[i] = (dnd_obj != NULL) ? dnd_obj->full_path : NULL;
		}

		aobj = EDVArchFIOGetListing(
		    core_ptr, arch_obj, &total_aobjs,
		    (const gchar **)strv, strc
		);

		/* Deallocate only the path strings pointer array, each
		 * string is shared.
		 */
		g_free(strv);
	    }


            /* Extract. */
            status = EDVArchOPExtract(
                core_ptr, arch_obj, aobj, total_aobjs,
                parent_path, &new_path, &total_new_paths,
                toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories
            );

            /* Get error message (if any) that might have occured in the
             * above operation.
             */
            error_mesg = EDVArchOPGetError();
            if(error_mesg != NULL)
            {
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Operation Error",
                    error_mesg,
                    NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
            }

            /* Report new disk objects added? */
            if((new_path != NULL) && !status)
            {
                struct stat lstat_buf;


                browser->processing = FALSE;

                for(i = 0; i < total_new_paths; i++)
                {
                    cstrptr = new_path[i];
                    if(cstrptr == NULL)
                        continue;
                    if(lstat(cstrptr, &lstat_buf))
                        continue;

                    objects_extracted++;

                    EDVObjectAddedEmit(
                        core_ptr, cstrptr, &lstat_buf
                    );
                }

                browser->processing = TRUE;
            }

            /* Deallocate coppies of new disk object's paths. */
            if(new_path != NULL)
            {
                for(i = 0; i < total_new_paths; i++)
                    g_free(new_path[i]);
                g_free(new_path);
                new_path = NULL;
                total_new_paths = 0;
            }


            /* Update status bar message. */
            EDVBrowserDNDUpdateStatusBar(
                browser, (gint)dc->action, info,
                total_aobjs, objects_extracted, status
            );

            /* Unmap progress dialog, it may have been mapped if any
             * operations occured in the above loop.
             */
            ProgressDialogBreakQuery(TRUE);
            ProgressDialogSetTransientFor(NULL);

            browser->processing = FALSE;
            EDVBrowserSetBusy(browser, FALSE);


	    /* Deallocate list of archive object stats. */
	    for(i = 0; i < total_aobjs; i++)
		EDVArchObjectDelete(aobj[i]);
	    g_free(aobj);
	    aobj = NULL;
	    total_aobjs = 0;
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      Location bar icon "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
gint EDVBrowserLocBarIconCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	edv_browser_struct *browser = (edv_browser_struct *)data;
        if((widget == NULL) || (crossing == NULL) || (browser == NULL))
            return(status);

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
	    gtk_widget_set_state(widget, GTK_STATE_PRELIGHT);
	    EDVStatusBarMessage(
		browser->status_bar,
#ifdef PROG_LANGUAGE_ENGLISH
"Drag this to create a link to this location",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Arrastre esto crear un eslabn a esta ubicacin",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Traner ceci pour crer un lien  cet emplacement",
#endif
		FALSE
	    );
	    if(browser->location_icon_pm != NULL)
	    {
		GdkBitmap *mask = NULL;
		GdkPixmap *pixmap = NULL;
		gtk_pixmap_get(
		    GTK_PIXMAP(browser->location_icon_pm),
		    &pixmap, &mask
		);
		if(pixmap != NULL)
		{
		    gint w = 15, h = 15;
		    gdk_window_get_size((GdkWindow *)pixmap, &w, &h);
		    GUIDNDSetDragIcon(
			pixmap, mask,
			(w / 2), (h / 2)
		    );
		}
	    }
	    status = TRUE;
	    break;

	  case GDK_LEAVE_NOTIFY:
            gtk_widget_set_state(widget, GTK_STATE_NORMAL);
            EDVStatusBarMessage(
                browser->status_bar, NULL, FALSE
            );
	    status = TRUE;
	    break;
	}

	return(status);
}



/*
 *	Sets the drag and drop icon based on the coordinates of the
 *	directory ctree's node given by row.
 */
void EDVBrowserDirTreeDNDSetIcon(
        edv_browser_struct *browser, gint row
)
{
        GtkCList *clist;
	GtkCTree *ctree;
	GtkCTreeNode *node;
	edv_object_struct *object;


        if(browser == NULL)
            return;

	if(!browser->initialized || !browser->map_state)
	    return;

	ctree = (GtkCTree *)browser->directory_ctree;
	if(ctree == NULL)
	    return;

        clist = GTK_CLIST(ctree);


	/* Get node that matches the given coordinates. */
	node = gtk_ctree_node_nth(ctree, row);
	if(node == NULL)
	    return;

	/* Get disk object structure referenced by matched node. */
	object = (edv_object_struct *)gtk_ctree_node_get_row_data(
	    ctree, node
	);
	if(object != NULL)
        {
            GdkPixmap *pixmap = NULL;
            GdkBitmap *mask = NULL;
	    GtkCTreeRow *ctree_row_ptr = GTK_CTREE_ROW(node);


	    /* Check if the node's row pointer is valid and obtain the
	     * pixmap and mask pair that will be used as the icon.
	     */
	    if(ctree_row_ptr != NULL)
	    {
		if(ctree_row_ptr->expanded)
		{
                    pixmap = ctree_row_ptr->pixmap_opened;
                    mask = ctree_row_ptr->mask_opened;
		}
		else
		{
		    pixmap = ctree_row_ptr->pixmap_closed;
		    mask = ctree_row_ptr->mask_closed;
		}
		/* Revert to closed icon if desired pixmap was not
		 * available.
		 */
		if(pixmap == NULL)
		{
		    pixmap = ctree_row_ptr->pixmap_closed;
                    mask = ctree_row_ptr->mask_closed;
		}
	    }

	    /* Is icon pixmap available? If so then set the new DND icon. */
            if(pixmap != NULL)
            {
                gint w = 15, h = 15;

                gdk_window_get_size((GdkWindow *)pixmap, &w, &h);

                GUIDNDSetDragIcon(
                    pixmap, mask,
                    (w / 2), (h / 2)
                );
            }
        }
}

/*
 *	Browser directory ctree DND "drag_data_get" signal callback.
 */
void EDVBrowserDirTreeDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
	gbool data_sent = FALSE;
	gint i;
        edv_core_struct *core_ptr;
	GList *glist;
	GtkWidget *w;
	GtkCList *clist;
	GtkCTree *ctree;
	GtkCListRow *row_ptr;
	edv_dnd_object_struct **dnd_object, *dnd_obj;
	gint total_dnd_objects;
	edv_object_struct *object;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((dc == NULL) || (browser == NULL))
            return;

        if(!browser->initialized || browser->processing)
	    return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

	/* Get directory ctree and make sure it matches the given widget. */
	w = browser->directory_ctree;
	if((w == NULL) || (w != widget))
	    return;

	ctree = GTK_CTREE(w);
	clist = GTK_CLIST(w);


        /* Sync data on browser so as to ensure we have the most up to
	 * date information to send out.
	 */
	EDVBrowserSyncData(browser);


	/* Iterate through selected rows, generating a list of DND disk
	 * object reference structures based on each selected row's disk
	 * object structure.
	 */
	total_dnd_objects = 0;
	dnd_object = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    /* Get pointer to selected row. */
	    row_ptr = (GtkCListRow *)glist->data;

	    /* Get disk object structure on selected row. */
            object = (edv_object_struct *)gtk_ctree_node_get_row_data(
		ctree, (GtkCTreeNode *)row_ptr
	    );
	    if((object != NULL) ? (object->full_path != NULL) : FALSE)
	    {
		/* Allocate more pointers for a new dnd disk object 
		 * reference structure.
		 */
		i = total_dnd_objects;
		total_dnd_objects = i + 1;
		dnd_object = (edv_dnd_object_struct **)g_realloc(
		    dnd_object,
		    total_dnd_objects * sizeof(edv_dnd_object_struct *)
		);
		if(dnd_object == NULL)
		{
		    total_dnd_objects = 0;
		    break;
		}
		/* Allocate a new dnd disk object reference structure. */
		dnd_object[i] = dnd_obj = EDVDNDObjectNew();
		if(dnd_obj != NULL)
		{
		    dnd_obj->full_path = g_strdup(object->full_path);
		}
	    }

	    /* Get next selected row. */
	    glist = glist->next;
	}

	/* Any DND disk object reference structures to send out? */
	if(total_dnd_objects > 0)
	{
	    guint8 *buf;
	    gint buf_len;


	    /* Allocate a serial buffer for DND data sending based on the
	     * list of DND disk object reference structures.
	     */
	    buf = EDVDNDObjectToBuffer(
		(const edv_dnd_object_struct **)dnd_object,
		total_dnd_objects, &buf_len
	    );
	    if(buf != NULL)
	    {
		/* Send out DND data buffer. */
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,		/* 8 bits per character. */
		    buf, buf_len
		);
		data_sent = TRUE;

		/* Deallocate DND data buffer. */
		g_free(buf);
		buf = NULL;
		buf_len = 0;
	    }
	}

	/* Deallocate all dnd disk object reference structures. */
	for(i = 0; i < total_dnd_objects; i++)
	    EDVDNDObjectDelete(dnd_object[i]);
	g_free(dnd_object);
	dnd_object = NULL;
	total_dnd_objects = 0;


        /* Failed to send out data? */
        if(!data_sent)
        {
	    /* Then send out an error response. */
            const gchar *strptr = "Error";

            gtk_selection_data_set(
                selection_data,
                GDK_SELECTION_TYPE_STRING,
                8,	/* 8 bits per character. */
                strptr, strlen(strptr)
            );
            data_sent = TRUE;
        }
}

/*
 *      Browser directory ctree DND "drag_data_received" signal callback.
 */
void EDVBrowserDirTreeDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
	static gbool reenterent = FALSE;
        GtkWidget *w;
        GtkCList *clist;
        GtkCTree *ctree;
	GtkCTreeNode *node;
	edv_object_struct *object;
        edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((dc == NULL) || (browser == NULL))
            return;

        if(!browser->initialized || browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get directory ctree and make sure it matches the given widget. */
        w = browser->directory_ctree;
        if((w == NULL) || (w != widget))
            return;

        ctree = GTK_CTREE(w);
        clist = GTK_CLIST(w);

        /* Check if received data is not empty. */
        if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;


        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
	{
	    reenterent = FALSE;
            return;
	}

        /* Sync data on browser so as to ensure we have the most up to
         * date information to apply the new drag data to.
         */
        EDVBrowserSyncData(browser);


	/* Find node that this drop occured on, then get disk object
	 * structure referenced by the node.
	 */
	node = EDVNodeGetByCoordinates(
	    ctree,
	    x,
            y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
                clist->column_title_area.height +
                clist->column_title_area.y : 0)
	);
	object = (node != NULL) ?
	    (edv_object_struct *)gtk_ctree_node_get_row_data(ctree, node) :
	    NULL;

	/* If unable to obtain disk object structure or node then that
	 * implies we have nothing to drop on. Since dropping on the
	 * directory ctree requires that the drop occurs on a node, we
	 * need to return due to having no node to drop on.
	 */
	if((object != NULL) ? (object->full_path == NULL) : TRUE)
	{
	    reenterent = FALSE;
	    return;
	}

        /* Handle received drag data. */
	EDVBrowserDragDataReceivedNexus(
            core_ptr, browser, dc, info, selection_data, object
	);

	reenterent = FALSE;
}

/*
 *      Browser directory ctree DND "drag_data_delete" signal callback.
 */
void EDVBrowserDirTreeDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	const gchar *cstrptr;
	gchar **strv;
	gint i, strc;
        edv_core_struct *core_ptr;
	GList *glist;
        GtkWidget *w;
        GtkCList *clist;
        GtkCTree *ctree;
	GtkCListRow *row_ptr;
        edv_object_struct *object;
	struct stat lstat_buf;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((dc == NULL) || (browser == NULL))
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get directory ctree and make sure it matches the given widget. */
        w = browser->directory_ctree;
        if((w == NULL) || (w != widget))
            return;

        ctree = GTK_CTREE(w);
        clist = GTK_CLIST(w);


	/* Iterate through selected rows, getting a list of paths. */
	strv = NULL;
	strc = 0;
        glist = clist->selection;
        while(glist != NULL)
        {
            /* Get pointer to selected row. */
            row_ptr = (GtkCListRow *)glist->data;

	    /* Get disk object structure on selected row. */
            object = (edv_object_struct *)gtk_ctree_node_get_row_data(
                ctree, (GtkCTreeNode *)row_ptr
            );
	    if((object != NULL) ? (object->full_path != NULL) : FALSE)
            {
		i = strc;
		strc = i + 1;
		strv = (gchar **)g_realloc(
		    strv, strc * sizeof(gchar *)
		);
		if(strv == NULL)
		{
		    strc = 0;
		    break;
		}
		strv[i] = g_strdup(object->full_path);
	    }

            /* Get next selected row. */
            glist = glist->next;
        }

	/* Emit signal indicating object removed for each path and
	 * deallocate each path.
	 */
	for(i = 0; i < strc; i++)
	{
	    cstrptr = strv[i];
	    if(cstrptr != NULL)
	    {
		/* Check if object really dosen't exist, if it really
		 * does not exist then emit a disk object removed signal.
		 */
		if(lstat(cstrptr, &lstat_buf))
		    EDVObjectRemovedEmit(core_ptr, cstrptr);
	    }

	    /* Deallocate this path. */
	    g_free(strv[i]);
	    strv[i] = NULL;
	}
	/* Deallocate paths string array. */
	g_free(strv);
	strv = NULL;
	strc = 0;
}



/*
 *      Sets the drag and drop icon based on the coordinates of the
 *      contents clist's node given by x and y.
 */
void EDVBrowserContentsDNDSetIcon(
        edv_browser_struct *browser, gint row, gint column
)
{
        GtkCList *clist;
        edv_object_struct *object;


        if(browser == NULL)
            return;

        if(!browser->initialized || !browser->map_state)
            return;

        clist = (GtkCList *)browser->contents_clist;
	if(clist == NULL)
	    return;

        /* Get disk object structure referenced by matched row. */
        object = (edv_object_struct *)gtk_clist_get_row_data(
            clist, row
        );
        if(object != NULL)
        {
            gint i, status;
            GdkPixmap *pixmap = NULL;
            GdkBitmap *mask = NULL;


	    /* Iterate through each cell of this row, looking for a
	     * useable pixmap.
	     */
	    for(i = 0; i < clist->columns; i++)
	    {
		status = (gint)gtk_clist_get_cell_type(
		    clist, row, i
		);
		if(status == GTK_CELL_PIXMAP)
		{
		    gtk_clist_get_pixmap(
			clist, row, i,
			&pixmap, &mask
		    );
		    break;
		}
		else if(status == GTK_CELL_PIXTEXT)
		{
		    gchar *text = NULL;
		    guint8 spacing = 0;

		    gtk_clist_get_pixtext(
			clist, row, i,
			&text, &spacing, &pixmap, &mask
		    );
		    break;
		}
	    }

            /* Is icon pixmap available? If so then set the new DND icon. */
            if(pixmap != NULL)
            {
                gint w = 15, h = 15;

                gdk_window_get_size((GdkWindow *)pixmap, &w, &h);

                GUIDNDSetDragIcon(
                    pixmap, mask,
                    (w / 2), (h / 2)
                );
            }
        }
}

/*
 *      Browser contents clist DND "drag_data_get" signal callback.
 */
void EDVBrowserContentsDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
        gbool data_sent = FALSE;
        gint i;
        edv_core_struct *core_ptr;
        GList *glist;
        GtkWidget *w;
        GtkCList *clist;
	gint row;
        edv_dnd_object_struct **dnd_object, *dnd_obj;
        gint total_dnd_objects;
        edv_object_struct *object;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((dc == NULL) || (browser == NULL))
            return;

        if(!browser->initialized || browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;
        /* Get contents clist and make sure it matches the given widget. */
        w = browser->contents_clist;
        if((w == NULL) || (w != widget))
            return;

        clist = GTK_CLIST(w);


        /* Sync data on browser so as to ensure we have the most up to
         * date information to send out.
         */
        EDVBrowserSyncData(browser);


        /* Iterate through selected rows, generating a list of DND disk
         * object reference structures based on each selected row's disk
         * object structure.
         */
        total_dnd_objects = 0;
        dnd_object = NULL;
        glist = clist->selection;
        while(glist != NULL)
        {
            /* Get pointer to selected row. */
            row = (gint)glist->data;

            /* Get disk object structure on selected row. */
            object = (edv_object_struct *)gtk_clist_get_row_data(
		clist, row
	    );
            if((object != NULL) ? (object->full_path != NULL) : FALSE)
            {
                /* Allocate more pointers for a new dnd disk object
                 * reference structure.
                 */
                i = total_dnd_objects;
                total_dnd_objects = i + 1;
                dnd_object = (edv_dnd_object_struct **)g_realloc(
                    dnd_object,
                    total_dnd_objects * sizeof(edv_dnd_object_struct *)
                );
                if(dnd_object == NULL)
                {
                    total_dnd_objects = 0;
                    break;
                }
                /* Allocate a new dnd disk object reference structure. */
                dnd_object[i] = dnd_obj = EDVDNDObjectNew();
                if(dnd_obj != NULL)
                {
                    dnd_obj->full_path = g_strdup(object->full_path);
                }
            }

            /* Get next selected row. */
            glist = glist->next;
        }

        /* Any DND disk object reference structures to send out? */
        if(total_dnd_objects > 0)
        {
            guint8 *buf;
            gint buf_len;


            /* Allocate a serial buffer for DND data sending based on the
             * list of DND disk object reference structures.
             */
            buf = EDVDNDObjectToBuffer(
                (const edv_dnd_object_struct **)dnd_object,
                total_dnd_objects, &buf_len
            );
            if(buf != NULL)
            {
                /* Send out DND data buffer. */
                gtk_selection_data_set(
                    selection_data,
                    GDK_SELECTION_TYPE_STRING,
                    8,          /* 8 bits per character. */
                    buf, buf_len
                );
                data_sent = TRUE;

                /* Deallocate DND data buffer. */
                g_free(buf);
                buf = NULL;
                buf_len = 0;
            }
        }

        /* Deallocate all dnd disk object reference structures. */
        for(i = 0; i < total_dnd_objects; i++)
            EDVDNDObjectDelete(dnd_object[i]);
        g_free(dnd_object);
        dnd_object = NULL;
        total_dnd_objects = 0;


        /* Failed to send out data? */
        if(!data_sent)
        {
            /* Then send out an error response. */
            const gchar *strptr = "Error";

            gtk_selection_data_set(
                selection_data,
                GDK_SELECTION_TYPE_STRING,
                8,      /* 8 bits per character. */
                strptr, strlen(strptr)
            );
            data_sent = TRUE;
        }
}

/*
 *      Browser contents clist DND "drag_data_received" signal callback.
 */
void EDVBrowserContentsDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
	static gbool reenterent = FALSE;
        GtkWidget *w;
        GtkCList *clist;
	gint row, column;
        edv_object_struct *object;
	edv_core_struct *core_ptr;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((dc == NULL) || (browser == NULL))
            return;

        if(!browser->initialized || browser->processing)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist and make sure it matches the given widget. */
        w = browser->contents_clist;
        if((w == NULL) || (w != widget))
            return;

        clist = GTK_CLIST(w);

        /* Check if received data is not empty. */
        if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


	/* Check and warn if write protect is enabled. */
	if(EDVCheckWriteProtect(core_ptr, TRUE))
	{
	    reenterent = FALSE;
	    return;
	}

        /* Sync data on browser so as to ensure we have the most up to
         * date information to apply the new drag data to.
         */
        EDVBrowserSyncData(browser);


        /* Find row that this drop occured on, then get disk object
         * structure referenced by the row.
         */
        /* Find row and column based on given coordinates. */
        if(!gtk_clist_get_selection_info(
	    clist,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
        ))
        {
            row = -1;
            column = 0;
        }
	/* Get disk object structure from selected row. */
	if((row >= 0) && (row < clist->rows))
	    object = (edv_object_struct *)gtk_clist_get_row_data(clist, row);
	else
	    object = NULL;

        /* Handle received drag data. */
        EDVBrowserDragDataReceivedNexus(
            core_ptr, browser, dc, info, selection_data,
	    object		/* Can be NULL. */
        );

	reenterent = FALSE;
}

/*
 *      Browser contents clist DND "drag_data_delete" signal callback.
 */
void EDVBrowserContentsDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
        const gchar *cstrptr;
        gchar **strv;
        gint i, strc;
        edv_core_struct *core_ptr;
        GList *glist;
        GtkWidget *w;
        GtkCList *clist;
	gint row;
        edv_object_struct *object;
        struct stat lstat_buf;
        edv_browser_struct *browser = (edv_browser_struct *)data;
        if((dc == NULL) || (browser == NULL))
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Get contents clist and make sure it matches the given widget. */
        w = browser->contents_clist;
        if((w == NULL) || (w != widget))
            return;

        clist = GTK_CLIST(w);


        /* Iterate through selected rows, getting a list of paths. */
        strv = NULL;
        strc = 0;
        glist = clist->selection;
        while(glist != NULL)
        {
            /* Get pointer to selected row. */
            row = (gint)glist->data;

            /* Get disk object structure on selected row. */
            object = (edv_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if((object != NULL) ? (object->full_path != NULL) : FALSE)
            {
                i = strc;
                strc = i + 1;
                strv = (gchar **)g_realloc(
                    strv, strc * sizeof(gchar *)
                );
                if(strv == NULL)
                {
                    strc = 0;
                    break;
                }
                strv[i] = g_strdup(object->full_path);
            }

            /* Get next selected row. */
            glist = glist->next;
        }

        /* Emit signal indicating object removed for each path and
         * deallocate each path.
         */
        for(i = 0; i < strc; i++)
        {
            cstrptr = strv[i];
            if(cstrptr != NULL)
            {
                /* Check if object really dosen't exist, if it really
                 * does not exist then emit a disk object removed signal.
                 */
                if(lstat(cstrptr, &lstat_buf))
                    EDVObjectRemovedEmit(core_ptr, cstrptr);
            }

            /* Deallocate this path. */
            g_free(strv[i]);
            strv[i] = NULL;
        }
        /* Deallocate paths string array. */
        g_free(strv);
        strv = NULL;
        strc = 0;
}
