#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "fprompt.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvcfg.h"
#include "edvid.h"
#include "edvobj.h"
#include "edvfop.h"
#include "edvmimetypes.h"
#include "browser.h"
#include "browsercb.h"
#include "browserdirtree.h"
#include "browsercontents.h"
#include "endeavour.h"
#include "edvopen.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


static void EDVBrowserContentsFPromptCancelCB(gpointer data);

static void EDVBrowserContentsResetColumns(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCList *clist
);
static void EDVBrowserContentsClear(GtkCList *clist);

static void EDVBrowserContentsSetCellName(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
	gint row, gint column
);
static void EDVBrowserContentsSetCellSize(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
	gint row, gint column,
	gbool hide_dir_size, gbool hide_link_size
);
static void EDVBrowserContentsSetCellType(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
	gint row, gint column
);
static void EDVBrowserContentsSetCellPermissions(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
	gint row, gint column, gbool hide_link_permissions
);
static void EDVBrowserContentsSetCellOwner(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellGroup(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellDateAccess(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellDateModified(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellDateChanged(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellHardLinks(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellLinkedTo(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellDevice(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellINode(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellDeviceType(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellBlockSize(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetCellBlocks(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
);
static void EDVBrowserContentsSetRow(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row
);

static gint EDVBrowserContentsAppendObject(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object
);
static void EDVBrowserContentsGetListing(
        edv_browser_struct *browser, const gchar *path,
	gbool update_status_bar
);

gint EDVBrowserContentsFindRowByPath(
	edv_browser_struct *browser, const gchar *path
);

void EDVBrowserContentsResetRows(edv_browser_struct *browser);
void EDVBrowserContentsDoUpdate(
	edv_browser_struct *browser, const gchar *path,
        gbool update_status_bar
);

void EDVBrowserContentsDoOpenObject(
	edv_browser_struct *browser, gint row, gint column, guint state
);
void EDVBrowserContentsDoOpenWithObject(
        edv_browser_struct *browser, gint row, gint column
);

static void EDVBrowserContentsFPromptRenameApplyCB(
        gpointer data, const gchar *value
);
void EDVBrowserContentsDoFPromptRename(
        edv_browser_struct *browser, gint row, gint column
);

void EDVBrowserContentsObjectAddedNotify(
        edv_browser_struct *browser, const gchar *path,
        const struct stat *lstat_buf
);
void EDVBrowserContentsObjectModifiedNotify(
        edv_browser_struct *browser, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
);
void EDVBrowserContentsObjectRemovedNotify(
        edv_browser_struct *browser, const gchar *path
);

void EDVBrowserContentsMountNotify(
        edv_browser_struct *browser, edv_device_struct *dev_ptr,
        gbool is_mounted
);



#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      All purpose fprompt cancel callback.
 */
static void EDVBrowserContentsFPromptCancelCB(gpointer data)
{
        gpointer *cb_data = (gpointer *)data;
        if(cb_data == NULL)
            return;

        /* Deallocate callback data. */
        g_free(cb_data);
        cb_data = NULL;
}


/*
 *	Clears the titles of each column on the given clist.
 */
static void EDVBrowserContentsResetColumns(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCList *clist
)
{
	gint i, m, column_type, width;
	const gchar *title;
	gint justify;
	edv_intlist_struct *column_types_intlist, *column_width_intlist;


	if((core_ptr == NULL) || (browser == NULL) || (clist == NULL))
	    return;

	/* Get column types mapping. */
	column_types_intlist = EDVCFGItemListGetValueIntList(
	    core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
	    return;

	column_width_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH
        );
        if(column_width_intlist == NULL)
            return;


	/* Update clist column settings. */
        gtk_clist_column_titles_active(clist);
        gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(clist, FALSE);
	gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING);

	/* Change clist selection mode to GTK_SELECTION_EXTENDED.
	 * The selection mode can change whenever the contents list is
	 * updated.
	 */
/* Already set */
/*        gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED); */

	/* Iterate through each column but not exceeding the number of
	 * columns specified by the column types or column widths 
	 * intlists.
	 */
	m = MIN(
	    MIN(clist->columns, column_types_intlist->total),
	    column_width_intlist->total
	);
	for(i = 0; i < m; i++)
	{
	    /* Get column's type which determines what will be listed on
	     * the current column i.
	     */
	    column_type = column_types_intlist->i[i];

	    /* Get width of the column displaying this type. */
	    if((column_type >= 0) && (column_type < column_width_intlist->total))
		width = column_width_intlist->i[column_type];
	    else
		width = 0;

	    /* Title based on column type. */
	    switch(column_type)
	    {
	      case EDV_BROWSER_COLUMN_TYPE_NAME:
                title = "Name";
		justify = GTK_JUSTIFY_LEFT;
		break;

              case EDV_BROWSER_COLUMN_TYPE_SIZE:
                title = "Size";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_TYPE:
                title = "Type";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_PERMISSIONS:
                title = "Permissions";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_OWNER:
                title = "Owner";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_GROUP:
                title = "Group";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_DATE_ACCESS:
                title = "Date Access";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
                title = "Date Modified";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_DATE_CHANGED:
                title = "Date Changed";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_HARD_LINKS:
                title = "Hard Links";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_LINKED_TO:
                title = "Linked To";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_DEVICE:
                title = "Device";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_INODE:
                title = "INode";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
                title = "Device Type";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
                title = "Block Size";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_BROWSER_COLUMN_TYPE_BLOCKS:
                title = "Blocks";
                justify = GTK_JUSTIFY_RIGHT;
                break;

	      default:
		title = "";
                justify = GTK_JUSTIFY_LEFT;
		break;
	    }

	    gtk_clist_set_column_auto_resize(
		clist, i, FALSE
	    );
	    gtk_clist_set_column_title(clist, i, title);
	    gtk_clist_set_column_width(clist, i, width);
	    gtk_clist_set_column_justification(clist, i, justify);
	}
	/* Reset the rest of the columns incase the number of columns in
	 * the intlists were less than the columns specified by the clist.
	 */
	for(; i < clist->columns; i++)
	{
            gtk_clist_set_column_auto_resize(
                clist, i, FALSE
            );
            gtk_clist_set_column_title(clist, i, "");
            gtk_clist_set_column_width(clist, i, 0);
	}
}

/*
 *	Deletes all items in the given clist.
 */
static void EDVBrowserContentsClear(GtkCList *clist)
{
	if(clist == NULL)
            return;

	gtk_clist_clear(clist);
}



/*
 *	Sets the object name for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellName(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
        GdkPixmap *pixmap;
        GdkBitmap *mask;
	GtkStyle *style;


	if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
	   (object == NULL)
	)
	    return;


        /* Begin checking for which GtkStyle to be used. */
        style = NULL;

	switch(object->type)
	{
	  case EDV_OBJECT_TYPE_DIRECTORY:
	    /* Directory not accessable? */
	    if(!EDVCheckObjectAccessable(core_ptr, object))
		style = browser->cell_style[
		    EDV_BROWSER_CELL_STYLE_NO_ACCESS
		];
	    break;

	  case EDV_OBJECT_TYPE_LINK:
	    if(!object->link_valid)
		style = browser->cell_style[
		    EDV_BROWSER_CELL_STYLE_DANGLING_LINK
		];
	    break;

	  default:

	    break;
	}


	/* Get pixmap and mask for the given object. */
	EDVMatchObjectIcon(
	    core_ptr->device, core_ptr->total_devices,
	    core_ptr->mimetype, core_ptr->total_mimetypes,
	    object->type,
	    object->full_path,
	    object->link_valid, object->permissions,
	    0,			/* Small icons. */
	    &pixmap, &mask,
	    NULL, NULL, NULL, NULL
	);

	if(pixmap != NULL)
	    gtk_clist_set_pixtext(
		clist, row, column,
		(object->name != NULL) ?
		    object->name : "(null)",
		EDV_LIST_PIXMAP_TEXT_SPACING,
		pixmap, mask
	    );
	else
	    gtk_clist_set_text(
		clist, row, column,
		(object->name != NULL) ?
		    object->name : "(null)"
	    );

        /* Set cell style. */
        gtk_clist_set_cell_style(clist, row, column, style);
}

/*
 *      Sets the object size for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellSize(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column,
	gbool hide_dir_size, gbool hide_link_size
)
{
	const gchar *s;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	if(hide_dir_size && (object->type == EDV_OBJECT_TYPE_DIRECTORY))
	    s = "";
	else if(hide_link_size && (object->type == EDV_OBJECT_TYPE_LINK))
            s = "";
	else
	    s = EDVGetObjectSizeStr(core_ptr, object->size);
	gtk_clist_set_text(
	    clist, row, column, s
	);
}

/*
 *	Sets the object type for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellType(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	gchar *type_str;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	/* Get MIME Type type string for the given object. */
	EDVMatchObjectTypeString(
	    core_ptr->mimetype, core_ptr->total_mimetypes,
	    object->type, object->permissions,
	    object->full_path,
	    &type_str
	);

        gtk_clist_set_text(
            clist, row, column, (type_str != NULL) ? type_str : ""
        );
}

/*
 *      Sets the object permissions for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellPermissions(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column, gbool hide_link_permissions
)
{
	gchar s[80];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        if(hide_link_permissions && (object->type == EDV_OBJECT_TYPE_LINK))
            *s = '\0';
	else
	    sprintf(
		s,
		"%c%c%c%c%c%c%c%c%c",
	    (object->permissions & EDV_PERMISSION_UREAD) ? 'r' : '-',
	    (object->permissions & EDV_PERMISSION_UWRITE) ? 'w' : '-',
	    (object->permissions & EDV_PERMISSION_SETUID) ?
		'S' :
		((object->permissions & EDV_PERMISSION_UEXECUTE) ? 'x' : '-'),
	    (object->permissions & EDV_PERMISSION_GREAD) ? 'r' : '-',
	    (object->permissions & EDV_PERMISSION_GWRITE) ? 'w' : '-',
	    (object->permissions & EDV_PERMISSION_SETGID) ?
		'G' :
		((object->permissions & EDV_PERMISSION_GEXECUTE) ? 'x' : '-'),
	    (object->permissions & EDV_PERMISSION_AREAD) ? 'r' : '-',
	    (object->permissions & EDV_PERMISSION_AWRITE) ? 'w' : '-',
	    (object->permissions & EDV_PERMISSION_STICKY) ?
		'T' :
		((object->permissions & EDV_PERMISSION_AEXECUTE) ? 'x' : '-')
	    );

        gtk_clist_set_text(
            clist, row, column, s
        );
}

/*
 *      Sets the object owner for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellOwner(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	const gchar *owner_name;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	/* Get owner name from object's user id. */
	owner_name = EDVUIDGetNameFromUID(
	    core_ptr->uid, core_ptr->total_uids,
	    object->owner_id, NULL
	);
	if(owner_name == NULL)
	    owner_name = "";

        gtk_clist_set_text(
            clist, row, column, owner_name
        );
}

/*
 *      Sets the object group for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellGroup(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	const gchar *group_name;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	/* Get group name from object's group id. */
	group_name = EDVGIDGetNameFromGID(
            core_ptr->gid, core_ptr->total_gids,
            object->group_id, NULL
        );
        if(group_name == NULL)
            group_name = "";

        gtk_clist_set_text(
            clist, row, column, group_name
        );
}

/*
 *	Sets the object access date for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellDateAccess(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->access_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the object modified date for the specified cell on the given
 *      clist.
 */
static void EDVBrowserContentsSetCellDateModified(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->modify_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the object change date for the specified cell on the given
 *      clist.
 */
static void EDVBrowserContentsSetCellDateChanged(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	gint relativity;
	const gchar *format, *date_str;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	/* Get date relativity and format. */
	relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
	format = EDVCFGItemListGetValueS(
	    core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
	);

	/* Get date string. */
	date_str = EDVDateFormatString(
	    object->change_time, format, relativity
	);

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the object hard links for the specified cell on the given
 *      clist.
 */
static void EDVBrowserContentsSetCellHardLinks(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	gchar num_str[80];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	sprintf(num_str, "%i", object->hard_links);

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}

/*
 *	Sets the object link destination for the specified cell on the
 *	given clist.
 */
static void EDVBrowserContentsSetCellLinkedTo(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	const gchar *link_dest;
        GtkStyle *style;


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Begin checking for which GtkStyle to be used. */
        style = NULL;

        /* Dangling link?
	 *
	 * Note that we use the no access style for the link
	 * destination of a dangling link. The actual dangling link
	 * style is only used for the name of a dangling link.
	 */
        if(!object->link_valid)
            style = browser->cell_style[EDV_BROWSER_CELL_STYLE_NO_ACCESS];


	/* Get link destination. */
	link_dest = (object->linked_to != NULL) ? object->linked_to : "";

        gtk_clist_set_text(
            clist, row, column, link_dest
        );


	/* Set cell style. */
	gtk_clist_set_cell_style(clist, row, column, style);
}

/*
 *      Sets the object device for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellDevice(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	gchar num_str[256];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	sprintf(num_str, "%ld", object->device);

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}

/*
 *      Sets the object inode for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellINode(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
        gchar num_str[256];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        sprintf(num_str, "%ld", object->inode);

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}

/*
 *      Sets the object device type for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellDeviceType(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
	gint major, minor;
        gchar num_str[256];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	if((object->type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
	   (object->type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	{
	    EDVGetDeviceNumbers(
		(dev_t)object->device_type, &major, &minor
	    );
	    sprintf(num_str, "%i, %i", major, minor);
	}
	else
	{
	    *num_str = '\0';
	}

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}

/*
 *      Sets the object block size for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellBlockSize(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
        gchar num_str[256];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        sprintf(num_str, "%ld", object->block_size);

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}

/*
 *      Sets the object blocks for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellBlocks(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
        gint row, gint column
)
{
        gchar num_str[256];


        if((core_ptr == NULL) || (browser == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        sprintf(num_str, "%ld", object->blocks);

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}

/*
 *	Sets all cells of the specified row of the given contents clist
 *	with the values of the given object.
 *
 *	The clist row's client data will not be updated.
 */
static void EDVBrowserContentsSetRow(
        edv_core_struct *core_ptr, edv_browser_struct *browser,
        GtkCList *clist, edv_object_struct *object,
	gint row
)
{
        gint i, column_type, columns;
	gbool hide_dir_size, hide_link_size, hide_link_permissions;
        edv_intlist_struct *column_types_intlist;


        if((core_ptr == NULL) || (browser == NULL) ||
           (clist == NULL) || (object == NULL)
        )
	    return;


        /* Get column types mapping. */
        column_types_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
        );
        if(column_types_intlist == NULL)
            return;

	/* Get additional display options. */
	hide_dir_size = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
	    EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_DIR_SIZE
        );
        hide_link_size = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_LINK_SIZE
        );
	hide_link_permissions = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
	    EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_LINK_PERMISSIONS
        );


        /* Get total number of columns on the contents clist. */
        columns = MIN(clist->columns, column_types_intlist->total);

        /* Iterate through each column but not more than the columns
	 * specified by the column types intlist.
	 */
        for(i = 0; i < columns; i++)
        {
            /* Check column references on configuration list, see which
             * one this column should show.
             */
            column_type = column_types_intlist->i[i];
            switch(column_type)
            {
              case EDV_BROWSER_COLUMN_TYPE_NAME:
                EDVBrowserContentsSetCellName(
                    core_ptr, browser, clist, object,
                    row, i
                );
		break;

              case EDV_BROWSER_COLUMN_TYPE_SIZE:
                EDVBrowserContentsSetCellSize(
                    core_ptr, browser, clist, object,
                    row, i, hide_dir_size, hide_link_size
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_TYPE:
                EDVBrowserContentsSetCellType(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_PERMISSIONS:
                EDVBrowserContentsSetCellPermissions(
                    core_ptr, browser, clist, object,
                    row, i, hide_link_permissions
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_OWNER:
                EDVBrowserContentsSetCellOwner(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_GROUP:
                EDVBrowserContentsSetCellGroup(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_DATE_ACCESS:
                EDVBrowserContentsSetCellDateAccess(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
                EDVBrowserContentsSetCellDateModified(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_DATE_CHANGED:
                EDVBrowserContentsSetCellDateChanged(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_HARD_LINKS:
                EDVBrowserContentsSetCellHardLinks(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_LINKED_TO:
                EDVBrowserContentsSetCellLinkedTo(
                    core_ptr, browser, clist, object,
                    row, i
                );
		break;

              case EDV_BROWSER_COLUMN_TYPE_DEVICE:
                EDVBrowserContentsSetCellDevice(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_INODE:
                EDVBrowserContentsSetCellINode(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
                EDVBrowserContentsSetCellDeviceType(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
                EDVBrowserContentsSetCellBlockSize(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

              case EDV_BROWSER_COLUMN_TYPE_BLOCKS:
                EDVBrowserContentsSetCellBlocks(
                    core_ptr, browser, clist, object,
                    row, i
                );
                break;

            }
        }
}

/*
 *	Appends the disk object to the contents clist. The given object will
 *	be transfered to the new row and should not be referenced again
 *	after this call.
 *
 *	Returns the row number that it was appended to or -1 on failure.
 */
static gint EDVBrowserContentsAppendObject(
	edv_core_struct *core_ptr, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *object
)
{
	gint row, columns;
	gchar **row_text;


	if((core_ptr == NULL) || (browser == NULL) ||
           (clist == NULL) || (object == NULL)
	)
	{
            EDVObjectDelete(object);
            return(-1);
	}

        /* Get total number of columns on the contents clist. */
        columns = MAX(clist->columns, 1);


	/* Allocate row text. */
	row_text = (gchar **)g_malloc0(columns * sizeof(gchar *));
	if(row_text != NULL)
	{
	    gint i;

	    /* Allocate blank strings for each cell. */
	    for(i = 0; i < columns; i++)
		row_text[i] = g_strdup("");

	    /* Append a new row. */
	    row = gtk_clist_append(clist, row_text);

	    /* Deallocate row text. */
	    for(i = 0; i < columns; i++)
                g_free(row_text[i]);
	    g_free(row_text);
	    row_text = NULL;
	}
	else
	{
	    row = -1;
	}
	/* Successfully appended row? */
	if(row < 0)
	{
            EDVObjectDelete(object);
            return(-1);
	}

	/* Update each cell of the new row with the values from the
	 * disk object structure.
	 */
	EDVBrowserContentsSetRow(
	    core_ptr, browser, clist, object, row
	);

	/* Set given disk object structure as the row data for this new
	 * row. The disk object structure should not be referenced again
	 * after this point.
	 */
	gtk_clist_set_row_data_full(
	    clist, row,
	    object, (GtkDestroyNotify)EDVBrowserContentsItemDestroyCB
	);
        /* Object pointer has now been transfered to the new row's
         * data. Mark it NULL so we do not reference it again.
         */
        object = NULL;

	return(row);
}

/*
 *	Gets all child disk objects of the specified path and appends them
 *	to the contents clist.
 */
static void EDVBrowserContentsGetListing(
	edv_browser_struct *browser, const gchar *path,
	gbool update_status_bar
)
{
        gbool hide_object_hidden, hide_object_noaccess;
	GtkCList *clist;
	gint strc;
	gchar **strv;
        edv_core_struct *core_ptr;


        if((browser == NULL) || (path == NULL))
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

	clist = (GtkCList *)browser->contents_clist;
	if(clist == NULL)
	    return;

        hide_object_hidden = !EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
        );
        hide_object_noaccess = !EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
        );


        /* Get listing of contents in the directory specified by path. */
        strv = GetDirEntNames2(path, &strc);
        if(strv != NULL)
        {
            gint i, objects_loaded = 0;
	    gint last_progress_percent, progress_percent;
	    const gchar *cstrptr, *cstrptr2;
	    gchar *obj_path;
	    edv_object_struct *object;
	    gint stat_result;
	    struct stat stat_buf, lstat_buf;


            /* Sort strings. */
            strv = StringQSort(strv, strc);
            if(strv != NULL)
            {
		/* Iterate through directory entry names and pick out
		 * just the directories for the first itteration.
		 */
		last_progress_percent = -1;	/* None/undefined. */
		progress_percent = 0;
		obj_path = NULL;
		for(i = 0; i < strc; i++)
                {
#define FREE_AND_CONTINUE       \
{ \
 g_free(obj_path); \
 obj_path = NULL; \
\
 /* Do not deallocate strv[i] on the first itteration pass. */ \
\
 continue; \
}

		    cstrptr = strv[i];
		    if(cstrptr == NULL)
			FREE_AND_CONTINUE

		    /* Skip special dir notations. */
		    if(!strcmp(cstrptr, "."))
			FREE_AND_CONTINUE

		    /* Allocate full path to object as obj_path. */
                    cstrptr2 = PrefixPaths(path, cstrptr);
                    obj_path = (cstrptr2 != NULL) ?
                        g_strdup(cstrptr2) : NULL;
                    if(obj_path == NULL)
			FREE_AND_CONTINUE

		    /* Get destination object statistics. */
		    if(stat(obj_path, &stat_buf))
			FREE_AND_CONTINUE
		    /* Skip if object's destination is not a directory. */
		    if(!S_ISDIR(stat_buf.st_mode))
			FREE_AND_CONTINUE

		    /* Object's destination is a directory, now get
		     * local stats.
		     */
		    lstat(obj_path, &lstat_buf);

		    /* Create a new disk object structure, set it up, and
                     * add it to the list.
                     */
		    object = EDVObjectNew();
		    if(object != NULL)
		    {
			gint row;


			/* Set disk object structure. */
                        EDVObjectSetPath(object, obj_path);
			EDVObjectSetStat(object, &lstat_buf);
			object->link_valid = TRUE;

                        /* Begin filter checks. */
                        if((hide_object_hidden ?
                            EDVCheckObjectHidden(core_ptr, object) : FALSE) ||
                           (hide_object_noaccess ?
                            !EDVCheckObjectAccessable(core_ptr, object) : FALSE)
                        )
                        {
                            /* Filter check failed, delete disk object
                             * structure instead of appending it.
                             */
                            EDVObjectDelete(object);
                            object = NULL;
                            row = -1;
                        }
                        else
                        {
                            /* Append row using this disk object structure,
                             * the object structure will be taken by this
                             * function and should not be referenced afterwards.
                             */
                            row = EDVBrowserContentsAppendObject(
                                core_ptr, browser, clist, object
                            );
                            object = NULL;

                            objects_loaded++;   /* Count this as an object loading. */
                        }
		    }

                    /* Update progress? */
                    if(update_status_bar && (strc > 0))
                    {
                        progress_percent = objects_loaded * 100 / strc;
                        if(progress_percent > last_progress_percent)
                        {
                            EDVStatusBarProgress(
                                browser->status_bar,
                                (gfloat)objects_loaded / (gfloat)strc,
                                TRUE
                            );
                            progress_percent = last_progress_percent;
                        }
                    }

		    FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
		}

		/* Now iterate through all other objects, skipping
		 * directories. Note that this iteration will also delete
		 * all strings in strv.
		 *
		 * Do not reset last_progress_percent, leave it as the last
		 * value from the previous loop.
		 */
		obj_path = NULL;
		for(i = 0; i < strc; i++)
		{
#define FREE_AND_CONTINUE       \
{ \
 g_free(obj_path); \
 obj_path = NULL; \
\
 g_free(strv[i]); \
 strv[i] = NULL; \
\
 continue; \
}
                    cstrptr = strv[i];
		    if(cstrptr == NULL)
			FREE_AND_CONTINUE

                    /* Skip special dir notations. */
                    if(!strcmp(cstrptr, "."))
                        FREE_AND_CONTINUE

                    /* Allocate full path to object as obj_path. */
                    cstrptr2 = PrefixPaths(path, cstrptr);
                    obj_path = (cstrptr2 != NULL) ?
                        g_strdup(cstrptr2) : NULL;
                    if(obj_path == NULL)
                        FREE_AND_CONTINUE

                    /* Get local stats of the object. */
                    if(lstat(obj_path, &lstat_buf))
			FREE_AND_CONTINUE

		    /* Get destination stats of the object. */
		    stat_result = stat(obj_path, &stat_buf);
		    if(!stat_result)
		    {
			/* Skip if object destination is a directory. */
			if(S_ISDIR(stat_buf.st_mode))
			    FREE_AND_CONTINUE
		    }

		    /* Create a new disk object structure, set it up, and
		     * add it to the list.
		     */
                    object = EDVObjectNew();
                    if(object != NULL)
                    {
			gint row;


			/* Set disk object structure. */
                        EDVObjectSetPath(object, obj_path);
                        EDVObjectSetStat(object, &lstat_buf);
                        object->link_valid = stat_result ? FALSE : TRUE;

                        /* Begin filter checks. */
                        if((hide_object_hidden ?
                            EDVCheckObjectHidden(core_ptr, object) : FALSE)
/*                         ||
                           (hide_object_noaccess ?
                            !EDVCheckObjectAccessable(core_ptr, object) : FALSE)
 */
                        )
                        {
                            /* Filter check failed, delete disk object
                             * structure instead of appending it.
                             */
                            EDVObjectDelete(object);
                            object = NULL;
                            row = -1;
                        }
			else
			{
			    /* Append row using this disk object structure,
			     * the object structure will be taken by this
			     * function and should not be referenced afterwards.
			     */
			    row = EDVBrowserContentsAppendObject(
				core_ptr, browser, clist, object
			    );
			    object = NULL;

			    objects_loaded++;	/* Count this as an object loaded. */
			}
                    }

		    /* Update progress? */
		    if(update_status_bar && (strc > 0))
		    {
			progress_percent = objects_loaded * 100 / strc;
			if(progress_percent > last_progress_percent)
			{
			    EDVStatusBarProgress(
				browser->status_bar,
				(gfloat)objects_loaded / (gfloat)strc,
				TRUE
			    );
			    progress_percent = last_progress_percent;
			}
		    }

		    FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
		}

		/* At this point all strings in the strv pointer array have
		 * been deallocated.
		 */
	    }

	    /* Deallocate the string pointer array, each string should
	     * already be deallocated.
	     */
	    g_free(strv);
	    strv = NULL;
	}

}

/*
 *	Returns the row index of the contents clist that contains a disk
 *	object structure who's path matches the given path.
 *
 *	Can return -1 on failed match.
 */
gint EDVBrowserContentsFindRowByPath(
        edv_browser_struct *browser, const gchar *path
)
{
	gint i;
	GtkCList *clist;
	edv_object_struct *object;


	if((browser == NULL) || (path == NULL))
	    return(-1);

	clist = (GtkCList *)browser->contents_clist;
	if(clist == NULL)
	    return(-1);

	/* Iterate through all rows. */
	for(i = 0; i < clist->rows; i++)
	{
	    object = (edv_object_struct *)gtk_clist_get_row_data(
		clist, i
	    );
	    if(object == NULL)
		continue;

	    /* Full path not specified on disk object structure? */
	    if(object->full_path == NULL)
		continue;

	    /* Full paths match? */
	    if(!strcmp(object->full_path, path))
		return(i);
	}

	return(-1);
}

/*
 *	Updates all rows on the contents clist by getting the row data
 *	and updates the clist's rows.
 *
 *	This is designed to be called whenever the displayed values
 *	of each row need to be set again from internal data, usually
 *	when a MIME Type has been added/modified/removed. This function
 *	should not be used to `refresh' the list (get new values of
 *	disk object structures), use EDVBrowserContentsDoUpdate() 
 *	instead.
 */
void EDVBrowserContentsResetRows(edv_browser_struct *browser)
{
	gint i;
        GtkWidget *w;
        GtkCList *clist;
        edv_core_struct *core_ptr;
	edv_object_struct *object;


        if(browser == NULL)
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

        w = browser->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

	for(i = 0; i < clist->rows; i++)
	{
	    object = (edv_object_struct *)gtk_clist_get_row_data(
		clist, i
	    );
	    if(object != NULL)
		EDVBrowserContentsSetRow(
		    core_ptr, browser, clist, object, i
		);
	}
}

/*
 *	Procedure to clear and reupdate the contents list on the browser
 *	with the contents found in the directory specified by the given
 *	path.
 */
void EDVBrowserContentsDoUpdate(
        edv_browser_struct *browser, const gchar *path,
        gbool update_status_bar
)
{
	GtkWidget *w;
	GtkCList *clist;
	edv_core_struct *core_ptr;


	if(browser == NULL)
	    return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

	w = browser->contents_clist;
	if(w == NULL)
	    return;
	else
	    clist = GTK_CLIST(w);


	if(update_status_bar)
	{
	    /* Reset progress. */
	    EDVStatusBarMessage(
		browser->status_bar,
		"Loading directory contents...",
		FALSE
	    );
	    EDVStatusBarProgress(
		browser->status_bar, 0.0, TRUE
	    );
	}


	/* Delete all entries. */
	EDVBrowserContentsClear(clist);


	/* Update columns. */
	EDVBrowserContentsResetColumns(
	    core_ptr, browser, clist
	);

	/* Get listing of objects in the location specified by path. */
	EDVBrowserContentsGetListing(browser, path, update_status_bar);


	if(update_status_bar)
	{
	    /* Reset progress. */
	    EDVStatusBarMessage(
		browser->status_bar,
		"Directory contents loaded",
		FALSE
	    );
	    EDVStatusBarProgress(
		browser->status_bar, 0.0, TRUE
	    );
	}
}


/*
 *	Procedure to open the object listed on the given browser's
 *	contents clist specified by row and column.
 */
void EDVBrowserContentsDoOpenObject(
        edv_browser_struct *browser, gint row, gint column, guint state
)
{
	GtkWidget *w;
	GtkCList *clist;
	edv_object_struct *object;
	const gchar *name;
	gchar *full_path;
	struct stat stat_buf;


	if(browser == NULL)
	    return;

	w = browser->contents_clist;
	if(w == NULL)
	    return;

	clist = GTK_CLIST(w);


	/* Get object from row data. */
	object = (edv_object_struct *)gtk_clist_get_row_data(
	    clist, row
	);
	if(object == NULL)
	    return;

	/* Check name of object for special notations. */
	name = object->name;
	if(name != NULL)
	{
	    if(!strcmp(name, "."))
	    {
		full_path = NULL;
	    }
	    else if(!strcmp(name, ".."))
	    {
		const gchar *cstrptr = GetParentDir(
		    EDVBrowserCurrentLocation(browser)
		);
		full_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	    }
	    else
	    {
		full_path = (object->full_path != NULL) ?
		    g_strdup(object->full_path) : NULL;
	    }
	}
	else
	{
	    /* No name available, then just get a copy of the full path. */
	    full_path = (object->full_path != NULL) ?
		g_strdup(object->full_path) : NULL;
	}
	if(full_path == NULL)
	    return;

	/* Get stats of destination object, the object structure may
	 * have that information already but we want to ensure we have
	 * the most up to date information.
	 */
	if(stat(full_path, &stat_buf))
	{
	    g_free(full_path);
	    return;
	}

	/* Is destination a directory? */
	if(S_ISDIR(stat_buf.st_mode))
	{
	    /* Change selected path on the directory ctree to the given
	     * object if possible.
	     */
	    EDVBrowserDirTreeDoSelectPath(browser, full_path);
	}
	else
	{
	    gchar	*stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;
	    const gchar *command_name = NULL;

	    if(state & GDK_CONTROL_MASK)
		command_name = "edit";
	    else if(state & GDK_SHIFT_MASK)
                command_name = "edit";

	    EDVBrowserSetBusy(browser, TRUE);
	    EDVOpenObjectPath(
		(edv_core_struct *)browser->core_ptr, full_path,
		command_name,		/* Command name. */
		browser->toplevel, TRUE,
		&stdout_path_rtn, &stderr_path_rtn
	    );
            EDVBrowserSetBusy(browser, FALSE);

	    g_free(stdout_path_rtn);
	    stdout_path_rtn = NULL;
	    g_free(stderr_path_rtn);
	    stderr_path_rtn = NULL;
	}

	/* Deallocate copy of full path. */
	g_free(full_path);
	full_path = NULL;
}

/*
 *      Procedure to `open with' the object listed on the given browser's
 *      contents clist specified by row and column.
 */
void EDVBrowserContentsDoOpenWithObject(
        edv_browser_struct *browser, gint row, gint column
)
{
        GtkWidget *w;
        GtkCList *clist;
        edv_object_struct *object;
        const gchar *name;
        gchar *full_path;
        struct stat stat_buf;


        if(browser == NULL)
            return;

        w = browser->contents_clist;
        if(w == NULL)
            return;

        clist = GTK_CLIST(w);


        /* Get object from row data. */
        object = (edv_object_struct *)gtk_clist_get_row_data(
            clist, row
        );
        if(object == NULL)
            return;

        /* Check name of object for special notations. */
        name = object->name;
        if(name != NULL)
        {
            if(!strcmp(name, "."))
	    {
                full_path = NULL;
            }
            else if(!strcmp(name, ".."))
            {
                const gchar *cstrptr = GetParentDir(
                    EDVBrowserCurrentLocation(browser)
                );
                full_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
            }
            else
            {
                full_path = (object->full_path != NULL) ?
                    g_strdup(object->full_path) : NULL;
            }
        }
        else
        {
            /* No name available, then just get a copy of the full path. */
            full_path = (object->full_path != NULL) ?
                g_strdup(object->full_path) : NULL;
        }
        if(full_path == NULL)
            return;

        /* Get stats of destination object, the object structure may
         * have that information already but we want to ensure we have
         * the most up to date information.
         */
        if(stat(full_path, &stat_buf))
        {
            g_free(full_path);
            return;
        }

	if(1)
	{
            gchar       *stdout_path_rtn = NULL,
                        *stderr_path_rtn = NULL;

            EDVOpenWithObjectPath(
                (edv_core_struct *)browser->core_ptr, full_path,
                NULL,                   /* Command name. */
                browser->toplevel, TRUE,
                &stdout_path_rtn, &stderr_path_rtn
            );

            g_free(stdout_path_rtn);
            stdout_path_rtn = NULL;
            g_free(stderr_path_rtn);
            stderr_path_rtn = NULL;
        }

        /* Deallocate copy of full path. */
        g_free(full_path);
        full_path = NULL;
}


/*
 *      FPrompt apply callback, set in EDVBrowserContentsDoFPromptRename().
 */
static void EDVBrowserContentsFPromptRenameApplyCB(
        gpointer data, const gchar *value
)
{
        edv_browser_struct *browser;
        GtkCList *clist;
	gint row;
        gpointer *cb_data = (gpointer *)data;
        if(cb_data == NULL)
            return;

        /* Get callback data. */
        browser = (edv_browser_struct *)cb_data[0];
        clist = (GtkCList *)cb_data[1];
	row = (gint)cb_data[2];

        /* Inputs valid? */
        if((browser != NULL) && (clist != NULL) && (row >= 0) &&
           (value != NULL)
        )
        {
	    edv_core_struct *core_ptr = (edv_core_struct *)browser->core_ptr;
            edv_object_struct *object = (edv_object_struct *)gtk_clist_get_row_data(
                clist, row
            );


	    /* Check if the selected object's disk object structure is
	     * valid.
	     */
            if((object != NULL) ? (object->full_path != NULL) : FALSE)
            {
		gchar *old_full_path = g_strdup(object->full_path);
		gchar *new_obj = NULL;
		const gchar *error_mesg;
		gbool yes_to_all = FALSE;
		gint status;


		/* Perform rename. */
		status = EDVFOPRename(
		    core_ptr, old_full_path, value,
		    &new_obj, browser->toplevel,
		    FALSE, TRUE,
		    &yes_to_all
		);

                /* Unmap progress dialog, it may have been mapped in the
                 * above operation.
                 */
                ProgressDialogBreakQuery(FALSE);
                ProgressDialogSetTransientFor(NULL);

		/* Get error message if any that might have occured in the
		 * above operation.
		 */
                error_mesg = EDVFOPGetError();
                if(error_mesg != NULL)
                {
                    CDialogSetTransientFor(browser->toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

		/* Got new object full path name (implying success)? */
		if((new_obj != NULL) && (old_full_path != NULL))
		{
		    struct stat lstat_buf;
		    const gchar *new_child, *old_child;


		    /* Get child object names. */
		    new_child = strrchr(new_obj, DIR_DELIMINATOR);
		    if(new_child == NULL)
			new_child = new_obj;
		    else
			new_child++;

		    old_child = strrchr(old_full_path, DIR_DELIMINATOR);
                    if(old_child == NULL)
                        old_child = old_full_path;
                    else
                        old_child++;

		    /* Get new local statistics for the renamed object. */
		    if(!lstat(new_obj, &lstat_buf))
		    {
                        gchar *buf = g_strdup_printf(
			    "Object \"%s\" renamed to \"%s\"",
			    old_child, new_child
			);
                        EDVStatusBarMessage(
                            browser->status_bar, buf, FALSE
                        );
                        g_free(buf);

                        /* Emit object modified signal to all windows. */
			EDVObjectModifiedEmit(
			    core_ptr, old_full_path,
			    new_obj, &lstat_buf
			);
		    }
		}
                else
                {
                    /* Did not get new object path new_obj, implying failed. */
                    EDVStatusBarMessage(
                        browser->status_bar, "Rename object failed", FALSE
                    );
                }

		/* The disk object structure may now be invalid if the
		 * object modified signal was emitted.
		 */
		object = NULL;


		/* Deallocate coppies of paths. */
		g_free(new_obj);
		g_free(old_full_path);
            }
        }

        /* Deallocate callback data, it is no longer needed. */
        g_free(cb_data);
        cb_data = NULL;
}

/*
 *	Procedure to map the floating prompt for renaming of the row
 *	specified by row.
 *
 *      If the column is -1 then it implies any column is valid.
 */
void EDVBrowserContentsDoFPromptRename(
        edv_browser_struct *browser, gint row, gint column
)
{
	gint i, column_type_name;
	gint cx, cy, px, py, pwidth, pheight;
        edv_intlist_struct *column_type_intlist;
        edv_object_struct *object;
	edv_core_struct *core_ptr;
        GtkWidget *w;
        GtkCList *clist;


        if((browser == NULL) || (row < 0))
            return;

        if(FPromptIsQuery())
            return;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;

	/* Get contents clist widget. */
        w = browser->contents_clist;
        if(w == NULL)
            return;
        clist = GTK_CLIST(w);


        /* Sync data on browser so as to ensure we have the most up to
         * date information to send out.
         */
        EDVBrowserSyncData(browser);


        /* Get column index for column that is displaying name (if any). */
        column_type_name = -1;
        column_type_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
        );
        if(column_type_intlist != NULL)
        {
	    /* Iterate through column type intlist. */
            for(i = 0; i < column_type_intlist->total; i++)
            {
		/* Is column i displaying the name? */
                if(column_type_intlist->i[i] == EDV_BROWSER_COLUMN_TYPE_NAME)
		{
                    column_type_name = i;
		    break;
		}
            }
        }
	/* No column displaying name? */
	if(column_type_name < 0)
	    return;

	/* Make sure given row index is in bounds. */
	if((row < 0) || (row >= clist->rows))
	    return;

        /* Get clist cell geometry. */
        if(GUICListGetCellGeometry(
            clist, column_type_name, row,
            &cx, &cy, &pwidth, &pheight
        ))
            return;

        /* Get root window relative coordinates. */
        px = 0;
        py = 0;
        gdk_window_get_deskrelative_origin(
            clist->clist_window, &px, &py
        );
        px += cx + 0;
        py += cy - 2;   /* Move up a little. */

        /* Get disk object structure from the contents clist cell. */
        object = (edv_object_struct *)gtk_clist_get_row_data(
            clist, row
        );
        if(object == NULL)
            return;

	/* Check if object name is a special notation that should not be
	 * allowed to be renamed.
	 */
	if(object->name != NULL)
	{
	    const gchar *name = object->name;
	    if(!strcmp(name, ".") || !strcmp(name, "..") ||
               !strcmp(name, "/")
	    )
		return;
	}

	/* Is the clicked on column the name column or given column is -1
	 * implying any column is acceptable?
	 */
	if((column == column_type_name) || (column < 0))
	{
            gpointer *cb_data = (gpointer *)g_malloc0(
                3 * sizeof(gpointer)
            );
            gchar *value = (object->name != NULL) ?
                g_strdup(object->name) : NULL;

            /* Set up callback data. */
            if(cb_data != NULL)
            {
                cb_data[0] = browser;
                cb_data[1] = clist;
                cb_data[2] = (gpointer)row;
            }

            /* Map floating prompt to change values. */
            FPromptSetTransientFor(browser->toplevel);
            FPromptSetPosition(px, py);
            FPromptMapQuery(
                NULL,                   /* No label. */
                value,
                NULL,                   /* No tooltip message. */
                FPROMPT_MAP_NO_MOVE,    /* Map code. */
                pwidth, -1,             /* Width and height. */
/*              FPROMPT_FLAG_OK_BTN | FPROMPT_FLAG_CANCEL_BTN, */
                0,      /* No buttons. */
                (gpointer)cb_data,      /* Callback data. */
                NULL,                   /* No browse callback. */
                EDVBrowserContentsFPromptRenameApplyCB,
                EDVBrowserContentsFPromptCancelCB
            );

	    /* Do not reference cb_data after this call, it will be passed
	     * to the callbacks where it will be deallocated.
	     */
	    cb_data = NULL;

            /* Deallocate original value. */
            g_free(value);
	    value = NULL;
        }
}


/*
 *      This should be called whenever a new object has been added, it
 *      will add a new row as needed to represent the new object.
 *
 *      The given path must be an absolute path to the object.
 */
void EDVBrowserContentsObjectAddedNotify(
        edv_browser_struct *browser, const gchar *path,
        const struct stat *lstat_buf
)
{
	const gchar *cstrptr;
	gchar *cur_path = NULL, *parent_path = NULL;
        GtkWidget *w;
        GtkCList *clist;
	gint row;
        edv_core_struct *core_ptr;
        edv_object_struct *object;
	gint stat_result;
	struct stat stat_buf;


        if((browser == NULL) || (path == NULL) || (lstat_buf == NULL))
            return;

        core_ptr = (edv_core_struct *)browser->core_ptr;
        if(core_ptr == NULL)
            return;

        w = browser->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

#define DO_FREE_LOCALS	\
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(path);
	parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	if((cur_path == NULL) || (parent_path == NULL))
	{
	    DO_FREE_LOCALS
	    return;
	}

	/* Is location of what contents list is displaying is different
	 * from the location of the new object to add?
	 */
	if(strcmp(cur_path, parent_path))
        {
            DO_FREE_LOCALS
            return;
        }


	/* Get destination stats. */
	stat_result = stat(path, &stat_buf);


	/* Check if the new path of the object to add reffers to an object
	 * that already exists in the list.
	 */
	row = EDVBrowserContentsFindRowByPath(browser, path);
	if((row >= 0) && (row < clist->rows))
	{
	    /* The new object to add already exists in the list, so just
	     * update the row.
	     */

	    /* Get disk object structure from matched node. */
	    object = (edv_object_struct *)gtk_clist_get_row_data(
		clist, row
	    );
	    if(object != NULL)
	    {
		/* Update disk object structure. */
		EDVObjectSetPath(object, path);
                EDVObjectSetStat(object, lstat_buf);
		object->link_valid = stat_result ? FALSE : TRUE;

		/* Update the contents clist row with the new values from
		 * this disk object structure.
		 */
		EDVBrowserContentsSetRow(
		    core_ptr, browser, clist, object,
		    row
		);
	    }
	}
	else
	{
	    /* Create a new disk object structure. */
	    object = EDVObjectNew();
	    if(object != NULL)
	    {
		gint new_row;


		/* Set disk object structure. */
		EDVObjectSetPath(object, path);
		EDVObjectSetStat(object, lstat_buf);
                object->link_valid = stat_result ? FALSE : TRUE;

		/* Append row using this disk object structure. */
		new_row = EDVBrowserContentsAppendObject(
		    core_ptr, browser, clist, object
		);
		/* The object structure is now invalid after passing it to
		 * EDVBrowserContentsAppendObject().
		 */
		object = NULL;
	    }
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      This should be called whenever a object has been modified, it will
 *      search for the object and then reupdate the matching row.
 *
 *      The given path must be an absolute path to the object and must be
 *      the path of the object's original name. The new_path must be an
 *      absolute path to the object at its new name, the new_path may be
 *      NULL if there was no name change.
 */
void EDVBrowserContentsObjectModifiedNotify(
        edv_browser_struct *browser, const gchar *path,
        const gchar *new_path,
        const struct stat *lstat_buf
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *parent_path = NULL;
        GtkWidget *w;
        GtkCList *clist;
        gint row;
	edv_core_struct *core_ptr;
	edv_object_struct *object;
	gint stat_result;
	struct stat stat_buf;


        if((browser == NULL) || (path == NULL) || (lstat_buf == NULL))
            return;

	if(new_path == NULL)
	    new_path = path;

	core_ptr = (edv_core_struct *)browser->core_ptr;
	if(core_ptr == NULL)
	    return;

        w = browser->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(new_path);
        parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (parent_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }


        /* Get destination stats. */
        stat_result = stat(new_path, &stat_buf);


        /* Is given path that has been modified the same as the current
         * location?
         */
        if(!strcmp(new_path, cur_path))
        {
	    /* Reget contents listing for the new path. */
	    gtk_clist_freeze(clist);
	    EDVBrowserContentsDoUpdate(browser, new_path, TRUE);
	    gtk_clist_thaw(clist);
	}
	else
	{
	    /* Look for a row who's disk object full path matches the
	     * old path.
	     */
	    row = EDVBrowserContentsFindRowByPath(browser, path);
	    if((row >= 0) && (row < clist->rows))
	    {
		/* Get disk object structure from matched row. */
		object = (edv_object_struct *)gtk_clist_get_row_data(
		    clist, row
		);
		if(object != NULL)
		{
		    /* Update disk object structure. */
                    EDVObjectSetPath(object, new_path);		/* Use new_path. */
		    EDVObjectSetStat(object, lstat_buf);
		    object->link_valid = stat_result ? FALSE : TRUE;

		    /* Update the contents clist row with the new values
		     * from the disk object structure.
		     */
		    EDVBrowserContentsSetRow(
			core_ptr, browser, clist, object, row
		    );
		}
	    }
        }

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      This should be called whenever a object has been removed, it will
 *      search for the object and then remove the matching row.
 *
 *      The given path must be an absolute path to the object.
 */
void EDVBrowserContentsObjectRemovedNotify(
        edv_browser_struct *browser, const gchar *path
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *parent_path = NULL;
        GtkWidget *w;
        GtkCList *clist;
	gint row;


        if((browser == NULL) || (path == NULL))
            return;

        w = browser->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(path);
        parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (parent_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }

	/* Is given path that has been removed the same as the current
	 * location?
	 */
        if(!strcmp(path, cur_path))
	{
	    /* Clear entire contents clist. */
	    gtk_clist_freeze(clist);
	    EDVBrowserContentsClear(clist);
	    gtk_clist_thaw(clist);
	}
	else
	{
	    /* Look for a row that matches the given path. */
	    row = EDVBrowserContentsFindRowByPath(browser, path);
	    if((row >= 0) && (row < clist->rows))
	    {
		/* Remove matched row. */
		gtk_clist_remove(clist, row);
	    }
	}

	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}


/*
 *      This should be called whenever a device has been mounted or
 *	unmounted.
 */
void EDVBrowserContentsMountNotify(
        edv_browser_struct *browser, edv_device_struct *dev_ptr,
	gbool is_mounted
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *mount_path = NULL;
        GtkWidget *w;
        GtkCList *clist;


        if((browser == NULL) || (dev_ptr == NULL))
            return;

        w = browser->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(mount_path); \
 mount_path = NULL; \
}
        /* Get coppies of current location and mount paths. */
        cstrptr = EDVBrowserCurrentLocation(browser);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	cstrptr = dev_ptr->mount_path;
	mount_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (mount_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }

	/* Need to simplify coppies of current and mount paths. */
	EDVSimplifyPath(cur_path);
	EDVSimplifyPath(mount_path);


	/* Check if mount path was the current location, if it was then
	 * the contents clist needs to be updated.
	 */
        if(!strcmp(cur_path, mount_path))
        {
            /* Reget contents listing for the new path. */
            gtk_clist_freeze(clist);
            EDVBrowserContentsDoUpdate(browser, cur_path, TRUE);
            gtk_clist_thaw(clist);
	}

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}
