#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvobj.h"
#include "edvcfg.h"
#include "edvconfirm.h"
#include "edvarch.h"
#include "edvarchfio.h"
#include "edvarchop.h"
#include "archiveopts.h"
#include "archiver.h"
#include "archivercb.h"
#include "archivercontents.h"
#include "archiverdnd.h"
#include "edvdde.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"


static gint EDVArchiverDNDConfirm(
        edv_core_struct *core_ptr,
	gint gdk_action, guint info,
        GtkWidget *toplevel,
        edv_dnd_object_struct **dnd_object, gint total_dnd_objects,
        const gchar *arch_path
);
static void EDVArchiverDNDUpdateStatusBar(
        edv_archiver_struct *archiver,
	gint gdk_action, guint info,
	gint total_src_objects,
	gint total_objects_processed, gint status
);

static void EDVArchiverDragDataReceivedNexus(
	edv_core_struct *core_ptr,
	edv_archiver_struct *archiver,	/* Can be NULL. */
        GdkDragContext *dc, guint info, GtkSelectionData *selection_data
);

gint EDVArchiverLocBarIconCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void EDVArchiverContentsDNDSetIcon(
        edv_archiver_struct *archiver, gint row, gint column
);
gboolean EDVArchiverContentsDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,
        gpointer data
);
void EDVArchiverContentsDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVArchiverContentsDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
);
void EDVArchiverContentsDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
);



/*
 *      All purpose initial operation confirmation procedure, prompts
 *      the user for response for confirmation of pending DND operation
 *      specified by the given inputs.
 *
 *      If the configuration will be checked first to se if confirmation
 *      is needed. If it is not needed then CDIALOG_RESPONSE_YES will
 *      be returned.
 *
 *      Returns a CDIALOG_RESPONSE_* code.
 */
static gint EDVArchiverDNDConfirm(
        edv_core_struct *core_ptr,
	gint gdk_action, guint info,
        GtkWidget *toplevel,
        edv_dnd_object_struct **dnd_object, gint total_dnd_objects,
	const gchar *arch_path
)
{
        const gchar *src_path = NULL;


        if(total_dnd_objects <= 0)
            return(CDIALOG_RESPONSE_NOT_AVAILABLE);

        /* If exactly one source object then get source path of that
         * object. Otherwise leave src_path as NULL.
         */
        if(total_dnd_objects == 1)
        {
            edv_dnd_object_struct *dnd_obj = dnd_object[0];
            if(dnd_obj != NULL)
                src_path = dnd_obj->full_path;
        }

        /* Do confirmation, handle by dnd target type. */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
        )
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_LINK:
		return(EDVConfirmArchiveAdd(
		    core_ptr, toplevel,
		    src_path, total_dnd_objects,
		    arch_path
		));
                break;
            }
        }
        else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        {
            /* Skip this, a recycled object cannot be added to the
             * archive.
             */
	}
        else if(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        {
            /* Skip this, an archive object cannot be added back into
             * the archive because it is already there.
             */
        }
        else
        {
            /* Unsupported target type. */
        }

        /* Unsupported DND target type or drag action, return response not
         * available just to be on the safe side so that the operation
         * does not continue.
         */
        return(CDIALOG_RESPONSE_NOT_AVAILABLE);
}

/*
 *      Updates the status bar to indicate the result of the dnd operation
 *      on the target archiver.
 */
static void EDVArchiverDNDUpdateStatusBar(
        edv_archiver_struct *archiver,
	gint gdk_action, guint info,
	gint total_src_objects,
        gint total_objects_processed, gint status
)
{
        gchar *buf = NULL;
        edv_status_bar_struct *sb;


        if(archiver == NULL)
            return;

        /* Get status bar. */
        sb = archiver->status_bar;
        if(sb == NULL)
            return;

        /* Begin formatting status message, handle by dnd target type. */
        if((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
           (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
           (info == EDV_DND_TYPE_INFO_STRING)
        )
        {
            switch(gdk_action)
            {
              case GDK_ACTION_COPY:
              case GDK_ACTION_MOVE:
              case GDK_ACTION_LINK:
		switch(status)
                {
                  case 0: case -5:
                    buf = g_strdup_printf(
                        "Added %i object%s",
                        total_objects_processed,
			(total_objects_processed == 1) ? "" : "s"
                    );
                    break;

                  case -4:  /* Cancel. */
		    buf = g_strdup_printf(
			"Add operation canceled"
		    );
		    break;

		  default:  /* Error. */
		    buf = g_strdup_printf(
                        "Unable to add object%s",
                        (total_src_objects == 1) ? "" : "s"
                    );
                    break;
                }
                break;
	    }
        }
        else if(info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        {
            /* Skip this, a recycled object cannot be added to the
	     * archive.
             */
	}
        else if(info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        {
            /* Skip this, an archive object cannot be added back into
             * the archive because it is already there.
             */
        }
        else
        {
            /* Unsupported target type. */
        }

        /* Set status bar message. */
        EDVStatusBarMessage(sb, buf, FALSE);

        /* Deallocate status bar message buffer. */
        g_free(buf);
	buf = NULL;
}


/*
 *	Data received nexus, all inputs assumed valid.
 */
static void EDVArchiverDragDataReceivedNexus(
        edv_core_struct *core_ptr,
	edv_archiver_struct *archiver,	/* Can be NULL. */
        GdkDragContext *dc, guint info, GtkSelectionData *selection_data
)
{
	gint initial_confirmation_result;
	GtkWidget *toplevel = (archiver != NULL) ? archiver->toplevel : NULL;
	edv_dnd_object_struct **dnd_object, *dnd_obj;
	gint total_dnd_objects;


#define DO_FREE_LOCALS	\
{ \
 gint i; \
\
 /* Deallocate dnd reference structures. */ \
 for(i = 0; i < total_dnd_objects; i++) \
  EDVDNDObjectDelete(dnd_object[i]); \
 g_free(dnd_object); \
 dnd_object = NULL; \
 total_dnd_objects = 0; \
\
}

        /* Parse the given selection buffer into an array of dnd object
         * structures.
         */
        dnd_object = EDVDNDBufferToObject(
            (const guint8 *)selection_data->data,
            selection_data->length,
            &total_dnd_objects
        );

	/* Check if there is exactly one dnd disk object parsed from the
	 * buffer and if it is an archive. If it is then we should give
	 * the user a choice to open this as an archive instead of adding
	 * it to the current archive.
	 */
	if((total_dnd_objects == 1) && (archiver != NULL))
	{
	    dnd_obj = dnd_object[0];
	    if((dnd_obj != NULL) ? (dnd_obj->full_path != NULL) : FALSE)
	    {
		if(EDVCheckEDVArchiverArchive(core_ptr, dnd_obj->full_path))
		{
		    /* There is exactly one dnd disk object and it is an
		     * archive. So query user for opening it as an archive
		     * instead of adding it to the current archive.
		     */
		    gint status;
		    const gchar *name;
		    gchar *buf;

		    name = strrchr(dnd_obj->full_path, DIR_DELIMINATOR);
		    if(name == NULL)
			name = dnd_obj->full_path;
		    else
			name++;

		    buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
			"Open archive `%s' instead of adding it?",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                        "El archivo abierto `%s' en vez de agregarlo?",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                        "L'archive ouverte `%s' au lieu de l'addition il?",
#endif
			name
		    );
                    CDialogSetTransientFor(toplevel);
                    status = CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
                        "Open Archive?",
#endif
#ifdef PROG_LANGUAGE_SPANISH
			"El Archivo Obierto",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                        "L'Archive Ouverte",
#endif
                        buf,
                        NULL,
                        CDIALOG_ICON_QUESTION,
                        CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
                        CDIALOG_BTNFLAG_YES
                    );
                    CDialogSetTransientFor(NULL);

		    g_free(buf);

		    if(status == CDIALOG_RESPONSE_YES)
		    {
			EDVArchiverSetBusy(archiver, TRUE);
			GUIBlockInput(toplevel, TRUE);

			/* Clear the contents clist and load the listing of
			 * the new archive.
			 */
			EDVArchiverSelectArchive(archiver, dnd_obj->full_path);

			GUIBlockInput(toplevel, FALSE);
			EDVArchiverSetBusy(archiver, FALSE);

			EDVArchiverUpdateMenus(archiver);


			DO_FREE_LOCALS

			return;
		    }
		}
	    }
	}

        /* Check and warn if write protect is enabled, this needs to be
	 * checked here because prior call to this function will not
	 * check this because the drag operation may be just to open an
	 * archive.
	 */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
        {
	    DO_FREE_LOCALS
            return;
        }

	/* Make initial user confirmation querying to proceed with this
	 * operation.
	 */
	initial_confirmation_result = EDVArchiverDNDConfirm(
	    core_ptr, (gint)dc->action, info,
	    toplevel,
	    dnd_object, total_dnd_objects,
	    EDVArchiverCurrentLocation(archiver)
	);
        /* User confirmed and dnd target type is a disk object? */
        if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
           ((info == EDV_DND_TYPE_INFO_TEXT_PLAIN) ||
            (info == EDV_DND_TYPE_INFO_TEXT_URI_LIST) ||
            (info == EDV_DND_TYPE_INFO_STRING)
           )
        )
        {
            gint i, n, status = -1;
            gbool yes_to_all = FALSE;
	    gbool recurse = TRUE;
	    gint compression = 50;	/* 0 to 100. */
	    gbool dereference_links = FALSE;
            gint objects_added = 0;
	    gchar **new_aobj_path;
	    gint total_new_aobj_paths;
	    const gchar *arch_obj;
            const gchar *error_mesg;


	    /* Get current archive path. */
	    arch_obj = EDVArchiverCurrentLocation(archiver);


	    if(archiver != NULL)
	    {
		EDVArchiverSetBusy(archiver, TRUE);
		archiver->processing = TRUE;
	    }


            /* Get additional add to archive options. */
            if(!EDVArchiveAddQueryOptions(
                core_ptr, toplevel, arch_obj,
		&recurse, &compression, &dereference_links
            ))
            {
                /* User canceled. */
		if(archiver != NULL)
		{
		    archiver->processing = FALSE;
		    EDVArchiverSetBusy(archiver, FALSE);
		}
		DO_FREE_LOCALS
		return;
	    }

	    /* Add dnd object references to archive. */
	    if(total_dnd_objects > 0)
	    {
		gint total_tar_paths = total_dnd_objects;
		gchar **tar_path = (gchar **)g_malloc0(
		    total_tar_paths * sizeof(gchar *)
		);
		for(i = 0; i < total_tar_paths; i++)
		{
		    dnd_obj = dnd_object[i];
		    if((dnd_obj != NULL) ? (dnd_obj->full_path == NULL) : TRUE)
			continue;

		    tar_path[i] = g_strdup(dnd_obj->full_path);
		}

		error_mesg = NULL;
                status = -1;
                new_aobj_path = NULL;
                total_new_aobj_paths = 0;
		switch((gint)dc->action)
		{
                  case GDK_ACTION_COPY:
                    /* Add object to the archive. */
                    status = EDVArchOPAdd(
                        core_ptr, arch_obj, tar_path, total_tar_paths,
                        &new_aobj_path, &total_new_aobj_paths,
                        toplevel, TRUE, TRUE, &yes_to_all,
                        recurse, compression, dereference_links
                    );
                    break;
                  case GDK_ACTION_MOVE:
                    error_mesg = "Cannot move an object into the archive";
                    break;
                  case GDK_ACTION_LINK:
                    error_mesg = "Cannot link an object into the archive";
                    break;
                  default:
                    error_mesg = "Unsupported drag operation";
                    break;
                }

                for(i = 0; i < total_tar_paths; i++)
		    g_free(tar_path[i]);
		g_free(tar_path);
		tar_path = NULL;
		total_tar_paths = 0;

                /* Get error message (if any) that might have occured
                 * in the above operation.
                 */
                if(error_mesg == NULL)
                    error_mesg = EDVArchOPGetError();
                if((error_mesg != NULL) && (status != -4))
                {
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }

                /* Object was added successfully? */
                if((total_new_aobj_paths > 0) && !status)
                {
                    objects_added += total_new_aobj_paths;
                }

                /* Deallocate returned new archive object paths. */
                for(n = 0; n < total_new_aobj_paths; n++)
                    g_free(new_aobj_path[n]);
                g_free(new_aobj_path);
                new_aobj_path = NULL;
                total_new_aobj_paths = 0;
	    }

	    /* Notify about archive's change in size, this will cause
	     * the archiver to reget the listing.
	     */
            if(!status && (arch_obj != NULL))
            {
                struct stat lstat_buf;

                if(!lstat(arch_obj, &lstat_buf))
		{
		    if(archiver != NULL)
                        archiver->processing = FALSE;

                    EDVObjectModifiedEmit(
                        core_ptr, arch_obj, arch_obj, &lstat_buf
                    );

                    if(archiver != NULL)
                        archiver->processing = TRUE;
		}
            }


            /* Update status bar message. */
	    if(archiver != NULL)
		EDVArchiverDNDUpdateStatusBar(
		    archiver, (gint)dc->action, info,
		    total_dnd_objects, objects_added, status
		);

            /* Unmap progress dialog, it may have been mapped if any
             * operations occured in the above loop.
             */
            ProgressDialogBreakQuery(TRUE);
            ProgressDialogSetTransientFor(NULL);

	    if(archiver != NULL)
	    {
		archiver->processing = FALSE;
		EDVArchiverSetBusy(archiver, FALSE);
	    }
        }
        /* User confirmed and dnd target type is a recycled object? */
        else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
                (info == EDV_DND_TYPE_INFO_RECYCLED_OBJECT)
        )
        {
            /* Recycled objects cannot be added to an archive. */
        }
        /* User confirmed and dnd target type is an archive object? */
        else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
                (info == EDV_DND_TYPE_INFO_ARCHIVE_OBJECT)
        )
        {
	    /* Archive objects cannot be added back into an archive. */
        }


	DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}


/*
 *      Location bar icon "enter_notify_event" or "leave_notify_event"
 *      signal callback.
 */
gint EDVArchiverLocBarIconCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        gint status = FALSE;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((widget == NULL) || (crossing == NULL) || (archiver == NULL))
            return(status);

        switch((gint)crossing->type)
        {
          case GDK_ENTER_NOTIFY:
            gtk_widget_set_state(widget, GTK_STATE_PRELIGHT);
            EDVStatusBarMessage(
                archiver->status_bar,
#ifdef PROG_LANGUAGE_ENGLISH
"Drag this to create a link to this location",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Arrastre esto crear un eslabn a esta ubicacin",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Traner ceci pour crer un lien  cet emplacement",
#endif
                FALSE
            );
            if(archiver->location_icon_pm != NULL)
            {
                GdkBitmap *mask = NULL;
                GdkPixmap *pixmap = NULL;
                gtk_pixmap_get(
                    GTK_PIXMAP(archiver->location_icon_pm),
                    &pixmap, &mask
                );
                if(pixmap != NULL)
                {
                    gint w = 15, h = 15;
                    gdk_window_get_size((GdkWindow *)pixmap, &w, &h);
                    GUIDNDSetDragIcon(
                        pixmap, mask,
                        (w / 2), (h / 2)
                    );
                }
            }
            status = TRUE;
            break;

          case GDK_LEAVE_NOTIFY:
            gtk_widget_set_state(widget, GTK_STATE_NORMAL);
            EDVStatusBarMessage(
                archiver->status_bar, NULL, FALSE
            );
            status = TRUE;
            break;
        }

        return(status);
}


/*
 *      Sets the drag and drop icon based on the coordinates of the
 *      contents clist's node given by row.
 */
void EDVArchiverContentsDNDSetIcon(
        edv_archiver_struct *archiver, gint row, gint column
)
{
        GtkCList *clist;
        edv_archive_object_struct *obj;


        if(archiver == NULL)
            return;

        if(!archiver->initialized || !archiver->map_state)
            return;

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

        /* Get archive object structure referenced by matched row. */
        obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
            clist, row
        );
        if(obj != NULL)
        {
            gint i, status;
            GdkPixmap *pixmap = NULL;
            GdkBitmap *mask = NULL;


            /* Iterate through each cell of this row, looking for a
             * useable pixmap.
             */
            for(i = 0; i < clist->columns; i++)
            {
                status = (gint)gtk_clist_get_cell_type(
                    clist, row, i
                );
                if(status == GTK_CELL_PIXMAP)
                {
                    gtk_clist_get_pixmap(
                        clist, row, i,
                        &pixmap, &mask
                    );
                    break;
                }
                else if(status == GTK_CELL_PIXTEXT)
                {
                    gchar *text = NULL;
                    guint8 spacing = 0;

                    gtk_clist_get_pixtext(
                        clist, row, i,
                        &text, &spacing, &pixmap, &mask
                    );
                    break;
                }
            }

            /* Is icon pixmap available? If so then set the new DND icon. */
            if(pixmap != NULL)
            {
                gint w = 15, h = 15;

                gdk_window_get_size((GdkWindow *)pixmap, &w, &h);

                GUIDNDSetDragIcon(
                    pixmap, mask,
                    (w / 2), (h / 2)
                );
            }
        }
}

/*
 *	Recycle bin contents clist DND "drag_motion" signal callback.
 *
 *	This is used to constrain all drags (regardless of its type
 *	or source data type) to be a drag action of move.
 */
gboolean EDVArchiverContentsDragMotionCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y, guint t,
        gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((dc == NULL) || (archiver == NULL))
            return(FALSE);

        if(dc->actions & GDK_ACTION_COPY)
            gdk_drag_status(dc, GDK_ACTION_COPY, t);
        else
            gdk_drag_status(dc, 0, t);

	return(TRUE);
}

/*
 *	Archiver contents clist DND "drag_data_get" signal callback.
 */
void EDVArchiverContentsDragDataGetCB(
        GtkWidget *widget, GdkDragContext *dc,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
        gbool data_sent = FALSE;
        gint i;
        edv_core_struct *core_ptr;
        GList *glist;
        GtkWidget *w;
        GtkCList *clist;
        gint row;
        edv_dnd_object_struct **dnd_object, *dnd_obj;
        gint total_dnd_objects;
        edv_archive_object_struct *obj;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((dc == NULL) || (archiver == NULL))
            return;

        if(!archiver->initialized || archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;
        /* Get contents clist and make sure it matches the given widget. */
        w = archiver->contents_clist;
        if((w == NULL) || (w != widget))
            return;

        clist = GTK_CLIST(w);


        /* Sync data on archiver so as to ensure we have the most up to
         * date information to send out.
         */
        EDVArchiverSyncData(archiver);


	/* Add first DND archive object reference as the archive itself. */
	total_dnd_objects = 1;
	dnd_object = (edv_dnd_object_struct **)g_malloc(
	    total_dnd_objects * sizeof(edv_dnd_object_struct *)
	);
	if(dnd_object == NULL)
	{
	    total_dnd_objects = 0;
	}
	else
	{
	    /* Allocate a new dnd archive object reference structure and
	     * set its path to reffer to the archive object itself.
	     */
	    dnd_object[0] = dnd_obj = EDVDNDObjectNew();
	    if(dnd_obj != NULL)
	    {
		dnd_obj->full_path = g_strdup(
		    EDVArchiverCurrentLocation(archiver)
		);
	    }
	}

        /* Iterate through selected rows, generating a list of DND archive
         * object reference structures based on each selected row's archive
         * object structure.
         */
        glist = clist->selection;
        while(glist != NULL)
        {
            /* Get pointer to selected row. */
            row = (gint)glist->data;

            /* Get  object structure on selected row. */
            obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if(obj != NULL)
            {
                /* Allocate more pointers for a new dnd archive object
                 * reference structure.
                 */
                i = total_dnd_objects;
                total_dnd_objects = i + 1;
                dnd_object = (edv_dnd_object_struct **)g_realloc(
                    dnd_object,
                    total_dnd_objects * sizeof(edv_dnd_object_struct *)
                );
                if(dnd_object == NULL)
                {
                    total_dnd_objects = 0;
                    break;
                }
                /* Allocate a new dnd recycled object reference structure. */
                dnd_object[i] = dnd_obj = EDVDNDObjectNew();
                if(dnd_obj != NULL)
                {
		    if(obj->name != NULL)
			dnd_obj->full_path = g_strdup(obj->full_path);
                }
            }

            /* Get next selected row. */
            glist = glist->next;
        }

        /* Any DND recycled object reference structures to send out? */
        if(total_dnd_objects > 0)
        {
            guint8 *buf;
            gint buf_len;


            /* Allocate a serial buffer for DND data sending based on the
             * list of DND recycled object reference structures.
             */
            buf = EDVDNDObjectToBuffer(
                (const edv_dnd_object_struct **)dnd_object,
                total_dnd_objects, &buf_len
            );
            if(buf != NULL)
            {
                /* Send out DND data buffer. */
                gtk_selection_data_set(
                    selection_data,
                    GDK_SELECTION_TYPE_STRING,
                    8,          /* 8 bits per character. */
                    buf, buf_len
                );
                data_sent = TRUE;

                /* Deallocate DND data buffer. */
                g_free(buf);
                buf = NULL;
                buf_len = 0;
            }
        }

        /* Deallocate all dnd recycled object reference structures. */
        for(i = 0; i < total_dnd_objects; i++)
            EDVDNDObjectDelete(dnd_object[i]);
        g_free(dnd_object);
        dnd_object = NULL;
        total_dnd_objects = 0;


        /* Failed to send out data? */
        if(!data_sent)
        {
            /* Then send out an error response. */
            const char *strptr = "Error";
            gtk_selection_data_set(
                selection_data,
                GDK_SELECTION_TYPE_STRING,
                8,      /* 8 bits per character. */
                strptr, strlen(strptr)
            );
            data_sent = TRUE;
        }
}

/*
 *	Recycle bin DND "drag_data_received" signal callback.
 */
void EDVArchiverContentsDragDataReceivedCB(
        GtkWidget *widget, GdkDragContext *dc,
        gint x, gint y,
        GtkSelectionData *selection_data, guint info, guint t,
        gpointer data
)
{
	static gbool reenterent = FALSE;
	edv_core_struct *core_ptr;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((dc == NULL) || (archiver == NULL))
            return;

        if(!archiver->initialized || archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Check if received data is not empty. */
        if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;


        /* Check and warn if write protect is enabled. */
/* Do not check this here, drag may be just to open the archive.
 * this will be checked later in EDVArchiverDragDataReceivedNexus().
        if(EDVCheckWriteProtect(core_ptr, TRUE))
	{
	    reenterent = FALSE;
            return;
	}
 */

        /* Sync data on recycle bin so as to ensure we have the most up to
         * date information to apply the new drag data to.
         */
        EDVArchiverSyncData(archiver);

        /* Handle received drag data. */
	EDVArchiverDragDataReceivedNexus(
	    core_ptr, archiver, dc, info, selection_data
	);

	reenterent = FALSE;
}

/*
 *      Recycle bin contents clist DND "drag_data_delete" signal callback.
 */
void EDVArchiverContentsDragDataDeleteCB(
        GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	/* Ignore this, archive objects are not to be deleted when
	 * extracted.
	 */
}
