#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvdate.h"
#include "edvcfg.h"
#include "edvid.h"
#include "edvobj.h"
#include "edvarch.h"
#include "edvarchfio.h"
#include "edvmimetypes.h"
#include "archiver.h"
#include "archivercb.h"
#include "archivercontents.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"



static void EDVArchiverContentsResetColumns(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist
);
static void EDVArchiverContentsClear(GtkCList *clist);

static void EDVArchiverContentsSetCellName(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellSize(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column,
	gbool hide_dir_size, gbool hide_link_size
);
static void EDVArchiverContentsSetCellType(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellPermissions(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column, gbool hide_link_permissions
);
static void EDVArchiverContentsSetCellOwner(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellGroup(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellDateAccess(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellDateModified(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellDateChanged(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellLocation(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellLinkedTo(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetCellDeviceType(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
);
static void EDVArchiverContentsSetRow(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row
);

static gint EDVArchiverContentsAppendObject(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object
);
static void EDVArchiverContentsGetListing(
        edv_archiver_struct *archiver, gbool update_status_bar
);

gint EDVArchiverContentsFindRowByPath(
        edv_archiver_struct *archiver, const gchar *path
);

void EDVArchiverContentsResetRows(edv_archiver_struct *archiver);
void EDVArchiverContentsDoUpdate(
        edv_archiver_struct *archiver, gbool update_status_bar
);

void EDVArchiverContentsObjectModifiedNotify(
        edv_archiver_struct *archiver, const gchar *path,
        const gchar *new_path, const struct stat *lstat_buf
);
void EDVArchiverContentsObjectRemovedNotify(
	edv_archiver_struct *archiver, const gchar *path
);

void EDVArchiverContentsArchiveObjectAddedNotify(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, edv_archive_object_struct *obj
);
void EDVArchiverContentsArchiveObjectModifiedNotify(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, const gchar *new_path,
        edv_archive_object_struct *obj
);
void EDVArchiverContentsArchiveObjectRemovedNotify(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Clears the titles of each column on the given clist.
 */
static void EDVArchiverContentsResetColumns(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist
)
{
        gint i, m, column_type, width;
        const gchar *title;
        edv_intlist_struct *column_types_intlist, *column_width_intlist;
        gint justify;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL))
            return;

        /* Get column types mapping. */
        column_types_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
        );
        if(column_types_intlist == NULL)
            return;

        column_width_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH
        );
        if(column_width_intlist == NULL)
            return;


        /* Update clist column settings. */
        gtk_clist_column_titles_active(clist);
        gtk_clist_column_titles_show(clist);
        gtk_clist_set_auto_sort(clist, FALSE);
        gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING);

        /* Change clist selection mode to GTK_SELECTION_EXTENDED.
         * The selection mode can change whenever the contents list is
         * updated.
         */
/* Already set. */
/*        gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED); */

        /* Iterate through each column but not exceeding the number of
         * columns specified by the column types or column widths
         * intlists.
         */
        m = MIN(
            MIN(clist->columns, column_types_intlist->total),
            column_width_intlist->total
        );
        for(i = 0; i < m; i++)
        {
            /* Get column's type which determines what will be listed on
             * the current column i.
             */
            column_type = column_types_intlist->i[i];

            /* Get width of the column displaying this type. */
            if((column_type >= 0) && (column_type < column_width_intlist->total))
                width = column_width_intlist->i[column_type];
            else
                width = 0;

            /* Title based on column type. */
            switch(column_type)
            {
              case EDV_ARCHIVER_COLUMN_TYPE_NAME:
                title = "Name";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_SIZE:
                title = "Size";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_TYPE:
                title = "Type";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_PERMISSIONS:
                title = "Permissions";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_OWNER:
                title = "Owner";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_GROUP:
                title = "Group";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DATE_ACCESS:
                title = "Date Access";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DATE_MODIFIED:
                title = "Date Modified";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DATE_CHANGED:
                title = "Date Changed";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_LOCATION:
                title = "Location";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_LINKED_TO:
                title = "Linked To";
                justify = GTK_JUSTIFY_LEFT;
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DEVICE_TYPE:
                title = "Device Type";
                justify = GTK_JUSTIFY_RIGHT;
                break;

              default:
                title = "";
                justify = GTK_JUSTIFY_LEFT;
                break;
            }

            gtk_clist_set_column_auto_resize(
                clist, i, FALSE
            );
            gtk_clist_set_column_title(clist, i, title);
            gtk_clist_set_column_width(clist, i, width);
            gtk_clist_set_column_justification(clist, i, justify);
        }
        /* Reset the rest of the columns incase the number of columns in
         * the intlists were less than the columns specified by the clist.
         */
        for(; i < clist->columns; i++)
        {
            gtk_clist_set_column_auto_resize(
                clist, i, FALSE
            );
            gtk_clist_set_column_title(clist, i, "");
            gtk_clist_set_column_width(clist, i, 0);
        }
}

/*
 *      Deletes all items in the given clist.
 */
static void EDVArchiverContentsClear(GtkCList *clist)
{
        if(clist == NULL)
            return;

        gtk_clist_clear(clist);
}


/*
 *	Sets the archive object name for the specified cell on the given
 *	clist.
 */
static void EDVArchiverContentsSetCellName(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
	const gchar *name;
        GdkPixmap *pixmap;
        GdkBitmap *mask;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	name = object->name;
	if((name != NULL) ? (*name == '\0') : TRUE)
	    name = object->full_path;

        /* Get pixmap and mask for the given object. */
        EDVMatchObjectIcon(
            NULL, 0,
            core_ptr->mimetype, core_ptr->total_mimetypes,
            object->type,
            name,
	    TRUE, object->permissions,
            0,                  /* Small icons. */
            &pixmap, &mask,
            NULL, NULL, NULL, NULL
        );

        if(pixmap != NULL)
            gtk_clist_set_pixtext(
                clist, row, column,
                (name != NULL) ? name : "(null)",
                EDV_LIST_PIXMAP_TEXT_SPACING,
                pixmap, mask
            );
        else
            gtk_clist_set_text(
                clist, row, column,
                (name != NULL) ? name : "(null)"
            );
}

/*
 *      Sets the archive object size for the specified cell on the given
 *      clist.
 */
static void EDVArchiverContentsSetCellSize(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column,
	gbool hide_dir_size, gbool hide_link_size
)
{
	const gchar *s;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        if(hide_dir_size && (object->type == EDV_OBJECT_TYPE_DIRECTORY))
            s = "";
        else if(hide_link_size && (object->type == EDV_OBJECT_TYPE_LINK))
            s = "";
	else
	    s = EDVGetObjectSizeStr(core_ptr, object->size);
        gtk_clist_set_text(
            clist, row, column, s
        );
}

/*
 *      Sets the archive object type for the specified cell on the given
 *      clist.
 */
static void EDVArchiverContentsSetCellType(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
	gchar *type_str;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get MIME Type type string for the given object. */
        EDVMatchObjectTypeString(
            core_ptr->mimetype, core_ptr->total_mimetypes,
            object->type, object->permissions,
            object->name,
            &type_str
        );

        gtk_clist_set_text(
            clist, row, column, (type_str != NULL) ? type_str : ""
        );
}

/*
 *      Sets the archive object permissions for the specified cell on the given
 *      clist.
 */
static void EDVArchiverContentsSetCellPermissions(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column, gbool hide_link_permissions
)
{
        gchar s[80];


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	if(hide_link_permissions && (object->type == EDV_OBJECT_TYPE_LINK))
	    *s = '\0';
	else
	    sprintf(
		s,
		"%c%c%c%c%c%c%c%c%c",
            (object->permissions & EDV_PERMISSION_UREAD) ? 'r' : '-',
            (object->permissions & EDV_PERMISSION_UWRITE) ? 'w' : '-',
            (object->permissions & EDV_PERMISSION_SETUID) ?
                'S' :
                ((object->permissions & EDV_PERMISSION_UEXECUTE) ? 'x' : '-'),
            (object->permissions & EDV_PERMISSION_GREAD) ? 'r' : '-',
            (object->permissions & EDV_PERMISSION_GWRITE) ? 'w' : '-',
            (object->permissions & EDV_PERMISSION_SETGID) ?
                'G' :
                ((object->permissions & EDV_PERMISSION_GEXECUTE) ? 'x' : '-'),
            (object->permissions & EDV_PERMISSION_AREAD) ? 'r' : '-',
            (object->permissions & EDV_PERMISSION_AWRITE) ? 'w' : '-',
            (object->permissions & EDV_PERMISSION_STICKY) ?
                'T' :
                ((object->permissions & EDV_PERMISSION_AEXECUTE) ? 'x' : '-')
	    );

        gtk_clist_set_text(
            clist, row, column, s
        );
}

/*
 *      Sets the archive object owner for the specified cell on the given
 *      clist.
 */
static void EDVArchiverContentsSetCellOwner(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        const gchar *owner_name;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;


        owner_name = object->owner_name;
        if(owner_name == NULL)
            owner_name = "";

        gtk_clist_set_text(
            clist, row, column, owner_name
        );
}

/*
 *      Sets the archive object owner for the specified cell on the given
 *      clist.
 */
static void EDVArchiverContentsSetCellGroup(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        const gchar *group_name;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;


        group_name = object->group_name;
        if(group_name == NULL)
            group_name = "";

        gtk_clist_set_text(
            clist, row, column, group_name
        );
}

/*
 *	Sets the archive object last access date for the specified cell
 *	on the given clist.
 */
static void EDVArchiverContentsSetCellDateAccess(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->access_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the archive object last modified date for the specified cell
 *      on the given clist.
 */
static void EDVArchiverContentsSetCellDateModified(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->modify_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *      Sets the archive object last changed date for the specified cell
 *      on the given clist.
 */
static void EDVArchiverContentsSetCellDateChanged(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        gint relativity;
        const gchar *format, *date_str;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get date relativity and format. */
        relativity = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_RELATIVITY
        );
        format = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_DATE_FORMAT
        );

        /* Get date string. */
        date_str = EDVDateFormatString(
            object->change_time, format, relativity
        );

        gtk_clist_set_text(
            clist, row, column, date_str
        );
}

/*
 *	Sets the archive object location relative to the archive for the
 *	specified cell on the given clist.
 */
static void EDVArchiverContentsSetCellLocation(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
	gchar *tmp_full_path;
        const gchar *full_path, *location;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

	full_path = object->full_path;
	if(full_path == NULL)
	    return;

	/* Create a copy of the full path. Prefix a ./ if it is not an
	 * absolute path.
	 */
	if(ISPATHABSOLUTE(full_path))
	    tmp_full_path = g_strdup(full_path);
	else
	    tmp_full_path = g_strdup_printf(".%c%s",
		DIR_DELIMINATOR, full_path
	    );

        /* Get location portion of our tempory modified full path. */
	location = GetParentDir(tmp_full_path);

        gtk_clist_set_text(
            clist, row, column, location
        );

	/* Deallocate tempory full path. */
	g_free(tmp_full_path);
}

/*
 *      Sets the archive object linked to path for the specified cell
 *      on the given clist.
 */
static void EDVArchiverContentsSetCellLinkedTo(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        const gchar *link_dest;


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
           (object == NULL)
        )
            return;

        /* Get link destination. */
        link_dest = (object->linked_to != NULL) ? object->linked_to : "";

        gtk_clist_set_text(
            clist, row, column, link_dest
        );
}

/*
 *      Sets the object device type for the specified cell on the given
 *      clist.
 */
static void EDVArchiverContentsSetCellDeviceType(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row, gint column
)
{
        gint major, minor;
        gchar num_str[256];


        if((core_ptr == NULL) || (archiver == NULL) || (clist == NULL) ||
            (object == NULL)
        )
            return;

        if((object->type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
           (object->type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
        )
        {
            EDVGetDeviceNumbers(
                (dev_t)object->device_type, &major, &minor
            );
            sprintf(num_str, "%i, %i", major, minor);
        }
        else
        {
            *num_str = '\0';
        }

        gtk_clist_set_text(
            clist, row, column, num_str
        );
}


/*
 *      Sets all cells of the specified row of the given contents clist
 *      with the values of the given object.
 *
 *      The clist row's client data will not be updated.
 */
static void EDVArchiverContentsSetRow(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object,
        gint row
)
{
        gint i, column_type, columns;
        gbool hide_dir_size, hide_link_size, hide_link_permissions;
        edv_intlist_struct *column_types_intlist;


        if((core_ptr == NULL) || (archiver == NULL) ||
           (clist == NULL) || (object == NULL)
        )
            return;


        /* Get column types mapping. */
        column_types_intlist = EDVCFGItemListGetValueIntList(
            core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
        );
        if(column_types_intlist == NULL)
            return;

        /* Get additional display options. */
        hide_dir_size = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_ARCHIVER_CONTENTS_HIDE_DIR_SIZE
        );
        hide_link_size = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_ARCHIVER_CONTENTS_HIDE_LINK_SIZE
        );
        hide_link_permissions = EDVCFGItemListGetValueI(
            core_ptr->cfg_list,
            EDV_CFG_PARM_ARCHIVER_CONTENTS_HIDE_LINK_PERMISSIONS
        );


        /* Get total number of columns on the contents clist. */
        columns = MIN(clist->columns, column_types_intlist->total);

        /* Iterate through each column but not more than the columns
         * specified by the column types intlist.
         */
        for(i = 0; i < columns; i++)
        {
            /* Check column references on configuration list, see which
             * one this column should show.
             */
            column_type = column_types_intlist->i[i];
            switch(column_type)
            {
              case EDV_ARCHIVER_COLUMN_TYPE_NAME:
                EDVArchiverContentsSetCellName(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_SIZE:
                EDVArchiverContentsSetCellSize(
                    core_ptr, archiver, clist, object,
                    row, i, hide_dir_size, hide_link_size
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_TYPE:
                EDVArchiverContentsSetCellType(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_PERMISSIONS:
                EDVArchiverContentsSetCellPermissions(
                    core_ptr, archiver, clist, object,
                    row, i, hide_link_permissions
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_OWNER:
                EDVArchiverContentsSetCellOwner(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_GROUP:
                EDVArchiverContentsSetCellGroup(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DATE_ACCESS:
                EDVArchiverContentsSetCellDateAccess(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DATE_MODIFIED:
                EDVArchiverContentsSetCellDateModified(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DATE_CHANGED:
                EDVArchiverContentsSetCellDateChanged(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_LOCATION:
                EDVArchiverContentsSetCellLocation(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_LINKED_TO:
                EDVArchiverContentsSetCellLinkedTo(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;

              case EDV_ARCHIVER_COLUMN_TYPE_DEVICE_TYPE:
                EDVArchiverContentsSetCellDeviceType(
                    core_ptr, archiver, clist, object,
                    row, i
                );
                break;
	    }
	}
}

/*
 *	Appends the archive object to the contents clist. The given
 *	archive object structure will be transfered to the new row and
 *	should not be referenced again after this call.
 *
 *      Returns the row number that it was appended to or -1 on failure.
 */
static gint EDVArchiverContentsAppendObject(
        edv_core_struct *core_ptr, edv_archiver_struct *archiver,
        GtkCList *clist, edv_archive_object_struct *object
)
{
        gint row, columns;
        gchar **row_text;


        if((core_ptr == NULL) || (archiver == NULL) ||
           (clist == NULL) || (object == NULL)
        )
        {
            EDVArchObjectDelete(object);
            return(-1);
        }

        /* Get total number of columns on the contents clist. */
        columns = MAX(clist->columns, 1);


        /* Allocate row text. */
        row_text = (gchar **)g_malloc0(columns * sizeof(gchar *));
        if(row_text != NULL)
        {
            gint i;

            /* Allocate blank strings for each cell. */
            for(i = 0; i < columns; i++)
                row_text[i] = g_strdup("");

            /* Append a new row. */
            row = gtk_clist_append(clist, row_text);

            /* Deallocate row text. */
            for(i = 0; i < columns; i++)
                g_free(row_text[i]);
            g_free(row_text);
            row_text = NULL;
        }
        else
        {
            row = -1;
        }
        /* Successfully appended row? */
        if(row < 0)
        {
            EDVArchObjectDelete(object);
            return(-1);
        }

        /* Update each cell of the new row with the values from the
         * disk object structure.
         */
        EDVArchiverContentsSetRow(
            core_ptr, archiver, clist, object, row
        );

	/* Set given archive object structure as the row data for this
	 * new row. The archive object structure should not be referenced
	 * again after this point.
         */
        gtk_clist_set_row_data_full(
            clist, row,
            object, (GtkDestroyNotify)EDVArchiverContentsItemDestroyCB
        );
        /* Archive object pointer has now been transfered to the new row's
         * data. Mark it NULL so we do not reference it again.
         */
        object = NULL;

        return(row);
}

/*
 *	Gets all archive objects in the currently opened archive and appends
 *	them to the contents clist.
 */
static void EDVArchiverContentsGetListing(
        edv_archiver_struct *archiver, gbool update_status_bar
)
{
        gint i, new_row, last_progress_percent, progress_percent;
	gint archive_objects_loaded, total_archive_objects;
        GtkCList *clist;
        edv_core_struct *core_ptr;
	edv_archive_object_struct **archive_object, *obj;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;


	/* Get listing of archive objects. */
	archive_object = EDVArchFIOGetListing(
	    core_ptr,
	    EDVArchiverCurrentLocation(archiver), &total_archive_objects,
	    NULL, 0		/* Get listing of all archive objects. */
	);

	last_progress_percent = -1;     /* None/undefined. */
	progress_percent = 0;
	archive_objects_loaded = 0;

	/* Iterate through archive objects. */
	for(i = 0; i < total_archive_objects; i++)
	{
	    obj = archive_object[i];
	    if(obj == NULL)
		continue;

	    /* Append row, transfering the archive object structure obj
	     * to the new row. The archive object structure should not
	     * be referenced after this point.
	     */
	    new_row = EDVArchiverContentsAppendObject(
		core_ptr, archiver, clist, obj
	    );
	    archive_object[i] = obj = NULL;

	    /* Increment total number of archive objects loaded. */
	    archive_objects_loaded++;


	    /* Update progress? */
	    if(update_status_bar && (total_archive_objects > 0))
	    {
		progress_percent = archive_objects_loaded * 100 /
		    total_archive_objects;
		if(progress_percent > last_progress_percent)
		{
		    EDVStatusBarProgress(
			archiver->status_bar,
			(gfloat)archive_objects_loaded /
			    (gfloat)total_archive_objects,
			TRUE
		    );
		    progress_percent = last_progress_percent;
		}
	    }
	}

	/* Deallocate archive objects pointer array but not each object
	 * since they have already been transfered or deallocated in
	 * the above loop.
	 */
	g_free(archive_object);
	archive_object = NULL;
	total_archive_objects = 0;
}

/*
 *      Returns the row index of the contents clist that contains an
 *	archive object structure who's full_path matches the given path.
 *
 *      Can return -1 on failed match.
 */
gint EDVArchiverContentsFindRowByPath(
        edv_archiver_struct *archiver, const gchar *path
)
{
        gint i;
        GtkCList *clist;
        edv_archive_object_struct *obj;


        if((archiver == NULL) || (path == NULL))
            return(-1);

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return(-1);

        /* Iterate through all rows. */
        for(i = 0; i < clist->rows; i++)
        {
            obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
                clist, i
            );
            if((obj != NULL) ? (obj->full_path == NULL) : TRUE)
                continue;

	    if(!strcmp(obj->full_path, path))
                return(i);
        }

        return(-1);
}

/*
 *      Updates all rows on the contents clist by getting the row data
 *      and updates the clist's rows.
 *
 *      This is designed to be called whenever the displayed values
 *      of each row need to be set again from internal data, usually
 *      when a MIME Type has been added/modified/removed. This function
 *      should not be used to `refresh' the list (get new values of
 *      disk object structures), use EDVBrowserContentsDoUpdate()
 *      instead.
 */
void EDVArchiverContentsResetRows(edv_archiver_struct *archiver)
{
        gint i;
        GtkWidget *w;
        GtkCList *clist;
        edv_core_struct *core_ptr;
        edv_archive_object_struct *object;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

        for(i = 0; i < clist->rows; i++)
        {
            object = (edv_archive_object_struct *)gtk_clist_get_row_data(
                clist, i
            );
            if(object != NULL)
                EDVArchiverContentsSetRow(
                    core_ptr, archiver, clist, object, i
                );
        }
}

/*
 *	Procedure to clear and re update the contents list with the
 *	archive objects found in the current archive.
 */
void EDVArchiverContentsDoUpdate(
        edv_archiver_struct *archiver, gbool update_status_bar
)
{
        GtkWidget *w;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);


        if(update_status_bar)
        {
            /* Reset progress. */
            EDVStatusBarMessage(
                archiver->status_bar,
                "Loading archive objects...",
                FALSE
            );
            EDVStatusBarProgress(
                archiver->status_bar, 0.0, TRUE
            );
        }


        /* Delete all entries. */
        EDVArchiverContentsClear(clist);


        /* Update columns. */
        EDVArchiverContentsResetColumns(
            core_ptr, archiver, clist
        );

        /* Get listing of archive objects. */
        EDVArchiverContentsGetListing(archiver, update_status_bar);


        if(update_status_bar)
        {
            /* Reset progress. */
            EDVStatusBarMessage(
                archiver->status_bar,
                "Archive objects loaded",
                FALSE
            );
            EDVStatusBarProgress(
                archiver->status_bar, 0.0, TRUE
            );
        }
}

/*
 *      Called whenever a disk object has been modified.
 */
void EDVArchiverContentsObjectModifiedNotify(
        edv_archiver_struct *archiver, const gchar *path,
        const gchar *new_path, const struct stat *lstat_buf
)
{
        const gchar *cstrptr;
        gchar *cur_path = NULL, *parent_path = NULL;
	GtkCList *clist;
        edv_core_struct *core_ptr;


        if((archiver == NULL) || (path == NULL) || (lstat_buf == NULL))
            return;

        if(new_path == NULL)
            new_path = path;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}

        /* Get parent of the given path and current location. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(new_path);
        parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (parent_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }

        /* Is given path that has been modified the same as the current
         * location?
         */
        if(!strcmp(new_path, cur_path))
        {
            /* Reget listing. */
	    gtk_clist_freeze(clist);
	    EDVArchiverContentsDoUpdate(archiver, TRUE);
	    gtk_clist_thaw(clist);
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *	Called whenever a disk object has been removed.
 */
void EDVArchiverContentsObjectRemovedNotify(
        edv_archiver_struct *archiver, const gchar *path
)
{
        GtkWidget *w;
        GtkCList *clist;
        edv_core_struct *core_ptr;
        const gchar *cstrptr;
        gchar *cur_path, *parent_path;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

#define DO_FREE_LOCALS  \
{ \
 g_free(cur_path); \
 cur_path = NULL; \
 g_free(parent_path); \
 parent_path = NULL; \
}
        /* Get parent of the given path and current location. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        cstrptr = GetParentDir(path);
        parent_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        if((cur_path == NULL) || (parent_path == NULL))
        {
            DO_FREE_LOCALS
            return;
        }

        /* Is given path that has been removed the same as the current
         * location?
         */
        if(!strcmp(path, cur_path))
        {
            /* Clear contents list. */
            gtk_clist_freeze(clist);
            EDVArchiverContentsClear(clist);
            gtk_clist_thaw(clist);
        }

        DO_FREE_LOCALS
#undef DO_FREE_LOCALS
}

/*
 *      Called whenever an archive object has been add to the
 *      specified archive arch_path which is gauranteed to the
 *	archive currently opened by the given archiver.
 */
void EDVArchiverContentsArchiveObjectAddedNotify(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, edv_archive_object_struct *obj
)
{
        GtkWidget *w;
	gint row;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);


        /* Check if the path of the added archive object is one that this
	 * archiver is already has in its contents list.
         */
	row = EDVArchiverContentsFindRowByPath(archiver, path);
	if((row >= 0) && (row < clist->rows))
	{
	    /* New archive object is already listed, so just update the row.
	     * First delete the old archive object structure and then copy
	     * the given archive object structure to it.
	     */
	    edv_archive_object_struct *old_obj;


	    /* Get old archive object from row data and delete it. */
	    old_obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
		clist, row
	    );
	    EDVArchObjectDelete(old_obj);
	    old_obj = NULL;

	    /* Copy given archive object and set it as the new archive
	     * object structure.
	     */
	    obj = EDVArchObjectCopy(obj);
	    gtk_clist_set_row_data_full(
		clist, row,
		obj, (GtkDestroyNotify)EDVArchiverContentsItemDestroyCB
	    );

	    /* Update row values. */
	    EDVArchiverContentsSetRow(
		core_ptr, archiver, clist, obj, row
	    );
        }
	else
	{
	    /* Copy given archive object structure. */
	    obj = EDVArchObjectCopy(obj);

	    /* Append row using coppied archive object structure. The
	     * archive object structure will be set as row data so it
	     * should not be referenced again.
	     */
	    row = EDVArchiverContentsAppendObject(
		core_ptr, archiver, clist, obj
	    );
	    obj = NULL;
	}
}

/*
 *      Called whenever an archive object has been modified in the
 *      specified archive arch_path which is gauranteed to the
 *      archive currently opened by the given archiver.
 */
void EDVArchiverContentsArchiveObjectModifiedNotify(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, const gchar *new_path,
        edv_archive_object_struct *obj
)
{
        GtkWidget *w;
	gint row;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);

        if(new_path == NULL)
            new_path = path;


        /* Check if the old path of the modified archive object is in
	 * this archiver's contents list.
         */
        row = EDVArchiverContentsFindRowByPath(archiver, path);
        if((row >= 0) && (row < clist->rows))
        {
            /* New archive object is already listed, so just update the row.
             * First delete the old archive object structure and then copy
             * the given archive object structure to it.
             */
            edv_archive_object_struct *old_obj;


            /* Get old archive object from row data and delete it. */
            old_obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            EDVArchObjectDelete(old_obj);
            old_obj = NULL;

            /* Copy given archive object and set it as the new archive
             * object structure.
             */
            obj = EDVArchObjectCopy(obj);
            gtk_clist_set_row_data_full(
                clist, row,
                obj, (GtkDestroyNotify)EDVArchiverContentsItemDestroyCB
            );

            /* Update row values. */
            EDVArchiverContentsSetRow(
                core_ptr, archiver, clist, obj, row
            );
        }
}

/*
 *      Called whenever an archive object has been removed from the
 *      specified archive arch_path which is gauranteed to the
 *      archive currently opened by the given archiver.
 */
void EDVArchiverContentsArchiveObjectRemovedNotify(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path
)
{
        GtkWidget *w;
	gint row;
        GtkCList *clist;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        w = archiver->contents_clist;
        if(w == NULL)
            return;
        else
            clist = GTK_CLIST(w);


	/* Check if the path of the removed archive object is in this
	 * archiver's contents list.
         */
        row = EDVArchiverContentsFindRowByPath(archiver, path);
        if((row >= 0) && (row < clist->rows))
        {
	    /* Remove matched row. */
	    gtk_clist_remove(clist, row);
        }
}
