#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "edvarch.h"
#include "edvstatusbar.h"
#include "archiver.h"
#include "archivercb.h"
#include "archiveropcb.h"
#include "archiverdnd.h"
#include "archivercontents.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvlistseek.h"
#include "edvcfglist.h"
#include "config.h"



void EDVArchiverContentsItemDestroyCB(gpointer data);

static gint EDVArchiverCListColumnSortDateNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
        gint sort_code
);
static gint EDVArchiverCListColumnSortDateAccessCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVArchiverCListColumnSortDateModifyCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVArchiverCListColumnSortDateChangeCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

gint EDVArchiverDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
void EDVArchiverDestroyCB(GtkObject *object, gpointer data);

gint EDVArchiverMapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
);
gint EDVArchiverUnmapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
);

gint EDVArchiverKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVArchiverButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVArchiverHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVArchiverHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVArchiverResizeColumnCB(
        GtkCList *clist, gint column, gint width, gpointer data
);
void EDVArchiverClickColumnCB(
        GtkCList *clist, gint column, gpointer data
);
void EDVArchiverSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);
void EDVArchiverUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);

void EDVArchiverComboActivateCB(GtkWidget *widget, gpointer data);

void EDVArchiverWriteProtectChangedCB(
        edv_archiver_struct *archiver, gbool state
);

void EDVArchiverObjectModifiedNotifyCB(
        edv_archiver_struct *archiver, const gchar *path,
        const gchar *new_path, const struct stat *lstat_buf
);
void EDVArchiverObjectRemovedNotifyCB(
        edv_archiver_struct *archiver, const gchar *path
);

void EDVArchiverArchiveObjectAddedNotifyCB(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, edv_archive_object_struct *obj
);
void EDVArchiverArchiveObjectModifiedNotifyCB(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, const gchar *new_path,
        edv_archive_object_struct *obj
);
void EDVArchiverArchiveObjectRemovedNotifyCB(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path
);

void EDVArchiverRecycledObjectAddedNotifyCB(
        edv_archiver_struct *archiver, guint index
);
void EDVArchiverRecycledObjectRemovedNotifyCB(
        edv_archiver_struct *archiver, guint index
);

void EDVArchiverReconfiguredNotifyCB(edv_archiver_struct *archiver);

void EDVArchiverMimeTypeAddedCB(
        edv_archiver_struct *archiver,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVArchiverMimeTypeModifiedCB(
        edv_archiver_struct *archiver,
        gint mt_num, edv_mimetype_struct *mt_ptr
);
void EDVArchiverMimeTypeRemovedCB(
        edv_archiver_struct *archiver, gint mt_num
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Recycle bin contents GtkCList item "destroy" signal callback.
 */
void EDVArchiverContentsItemDestroyCB(gpointer data)
{
	EDVArchObjectDelete((edv_archive_object_struct *)data);
}


/*
 *      Returns the sort code for the rows sorted by date.
 *
 *      The recycled object structures are obtained from each row and the
 *      dates are compared.
 */
static gint EDVArchiverCListColumnSortDateNexus(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
        gint sort_code
)
{
        gint sort_column;
        const GtkCListRow *row1 = (const GtkCListRow *)ptr1;
        const GtkCListRow *row2 = (const GtkCListRow *)ptr2;
        edv_archive_object_struct *obj1, *obj2;


        if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
            return(-1);

        sort_column = clist->sort_column;
        if((sort_column < 0) || (sort_column >= clist->columns))
            return(-1);

        obj1 = (edv_archive_object_struct *)row1->data;
        obj2 = (edv_archive_object_struct *)row2->data;
        if((obj1 == NULL) || (obj2 == NULL))
            return(-1);

        switch(sort_code)
        {
          case 0:       /* Access time. */
            if(obj1->access_time <= obj2->access_time)
                    return((gint)(obj1->access_time < obj2->access_time));
                else
                    return(-1);
            break;

          case 1:       /* Modify time. */
            if(obj1->modify_time <= obj2->modify_time)
                    return((gint)(obj1->modify_time < obj2->modify_time));
                else
                    return(-1);
            break;

          case 2:       /* Change time. */
            if(obj1->change_time <= obj2->change_time)
                    return((gint)(obj1->change_time < obj2->change_time));
                else
                    return(-1);
            break;
        }

        return(-1);
}

static gint EDVArchiverCListColumnSortDateAccessCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVArchiverCListColumnSortDateNexus(
            clist, ptr1, ptr2, 0
        ));
}

static gint EDVArchiverCListColumnSortDateModifyCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVArchiverCListColumnSortDateNexus(
            clist, ptr1, ptr2, 1
        ));
}

static gint EDVArchiverCListColumnSortDateChangeCB(
        GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
        return(EDVArchiverCListColumnSortDateNexus(
            clist, ptr1, ptr2, 2
        ));
}


/*
 *	Recycle bin toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVArchiverDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
            return(TRUE);

        if(archiver->processing)
            return(TRUE);

        EDVArchiverOPClose(archiver);

        return(TRUE);
}

/*
 *	Recycle bin toplevel GtkWidget "destroy" signal callback.
 */
void EDVArchiverDestroyCB(GtkObject *object, gpointer data)
{
        return;
}


/*
 *	Recycle bin toplevel "map" signal callback.
 */
gint EDVArchiverMapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
	    return(FALSE);

	return(FALSE);
}

/*
 *      Recycle bin toplevel "unmap" signal callback.
 */
gint EDVArchiverUnmapCB(
	GtkWidget *widget, GdkEventAny *event, gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if(archiver == NULL)
            return(FALSE);

	return(FALSE);
}


/*
 *	Recycle bin "key_press_event" or "key_release_event" signal callback.
 */
gint EDVArchiverKeyEventCB(
         GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
        static gbool reenterent = FALSE;
        gint status = FALSE;
        gint etype;
        guint keyval, state;
        gbool press;
        edv_core_struct *core_ptr;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((widget == NULL) || (key == NULL) || (archiver == NULL))
            return(status);

        if(archiver->processing)
            return(status);

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

        /* Get event type. */
        etype = key->type;

        /* Get other key event values. */
        press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
        keyval = key->keyval;
        state = key->state;

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type.
 */
#define DO_STOP_KEY_SIGNAL_EMIT \
{ \
 gtk_signal_emit_stop_by_name( \
  GTK_OBJECT(widget), \
  press ? "key_press_event" : "key_release_event" \
 ); \
}

/* Macro to clamp the GtkAdjustment adj and emit a "value_changed"
 * signal.
 */
#define DO_ADJ_CLAMP_EMIT       \
{ \
 if(adj->value > (adj->upper - adj->page_size)) \
  adj->value = adj->upper - adj->page_size; \
\
 if(adj->value < adj->lower) \
  adj->value = adj->lower; \
\
 gtk_signal_emit_by_name( \
  GTK_OBJECT(adj), "value_changed" \
 ); \
}

/* Macro to clamp the GtkCList clist's focus_row. */
#define DO_CLIST_FOCUS_ROW_CLAMP	\
{ \
 if(clist->focus_row >= clist->rows) \
  clist->focus_row = clist->rows - 1; \
 if(clist->focus_row < 0) \
  clist->focus_row = 0; \
}


        /* Check which ctree this signal is for. */

	/* Contents clist. */
	if(widget == archiver->contents_clist)
        {
            gint row;
	    GList *glist;
            GtkCList *clist = GTK_CLIST(widget);


            /* Get last selected row. */
	    row = EDVCListGetSelectedLast(clist, NULL);

            /* Handle by key value. */
            switch(keyval)
            {
              case GDK_Return:
              case GDK_KP_Enter:
              case GDK_ISO_Enter:
              case GDK_3270_Enter:
                /* Handle only if control key modifier is not held, so
                 * that accelerator keys will get handled properly.
                 */
                if(!(state & GDK_CONTROL_MASK))
                {
		    if(press)
			EDVArchiverOPExtract(archiver);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

	      case GDK_plus:
	      case GDK_KP_Add:
                if(1)
                {
                    if(press)
                        EDVArchiverOPAdd(archiver);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

	      case GDK_Delete:
                if(press)
                {
		    EDVArchiverOPDelete(archiver);
                }
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
                break;

              case GDK_space:
              case GDK_KP_Space:
                row = clist->focus_row;
                if((row >= 0) && (row < clist->rows) && press)
                {
                    gbool already_selected = FALSE;

                    /* Check if this row is already selected. */
                    glist = clist->selection;
                    while(glist != NULL)
                    {
                        if(row == (gint)glist->data)
                        {
                            already_selected = TRUE;
                            break;
                        }
                        glist = glist->next;
                    }

                    gtk_clist_freeze(clist);
                    if(already_selected)
                        gtk_clist_unselect_row(clist, row, 0);
                    else
                        gtk_clist_select_row(clist, row, 0);
                    gtk_clist_thaw(clist);
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Up:
              case GDK_KP_Up:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    if(press)
                    {
                        gint prev_focus_row = clist->focus_row;

                        gtk_clist_freeze(clist);
                        clist->focus_row--;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,   /* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );

                        if(state & GDK_SHIFT_MASK)
                        {
                            gtk_clist_select_row(
                                clist, prev_focus_row, 0
                            );
                            gtk_clist_select_row(
                                clist, clist->focus_row, 0
                            );
                        }
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Down:
              case GDK_KP_Down:
                if(state & GDK_CONTROL_MASK)
                {
                    /* Get adjustment and scroll down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                else
                {
                    if(press)
                    {
                        gint prev_focus_row = clist->focus_row;

                        gtk_clist_freeze(clist);
                        clist->focus_row++;
                        DO_CLIST_FOCUS_ROW_CLAMP

                        if(gtk_clist_row_is_visible(
                            clist, clist->focus_row) != GTK_VISIBILITY_FULL
                        )
                            gtk_clist_moveto(
                                clist,
                                clist->focus_row, -1,   /* Row, column. */
                                0.5, 0.0                /* Row, column. */
                            );

                        if(state & GDK_SHIFT_MASK)
                        {
                            gtk_clist_select_row(
                                clist, prev_focus_row, 0
                            );
                            gtk_clist_select_row(
                                clist, clist->focus_row, 0
                            );
                        }
                        gtk_clist_thaw(clist);
                    }
                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

              case GDK_Left:
              case GDK_KP_Left:
                if(1)
                {
                    /* Get adjustment and scroll left. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value -= adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->lower;
                        else
                            adj->value -= adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Right:
              case GDK_KP_Right:
                if(1)
                {
                    /* Get adjustment and scroll right. */
                    GtkAdjustment *adj = clist->hadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        if(state & GDK_CONTROL_MASK)
                            adj->value += adj->step_increment * 4;
                        else if(state & GDK_SHIFT_MASK)
                            adj->value = adj->upper - adj->page_size;
                        else
                            adj->value += adj->step_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Up:
              case GDK_KP_Page_Up:
                if(1)
                {
                    /* Get adjustment and scroll up one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value -= adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Page_Down:
              case GDK_KP_Page_Down:
                if(1)
                {
                    /* Get adjustment and scroll down one page. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value += adj->page_increment;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_Home:
              case GDK_KP_Home:
                if(1)
                {
                    /* Get adjustment and scroll all the way up. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->lower;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              case GDK_End:
              case GDK_KP_End:
                if(1)
                {
                    /* Get adjustment and scroll all the way down. */
                    GtkAdjustment *adj = clist->vadjustment;
                    if((adj != NULL) && press)
                    {
                        gtk_clist_freeze(clist);
                        adj->value = adj->upper - adj->page_size;
                        DO_ADJ_CLAMP_EMIT
                        gtk_clist_thaw(clist);
                    }
                }
                DO_STOP_KEY_SIGNAL_EMIT
                status = TRUE;
                break;

              default:
                /* For all other alphanumeric character keys and while
                 * no modifier keys are held, attempt to seek to the
                 * item who's name starts with the letter of the key
                 * that was pressed.
                 */
                if(isalnum((char)keyval) &&
                   ((state == 0x00000000) || (state == GDK_SHIFT_MASK))
                )
                {
                    /* Get the column that is displaying the object
                     * name as column_type_name.
                     */
                    gint i, column_type_name = -1;
                    edv_intlist_struct *column_type_intlist =
                        EDVCFGItemListGetValueIntList(
                            core_ptr->cfg_list,
                            EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
                        );
                    if(column_type_intlist != NULL)
                    {
                        /* Iterate through column type intlist. */
                        for(i = 0; i < column_type_intlist->total; i++)
                        {
                            /* Is column i displaying the name? */
                            if(column_type_intlist->i[i] == EDV_ARCHIVER_COLUMN_TYPE_NAME)
                            {
                                column_type_name = i;
                                break;
                            }
                        }
                    }

                    gtk_clist_freeze(clist);
                    EDVCListSeekCharacter(
                        clist, column_type_name, 0,
                        (gchar)((state & GDK_SHIFT_MASK) ?
                            toupper((char)keyval) : keyval
                        )
                    );
                    gtk_clist_thaw(clist);

                    DO_STOP_KEY_SIGNAL_EMIT
                    status = TRUE;
                }
                break;

            }
        }

#undef DO_CLIST_FOCUS_ROW_CLAMP
#undef DO_ADJ_CLAMP_EMIT
#undef DO_STOP_KEY_SIGNAL_EMIT

        reenterent = FALSE;
        return(status);
}

/*
 *	Recycle bin GtkWidget "button_press_event" signal callback.
 */
gint EDVArchiverButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
        static gbool reenterent = FALSE;
	gint status = FALSE;
        gint etype;
        edv_core_struct *core_ptr;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((widget == NULL) || (archiver == NULL))
            return(status);

        if(archiver->processing)
            return(status);

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

	/* Get event type. */
        etype = button->type;

	/* Check which widget this signal is for. */
	if(widget == archiver->contents_clist)
        {
            gint row, column;
            gint rows_selected = 0, selected_row = -1;
            GList *glist;
            GtkCList *clist = GTK_CLIST(widget);


            /* Find row and column based on given coordinates. */
            if(!gtk_clist_get_selection_info(
                clist, button->x, button->y, &row, &column
            ))
            {
                row = -1;
                column = 0;
            }

            /* Get number of selected rows and highest selected row. */
            glist = clist->selection;
            while(glist != NULL)
            {
                rows_selected++;
                selected_row = (gint)glist->data;
                glist = glist->next;
            }

            /* Handle by button number. */
            switch(button->button)
            {
              case 3:
                if(etype == GDK_BUTTON_PRESS)
                {
                    GtkMenu *menu;

                    /* Select item before mapping menu? */
                    if(EDVCFGItemListGetValueI(
                        core_ptr->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
                    ) && (row >= 0) && (row < clist->rows))
                    {
                        /* Select the row that the button was pressed over.
                         * if no key modifiers are held then this will also
                         * unselect all previously selected rows.
                         */
                        gtk_clist_freeze(clist);
                        if(!(button->state & GDK_CONTROL_MASK) &&
                           !(button->state & GDK_SHIFT_MASK)
                        )
                            gtk_clist_unselect_all(clist);
                        clist->focus_row = row;
                        gtk_clist_select_row(clist, row, 0);
                        gtk_clist_thaw(clist);
                    }

                    /* Update all menus and map right click menu. */
                    EDVArchiverUpdateMenus(archiver);
                    menu = (GtkMenu *)archiver->contents_clist_menu;
                    if(menu != NULL)
                        gtk_menu_popup(
                            menu, NULL, NULL,
                            NULL, NULL,
                            button->button, button->time
                        );
                }
		status = TRUE;
                break;

              case 2:
                if(etype == GDK_BUTTON_PRESS)
                {
                    if((row >= 0) && (row < clist->rows))
		    {
/* Renaming not allowed. */
		    }
                }
                break;

              case 1:
		/* Double click? */
                if(etype == GDK_2BUTTON_PRESS)
                {
                    if((row >= 0) && (row < clist->rows))
		    {
			/* Do extract archive object. */
			EDVArchiverOPExtract(archiver);
			status = TRUE;
		    }
                }
                break;
            }

            if(etype == GDK_BUTTON_PRESS)
                gtk_widget_grab_focus(widget);
        }

        reenterent = FALSE;
        return(status);
}


/*
 *      GtkHandleBox "child_attached" signal callback.
 */
void EDVArchiverHandleChildAttachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((handle_box == NULL) || (archiver == NULL))
            return;

        gtk_widget_queue_resize(
            gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
        );
}

/*
 *      GtkHandleBox "child_detached" signal callback.
 */
void EDVArchiverHandleChildDetachedCB(
        GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((handle_box == NULL) || (archiver == NULL))
            return;

        gtk_widget_queue_resize(
            gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
        );
}


/*
 *      Recycle bin GtkCList "resize_column" signal callback.
 */
void EDVArchiverResizeColumnCB(
        GtkCList *clist, gint column, gint width, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((clist == NULL) || (archiver == NULL))
            return;

        if(archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == archiver->contents_clist)
        {
            gint column_type = -1;
            edv_intlist_struct *column_types_intlist, *column_width_intlist;


            /* Get column_type from the given column index. */
            column_types_intlist = EDVCFGItemListGetValueIntList(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
            );
            if(column_types_intlist != NULL)
            {
                if((column >= 0) && (column < column_types_intlist->total))
                    column_type = column_types_intlist->i[column];
            }

            /* Get column widths intlist. */
            column_width_intlist = EDVCFGItemListGetValueIntList(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN_WIDTH
            );
            if(column_width_intlist != NULL)
            {
                /* Need to increase column_width_intlist allocation? */
                if((column_type >= column_width_intlist->total) &&
                   (column_type >= 0)
                )
                {
                    gint n;
                    gint prev_total = column_width_intlist->total;

                    /* Increase array allocation. */
                    column_width_intlist->total = column_type + 1;
                    column_width_intlist->i = (gint *)g_realloc(
                        column_width_intlist->i,
                        column_width_intlist->total * sizeof(gint)
                    );
                    /* Reset newly allocated indexes. */
                    for(n = prev_total; n < column_width_intlist->total; n++)
                        column_width_intlist->i[n] = 0;
                }

                /* Got column type in bounds as index on column width
                 * intlist?
                 */
                if((column_type >= 0) &&
                   (column_type < column_width_intlist->total)
                )
                {
                    /* Record new width of column on the column width
                     * intlist.
                     */
                    column_width_intlist->i[column_type] = width;
                }
            }
        }

        reenterent = FALSE;
}

/*
 *      Recycle bin GtkCList "click_column" signal callback.
 */
void EDVArchiverClickColumnCB(
        GtkCList *clist, gint column, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_core_struct *core_ptr;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((clist == NULL) || (archiver == NULL))
            return;

        if(archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == archiver->contents_clist)
        {
            edv_intlist_struct *column_types_intlist;
            GtkCListCompareFunc cmp_func = NULL;
            GtkCListCompareFunc cmp_func_str =
                 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
            GtkCListCompareFunc cmp_func_num =
                (GtkCListCompareFunc)EDVCListColumnSortNumericCB;


            EDVArchiverSetBusy(archiver, TRUE);
	    GUIBlockInput(archiver->toplevel, TRUE);
            archiver->processing = TRUE;


            /* Get column types mapping. */
            column_types_intlist = EDVCFGItemListGetValueIntList(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_CONTENTS_COLUMN
            );
            if(column_types_intlist != NULL)
            {
                if((column >= 0) && (column < column_types_intlist->total))
                {
                    gint column_type = column_types_intlist->i[column];
                    switch(column_type)
                    {
                      case EDV_ARCHIVER_COLUMN_TYPE_NAME:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_SIZE:
                        cmp_func = cmp_func_num;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_TYPE:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_PERMISSIONS:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_OWNER:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_GROUP:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_DATE_ACCESS:
                        cmp_func = EDVArchiverCListColumnSortDateAccessCB;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_DATE_MODIFIED:
                        cmp_func = EDVArchiverCListColumnSortDateModifyCB;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_DATE_CHANGED:
                        cmp_func = EDVArchiverCListColumnSortDateChangeCB;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_LOCATION:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_LINKED_TO:
                        cmp_func = cmp_func_str;
                        break;
                      case EDV_ARCHIVER_COLUMN_TYPE_DEVICE_TYPE:
                        cmp_func = cmp_func_str;
                        break;
		    }
		}
	    }


            gtk_clist_freeze(clist);

            /* Set sort column settings on the clist. */
            if(column != clist->sort_column)
                gtk_clist_set_sort_column(clist, column);
            else
                gtk_clist_set_sort_type(
                    clist,
                    (clist->sort_type == GTK_SORT_ASCENDING) ?
                        GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
                );
            if(cmp_func != NULL)
                gtk_clist_set_compare_func(clist, cmp_func);

            /* Sort rows. */
            gtk_clist_sort(clist);

            gtk_clist_thaw(clist);

            archiver->processing = FALSE;
            GUIBlockInput(archiver->toplevel, FALSE);
            EDVArchiverSetBusy(archiver, FALSE);

            EDVArchiverUpdateMenus(archiver);
        }

        reenterent = FALSE;
}

/*
 *      Recycle bin GtkCList "select_row" signal callback.
 */
void EDVArchiverSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((clist == NULL) || (archiver == NULL))
            return;

        if(archiver->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == archiver->contents_clist)
        {
            gint total_selected = 0;
            GList *glist;
            edv_archive_object_struct *obj;


            /* Get total number of objects selected. */
            glist = clist->selection;
            while(glist != NULL)
            {
                total_selected++;
                glist = glist->next;
            }


            /* Update selected row. */
            archiver->contents_clist_selected_row = row;

            /* Update DND icon for contents clist. */
	    EDVArchiverContentsDNDSetIcon(archiver, row, column);

            /* Get disk object structure from selected row. */
            obj = (edv_archive_object_struct *)gtk_clist_get_row_data(
                clist, row
            );
            if(obj != NULL)
            {
                /* Update status bar message. */
                if(obj->name != NULL)
                {
                    gchar *buf;
                    const gchar *type_str = "Object";
                    gchar size_str[80];


                    /* Get object type string and size string. */
                    *size_str = '\0';
                    switch(obj->type)
                    {
                      case EDV_OBJECT_TYPE_FILE:
                        type_str = "File";
                        sprintf(size_str, " (%s bytes)",
                            EDVGetObjectSizeStr(
                                (edv_core_struct *)archiver->core_ptr,
                                obj->size
                            )
			);
                        break;
                      case EDV_OBJECT_TYPE_DIRECTORY:
                        type_str = "Directory";
                        break;
                      case EDV_OBJECT_TYPE_LINK:
                        type_str = "Link";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
                        type_str = "Block device";
                        break;
                      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
                        type_str = "Character device";
                        break;
                      case EDV_OBJECT_TYPE_FIFO:
                        type_str = "FIFO Pipe";
                        break;
                      case EDV_OBJECT_TYPE_SOCKET:
                        type_str = "Socket";
                        break;
                    }

                    /* Format status bar message. */
                    if(total_selected > 1)
                        buf = g_strdup_printf(
                            "%i objects selected",
                            total_selected
                        );
                    else if(!strcmp(obj->name, ".."))
                        buf = g_strdup_printf(
                            "Parent directory selected"
                        );
                    else
                        buf = g_strdup_printf(
                            "%s \"%s\" selected%s",
                            type_str, obj->name, size_str
                        );

                    /* Set status bar message. */
                    EDVStatusBarMessage(
                        archiver->status_bar, buf, FALSE
                    );

                    /* Deallocate status bar message. */
                    g_free(buf);
                }
                else
                {
                    EDVStatusBarMessage(
                        archiver->status_bar,
                        "Object with NULL name selected",
                        FALSE
                    );
                }
            }

            /* Check if selected row is fully visible, if not then adjust
             * scroll position to try and make it visible.
             */
            if(gtk_clist_row_is_visible(clist, row) !=
                GTK_VISIBILITY_FULL
            )
                gtk_clist_moveto(
                    clist,
                    row, -1,	/* Row, column. */
                    0.5, 0.0	/* Row, column. */
                );

            EDVArchiverUpdateMenus(archiver);
        }

        reenterent = FALSE;
}

/*
 *      Recycle bin GtkCList "unselect_row" signal callback.
 */
void EDVArchiverUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((clist == NULL) || (archiver == NULL))
            return;

        if(archiver->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which clist this signal is for. */
        if(GTK_WIDGET(clist) == archiver->contents_clist)
        {


            EDVArchiverUpdateMenus(archiver);
        }

        reenterent = FALSE;
}


/*
 *      Location combo activate callback.
 */
void EDVArchiverComboActivateCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        GtkCombo *combo = (GtkCombo *)widget;
        edv_archiver_struct *archiver = (edv_archiver_struct *)data;
        if((widget == NULL) || (archiver == NULL))
            return;

        if(archiver->processing)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Check which widget was activated. */
        if(widget == archiver->location_combo)
        {
            gchar *new_path = EDVCopyEvaluateInputPath(
                NULL,           /* No parent path, imply use toplevel. */
                gtk_entry_get_text(GTK_ENTRY(combo->entry))
            );
            if(new_path != NULL)
            {
                EDVArchiverSetBusy(archiver, TRUE);
                GUIBlockInput(archiver->toplevel, TRUE);


                /* Update location combo, this is just to record history
		 * of the new path on the location combo.
		 */
                EDVArchiverSetLocation(archiver, new_path, TRUE);

		/* Clear the contents clist and load the listing of the
		 * new archive specified by new_path.
		 */
		EDVArchiverSelectArchive(archiver, new_path);


                GUIBlockInput(archiver->toplevel, FALSE);
                EDVArchiverSetBusy(archiver, FALSE);

		EDVArchiverUpdateMenus(archiver);

                /* Deallocate copy of the new path. */
                g_free(new_path);
                new_path = NULL;
            }
        }

        reenterent = FALSE;
}


/*
 *      Called whenever the global write protect has changed.
 *
 *      The new state is given as state.
 */
void EDVArchiverWriteProtectChangedCB(
        edv_archiver_struct *archiver, gbool state
)
{
        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        EDVArchiverUpdateMenus(archiver);
}


/*
 *      Called whenever a disk object has been modified.
 */
void EDVArchiverObjectModifiedNotifyCB(
        edv_archiver_struct *archiver, const gchar *path,
        const gchar *new_path, const struct stat *lstat_buf
)
{
        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        /* Check current location first, if the old path matches the
         * current location then the current location needs to be updated
         * to reflect the new path before notifying the contents clist.
         */
        if(path != NULL)
        {
            const gchar *cstrptr;


            if(new_path == NULL)
                new_path = path;

            /* Check if the old path matches the current location, if it
             * does then the current location needs to be updated to the
             * new path.
             */
            cstrptr = EDVArchiverCurrentLocation(archiver);
            if((cstrptr != NULL) ? !strcmp(cstrptr, path) : FALSE)
            {
                /* Old path matches current location, change values
                 * reflecting the current location to the value of
                 * new_path.
                 */
                EDVArchiverSetTitle(archiver, new_path);
                EDVArchiverSetLocation(archiver, new_path, FALSE);
                EDVArchiverUpdateLocationIcon(archiver, new_path);
            }
        }

	EDVArchiverContentsObjectModifiedNotify(
	    archiver, path, new_path, lstat_buf
	);

/*	EDVArchiverUpdateMenus(archiver); */
}

/*
 *      Called whenever a disk object has been removed.
 */
void EDVArchiverObjectRemovedNotifyCB(
        edv_archiver_struct *archiver, const gchar *path
)
{
        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        EDVArchiverContentsObjectRemovedNotify(archiver, path);

        /* Check if the path that was removed is the current location
         * (after letting contents clist and directory ctree check on it
         * first), if the path that was removed is the location then
         * set the current location to be the parent of the path that
         * was removed.
         */
        if(path != NULL)
        {
            const gchar *cstrptr;

            /* Check if the removed object path matches the current
             * location. If it matches then the current location needs to
             * be updated to reflect that.
             */
            cstrptr = EDVArchiverCurrentLocation(archiver);
            if((cstrptr != NULL) ? !strcmp(cstrptr, path) : FALSE)
            {
                /* Removed object path matches the current location. */

/* Do not need to update title or location because of this. */

            }
        }

/*	EDVArchiverUpdateMenus(archiver); */
}


/*
 *      Called whenever an archive object has been add to the
 *      specified archive arch_path.
 */
void EDVArchiverArchiveObjectAddedNotifyCB(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, edv_archive_object_struct *obj
)
{
	const gchar *cur_path;


        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

	/* Compare the archive path with the current archive opened on
	 * this archiver. If they are the same then report this signal
	 * to this archiver's contents list.
	 */
	cur_path = EDVArchiverCurrentLocation(archiver);
	if((cur_path != NULL) && (arch_path != NULL))
	{
	    if(!strcmp(cur_path, arch_path))
	    {
		EDVArchiverContentsArchiveObjectAddedNotify(
		    archiver, arch_path, path, obj
		);
	    }
	}

/*	EDVArchiverUpdateMenus(archiver); */
}

/*
 *	Called whenever an archive object has been modified in the
 *      specified archive arch_path.
 */
void EDVArchiverArchiveObjectModifiedNotifyCB(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path, const gchar *new_path,
        edv_archive_object_struct *obj
)
{
        const gchar *cur_path;


        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        /* Compare the archive path with the current archive opened on
         * this archiver. If they are the same then report this signal
         * to this archiver's contents list.
         */
        cur_path = EDVArchiverCurrentLocation(archiver);
        if((cur_path != NULL) && (arch_path != NULL))
        {
            if(!strcmp(cur_path, arch_path))
            {
                EDVArchiverContentsArchiveObjectModifiedNotify(
                    archiver, arch_path, path, new_path, obj
                );
            }
        }

/*	EDVArchiverUpdateMenus(archiver); */
}

/*
 *	Called whenever an archive object has been removed from the
 *	specified archive arch_path.
 */
void EDVArchiverArchiveObjectRemovedNotifyCB(
        edv_archiver_struct *archiver, const gchar *arch_path,
        const gchar *path
)
{
        const gchar *cur_path;


        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        /* Compare the archive path with the current archive opened on
         * this archiver. If they are the same then report this signal
         * to this archiver's contents list.
         */
        cur_path = EDVArchiverCurrentLocation(archiver);
        if((cur_path != NULL) && (arch_path != NULL))
        {
            if(!strcmp(cur_path, arch_path))
            {
                EDVArchiverContentsArchiveObjectRemovedNotify(
                    archiver, arch_path, path
                );
            }
        }

/*	EDVArchiverUpdateMenus(archiver); */
}

/*
 *      Called whenever an object has been added to the recycle bin.
 */
void EDVArchiverRecycledObjectAddedNotifyCB(
        edv_archiver_struct *archiver, guint index
)
{
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;


        /* There is not much interest when a recycled object has been
         * added or removed. Only the menus need to be updated if there
         * is a change in the number of recycled objects.
         */
        if(core_ptr->last_recbin_items != archiver->last_recbin_items)
            EDVArchiverUpdateMenus(archiver);
}

/*
 *      Called whenever an object has been removed from the recycle bin.
 */
void EDVArchiverRecycledObjectRemovedNotifyCB(
        edv_archiver_struct *archiver, guint index
)
{
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        /* There is not much interest when a recycled object has been
         * added or removed. Only the menus need to be updated if there
         * is a change in the number of recycled objects.
         */
        if(core_ptr->last_recbin_items != archiver->last_recbin_items)
            EDVArchiverUpdateMenus(archiver);
}


/*
 *      Called whenever the global configuration has changed.
 */
void EDVArchiverReconfiguredNotifyCB(edv_archiver_struct *archiver)
{
        GtkWidget *w;
        GtkCList *clist;
        edv_status_bar_struct *status_bar;
        edv_core_struct *core_ptr;


        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring.
	 */
	archiver->last_recbin_items = -1;
	archiver->last_write_protect_state = -1;


        /* Update title. */
        EDVArchiverSetTitle(
	    archiver,
	    EDVArchiverCurrentLocation(archiver)
	);

        /* Regenerate tool bar. */
        EDVArchiverToolbarRegenerate(archiver);

        /* Show tool bar? */
        w = archiver->tool_bar_handle;
        if(w != NULL)
        {
            archiver->tool_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
            );
            if(archiver->tool_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show location bar? */
        w = archiver->location_bar_handle;
        if(w != NULL)
        {
            archiver->location_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
            );
            if(archiver->location_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show find bar? */
        w = archiver->find_bar_handle;
        if(w != NULL)
        {
            archiver->find_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
            );
            if(archiver->find_bar_map_state)
                gtk_widget_show(w);
            else
                gtk_widget_hide(w);
        }

        /* Show status bar? */
        status_bar = archiver->status_bar;
        if(status_bar != NULL)
        {
            archiver->status_bar_map_state = EDVCFGItemListGetValueI(
                core_ptr->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
            );
            if(archiver->status_bar_map_state)
                EDVStatusBarMap(status_bar);
            else
                EDVStatusBarUnmap(status_bar);
        }


        /* Reset displayed row values of the contents clist. */
        clist = (GtkCList *)archiver->contents_clist;
        if(clist != NULL)
        {
            gtk_clist_freeze(clist);
	    EDVArchiverContentsResetRows(archiver);
            gtk_clist_thaw(clist);
        }

        /* Update menus. */
        EDVArchiverUpdateMenus(archiver);


        /* Notify archiver's toplevel widget to resize. */
        w = archiver->toplevel;
        if(w != NULL)
            gtk_widget_queue_resize(w);
}


/*
 *      Called whenever a MIME Type has been added to the core
 *      structure's list of MIME Types.
 */
void EDVArchiverMimeTypeAddedCB(
        edv_archiver_struct *archiver,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        /* Treat a MIME Type added the same as it would be for a MIME Type
         * modified, forward signal to the MIME Type modified callback.
         */
        EDVArchiverMimeTypeModifiedCB(archiver, mt_num, mt_ptr);
}

/*
 *      Called whenever a MIME Type has been modified on the core
 *      structure's list of MIME Types.
 */
void EDVArchiverMimeTypeModifiedCB(
        edv_archiver_struct *archiver,
        gint mt_num, edv_mimetype_struct *mt_ptr
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        edv_core_struct *core_ptr;
	GtkCList *clist;


        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        clist = (GtkCList *)archiver->contents_clist;
	if(clist == NULL)
	    return;


        /* Get copy of current location path. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        /* Reget listings. */

	/* Reset displayed row values of the contents clist. */
	gtk_clist_freeze(clist);
	EDVArchiverContentsResetRows(archiver);
	gtk_clist_thaw(clist);


/*	EDVArchiverUpdateMenus(archiver); */

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}

/*
 *      Called whenever a MIME Type has been removed from the core
 *      structure's list of MIME Types.
 */
void EDVArchiverMimeTypeRemovedCB(
        edv_archiver_struct *archiver, gint mt_num
)
{
        const gchar *cstrptr;
        gchar *cur_path;
        edv_core_struct *core_ptr;
        GtkCList *clist;


        if(archiver == NULL)
            return;

        if(archiver->processing)
            return;

        core_ptr = (edv_core_struct *)archiver->core_ptr;
        if(core_ptr == NULL)
            return;

        clist = (GtkCList *)archiver->contents_clist;
        if(clist == NULL)
            return;


        /* Get copy of current location path. */
        cstrptr = EDVArchiverCurrentLocation(archiver);
        cur_path = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;


        /* Reget listings. */

        /* Reset displayed row values of the contents clist. */
        gtk_clist_freeze(clist);
        EDVArchiverContentsResetRows(archiver);
        gtk_clist_thaw(clist);


/*	EDVArchiverUpdateMenus(archiver); */

        /* Deallocate copy of current location path. */
        g_free(cur_path);
        cur_path = NULL;
}

