# This file is part of Email-Reminder.
#
# Email-Reminder is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 3 of the
# License, or (at your option) any later version.
#
# Email-Reminder is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Email-Reminder; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

package EmailReminder::MonthlyEvent;

use strict;
use warnings;

use Date::Manip;
use POSIX;

use EmailReminder::Event;
use EmailReminder::Utils;

require Exporter;
our @ISA    = ("EmailReminder::Event", "Exporter");
#our @EXPORT = qw(get_name get_current_occurence_date valid_day);

# XML tags, attributes and values
my $DAY_TAG   = 'day';
my $NAME_TAG  = 'name';

# Global date variables
my $current_time = ParseDate("now");
my $current_date = ParseDate(UnixDate($current_time, "\%x"));
my $current_month = UnixDate($current_time, "\%m");
my $current_year = UnixDate($current_time, "\%Y");

# Hard-coded value for this event's type (class method)
sub get_type
{
    return 'monthly';
}

# Number of fields this event adds to its parent (class method)
sub get_nb_fields
{
    return EmailReminder::Event->get_nb_fields() + 2;
}

sub get_name
{
    my ($this) = @_;
    my $name  = EmailReminder::Utils::get_node_value($this->{XML_NODE}, $NAME_TAG);
    return $name;
}

sub set_name
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $NAME_TAG, $new_value);
}

sub valid_day
{
    my ($new_value) = @_;

    # Make sure the value is a valid number
    if ($new_value > 31) {
        $new_value = 31;
    } elsif ($new_value < 1) {
        $new_value = 1;
    }
    
    return $new_value;
}

sub get_day
{
    my ($this) = @_;
    my $day = EmailReminder::Utils::get_node_value($this->{XML_NODE}, $DAY_TAG);
    return valid_day($day);
}

sub set_day
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $DAY_TAG, valid_day($new_value));
}

sub get_current_occurence_date
{
    my ($this, $modifier) = @_;

    my $day = $this->get_day();
    return undef unless $day;

    # Set the day of the month where the event occurs, make sure the date is valid
    # (e.g. fix-up for event on the 31st when the month has only 30 days)
    my $current_occurence_date = ParseDate("$current_year-$current_month-$day");
    while (UnixDate($current_occurence_date, "\%d") != $day) {
        $day -= 1;
        $current_occurence_date = ParseDate("$current_year-$current_month-$day");
    }

    my $modified_date = $current_occurence_date;
    if ($modifier) {
        if ($modifier >= $day) {
            # We are warning about an event in a few days, past the end of the current month
            $modified_date = DateCalc($modified_date, " + 1 month");
        }
        $modified_date = DateCalc($modified_date, " - $modifier days");
    }

    return $modified_date;
}

sub get_message_body
{
    my $this = shift;

    # destination user
    my $first_name = shift;

    # event details
    my $when      = $this->{"WHEN"};
    my $name      = $this->get_name();
    
    my $message = <<MESSAGEEND;
Subject: $name

Hi $first_name,

I just want to remind you of the following event $when:

$name
MESSAGEEND

    return $message;
}

# Returns 1 if the event will occur in X days (X is a param)
sub will_occur
{
    my $this = shift;
    my $modifier = shift;
    
    # Apply the modifier to the event date
    my $current_occurence_date = $this->get_current_occurence_date($modifier);
    return 0 unless $current_occurence_date;
    
    my $tomorrow = DateCalc($current_date, " + 1 day");
    if (Date_Cmp($current_date, $current_occurence_date) == 0) {
        return 1;
    } elsif (Date_Cmp($current_date, $current_occurence_date) < 0 and
        Date_Cmp($tomorrow, $current_occurence_date) > 0) {
        # e.g. if today is the 30, the reminder is on the 31st and tomorrow is the 1st
        return 1;
    } else {
        return 0;
    }
}

1;
