# This file is part of Email-Reminder.
#
# Email-Reminder is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 3 of the
# License, or (at your option) any later version.
#
# Email-Reminder is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Email-Reminder; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

package EmailReminder::BirthdayEvent;

use strict;
use warnings;

use EmailReminder::Utils;
use EmailReminder::YearlyEvent;

require Exporter;
our @ISA = ("EmailReminder::YearlyEvent", "Exporter");
our @EXPORT = qw(get_th get_email);

# XML tags, attributes and values
my $EMAIL_TAG = 'email';

# Hard-coded value for this event's type (class method)
sub get_type
{
    return 'birthday';
}

# Number of fields this event adds to its parent (class method)
sub get_nb_fields
{
    return EmailReminder::YearlyEvent->get_nb_fields() + 1;
}

sub get_email
{
    my ($this) = @_;
    return EmailReminder::Utils::get_node_value($this->{XML_NODE}, $EMAIL_TAG);
}

sub set_email
{
    my ($this, $new_value) = @_;
    return EmailReminder::Utils::set_node_value($this->{XML_NODE}, $EMAIL_TAG, $new_value);
}

# Returns the age of the person (starts at 0 years old)
sub get_occurence
{
    my $this = shift;

    my $age = $this->EmailReminder::YearlyEvent::get_occurence();

    return defined($age) ? ($age - 1) : undef;
}

sub get_message_body
{
    my $this = shift;

    # destination user
    my $first_name = shift;

    # birthday person
    my $name  = $this->get_name();
    my $email = $this->get_email();
    my $age   = $this->get_occurence();
    my $when  = $this->{"WHEN"};
    
    my $subject = ($age and $when eq "today") ? "$name is now $age" : "${name}'s birthday";
    my $age_string = $age ? "turning $age" : "getting one year older";
    my $email_message = $email ? "\n\nYou can reach $name at $email." : "";

    my $message = <<MESSAGEEND;
Subject: $subject

Hi $first_name,

I just want to remind you that $name is $age_string $when.$email_message
MESSAGEEND

    return $message;
}

1;
