# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

from elisa.core.common import application
from elisa.core.media_uri import MediaUri
from elisa.core.components.model import Model
from elisa.core.default_config import PICTURES_CACHE
from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController

from elisa.plugins.youtube.models import YoutubeVideoModel, \
                                         YoutubeVideoListModel
from elisa.plugins.youtube.actions import SearchAction
from elisa.plugins.poblesec.actions import Action

from twisted.internet import defer

import os

try:
    from hashlib import md5
except ImportError:
    # hashlib is new in Python 2.5
    from md5 import md5


_ = install_translation('youtube')
_p = install_translation('poblesec')


class YoutubeVideoFeedModel(Model):

    def __init__(self):
        super(Model, self).__init__()
        self.title = None
        self.uri = None


def youtube_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/youtube'
    link.label = _('Youtube')
    link.icon = 'elisa.plugins.youtube.logo'
    controller.model.append(link)
    return defer.succeed(None)


# TODO: factorize this code in the core of Elisa in a caching utilities module
def _thumbnail_file(thumbnail_uri):
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)


class YoutubeController(HierarchyController):

    start_uri = MediaUri('http://gdata.youtube.com/feeds/')

    def initialize(self, videos=None, uri=start_uri):
        deferred = super(YoutubeController, self).initialize()
        self.uri = uri

        def add_feeds(self):
            # search
            self.actions.append(SearchAction(self))

            feeds_root = 'http://gdata.youtube.com/feeds/api/standardfeeds/'
            feeds = [(_('Top rated'), 'top_rated'),
                     (_('Top favorites'), 'top_favorites'),
                     (_('Most viewed'), 'most_viewed'),
                     (_('Most popular'), 'most_popular'),
                     (_('Most recent'), 'most_recent')]
            feed_models = []
            for feed in feeds:
                feed_model = YoutubeVideoFeedModel()
                feed_model.title = feed[0]
                feed_model.uri = MediaUri(feeds_root + feed[1])
                feed_models.append(feed_model)
            self.model.extend(feed_models)

            return self

        def resource_loaded(resource):
            if isinstance(resource, YoutubeVideoListModel):
                self.model.extend(resource.videos)
            return self

        def load_resource(self):
            resource, get_deferred = application.resource_manager.get(self.uri)
            return get_deferred

        def extend_model(self, videos):
            self.model.extend(videos)
            return self

        # A list of videos
        if videos is not None:
            deferred.addCallback(extend_model, videos)
        elif uri == self.start_uri:
            # List of the feeds
            deferred.addCallback(add_feeds)
        else:
            # One particular feed
            deferred.addCallback(load_resource)
            deferred.addCallback(resource_loaded)

        return deferred

    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            item.run()
        elif isinstance(item, YoutubeVideoFeedModel):
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            dfr = browser.history.append_controller(self.path, item.title, uri=item.uri)

        elif isinstance(item, YoutubeVideoModel):
            def play_video(playable_model):
                controllers = self.frontend.retrieve_controllers('/poblesec/video_player')
                player = controllers[0]
                player.player.play_model(playable_model)

                controllers = self.frontend.retrieve_controllers('/poblesec')
                main = controllers[0]
                main.show_video_player()
                self.stop_loading_animation()

            if item.playable_model:
                # We already know the real URL of the playable video, no
                # need to query the resource manager
                play_video(item.playable_model)
                dfr = defer.succeed(True)
            else:
                playable_model, dfr = application.resource_manager.get(item.playable_uri, item)
                dfr.addCallback(play_video)


class YoutubeViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        return defer.succeed(item.title)

    def get_default_image(self, item):
        if isinstance(item, YoutubeVideoFeedModel):
            resource = 'elisa.plugins.poblesec.video_folders'
        elif isinstance(item, YoutubeVideoModel):
            resource = 'elisa.plugins.poblesec.file_video'
        elif isinstance(item, Action):
            resource = item.icon
        return resource

    def get_image(self, item, theme):
        if isinstance(item, YoutubeVideoModel):
            # Get a thumbnail for the video
            thumbnail_uri = item.thumbnails[0].references[0]
            try:
                thumbnail_file = item.thumbnail_file
            except AttributeError:
                item.thumbnail_file = _thumbnail_file(thumbnail_uri)
                thumbnail_file = item.thumbnail_file
            if os.path.exists(thumbnail_file):
                return defer.succeed(thumbnail_file)
            else:
                # Download the thumbnail first
                def got_thumbnail(data_model):
                    fd = file(thumbnail_file, 'wb')
                    fd.write(data_model.data)
                    fd.close()
                    return thumbnail_file

                data_model, thumbnail_deferred = \
                    application.resource_manager.get(thumbnail_uri)
                return thumbnail_deferred.addCallback(got_thumbnail)
        else:
            return None

    def get_preview_image(self, item, theme):
        if isinstance(item, YoutubeVideoModel):
            try:
                return item.thumbnail_file
            except AttributeError:
                return None
        else:
            return None


class YoutubePreviewListController(YoutubeController, MenuItemPreviewListController):
    view_mode = YoutubeViewMode


class YoutubeCoverflowController(YoutubeController, ImageWithReflectionCoverflowController):
    view_mode = YoutubeViewMode


class YoutubeGridController(YoutubeController, GridItemGridController):
    view_mode = YoutubeViewMode


class YoutubeListSwitcherController(ListSwitcherController):
    modes = [YoutubePreviewListController,
             YoutubeCoverflowController,
             YoutubeGridController]
    default_config = {'view_mode': 'preview_list'}


def use_me_hook(frontend):
    """
    'Use me' hook that takes the user to the Video/Internet/Youtube section.

    @param frontend: the current frontend (poblesec)
    @type frontend:  L{elisa.plugins.pigment.pigment_frontend.PigmentFrontend}

    @return:         a deferred fired when the action is complete
    @rtype:          L{twisted.internet.defer.Deferred}
    """
    browser = frontend.retrieve_controllers('/poblesec/browser')[0]
    browser.history.clear()

    paths = [('/poblesec/video_menu', _p('Video')),
             ('/poblesec/video/internet', _p('Internet')),
             ('/poblesec/youtube', _('Youtube'))]

    # Select the video section in Elisa's main menu
    for index, link in enumerate(browser.history.current.model):
        if link.controller_path == paths[0][0]:
            browser.history.current.menu.selected_item_index = index
            break

    def append_controller_cb(result, path, label, **kwargs):
        return browser.history.append_controller(path, label, **kwargs)

    dfr = defer.succeed(None)
    for path, label in paths:
        dfr.addCallback(append_controller_cb, path, label)
    return dfr
