# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Common models related to media.
"""

from elisa.core.components.model import Model
from elisa.core.utils import notifying_list

from twisted.internet import defer

class PlayableModel(Model):

    """
    A playable model contains the real URI of a playable media, that is,
    typically, a URI directly usable by gstreamer to play streamed data.
    It also potentially contains a title for the media.

    @ivar uri:   the real URI of the playable media
    @type uri:   L{elisa.core.media_uri.MediaUri}
    @ivar title: the title of the playable media
    @type title: C{unicode}
    @ivar allow_pause: whether the player can pause the media playback or not
    @type allow_pause: C{bool}
    @ivar allow_seek: whether the player can seek inside the media playback or not
    @type allow_seek: C{bool}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(PlayableModel, self).__init__()
        self.uri = None
        self.title = None
        self.allow_pause = True
        self.allow_seek = True

    def __eq__(self, other):
        """
        Compare two playable models.

        Two playable models are considered equal if and only if their URIs are
        equal.

        @param other: another playable model
        @type other:  L{elisa.plugins.base.models.media.PlayableModel}

        @return:      C{True} if the two models are equal, C{False} otherwise
        @rtype:       C{bool}
        """
        return (self.uri == other.uri)

class PlaylistModel(Model, notifying_list.List):
    """
    A playlist model is a list of models that can be played by
    a player. The models stored in this list *must* implement the
    get_playable_model method, like
    L{elisa.plugins.base.models.audio.TrackModel} and
    L{elisa.plugins.base.models.video.VideoModel}.

    @ivar allow_previous: if it is possible to set as current the previous track
                          in the playlist relatively to the current one by
                          calling the method 'previous_track'
    @type allow_previous: C{bool}
    @ivar allow_next: if it is possible to set as current the next track
                      in the playlist relatively to the current one by
                      calling the method 'next_track'
    @type allow_next: C{bool}
    @ivar allow_jump: if it is possible to set as current any track knowing its
                      index in the playlist by calling the method
                      'set_current_index'
    @type allow_jump: C{bool}
    @ivar current_track: playable model corresponding to the current track in
                         the playlist; it differs from the model stored in the
                         playlist because it is not the model itself but only
                         the shell containing the necessary playback information
    @type current_track: L{elisa.plugins.base.models.media.PlayableModel}
    @ivar current_index: index in the playlist of the current model being played
    @type current_index: C{int}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(PlaylistModel, self).__init__()
        self.allow_previous = True
        self.allow_next = True
        self.allow_jump = True
        self.current_track = None
        self.current_index = -1

    def previous_track(self):
        """
        Decrement L{current_index} of the playlist and return the
        PlayableModel located at the new position in the playlist.

        @rtype: L{twisted.internet.defer.Deferred}
        """
        if self.current_index > 0:
            dfr = self.set_current_index(self.current_index-1)
        else:
            dfr = defer.fail(Exception("Beginning of playlist reached"))
        return dfr

    def next_track(self):
        """
        Increment L{current_index} of the playlist and return the
        PlayableModel located at the new position in the playlist.

        @rtype: L{twisted.internet.defer.Deferred}
        """
        if self.current_index < len(self)-1:
            dfr = self.set_current_index(self.current_index+1)
        else:
            dfr = defer.fail(Exception("End of playlist reached"))
        return dfr

    def set_current_index(self, index):
        """
        Set L{current_index} of the playlist and return the
        PlayableModel located at the new position in the playlist.

        @rtype: L{twisted.internet.defer.Deferred}
        """
        def got_track(track):
            self.current_track = track
            return track

        self.current_index = index

        dfr = self._retrieve_playable_model_at(index)
        dfr.addCallback(got_track)
        return dfr

    def _retrieve_playable_model_at(self, index):
        track = self[index]
        if not isinstance(track, PlayableModel):
            dfr = track.get_playable_model()
        else:
            dfr = defer.succeed(track)
        return dfr


class RawDataModel(Model):

    """
    A raw data model contains raw data from a media file (can be binary, text,
    etc...).

    @ivar data: the raw data
    @type data: C{str}
    @ivar size: the total size of the data
    @type size: C{int}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(RawDataModel, self).__init__()
        self.data = None
        self.size = None


# TODO: subclass the RawDataModel to have a model that reads its data
#       asynchronously.
