# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Benjamin Kampmann <benjamin@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>
#          Philippe Normand <philippe@fluendo.com>

"""
Common models related to image.
"""

import math, md5, os
from elisa.core.components.model import Model
from elisa.core.utils.i18n import install_translation

from elisa.plugins.base.models.file import FileModel
from elisa.plugins.base.models.media import RawDataModel

from twisted.internet import defer

pi = math.pi

ROTATED_0=1
HORIZ_MIRROR=2
ROTATED_180=3
VERT_MIRROR=4
HORIZ_MIRROR_ROTATED_90_CCW=5
ROTATED_90_CW=6
HORIZ_MIRROR_ROTATED_90_CW=7
ROTATED_90_CCW=8

_ = install_translation('base')

# Constants detailing the various possible Image orientations
# These come from the EXIF spec and contain PgmImage rotation settings
# for earch orientation value.
IMAGE_ORIENTATIONS={ROTATED_0: (_('Horizontal (normal)'),
                                {'rx': 0, 'ry': 0, 'rz': 0}),
                    HORIZ_MIRROR: (_('Mirrored horizontally'),
                                   {'rx': 0, 'ry': pi, 'rz': 0}),
                    ROTATED_180: (_('Rotated 180°'), {'rx': 0, 'ry': 0, 'rz': pi}),
                    VERT_MIRROR: (_('Mirrored vertically'),
                                  {'rx': pi, 'ry': 0, 'rz': 0}),
                    HORIZ_MIRROR_ROTATED_90_CCW: (_('Mirrored horizontally then rotated 90° counter-clock-wise'),
                                                  {'rx': 0, 'ry': pi, 'rz': pi * 1.5}),
                    ROTATED_90_CW: (_('Rotated 90° clock-wise'),
                                    {'rx': 0, 'ry': 0, 'rz': pi * 0.5}),
                    HORIZ_MIRROR_ROTATED_90_CW: (_('Mirrored horizontally then rotated 90° clock-wise'),
                                                 {'rx': 0, 'ry': pi, 'rz': pi * 0.5}),
                    ROTATED_90_CCW: (_('Rotated 90° counter-clock-wise'),
                                     {'rx': 0, 'ry': 0, 'rz': pi * 1.5})
                    }


# for the caching we have to make sure that the picture cache is existing.
from elisa.core.default_config import PICTURES_CACHE

if not os.path.exists(PICTURES_CACHE):
    os.makedirs(PICTURES_CACHE, 0755)


class ImageModel(Model):

    """
    Representation of an image.

    An image model contains a list of references to image files that are in
    fact one image in various dimensions.
    This list is ordered by increasing size of image.
    This allows to easily retrieve the largest or the smallest representation
    of an image.

    'A simple example': the image is used as the cover art of an
    L{elisa.plugins.base.models.audio.AlbumModel}. A resource provider fills
    this image model with a thumbnail and a huge high contrast image. Now the
    UI can decide that it does not want to show such a high quality image
    because it is going to be used as an icon in a list. It uses
    C{model.cover.references[0]} (the first one, the smallest image) and the
    amount of data to load is minimal. Later the user decides to play a track
    of the album and the UI wants to show it in fullscreen, it always uses the
    last image in the list because it is the largest one (and very probably the
    one with the best quality for a huge picture):
    C{model.cover.references[-1]}.

    @ivar references:  images ordered by increasing size of raw data
    @type references:  C{list} of L{elisa.core.media_uri.MediaUri}
    @ivar orientation: orientation of the image, its meaning is the same as
                       the orienation EXIF field
    @type orientation: one of the keys of C{IMAGE_ORIENTATIONS}
    @ivar can_rotate:  can the image represented by the model be rotated?
    @type can_rotate:  C{bool}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(ImageModel, self).__init__()
        self.references = []
        self.orientation = ROTATED_0
        self.can_rotate = True

    def get_cached_data_path(self, reference_num):
        uri = self.references[reference_num]
        path = self._compute_cached_path(uri)

        if os.path.exists(path):
            # already cached
            return defer.succeed(path)

        dfr = self._retrieve_raw_data_model(uri)
        dfr.addCallback(self._cache_raw_data, path)
        return dfr

    
    def _compute_cached_path(self, uri):
        filename = md5.new(str(uri)).hexdigest() + '.' + uri.extension
        path = os.path.join(PICTURES_CACHE, filename)
        return path

    def _cache_raw_data(self, raw_data_model, path):
        if isinstance(raw_data_model, FileModel):
            # this is conceptionally wrong. we should not get it here at all!
            return raw_data_model.uri.path

        fd = open(path, 'wb')
        fd.write(raw_data_model.data)
        fd.close()
        return path

    def _retrieve_raw_data_model(self, uri):
        from elisa.core import common
        resource_manager = common.application.resource_manager

        model, dfr = resource_manager.get(uri)
        return dfr
