# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Philippe Normand <philippe@fluendo.com>

"""
Models managed by TheTvDB.com resource_provider
"""

from elisa.core.components.model import Model
from elisa.core import media_uri

from elisa.plugins.thetvdb.constants import *
from elisa.plugins.thetvdb import utils

class TvSeriesListModel(Model):

    def __init__(self):
        super(TvSeriesListModel, self).__init__()
        self.series = []

class TvSeriesModel(Model):

    def __init__(self):
        """

        @ivar id: tvdb.com ID of the show
        @type id: C{int}
        @ivar imdbid: imdb.com ID of the show
        @type imdbid: C{unicode}
        @ivar zap2itid: zap2it ID of the show
        @type zap2itid: C{unicode}
        @ivar name: name of the show, translated in L{language}, if present
        @type name: C{unicode}
        @ivar overview: overview of the show, translated in L{language}, if present
        @type overview: C{unicode}
        @ivar first_aired_date: date at which the show was first aired
        @type first_aired_date: C{datetime.date}
        @ivar banner_url: banner of the show
        @type banner_url: C{elisa.core.media_uri.MediaUri}
        @ivar runtime: usual runtime of the show, in minutes
        @type runtime: C{int}
        @ivar actors: permanent actors of the show
        @type actors: C{list} of C{unicode}
        @ivar airs_day_of_week: day of the week at which the show airs
        @type airs_day_of_week: C{unicode}
        @ivar airs_time: hour of the day at which the show airs
        @type airs_time: C{datetime.time}
        @ivar genres: list of genres corresponding to the show
        @type genres: C{list} of C{unicode}
        @ivar network: network name on which the show airs
        @type network: C{unicode}
        @ivar rating: user rating of the show
        @type rating: C{float} between 0 and 10
        @ivar status: is the show still airing or discontinued
        @type status: C{unicode}
        @ivar last_updated: date at which the show entry was last edited
        @type last_updated: C{datetime.date}
        @ivar fanart_url: url of user supplied fanart picture
        @type fanart_url: C{elisa.core.media_uri.MediaUri}
        @ivar poster_url: url of the show poster
        @type poster_url: C{elisa.core.media_uri.MediaUri}
        @ivar language: language for which the show data was retrieved
        @type language: C{unicode}
        @ivar season_banners: banners sorted by season and language
        @type season_banners: C{dict}. Keys: season numbers, Values:
                              C{dict} of
                              C{unicode}:C{elisa.core.media_uri.MediaUri}
        """
        super(TvSeriesModel, self).__init__()
        self.id = None
        self.imdbid = None
        self.zap2itid = None
        self.name = None
        self.overview = None
        self.first_aired_date = None
        self.banner_url = None

        # optional:
        self.runtime = None
        self.actors = [] # of strings
        self.airs_day_of_week = None
        self.airs_time = None
        self.genres = [] # of strings
        self.network = None
        self.rating = None
        self.status = None
        self.last_updated = None
        self.fanart_url = None
        self.poster_url = None
        self.language = None
        self.season_banners = {}

    def extended_infos_url(self, lang):
        url = 'http://%s/api/%s/series/%s/%s.xml' % (API_SERVER, API_KEY,
                                                     self.id, lang)
        return media_uri.MediaUri(url)

    @property
    def season_banners_url(self):
        url = 'http://%s/api/%s/series/%s/banners.xml' % (API_SERVER, API_KEY,
                                                          self.id)
        return media_uri.MediaUri(url)

    def episode_url(self, season, episode, lang):
        return utils.get_episode_url(self.id, season, episode, lang)

class TvEpisodeModel(Model):

    def __init__(self):
        """

        @ivar id: tvdb.com ID of the episode
        @type id: C{int}
        @ivar imdbid: imdb.com ID of the episode
        @type imdbid: C{unicode}
        @ivar seriesid: tvdb.comm ID of the series
        @type seriesid: C{int}
        @ivar seasonid: tvdb.comm ID of the season
        @type seasonid: C{int}
        @ivar last_updated: date at which the entry was last updated
        @type last_updated: C{datetime.date}
        @ivar season_number: season number during which the episode was aired
        @type season_number: C{int}
        @ivar episode_number: number of the episode in the season
        @type episode_number: C{int}
        @ivar director: name of the episode director
        @type director: C{unicode}
        @ivar name: episode name (translated in L{language})
        @type name: C{unicode}
        @ivar overview: episode overview (translated in L{language})
        @type overview: C{unicode}
        @ivar guest_stars: guest stars appearing in the episode
        @type guest_stars: C{list} of C{unicode}
        @ivar language: language for which the episode data was retrieved
        @type language: C{unicode}
        @ivar rating: user-rating of the episode
        @type rating: L{float} (between 0 and 10)
        @ivar writer: name of the person who wrote the episode
        @type writer: L{unicode}
        @ivar poster_url: url of the episode poster (usually jpeg data)
        @type poster_url: C{elisa.core.media_uri.MediaUri}
        """
        super(TvEpisodeModel, self).__init__()
        self.id = None
        self.imdbid = None
        self.seriesid = None
        self.seasonid = None
        self.last_updated = None
        self.season_number = None
        self.episode_number = None
        self.director = None
        self.name = None
        self.guest_stars = [] # of strings
        self.overview = None
        self.language = None
        self.rating = None
        self.writer = None
        self.poster_url = None
