# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Widget and controller for a popup dialog.
"""

from elisa.core.input_event import EventValue

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text

from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.widgets.button import TextButton

import pgm
from pgm.timing import implicit


class PopupButton(TextButton):
    pass


class PopupWidget(Widget):

    """
    Popup widget (containing an icon, a title, a body text, and buttons).
    """

    def __init__(self, icon, title, text, buttons):
        super(PopupWidget, self).__init__()

        self.background = Image()
        self.add(self.background)
        self.background.position = (0.0, 0.0, 0.0)
        self.background.size = (1.0, 1.0)
        self.background.visible = True

        self.icon = Image()
        self.add(self.icon)
        theme = Theme.get_default()
        image = theme.get_resource(icon)
        self.icon.set_from_file(image)
        self.icon.visible = True

        self.title = Text()
        self.add(self.title)
        self.title.label = title
        self.title.visible = True

        self.text = Text()
        self.add(self.text)
        self.text.label = text
        self.text.visible = True

        self.buttons = buttons
        self._buttons = []
        self.callbacks = {}

        def button_clicked_cb(button_widget, x, y, z, button, time, pressure):
            self.selected_item_index = self._buttons.index(button_widget)
            self.callbacks[button_widget]()
            return True

        for text, callback in self.buttons:
            button = PopupButton()
            self._buttons.append(button)
            self.add(button)
            button.label = text
            button.bg_a = 0
            button.visible = True

            self.callbacks[button] = callback
            button.connect('clicked', button_clicked_cb)

        self._selected_item_index = -1
        self.selector = SlicedImageHorizontal()
        self.add(self.selector)
        resource_root = 'elisa.plugins.poblesec.selector'
        left_cap = theme.get_resource('%s_left_cap' % resource_root)
        right_cap = theme.get_resource('%s_right_cap' % resource_root)
        body = theme.get_resource('%s_body' % resource_root)
        self.selector.left_cap.set_from_file(left_cap)
        self.selector.right_cap.set_from_file(right_cap)
        self.selector.body.set_from_file(body)
        self.selector.visible = True

        self._animated_selector = implicit.AnimatedObject(self.selector)
        self._animated_selector.setup_next_animations(duration=200)

        self.update_style_properties(self.style.get_items())

    def update_style_properties(self, props=None):
        if props is None:
            return

        remaining_props = {}

        for key, value in props.iteritems():
            try:
                subwidget, attribute = key.split('-')
            except ValueError:
                remaining_props[key] = value
                continue
            if subwidget == 'buttons' and attribute == 'position':
                # Change the position of all the buttons
                for index, button in enumerate(self._buttons):
                    button.x = value[0] + index * props['button-size'][0]
                    button.y = value[1]
                    button.z = value[2]
                self.selector.position = value
            elif subwidget == 'button':
                for button in self._buttons:
                    setattr(button, attribute, value)
            else:
                remaining_props[key] = value

        if len(remaining_props) > 0:
            return super(PopupWidget, self).update_style_properties(remaining_props)

    def clean(self):
        self._animated_selector = None
        self.selector.clean()
        self.selector = None
        self.text.clean()
        self.text = None
        self.title.clean()
        self.title = None
        self.icon.clean()
        self.icon = None
        self.background.clean()
        self.background = None
        return super(PopupWidget, self).clean()

    def _set_selected_item_index(self, value):
        if value >= 0 and value < len(self._buttons):
            self._selected_item_index = value
            button = self._buttons[value]
            button.focus = True
            self._animated_selector.x = button.x

    def _get_selected_item_index(self):
        return self._selected_item_index

    selected_item_index = property(fget=_get_selected_item_index,
                                   fset=_set_selected_item_index)


class PopupController(PigmentController):

    """
    Popup controller.
    """

    def initialize(self, icon, title, text, buttons):
        dfr = super(PopupController, self).initialize()
        self.icon = icon
        self.title = title
        self.text = text
        self.buttons = buttons
        dfr.addCallback(self._create_widget)
        dfr.addCallback(self._connect_to_signals)
        return dfr

    def _create_widget(self, result):
        self.popup = PopupWidget(self.icon, self.title,
                                 self.text, self.buttons)
        self.widget.add(self.popup)
        self.popup.position = (0.0, 0.0, 0.1)
        self.popup.size = (1.0, 1.0)
        self.popup.visible = True
        return result

    def _connect_to_signals(self, result):
        self._mouse_event_handler_ids = []
        for event in ['double-clicked', 'drag-end', 'released', 'clicked',
                      'pressed', 'drag-begin', 'drag-motion']:
            event_id = self.popup.background.connect(event,
                                                     self._mouse_event_cb)
            self._mouse_event_handler_ids.append(event_id)

        scrolled_event_id = self.popup.background.connect('scrolled',
                                                          self._scrolled_cb)
        self._mouse_event_handler_ids.append(scrolled_event_id)

        # By default the focus is forwarded to the first button of the popup
        self._focus_handler_id = self.widget.connect('focus', self._on_focus)

        return result

    def clean(self):
        self.widget.disconnect(self._focus_handler_id)
        for event_id in self._mouse_event_handler_ids:
            self.popup.background.disconnect(event_id)
        self.popup.clean()
        self.popup = None
        return super(PopupController, self).clean()

    def _mouse_event_cb(self, *args):
        # Swallow the event
        return True

    def _scrolled_cb(self, widget, x, y, z, direction, time):
        if direction == pgm.SCROLL_DOWN:
            self.popup.selected_item_index -= 1
        elif direction == pgm.SCROLL_UP:
            self.popup.selected_item_index += 1
        return True

    def _on_focus(self, widget, focus):
        # always forward the focus to the buttons of the popup
        if focus:
            self.popup.selected_item_index = 0

    def handle_input(self, manager, input_event):
        if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
            self.buttons[self.popup.selected_item_index][1]()
        elif input_event.value == EventValue.KEY_GO_LEFT:
            self.popup.selected_item_index -= 1
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            self.popup.selected_item_index += 1
        elif input_event.value in (EventValue.KEY_f, EventValue.KEY_EXIT):
            # Propagate only the 'fullscreen' and 'exit' events. All the other
            # events are swallowed because the popup is modal.
            su = super(PopupController, self)
            return su.handle_input(manager, input_event)
        return True
