# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.pigment_frontend import ControllerNotFound
from elisa.plugins.poblesec.transitions import Transition, FadeIn, FadeOut
from elisa.plugins.poblesec.configuration_wizard import ConfigurationWizard

from elisa.core.media_uri import MediaUri
from elisa.plugins.gstreamer.amp_master import GstMetadataAmpClient

from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.base.models.image import ImageModel

from elisa.plugins.pigment.message import PigmentFrontendLoadedMessage

from elisa.core.input_event import *
from elisa.core.utils import defer
from elisa.core.default_config import CONFIG_DIR
from elisa.core.application import NewElisaVersionMessage
from elisa.core.utils.update_checker import AvailablePluginUpdatesMessage

from elisa.core import common
from elisa.core.utils import typefinding
from elisa.core.utils.i18n import install_translation

_ = install_translation('poblesec')

import platform
import os

from twisted.internet import task

import pgm


try:
    import dbus
    from elisa.plugins.poblesec.dbus_player import Player as DBusPlayer
except ImportError:
    dbus = None

class PartialFadeOut(Transition):
    def apply(self, controller):
        def cancel(dfr):
            self._stop(dfr, controller)
        dfr = defer.Deferred(canceller=cancel)

        def end_animation(timer):
            if not dfr.called:
                dfr.callback(controller)

        controller.widget_animated.update_animation_settings(duration=400,
                                                    end_callback=end_animation)
        controller.widget_animated.opacity = 50

        return dfr


class PoblesecController(PigmentController):

    default_config = {'viewport_buttons_mode': 'desktop',
                      'enable_remove': '0'}
    config_doc = {'viewport_buttons_mode': 'what to display in the top-right'\
                  ' corner of the viewport. Two modes available: "desktop" and'\
                  ' "embedded". In "desktop" we show the minimize/maxmimize/close'\
                  ' buttons and in "embedded" we show only an home button',
                  'enable_remove': 'Whether or not to allow the user to '\
                  ' remove/hide videos/tracks/photos'}

    def __init__(self):
        super(PoblesecController, self).__init__()
        self._overlay_in = FadeIn()
        self._overlay_out = FadeOut()
        self._background_in = FadeIn()
        self._background_out = PartialFadeOut()

        self.current_player = None

        self._popup_visible = False
        self._popups_queue = []
        self._popups_enabled = True

    def initialize(self):
        bus = common.application.bus
        bus.register(self._got_new_version, NewElisaVersionMessage)
        bus.register(self._frontend_loaded, PigmentFrontendLoadedMessage)
        bus.register(self._plugin_updates_available,
                     AvailablePluginUpdatesMessage)
        return super(PoblesecController, self).initialize()

    def _got_new_version(self, message, sender):
        if platform.system().lower() == 'windows' and message.installer_url:
            self._download_installer(message.installer_url, message.version)

    def _download_installer(self, url, version):
        uri = MediaUri(url)
        installer_path = os.path.join(CONFIG_DIR, uri.filename)

        # avoid downloading twice the same file
        if not os.path.exists(installer_path):

            def got_installer(model):
                fd = open(installer_path, 'wb')
                fd.write(model.data)
                fd.close()
                return self._show_installer_popup(installer_path, version)

            model, dfr = common.application.resource_manager.get(uri)
            dfr.addCallback(got_installer)
        else:
            self._show_installer_popup(installer_path, version)

    def _show_installer_popup(self, installer_path, version):
        icon = 'elisa.plugins.poblesec.warning'
        title = _('New Elisa version available')
        text = _('Version %s of Elisa is available. Would you like to install it now?')
        text = text % version

        def ok_clicked():
            # execute installer in background and stop Elisa
            from win32api import ShellExecute
            import pywintypes

            try:
                ShellExecute(0, 'open', installer_path, '', None, 1)
            except pywintypes.error, error:
                self.warning(error)
                self.hide_popup()
            else:
                common.application.stop()

        buttons = [(_('Yes'), ok_clicked), (_('Later'), self.hide_popup)]
        self.enqueue_popup(icon, title, text, buttons)

    def _frontend_loaded(self, message, sender):
        if not common.application.first_run:
            return
        self.debug('First run, showing the configuration wizard')

        # While the wizard is visible, no popups can be shown
        #self.enable_popups(False)

        def start_wizard(wizard):
            self._wizard_close_handler_id = wizard.connect('closed',
                                                           self._close_wizard_cb)
            wizard.set_frontend(self.frontend)
            return wizard.start()

        def show_wizard(wizard):
            self.current_controller = self.focused_controller
            wizard.widget.visible = False
            self.widget.add(wizard.widget)
            wizard.widget.position = (0.0, 0.0, 1.0)
            wizard.widget.size = (1.0, 1.0)
            wizard.widget.regenerate()
            self._background_in.apply(wizard)
            self.focused_controller = wizard
            self.focused_controller.widget.focus = True
            return wizard

        # This is a sample configuration wizard.
        #from elisa.plugins.poblesec.configuration.welcome import WelcomeScreen
        #from elisa.plugins.poblesec.configuration.finished import FinalScreen
        #welcome = (WelcomeScreen, {'next': 'final', 'quit': 'quit'})
        #final = (FinalScreen, {'next': 'done', 'prev': 'welcome'})
        #screens = {'welcome': welcome, 'final': final}
        #dfr = ConfigurationWizard.create(None, screens=screens,
        #                                 start='welcome')
        #dfr.addCallback(start_wizard)
        #dfr.addCallback(show_wizard)
        dfr = defer.succeed(self)

        return dfr

    def _close_wizard_cb(self, wizard):
        def faded(controller):
            self.focused_controller = self.current_controller
            self.focused_controller.widget.focus = True
            self.widget.remove(controller.widget)
            # Now popups are allowed to show up
            self.enable_popups(True)
            return controller.clean()

        return self._background_out.apply(wizard).addCallback(faded)

    def _plugin_updates_available(self, message, sender):
        # Plugin updates and/or new recommended plugins are available for
        # download from the plugin repository.

        def iterate_plugins(available_updates, new_recommended, updated):
            plugin_registry = common.application.plugin_registry

            def _plugin_updated(result, plugin_dict):
                self.debug('Updated %s to version %s.' % \
                           (plugin_dict['name'], plugin_dict['version']))
                updated.append(plugin_dict['name'])
                return result

            def _update_failed(failure, plugin_dict):
                self.debug('Failed to update %s to version %s.' % \
                           (plugin_dict['name'], plugin_dict['version']))
                # Swallow the failure, to go on with the remaining updates.
                return None

            for plugin_dict in available_updates:
                dfr = plugin_registry.update_plugin(plugin_dict)
                dfr.addCallback(_plugin_updated, plugin_dict)
                dfr.addErrback(_update_failed, plugin_dict)
                yield dfr

            def _plugin_installed(result, plugin_dict):
                self.debug('Installed %s %s.' % \
                           (plugin_dict['name'], plugin_dict['version']))
                return result

            def _install_failed(failure, plugin_dict):
                self.debug('Failed to install %s %s.' % \
                           (plugin_dict['name'], plugin_dict['version']))
                # Swallow the failure, to go on with the remaining plugins.
                return None

            def _install_plugin(egg_file, plugin_dict):
                install_dfr = \
                    plugin_registry.install_plugin(egg_file,
                                                   plugin_dict['name'])
                install_dfr.addErrback(_install_failed)
                return install_dfr

            for plugin_dict in new_recommended:
                dfr = plugin_registry.download_plugin(plugin_dict)
                dfr.addCallback(_install_plugin, plugin_dict)
                dfr.addCallback(_plugin_installed, plugin_dict)
                yield dfr

        def process_plugins(result, updates, new, updated):
            return task.coiterate(iterate_plugins(updates, new, updated))

        def show_restart_popup(result, updated):
            if len(updated) == 0:
                # Restart is needed only for plugin updates
                return result

            icon = 'elisa.plugins.poblesec.plugin'
            title = _('Need To Restart Elisa')
            if len(updated) == 1:
                text = _('An update has been installed. For it to take ' \
                         'effect, please exit and restart Elisa.')
            else:
                text = _('%d updates have been installed. For them to ' \
                         'take effect, please exit and restart Elisa.') % \
                         len(updated)

            buttons = [(_('OK'), self.hide_popup)]
            import sys
            if platform.system() == 'Windows' and hasattr(sys, 'frozen'):
                # Running an installed Elisa on Windows: we are able to restart
                # Elisa automatically.
                buttons[0] = (_('Later'), self.hide_popup)
                from mswin32 import tools
                restart_button = (_('Restart'), lambda: tools.exit(1, True))
                buttons.insert(0, restart_button)

            self.enqueue_popup(icon, title, text, buttons)
            return result

        def do_process_plugins(updates, new):
            if self._popup_visible:
                dfr = self.hide_popup()
            else:
                dfr = defer.succeed(None)
            updated = []
            dfr.addCallback(process_plugins, updates, new, updated)
            dfr.addCallback(show_restart_popup, updated)
            return dfr

        config = common.application.config
        silent_update = \
            config.get_option('auto_update_plugins', section='plugin_registry')
        silent_install = \
            config.get_option('auto_install_new_recommended_plugins',
                              section='plugin_registry')

        if silent_update and silent_install:
            # Do not ask the user's confirmation
            return do_process_plugins(message.available_updates,
                                      message.new_recommended)

        elif silent_update:
            if len(message.new_recommended) == 0:
                return do_process_plugins(message.available_updates, [])

            # Ask the user's confirmation to install new recommended plugins
            title = _('New Recommended Plugins Available')
            if len(message.new_recommended) == 1:
                text = _('A new recommended plugin is available for ' \
                         'installation. Do you want to install it?')
            else:
                text = _('%d new recommended plugins are available for ' \
                         'installation. Do you want to install them?') % \
                         len(message.new_recommended)
            icon = 'elisa.plugins.poblesec.plugin'
            buttons = [(_('Now'),
                        lambda: do_process_plugins(message.available_updates,
                                                   message.new_recommended)),
                       (_('Later'),
                        lambda: do_process_plugins(message.available_updates,
                                                   []))]
            return self.enqueue_popup(icon, title, text, buttons)

        elif silent_install:
            if len(message.available_updates) == 0:
                return do_process_plugins([], message.new_recommended)

            # Ask the user's confirmation to update plugins
            title = _('Updates Available')
            if len(message.available_updates) == 1:
                text = _('An update is available for installation. ' \
                         'Do you want to install it?')
            else:
                text = _('%d updates are available for installation. ' \
                         'Do you want to install them?') % \
                         len(message.available_updates)
            icon = 'elisa.plugins.poblesec.plugin'
            buttons = [(_('Now'),
                        lambda: do_process_plugins(message.available_updates,
                                                   message.new_recommended)),
                       (_('Later'),
                        lambda: do_process_plugins([],
                                                   message.new_recommended))]
            return self.enqueue_popup(icon, title, text, buttons)

        if len(message.new_recommended) == 0:
            title = _('Updates Available')
            if len(message.available_updates) == 1:
                text = _('An update is available for installation. ' \
                         'Do you want to install it?')
            else:
                text = _('%d updates are available for installation. ' \
                         'Do you want to install them?') % \
                         len(message.available_updates)
        elif len(message.available_updates) == 0:
            title = _('New Recommended Plugins Available')
            if len(message.new_recommended) == 1:
                text = _('A new recommended plugin is available for ' \
                         'installation. Do you want to install it?')
            else:
                text = _('%d new recommended plugins are available for ' \
                         'installation. Do you want to install them?') % \
                         len(message.new_recommended)
        else:
            title = _('Updates and New Recommended Plugins Available')
            if len(message.available_updates) == 1 and \
                len(message.new_recommended) == 1:
                text = _('An update and a new recommended plugin are ' \
                 'available for installation. Do you want to install them?')
            elif len(message.available_updates) == 1:
                text = _('An update and %d new recommended plugins ' \
                         'are available for installation. Do you want to ' \
                         'install them?') % len(message.new_recommended)
            elif len(message.new_recommended) == 1:
                text = _('%d updates and a new recommended plugin ' \
                         'are available for installation. Do you want to ' \
                         'install them?') % len(message.available_updates)
            else:
                text = _('%d updates and %d new recommended plugins ' \
                         'are available for installation. Do you want to ' \
                         'install them?') % (len(message.available_updates), \
                                             len(message.new_recommended))

        icon = 'elisa.plugins.poblesec.plugin'
        buttons = [(_('Now'),
                    lambda: do_process_plugins(message.available_updates,
                                               message.new_recommended)),
                   (_('Later'), self.hide_popup)]
        self.enqueue_popup(icon, title, text, buttons)

    def set_frontend(self, frontend):
        super(PoblesecController, self).set_frontend(frontend)
        
        deferreds = []

        def browser_created(controller):
            self.browser = controller
            self.widget.add(self.browser.widget)
            self.browser.widget.width = 1.0
            self.browser.widget.height = 1.0
            self.browser.widget.x = 0.0
            self.browser.widget.y = 0.0
            self.focused_controller = controller

        dfr = self.frontend.create_controller('/poblesec/browser',
                                              home_path='/poblesec/sections_menu')
        dfr.addCallback(browser_created)
        deferreds.append(dfr)

        def video_player_created(controller):
            self.video_player = controller
            self.current_player = controller
            self.video_player.widget.layer = pgm.DRAWABLE_FAR
            self.widget.add(self.video_player.widget)
            self.video_player.widget.width = 1.0
            self.video_player.widget.height = 1.0
            self.video_player.widget.x = 0.0
            self.video_player.widget.y = 0.0
            self.video_player.widget.visible = True
            self._background_out.apply(self.video_player)

        dfr = self.frontend.create_controller('/poblesec/video_player')
        dfr.addCallback(video_player_created)
        deferreds.append(dfr)

        def dvd_player_created(controller):
            self.dvd_player = controller
            self.current_player = controller
            self.dvd_player.widget.layer = pgm.DRAWABLE_FAR
            self.widget.add(self.dvd_player.widget)
            self.dvd_player.widget.width = 1.0
            self.dvd_player.widget.height = 1.0
            self.dvd_player.widget.x = 0.0
            self.dvd_player.widget.y = 0.0
            self.dvd_player.widget.visible = True
            self._background_out.apply(self.dvd_player)


        # We don't want the dvd plugin to be a hard dependency for poblesec.
        try:
            dfr = self.frontend.create_controller('/poblesec/dvd_player')
            dfr.addCallback(dvd_player_created)
            deferreds.append(dfr)
        except ControllerNotFound:
            self.debug("Not loading /poblesec/dvd_player because its " +
                       "controller could not be found")
            self.dvd_player = None


        def music_player_created(controller):
            self.music_player = controller
            self.music_player.widget.layer = pgm.DRAWABLE_FAR
            self.widget.add(self.music_player.widget)
            self.music_player.widget.width = 1.0
            self.music_player.widget.height = 1.0
            self.music_player.widget.x = 0.0
            self.music_player.widget.y = 0.0
            self._background_out.apply(self.music_player)

        dfr = self.frontend.create_controller('/poblesec/music_player')
        dfr.addCallback(music_player_created)
        dfr.addCallback(self._initialize_dbus)
        deferreds.append(dfr)

        def slideshow_player_created(controller):
            self.slideshow_player = controller
            self.slideshow_player.widget.layer = pgm.DRAWABLE_FAR
            self.widget.add(self.slideshow_player.widget)
            self.slideshow_player.widget.width = 1.0
            self.slideshow_player.widget.height = 1.0
            self.slideshow_player.widget.x = 0.0
            self.slideshow_player.widget.y = 0.0

        dfr = self.frontend.create_controller('/poblesec/slideshow_player')
        dfr.addCallback(slideshow_player_created)
        deferreds.append(dfr)

        dfr = self._create_gst_metadata()
        deferreds.append(dfr)
        
        dfr_list = defer.DeferredList(deferreds)
        dfr_list.addCallback(self._process_command_line_files)

    def _process_command_line_files(self, result):
        files = common.application.options['files']
        if len(files) > 0:
            self.play_files(files)

    def _create_gst_metadata(self):
        def gst_metadata_created(gst_metadata):
            self.gst_metadata = gst_metadata

        dfr = GstMetadataAmpClient.create()
        dfr.addCallback(gst_metadata_created)
        return dfr

    def _initialize_dbus(self, result=None):
        if dbus is None:
            # no dbus support
            return

        bus = dbus.SessionBus()
        self.bus_name = dbus.service.BusName('com.fluendo.Elisa', bus)

        self.dbus_player = DBusPlayer(self, bus,
                '/com/fluendo/Elisa/Plugins/Poblesec/AudioPlayer', self.bus_name)
        
        from elisa.plugins.poblesec.dbus_player import FilePlayer
        self.dbus_fileplayer = FilePlayer(self, bus,
                '/com/fluendo/Elisa/Plugins/Poblesec/FilePlayer', self.bus_name)

    def clean(self):
        self._clean_dbus()
        bus = common.application.bus
        bus.unregister(self._plugin_updates_available)
        bus.unregister(self._frontend_loaded)
        bus.unregister(self._got_new_version)
        return super(MainController, self).clean()

    def _clean_dbus(self):
        if dbus is None:
            # no dbus support
            return

        bus = dbus.SessionBus()
        self.dbus_player.remove_from_connection(bus,
                '/com/fluendo/Elisa/Plugins/Poblesec/AudioPlayer')
        self.dbus_fileplayer.remove_from_connection(bus,
                '/com/fluendo/Elisa/Plugins/Poblesec/FilePlayer')
        # BusName implements __del__, eew
        del self.bus_name

        # remove the reference cycle
        self.dbus_player = None
        self.dbus_fileplayer = None

    def show_video_player(self):
        # stop slideshow player
        self.slideshow_player.player.stop()
        FadeOut().apply(self.slideshow_player)

        # stop music player
        self.music_player.player.stop()
        FadeOut().apply(self.music_player)

        # stop dvd player
        if self.dvd_player:
            self.dvd_player.player.stop()
            FadeOut().apply(self.dvd_player)

        # show video player
        self.current_player = self.video_player
        self.show_current_player()

    def show_dvd_player(self):
        if not self.dvd_player:
            return

        # stop slideshow player
        self.slideshow_player.player.stop()
        FadeOut().apply(self.slideshow_player)

        # stop music player
        self.music_player.player.stop()
        FadeOut().apply(self.music_player)

        # stop video player
        self.video_player.player.stop()
        FadeOut().apply(self.video_player)

        # show dvd player
        self.current_player = self.dvd_player
        self.show_current_player()

    def show_music_player(self):
        # hide slideshow player
        FadeOut().apply(self.slideshow_player)

        # stop video player
        self.video_player.player.stop()
        FadeOut().apply(self.video_player)

        # stop dvd player
        if self.dvd_player:
            self.dvd_player.player.stop()
            FadeOut().apply(self.dvd_player)

        # show music player
        self.current_player = self.music_player
        self.show_current_player()

    def show_slideshow_player(self):
        # hide music player
        FadeOut().apply(self.music_player)

        # stop video player
        self.video_player.player.stop()
        FadeOut().apply(self.video_player)

        # stop dvd player
        if self.dvd_player:
            self.dvd_player.player.stop()
            FadeOut().apply(self.dvd_player)

        # show slideshow player
        self.current_player = self.slideshow_player
        self.show_current_player()

    def show_current_player(self):
        self._overlay_out.apply(self.browser)
        self._background_in.apply(self.current_player)

        self.focused_controller = self.current_player
        self.focused_controller.widget.focus = True

    def hide_current_player(self):
        self._overlay_in.apply(self.browser)
        self._background_out.apply(self.current_player)
        self.focused_controller = self.browser
        self.focused_controller.widget.focus = True

    def enable_popups(self, enabled):
        """
        Enable or disable popups. When popups are disabled, they will not show
        up but can still be enqueued, and will start being dequeued when they
        are enabled again.
        
        @warning: disabling popups will not hide the current popup if there is
                  one.

        @param enabled: whether popups should be enabled
        @type enabled:  C{bool}
        """
        self._popups_enabled = enabled
        if enabled and len(self._popups_queue) > 0:
            self._show_popup()

    def enqueue_popup(self, icon, title, text, buttons):
        """
        Enqueue a popup for later display.
        If the popup can be displayed right away, it will be.

        @param icon:    the icon resource of the popup
        @type icon:     C{str}
        @param title:   the title of the popup
        @type title:    C{unicode}
        @param text:    the text of the popup
        @type text:     C{unicode}
        @param buttons: a list of buttons for the popup
        @type buttons:  C{list} of (C{str}, C{callable})
        """
        self._popups_queue.append((icon, title, text, buttons))
        if self._popups_enabled and not self._popup_visible:
            self._show_popup()

    def _show_popup(self):
        # Dequeue one popup and show it
        try:
            icon, title, text, buttons = self._popups_queue.pop(0)
        except IndexError:
            return defer.fail(AssertionError('No popup in queue.'))

        def popup_created(controller):
            self._popup_visible = True
            self.current_controller = self.focused_controller
            self.popup_controller = controller
            controller.widget.visible = False
            self.widget.add(controller.widget)
            controller.widget.x, controller.widget.y = (0.0, 0.0)
            controller.widget.width, controller.widget.height = (1.0, 1.0)
            controller.widget.regenerate()
            self._background_in.apply(controller)
            self.focused_controller = controller
            self.focused_controller.widget.focus = True
            return controller

        popup_dfr = self.frontend.create_controller('/poblesec/popup',
                                                    icon=icon, title=title,
                                                    text=text, buttons=buttons)
        popup_dfr.addCallback(popup_created)
        return popup_dfr

    def hide_popup(self):
        """
        Hide the current popup.

        @return: a deferred fired when the popup is hidden, or when the next
                 popup in the queue, if there is one, is shown
        @rtype:  L{elisa.core.utils.defer.Deferred}

        @raise AssertionError: if there is no popup currently visible
        """
        if not self._popup_visible:
            # No popup to hide.
            return defer.fail(AssertionError('No popup to hide.'))

        def faded(controller):
            self.focused_controller = self.current_controller
            self.focused_controller.widget.focus = True
            self.widget.remove(controller.widget)
            self._popup_visible = False
            return controller

        def show_next_popup(controller):
            if len(self._popups_queue) > 0:
                return self._show_popup()
            return controller

        hide_dfr = self._background_out.apply(self.popup_controller)
        hide_dfr.addCallback(faded)
        hide_dfr.addCallback(show_next_popup)
        return hide_dfr
    
    def stop_all_players(self):
        self.slideshow_player.player.stop()
        self.music_player.player.stop()
        self.video_player.player.stop()
        self.dvd_player.player.stop()

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_OK and \
                input_event.modifier == EventModifier.ALT:
            self.frontend.viewport.fullscreen = not self.frontend.viewport.fullscreen
            return True
        
        if self.focused_controller.handle_input(manager, input_event):
            return True
        
        if input_event.value == EventValue.KEY_ESCAPE:
            self.frontend.viewport.fullscreen = not self.frontend.viewport.fullscreen
            return True
        elif input_event.value == EventValue.KEY_F11:
            self.frontend.viewport.fullscreen = not self.frontend.viewport.fullscreen
            return True
        elif input_event.value == EventValue.KEY_f:
            self.frontend.viewport.fullscreen = not self.frontend.viewport.fullscreen
            return True
        elif input_event.value == EventValue.KEY_MENU:
            if self.focused_controller == self.browser:
                if self.current_player.player.status != \
                                            self.current_player.player.STOPPED:
                    self.show_current_player()
            else:
                self.hide_current_player()
            return True
        elif input_event.value == EventValue.KEY_SPACE and self.current_player:
            if self.current_player.player.status != \
                                            self.current_player.player.STOPPED:
                    self.current_player.toggle_play_pause()
            return True

        return super(PoblesecController, self).handle_input(manager, input_event)
    
    def get_playable_model(self, file_path):
        file_uri = MediaUri("file://%s" % file_path)
        playable_model = PlayableModel()
        playable_model.uri = file_uri
        playable_model.title = os.path.basename(file_path)
        return playable_model
    
    def get_image_model(self, file_path):
        file_uri = MediaUri("file://%s" % file_path)
        image = ImageModel()
        image.references.append(file_uri)
        image.title = file_path
        return image
    
    def play_files(self, files):
        #import pdb; pdb.set_trace()
        try:
            file_path = files[0]
        except IndexError, e:
            return
        try:
            file_uri = MediaUri("file://%s" % file_path)
            media_type = typefinding.typefind(file_uri)
        except typefinding.UnknownFileExtension, e:
            self.warning("Can't retrieve file type: %s" % file_path)
            return
         
        playable_model = PlayableModel()
        playable_model.uri = file_uri
        playable_model.title = os.path.basename(file_path)
            
        if "video" in media_type:
            self.video_player.player.play_model(playable_model)
            self.show_video_player()
            for file in files[1:]:
                playable_model = self.get_playable_model(file)
                self.video_player.player.enqueue_to_playlist(playable_model)
        elif "audio" in media_type:
            self.music_player.player.play_model(playable_model)
            self.show_music_player()
            for file in files[1:]:
                playable_model = self.get_playable_model(file)
                self.music_player.player.enqueue_to_playlist(playable_model)
        elif "image" in media_type:
            self.slideshow_player.player.clear_playlist()
            image = ImageModel()
            image.references.append(file_uri)
            image.title = file_path
            self.slideshow_player.player.enqueue_to_playlist(image)
            for file in files[1:]:
                image = self.get_image_model(file)
                self.slideshow_player.player.enqueue_to_playlist(image)
            self.show_slideshow_player()
        else:
            self.warning("Unknown media type: %s" % file_path)
