# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


import gobject
import weakref
import copy


class Style(gobject.GObject):
    """
    A very simple and dynamic style object.

    Just a set of properties (object attributes) and a reference to the widget
    owning it. When changed, a property emit the 'property-changed' signal.

    @ivar widget: the widget owning the style
    @type widget: L{elisa.plugins.pigment.widgets.Widget}
    """

    __gsignals__ = {'property-changed': (gobject.SIGNAL_RUN_LAST,
                                         gobject.TYPE_BOOLEAN,
                                         (gobject.TYPE_STRING,
                                          gobject.TYPE_PYOBJECT))}

    def __init__(self, **kwargs):
        """
        Initialize the style object with the given properties

        @param kwargs: the properties dictionary
        @type kwargs: dictionary
        """

        self.widget = None
        self._properties = {}

        self._initializing = True

        super(Style, self).__init__()

        for key, value in kwargs.iteritems():
            setattr(self, key, value)

        self._initializing = False

    def __setattr__(self, key, value):
        """
        Set the property's value, emitting a 'property-changed' signal

        @param key: the property to set
        @type key: the string
        @param value: the property's value
        @type value: any
        """
        old_key = key
        key = key.replace('-', '_')
        gobject.GObject.__setattr__(self, key, value)

        if key[0] != '_' and key != 'widget':
            self._properties[old_key] = value
            if not self._initializing:
                self.emit("property-changed", old_key, value)

    def widget__set(self, value):
        if value is not None:
            self._widget = weakref.proxy(value)
        else:
            self._widget = None

    def widget__get(self):
        return self._widget

    widget = property(widget__get, widget__set)

    def update(self, other):
        """Merge in-place the properties of another style."""
        self.merge(other, inplace=True)

    def merge(self, other, inplace=True):
        """
        Merge the properties of another style and returns a new one.

        The new style will have all the properties of the current style, with
        replaced values from the second, plus further properties coming from the
        other style.

        @param other:   the style to merge
        @type other:    L{elisa.plugins.pigment.widgets.Style}
        @param inplace: whether to build another style, or update the current
                        one
        @type inplace:  C{bool}

        @returns:       the new style
        @rtype:         L{elisa.plugins.pigment.widgets.Style}
        """
        if inplace:
            new = self
        else:
            new = Style(**self._properties)

        for key, value in other:
            setattr(new, key, value)

        return new

    def __iter__(self):
        return self._properties.iteritems()

    def get_properties(self):
        """
        The list of set properties.

        @returns: the list of set properties
        @rtype: list of strings
        """
        return self._properties.keys()

    def get_items(self):
        """
        The dictionary of properties => property values.

        @returns: the dictionary of properties
        @rtype: dict
        """
        return self._properties

    def __repr__(self):
        return "Style: " + str(self._properties)

    def __deepcopy__(self, memo):
        new_style = Style()
        new_style.widget = self.widget
        new_style._properties = copy.deepcopy(self._properties, memo)
        return new_style


if __name__ == '__main__':
    style = Style()

