# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>


from elisa.core.common import application
from elisa.core.resource_manager import NoMatchingResourceProvider

from elisa.core.media_uri import MediaUri
from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.widgets.menu_item import MenuItemWidget

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.database.music_controller import GenericAlbumsDbController, \
                                                    GenericArtistsController, \
                                                    GenericTracksDbController

from elisa.plugins.database.music_controller import ArtistsDbViewMode, \
                                                    AlbumsDbViewMode, \
                                                    DBTracksViewMode

from elisa.plugins.database.photo_controller import GenericPhotoAlbumsController, \
                                                    GenericPhotosController

from elisa.plugins.database.photo_controller import PhotoViewMode, \
                                                    PhotoAlbumViewMode
                                                    
from elisa.plugins.database.video_controller import VideoViewMode, \
                                                    VideoController

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.section import SectionMenuController

from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.database.models import *
from elisa.plugins.favorites.models import *

from elisa.plugins.shoutcast.models import ShoutcastRadioStationModel

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.shoutcast.controller import GenericShoutcastController
from elisa.plugins.shoutcast.controller import ShoutcastViewMode

from twisted.internet import defer

from storm.locals import Select

from elisa.core.utils.i18n import install_translation

_ = install_translation('favorites')


def music_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/music/favorites'
    link.controller_args = {'section' : '/music'}
    link.label = _('Favorites')
    link.icon = 'elisa.plugins.favorites.icon'
    controller.model.append(link)

    return defer.succeed(None)

def music_internet_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/music/internet/favorites'
    link.label = _('Radios')
    link.icon = 'elisa.plugins.poblesec.radio'
    controller.model.append(link)

    return defer.succeed(None)

def pictures_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/pictures/favorites'
    link.controller_args = {'section' : '/pictures'}
    link.label = _('Favorites')
    link.icon = 'elisa.plugins.favorites.icon'
    controller.model.append(link)

    return defer.succeed(None)
    
def videos_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/videos/favorites/videos'
    link.controller_args = {} 
    link.label = _('Favorites')
    link.icon = 'elisa.plugins.favorites.icon'
    controller.model.append(link)

    return defer.succeed(None)


class FavoritesController(SectionMenuController):

    def initialize(self, section=None):
        self.section = section

        dfr = super(FavoritesController, self).initialize()

        if section == '/music':
            link = Link()
            link.controller_path = '/poblesec/music/favorites/artists'
            link.controller_args = {}
            link.label = _('Artists')
            link.icon = 'elisa.plugins.poblesec.by_artist'

            self.model.append(link)

            link = Link()
            link.controller_path = '/poblesec/music/favorites/albums'
            link.controller_args = {}
            link.label = _('Albums')
            link.icon = 'elisa.plugins.poblesec.by_album'

            self.model.append(link)

            link = Link()
            link.controller_path = '/poblesec/music/favorites/tracks'
            link.controller_args = {}
            link.label = _('Tracks')
            link.icon = 'elisa.plugins.poblesec.by_track'

            self.model.append(link)

        elif section == '/pictures':
            link = Link()
            link.controller_path = '/poblesec/pictures/favorites/albums'
            link.controller_args = {}
            link.label = _('Albums')
            link.icon = 'elisa.plugins.poblesec.photo_album'

            self.model.append(link)

            link = Link()
            link.controller_path = '/poblesec/pictures/favorites/photos'
            link.controller_args = {}
            link.label = _('Photos')
            link.icon = 'elisa.plugins.poblesec.photo'

            self.model.append(link)
        elif section == '/videos':
            link = Link()
            link.controller_path = '/poblesec/videos/favorites/videos'
            link.controller_args = {}
            link.label = _('Videos')
            link.icon = 'elisa.plugins.poblesec.file_video'

            self.model.append(link)

        return dfr


# This function is meant to be connected to the "focus" signal emitted from the
# main widget of a controller, in order to trigger a refresh of its model -
# except the first time the controller is put on the screen.
# It is a hack to deal with the dumb history navigation system we have in place
# at present.
def focus_refresh_cb(widget, focus, controller):
    if focus:
        if getattr(controller, '_first_initialize', True):
            controller._first_initialize = False
        else:
            controller.refresh()


class FavoritesAlbumsController(GenericAlbumsDbController):

    empty_label = _('There are no favorite albums')
    empty_icon = 'elisa.plugins.poblesec.by_album'

    def initialize(self, artist=None):
        self.artist = artist

        dfr = super(FavoritesAlbumsController, self).initialize()
        dfr.addCallback(self.load_data)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result):

        def get_db_albums(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'MusicAlbum')
            dfr = application.store.find(MusicAlbum, MusicAlbum.name.is_in(subselect))
            dfr.addCallback(lambda rs: rs.all())
            return dfr

        def got_db_albums(albums):
            self.model.extend(albums)
            return self

        def get_other_albums(result):
            dfr = application.store.find(FavoritesItem,
                                         FavoritesItem.type == u'music_album',
                                         FavoritesItem.uri != None)
            dfr.addCallback(lambda rs: rs.all())
            return dfr

        def got_other_albums(items):
            # FIXME: activate me and reimplement insert() when the model itself
            # implements sorted insertion
            # Asynchronously inserting items in the model: it isn't necessary to
            # wait for completion
            for item in items:
                try:
                    model, dfr = application.resource_manager.get(MediaUri(item.uri))
                except NoMatchingResourceProvider:
                    return

                def callback(result, i=item):
                    insert(result, i.title)

                dfr.addCallback(callback)

        def got_other_albums_waiting(items):
            dfrs = []
            for item in items:
                try:
                    model, dfr = application.resource_manager.get(MediaUri(item.uri))
                except NoMatchingResourceProvider:
                    return

                def callback(result, i=item):
                    insert(result, i.title)
                dfr.addCallback(callback)
                dfrs.append(dfr)

            return defer.DeferredList(dfrs)

        def insert(item, title):
            self.model.append(item)

        def sort_albums(results):
            def key(item):
                if isinstance(item, Action):
                    return ''
                return item.name.lower()

            self.model.sort(key=key)
            return self

        dfr = get_db_albums(None)
        dfr.addCallback(got_db_albums)
        dfr.addCallback(get_other_albums)
        dfr.addCallback(got_other_albums_waiting)
        dfr.addCallback(sort_albums)

        return dfr


class FavoritesArtistsController(GenericArtistsController):

    empty_label = _('There are no favorite artists')
    empty_icon = 'elisa.plugins.poblesec.by_artist'

    def initialize(self):
        dfr = super(FavoritesArtistsController, self).initialize()
        dfr.addCallback(self.load_data)
        
        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result):

        def get_db_artists(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'Artist')
            return application.store.find(Artist, Artist.name.is_in(subselect))

        def got_db_artists(artists):
            self.model.extend(artists)
            return self

        def sort_db_artists(result_set):
            result_set.order_by(Artist.name)
            return result_set.all()

        def get_other_artists(result):
            dfr = application.store.find(FavoritesItem,
                                         FavoritesItem.type == u'artist',
                                         FavoritesItem.uri != None)
            dfr.addCallback(lambda rs: rs.all())
            return dfr

        def got_other_artists(items):
            # FIXME: activate me and reimplement insert() when the model itself
            # implements sorted insertion
            # Asynchronously inserting items in the model: it isn't necessary to
            # wait for completion
            for item in items:
                try:
                    model, dfr = application.resource_manager.get(MediaUri(item.uri))
                except NoMatchingResourceProvider:
                    return

                def callback(result, i=item):
                    insert(result)
                dfr.addCallback(callback)

        def got_other_artists_waiting(items):
            dfrs = []
            for item in items:
                try:
                    model, dfr = application.resource_manager.get(MediaUri(item.uri))
                except NoMatchingResourceProvider:
                    return

                def callback(result, i=item):
                    insert(result)
                dfr.addCallback(callback)

                dfrs.append(dfr)

            return defer.DeferredList(dfrs)

        def insert(item):
            self.model.append(item)

        def sort_artists(results):
            def key(item):
                if isinstance(item, Action):
                    return ''
                return (item.name or '').lower()

            self.model.sort(key=key)
            return self

        dfr = get_db_artists(None)
        dfr.addCallback(sort_db_artists)
        dfr.addCallback(got_db_artists)
        dfr.addCallback(get_other_artists)
        dfr.addCallback(got_other_artists_waiting)
        dfr.addCallback(sort_artists)

        return dfr


class FavoritesTracksController(GenericTracksDbController):

    empty_label = _('There are no favorite tracks')
    empty_icon = 'elisa.plugins.poblesec.file_music'

    def initialize(self, album=None, track=None, tracks=None):
        self.album = album
        self.track = track
        self.tracks = tracks

        dfr = super(FavoritesTracksController, self).initialize()
        dfr.addCallback(self.load_data)
        
        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result):
        def get_db_tracks(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'MusicTrack')
            return application.store.find(MusicTrack,
                                          MusicTrack.file_path.is_in(subselect),
                                          MusicTrack.file_path == File.path,
                                          File.hidden == False)

        def got_db_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_db_tracks_title(result_set):
            result_set.order_by(MusicTrack.title)
            return result_set.all()

        def get_other_tracks(result):
            dfr = application.store.find(FavoritesItem,
                                         FavoritesItem.type == u'music_track',
                                         FavoritesItem.uri != None)
            dfr.addCallback(lambda rs: rs.all())
            return dfr

        def got_other_tracks(items):
            # FIXME: activate me and reimplement insert() when the model itself
            # implements sorted insertion
            # Asynchronously inserting items in the model: it isn't necessary to
            # wait for completion
            for item in items:
                try:
                    model, dfr = application.resource_manager.get(MediaUri(item.uri))
                except NoMatchingResourceProvider:
                    return

                def callback(result, i=item):
                    insert(result)
                dfr.addCallback(callback)

        def got_other_tracks_waiting(items):
            dfrs = []
            for item in items:
                try:
                    model, dfr = application.resource_manager.get(MediaUri(item.uri))
                except NoMatchingResourceProvider:
                    return

                def callback(result, i=item):
                    insert(result)
                dfr.addCallback(callback)

                dfrs.append(dfr)

            return defer.DeferredList(dfrs)

        def insert(item):
            self.model.append(item)

        def sort_tracks(results):
            def key(item):
                if isinstance(item, Action):
                    return ''
                return (item.title or '').lower()

            self.model.sort(key=key)
            return self

        if self.track is None:
            dfr = get_db_tracks(None)
            dfr.addCallback(sort_db_tracks_title)
            dfr.addCallback(got_db_tracks)
            dfr.addCallback(get_other_tracks)
            dfr.addCallback(got_other_tracks_waiting)
            dfr.addCallback(sort_tracks)

            return dfr


class FavoritesRadiosController(GenericShoutcastController):

    empty_label = _('There are no favorite radios')
    empty_icon = 'elisa.plugins.poblesec.radio'

    def initialize(self, uri=None, station=None):
        dfr = super(FavoritesRadiosController, self).initialize(uri=uri, station=station)
        self.station = station

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        dfr.addCallback(self.load_data)
        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result):

        if self.station is None:
            def get_radios(result):
                dfr = application.store.find(FavoritesItem,
                                             FavoritesItem.foreign_class == u'InternetRadio')
                return dfr

            def got_radios(radios):
                shoutcast_radios = []
                for radio in radios:
                    shoutcast_radio = ShoutcastRadioStationModel()
                    shoutcast_radio.name = radio.title
                    shoutcast_radio.get_playable = MediaUri(radio.uri)
                    shoutcast_radios.append(shoutcast_radio)

                self.model.extend(shoutcast_radios)
                return self

            def sort_radios(result_set):
                result_set.order_by(FavoritesItem.title)
                return result_set.all()

            dfr = get_radios(None)
            dfr.addCallback(sort_radios)
            dfr.addCallback(got_radios)

            return dfr


class FavoritesAlbumsVerticalWithPreview(FavoritesAlbumsController, DoubleLineMenuItemPreviewListController):
    view_mode = AlbumsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name


class FavoritesAlbumsCoverflow(FavoritesAlbumsController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode


class FavoritesAlbumsGrid(FavoritesAlbumsController, GridItemGridController):
    view_mode = AlbumsDbViewMode


class FavoritesAlbumsListSwitcher(ListSwitcherController):
    modes = [FavoritesAlbumsVerticalWithPreview, FavoritesAlbumsCoverflow, FavoritesAlbumsGrid]
    default_mode = FavoritesAlbumsVerticalWithPreview


class FavoritesArtistsVerticalWithPreview(FavoritesArtistsController, MenuItemPreviewListController):
    view_mode = ArtistsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name


class FavoritesArtistsCoverflow(FavoritesArtistsController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name


class FavoritesArtistsGrid(FavoritesArtistsController, GridItemGridController):
    view_mode = ArtistsDbViewMode


class FavoritesArtistsListSwitcher(ListSwitcherController):
    modes = [FavoritesArtistsVerticalWithPreview, FavoritesArtistsCoverflow, FavoritesArtistsGrid]
    default_mode = FavoritesArtistsVerticalWithPreview


class FavoritesTracksVerticalWithPreview(FavoritesTracksController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.title

class FavoritesTracksCoverflow(FavoritesTracksController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksViewMode


class FavoritesTracksGrid(FavoritesTracksController, GridItemGridController):
    view_mode = DBTracksViewMode


class FavoritesTracksListSwitcher(ListSwitcherController):
    modes = [FavoritesTracksVerticalWithPreview, FavoritesTracksCoverflow, FavoritesTracksGrid]
    default_mode = FavoritesTracksVerticalWithPreview


class FavoritesRadiosVerticalWithPreview(FavoritesRadiosController, MenuItemPreviewListController):
    view_mode = ShoutcastViewMode

class FavoritesRadiosCoverflow(FavoritesRadiosController, ImageWithReflectionCoverflowController):
    view_mode = ShoutcastViewMode


class FavoritesRadiosGrid(FavoritesRadiosController, GridItemGridController):
    view_mode = ShoutcastViewMode


class FavoritesRadiosListSwitcher(ListSwitcherController):
    modes = [FavoritesRadiosVerticalWithPreview, FavoritesRadiosCoverflow, FavoritesRadiosGrid]
    default_mode = FavoritesRadiosVerticalWithPreview


class FavoritesPhotosController(GenericPhotosController):

    empty_label = _('There are no favorite photos')
    empty_icon = 'elisa.plugins.poblesec.photo'

    def initialize(self, album=None, photo=None, photos=None):
        dfr = super(FavoritesPhotosController, self).initialize(album=album, photo=photo, photos=photos)
        dfr.addCallback(self.load_data, album, photo, photos)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result, album=None, photo=None, photos=None):

        def get_photos(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'Image')
            return application.store.find(Image, Image.file_path.is_in(subselect),
                                          Image.file_path == File.path,
                                          File.hidden == False)

        def got_photos(photos):
            self.model.extend(photos)
            return self

        def sort_photos(result_set):
            result_set.order_by(Image.shot_time)
            return result_set.all()

        if photo is None:
            dfr = get_photos(None)
            dfr.addCallback(sort_photos)
            dfr.addCallback(got_photos)
            return dfr


class FavoritesPhotoAlbumsController(GenericPhotoAlbumsController):

    empty_label = _('There are no favorite albums')
    empty_icon = 'elisa.plugins.poblesec.photo_album'

    def initialize(self):
        dfr = super(FavoritesPhotoAlbumsController, self).initialize()
        dfr.addCallback(self.load_data)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result):

        def get_albums(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'PhotoAlbum')
            return application.store.find(PhotoAlbum, PhotoAlbum.name.is_in(subselect))

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_albums_title(result_set):
            result_set.order_by(PhotoAlbum.name)
            return result_set.all()

        dfr = get_albums(None)
        dfr.addCallback(sort_albums_title)
        dfr.addCallback(got_albums)
        return dfr


class FavoritesPhotosVerticalWithPreview(FavoritesPhotosController, MenuItemPreviewListController):
    view_mode = PhotoViewMode


class FavoritesPhotosCoverflow(FavoritesPhotosController, ImageWithReflectionCoverflowController):
    view_mode = PhotoViewMode


class FavoritesPhotosGrid(FavoritesPhotosController, GridItemGridController):
    view_mode = PhotoViewMode


class FavoritesPhotosListSwitcher(ListSwitcherController):
    modes = [FavoritesPhotosVerticalWithPreview, FavoritesPhotosCoverflow, FavoritesPhotosGrid]
    default_mode = FavoritesPhotosVerticalWithPreview


class FavoritesPhotoAlbumsVerticalWithPreview(FavoritesPhotoAlbumsController, MenuItemPreviewListController):
    view_mode = PhotoAlbumViewMode


class FavoritesPhotoAlbumsCoverflow(FavoritesPhotoAlbumsController, ImageWithReflectionCoverflowController):
    view_mode = PhotoAlbumViewMode


class FavoritesPhotoAlbumsGrid(FavoritesPhotoAlbumsController, GridItemGridController):
    view_mode = PhotoAlbumViewMode


class FavoritesPhotoAlbumsListSwitcher(ListSwitcherController):
    modes = [FavoritesPhotoAlbumsVerticalWithPreview, FavoritesPhotoAlbumsCoverflow, FavoritesPhotoAlbumsGrid]
    default_mode = FavoritesPhotoAlbumsVerticalWithPreview

    
class FavoritesVideosController(VideoController):

    empty_label = _('There are no favorite videos')
    empty_icon = 'elisa.plugins.poblesec.file_video'

    def initialize(self):
        dfr = super(FavoritesVideosController, self).initialize()
        dfr.addCallback(self.load_data)

        if not getattr(self, '_focus_refresh_cb_handle', None):
            self._focus_refresh_cb_handle = self.widget.connect('focus', focus_refresh_cb, self)

        return dfr

    def refresh(self):
        self.model[:] = []
        self.actions[:] = self.actions
        self.load_data(None)

    def load_data(self, result):
        def get_videos(result):
            subselect = Select(FavoritesItem.foreign_id,
                               FavoritesItem.foreign_class == u'Video')
            return application.store.find(Video, Video.file_path.is_in(subselect),
                                          Video.file_path == File.path,
                                          File.hidden == False)
                                          
        def got_videos(videos):
            self.model.extend(videos)
            return self

        def sort_videos_title(result_set):
            result_set.order_by(Video.name)
            return result_set.all()

        dfr = get_videos(None)
        dfr.addCallback(sort_videos_title)
        dfr.addCallback(got_videos)

        return dfr


class FavoritesVideosVerticalWithPreview(FavoritesVideosController, MenuItemPreviewListController):
    view_mode = VideoViewMode


class FavoritesVideosCoverflow(FavoritesVideosController, ImageWithReflectionCoverflowController):
    view_mode = VideoViewMode


class FavoritesVideosGrid(FavoritesVideosController, GridItemGridController):
    view_mode = VideoViewMode


class FavoritesVideosListSwitcher(ListSwitcherController):
    modes = [FavoritesVideosVerticalWithPreview, FavoritesVideosCoverflow, FavoritesVideosGrid]
    default_mode = FavoritesVideosVerticalWithPreview


