/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.elasticsearch.gateway.local;

import org.apache.lucene.util.LuceneTestCase;
import org.elasticsearch.Version;
import org.elasticsearch.action.admin.indices.recovery.RecoveryResponse;
import org.elasticsearch.action.admin.indices.recovery.ShardRecoveryResponse;
import org.elasticsearch.action.count.CountResponse;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.routing.allocation.allocator.BalancedShardsAllocator;
import org.elasticsearch.cluster.routing.allocation.decider.EnableAllocationDecider;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.indices.flush.IndicesSyncedFlushResult;
import org.elasticsearch.indices.flush.SyncedFlushUtil;
import org.elasticsearch.indices.recovery.RecoveryState;
import org.elasticsearch.test.ElasticsearchBackwardsCompatIntegrationTest;
import org.elasticsearch.test.ElasticsearchIntegrationTest;
import org.junit.Test;

import java.util.HashMap;

import static org.elasticsearch.test.hamcrest.ElasticsearchAssertions.assertAcked;
import static org.elasticsearch.test.hamcrest.ElasticsearchAssertions.assertHitCount;
import static org.hamcrest.Matchers.*;

@ElasticsearchIntegrationTest.ClusterScope(numDataNodes = 0, scope = ElasticsearchIntegrationTest.Scope.TEST, numClientNodes = 0, transportClientRatio = 0.0)
public class RecoveryBackwardsCompatibilityTests extends ElasticsearchBackwardsCompatIntegrationTest {


    protected Settings nodeSettings(int nodeOrdinal) {
        return ImmutableSettings.builder()
                .put(super.nodeSettings(nodeOrdinal))
                .put("action.admin.cluster.node.shutdown.delay", "10ms")
                .put("gateway.recover_after_nodes", 2)
                .put(BalancedShardsAllocator.SETTING_THRESHOLD, 100.0f).build(); // use less aggressive settings
    }

    protected int minExternalNodes() {
        return 2;
    }

    protected int maxExternalNodes() {
        return 3;
    }


    @Test
    @LuceneTestCase.Slow
    public void testReusePeerRecovery() throws Exception {
        assertAcked(prepareCreate("test").setSettings(ImmutableSettings.builder().put(indexSettings()).put(IndexMetaData.SETTING_NUMBER_OF_REPLICAS, 0)));
        logger.info("--> indexing docs");
        int numDocs = scaledRandomIntBetween(100, 1000);
        IndexRequestBuilder[] builders = new IndexRequestBuilder[numDocs];
        for (int i = 0; i < builders.length; i++) {
            builders[i] = client().prepareIndex("test", "type").setSource("field", "value");
        }
        indexRandom(true, builders);
        ensureGreen();
        if (randomBoolean()) { // just make sure it doesn't break anything - we seal before we actually bump replicas
            logger.info("--> trying to sync flush");
            // we should not try a synced flush if we do not have at least one node with current version in it
            // because old versions might not have an instance of SyncedFlushService (added in 1.6)
            if (backwardsCluster().numNewDataNodes() > 0) {
                // we try and sync flush but do not check the result as some of the shards might fail if they are on an older version
                SyncedFlushUtil.attemptSyncedFlush(backwardsCluster().internalCluster(), "test");
            }
        }
        logger.info("--> bump number of replicas from 0 to 1");
        client().admin().indices().prepareOptimize("test").setMaxNumSegments(100).get(); // just wait for merges
        client().admin().indices().prepareFlush().setWaitIfOngoing(true).setForce(true).get();
        client().admin().indices().prepareUpdateSettings("test").setSettings(ImmutableSettings.builder().put(IndexMetaData.SETTING_NUMBER_OF_REPLICAS, "1").build()).get();
        ensureGreen();

        assertAllShardsOnNodes("test", backwardsCluster().backwardsNodePattern());

        logger.info("--> upgrade cluster");
        logClusterState();
        CountResponse countResponse = client().prepareCount().get();
        assertHitCount(countResponse, numDocs);

        client().admin().cluster().prepareUpdateSettings().setTransientSettings(ImmutableSettings.settingsBuilder().put(EnableAllocationDecider.CLUSTER_ROUTING_ALLOCATION_ENABLE, "none")).execute().actionGet();
        backwardsCluster().upgradeAllNodes();
        client().admin().cluster().prepareUpdateSettings().setTransientSettings(ImmutableSettings.settingsBuilder().put(EnableAllocationDecider.CLUSTER_ROUTING_ALLOCATION_ENABLE, "all")).execute().actionGet();
        ensureGreen();

        countResponse = client().prepareCount().get();
        assertHitCount(countResponse, numDocs);

        RecoveryResponse recoveryResponse = client().admin().indices().prepareRecoveries("test").setDetailed(true).get();
        HashMap<String, String> map = new HashMap<>();
        map.put("details", "true");
        final ToXContent.Params params = new ToXContent.MapParams(map);
        for (ShardRecoveryResponse response : recoveryResponse.shardResponses().get("test")) {
            RecoveryState recoveryState = response.recoveryState();

            final String recoverStateAsJSON = XContentHelper.toString(recoveryState, params);
            if (!recoveryState.getPrimary()) {
                RecoveryState.Index index = recoveryState.getIndex();
                if (compatibilityVersion().onOrAfter(Version.V_1_2_0)) {
                    assertThat(recoverStateAsJSON, index.recoveredBytes(), equalTo(0l));
                    assertThat(recoverStateAsJSON, index.reusedBytes(), greaterThan(0l));
                    assertThat(recoverStateAsJSON, index.reusedBytes(), equalTo(index.totalBytes()));
                    assertThat(recoverStateAsJSON, index.recoveredFileCount(), equalTo(0));
                    assertThat(recoverStateAsJSON, index.reusedFileCount(), equalTo(index.totalFileCount()));
                    assertThat(recoverStateAsJSON, index.reusedFileCount(), greaterThan(0));
                    assertThat(recoverStateAsJSON, index.recoveredBytesPercent(), equalTo(100.f));
                    assertThat(recoverStateAsJSON, index.recoveredFilesPercent(), equalTo(100.f));
                    assertThat(recoverStateAsJSON, index.reusedBytes(), greaterThan(index.recoveredBytes()));
                } else {
                    /* We added checksums on 1.3 but they were available on 1.2 already since this uses Lucene 4.8.
                     * yet in this test we upgrade the entire cluster and therefor the 1.3 nodes try to read the checksum
                     * from the files even if they haven't been written with ES 1.3. Due to that we don't have to recover
                     * the segments files if we are on 1.2 or above...*/
                    assertThat(recoverStateAsJSON, index.recoveredBytes(), greaterThan(0l));
                    assertThat(recoverStateAsJSON, index.recoveredFileCount(), greaterThan(0));
                    assertThat(recoverStateAsJSON, index.reusedBytes(), greaterThan(0l));
                    assertThat(recoverStateAsJSON, index.recoveredBytesPercent(), greaterThan(0.0f));
                    assertThat(recoverStateAsJSON, index.recoveredBytesPercent(), equalTo(100.f));
                    assertThat(recoverStateAsJSON, index.recoveredFilesPercent(), equalTo(100.f));
                    assertThat(recoverStateAsJSON, index.reusedBytes(), greaterThan(index.recoveredBytes()));
                    assertThat(recoverStateAsJSON, index.recoveredBytes(), lessThan(index.totalBytes()));
                }
                // TODO upgrade via optimize?
            }
        }
    }
}
