/* Copyright (C) 2005, 2006 and 2009 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#ifndef CGU_IO_WATCH_H
#define CGU_IO_WATCH_H

/**
 */

/**
 * @defgroup io_watch io_watch
 *
 * \#include <c++-gtk-utils/io_watch.h>
 *
 * The start_iowatch() function connects a Unix file descriptor to an
 * event loop owned by a GMainContext object (normally the main
 * program loop).  It both saves the overhead of having to construct a
 * GIOChannel object where the only thing wanted is to execute a
 * callback when there is something to be read from a pipe, fifo or
 * socket or a pipe or fifo can be written to, and it also provides
 * for automatic disconnection when an object whose function the
 * callback represents is destroyed.
 *
 * As this method has been written, the functions should not check for
 * both input and output (in other words pass G_IO_IN or G_IO_OUT to
 * the GIOCondition argument in start_iowatch(), but not a
 * bitwise-ORed version of both).  If the same descriptor is used for
 * both then there should be separate watches for reading and writing.
 *
 * G_IO_IN can be bitwise-ORed with G_IO_HUP, and should be if the
 * callback has the task of cleaning up if EOF is reached (see
 * http://www.greenend.org.uk/rjk/2001/06/poll.html ), which is
 * detected by read() returning 0.  A cast will be required to do this
 * for the third argument of start_iowatch() (that is, pass
 * GIOCondition(G_IO_IN | G_IO_HUP)).
 *
 * start_iowatch() is thread-safe (it may be called in any thread)
 * provided that the glib main loop has been made thread-safe by a
 * call to g_thread_init().
 *
 * start_iowatch() takes ownership of the passed Callback object.  The
 * function comes in two versions.  The one which takes a
 * Callback::Releaser object as its third argument provides for
 * automatic ceasing of the watch if the target object which has the
 * Releaser as a member is destroyed.  (Note that for this to be race
 * free, the lifetime of the remote target object whose method is to be
 * invoked must be determined by the thread to whose main loop the
 * watch has been attached.  When the main loop begins invoking the
 * execution of the watch callback, the remote object must either
 * wholly exist, in which case the callback will be invoked, or have
 * been destroyed, in which case the callback will be ignored, and not
 * be in some transient half-state governed by another thread.)
 *
 * The connected function encapsulated by the callback passed to
 * start_iowatch() and executed by the iowatch should take a single
 * bool& argument, or have its last argument as a bool& unbound
 * argument (with all other arguments bound in the callback).  If that
 * bool& argument is set by the connected function to false, say
 * because end-of-file has been reached, then the watch will be ended
 * and all resources connected with it deleted without further user
 * action being required (there is no need for the connected function
 * to set it to true if the watch is to continue, as that is the
 * default).  In addition, the watch will be ended automatically and
 * resources deleted if (i) as mentioned above, the callback passed to
 * start_iowatch() is protected by a Releaser object and the target
 * object whose method is encapsulated by the callback is destroyed,
 * or (ii) g_source_remove() is called on the source id returned by
 * start_iowatch() (where the watch is attached to the default main
 * context) or g_source_destroy() is called on the GSource object
 * obtained from that id with g_main_context_find_source_by_id()
 * (where the watch has been attached to a non-default main context).
 * If the source has been removed automatically by virtue of the bool&
 * argument being set to false or by virtue of a Releaser object
 * releasing, g_source_remove() or g_source_destroy() should not
 * afterwards be called in respect of the id value returned by
 * start_iowatch() in case it has been reused by the main context
 * concerned in the meantime.
 */

#include <glib.h>
#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

class Releaser;

/**
 * Starts an io watch in the glib main loop on a file descriptor, and
 * executes the callback if the condition in io_condition is met.  It
 * is thread-safe (it may be called in any thread) provided that the
 * glib main loop has been made thread-safe by a call to
 * g_thread_init().
 * @param fd The file descriptor.
 * @param cb The callback object.  Ownership is taken of this object,
 * and it will be deleted when it has been finished with
 * @param io_condition The condition to be watched for (G_IO_IN may be
 * bitwise-ored with G_IO_HUP).
 * @param priority The priority to be given to the watch in the main
 * loop.  In ascending order of priorities, priorities are
 * G_PRIORITY_LOW, G_PRIORITY_DEFAULT_IDLE, G_PRIORITY_HIGH_IDLE,
 * G_PRIORITY_DEFAULT and G_PRIORITY_HIGH. The default is
 * G_PRIORITY_DEFAULT.  This determines the order in which the
 * callback will appear in the event list in the main loop, not the
 * priority which the OS will adopt
 * @param context The glib main context to which the watch is to be
 * attached (the default of NULL will cause the watch to be attached
 * to the main program loop, and this is almost always what is
 * wanted).
 * @return The glib source id of the watch.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  If it does
 * so, the CallbackArg object will be disposed of.
 * @note Cancellation of the thread to which the watch is attached is
 * blocked during execution of the callback.
 * @ingroup io_watch
 * 
 * Since 0.9.2 choice of priority available.
 */
guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb,
		    GIOCondition io_condition, gint priority = G_PRIORITY_DEFAULT,
		    GMainContext* context = 0);

/**
 * Starts an io watch in the glib main loop on a file descriptor, and
 * executes the callback if the condition in io_condition is met.
 * This version provides for automatic watch disconnection when the
 * object whose function the callback represents is destroyed, via the
 * Releaser object.  It is thread-safe (it may be called in any
 * thread) provided that the glib main loop has been made thread-safe
 * by a call to g_thread_init().
 * @param fd The file descriptor.
 * @param cb The callback object.  Ownership is taken of this object,
 * and it will be deleted when it has been finished with.
 * @param r A Releaser object which the protected object has as a
 * public member.
 * @param io_condition The condition to be watched for (G_IO_IN may be
 * bitwise-ored with G_IO_HUP).
 * @param priority The priority to be given to the watch in the main
 * loop.  In ascending order of priorities, priorities are
 * G_PRIORITY_LOW, G_PRIORITY_DEFAULT_IDLE, G_PRIORITY_HIGH_IDLE,
 * G_PRIORITY_DEFAULT and G_PRIORITY_HIGH. The default is
 * G_PRIORITY_DEFAULT.  This determines the order in which the
 * callback will appear in the event list in the main loop, not the
 * priority which the OS will adopt
 * @param context The glib main context to which the watch is to be
 * attached (the default of NULL will cause the watch to be attached
 * to the main program loop, and this is almost always what is
 * wanted).
 * @return The glib source id of the watch.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  If it does
 * so, the CallbackArg object will be disposed of.
 * @note Cancellation of the thread to which the watch is attached is
 * blocked during execution of the callback.
 * @ingroup io_watch
 * 
 * Since 0.9.2 choice of priority available.
 */
guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb, Releaser& r,
		    GIOCondition io_condition, gint priority = G_PRIORITY_DEFAULT,
		    GMainContext* context = 0);

/**
 * @b DEPRECATED: Don't use this function, it is just provided to
 * maintain API compatibility with version 0.9.0 and 0.9.1 of
 * c++-gtk-utils and will be dropped at some stage.  The signature of
 * start_iowatch() was changed in version 0.9.2 to permit a priority
 * to be specified.
 * @ingroup io_watch
 */
// we have been lucky with this - in the very unlikely event of a user
// having previously passed an explicit context argument of 0 (the
// previous default) to this deprecated version of start_iowatch()
// which will now be interpreted by the compiler on overload
// resolution as a call to the undeprecated version with a priority
// of 0 and a default GMainContext of NULL, the effect will be
// identical since G_PRIORITY_DEFAULT is defined as 0 (phew!)
inline guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb,
			   GIOCondition io_condition, GMainContext* context) {
  return start_iowatch(fd, cb, io_condition, G_PRIORITY_DEFAULT, context);
}

/**
 * @b DEPRECATED: Don't use this function, it is just provided to
 * maintain API compatibility with version 0.9.0 and 0.9.1 of
 * c++-gtk-utils and will be dropped at some stage.  The signature of
 * start_iowatch() was changed in version 0.9.2 to permit a priority
 * to be specified.
 * @ingroup io_watch
 */
// we have been lucky with this - in the very unlikely event of a user
// having previously passed an explicit context argument of 0 (the
// previous default) to this deprecated version of start_iowatch()
// which will now be interpreted by the compiler on overload
// resolution as a call to the undeprecated version with a priority
// of 0 and a default GMainContext of NULL, the effect will be
// identical since G_PRIORITY_DEFAULT is defined as 0 (phew!)
inline guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb, Releaser& r,
			   GIOCondition io_condition, GMainContext* context) {
  return start_iowatch(fd, cb, r, io_condition, G_PRIORITY_DEFAULT, context);
}

} // namespace Cgu

#endif
