/* Copyright (C) 2005 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the src/utils sub-directory);
   if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#include <utility>

#include <glib/gmain.h>

#include "notifier.h"
#include "mutex.h"

int Notifier::count = -1;
InstanceMap Notifier::instance_map;
PipeFifo Notifier::pipe;
Thread::Mutex* Notifier::write_mutex_p;
guint Notifier::iowatch_tag;

Notifier::Notifier(void) {

  count++;

  if (count < 0) {
    throw NotifierError("More than INT_MAX Notifier objects created");
  }

  id = make_id();
  if (id == -1) {
    throw NotifierError("Mapping count error in Notifier::make_id()\n");
  }

  instance_map.insert(std::pair<int, Notifier*>(id, this));

  if (!count) {

    try {
      pipe.open(PipeFifo::block);
    }
    catch (PipeError&) {
      throw PipeError("PipeFifo::open() call in Notifier::Notifier() cannot open pipe\n");
    }

    write_mutex_p = new Thread::Mutex;
    iowatch_tag = start_iowatch(pipe.get_read_fd(),
				sigc::ptr_fun(&Notifier::read_pipe_slot),
				G_IO_IN);
  }
}

Notifier::~Notifier(void) {

  count--;
  if (count < 0) {
    g_source_remove(iowatch_tag);
    pipe.close();
    delete write_mutex_p;
  }
  if (id != -1) instance_map.erase(id);
}

sigc::connection Notifier::connect(const sigc::slot<void>& cb) {

  return sig.connect(cb);
}

void Notifier::emit(void) {

  if (id != -1) {
    // make sure the write is atomic between threads
    Thread::Mutex::Lock lock(*write_mutex_p);
    pipe.write(reinterpret_cast<char*>(&id), sizeof(int));
  }
}

int Notifier::make_id(void) {

  // valid values of id are 0 to INT_MAX
  // as an approximation, start with the current count for new id
  int new_id = count;

  int iteration_count = -1;

  InstanceMap::iterator map_iter;
  do {
    iteration_count++;
    map_iter = instance_map.find(new_id);

    // map_iter == instance_map.end() if the id is not yet in use
    if (map_iter != instance_map.end()) {
      if (new_id < 0) new_id = 0;
      else new_id++;
    }
  } while (map_iter != instance_map.end() && iteration_count >= 0);
  
  if (iteration_count < 0) {
    new_id = -1;
  }
  
  return new_id;
}

bool Notifier::read_pipe_slot(void) {
  int emitted_id;
  int remaining = sizeof(int);
  ssize_t result;
  char* temp_p = reinterpret_cast<char*>(&emitted_id);
  do {
    result = Notifier::pipe.read(temp_p, remaining);
    if (result > 0) {
      temp_p += result;
      remaining -= result;
    }
  } while (remaining             // more to come
	   && result             // not end of file
	   && result != -1);     // no error

  if (result > 0) {
    // extract the character from the pipe and iterators for the slots to which it refers
    InstanceMap::iterator map_iter = instance_map.find(emitted_id);
    if (map_iter == instance_map.end()) {
      // throwing an exception in this slot is probably too extreme as it cannot readily
      // be handled - so report the error
      write_error("Yikes! There is a mapping error in Notifier::read_pipe_slot()\n");
    }

    else {
      // signal to the connected slot(s)
      map_iter->second->sig();
    }
  }
  else { // end of file or error on pipe
    // throwing an exception in this slot is probably too extreme as it cannot readily
    // be handled - so report the error
    write_error("IO error in Notifier::read_pipe_slot()\n");
  }
  return true; // multishot
}
