/* Copyright (C) 2001 2002 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include "prog_defs.h"

#include <unistd.h>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <strstream>
#include <string>

#include <gtkmm/main.h>
#include <glibmm/ustring.h>
#include <gdk/gdkx.h>
#include <glibmm/convert.h>
#include <X11/Xlib.h>

#include "mainwindow.h"
#include "dialogs.h"

#ifdef HAVE_GETTEXT
#include <locale.h>
#include <libintl.h>
#endif

Prog_config prog_config;

bool get_prog_parm(const char*, std::string&, Glib::ustring&, Glib::ustring(*)(const std::string&));

inline bool get_prog_parm(const char* name, std::string& line, Glib::ustring& result) {
  return get_prog_parm(name, line, result, Glib::locale_to_utf8);
}

void get_fonts(void);

int main(int argc, char* argv[]) {

  // set up the locale for gettext() and base locale domain name of this program
#ifdef HAVE_GETTEXT
  bindtextdomain("efax-gtk", LOCALEDIR);
  bind_textdomain_codeset("efax-gtk", "UTF-8");
  textdomain("efax-gtk");
  setlocale(LC_ALL,"");
#endif

  string messages(configure_prog(false));

  Gtk::Main app(argc, argv);

  get_fonts();
  if (!prog_config.GPL_flag) {
    GplDialog gpl_dialog(24);
    int result = gpl_dialog.run();
    if (result == GplDialog::accepted) {
      string gpl_file(prog_config.homedir + "/.efax-gtk_gpl");
      ofstream gplfile(gpl_file.c_str(), ios::out);
      prog_config.GPL_flag = true;
    }
    else beep();
  }
  if (prog_config.GPL_flag) {

    const char* filename = 0;
    if (argc > 1) filename = argv[argc - 1];
    MainWindow mainwindow(messages, filename);

    // everything is set up
    // now enter the main program loop
    
    app.run(mainwindow); // main loop will terminate when mainwindow hides
  }
  return 0;
}

Glib::ustring configure_prog(bool reread) {

  if (!reread) {

    // check to see if GPL terms have been accepted

    char* home = getenv("HOME");

    if (!home) cerr << "Your HOME environmental variable is not defined!" << endl;
    else prog_config.homedir = Glib::filename_to_utf8(home);

    string gpl_file(prog_config.homedir + "/.efax-gtk_gpl");
    int result = access(gpl_file.c_str(), F_OK);
  
    if (!result) prog_config.GPL_flag = true;
    else prog_config.GPL_flag = false;
  }

// now find rc file

  prog_config.found_rcfile = false;
  ifstream filein;
  string rcfile;
  Glib::ustring return_val;

  if (!prog_config.homedir.empty()) {
    rcfile = prog_config.homedir + "/.";
    rcfile += RC_FILE;

#ifdef HAVE_IOS_NOCREATE
    filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
    // we must have Std C++ so we probably don't need a ios::nocreate
    // flag on a read open to ensure uniqueness
    filein.open(rcfile.c_str(), ios::in);
#endif

    if (filein) prog_config.found_rcfile = true;
    else filein.clear();
  }

  if (!prog_config.found_rcfile) {

    rcfile = "/usr/local/etc/";
    rcfile += RC_FILE;

#ifdef HAVE_IOS_NOCREATE
    filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
    // we must have Std C++ so we probably don't need a ios::nocreate
    // flag on a read open to ensure uniqueness
    filein.open(rcfile.c_str(), ios::in);
#endif

    if (filein) prog_config.found_rcfile = true;
    else filein.clear();
  }

  if (!prog_config.found_rcfile) {

    rcfile = "/etc/";
    rcfile += RC_FILE;

#ifdef HAVE_IOS_NOCREATE
    filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
    // we must have Std C++ so we probably don't need a ios::nocreate
    // flag on a read open to ensure uniqueness
    filein.open(rcfile.c_str(), ios::in);
#endif

    if (filein) prog_config.found_rcfile = true;
    else filein.clear();
  }

  if (!prog_config.found_rcfile) {
    return_val = "Can't find or open file /etc/" RC_FILE ",\n"
                 "/usr/local/etc/" RC_FILE;
    if (!prog_config.homedir.empty()) {
      return_val +=  " or ";
      return_val += prog_config.homedir + "/." RC_FILE;
    }
    return_val += "\n";
  }
  
  else {

    // if we are re-reading efax-gtkrc, we need to clear the old settings
    if (reread) {
      prog_config.my_name = "";
      prog_config.my_number = "";
      prog_config.parms.clear();
      prog_config.lock_file = "";
      prog_config.page_size = "";
      prog_config.page_dim = "";
      prog_config.resolution = "";
      prog_config.print_cmd = "";
      prog_config.print_shrink = "";
      prog_config.ps_view_cmd = "";
      prog_config.logfile_name = "";
    }
    // if we are setting up for first time, initialise prog_config.receive_dirname
    else *prog_config.receive_dirname = 0;

// now extract settings from file

    string file_read;
    Glib::ustring device;
    Glib::ustring modem_class;
    Glib::ustring rings;
    Glib::ustring dialmode;
    Glib::ustring init;
    Glib::ustring reset;
    Glib::ustring capabilities;
    Glib::ustring extra_parms;
    
    while (getline(filein, file_read)) {

      if (!file_read.empty() && file_read[0] != '#') { // valid line to check
	// now check for other comment markers
	string::size_type pos = file_read.find_first_of('#');
	if (pos != string::npos) file_read.resize(pos); // truncate
	
	// look for "NAME:"
	if (get_prog_parm("NAME:", file_read, prog_config.my_name));
	
	// look for "NUMBER:"
	else if (get_prog_parm("NUMBER:", file_read, prog_config.my_number));
	
	// look for "DEVICE:"
	else if (get_prog_parm("DEVICE:", file_read, device));
	
	// look for "LOCK:"
	else if (get_prog_parm("LOCK:", file_read, prog_config.lock_file,
			       Glib::filename_to_utf8));

	// look for "CLASS:"
	else if (get_prog_parm("CLASS:", file_read, modem_class));

	// look for "PAGE:"
	else if (get_prog_parm("PAGE:", file_read, prog_config.page_size));

	// look for "RES:"
	else if (get_prog_parm("RES:", file_read, prog_config.resolution));
	
	// look for "RINGS:"
	else if (get_prog_parm("RINGS:", file_read, rings));
	
	// look for "DIALMODE:"
	else if (get_prog_parm("DIALMODE:", file_read, dialmode));
	
	// look for "INIT:"
	else if (get_prog_parm("INIT:", file_read, init));
	
	// look for "RESET:"
	else if (get_prog_parm("RESET:", file_read, reset));

	// look for "CAPABILITIES:"
	else if (get_prog_parm("CAPABILITIES:", file_read, capabilities));

	// look for "PARMS:"
	else if (get_prog_parm("PARMS:", file_read, extra_parms));

	// look for "PRINT_CMD:"
	else if (get_prog_parm("PRINT_CMD:", file_read, prog_config.print_cmd));

	// look for "PRINT_SHRINK:"
	else if (get_prog_parm("PRINT_SHRINK:", file_read, prog_config.print_shrink));

	// look for "PS_VIEWER:"
	else if (get_prog_parm("PS_VIEWER:", file_read, prog_config.ps_view_cmd));

	// look for "LOG_FILE:"
	else if (get_prog_parm("LOG_FILE:", file_read, prog_config.logfile_name,
			       Glib::filename_to_utf8));
      }
    }

    // we have finished reading the configuration file    
    // now enter parameters common to send and receive of faxes	
    prog_config.parms.push_back("efax");

    prog_config.parms.push_back("-vew");  // stderr -- errors and warnings
    
    prog_config.parms.push_back("-vin");  // stdin -- information and negotiations
    //prog_config.parms.push_back("-vina");  // this will also report the parameters passed to efax
                                             // uncomment it and comment out preceding line for debugging
    
    string temp;

    if (!prog_config.my_number.empty()) {
      temp = "-l";
      temp += prog_config.my_number;
      prog_config.parms.push_back(temp);
    }

    if (!init.empty()) {
      string::size_type start = 0;
      string::size_type end;
      
      while (start != string::npos) {
	temp = "-i";
	end = init.find_first_of(" \t", start);
	if (end != string::npos) {
	  temp.append(init, start, end - start);
	  start = init.find_first_not_of(" \t", end); // prepare for the next iteration
	}
	else {
	  temp.append(init, start, init.size() - start);
	  start = end;
	}
	prog_config.parms.push_back(temp);
      }
    }
    
    else {
      prog_config.parms.push_back("-iZ");
      prog_config.parms.push_back("-i&FE&D2S7=120");
      prog_config.parms.push_back("-i&C0");
      prog_config.parms.push_back("-iM1L0");
      
    }
    
    if (!reset.empty()) {
      string::size_type start = 0;
      string::size_type end;
    
      while (start != string::npos) {
	temp = "-k";
	end = reset.find_first_of(" \t", start);
	if (end != string::npos) {
	  temp.append(reset, start, end - start);
	  start = reset.find_first_not_of(" \t", end); // prepare for the next iteration
	}
	else {
	  temp.append(reset, start, reset.size() - start);
	  start = end;
	}
	prog_config.parms.push_back(temp);
      }
    }

    else prog_config.parms.push_back("-kZ");
    
    if (!modem_class.empty()) {
      temp = "-o";
      if (!modem_class.compare("2.0")) temp += '0';
      else if (!modem_class.compare("1")) temp += '1';
      else if (!modem_class.compare("2")) temp += '2';
      else {
	return_val += gettext("Invalid modem class specified\n"
			      "Adopting default of Class 2\n");
        temp += '2';
      }
      prog_config.parms.push_back(temp);
    }
      
    if (device.empty()) {
      if (access("/dev/modem", F_OK) == -1) {
	return_val += gettext("No serial port device specified in " RC_FILE " configuration file\n"
			      "and /dev/modem does not exist\n");
      }
      
      else {
	return_val += gettext("No serial port device specified in " RC_FILE " configuration file\n"
			      "Using default of /dev/modem\n");
	device = "modem";
      }
    }
      
    if (!device.empty()) {
      if (prog_config.lock_file.empty()) prog_config.lock_file = "/var/lock";
      prog_config.lock_file += "/LCK..";
      prog_config.lock_file += device;
    
      temp = "-d/dev/";
      temp += device;
      prog_config.parms.push_back(temp);
    }

    temp = "-x";
    temp += filename_from_utf8(prog_config.lock_file);
    prog_config.parms.push_back(temp);

    if (!capabilities.empty()) {
      temp = "-c";
      temp += capabilities;
      prog_config.parms.push_back(temp);
    }

    if (prog_config.resolution.empty()) {
      return_val += gettext("Adopting default fax resolution of 204x196\n");
      prog_config.resolution = "204x196";
    }
    else if (!prog_config.resolution.compare("fine")) prog_config.resolution = "204x196";
    else if (!prog_config.resolution.compare("standard")) prog_config.resolution = "204x98";
    else {
      return_val += gettext("Invalid fax resolution specified\n"
			    "Adopting default fax resolution of 204x196\n");
      prog_config.resolution = "204x196";
    }
    
    if (rings.empty()) prog_config.rings = '1';
    else if (rings.size() > 1 || rings[0] < '1' || rings[0] > '9') {
      return_val += gettext("Invalid ring number specified\n"
			    "Will answer after one ring\n");
      prog_config.rings = '1';
    }
    else prog_config.rings = rings[0];

    if (prog_config.page_size.empty()) {
      return_val += gettext("Adopting default page size of a4\n");
      prog_config.page_size = "a4";
      prog_config.page_dim = "210x297mm";
    }
    else if (!prog_config.page_size.compare("a4")) prog_config.page_dim = "210x297mm";
    else if (!prog_config.page_size.compare("letter")) prog_config.page_dim = "216x279mm";
    else if (!prog_config.page_size.compare("legal")) prog_config.page_dim = "216x356mm";
    else {
      return_val += gettext("Invalid page size specified\n"
			    "Adopting default page size of a4\n");
      prog_config.page_size = "a4";
      prog_config.page_dim = "210x297mm";
    }
    
    if (dialmode.empty()) prog_config.tone_dial = true;
    else if (!dialmode.compare("tone")) prog_config.tone_dial = true;
    else if (!dialmode.compare("pulse")) prog_config.tone_dial = false;
    else {
      return_val += gettext("Invalid dialmode specified\n"
			    "Adopting default of tone dialling\n");
      prog_config.tone_dial = true;
    }    

    if (!extra_parms.empty()) {
      string::size_type start = 0;
      string::size_type end;
    
      while (start != string::npos) {
	end = extra_parms.find_first_of(" \t", start);
	if (end != string::npos) {
	  temp.assign(extra_parms, start, end - start);
	  start = extra_parms.find_first_not_of(" \t", end); // prepare for the next iteration
	}
	else {
	  temp.assign(extra_parms, start, extra_parms.size() - start);
	  start = end;
	}
	prog_config.parms.push_back(temp);
      }
    }
    if (prog_config.print_cmd.empty()) {
      return_val += gettext("Adopting default printer command of 'lpr'\n");
      prog_config.print_cmd = "lpr";
    }
    
    if (prog_config.print_shrink.empty()) {
      prog_config.print_shrink = "100";
    }
    else if (atoi(prog_config.print_shrink.c_str()) < 50 || atoi(prog_config.print_shrink.c_str()) > 100) {
      return_val += gettext("Invalid print shrink specified: adopting default value of 100\n");
      prog_config.print_shrink = "100";
    }
    
    if (prog_config.ps_view_cmd.empty()) {
      return_val += gettext("Adopting default postscript view command of 'gv'\n");
      prog_config.ps_view_cmd = "gv";
    }
  }
  return return_val;
}

void get_fonts(void) {

  // this will get a suitable fixed font for GplDialog and HelpDialog to use with Pango
  const int MAX_FONTS = 10000;
  int num_fonts;
  char** fonts = XListFonts(GDK_DISPLAY(), "-*", MAX_FONTS, &num_fonts);

  if (fonts) {
    int count;
    string inspect_name;
    prog_config.fixed_font = "";

    //try for courier font
    for (count = 0; count < num_fonts; count++) {
      inspect_name = fonts[count];
      string::size_type pos = pos = inspect_name.find("courier-medium-r-normal-");
      if (pos != string::npos) {
	prog_config.fixed_font = "courier";
	break;
      }
    }

    // unsuccessful -- go for the generic "fixed" font
    if (prog_config.fixed_font.empty()) prog_config.fixed_font = "fixed";
  }
  XFreeFontNames(fonts);
}

bool get_prog_parm(const char* name, std::string& line, Glib::ustring& result,
		   Glib::ustring(*convert_func)(const std::string&)) {
// This function looks for a setting named `name' in the string `line'
// and returns the values stated after it in string `result'.  It returns
// `true' if the setting was found.  If there are trailing spaces or tabs,
// string `line' will be modified.  string `result' is only modified if
// the `name' setting is found.  Anything extracted from `line' will be
// converted (when placed into `result') to UTF-8 as maintained by
// Glib::ustring, using the function assigned to function pointer
// convert_func (you would normally use Glib::locale_to_utf8() or
// Glib::filename_to_utf8(), and there is a default inline function
// using Glib::locale_to_utf8()

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) {
    // erase any trailing space or tab
    while (line.find_last_of(" \t") == line.size() - 1) line.resize(line.size() - 1);
    if (line.size() > length) {
      // ignore any preceding space or tab from the setting value given
      string::size_type pos = line.find_first_not_of(" \t", length); // pos now is set to beginning of setting value
      if (pos != string::npos) result.assign(convert_func(line.substr(pos)));
    }
    return true;
  }
  return false;
}

void beep(void) {
  XBell(GDK_DISPLAY(), 0);
}
