/* Copyright (C) 2001 2002 2003 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <gdk/gdkkeysyms.h> // the key codes are here
#include <gdkmm/pixbuf.h>
#include <gtkmm/stock.h>

#include "helpfile.h"
#include "window_icon.h"

#ifdef HAVE_GETTEXT
#include <libintl.h>
#endif

HelpDialog::HelpDialog(const int standard_size): Gtk::Window(Gtk::WINDOW_TOPLEVEL),
			     close_button(Gtk::Stock::CLOSE),
			     button_box(Gtk::BUTTONBOX_SPREAD) { 

  sending_editbox.set_editable(false);
  sending_editbox.modify_font(Pango::FontDescription(prog_config.fixed_font));
  sending_editbox.get_buffer()->set_text(get_sending_help());
  sending_editbox.unset_flags(Gtk::CAN_FOCUS);

  sending_scrolled_window.set_shadow_type(Gtk::SHADOW_IN);
  sending_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  sending_scrolled_window.add(sending_editbox);

  receiving_editbox.set_editable(false);
  receiving_editbox.modify_font(Pango::FontDescription(prog_config.fixed_font));
  receiving_editbox.get_buffer()->set_text(get_receiving_help());
  receiving_editbox.unset_flags(Gtk::CAN_FOCUS);

  receiving_scrolled_window.set_shadow_type(Gtk::SHADOW_IN);
  receiving_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  receiving_scrolled_window.add(receiving_editbox);

  addressbook_editbox.set_editable(false);
  addressbook_editbox.modify_font(Pango::FontDescription(prog_config.fixed_font));
  addressbook_editbox.get_buffer()->set_text(get_addressbook_help());
  addressbook_editbox.unset_flags(Gtk::CAN_FOCUS);

  addressbook_scrolled_window.set_shadow_type(Gtk::SHADOW_IN);
  addressbook_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  addressbook_scrolled_window.add(addressbook_editbox);

  fax_list_editbox.set_editable(false);
  fax_list_editbox.modify_font(Pango::FontDescription(prog_config.fixed_font));
  fax_list_editbox.get_buffer()->set_text(get_fax_list_help());
  fax_list_editbox.unset_flags(Gtk::CAN_FOCUS);

  fax_list_scrolled_window.set_shadow_type(Gtk::SHADOW_IN);
  fax_list_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  fax_list_scrolled_window.add(fax_list_editbox);

  settings_editbox.set_editable(false);
  settings_editbox.modify_font(Pango::FontDescription(prog_config.fixed_font));
  settings_editbox.get_buffer()->set_text(get_settings_help());
  settings_editbox.unset_flags(Gtk::CAN_FOCUS);

  settings_scrolled_window.set_shadow_type(Gtk::SHADOW_IN);
  settings_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
  settings_scrolled_window.add(settings_editbox);

  button_box.add(close_button);

  notebook.set_tab_pos(Gtk::POS_TOP);
  notebook.set_scrollable(true);
  // set up the notebook pages
  {
    using namespace Gtk::Notebook_Helpers;
    PageList& page_list = notebook.pages();
    page_list.push_back(TabElem(sending_scrolled_window, gettext("Sending")));
    page_list.push_back(TabElem(receiving_scrolled_window, gettext("Receiving")));
    page_list.push_back(TabElem(addressbook_scrolled_window, gettext("Address Book")));
    page_list.push_back(TabElem(fax_list_scrolled_window, gettext("Fax Lists")));
    page_list.push_back(TabElem(settings_scrolled_window, gettext("Settings")));
  }

  vbox.pack_start(notebook, Gtk::PACK_EXPAND_WIDGET, standard_size/3);
  vbox.pack_start(button_box, Gtk::PACK_SHRINK, standard_size/3);

  close_button.signal_clicked().connect(SigC::slot(*this, &HelpDialog::kill_dialog));
  
  add(vbox);
  
  set_title(gettext("efax-gtk: Help"));
  set_default_size(standard_size * 25, standard_size * 14);
  
  set_border_width(standard_size/4);
  set_position(Gtk::WIN_POS_NONE);
  
  grab_focus();
  close_button.unset_flags(Gtk::CAN_FOCUS);

  set_icon(Gdk::Pixbuf::create_from_xpm_data(window_icon_xpm));

  show_all();
}

void HelpDialog::kill_dialog(void) {
  hide_all();
  delete this;  // this a modeless dialog, and so safe
}

bool HelpDialog::on_delete_event(GdkEventAny*) {
  kill_dialog();
  return true; // returning true prevents destroy sig being emitted
}

bool HelpDialog::on_key_press_event(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  
  if (keycode == GDK_Escape) kill_dialog();

  else if (keycode == GDK_Home || keycode == GDK_End
	   || keycode == GDK_Up || keycode == GDK_Down
	   || keycode == GDK_Page_Up || keycode == GDK_Page_Down) {
    static_cast<HelpEditor*>(notebook.get_nth_page(notebook.get_current_page()))->on_key_press_event(event_p);
    return false;
  }
  return true;
}

const char* HelpDialog::get_sending_help(void) {

  return gettext("\n"
		 "Sending faxes\n"
		 "-------------\n"
		 "\n"
		 "Before sending a fax, the name of the file to be sent must be\n"
		 "specified in the \"File to fax\" box.  The file specified must be in\n"
		 "postscript format, and will be converted by the program into the\n"
		 "correct tiffg3 fax format.\n"
		 "\n"
		 "The file can be entered into the \"File to fax\" box by pressing the\n"
		 "\"File to fax\" button (or from the `File/Find file' pull down menu\n"
		 "item), by starting the program with the filename as the last parameter,\n"
		 "or by entering it by hand into the box.  Files can be more easily\n"
		 "found with the Find file dialog if they are placed in the $HOME/faxout\n"
		 "directory.\n"
		 "\n"
		 "In addition, the telephone number to which the fax is to be sent must\n"
		 "be entered into the \"Tel number\" box.  This can be entered directly\n"
		 "into the box, or by using the built-in addressbook.  The addressbook\n"
		 "can be invoked by pressing the \"Tel number\" button, or from the\n"
		 "`File/Address book' pull-down menu item.  See \"Using the address book\"\n"
		 "further below.\n"
		 "\n"
		 "To make the sending of faxes direct from a word processor program more\n"
		 "easy, see the README file which comes with the distribution.\n"
		 "\n"
		 "Successfully sent faxes are copied to a directory in the $HOME/faxsent\n"
		 "directory, which has a name derived from the year, month, day, hour\n"
		 "and seconds when the sending of the fax was completed, and will appear\n"
		 "in the faxes sent list.  They are only included in that list if they\n"
		 "have been sent without error.  The efax message display box will\n"
		 "report on the progress of a fax being sent.  The fax list can be\n"
		 "brought up from the `File/List sent faxes' pull down menu item.  See\n"
		 "\"Using the fax lists\" further below.\n");

}

const char* HelpDialog::get_receiving_help(void) {

  return gettext("\n"
		 "Receiving faxes\n"
		 "---------------\n"
		 "\n"
		 "Three ways of receiving faxes are provided for.\n"
		 "\n"
		 "First, the program can be set to answer a fax call which is ringing\n"
		 "but has not been answered, by pressing the \"Answer call\" button.\n"
		 "\n"
		 "Secondly, the program can take over a call which has already been\n"
		 "answered (say, by a telephone hand set) by pressing the \"Take over\n"
		 "call\" button.\n"
		 "\n"
		 "Thirdly, the program can be placed in standby mode by pressing the\n"
		 "\"Standby\" button.  This will automatically answer any call after the\n"
		 "number of rings specified in the efax-gtkrc file, and receive the fax.\n"
		 "The program will keep on receiving faxes until the \"Stop\" button is\n"
		 "pressed.\n"
		 "\n"
		 "Received faxes in tiffg3 format (one file for each page) are placed in\n"
		 "a directory in the $HOME/faxin directory, which has a name derived\n"
		 "from the year, month, day, hour and seconds when the relevant \"Answer\n"
		 "call\", \"Take over call\" or \"Standy\" button was pressed.  (Where in\n"
		 "standby mode after a fax has been received, any further fax will\n"
		 "derive its name from the time when receipt of the last received fax\n"
		 "has been completed and the program goes back into standby mode.)\n"
		 "\n"
		 "Received faxes can be printed, viewed, described and managed using the\n"
		 "built in fax list facility.  This can be brought up from the\n"
		 "`File/List received faxes' pull down menu item.  See \"Using the fax\n"
		 "lists\" further below.\n");
}

const char* HelpDialog::get_addressbook_help(void) {

  return gettext("\n"
		 "Using the address book\n"
		 "----------------------\n"
		 "\n"
		 "To pick a telephone number from the address book, highlight the\n"
		 "relevant address by pressing the left mouse button over it, and then\n"
		 "press the \"OK\" button.\n"
		 "\n"
		 "Addresses can be added to the address book by pressing the add button,\n"
		 "and then completing the relevant dialog which will appear.  To delete\n"
		 "an address from the address book, highlight the relevant address and\n"
		 "press the delete (trashcan) button.  The addressbook can be sorted by\n"
		 "using the up and down arrow buttons on a highlighted address.\n");
}

const char* HelpDialog::get_fax_list_help(void) {

  return gettext("\n"
		 "Using the fax lists\n"
		 "------------------\n"
		 "\n"
		 "To bring up the fax lists, go to the the `File' menu and pick the\n"
		 "`List received faxes' or `List sent faxes' menu item.  Highlight the\n"
		 "fax to printed or viewed by pressing the left mouse button.  The\n"
		 "programs to be used to print and view the fax are specifed in the\n"
		 "efax-gtkrc configuration file, or if none are specified, the program\n"
		 "will print using lpr (which will work for most Unix systems) and view\n"
		 "with gv.\n"
		 "\n"
		 "To print faxes, a PRINT_SHRINK parameter can be specifed in efax-gtkrc\n"
		 "to enable the fax page to fit within the printer margins.  A parameter\n"
		 "of 98 will work with most printers.  This can be changed while the\n"
		 "program is running by bringing up the `Settings' dialog and entering\n"
		 "it into the `Print/Print Shrink' box.\n"
		 "\n"
		 "A fax can be deleted from a fax list by pressing the delete (trashcan)\n"
		 "button.  This will place the deleted fax in the $HOME/faxin/oldfax\n"
		 "or $HOME/faxsent/oldfax directory.\n"
		 "\n"
		 "NB: link()/unlink() are used to move a fax, so $HOME/faxin/oldfax must\n"
		 "be on the same file system as $HOME/faxin, and $HOME/faxsent/oldfax\n"
		 "must be on the same file system as $HOME/faxsent.  If they are not,\n"
		 "the fax will remain in $HOME/faxin or $HOME/faxsent.\n"
		 "\n"
		 "A description can be added to a received fax when appearing in a fax\n"
		 "list (or subsequently amended) by pressing the relevant button -- this\n"
		 "will enable faxes to be more easily identified.\n"
		 "\n"
		 "To refresh a fax list (say, to update with any faxes received or sent\n"
		 "since the fax list was brought up), press the refresh button.\n");
}

const char* HelpDialog::get_settings_help(void) {

  return gettext("\n"
		 "Settings\n"
		 "--------\n"
		 "\n"
		 "The program settings can be changed by manually editing the efax-gtk\n"
		 "configuration file comprising $HOME/.efax-gtkrc,\n"
		 "/usr/local/etc/efax-gtkrc or /etc/efax-gtkrc.  The file is searched\n"
		 "for in that order, so $HOME/.efax-gtkrc takes precedence over the\n"
		 "other two.\n"
		 "\n"
		 "The configuration file can also be set by using the Settings dialog\n"
		 "launched from the `File/Settings' pull down menu item.  The settings\n"
		 "entered using this dialog are always stored as $HOME/.efax-gtkrc.\n"
		 "Accordingly, if the Settings dialog has been used, and you want to\n"
		 "revert to the global settings, this can be done either by deleting the\n"
		 "$HOME/.efax-gtkrc file, or by pressing the `Reset' button in the\n"
		 "Settings dialog, which will reload the Settings dialog from\n"
		 "/etc/efax-gtkrc or /usr/local/etc/efax-gtkrc.\n"
		 "\n"
		 "Help can be obtained when filling out the Settings dialog by holding\n"
		 "the mouse over the relevant help (?) button, which will bring up a\n"
		 "\"Tips\" display, or by pressing the button, which will bring up an\n"
		 "information display.\n");
}
