/*============================================================================
 *  Définitions des fonctions de base
 *   associées à la structure `ecs_cmd_t' décrivant
 *   les options de la ligne de commande
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilite
 *============================================================================*/

#include "ecs_config.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <ctype.h>       /* isdigit()          */
#include <math.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>      /* atoi()             */
#include <string.h>      /* strlen()           */
#include <time.h>        /* time(), strftime() */

#include <bft_error.h>
#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>
#include <bft_sys_info.h>
#include <bft_version.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' systeme
 *----------------------------------------------------------------------------*/

#if defined(HAVE_MKDIR)
#include <sys/stat.h>
#include <sys/types.h>
#endif

#if defined(HAVE_DUP2)
#include <unistd.h>
#endif

#if defined(HAVE_STAT)
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#endif

#if defined(HAVE_GETCWD)
#include <unistd.h>
#endif

#if defined(HAVE_GETPWUID) && defined(HAVE_GETEUID)
#include <pwd.h>
#endif

#if defined(HAVE_UNAME)
#include <sys/utsname.h>
#endif

/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_pre.h"

#if defined(HAVE_MED)
#include "ecs_med.h"
#endif


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_cmd.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_cmd_priv.h"


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui complète la ligne de commande par les données d'un fichier
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__lit_arg_fic(char  *  nomfic,
                         int   *  argc,
                         char  ** argv[])
{
  ecs_int_t    iarg;
  ecs_int_t    ideb;
  ecs_int_t    ifin;
  ecs_int_t    ipos;

  ecs_int_t    nbr_arg;
  char       **lst_arg;

  char         ligne[ECS_CMD_LIGNE_MAX + 1];
  char        *res;

  bool         bool_err;

  FILE        *ptr = NULL;
  bft_file_t  *fic_cmd = NULL;

  nbr_arg = *argc;
  lst_arg = *argv;

  bool_err = false;

  if (nomfic != NULL) {

    fic_cmd = bft_file_open(nomfic,
                            BFT_FILE_MODE_READ,
                            BFT_FILE_TYPE_TEXT);

  }
  else {

    ptr = stdin;

  }


  /* Recopie du début de la ligne de commande */

  BFT_MALLOC(lst_arg, nbr_arg, char *);

  for (iarg = 0; iarg < nbr_arg; iarg++) {
    BFT_MALLOC(lst_arg[iarg], strlen((*argv)[iarg]) + 1, char);
    strcpy(lst_arg[iarg], (*argv)[iarg]);
  }

  /* Lecture du fichier (ligne à ligne) */

  do {

    if (ptr != NULL)
      res = fgets(ligne, ECS_CMD_LIGNE_MAX, ptr);

    else if (fic_cmd != NULL)
      res = bft_file_gets_try(ligne, ECS_CMD_LIGNE_MAX, fic_cmd, NULL);

    /*
      Traitement de la ligne; on ignore ce qui se trouve après
      le caractère '#' (commentaire)
    */

    if (res != NULL) {

      ideb = 0;
      ifin = 0;

      while (   ligne[ifin] != '\0' && ligne[ifin] != '#'
             && ligne[ifin] != '\n' && ligne[ifin] != '\r') {

        for (ideb = ifin;
             ligne[ideb] != '\0' && (ligne[ideb] == ' ' || ligne[ideb] == '\t');
             ideb++);

        for (ifin = ideb;
             (   ligne[ifin] != '\0'  && ligne[ifin] != '#'
              && ligne[ifin] != ' '   && ligne[ifin] != '\t'
              && ligne[ifin] != '\n'  && ligne[ifin] != '\r');
             ifin++);

        if (   ligne[ideb] != '\0' && ligne[ideb] != '\n' && ligne[ideb] != '\r'
            && ifin > ideb) {

          BFT_REALLOC(lst_arg, nbr_arg + 1, char *);
          BFT_MALLOC(lst_arg[nbr_arg], ifin - ideb + 1, char);

          for (ipos = 0; ipos < ifin - ideb; ipos++)
            lst_arg[nbr_arg][ipos] = ligne[ideb + ipos];
          lst_arg[nbr_arg][ipos] = '\0';

          nbr_arg++;

        }

      }

    }

  } while (res != NULL);


  if (fic_cmd != NULL)
    bft_file_free(fic_cmd);


  /* Valeurs de retour */

  *argc = nbr_arg;
  *argv = lst_arg;

}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche le numero de version
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__aff_version(void)
{

  char str_date [ECS_STR_SIZE];

  int               ind_mois;
  int               nb_extensions = 0;
  struct tm         time_cnv;

  const char nom_mois[12][4]
    = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
       "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  strcpy(str_date, ecs_glob_build_date);

  /* Date de compilation */

  for (ind_mois = 0; ind_mois < 12; ind_mois++) {
    if (strncmp(str_date, nom_mois[ind_mois], 3) == 0) {
      time_cnv.tm_mon = ind_mois;
      break;
    }
  }

  sscanf(str_date + 3, "%d", &(time_cnv.tm_mday));
  sscanf(str_date + 6, "%d", &(time_cnv.tm_year));

  time_cnv.tm_year -= 1900;

  strcpy(str_date, __TIME__);

  sscanf(str_date, "%d", &(time_cnv.tm_hour));
  sscanf(str_date + 3, "%d", &(time_cnv.tm_min));
  sscanf(str_date + 6, "%d", &(time_cnv.tm_sec));

  time_cnv.tm_isdst = -1;

  /* Date de compilation recalculée et internationalisée */

  mktime(&time_cnv);
  strftime(str_date, ECS_STR_SIZE - 1, "%c", &time_cnv);

#if defined (PACKAGE_VERSION)
  bft_printf("\n  %s %s   (%s)\n",
             _("ECS   version"), PACKAGE_VERSION, str_date);
#else
  bft_printf("\n  %s    (%s)\n",
             _("ECS   version"), str_date);
#endif

  bft_printf("\n");

#if defined(HAVE_CCM)
  bft_printf(_("  STAR-CCM+ file format support\n"));
  nb_extensions++;
#endif

#if defined(HAVE_CGNS)
  if (ecs_glob_cgns_ver_rel >= 0)
    bft_printf(_("  CGNS %d.%d.%d file format support\n"),
               ecs_glob_cgns_ver_maj, ecs_glob_cgns_ver_min,
               ecs_glob_cgns_ver_rel);
  else
    bft_printf(_("  CGNS %d.%d file format support\n"),
               ecs_glob_cgns_ver_maj, ecs_glob_cgns_ver_min);
  nb_extensions++;
#endif

#if defined(HAVE_MED)
  ecs_med__version_shlib();
  if (ecs_glob_med_ver_rel < 0)
    bft_printf(_("  MED %d.%d (HDF5 %d.%d.%d) file format support\n"),
               ecs_glob_med_ver_maj, ecs_glob_med_ver_min,
               ecs_glob_hdf5_ver_maj, ecs_glob_hdf5_ver_min,
               ecs_glob_hdf5_ver_rel);
  else
    bft_printf(_("  MED %d.%d.%d (HDF5 %d.%d.%d) file format support\n"),
               ecs_glob_med_ver_maj, ecs_glob_med_ver_min,
               ecs_glob_med_ver_rel, ecs_glob_hdf5_ver_maj,
               ecs_glob_hdf5_ver_min, ecs_glob_hdf5_ver_rel);
  nb_extensions++;
#endif

  if (bft_version_zlib() != NULL) {
    bft_printf(_("  Reading of compressed files ('.gz') with Zlib %s\n"),
               bft_version_zlib());
    if (strcmp(bft_version_zlib(),
               bft_version_build_zlib()) != 0)
    bft_printf(_("    (BFT library support compiled with Zlib %s)\n"),
               bft_version_build_zlib());
    nb_extensions++;
  }
  if (nb_extensions > 0)
    bft_printf("\n");

}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche la ligne de commande, le titre et la version
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__aff_titre(int    argc,
                       char  *argv[])
{

  int       iarg;
  int       ltot;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  ltot = 0;

  for (iarg = 0; iarg < argc; iarg++) {

    ltot += strlen(argv[iarg]) + 1;

    if (ltot > 80) {
      bft_printf("\n");
      ltot = strlen(argv[iarg]) + 1;
    }

    bft_printf("%s ", argv[iarg]);

  }

  bft_printf(_("\n\n"
               "  .------------------------------.\n"
               "  |                              |\n"
               "  |   Code_Saturne Preprocessor  |\n"
               "  |                              |\n"
               "  `------------------------------'\n"));

  ecs_loc_cmd__aff_version();


}


/*----------------------------------------------------------------------------
 *  Fonction utilitataire pour les affichages
 *----------------------------------------------------------------------------*/

static void
_fct_prt(const char  *opt,
         const char  *arg,
         const char  *texte)
{
  size_t l = strlen(opt);
  size_t pad = (l > 12) ? 16+12-l :16;

  bft_printf("  %-12s ", opt);
  ecs_print_padded_str(arg, pad);
  bft_printf(" %s\n", texte);
}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche l'usage et les options
 *----------------------------------------------------------------------------*/

static void ecs_loc_cmd__aff_aide
(
 void
)
{

  char opt_str[81];

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf(_("\n\nUSAGE :"));


  /* Usage en PRE-TRAITEMENT */

  bft_printf(_("  %s [<options>]"),
             ECS_CMD_EXEC_NAME);
  bft_printf(" %s <%s>",
             ECS_CMD_OPTION_MESH_FILE_1, _("file"));


  /* Arguments nécessaires */
  /*-----------------------*/

  bft_printf(_("\n\n\nwith :\n\n"));


  _fct_prt(ECS_CMD_OPTION_MESH_FILE_1, _("<file> [...]"),
           _(": Preprocessor input mesh file"));
  _fct_prt("", "", _("  (mesh file format determined"));
  _fct_prt("", "", _("   automatically by filename extension"));
  _fct_prt("", "", _("   or forced by \"-format\" sub-option)"));
  _fct_prt(ECS_CMD_OPTION_MESH_FILE, _("<file> [...]"), _(": same"));

  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_NULL_COMM_1, "",
           _(": simulate messages without file output"));
  _fct_prt(ECS_CMD_OPTION_NULL_COMM, "", _(": same"));


  bft_printf("\n");


  /* Options generales */
  /*-------------------*/

  bft_printf(_("\ngeneral options:\n\n"));


#if defined(HAVE_CHDIR)

  _fct_prt(ECS_CMD_OPTION_CWD, _("<directory>"), _(": working directory"));
  _fct_prt("", "", _("  (if different from current directory !)"));

  bft_printf("\n");

#endif


  _fct_prt(ECS_CMD_OPTION_HELP_1, "", _(": this help message"));
  _fct_prt(ECS_CMD_OPTION_HELP, "", _(": same"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_INPUT_FILE_1, _("[<file>]"),
           _(": command line completed by a file"));

  _fct_prt("", "", _("  (default file: standard input)"));

  _fct_prt(ECS_CMD_OPTION_INPUT_FILE, _("[<file>]"), _(": same"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_OUTPUT_FILE, _("[<file>]"),
           _(": redirect terminal output to a file"));

  sprintf(opt_str, _("  (default file: \"%s\")"),
          ECS_CMD_OUTFILE_NAME_DEFAULT);
  _fct_prt("", "", opt_str);

  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_SELECT_FAC_INT, _("[<sub-options>]"),
           _(": count and display interior faces"));

  _fct_prt("", "", _("  these faces may be selected with"));
  _fct_prt("", "", _("  face-selection sub-options"));

  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_RC_2, _("[<sub-options>]"),
           _(": join non-conforming faces;"));

  _fct_prt("", "", _("  select faces to join with"));
  _fct_prt("", "", _("  face-selection sub-options"));

  _fct_prt(ECS_CMD_OPTION_RC, _("[<sub-options>]"), _(": same"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_PERIO_2, _("<sub-options>"),
           _(": mesh periodicity defined by argument;"));

  _fct_prt("", "", _("  joining sub-options for tolerances"));
  _fct_prt("", "", _("  and selection of periodic faces with"));
  _fct_prt("", "", _("  face-selection sub-options"));

  _fct_prt(ECS_CMD_OPTION_PERIO, _("<sub-options>"), _(": same"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_ORIENT_CORREC, "",
           _(": if necessary, correct orientation of"));
  _fct_prt("", "", _("  cells and faces"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_VERSION, "",
           _(": print version number"));


  /* Options de POST-TRAITEMENT */
  /*----------------------------*/

  bft_printf(_("\n\nPOSTPROCESSING options:\n\n"));


  _fct_prt(ECS_CMD_OPTION_CASE,
           _("<name>"),
           _(": case name (without this option,"));

  sprintf(opt_str,
          _("  the default name is: \"%s\""),
          ECS_CMD_POST_CASE_DEFAULT);
  _fct_prt("", "", opt_str);

  bft_printf("\n");

#if defined(HAVE_CGNS)

  sprintf(opt_str,
          _(": %s geometry output"), "CGNS");
  _fct_prt(ECS_CMD_OPTION_POST_CGNS, _("[<sub-options>]"), opt_str);

  bft_printf("\n");

#endif /* HAVE_CGNS */


  sprintf(opt_str,
          _(": %s geometry output"), "EnSight Gold");
  _fct_prt(ECS_CMD_OPTION_POST_ENS, _("[<sub-options>]"), opt_str);


#if defined(HAVE_MED)

  bft_printf("\n");

  sprintf(opt_str,
          _(": %s geometry output"), "MED 2.3");
  _fct_prt(ECS_CMD_OPTION_POST_MED, _("[<sub-options>]"), opt_str);

#endif /* HAVE_MED */


  /* Sous-options de sélection de maillage */
  /*---------------------------------------*/

  bft_printf(_("\n\nmesh selection sub-options\n"
               " (-m, -mesh):\n\n"));


  _fct_prt(ECS_CMD_OPTION_FMT_MESH_FILE, _("<keyword>"),
           _(": selection of mesh file format"));


  _fct_prt(ECS_CMD_OPTION_NUM_MESH, "<n>",
           _(": selection of mesh number in file"));
  _fct_prt("", "",
           _("  (if the format allows it)"));

  _fct_prt(ECS_CMD_OPTION_GRP_CEL_MESH, _("<keyword>"),
           _(": add groups of cells"));
  _fct_prt("", "", _("   * based on sections: keyword \""
                     ECS_CMD_KEY_MESH_GRP_SECTION"\""));
  _fct_prt("", "", _("   * based on zones:    keyword \""
                     ECS_CMD_KEY_MESH_GRP_ZONE"\""));
  _fct_prt("", "", _("  (based on format features/conventions)"));

  _fct_prt(ECS_CMD_OPTION_GRP_FAC_MESH, _("<keyword>"),
           _(": add groups of faces"));
  _fct_prt("", "", _("   * based on sections: keyword \""
                     ECS_CMD_KEY_MESH_GRP_SECTION"\""));
  _fct_prt("", "", _("   * based on zones:    keyword \""
                     ECS_CMD_KEY_MESH_GRP_ZONE"\""));
  _fct_prt("", "", _("  (based on format features/conventions)"));


  bft_printf(_("\n\navailable mesh formats\n"
               " (-m, -mesh; -format):\n"));
  bft_printf(_("                                  extension:    keyword:\n"));


  ecs_pre__aff_formats();


  /* Sous-options de post traitement */
  /*---------------------------------*/

  opt_str[0] = '\0';

#if defined(HAVE_CGNS)
  strcat(opt_str, ECS_CMD_OPTION_POST_CGNS);
  strcat(opt_str, ", ");
#endif

  strcat(opt_str, ECS_CMD_OPTION_POST_ENS);

#if defined(HAVE_MED)
  strcat(opt_str, ", ");
  strcat(opt_str, ECS_CMD_OPTION_POST_MED);
#endif

  bft_printf(_("\n\npostprocessing selection sub-options\n"
               " (%s):\n\n"), opt_str);


  _fct_prt(ECS_CMD_OPTION_POST_NO_POLY,
           "", _(": discard polygonal faces (with more"));
  _fct_prt("", "", _("  than 4 edges) or polyhedra"));

  _fct_prt(ECS_CMD_OPTION_POST_SINGLE_PART,
           "", _(": force mesh output on a single"));
  _fct_prt("", "", _("  part (EnSight)"));

  bft_printf("\n");

  _fct_prt(ECS_CMD_OPTION_POST_TEXT,
           "", _(": force text output (EnSight)"));

  _fct_prt(ECS_CMD_OPTION_POST_BIG_ENDIAN,
           "", _(": force binary output to big-endian (EnSight)"));

  bft_printf("\n");

  _fct_prt(ECS_CMD_OPTION_POST_COLOR_TO_GROUP,
           "", _(": transform colors to groups (MED)"));

  bft_printf("\n");

  _fct_prt(ECS_CMD_OPTION_POST_VOLUME,
           "", _(": activate output of volume mesh"));
  sprintf(opt_str, _("  (by default if %s or %s not given)"),
          ECS_CMD_OPTION_POST_BORD, ECS_CMD_OPTION_POST_INFO);
  _fct_prt("", "", opt_str);

  _fct_prt(ECS_CMD_OPTION_POST_BORD,
           "", _(": activate output of boundary mesh"));
  sprintf(opt_str, _("  (by default if %s or %s not given)"),
          ECS_CMD_OPTION_POST_VOLUME, ECS_CMD_OPTION_POST_INFO);
  _fct_prt("", "", opt_str);

  _fct_prt(ECS_CMD_OPTION_POST_INFO,
              "", _(": activate output of information meshes"));
  sprintf(opt_str, _("  (by default if %s or %s not given)"),
          ECS_CMD_OPTION_POST_VOLUME, ECS_CMD_OPTION_POST_BORD);
  _fct_prt("", "", opt_str);


  /* Sous-options de périodicité */
  /*-----------------------------*/

  bft_printf(_("\n\nperiodicity sub-options\n"
               " (-p, --perio):\n\n"));


  _fct_prt(ECS_CMD_OPTION_PERIO_TRANS, _("<3 real numbers>"),
           _(": translation vector"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_PERIO_ROTA, _("[<sub-options>]"),
           _(": rotation defined"));

  _fct_prt("", " ",
           _("  - either by an angle (in degrees) and"));
  _fct_prt("", "["ECS_CMD_OPTION_PERIO_ANGLE", "ECS_CMD_OPTION_PERIO_DIR"]",
           _("    a direction vector, and eventually"));
  _fct_prt("", "["ECS_CMD_OPTION_PERIO_PTINV"]",
           _("    by an invariant point (origin by default)"));
  _fct_prt("", "",
           _("  - either by the rotation matrix (given"));
  _fct_prt("", "["ECS_CMD_OPTION_PERIO_MAT"]",
           _("    line by line), and eventually"));
  _fct_prt("", "["ECS_CMD_OPTION_PERIO_PTINV"]",
           _("    by an invariant point (origin by default)"));


  bft_printf("\n");


  /* Sous-options de recollement de faces */
  /*--------------------------------------*/

  bft_printf(_("\nnon-conformal face joining sub-options\n"
               " (-j, --join, -p, --perio):\n\n"));


  _fct_prt(ECS_CMD_OPTION_RC_FRACTION_DIST, _("<real number>"),
           _(": fraction of the length of the smallest"));

  _fct_prt("", "", _("  edge under which we consider that"));
  _fct_prt("", "", _("  vertices may be fused"));
  sprintf(opt_str, _("  (default value: %s)"), ECS_CMD_INIT_RC_FRACTION_DIST);
  _fct_prt("", "", opt_str);


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_RC_EPSILON_PLAN, _("<real number>"),
           _(": minimal cosine of the normals of 2"));
  sprintf(opt_str, _("  coplanar faces (default value: %s)"),
          ECS_CMD_INIT_RC_EPSILON_PLAN);
  _fct_prt("", "", opt_str);

  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_RC_SEMI_CONFORME, "",
           _(": fast joining algorithm (allowed if"));

  _fct_prt("", "", _("  faces already share vertices)"));


  /* Sous-options de sélection de faces */
  /*------------------------------------*/

  bft_printf(_("\n\nface selection sub-options\n"
               " (for joining, selection, and periodicity):"
               "\n\n"));


  _fct_prt(ECS_CMD_OPTION_FAC_COLOR,
           _("<number(s)>"),
           _(": color numbers of faces to select"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_FAC_GROUP,
           _("<name(s)>"),
           _(": group names of faces to select"));


  bft_printf("\n");


  _fct_prt(ECS_CMD_OPTION_INVSEL, "", _(": invert selection"));


  /* Variables d'environnement */
  /*---------------------------*/

  bft_printf("\n\n%s:\n\n",
             _("Environment variables"));

  bft_printf(_("  CS_PREPROCESS_MIN_EDGE_LEN    : "
               "length under which an edge is considered\n"
               "                                  "
               "degenerate (default: 1.e-15)\n\n"));

  bft_printf(_("  CS_PREPROCESS_MEM_LOG         : "
               "name of memory operations trace file\n\n"));

  bft_printf(_("  CS_PREPROCESS_IGNORE_IDEAS_COO_SYS : "
               "ignore I-deas coordinate systems\n\n"));

  bft_printf(_("  CS_PREPROCESS_JOIN_MAX_SUB_FACES : "
               "number of sub-faces above which we\n"
               "                                  "
               "consider that conformal joining face\n"
               "                                  "
               "reconstruction has gone into an infinite loop\n"
               "                                  "
               "(default: 100)\n\n"));
}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche un message indiquant
 *   qu'une meme option est mentionnee au moins 2 fois
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__aff_opt_en_double(int          argc,
                               char        *argv[],
                               const char  *opt)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  ecs_loc_cmd__aff_titre(argc, argv);

  ecs_loc_cmd__aff_aide();

  bft_error(__FILE__, __LINE__, 0,
            _("Error in command line specification.\n\n"
              "Option \"%s\" is set at least twice."), opt);


}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche un message indiquant
 *   qu'il manque un argument a une option de la ligne de commande
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__aff_manque_arg(int          argc,
                            char        *argv[],
                            const char  *opt)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  ecs_loc_cmd__aff_titre(argc, argv);

  ecs_loc_cmd__aff_aide();

  bft_error(__FILE__, __LINE__, 0,
            _("Error in command line specification.\n\n"
              "Option \"%s\" requires an argument."), opt);

}


/*----------------------------------------------------------------------------
 *  Fonction qui lit les sous-options d'une sélection de maillage
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__lit_arg_maillage(ecs_cmd_t  *cmd,
                              int         argc,
                              char       *argv[],
                              int        *argpos)
{
  int          iarg;
  int          iarg_prec;

  size_t       ific;
  size_t       iloc;

  char        *cle_fmt = NULL;
  bool         bool_fin = false;
  bool         bool_num = false;

  bool         grp_cel_section = false;
  bool         grp_cel_zone    = false;
  bool         grp_fac_section = false;
  bool         grp_fac_zone    = false;

  ecs_int_t         num_maillage = 0;

  const char arg_err_keyword[]
    = N_("Error in command line specification.\n\n"
         "Keyword \"%s\" of option \"%s\" is not recognized.\n");

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  iarg_prec = *argpos - 1;

  ific = cmd->liste_fic_maillage.nbr;

  for (iarg = *argpos; iarg < argc && bool_fin == false; iarg++) {

    /* Sous-option de choix de format */

    if (!strcmp (ECS_CMD_OPTION_FMT_MESH_FILE, argv[iarg])) {

      if (cle_fmt == NULL)
        cle_fmt = argv[iarg + 1];
      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      iarg++; /* Un sous-argument lu -> avancer iarg */

    }

    /* Sous-option de numéro de maillage */

    else if (!strcmp (ECS_CMD_OPTION_NUM_MESH, argv[iarg])) {

      if (bool_num == false)
        bool_num = true;
      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      if (iarg + 1 < argc && atoi(argv[iarg + 1]) > 0)

        num_maillage = atoi(argv[iarg + 1]);

      else {

        ecs_loc_cmd__aff_titre(argc, argv);

        ecs_loc_cmd__aff_aide();

        bft_error(__FILE__, __LINE__, 0,
                  _(arg_err_keyword), argv[iarg + 1], argv[iarg]);

      }

      iarg++; /* Un sous-argument lu -> avancer iarg */

    }

    /* Sous-options de création de groupes de cellules */

    else if (!strcmp (ECS_CMD_OPTION_GRP_CEL_MESH, argv[iarg])) {

      if (   iarg + 1 < argc
          && !strcmp(ECS_CMD_KEY_MESH_GRP_SECTION, argv[iarg + 1]))

        grp_cel_section = true;

      else if (   iarg + 1 < argc
          && !strcmp(ECS_CMD_KEY_MESH_GRP_ZONE, argv[iarg + 1]))

        grp_cel_zone = true;

      else {

        ecs_loc_cmd__aff_titre(argc, argv);

        ecs_loc_cmd__aff_aide();

        bft_error(__FILE__, __LINE__, 0,
                  _(arg_err_keyword), argv[iarg + 1], argv[iarg]);

      }

      iarg++; /* Un sous-argument lu -> avancer iarg */

    }

    /* Sous-options de création de groupes de faces */

    else if (!strcmp (ECS_CMD_OPTION_GRP_FAC_MESH, argv[iarg])) {

      if (   iarg + 1 < argc
          && !strcmp(ECS_CMD_KEY_MESH_GRP_SECTION, argv[iarg + 1]))

        grp_fac_section = true;

      else if (   iarg + 1 < argc
          && !strcmp(ECS_CMD_KEY_MESH_GRP_ZONE, argv[iarg + 1]))

        grp_fac_zone = true;

      else {

        ecs_loc_cmd__aff_titre(argc, argv);

        ecs_loc_cmd__aff_aide();

        bft_error(__FILE__, __LINE__, 0,
                  _(arg_err_keyword), argv[iarg + 1], argv[iarg]);

      }

      iarg++; /* Un sous-argument lu -> avancer iarg */

    }

    /* Nom de fichier */

    else if (argv[iarg][0] != '-') {

      BFT_REALLOC(cmd->liste_fic_maillage.val, ific + 1, char *);

      BFT_MALLOC(cmd->liste_fic_maillage.val[ific],
                 strlen(argv[iarg]) + 1, char);
      strcpy(cmd->liste_fic_maillage.val[ific], argv[iarg]);

      ific++;

    }

    /* Autre option (pas une sous-option) -> on a fini */

    else {

      iarg--;
      bool_fin = true;

    }

  }

  if (cmd->liste_fic_maillage.nbr == ific)
    ecs_loc_cmd__aff_manque_arg(argc, argv, argv[iarg_prec]);

  BFT_REALLOC(cmd->liste_num_maillage, ific + 1, ecs_int_t);

  BFT_REALLOC(cmd->liste_fmt_maillage, ific + 1, ecs_pre_format_t);

  BFT_REALLOC(cmd->liste_grp_maillage, (ific + 1) * 4, bool);

  for (iloc = cmd->liste_fic_maillage.nbr; iloc < ific; iloc++)
    cmd->liste_num_maillage[iloc] = num_maillage;

  for (iloc = cmd->liste_fic_maillage.nbr; iloc < ific; iloc++)
     cmd->liste_fmt_maillage[iloc]
       = ecs_pre__type_format(cmd->liste_fic_maillage.val[iloc],
                              cle_fmt);

  for (iloc = cmd->liste_fic_maillage.nbr; iloc < ific; iloc++) {
    cmd->liste_grp_maillage[iloc*4    ] = grp_cel_section;
    cmd->liste_grp_maillage[iloc*4 + 1] = grp_cel_zone;
    cmd->liste_grp_maillage[iloc*4 + 2] = grp_fac_section;
    cmd->liste_grp_maillage[iloc*4 + 3] = grp_fac_zone;
  }

  cmd->liste_fic_maillage.nbr = ific;

  *argpos = iarg - 1;

}


/*----------------------------------------------------------------------------
 *  Fonction qui lit les sous-options d'un post traitement
 *----------------------------------------------------------------------------*/

static ecs_cmd_post_t  *
ecs_loc_cmd__lit_arg_post(int    argc,
                          char  *argv[],
                          int   *argpos)
{
  int         iarg;
  int         iarg_prec;

  bool        bool_fin = false;

  ecs_cmd_post_t  * cmd_post;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  iarg_prec = *argpos;

  cmd_post = NULL;

  BFT_MALLOC(cmd_post, 1, ecs_cmd_post_t);

  cmd_post->no_poly        = false;
  cmd_post->simple         = false;
  cmd_post->text           = false;
  cmd_post->big_endian     = false;
  cmd_post->color_to_group = false;

  cmd_post->volume = false;
  cmd_post->bord   = false;
  cmd_post->info   = false;

  for (iarg = *argpos + 1; iarg < argc && bool_fin == false; iarg++) {

    /* Sous-option de choix de format */

    if      (!strcmp (ECS_CMD_OPTION_POST_NO_POLY, argv[iarg])) {
      cmd_post->no_poly = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_SINGLE_PART, argv[iarg])) {
      cmd_post->simple = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_TEXT, argv[iarg])) {
      cmd_post->text = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_BIG_ENDIAN, argv[iarg])) {
      cmd_post->big_endian = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_COLOR_TO_GROUP, argv[iarg])) {
      cmd_post->color_to_group = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_VOLUME, argv[iarg])) {
      cmd_post->volume = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_BORD, argv[iarg])) {
      cmd_post->bord = true;
    }

    else if (!strcmp (ECS_CMD_OPTION_POST_INFO, argv[iarg])) {
      cmd_post->info = true;
    }

    else {

      /* Autre option (pas une sous-option) -> on a fini */

      iarg--;
      bool_fin = true;

    }

  }

  /* Si aucune option de filtrage du post traitement n'est
     activée, tous les types post traitements sont actifs */

  if (   cmd_post->volume == false
      && cmd_post->bord   == false
      && cmd_post->info   == false) {
    cmd_post->volume = true;
    cmd_post->bord   = true;
    cmd_post->info   = true;
  }

  *argpos = iarg - 1;

  return cmd_post;

}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne un booleen selon qu'un argument est de type
 *  numerique (true) ou non (false)
 *----------------------------------------------------------------------------*/

static bool
ecs_loc_cmd__est_num_arg(char  *aux_arg)
{

  bool         est_num;


  est_num = false;

  if (aux_arg[0] == '-') {
    if ( aux_arg[1] == '0' || aux_arg[1] == '1' || aux_arg[1] == '2' ||
         aux_arg[1] == '3' || aux_arg[1] == '4' || aux_arg[1] == '5' ||
         aux_arg[1] == '6' || aux_arg[1] == '7' || aux_arg[1] == '8' ||
         aux_arg[1] == '9' || aux_arg[1] == '.') {
      est_num = true;
    };
  }
  else if ( aux_arg[0] == '0' || aux_arg[0] == '1' || aux_arg[0] == '2' ||
            aux_arg[0] == '3' || aux_arg[0] == '4' || aux_arg[0] == '5' ||
            aux_arg[0] == '6' || aux_arg[0] == '7' || aux_arg[0] == '8' ||
            aux_arg[0] == '9' || aux_arg[0] == '.') {
    est_num = true;
  }

  return est_num;

}


/*----------------------------------------------------------------------------
 *  Fonction qui lit les argument des sous-options d'un recollement, ...
 *  et retourne le nombre d'argument de type numerique
 *----------------------------------------------------------------------------*/

static ecs_int_t
ecs_loc_cmd__nbr_arg_num_ss_opt(int   argc,
                                char *argv[],
                                int  *argpos)
{

  int           nbr_arg_num;
  int           iarg   ;


  nbr_arg_num = 0;
  iarg = *argpos;

  while (iarg < argc-1 && ecs_loc_cmd__est_num_arg(argv[iarg+1])) {
    iarg++;
    nbr_arg_num++;
  }

  return nbr_arg_num;

}


/*----------------------------------------------------------------------------
 *  Fonction qui lit les sous-options de selection de faces
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__lit_arg_select_fac(ecs_select_fac_t  *select_fac,
                                int                argc,
                                char              *argv[],
                                int               *argpos)
{

  bool             bool_fin;

  int              iarg;
  int              iarg_sauv;
  int              igrp;
  int              lng;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisation */

  bool_fin = false;

  /* Boucle sur les arguments */

  for (iarg = *argpos; iarg < argc && bool_fin == false; iarg++) {


    if (!strcmp (ECS_CMD_OPTION_FAC_COLOR, argv[iarg])) {

      iarg_sauv = iarg;

      while (argc - 1 > iarg && strncmp(argv[iarg + 1], "-", 1)) {

        BFT_REALLOC(select_fac->liste_couleur_fac.val,
                    select_fac->liste_couleur_fac.nbr + 1, ecs_int_t);

        select_fac->liste_couleur_fac.val[select_fac->liste_couleur_fac.nbr++]
          = atoi(argv[++iarg]);

      }

      if (iarg == iarg_sauv) {
        ecs_loc_cmd__aff_manque_arg(argc, argv, argv[iarg]);
      }

    }
    else if (!strcmp (ECS_CMD_OPTION_FAC_GROUP, argv[iarg])) {

      iarg_sauv = iarg;

      while (argc - 1 > iarg && strncmp(argv[iarg + 1], "-", 1)) {

        BFT_REALLOC(select_fac->liste_groupe_fac.val,
                    select_fac->liste_groupe_fac.nbr + 1, char *);

        lng = strlen(argv[iarg + 1]);
        igrp = select_fac->liste_groupe_fac.nbr;
        BFT_MALLOC(select_fac->liste_groupe_fac.val[igrp], lng + 1, char);
        strcpy(select_fac->liste_groupe_fac.val[igrp], argv[++iarg]);

        select_fac->liste_groupe_fac.nbr++;

      }

      if (iarg == iarg_sauv) {
        ecs_loc_cmd__aff_manque_arg(argc, argv, argv[iarg]);
      }

    }
    else if (!strcmp (ECS_CMD_OPTION_INVSEL, argv[iarg])) {

      select_fac->inv_selection = true;

    }
    else {

      bool_fin = true;

    }

  }


  if (bool_fin == true) {

    /* Il y a encore au moins une option a lire */
    /*  qui ne fait pas partie des sous-options */

    *argpos = iarg - 2;

  }
  else {

    /* Il n'y a plus d'options a lire */

    *argpos = iarg;

  }


}


/*----------------------------------------------------------------------------
 *  Lecture des sous-options d'un recollement conforme ou d'une périodicité
 *----------------------------------------------------------------------------*/

static ecs_cmd_rc_t *
ecs_loc_cmd__lit_arg_rc_perio(int          argc,
                              char        *argv[],
                              int         *argpos,
                              ecs_int_t    num_rc_perio,
                              bool         bool_perio)
{
  bool                 bool_fin;
  bool                 bool_fin_rot;

  bool                 angle_est_present;
  bool                 dir_est_present;
  bool                 mat_est_present;

  int                  iarg;
  int                  iarg_sauv;
  int                  ind;
  int                  ind_i;
  int                  ind_j;

  ecs_int_t            nbr_rot;
  ecs_int_t            nbr_trans;

  ecs_real_t           fraction_dist;
  ecs_real_t           epsilon_plan;

  ecs_param_perio_t  * param_perio = NULL;

  ecs_cmd_rc_t       * cmd_rc;


  /* Def d'une macro calculant la norme d'un vecteur pour
     tester la non-nullite de celui-ci */

  enum {X, Y, Z};

#define ECS_LOC_MODULE(vect) \
     sqrt(vect[X] * vect[X] + vect[Y] * vect[Y] + vect[Z] * vect[Z])


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(cmd_rc, 1, ecs_cmd_rc_t);

  /* Initialisation de la structure */

  if (bool_perio == true)
    cmd_rc->param_rc.num_rc = 0;
  else
    cmd_rc->param_rc.num_rc = num_rc_perio;

  cmd_rc->param_rc.fraction_dist = atof(ECS_CMD_INIT_RC_FRACTION_DIST);
  cmd_rc->param_rc.epsilon_plan  = atof(ECS_CMD_INIT_RC_EPSILON_PLAN);
  cmd_rc->param_rc.param_perio   = NULL;
  cmd_rc->param_rc.semi_conforme = false;

  cmd_rc->select_fac_rc.inv_selection = false;
  cmd_rc->select_fac_rc.type_sel      = ECS_SELECT_FAC_BORD;

  cmd_rc->select_fac_rc.liste_couleur_fac.nbr = 0;
  cmd_rc->select_fac_rc.liste_couleur_fac.val = NULL;

  cmd_rc->select_fac_rc.liste_groupe_fac.nbr  = 0;
  cmd_rc->select_fac_rc.liste_groupe_fac.val  = NULL;


  /* Initialisation de la sous-structure de périodicité */

  if (bool_perio == true) {

    BFT_MALLOC(param_perio, 1, ecs_param_perio_t);

    param_perio->num_perio  = num_rc_perio;

    param_perio->type_perio = ECS_PERIO_TYPE_NUL;
    param_perio->angle      = 0.0;

    for (ind = 0; ind < 3; ind++) {
      param_perio->translation[ind] = 0.0;
      param_perio->direction[ind]   = 0.0;
      param_perio->point_inv[ind]   = 0.0;
    }

    for (ind_i = 0; ind_i < 3; ind_i++) {
      for (ind_j = 0; ind_j < 3; ind_j++) {
        if (ind_i != ind_j)
          param_perio->matrice[ind_i][ind_j] = 0.0;
        else
          param_perio->matrice[ind_i][ind_j] = 1.0;
      }
    }

  }


  /* Initialisation d'indicateurs d'état */

  angle_est_present = false;
  dir_est_present   = false;
  mat_est_present   = false;

  bool_fin = false;

  nbr_rot   = 0;
  nbr_trans = 0;


  /*--------------------------*/
  /* Boucle sur les arguments */
  /*--------------------------*/

  for (iarg = *argpos; iarg < argc && bool_fin == false; iarg++) {

    /* Parametres du recollement périodique */
    /*--------------------------------------*/

    if (!strcmp (ECS_CMD_OPTION_RC_FRACTION_DIST, argv[iarg])) {

      switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

      case 0 :

        cmd_rc->param_rc.fraction_dist = atof(ECS_CMD_INIT_RC_FRACTION_DIST);
        break;

      case 1 :

        fraction_dist = atof(argv[++iarg]);
        /* `atof()' renvoie 0 si un `double' ne peut pas être reconnu */

        if (fraction_dist <= 0.                              ||
            fraction_dist <= ECS_CMD_INIT_RC_FRACTION_DIST_I) {

          ecs_loc_cmd__aff_titre(argc, argv);

          ecs_loc_cmd__aff_aide();

          bft_error
            (__FILE__, __LINE__, 0,
             _("Error in command line specification.\n\n"
               "The conformal joining parameter representing the fraction\n"
               "of the minimal incident edge length for a given vertex\n"
               "must be a real number greater than %g\n"
               "(and for a normal use, less than %g).\n"
               "Parameter value read: %s"),
             ECS_CMD_INIT_RC_FRACTION_DIST_I,
             ECS_CMD_INIT_RC_FRACTION_DIST_S,
             argv[iarg]);
        }
        else
          cmd_rc->param_rc.fraction_dist = fraction_dist;

        break;

      default :

        bft_error(__FILE__, __LINE__, 0,
                  _("Error in command line specification.\n\n"
                    "Option \"%s\" requires a real number as a parameter."),
                    ECS_CMD_OPTION_RC_FRACTION_DIST);

      }

    }
    else if (!strcmp (ECS_CMD_OPTION_RC_EPSILON_PLAN, argv[iarg])) {

      switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

      case 0 :

        cmd_rc->param_rc.epsilon_plan = atof(ECS_CMD_INIT_RC_EPSILON_PLAN);
        break;

      case 1 :

        epsilon_plan = atof(argv[++iarg]);
        /* `atof()' renvoie 0 si un `double' ne peut pas etre reconnu */

        if (epsilon_plan <= 0.                             ||
            epsilon_plan <  ECS_CMD_INIT_RC_EPSILON_PLAN_I ||
            epsilon_plan >  ECS_CMD_INIT_RC_EPSILON_PLAN_S) {

          ecs_loc_cmd__aff_titre(argc, argv);

          ecs_loc_cmd__aff_aide();

          bft_error(__FILE__, __LINE__, 0,
                    _("Error in command line specification.\n\n"
                      "The conformal joining parameter used to determine if "
                      "2 edges\n"
                      "are coplanar must be a real number in the interval "
                      " [%g,%g].\n"
                      "Parameter value read: %s"),
                    ECS_CMD_INIT_RC_EPSILON_PLAN_I,
                    ECS_CMD_INIT_RC_EPSILON_PLAN_S,
                    argv[iarg]);
        }
        else
          cmd_rc->param_rc.epsilon_plan = epsilon_plan;

        break;

      default :

        bft_error(__FILE__, __LINE__, 0,
                  _("Error in command line specification.\n\n"
                    "Option \"%s\" requires a real number as a parameter."),
                  ECS_CMD_OPTION_RC_EPSILON_PLAN);

      }

    }
    else if (!strcmp (ECS_CMD_OPTION_RC_SEMI_CONFORME, argv[iarg])) {

      cmd_rc->param_rc.semi_conforme = true;

    }


    /* En cas de périodicité */
    /*-----------------------*/

    /* Cas d'une translation */

    else if (   bool_perio == true
             && (!strcmp (ECS_CMD_OPTION_PERIO_TRANS, argv[iarg]))) {

      nbr_trans++;

      if (nbr_trans > 1)
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      if (param_perio->type_perio == ECS_PERIO_TYPE_NUL)
        param_perio->type_perio = ECS_PERIO_TYPE_TRANS;

      switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

      case 3 :

        for (ind = 0; ind < 3; ind++)
          param_perio->translation[ind] = atof(argv[++iarg]);

        if (ECS_LOC_MODULE(param_perio->translation) < 1.e-12)
          bft_error(__FILE__, __LINE__, 0,
                    _("Error in command line specification.\n\n"
                      "The translation vector may not be null."));

        break;

      default :

        bft_error(__FILE__, __LINE__, 0,
                  _("Error in command line specification.\n\n"
                    "The translation vector must be defined by 3 real number "
                    "parameters."));

      }

    } /* Fin : Cas d'une translation */


    /* Cas d'une rotation */

    else if (   bool_perio == true
             && (!strcmp (ECS_CMD_OPTION_PERIO_ROTA, argv[iarg]))) {

      nbr_rot++;

      if (nbr_rot > 1)
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      param_perio->type_perio = ECS_PERIO_TYPE_ROTA;
      bool_fin_rot = false;

      while (bool_fin_rot == false && ++iarg < argc) {

        /* Point invariant */

        if (!strcmp (ECS_CMD_OPTION_PERIO_PTINV, argv[iarg])) {

          switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

          case 3 :

            for (ind = 0; ind < 3; ind++)
              param_perio->point_inv[ind] = atof(argv[++iarg]);

            break;

          default :

            bft_error(__FILE__, __LINE__, 0,
                      _("Error in command line specification.\n\n"
                        "The invariant point must be defined by 3 real number "
                        "parameters."));

          }

        }

        /* Angle */

        else if (!strcmp (ECS_CMD_OPTION_PERIO_ANGLE, argv[iarg])) {

          switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

          case 1 :

            angle_est_present = true;
            param_perio->angle = atof(argv[++iarg]);

            if (   param_perio->angle < ECS_CMD_INIT_PERIO_ROTA_ANGLE_I
                || param_perio->angle > ECS_CMD_INIT_PERIO_ROTA_ANGLE_S
                || ECS_ABS(param_perio->angle) < 1.e-12)

              bft_error(__FILE__, __LINE__, 0,
                        _("Error in command line specification.\n\n"
                          "The rotation angle must be a nonzero real number\n"
                          "in interval ]%g,%g[.\n"
                          "Parameter value read: %s"),
                        ECS_CMD_INIT_PERIO_ROTA_ANGLE_I,
                        ECS_CMD_INIT_PERIO_ROTA_ANGLE_S,
                        argv[iarg]);

            break;

          default :

            bft_error(__FILE__, __LINE__, 0,
                      _("Error in command line specification.\n\n"
                        "The angle requires a real number as a parameter."));

          }

        }

        /* Direction */

        else if (!strcmp (ECS_CMD_OPTION_PERIO_DIR, argv[iarg])) {

          switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

          case 3 :

            dir_est_present = true;

            for (ind = 0; ind < 3; ind++)
              param_perio->direction[ind] = atof(argv[++iarg]);

            if (ECS_LOC_MODULE(param_perio->direction) < 1.e-12)
              bft_error(__FILE__, __LINE__, 0,
                        _("Error in command line specification.\n\n"
                          "The rotation vector may not be null."));

            break;

          default :

            bft_error(__FILE__, __LINE__, 0,
                      _("Error in command line specification.\n\n"
                        "The rotation vector must be defined by 3 real number "
                        "parameters."));

          }

        }

        /* Matrice */

        else if (!strcmp (ECS_CMD_OPTION_PERIO_MAT, argv[iarg])) {

          switch(ecs_loc_cmd__nbr_arg_num_ss_opt(argc,argv,&iarg)) {

          case 9 :

            mat_est_present = true;

            for (ind_i = 0; ind_i < 3; ind_i++)
              for (ind_j = 0; ind_j < 3; ind_j++)
                param_perio->matrice[ind_i][ind_j] = atof(argv[++iarg]);

            break;

          default :

            bft_error(__FILE__, __LINE__, 0,
                      _("Error in command line specification.\n\n"
                        "The rotation marix must be defined by 9 real number "
                        "parameters."));

          }

        }
        else

          bool_fin_rot = true;

      }

      iarg--;


    } /* Fin : Cas d'une rotation */


    /* Fin des option périodiques */
    /*----------------------------*/


    else {

      iarg_sauv = iarg;

      ecs_loc_cmd__lit_arg_select_fac(&cmd_rc->select_fac_rc,
                                      argc,
                                      argv,
                                      &iarg);

      /* Si on n'a pas trouvé d'option concernant la sélection des faces */
      /*  il faut sortir de la fonction                                  */

      if (iarg <= iarg_sauv) {

        bool_fin = true;

      }

    }

  }


  *argpos = iarg - 1;


  if (param_perio != NULL) {

    /* On vérifie que l'utilisateur a renseigné soit les paramètres
       d'une translation, soit ceux d'une rotation */

    if (param_perio->type_perio == ECS_PERIO_TYPE_NUL)

      bft_error(__FILE__, __LINE__, 0,
                _("Error in command line specification.\n\n"
                  "With the \"%s\" option, it is necessary to set\n"
                  "options \"%s\" or \"%s\"."),
                ECS_CMD_OPTION_PERIO_2,
                ECS_CMD_OPTION_PERIO_TRANS,
                ECS_CMD_OPTION_PERIO_ROTA);

    /* On vérifie enfin la cohérence des paramètres de la rotation */

    if (param_perio->type_perio == ECS_PERIO_TYPE_ROTA)

      if (!(mat_est_present == false &&
            angle_est_present == true && dir_est_present == true) &&
          !(mat_est_present == true &&
          angle_est_present == false && dir_est_present == false))

        bft_error(__FILE__, __LINE__, 0,
                  _("Error in command line specification.\n\n"
                    "The rotation may be defined in 2 ways:\n"
                    "-either by an angle and direction (-angle . -dir . . .)\n"
                    "-either by a matrix (-matrice . . .  . . .  . . .)\n"
                    "In either case, an invariant point may be chosen "
                    "(-invpt . . .)"));


    /* On étend les paramètres de recollement par ceux de périodicité */

    cmd_rc->param_rc.param_perio = param_perio;

  }


  return cmd_rc;


}


/*----------------------------------------------------------------------------
 *  Fonction qui lit les sous-options d'une sélection simple
 *----------------------------------------------------------------------------*/

static ecs_select_fac_t *
ecs_loc_cmd__lit_arg_select_int(int    argc,
                                char  *argv[],
                                int   *argpos)
{

  ecs_select_fac_t   * cmd_select;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(cmd_select, 1, ecs_select_fac_t);


  /* Initialisation */

  cmd_select->inv_selection = false;
  cmd_select->type_sel      = ECS_SELECT_FAC_INT;

  cmd_select->liste_couleur_fac.nbr = 0;
  cmd_select->liste_couleur_fac.val = NULL;

  cmd_select->liste_groupe_fac.nbr  = 0;
  cmd_select->liste_groupe_fac.val  = NULL;


  ecs_loc_cmd__lit_arg_select_fac(cmd_select,
                                  argc,
                                  argv,
                                  argpos);

  return cmd_select;

}


/*----------------------------------------------------------------------------
 *  Fonction qui initialise les options de commande
 *----------------------------------------------------------------------------*/

static ecs_cmd_t *
ecs_loc_cmd__initialise(void)
{

  size_t   lng;

  ecs_cmd_t  * cmd;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(cmd, 1, ecs_cmd_t);


  /* Initialisations */
  /*=================*/


  lng =   strlen(ECS_CMD_POST_CASE_DEFAULT) + 1;
  BFT_MALLOC(cmd->nom_cas, lng, char);
  strcpy(cmd->nom_cas,
         ECS_CMD_POST_CASE_DEFAULT);

  cmd->liste_num_maillage            = NULL;
  cmd->liste_fmt_maillage            = NULL;
  cmd->liste_grp_maillage            = NULL;

  cmd->liste_fic_maillage.val        = NULL;
  cmd->liste_fic_maillage.nbr        = 0;

#if defined(HAVE_CGNS)

  cmd->post_cgns                     = NULL;

#endif /* HAVE_CGNS */

  cmd->post_ens                      = NULL;

#if defined(HAVE_MED)

  cmd->post_med                      = NULL;

#endif /* HAVE_MED */

  cmd->correct_orient                = false;

  cmd->nbr_dump                      = 0;

  cmd->sim_comm                      = false;

  cmd->nbr_cmd_rc                    = 0;
  cmd->liste_cmd_rc                  = NULL;

  cmd->nbr_cmd_perio                 = 0;

  cmd->cmd_select_fac_int            = NULL;


  return cmd;

}


/*----------------------------------------------------------------------------
 *  Fonction qui affiche la configuration du cas de lancement
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__aff_config(ecs_cmd_t  *cmd)
{

  size_t           ific;
  time_t           date;

#if !defined(PATH_MAX)
#define PATH_MAX 1024
#endif

  char             str_date     [ECS_STR_SIZE]  = "";
  char             str_system   [ECS_STR_SIZE]  = "";
  char             str_machine  [ECS_STR_SIZE]  = "";
  char             str_ram      [ECS_STR_SIZE]  = "";
  char             str_user     [ECS_STR_SIZE]  = "";
  char             str_directory[PATH_MAX] = "";

  int              l_user;
  size_t           ram;

#if defined(HAVE_UNAME)
  struct utsname   sys_config;
#endif

#if defined(HAVE_GETPWUID) && defined(HAVE_GETEUID)
  struct passwd   *pwd_user = NULL;
#endif


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Determination de la configuration du cas de lancement */
  /*-------------------------------------------------------*/


  /* Date */

  if (time(&date) == -1 ||
      strftime(str_date, ECS_STR_SIZE - 1, "%c", localtime(&date)) == 0)
    strcpy(str_date, "");


  /* Systeme et machine */

#if defined(HAVE_UNAME)
  if (uname(&sys_config) != -1) {
    strcpy(str_system, sys_config.sysname);
    strcat(str_system, " ");
    strcat(str_system, sys_config.release);
    strcpy(str_machine, sys_config.nodename);
  }
#endif


  /* Nom de login */

#if defined(HAVE_GETPWUID) && defined(HAVE_GETEUID)

  int l_info;

  pwd_user = NULL;

  pwd_user = getpwuid(geteuid());

  if (pwd_user != NULL) {

    str_user[ECS_STR_SIZE - 1] = '\0';
    strncpy(str_user, pwd_user->pw_name, ECS_STR_SIZE - 1);

    if (pwd_user->pw_gecos != NULL) {

      l_user = strlen(str_user);
      for (l_info = 0;
           (   pwd_user->pw_gecos[l_info] != '\0'
            && pwd_user->pw_gecos[l_info] != ',');
           l_info++);

      if (l_user + l_info + 3 < ECS_STR_SIZE) {
        strcat(str_user, " (");
        strncpy(str_user + l_user + 2, pwd_user->pw_gecos, l_info);
        str_user[l_user + 2 + l_info]     = ')';
        str_user[l_user + 2 + l_info + 1] = '\0';
      }

    }

  }

#endif /* defined(HAVE_GETPWUID) && defined(HAVE_GETEUID) */


  /* Mémoire vive */

  ram = bft_sys_info_mem_ram();
  if (ram > 1)
    sprintf(str_ram, "%lu", (unsigned long)ram);

  /* Repertoire courant */

  str_directory[0] = '\0';

#if defined(HAVE_GETCWD)
  if (getcwd(str_directory, PATH_MAX) == NULL)
    str_directory[0] = '\0';
#endif


  /* Affichage de la configuration du cas de lancement */
  /*---------------------------------------------------*/

  bft_printf("\n\n%s\n", _("Case configuration\n"
                           "------------------\n"));

  if (strlen(str_date) > 0) {
    bft_printf("  ");
    ecs_print_padded_str(_("Date"), 19);
    bft_printf(" : %s\n", str_date);
  }

  if (strlen(str_system) > 0) {
    bft_printf("  ");
    ecs_print_padded_str(_("System"), 19);
    bft_printf(" : %s\n", str_system);
  }

  if (strlen(str_machine) > 0) {
    bft_printf("  ");
    ecs_print_padded_str(_("Machine"), 19);
    bft_printf(" : %s\n", str_machine);
  }

  bft_printf("  ");
  ecs_print_padded_str(_("Processor"), 19);
  bft_printf(" : %s\n", bft_sys_info_cpu());

  if (ram > 0) {
    bft_printf("  ");
    ecs_print_padded_str(_("Memory"), 19);
    bft_printf(" : %s\n", str_ram);
  }

  if (strlen(str_user) > 0) {
    bft_printf("  ");
    ecs_print_padded_str(_("User"), 19);
    bft_printf(" : %s\n", str_user);
  }

  if (strlen(str_directory) > 0) {
    bft_printf("  ");
    ecs_print_padded_str(_("Directory"), 19);
    bft_printf(" : %s\n", str_directory);
  }

  bft_printf("\n");

  bft_printf("  ");
  ecs_print_padded_str(_("Case name"), 19);
  bft_printf(" : %s\n", cmd->nom_cas);

  bft_printf("  ");
  ecs_print_padded_str(_("Mesh file"), 19);
  bft_printf(" : %s\n", cmd->liste_fic_maillage.val[0]);

  for (ific = 1; ific < cmd->liste_fic_maillage.nbr; ific++)
    bft_printf("  %-*s   %s\n", 19, "", cmd->liste_fic_maillage.val[ific]);

  bft_printf("\n");

}


/*----------------------------------------------------------------------------
 *  Fonction qui diagnostique l'erreur de non accessibilite
 *   du fichier dont le nom est donne
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__diagnost_non_acces(const char  *fic_name,
                                const char  *msg_cmd_err)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


#if defined(ENOENT)
  if (errno == ENOENT)
    bft_error(__FILE__, __LINE__, 0,
              msg_cmd_err, fic_name, _("The file does not exist."));
#endif

#if defined(EACCES)
  if (errno == EACCES)
    bft_error(__FILE__, __LINE__, 0,
              msg_cmd_err, fic_name,
              _("Read permission on this file is refused."));
#endif

#if defined(ENAMETOOLONG)
  if (errno == ENAMETOOLONG)
    bft_error(__FILE__, __LINE__, 0,
              msg_cmd_err, fic_name,
              _("The filename is too long (system limit)."));
#endif

  bft_error(__FILE__, __LINE__, errno,
            msg_cmd_err, fic_name, "");
}


/*----------------------------------------------------------------------------
 *  Fonction qui verifie que le nom du fichier donne
 *   correspond bien a un fichier physiquement existant
 *----------------------------------------------------------------------------*/

static void
ecs_loc_cmd__teste_exist_fic(const char  *fic_name,
                             const char  *msg_cmd_err)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


#if defined(HAVE_STAT)

  struct stat buf;

  if (stat(fic_name, &buf) != 0) {

    ecs_loc_cmd__diagnost_non_acces(fic_name,
                                    msg_cmd_err);

  }
  else {

    if (S_ISREG(buf.st_mode) != true) {

      bft_error(__FILE__, __LINE__, 0,
                msg_cmd_err,
                fic_name, _("The file is not a regular file."));

    }

  }

#else /* HAVE_STAT */

  FILE * fic;

  if ((fic = fopen(fic_name, "r")) == NULL) {

    ecs_loc_cmd__diagnost_non_acces(fic_name,
                                    msg_cmd_err);

  }
  else {

    fclose(fic);

  }

#endif /* HAVE_STAT */

}


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui lit la ligne de commande
 *----------------------------------------------------------------------------*/

ecs_cmd_t * ecs_cmd__lit_arg
(
 int     argc,
 char  * argv[]
)
{

  int        iarg;
  size_t     ific;
  size_t     lng;
  bool       bool_cmd_option_case;
  bool       bool_cmd_option_cwd;
  bool       bool_cmd_option_help;
  bool       bool_cmd_option_input_file;
  bool       bool_cmd_option_mesh_file;
  bool       bool_cmd_option_output_file;
  bool       bool_cmd_option_ncs_file;
  bool       bool_cmd_option_parall;
  bool       bool_cmd_option_verif;
  bool       bool_cmd_option_version;

  ecs_cmd_t  *cmd;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /*-----------------*/
  /* Initialisations */
  /*-----------------*/

  bool_cmd_option_case        = false;
  bool_cmd_option_input_file  = false;
  bool_cmd_option_cwd         = false;
  bool_cmd_option_help        = false;
  bool_cmd_option_mesh_file   = false;
  bool_cmd_option_output_file = false;
  bool_cmd_option_parall      = false;
  bool_cmd_option_ncs_file    = false;
  bool_cmd_option_verif       = false;
  bool_cmd_option_version     = false;


  /*----------------------------------------*/
  /* Initialisation des options de commande */
  /*----------------------------------------*/

  cmd = ecs_loc_cmd__initialise();


  /*---------------------------------------------*/
  /* Lecture des options de la ligne de commande */
  /*---------------------------------------------*/

  for (iarg = 1; iarg < argc; iarg++) {


    if (!strcmp (ECS_CMD_OPTION_CASE, argv[iarg])) {

      if (bool_cmd_option_case == false)
        bool_cmd_option_case = true;
      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);


      if (argc - 1 > iarg && strncmp(argv[iarg + 1], "-", 1)) {

        lng = strlen(argv[++iarg]) + 1;
        BFT_REALLOC(cmd->nom_cas, lng, char);
        strcpy(cmd->nom_cas, argv[iarg]);

      }
      else {
        ecs_loc_cmd__aff_manque_arg(argc, argv, argv[iarg]);
      }

    }
    else if (!strcmp (ECS_CMD_OPTION_CWD, argv[iarg])) {


      if (bool_cmd_option_cwd == false)
        bool_cmd_option_cwd = true;
      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);


#if defined(HAVE_CHDIR)

      if (argc - 1 > iarg && strncmp(argv[iarg + 1], "-", 1)) {

        if (chdir(argv[++iarg]) != 0) {

          ecs_loc_cmd__aff_titre(argc, argv);

          bft_error(__FILE__, __LINE__, errno,
                    _("Failed to switch to directory :\n%s"),
                    &argv[iarg][2]);
        }

      }
      else {
        ecs_loc_cmd__aff_manque_arg(argc, argv, argv[iarg]);
      }

#else

      ecs_loc_cmd__aff_titre(argc, argv);

      ecs_loc_cmd__aff_aide();

      bft_error(__FILE__, __LINE__, 0,
                _("Error in command line specification.\n\n"
                  "Option \"%s\" is not available."),
                ECS_CMD_OPTION_CWD);

#endif /* HAVE_CHDIR */


    }
    else if (!strcmp (ECS_CMD_OPTION_DUMP, argv[iarg])) {

      /* Option non documentee */

      cmd->nbr_dump = 1;

      /* 1 argument optionnel : nombre n d'elements affiches en echo */

      if (argc -1 > iarg && strncmp(argv[iarg + 1], "-", 1)) {

        cmd->nbr_dump = (ecs_int_t) atoi(argv[++iarg]);

      }

    }
    else if (!strcmp (ECS_CMD_OPTION_NULL_COMM_1, argv[iarg]) ||
             !strcmp (ECS_CMD_OPTION_NULL_COMM, argv[iarg])) {

      cmd->sim_comm = true;

    }
    else if (!strcmp(ECS_CMD_OPTION_HELP_1, argv[iarg]) ||
             !strcmp(ECS_CMD_OPTION_HELP, argv[iarg])) {

      bool_cmd_option_help = true;

    }

#if defined(HAVE_CGNS)

    else if (!strcmp (ECS_CMD_OPTION_POST_CGNS, argv[iarg])) {

      if (cmd->post_cgns == NULL)
        cmd->post_cgns = ecs_loc_cmd__lit_arg_post(argc,
                                                   argv,
                                                   &iarg);

      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

    }

#endif /* HAVE_CGNS */

    else if (!strcmp (ECS_CMD_OPTION_POST_ENS, argv[iarg])) {

      if (cmd->post_ens == NULL)
        cmd->post_ens = ecs_loc_cmd__lit_arg_post(argc,
                                                  argv,
                                                  &iarg);

      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

    }

#if defined(HAVE_MED)

    else if (!strcmp (ECS_CMD_OPTION_POST_MED, argv[iarg])) {

      if (cmd->post_med == NULL)
        cmd->post_med = ecs_loc_cmd__lit_arg_post(argc,
                                                  argv,
                                                  &iarg);

      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

    }

#endif /* HAVE_MED */

    else if (!strcmp (ECS_CMD_OPTION_MESH_FILE_1, argv[iarg]) ||
             !strcmp (ECS_CMD_OPTION_MESH_FILE, argv[iarg])) {

      if (bool_cmd_option_mesh_file == false)
        bool_cmd_option_mesh_file = true;

      iarg++;

      ecs_loc_cmd__lit_arg_maillage(cmd,
                                    argc,
                                    argv,
                                    &iarg);

    }

    else if (!strcmp (ECS_CMD_OPTION_INPUT_FILE_1, argv[iarg]) ||
             !strcmp (ECS_CMD_OPTION_INPUT_FILE, argv[iarg])) {

      if (bool_cmd_option_input_file == false)
        bool_cmd_option_input_file = true;
      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      if (argc - 1 > iarg && strncmp(argv[iarg + 1], "-", 1)) {

        ecs_loc_cmd__lit_arg_fic(argv[++iarg], &argc, &argv);

      }
      else {

        ecs_loc_cmd__lit_arg_fic(NULL, &argc, &argv);

      }

    }
    else if (!strcmp (ECS_CMD_OPTION_OUTPUT_FILE, argv[iarg])) {

      const char * outfic;
      char * outfic_err;
      FILE * ptrfic;

      if (bool_cmd_option_output_file == false)
        bool_cmd_option_output_file = true;
      else
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      if (argc - 1 > iarg && strncmp(argv[iarg + 1], "-", 1))
        outfic = argv[++iarg];
      else
        outfic = ECS_CMD_OUTFILE_NAME_DEFAULT;

      BFT_MALLOC(outfic_err, strlen(outfic) + strlen(".err") + 1, char);

      sprintf(outfic_err, "%s.err", outfic);

#if defined(HAVE_DUP2)
      if ((ptrfic = freopen(outfic, "w", stdout)) == NULL ||
          dup2(fileno(ptrfic), fileno(stderr)) == -1) {
#else
      if (freopen(outfic, "w", stdout) == NULL ||
          freopen(outfic_err, "w", stderr) == NULL) {
#endif

        ecs_loc_cmd__aff_titre(argc, argv);

        bft_error(__FILE__, __LINE__, errno,
                  _("It is impossible to redirect the standard output "
                    "to file:\n%s"), outfic);
      }

      BFT_FREE(outfic_err);

    }
    else if (!strcmp (ECS_CMD_OPTION_RC_2, argv[iarg]) ||
             !strcmp (ECS_CMD_OPTION_RC, argv[iarg])) {

      BFT_REALLOC(cmd->liste_cmd_rc,
                  cmd->nbr_cmd_rc + 1, ecs_cmd_rc_t *);

      iarg++;

      cmd->liste_cmd_rc[cmd->nbr_cmd_rc]
        = ecs_loc_cmd__lit_arg_rc_perio(argc,
                                        argv,
                                        &iarg,
                                        (  cmd->nbr_cmd_rc
                                         - cmd->nbr_cmd_perio + 1),
                                        false);

      cmd->nbr_cmd_rc++;

    }
    else if (!strcmp (ECS_CMD_OPTION_PERIO_2, argv[iarg]) ||
             !strcmp (ECS_CMD_OPTION_PERIO, argv[iarg])) {

      BFT_REALLOC(cmd->liste_cmd_rc,
                  cmd->nbr_cmd_rc + 1, ecs_cmd_rc_t *);

      iarg++;

      cmd->liste_cmd_rc[cmd->nbr_cmd_rc]
        = ecs_loc_cmd__lit_arg_rc_perio(argc,
                                        argv,
                                        &iarg,
                                        cmd->nbr_cmd_perio + 1,
                                        true);

      cmd->nbr_cmd_rc++;
      cmd->nbr_cmd_perio++;

    }
    else if (!strcmp (ECS_CMD_OPTION_SELECT_FAC_INT, argv[iarg])) {

      if (cmd->cmd_select_fac_int != NULL)
        ecs_loc_cmd__aff_opt_en_double(argc, argv, argv[iarg]);

      else {

        iarg++;

        cmd->cmd_select_fac_int = ecs_loc_cmd__lit_arg_select_int(argc,
                                                                  argv,
                                                                  &iarg);
      }

    }
    else if (!strcmp (ECS_CMD_OPTION_ORIENT_CORREC, argv[iarg])) {

      cmd->correct_orient = true;

    }
    else if (!strcmp (ECS_CMD_OPTION_VERSION, argv[iarg])) {

      bool_cmd_option_version = true;

    }
    else {

      ecs_loc_cmd__aff_titre(argc, argv);

      ecs_loc_cmd__aff_aide();

      bft_error(__FILE__, __LINE__, 0,
                _("Option \"%s\" is not recognized.\n"), argv[iarg]);

    }

  } /* Fin : boucle sur les arguments */


  /*---------------------------------------------------------------*/
  /* Affichage de la ligne de commande, du titre et de la version */
  /*---------------------------------------------------------------*/

  ecs_loc_cmd__aff_titre(argc, argv);


  /*
    Si l'on a lu un fichier de commandes, on a recopié argc et argv,
    que l'on peut libérer maintenant.
  */

  if (bool_cmd_option_input_file == true) {
    for (iarg = 0; iarg < argc; iarg++)
      BFT_FREE(argv[iarg]);
    BFT_FREE(argv);
  }


  /*-----------------------------------------*/
  /* Options devant etre traitees en premier */
  /*-----------------------------------------*/

  if (bool_cmd_option_help == true) {

    ecs_loc_cmd__aff_aide();

  }


  /*-------------------------------------------*/
  /* Options provoquant l'arret de l'execution */
  /*-------------------------------------------*/

  if ((argc <= 1)                                        ||
      (             bool_cmd_option_help    == true) ||
      (argc == 2 && bool_cmd_option_version == true) ||
      (argc == 3 && bool_cmd_option_help    == true
                 && bool_cmd_option_version == true)) {

    ecs_cmd__detruit(cmd);
    ecs_exit(EXIT_SUCCESS);

  }


  /*-----------------------------------------------------------*/
  /* Verification que les donnees necessaires ont ete fournies */
  /*-----------------------------------------------------------*/


  if (cmd->liste_fic_maillage.nbr == 0) {

    ecs_loc_cmd__aff_aide();

    bft_error(__FILE__, __LINE__, 0,
              _("Error in command line specification.\n\n"
                "Option \"%s\" must always be set, as the\n"
                "specification of an input mesh file is necessary."),
              ECS_CMD_OPTION_MESH_FILE_1);

  }


  /*--------------------------*/
  /* Verification des donnees */
  /*--------------------------*/


  /* Verification que le fichier de maillage existe */
  /*------------------------------------------------*/

  for (ific = 0; ific < cmd->liste_fic_maillage.nbr; ific++) {

    ecs_loc_cmd__teste_exist_fic(cmd->liste_fic_maillage.val[ific],
                                 _("Mesh file \"%s\" "
                                   "is not accessible.\n%s"));

  }


  /*---------------------------------------------------*/
  /* Affichage de la configuration du cas de lancement */
  /*---------------------------------------------------*/


  ecs_loc_cmd__aff_config(cmd);


  return cmd;


}


/*----------------------------------------------------------------------------
 *  Fonction liberant une structure `ecs_cmd_t' donnee en argument
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_cmd_t * ecs_cmd__detruit
(
 ecs_cmd_t * cmd
)
{

  size_t  ific;
  size_t  igrp;
  int     irc;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(cmd != NULL);


  /* Liberation du contenu de la structure `ecs_cmd_t' */
  /*===============================================*/


  if (cmd->nom_cas != NULL)
    BFT_FREE(cmd->nom_cas);

  for (ific = 0; ific < cmd->liste_fic_maillage.nbr; ific++)

    BFT_FREE(cmd->liste_fic_maillage.val[ific]);


  if (cmd->liste_fic_maillage.nbr != 0) {

    BFT_FREE(cmd->liste_fic_maillage.val);

    BFT_FREE(cmd->liste_num_maillage);
    BFT_FREE(cmd->liste_fmt_maillage);
    BFT_FREE(cmd->liste_grp_maillage);

  }

#if defined(HAVE_CGNS)
  if (cmd->post_cgns != NULL)
    BFT_FREE(cmd->post_cgns);
#endif /* HAVE_CGNS */

  if (cmd->post_ens != NULL)
    BFT_FREE(cmd->post_ens);

#if defined(HAVE_MED)

  if (cmd->post_med != NULL)
    BFT_FREE(cmd->post_med);

#endif /* HAVE_MED */

  if (cmd->nbr_cmd_rc != 0) {

    for (irc = 0; irc < cmd->nbr_cmd_rc; irc++) {

      ecs_select_fac_t select_fac_rc = cmd->liste_cmd_rc[irc]->select_fac_rc;

      if (select_fac_rc.liste_couleur_fac.nbr != 0)
        BFT_FREE(select_fac_rc.liste_couleur_fac.val);

      for (igrp = 0; igrp < select_fac_rc.liste_groupe_fac.nbr; igrp++)
        BFT_FREE(select_fac_rc.liste_groupe_fac.val[igrp]);

      if (select_fac_rc.liste_groupe_fac.nbr != 0)
        BFT_FREE(select_fac_rc.liste_groupe_fac.val);

      if (cmd->liste_cmd_rc[irc]->param_rc.param_perio != NULL)
        BFT_FREE(cmd->liste_cmd_rc[irc]->param_rc.param_perio);

      BFT_FREE(cmd->liste_cmd_rc[irc]);

    }

    BFT_FREE(cmd->liste_cmd_rc);

  }


  if (cmd->cmd_select_fac_int != NULL) {

    ecs_select_fac_t * select_fac = cmd->cmd_select_fac_int;

    if (select_fac->liste_couleur_fac.nbr != 0)
      BFT_FREE(select_fac->liste_couleur_fac.val);

    for (igrp = 0; igrp < select_fac->liste_groupe_fac.nbr; igrp++)
      BFT_FREE(select_fac->liste_groupe_fac.val[igrp]);

    if (select_fac->liste_groupe_fac.nbr != 0)
      BFT_FREE(select_fac->liste_groupe_fac.val);

    BFT_FREE(cmd->cmd_select_fac_int);

  }


  /* Liberation de la structure `ecs_cmd_t' */
  /*====================================*/

  BFT_FREE(cmd);


  return cmd;

}

