#ifndef _ECS_VEC_DEF_H_
#define _ECS_VEC_DEF_H_

/*============================================================================
 *  Prototypes des fonctions
 *   associées aux structures `ecs_vec_int_t' et `ecs_vec_real_t' décrivant
 *   les vecteurs indexes entier et réel
 *   et propres aux vecteurs indexes
 *      lies aux champs principaux de type "définition"
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_param_perio_glob.h"
#include "ecs_vec_int.h"
#include "ecs_vec_real.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction réalisant la décomposition des cellules en faces
 *   sur les vecteurs `ecs_vec_int_t' associés aux champs principaux
 *----------------------------------------------------------------------------*/

void
ecs_vec_def__decompose_cel(ecs_vec_int_t  **vec_def_fac,
                           ecs_vec_int_t  **vec_cel_def_fac,
                           ecs_vec_int_t   *vec_def_cel);


/*----------------------------------------------------------------------------
 *  Fonction qui réalise la fusion des définitions des éléments
 *
 *  Les valeurs absolues sont triées si `bool_abs' est a `true'
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_vec_def__fusionne(ecs_vec_int_t  **this_vec_def,
                      ecs_tab_int_t   *signe_elt);


/*----------------------------------------------------------------------------
 *  Fonction qui fusionne des sommets équivalents
 *  Le pointeur sur un tableau de tolérances peut être à NULL, dans quel cas
 *  on prendra le même poids pour tous les sommets.
 *  Le pointeur sur le tableau liste_som_new peut aussi être à NULL, si l'on
 *  n'est pas intéressé par la liste des sommets fusionnés.
 *
 *  Remarque : le tableau d'équivalence (fusion) des sommets est construit de
 *             manière à ce qu'à un sommet ne comportant pas d'équivalent
 *             où de plus petit indice parmi ses équivalents corresponde la
 *             valeur -1, alors qu'un un sommet possédant des équivalents de
 *             plus petit indice corresponde le plus grand indice parmi ces
 *             équivalents (ce qui constitue une sorte de liste chaînée).
 *----------------------------------------------------------------------------*/

ecs_vec_int_t *
ecs_vec_def__fusion_som(ecs_vec_real_t        *vec_def_som,
                        ecs_tab_int_t         *tab_equiv_som,
                        const ecs_tab_real_t  *dist_max_som,
                        ecs_tab_int_t         *liste_som_new);


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas à la connectivité
 *   (renvoie le vecteur de renumérotation des sommets, ou NULL si aucune
 *   renumérotation n'a été nécessaire).
 *----------------------------------------------------------------------------*/

ecs_vec_int_t *
ecs_vec_def__nettoie_nodal(ecs_vec_real_t       *vec_def_som,
                           const ecs_vec_int_t  *vec_def_fac,
                           const ecs_vec_int_t  *vec_def_cel);


/*----------------------------------------------------------------------------
 *  Correction si nécessaire de l'orientation des éléments en
 *   connectivité nodale.
 *----------------------------------------------------------------------------*/

void
ecs_vec_def__orient_nodal(ecs_vec_real_t       *vec_def_som,
                          const ecs_vec_int_t  *vec_def_fac,
                          const ecs_vec_int_t  *vec_def_cel,
                          ecs_tab_int_t        *liste_cel_err,
                          ecs_tab_int_t        *liste_cel_cor,
                          bool                  correc_orient);


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnées minimales et maximales
 *----------------------------------------------------------------------------*/

void
ecs_vec_def__calc_coo_ext(ecs_vec_real_t  *vec_som);


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnées du maillage
 *----------------------------------------------------------------------------*/

void
ecs_vec_def__transf_coo(ecs_vec_real_t  *vec_som,
                        const double     matrice[3][4]);


/*----------------------------------------------------------------------------
 *  Fonction qui réalise la fusion des définitions des éléments
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_vec_real_def__fusionne(ecs_vec_real_t  **this_vec_def);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un tableau associant un type à chaque face, sous
 * forme de masque : 0 pour face isolée, 1 ou 2 pour face de bord (1 si
 * cellule avec cette face normale sortante, 2 si cellule avec cette face
 * normale entrante), 1+2 = 3 pour face interne, et 4 ou plus pour tous
 * les autres cas, correspondant à une erreur de connectivité (+4 pour faces
 * voyant au moins deux cellules avec face normale sortante, +8 pour faces
 * voyant au moins deux cellules avec face normale entrante).
 *
 *  Le type de chaque face pourra être modifié ultérieurement en fonction
 * des informations de périodicité.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_vec_def__typ_fac_cel(const ecs_vec_int_t  *vec_def_cel,
                         size_t                nbr_fac);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un tableau associant à chaque face les
 * numéros des cellules définies par cette face (normale sortante,
 * puis normale entrante). On affecte une valeur 0 lorsqu'il n'y a pas de
 * cellule correspondante directe (la périodicité n'est donc pas prise en
 * compte à ce niveau).
 *
 * On suppose que la cohérence du maillage a déjà été véridifiée et
 * qu'aucune face n'appartient à plus d'une cellule par côté.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_vec_def__fac_cel(const ecs_vec_int_t  *vec_def_cel,
                     size_t                nbr_fac);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un tableau associant à chaque cellule un code
 * en fonction des erreurs de connectivité éventuelles associées à cette
 * cellule (0 si pas d'erreur, 1 si une des faces définissant cette cellule
 * s'appuie sur plusieurs cellules du même côté).
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_vec_def__err_cel_connect(const ecs_vec_int_t  *vec_def_cel,
                             const ecs_tab_int_t  *typ_fac_cel);


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attachées à une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t
ecs_vec_def__liste_cel_fac(size_t                nbr_fac,
                           const ecs_vec_int_t  *vec_def_cel,
                           const ecs_tab_int_t   liste_fac);


/*----------------------------------------------------------------------------
 *  Fonction qui construit la table de connectivité "faces -> sommets"
 *----------------------------------------------------------------------------*/

ecs_vec_int_t *
ecs_vec_def__cree_fac_som(const ecs_vec_int_t  *vec_def_fac,
                          const ecs_vec_int_t  *vec_def_are);


/*----------------------------------------------------------------------------
 *  Fonction qui construit un tableau de booléens conforme a une liste
 *   de sous-éléments
 *  Un sous-élément est a `true'
 *   s'il intervient dans la définition des éléments
 *----------------------------------------------------------------------------*/

void
ecs_vec_def__cree_masque(const ecs_tab_bool_t   bool_sselt_select,
                         const ecs_vec_int_t   *vec_def_elt);


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus appartenant à des mêmes faces.
 *  La connectivité des faces est mise à jour.
 *----------------------------------------------------------------------------*/

ecs_vec_int_t *
ecs_vec_def__nettoie_som_fac(ecs_vec_real_t  *vec_def_som,
                             ecs_vec_int_t   *vec_def_fac);


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les éventuelles faces dégénérées
 *----------------------------------------------------------------------------*/

ecs_vec_int_t  *
ecs_vec_def__nettoie_fac(ecs_vec_int_t  *vec_def_fac);


/*----------------------------------------------------------------------------
 *  Fonction qui met à jour le tableau des fusions de sommets
 *
 *  Remarque : le tableau d'équivalence (fusion) des sommets est construit de
 *             manière à ce qu'à un sommet ne comportant pas d'équivalent
 *             où de plus grand indice parmi ses équivalents corresponde la
 *             valeur -1, alors qu'un un sommet possédant des équivalents de
 *             plus grand indice corresponde le plus petit indice parmi ces
 *             équivalents (ce qui constitue une sorte de liste chaînée).
 *----------------------------------------------------------------------------*/

void
ecs_vec_def__maj_equiv_som(size_t          ind_som_0,
                           size_t          ind_som_1,
                           ecs_tab_int_t  *tab_equiv_som);


#endif /* _ECS_VEC_DEF_H_ */
