/*============================================================================
 *  Dfinitions des fonctions
 *   de calcul et d'affichage de temps d'excution (chronomtrie)
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include <bft_printf.h>
#include <bft_timer.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_chrono.h"


/*============================================================================
 *                       Variables globales
 *============================================================================*/

size_t     _ecs_chrono_id ;
char    ** _ecs_chrono_nom ;
double   * _ecs_chrono_temps ;
double     _ecs_chrono_cpu ;
double     _ecs_chrono_cpu_ini ;
double     _ecs_chrono_cpu_prec ;


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/


/*----------------------------------------------------------------------------
 *    Fonction qui initialise les temps de rfrence
 *----------------------------------------------------------------------------*/

void ecs_chrono__initialise
(
 void
)
{

  (void)bft_timer_wtime() ;

  _ecs_chrono_id  = 0 ;
  _ecs_chrono_nom = NULL ;
  _ecs_chrono_temps = NULL ;

  _ecs_chrono_cpu_ini  = bft_timer_cpu_time() ;
  _ecs_chrono_cpu_prec = _ecs_chrono_cpu_ini ;

}


/*----------------------------------------------------------------------------
 *    Fonction qui affiche les temps d'excution et termine la chronomtrie
 *----------------------------------------------------------------------------*/

void ecs_chrono__affiche
(
 void
)
{

  double  utime ;
  double  stime ;
  double  time_cpu ;
  double  time_tot ;

  char  *nom_fic ;
  FILE  *fic_time = NULL ;

  float  pc ;
  size_t i ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf(_("\n\nTime and memory summary\n"
               "-----------------------\n\n")) ;


  bft_timer_cpu_times(&utime, &stime) ;

  if (utime > 0. || stime > 0.)
    time_cpu = utime + stime;

  else
    time_cpu = bft_timer_cpu_time() ;


  /* (heure de fin d'execution) - (heure de debut d'execution) */


  if (utime > 0. || stime > 0.) {

    bft_printf("  ") ;
    ecs_print_padded_str(_("User CPU time                       (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)utime);

    bft_printf("  ") ;
    ecs_print_padded_str(_("System CPU time                     (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)stime);

  }

  else if (time_cpu > 0.) {

    bft_printf("  ") ;
    ecs_print_padded_str(_("Total CPU time                      (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)time_cpu);

  }


  /* Dure d'excution  */

  time_tot = bft_timer_wtime();

  if (time_tot > 0.) {

    bft_printf("  ") ;
    ecs_print_padded_str(_("Total time                          (sec)"),
                         ECS_LNG_AFF_STR) ;
    bft_printf(" : %*.*f\n",
               ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
               (float)time_tot);

    if (time_cpu > 0.) {

      bft_printf("  ") ;
      ecs_print_padded_str(_("Total CPU time / Total time              "),
                           ECS_LNG_AFF_STR) ;
      bft_printf(" : %*.*f\n",
                 ECS_LNG_AFF_REE_MANTIS, ECS_LNG_AFF_REE_PRECIS,
                 (float)(time_cpu/time_tot));

    }

  }

  bft_printf("\n");

  if (time_cpu > 0. && (nom_fic = getenv("ECS_FIC_CHRONO")) != NULL) {

    fic_time = fopen(nom_fic, "w");

    if (fic_time == NULL) {

      ecs_warn() ;
      bft_printf(_("The timing file defined by the\n"
                   "ECS_FIC_CHRONO=%s environment variable.\n"
                   "could not be opened for writing."), nom_fic) ;

    }

  }

  if (fic_time != NULL) {

    _ecs_chrono_cpu_prec = _ecs_chrono_cpu_ini ;

    ECS_CHRONO("TOTAL");

    for (i = 0 ; i < _ecs_chrono_id ; i++) {

      if (_ecs_chrono_temps[_ecs_chrono_id - 1] > 1.e-16)
        pc = 100. * _ecs_chrono_temps[i]
                  /  _ecs_chrono_temps[_ecs_chrono_id - 1] ;
      else
        pc = 0. ;

      fprintf(fic_time,
              "%2lu. %14.6f : %7.3f %% : %s\n",
              (unsigned long)i,
              (float)_ecs_chrono_temps[i],
              (float)pc,
              _ecs_chrono_nom[i]) ;

    }

    fclose(fic_time);

  }

  /* Libration de la structure */

  for (i = 0 ; i < _ecs_chrono_id ; i++)

    BFT_FREE(_ecs_chrono_nom[i]) ;


  BFT_FREE(_ecs_chrono_temps) ;
  BFT_FREE(_ecs_chrono_nom) ;

}

