/*============================================================================
 *  Dfinition de la fonction
 *   de lecture d'un fichier de maillage au format MED
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

#include "ecs_config.h"

#if defined(HAVE_MED)


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_elt_typ_liste.h"
#include "bft_error.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"
#include "ecs_descr.h"
#include "ecs_descr_chaine.h"
#include "ecs_entmail.h"
#include "ecs_famille.h"
#include "ecs_famille_chaine.h"
#include "ecs_maillage.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_pre.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_pre_med.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_med_priv.h"



/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Cration d'une structure `ecs_med_t' et ouverture d'un fichier MED
 *  en lecture
 *----------------------------------------------------------------------------*/

ecs_med_t *ecs_pre_med__cree
(
 const char  *const nom_fichier
);


/*----------------------------------------------------------------------------
 *  Fermeture d'un fichier MED en lecture et destruction de la structure
 *  `ecs_med_t' associe
 *----------------------------------------------------------------------------*/

ecs_med_t * ecs_pre_med__detruit
(
 ecs_med_t  * fic
);


/*----------------------------------------------------------------------------
 *                        Lecture des noeuds
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * ecs_loc_pre_med__lit_noeud
(
 const ecs_med_t  *const fic_maillage ,
 const char       *const nom_maillage ,
       ecs_dim_t         dim_e
);


/*----------------------------------------------------------------------------
 *                        Lecture des lments
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * * ecs_loc_pre_med__lit_maille
(
 const ecs_med_t  *const fic_maillage ,
 const char       *const nom_maillage ,
       ecs_dim_t         dim_e
);


/*----------------------------------------------------------------------------
 *                        Lecture des familles
 *----------------------------------------------------------------------------*/

static ecs_famille_t * * ecs_loc_pre_med__lit_famille
(
 const ecs_med_t *const fic_maillage ,
 const char      *const nom_maillage
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un pointeur sur un tableau de type `ecs_int_t'
 *   dont les valeurs sont converties si ncessaire
 *    partir du tableau de connectivit `val_med' ayant `nbr_val' valeurs
 *   de type `med_int', avec `pas_med' valeurs MED et `pas_ecs' valeurs
 *   locales par lement.
 *
 *  Cette fonction supprime donc les rfrences inutiles (noeuds
 *   quadratiques ou connectivits supplmentaires ventuelles)
 *
 *  Le tableau en entree `val_med' est libr si le tableau renvoy est allou
 *----------------------------------------------------------------------------*/

static ecs_int_t * ecs_loc_convert_connect_med_ecs
(
       med_int    * val_med ,
 const ecs_int_t    nbr_val,
 const ecs_int_t    pas_med,
 const ecs_int_t    pas_ecs
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un pointeur sur un tableau de type `ecs_real_t'
 *   dont les valeurs sont converties si ncessaire
 *    partir du tableau `val_med' ayant `nbr_val' valeurs de type `med_float'
 *
 *  Le tableau en entre `val_med' est liber si le tableau renvoy est allou
 *----------------------------------------------------------------------------*/

static ecs_real_t * ecs_loc_convert_real_med_ecs
(
       med_float  * val_med ,
 const ecs_int_t    nbr_val
);


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Lecture d'un fichier au format MED
 *   et affectation des donnees dans la structure de maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_pre_med__lit_maillage
(
 const char       *const nom_fic_maillage ,      /* --> Nom du fichier  lire */
 const ecs_int_t         num_maillage            /* --> Numro du maillage    */
)
{

  char             * nom_maillage;

  int                ind;

  ecs_dim_t          dim_e;

  ecs_int_t          ient;

  ecs_med_t        * fic_maillage;

  ecs_entmail_t    * vect_entmail[ECS_ENTMAIL_FIN];
  ecs_entmail_t  * * vect_entmail_ret;
  ecs_famille_t  * * vect_famille;
  ecs_maillage_t   * maillage;


  /* Declarations des variables pour MED */
  /*-------------------------------------*/

  char           nom_maillage_med[MED_TAILLE_NOM + 1];
  char           desc_maillage_med[MED_TAILLE_DESC + 1];

  med_err        ret_med = 0;

  med_int        mdim_med;
  med_int        nbr_maillages_med;
  int            num_maillage_med;
  med_maillage   type_maillage_med;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf(_("\n\n"
               "Reading mesh from file in MED (EDF/CEA) format\n"
               "----------------------\n"));

  bft_printf(_("  Mesh file: %s\n\n\n"),
             nom_fic_maillage);


  /* Ouverture du fichier MED en lecture */
  /*-------------------------------------*/

  fic_maillage = ecs_pre_med__cree(nom_fic_maillage);

  /* Vrification et traitements selon le nombre de maillages */

  nbr_maillages_med = MEDnMaa(fic_maillage->fid);
  if (nbr_maillages_med < 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\"."),
              nom_fic_maillage);

  if (nbr_maillages_med > 1) {
    bft_printf(_("\n  The file contains multiple meshes:\n"));
    for (ind = 0; ind < nbr_maillages_med; ind++) {
      if (MEDmaaInfo(fic_maillage->fid,
                     ind + 1,
                     nom_maillage_med,
                     &mdim_med,
                     &type_maillage_med,
                     desc_maillage_med) != 0)
        bft_error(__FILE__, __LINE__, 0,
                  _("MED: error reading file \"%s\"."),
                  nom_fic_maillage);
      bft_printf(_("    Mesh %2d: %s\n"), ind + 1, nom_maillage_med);
    }
    if (num_maillage == 0)
      bft_printf(_("\n  No mesh was specified; the first is read.\n\n"));
    else if (num_maillage > 0)
      bft_printf(_("\n  Mesh number %d was specified.\n\n"), num_maillage);
  }
  else if (nbr_maillages_med == 0)
    bft_error(__FILE__, __LINE__, 0,
              _("No mesh in file.\n"));

  assert (num_maillage >= 0);
  if (num_maillage > nbr_maillages_med)
    bft_error(__FILE__, __LINE__, 0,
              _("The specified mesh number (%d) is greater than\n"
                "the number of meshes defined (%d) in file\n%s.\n"),
              num_maillage, nbr_maillages_med, nom_fic_maillage);
  else
    num_maillage_med = ECS_MAX(1, num_maillage);


  ret_med = MEDmaaInfo(fic_maillage->fid,
                       num_maillage_med,
                       nom_maillage_med,
                       &mdim_med,
                       &type_maillage_med,
                       desc_maillage_med);

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\".\n"
                "Name of mesh read : \"%s\"\n"
                "Dimension read    : \"%d\""),
              nom_fic_maillage, nom_maillage_med, mdim_med);


  ret_med = MEDdimEspaceLire(fic_maillage->fid, nom_maillage_med);

  if (ret_med != -1)
    mdim_med = ret_med;


  nom_maillage_med[MED_TAILLE_NOM] = '\0';

  bft_printf(_("  Mesh name: %s\n\n"), nom_maillage_med);


  assert((ecs_dim_t)mdim_med == ECS_DIM_2 ||
         (ecs_dim_t)mdim_med == ECS_DIM_3   );

  dim_e = (ecs_dim_t)mdim_med;
  BFT_MALLOC(nom_maillage, strlen(nom_maillage_med) + 1, char);
  strcpy(nom_maillage, nom_maillage_med);


  /* Lecture des noeuds */
  /*--------------------*/

  vect_entmail[ECS_ENTMAIL_SOM] = ecs_loc_pre_med__lit_noeud(fic_maillage,
                                                             nom_maillage,
                                                             dim_e);

  /* Lecture des elements */
  /*----------------------*/

  vect_entmail_ret = ecs_loc_pre_med__lit_maille(fic_maillage,
                                                 nom_maillage,
                                                 dim_e);


  for (ient = ECS_ENTMAIL_ARE; ient < ECS_ENTMAIL_FIN; ient++)
    vect_entmail[ient] = vect_entmail_ret[ient];

  BFT_FREE(vect_entmail_ret);


  maillage = ecs_maillage__cree_nodal(dim_e,
                                      vect_entmail);


  /* Lecture des familles */
  /*----------------------*/

  vect_famille = ecs_loc_pre_med__lit_famille(fic_maillage,
                                              nom_maillage);

  ecs_maillage__definit_famille(maillage,
                                vect_famille);

  BFT_FREE(vect_famille);
  BFT_FREE(nom_maillage);


  /* Fermeture du fichier de lecture du maillage */
  /*---------------------------------------------*/

  ecs_pre_med__detruit(fic_maillage);


  /* Transformation des familles en attributs "groupe" et "couleur" */
  /*----------------------------------------------------------------*/

  ecs_maillage__cree_attribut(maillage,
                              ECS_ENTMAIL_SOM,
                              ECS_ENTMAIL_SOM);

  ecs_maillage__cree_attribut(maillage,
                              ECS_ENTMAIL_ARE,
                              ECS_ENTMAIL_CEL);


  return maillage;


}


/*============================================================================
 *
 *                              Fonctions privees
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Cration d'une structure `ecs_med_t' et ouverture d'un fichier MED
 *  en lecture
 *----------------------------------------------------------------------------*/

ecs_med_t *ecs_pre_med__cree
(
 const char  *const nom_fichier
)
{

  ecs_med_t  * fic;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  BFT_MALLOC(fic, 1, ecs_med_t);


  BFT_MALLOC(fic->nom_fic, strlen(nom_fichier) + 1, char);
  strcpy(fic->nom_fic, nom_fichier);


  fic->fid = MEDouvrir(fic->nom_fic,
                       MED_LECTURE);

  if (fic->fid < 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error opening file \"%s\"."), nom_fichier);

  fic->nbr_maillages = 0;
  fic->tab_maillages = NULL;

  return fic;

}


/*----------------------------------------------------------------------------
 *  Fermeture d'un fichier MED en lecture et destruction de la structure
 *  `ecs_med_t' associe
 *----------------------------------------------------------------------------*/

ecs_med_t * ecs_pre_med__detruit
(
 ecs_med_t  * fic
)
{


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(fic != NULL);


  if (MEDfermer(fic->fid) != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error closing file \"%s\"."),
              fic->nom_fic);


  BFT_FREE(fic->nom_fic);

  BFT_FREE(fic);


  return fic;


}


/*----------------------------------------------------------------------------
 *                        Lecture des noeuds
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * ecs_loc_pre_med__lit_noeud
(
 const ecs_med_t  *const fic_maillage ,
 const char       *const nom_maillage ,
       ecs_dim_t         dim_e
)
{

  /* Declarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  ecs_int_t     nbr_som;

  ecs_real_t  * som_val_coord;                   /* Coordonnees des sommets  */


  /* Declarations des variables pour MED */
  /*-------------------------------------*/

  med_booleen   inom_noe_med;
  med_booleen   inum_noe_med;
  med_repere    repere_med;

  med_int       mdim_med;

  med_int       nbr_noe_med;
  med_int     * fam_noe_med;
  med_int     * num_noe_med;

  med_float   * coord_med;

  med_err       ret_med = 0;

  char          nom_coord_med[3 * MED_TAILLE_PNOM + 1];
  char          nom_maillage_med[MED_TAILLE_NOM + 1];
  char        * nom_noe_med;
  char          uni_coord_med[3 * MED_TAILLE_PNOM + 1];

  ecs_entmail_t  * entmail_som;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  strcpy(nom_maillage_med, nom_maillage);


  /* On recupere le nombre de noeuds */

  nbr_noe_med = MEDnEntMaa(fic_maillage->fid,
                           nom_maillage_med,
                           MED_COOR,
                           MED_NOEUD,
                           (med_geometrie_element)0,
                           (med_connectivite)0);

  if (nbr_noe_med < 0)
    ret_med = -1;

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\".\n"
                "Number of vertices of entity read: \"%d\"."),
              fic_maillage->nom_fic, (int)nbr_noe_med);



  /*------------------------------------*/
  /* Lecture des coordonnes des noeuds */
  /*------------------------------------*/

  mdim_med = (med_int)dim_e;


  BFT_MALLOC(coord_med  , nbr_noe_med * mdim_med           , med_float);
  BFT_MALLOC(fam_noe_med, nbr_noe_med                      , med_int  );
  BFT_MALLOC(num_noe_med, nbr_noe_med                      , med_int  );
  BFT_MALLOC(nom_noe_med, nbr_noe_med * MED_TAILLE_PNOM + 1, char     );

  ret_med = MEDnoeudsLire(fic_maillage->fid,
                          nom_maillage_med,
                          mdim_med,
                          coord_med,
                          MED_FULL_INTERLACE,
                          &repere_med,
                          nom_coord_med,
                          uni_coord_med,
                          nom_noe_med,
                          &inom_noe_med,
                          num_noe_med,
                          &inum_noe_med,
                          fam_noe_med,
                          nbr_noe_med);

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\".\n"
                "Error reading coordinates."),
              fic_maillage->nom_fic);

  BFT_FREE(fam_noe_med);
  BFT_FREE(num_noe_med);
  BFT_FREE(nom_noe_med);



  /* Transformation du tableau des coord au type `med_float' */
  /*             en un tableau des coord au type `ecs_real_t'      */
  /*---------------------------------------------------------*/

  nbr_som = (ecs_int_t)nbr_noe_med;


  som_val_coord = ecs_loc_convert_real_med_ecs(coord_med,
                                               nbr_som * dim_e);

  if (dim_e != ECS_DIM_3) {

    ecs_int_t isom;

    BFT_REALLOC(som_val_coord, nbr_som * ECS_DIM_3, ecs_real_t);

    if (dim_e == ECS_DIM_1) {

      for (isom = nbr_som-1; isom >= 0; isom--) {
        som_val_coord[isom*3    ] = som_val_coord[isom];
        som_val_coord[isom*3 + 1] = 0.0;
        som_val_coord[isom*3 + 2] = 0.0;
      }

    }
    else if (dim_e == ECS_DIM_2) {

      for (isom = nbr_som-1; isom >= 0; isom--) {
        som_val_coord[isom*3    ] = som_val_coord[isom*2    ];
        som_val_coord[isom*3 + 1] = som_val_coord[isom*2 + 1];
        som_val_coord[isom*3 + 2] = 0.0;
      }

    }

  }


  /* Transfert des valeurs lues dans la structure d'entite de maillage */
  /*===================================================================*/


  entmail_som = ecs_entmail_pre__cree_som(dim_e,
                                          nbr_som,
                                          som_val_coord,
                                          NULL);


  return entmail_som;


}


/*----------------------------------------------------------------------------
 *                        Lecture des lments
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * * ecs_loc_pre_med__lit_maille
(
 const ecs_med_t  *const fic_maillage ,
 const char       *const nom_maillage ,
       ecs_dim_t         dim_e
)
{

  ECS_ENTMAIL_E    entmail_e;

  ecs_int_t      cpt_som;
  ecs_int_t      ient;
  ecs_int_t      ielt;
  ecs_int_t      isom;
  ecs_int_t      ityp;
  ecs_int_t      nbr_elt;
  ecs_int_t      nbr_som_elt;
  ecs_int_t      nbr_som_fac;
  ecs_int_t      pos_elt;
  ecs_int_t      renum_som;
  ecs_int_t      taille;
  ecs_int_t      typ_geo_ecs;

  ecs_int_t    * elt_val_som  ;


  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  size_t       cpt_elt_ent        [ECS_ENTMAIL_FIN]; /* Nbr. elts par entite */
  ecs_int_t    cpt_coul_ent       [ECS_ENTMAIL_FIN]; /* Compteur de couleurs */
  ecs_int_t  * val_coul_ent       [ECS_ENTMAIL_FIN]; /* Tableau des couleurs */
  ecs_size_t * cpt_elt_coul_ent   [ECS_ENTMAIL_FIN];
  ecs_size_t * elt_pos_som_ent    [ECS_ENTMAIL_FIN]; /* Positions sommets    */
  ecs_int_t  * elt_val_som_ent    [ECS_ENTMAIL_FIN]; /* Numeros des sommets  */
  ecs_int_t  * elt_val_label_ent  [ECS_ENTMAIL_FIN]; /* Etiquettes elements  */
  ecs_int_t  * elt_val_fam_ent    [ECS_ENTMAIL_FIN]; /* Familles elements    */
  ecs_int_t  * elt_val_color_ent  [ECS_ENTMAIL_FIN]; /* Couleurs elements    */

  ecs_size_t * pos_fac_connect;
  ecs_int_t  * val_fac_connect;

  ecs_entmail_t ** vect_entmail;


  /* Dclarations des variables pour MED */
  /*-------------------------------------*/

  char               nom_maillage_med[MED_TAILLE_NOM + 1] = "";
  char               nom_equiv[MED_TAILLE_NOM+1] = "";

  med_geometrie_element  typ_geo_med;
  med_entite_maillage    typ_ent_med;

  med_err            ret_med = 0;

  med_int            nbr_equiv = 0;
  med_int            ind_equiv = -1;
  med_int            edim_med;
  med_int            mdim_med;
  med_int            nbr_ele_med;
  med_int            taille_med;

  med_int          * connect_med = NULL;
  med_int          * fam_ele_med = NULL;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /*====================================================*/
  /* Initialisations et allocations des tableaux locaux */
  /*====================================================*/


  /* Attention au decalage de `1' !!!         */
  /* On n'alloue pas les tableaux locaux pour */
  /* `ECS_ENTMAIL_DEB = ECS_ENTMAIL_SOM'      */

  for (ient = ECS_ENTMAIL_ARE; ient < ECS_ENTMAIL_FIN; ient++) {

    cpt_elt_ent        [ient] = 0   ;

    elt_pos_som_ent    [ient] = NULL;
    elt_val_som_ent    [ient] = NULL;
    elt_val_label_ent  [ient] = NULL;
    elt_val_fam_ent    [ient] = NULL;
    elt_val_color_ent  [ient] = NULL;

    cpt_coul_ent       [ient] = 0   ;
    val_coul_ent       [ient] = NULL;
    cpt_elt_coul_ent   [ient] = NULL;

  }

  pos_fac_connect = NULL;
  val_fac_connect = NULL;

  mdim_med = (med_int)dim_e;
  strcpy(nom_maillage_med, nom_maillage);


  /*------------------------------------------------*/
  /* Lecture de la connectivit nodale des elements */
  /*------------------------------------------------*/


  /* On regarde si le maillage MED comporte des points */
  /*---------------------------------------------------*/

  nbr_ele_med = MEDnEntMaa(fic_maillage->fid,
                           nom_maillage_med,
                           MED_CONN,
                           MED_NOEUD,
                           MED_POINT1,
                           MED_NOD);

  if (nbr_ele_med < 0)
    ret_med = -1;

  else if (nbr_ele_med == 0) {

    nbr_ele_med = MEDnEntMaa(fic_maillage->fid,
                             nom_maillage_med,
                             MED_CONN,
                             MED_MAILLE,
                             MED_POINT1,
                             MED_NOD);

    if (nbr_ele_med < 0)
      ret_med = -1;

  }

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\".\n"
                "Number of elements of entity read: \"%d\"."),
              fic_maillage->nom_fic, (int)nbr_ele_med);


  if (nbr_ele_med != 0) {

    /* Le cas de points isols (n'etant pas dfinis comme sommets) */
    /*  n'est pas prvu                                            */

    ecs_warn();
    bft_printf(_("The MED mesh contains %d elements of type `point'\n"
                 "which are ignored by the Preprocessor.\n"),
               nbr_ele_med);

  }


  /* Vrification du nombre d'quivalences
     (pour connectivit faces non conformes ventuelle) */

  nbr_equiv = MEDnEquiv(fic_maillage->fid,
                        nom_maillage_med);

  if (nbr_equiv < 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\".\n"
                "Error reading equivalence information."),
              fic_maillage->nom_fic);

  else if (nbr_equiv > 0) {

    size_t  ind;
    char nom_equiv_cmp[MED_TAILLE_NOM+1];
    char desc_equiv[MED_TAILLE_DESC+1];

    for (ind_equiv = 0; ind_equiv < nbr_equiv; ind_equiv++) {

      ret_med = MEDequivInfo(fic_maillage->fid,
                             nom_maillage_med,
                             ind_equiv + 1,
                             nom_equiv,
                             desc_equiv);

      if (ret_med < 0)
        bft_error(__FILE__, __LINE__, 0,
                  _("MED: error reading file \"%s\".\n"
                    "Error reading equivalence information."),
                  fic_maillage->nom_fic);

      nom_equiv[MED_TAILLE_NOM] = '\0';

      for (ind = 0; ind < MED_TAILLE_NOM && nom_equiv[ind] != '\0'; ind++)
        nom_equiv_cmp[ind] = tolower(nom_equiv[ind]);
      nom_equiv_cmp[strlen(nom_equiv)] = '\0';

      if (   strcmp(nom_equiv_cmp, "face_connectivity") == 0
          || strcmp(nom_equiv_cmp, "face connectivity") == 0)
        break;

    }

    if (ind_equiv >= nbr_equiv) {
      nom_equiv[0] = '\0';
      ind_equiv = -1;
    }

  }


  /* Boucle sur tous les types gomtriques MED rfrencs dans l'Enveloppe */
  /*------------------------------------------------------------------------*/


  for (ityp = 0; ityp < ECS_MED_NBR_TYP_ELT; ityp++) {


    typ_geo_med = ecs_fic_med_init_elt_liste_c[ityp].med_typ;


    /*
     * On essaiera de lire en priorit les mailles, les artes ou faces
     * correspondant  une connectivit descendante tant lues en second choix.
     */

    edim_med = typ_geo_med / 100;

    typ_ent_med = MED_MAILLE;

    if (typ_geo_med != MED_POINT1) {

      nbr_ele_med = MEDnEntMaa(fic_maillage->fid,
                               nom_maillage_med,
                               MED_CONN,
                               typ_ent_med,
                               typ_geo_med,
                               MED_NOD);

      if (nbr_ele_med < 0)
        ret_med = -1;

      else if (nbr_ele_med == 0 && edim_med <= 2) {

        if (edim_med == 1)
          typ_ent_med = MED_ARETE;
        else if (edim_med == 2)
          typ_ent_med = MED_FACE;

        nbr_ele_med = MEDnEntMaa(fic_maillage->fid,
                                 nom_maillage_med,
                                 MED_CONN,
                                 typ_ent_med,
                                 typ_geo_med,
                                 MED_NOD);

        if (nbr_ele_med < 0)
          ret_med = -1;

      }

      if (ret_med != 0)
        bft_error(__FILE__, __LINE__, 0,
                  _("MED: error reading file \"%s\".\n"
                    "Number of elements of entity read: \"%d\"."),
                  fic_maillage->nom_fic, (int)nbr_ele_med);


      if (nbr_ele_med != 0) {


        nbr_elt = (ecs_int_t)nbr_ele_med;


        /* Type gometrique de l'Enveloppe */
        /*---------------------------------*/

        typ_geo_ecs = ecs_fic_med_init_elt_liste_c[ityp].ecs_typ;


        /* Identification de l'entit concerne */
        /*--------------------------------------*/

        entmail_e = ecs_entmail_pre__ret_typ_geo(typ_geo_ecs);


        BFT_REALLOC(elt_pos_som_ent[entmail_e],
                    cpt_elt_ent[entmail_e] + 1 + nbr_elt, ecs_size_t);

        if (cpt_elt_ent[entmail_e] == 0) {
          /* On est au 1er tour */
          elt_pos_som_ent[entmail_e][0] = 1;
        }


        BFT_REALLOC(elt_val_fam_ent[entmail_e],
                    cpt_elt_ent[entmail_e] + nbr_elt, ecs_int_t);


        BFT_MALLOC(fam_ele_med, nbr_ele_med, med_int);


        /* Traitement des lments "classiques" */
        /*--------------------------------------*/

        if (typ_geo_med != MED_POLYGONE && typ_geo_med != MED_POLYEDRE) {

          nbr_som_elt = ecs_fic_elt_typ_liste_c[typ_geo_ecs].nbr_som;

          taille = elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e]] - 1;

          BFT_REALLOC(elt_val_som_ent[entmail_e],
                      taille + nbr_elt * nbr_som_elt, ecs_int_t);


          /* Convention sur la taille des mailles */

          edim_med = typ_geo_med / 100;
          taille_med = typ_geo_med % 100;

          taille = (ecs_int_t)nbr_ele_med * (ecs_int_t)taille_med;
          BFT_MALLOC(connect_med, taille, med_int);

          ret_med = MEDconnLire(fic_maillage->fid,
                                nom_maillage_med,
                                mdim_med,
                                connect_med,
                                MED_FULL_INTERLACE,
                                NULL, /* Pas de profil */
                                0,
                                typ_ent_med,
                                typ_geo_med,
                                MED_NOD);

          if (ret_med != 0)
            bft_error(__FILE__, __LINE__, 0,
                      _("MED: error reading file \"%s\".\n"
                        "Error reading connectivity."),
                      fic_maillage->nom_fic);

          /* Les rfrences aux noeuds milieux des lments     */
          /*  quadratiques ne sont pas conserves               */

          elt_val_som = ecs_loc_convert_connect_med_ecs(connect_med,
                                                        taille,
                                                        taille_med,
                                                        nbr_som_elt);

          /* Remplissage de la connectivit */

          cpt_som = 0;

          for (ielt = 0; ielt < nbr_elt; ielt++) {

            pos_elt = elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e]
                                                 + ielt];

            elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e] + ielt + 1]
              = pos_elt + nbr_som_elt;

            for (isom = 0; isom < nbr_som_elt; isom++) {

              renum_som = ecs_fic_med_init_elt_liste_c[ityp].num_som[isom] - 1;
              elt_val_som_ent[entmail_e][pos_elt - 1 + isom]
                = elt_val_som[cpt_som + renum_som];

            }

            cpt_som += nbr_som_elt;

          }

          /* Libration connectivit temporaire */

          BFT_FREE(elt_val_som);

        }

        /* Traitement des polygones */
        /*--------------------------*/

        else if (typ_geo_med == MED_POLYGONE) {

          ecs_int_t    ival;
          ecs_int_t    nbr_val_elt;
          med_int    * index_med = NULL;

          /* Taille du tableau des connectivites */

          ret_med = MEDpolygoneInfo(fic_maillage->fid,
                                    nom_maillage_med,
                                    typ_ent_med,
                                    MED_NOD,
                                    &taille_med);

          if (ret_med != 0)
            bft_error(__FILE__, __LINE__, 0,
                      _("MED: error reading file \"%s\".\n"
                        "(polygons information)."),
                      fic_maillage->nom_fic);

          taille = elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e]] - 1;

          BFT_REALLOC(elt_val_som_ent[entmail_e],
                      taille + (ecs_int_t)taille_med, ecs_int_t);

          BFT_MALLOC(index_med, (ecs_int_t)(nbr_ele_med + 1), med_int);
          BFT_MALLOC(connect_med, (ecs_int_t)taille_med, med_int);

          /* Lecture de la connectivit des polygones */

          ret_med = MEDpolygoneConnLire(fic_maillage->fid,
                                        nom_maillage_med,
                                        index_med,
                                        nbr_ele_med + 1,
                                        connect_med,
                                        typ_ent_med,
                                        MED_NOD);

          if (ret_med != 0)
            bft_error(__FILE__, __LINE__, 0,
                      _("MED: error reading file \"%s\".\n"
                        "(polygons connectivity)."),
                      fic_maillage->nom_fic);

          /* Remplissage de la connectivit */

          pos_elt = elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e]];

          for (ielt = 0; ielt < nbr_elt; ielt++)
            elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e] + ielt + 1]
              = pos_elt + index_med[ielt + 1] - index_med[0];

          nbr_val_elt = index_med[nbr_elt] - index_med[0];

          for (ival = 0; ival < nbr_val_elt; ival++)
            elt_val_som_ent[entmail_e][pos_elt - 1 + ival]
              = connect_med[ival];

          /* Libration connectivit temporaire */

          BFT_FREE(index_med);
          BFT_FREE(connect_med);

        }

        /* Traitement des polydres */
        /*--------------------------*/

        else if (typ_geo_med == MED_POLYEDRE) {

          ecs_int_t    ifac, ival;
          ecs_int_t    num_som_prec, num_som_deb;
          ecs_int_t    cpt_som_poly_dup = 0;
          ecs_int_t    cpt_fac_poly_dgn = 0;
          med_int      n_index_f_med = 0;
          med_int    * index_med = NULL;
          med_int    * index_f_med = NULL;

          /* Taille du tableau des connectivites */

          ret_med = MEDpolyedreInfo(fic_maillage->fid,
                                    nom_maillage_med,
                                    MED_NOD,
                                    &n_index_f_med,
                                    &taille_med);

          if (ret_med != 0)
            bft_error(__FILE__, __LINE__, 0,
                      _("MED: error reading file \"%s\".\n"
                        "(polyhedra information)."),
                      fic_maillage->nom_fic);

          /* On ajoute le nombre de faces (n_index_f_med - 1)  la taille du
             tableau destin  recevoir la connectivit afin de faire apparatre
             une seconde fois le numro du premier sommet  la fin de la liste
             des sommets de chaque face dans la dfinition des cellules (pour
             reprer la fin de dfinition de chaque face d'une cellule). */

          taille = elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e]] - 1;

          BFT_REALLOC(elt_val_som_ent[entmail_e],
                      taille + (ecs_int_t)(taille_med + n_index_f_med - 1),
                      ecs_int_t);

          BFT_MALLOC(index_med, (ecs_int_t)(nbr_ele_med + 1), med_int);
          BFT_MALLOC(index_f_med, (ecs_int_t)(n_index_f_med), med_int);
          BFT_MALLOC(connect_med, (ecs_int_t)taille_med, med_int);

          /* Lecture de la connectivit des polydres */

          ret_med = MEDpolyedreConnLire(fic_maillage->fid,
                                        nom_maillage_med,
                                        index_med,
                                        nbr_ele_med + 1,
                                        index_f_med,
                                        n_index_f_med,
                                        connect_med,
                                        MED_NOD);

          if (ret_med != 0)
            bft_error(__FILE__, __LINE__, 0,
                      _("MED: error reading file \"%s\".\n"
                        "(polyhedra connectivity)."),
                      fic_maillage->nom_fic);

          if (index_med[nbr_ele_med] - index_med[0] + 1 != n_index_f_med)
            bft_error
              (__FILE__, __LINE__, 0,
               _("MED: inconsistency in polyhedra connectivity;\n"
                 "number of polyhedra: %d\n"
                 "the cells->faces end index (%d) should be equal to\n"
                 "the size of the faces->vertices index array (%d)\n"),
               (int) nbr_ele_med,
               (int)(index_med[nbr_ele_med] - index_med[0] + 1),
               (int) n_index_f_med);

          /* Remplissage de la connectivit */

          pos_elt = elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e]];

          ival = pos_elt - 1;

          for (ielt = 0; ielt < nbr_elt; ielt++) {

            for (ifac = index_med[ielt    ] - index_med[0];
                 ifac < index_med[ielt + 1] - index_med[0];
                 ifac++) {

              /* Premier sommet de la face */

              isom = index_f_med[ifac    ] - index_f_med[0];

              elt_val_som_ent[entmail_e][ival++] = connect_med[isom];

              num_som_prec = connect_med[isom];
              num_som_deb  = connect_med[isom];

              nbr_som_fac = 1;

              /* Autres sommets de la face */

              for (isom = index_f_med[ifac    ] - index_f_med[0] + 1;
                   isom < index_f_med[ifac + 1] - index_f_med[0];
                   isom++) {

                if (   connect_med[isom] != num_som_prec
                    && connect_med[isom] != num_som_deb) {
                  elt_val_som_ent[entmail_e][ival++] = connect_med[isom];
                  num_som_prec = connect_med[isom];
                  nbr_som_fac += 1;
                }
                else
                  cpt_som_poly_dup += 1;

              }

              /* Si la face est dgnre (1 ou 2 sommets), on la supprime */

              if (nbr_som_fac < 3) {

                ival -= nbr_som_fac;
                cpt_fac_poly_dgn += 1;

              }
              else {

                /* On rajoute une seconde rfrence au premier sommet de
                   chaque face en fin de liste pour "fermer" cette face
                   (convention polydre Enveloppe Code_Saturne pour reprer
                   les faces d'un polydre en connectivit nodale) */

                isom = index_f_med[ifac    ] - index_f_med[0];

                elt_val_som_ent[entmail_e][ival++] = connect_med[isom];

              }

            }

            elt_pos_som_ent[entmail_e][cpt_elt_ent[entmail_e] + ielt + 1]
              = ival + 1;

          }

          assert((size_t)(taille_med + n_index_f_med - 1)
                 >= (  elt_pos_som_ent[entmail_e]
                                      [cpt_elt_ent[entmail_e] + ielt]
                     - elt_pos_som_ent[entmail_e]
                                      [cpt_elt_ent[entmail_e]])
                    + cpt_som_poly_dup + cpt_fac_poly_dgn);

          if (cpt_som_poly_dup > 0) {

            ecs_warn();
            bft_printf(_("While reading file \"%s\",\n"
                         "%d repeated references to the same vertices\n"
                         "and %d degenerate faces were encountered\n"
                         "in the definition of %d polyhedra.\n"),
                       fic_maillage->nom_fic, (int) cpt_som_poly_dup,
                       (int) cpt_fac_poly_dgn, (int) nbr_elt);

            BFT_REALLOC (elt_val_som_ent[entmail_e], ival + 1, ecs_int_t);

          }

          /* Libration connectivit temporaire */

          BFT_FREE(index_med);
          BFT_FREE(index_f_med);
          BFT_FREE(connect_med);

        }

        /* Lecture des numros de familles */

        ret_med = MEDfamLire(fic_maillage->fid,
                             nom_maillage_med,
                             fam_ele_med,
                             nbr_ele_med,
                             typ_ent_med,
                             typ_geo_med);


        /* Convention MED 2.2 : si pas de familles, numros = 0 */

        if (ret_med < 0) {
          for (ielt = 0; ielt < nbr_elt; ielt++)
            fam_ele_med[ielt] = 0;
          ret_med = 0;
        }
        else if (ret_med > 0) {
          ecs_warn();
          bft_printf(_("MED: erreur reading file \"%s\".\n"
                       "(families)."),
                     fic_maillage->nom_fic);
        }


        /* Changement de signe pour les familles des lments */

        for (ielt = 0; ielt < nbr_elt; ielt++) {

          elt_val_fam_ent[entmail_e][cpt_elt_ent[entmail_e] + ielt]
            = -fam_ele_med[ielt];

        }


        BFT_FREE(fam_ele_med);

        /* Lecture ventuelle des connectivits */

        if (edim_med == 2 && ind_equiv >= 0) {

          med_int ind_corres;
          med_int nbr_corres;
          med_int *corres = NULL;
          ecs_int_t *cpt_fac_connect = NULL;
          size_t nbr_fac = cpt_elt_ent[entmail_e];

          nbr_corres = MEDnCorres(fic_maillage->fid,
                                  nom_maillage_med,
                                  nom_equiv,
                                  typ_ent_med,
                                  typ_geo_med);

          if (nbr_corres < 0)
            ret_med = -1;

          else if (nbr_corres > 0) {

            BFT_MALLOC(corres, nbr_corres*2, med_int);

            ret_med = MEDequivLire(fic_maillage->fid,
                                   nom_maillage_med,
                                   nom_equiv,
                                   corres,
                                   nbr_corres,
                                   typ_ent_med,
                                   typ_geo_med);

          }

          if (ret_med < 0)
            bft_error(__FILE__, __LINE__, 0,
                      _("MED: error reading file \"%s\".\n"
                        "Error reading equivalence information."),
                      fic_maillage->nom_fic);

          /* Redimensionnement des structures */

          BFT_REALLOC(pos_fac_connect,
                      cpt_elt_ent[entmail_e] + 1 + nbr_elt, ecs_size_t);

          BFT_MALLOC(cpt_fac_connect, nbr_elt, ecs_int_t);

          for (ielt = 0; ielt < nbr_elt; ielt++)
            cpt_fac_connect[ielt] = 0;

          for (ind_corres = 0; ind_corres < nbr_corres; ind_corres++) {
            cpt_fac_connect[corres[ind_corres*2] - 1] += 1;
            cpt_fac_connect[corres[ind_corres*2 + 1] - 1] += 1;
          }

          if (nbr_fac == 0)
            pos_fac_connect[0] = 1;

          for (ielt = 0; ielt < nbr_elt; ielt++) {
            pos_fac_connect[nbr_fac + 1 + ielt]
              = pos_fac_connect[nbr_fac + ielt] + cpt_fac_connect[ielt];
            cpt_fac_connect[ielt] = 0;
          }

          BFT_REALLOC(val_fac_connect,
                      pos_fac_connect[nbr_fac + nbr_elt],
                      ecs_int_t);

          /* Mise  jour des structures */

          for (ind_corres = 0; ind_corres < nbr_corres; ind_corres++) {
            med_int icor1 = corres[ind_corres*2] - 1;
            med_int icor2 = corres[ind_corres*2 + 1] - 1;
            val_fac_connect[  pos_fac_connect[nbr_fac + icor1]
                            + cpt_fac_connect[icor1] - 1]
              = nbr_fac + icor2 + 1;
            val_fac_connect[  pos_fac_connect[nbr_fac + icor2]
                            + cpt_fac_connect[icor2] - 1]
              = nbr_fac + icor1 + 1;
            cpt_fac_connect[icor1] += 1;
            cpt_fac_connect[icor2] += 1;
          }

          BFT_FREE(cpt_fac_connect);
          BFT_FREE(corres);
        }

        cpt_elt_ent[entmail_e] += nbr_elt;


      } /* Fin : si le nombre d'lments de ce type MED n'est pas nul */


    } /* Fin : si le type gomtrique MED n'est pas le type `point' */


  } /* Fin : boucle sur les types d'elements MED */


  /* Transfert des valeurs lues dans les structures d'entit de maillage */
  /*=====================================================================*/


  vect_entmail = ecs_entmail_pre__cree_elt(cpt_elt_ent,
                                           elt_pos_som_ent,
                                           elt_val_som_ent,
                                           elt_val_label_ent,
                                           elt_val_fam_ent,
                                           elt_val_color_ent,
                                           cpt_coul_ent,
                                           val_coul_ent,
                                           cpt_elt_coul_ent);


  /* Ajout des connectivits de face ventuelles */

  if (pos_fac_connect != NULL) {

    ecs_champ_t *champ_connect_fac;

    champ_connect_fac = ecs_champ__cree(cpt_elt_ent[ECS_ENTMAIL_FAC],
                                        0,
                                        pos_fac_connect,
                                        val_fac_connect,
                                        ECS_TYPE_ecs_int_t,
                                        ECS_CHAMP_NOM_CONNECT,
                                        NULL,
                                        ECS_CHAMP_STATUT_REF_ELT);

  /* Transfert dans l'entit de maillage */

    ecs_entmail__ajoute_champ(vect_entmail[ECS_ENTMAIL_FAC],
                              champ_connect_fac,
                              ECS_CHAMP_ATT);

  }

  return vect_entmail;

}


/*----------------------------------------------------------------------------
 *                        Lecture des familles
 *----------------------------------------------------------------------------*/

static ecs_famille_t * * ecs_loc_pre_med__lit_famille
(
 const ecs_med_t  *const fic_maillage ,
 const char       *const nom_maillage
)
{

  char                * nom;

  ecs_int_t             deb_pos_sans_blanc;
  ecs_int_t             fin_pos_sans_blanc;
  ecs_int_t             icar;
  ecs_int_t             ide;
  ecs_int_t             ifam;
  ecs_int_t             ifam_ent;
  ecs_int_t             nbr_car;
  ecs_int_t             nbr_famille_ent[ECS_FAMILLE_FIN];
  ecs_int_t             num_fam;
  ecs_int_t             num_fam_ent;

  ecs_descr_t           * descr;
  ecs_descr_t           * descr_tete;

  ecs_famille_t         * famille;
  ecs_famille_t       * * vect_famille_tete;
  ecs_famille_t       * * liste_famille_ent[ECS_FAMILLE_FIN];


  /* Declarations des variables pour MED */
  /*-------------------------------------*/

  char        nom_maillage_med[MED_TAILLE_NOM + 1];
  char      * att_des_med;
  char      * grp_des_med;
  char        nom_fam_med[MED_TAILLE_NOM + 1];
  char        un_att_des_med[MED_TAILLE_DESC + 1];
  char        un_grp_des_med[MED_TAILLE_LNOM + 1];

  med_err     ret_med = 0;

  med_int     iatt_med;
  med_int     ifam_med;
  med_int     igrp_med;
  med_int     nbr_fam_med;
  med_int     nbr_att_med;
  med_int     nbr_grp_med;
  med_int     num_fam_med;
  med_int   * att_ide_med;
  med_int   * att_val_med;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  strcpy(nom_maillage_med, nom_maillage);


  /* On rcupere le nombre de familles */

  nbr_fam_med = MEDnFam(fic_maillage->fid,
                        nom_maillage_med);

  if (nbr_fam_med < 0)
    ret_med = -1;

  if (ret_med != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("MED: error reading file \"%s\".\n"
                "Number of families read: \"%d\""),
              fic_maillage->nom_fic, (int)nbr_fam_med);


  for (ifam_ent = ECS_FAMILLE_DEB; ifam_ent < ECS_FAMILLE_FIN; ifam_ent++) {

    nbr_famille_ent[ifam_ent] = 0;
    BFT_MALLOC(liste_famille_ent[ifam_ent],
               (ecs_int_t)nbr_fam_med, ecs_famille_t *);

  }


  BFT_MALLOC(vect_famille_tete, ECS_FAMILLE_FIN, ecs_famille_t *);
  for (ifam_ent = 0; ifam_ent < ECS_FAMILLE_FIN; ifam_ent++)
    vect_famille_tete[ifam_ent] = NULL;


  /*----------------------*/
  /* Lecture des familles */
  /*----------------------*/

  for (ifam_med = 0; ifam_med < nbr_fam_med; ifam_med++) {


    /*------------------------------------*/
    /* Rcupration du nombre d'attributs */
    /*------------------------------------*/

    nbr_att_med = MEDnAttribut(fic_maillage->fid,
                               nom_maillage_med,
                               ifam_med + 1);


    if (nbr_att_med < 0)
      ret_med = -1;

    if (ret_med != 0)
      bft_error(__FILE__, __LINE__, 0,
                _("MED: error reading file \"%s\".\n"
                  "Number of attributes read: \"%d\""),
                fic_maillage->nom_fic, (int)nbr_att_med);


    /*-----------------------------------*/
    /* Rcupration du nombre de groupes */
    /*-----------------------------------*/

    nbr_grp_med = MEDnGroupe(fic_maillage->fid,
                             nom_maillage_med,
                             ifam_med + 1);

    if (nbr_grp_med < 0)
      ret_med = -1;

    if (ret_med != 0)
      bft_error(__FILE__, __LINE__, 0,
                _("MED: error reading file \"%s\".\n"
                  "Number of groups read: \"%d\""),
                fic_maillage->nom_fic, (int)nbr_grp_med);


    /*--------------------------------------*/
    /* Lecture des attributs et des groupes */
    /*--------------------------------------*/

    if (nbr_att_med != 0) {

      BFT_MALLOC(att_ide_med, nbr_att_med, med_int);
      BFT_MALLOC(att_val_med, nbr_att_med, med_int);
      BFT_MALLOC(att_des_med, nbr_att_med * MED_TAILLE_DESC + 1, char);

    }
    else {

      att_ide_med = NULL;
      att_val_med = NULL;
      att_des_med = NULL;

    }


    if (nbr_grp_med != 0) {

      BFT_MALLOC(grp_des_med, nbr_grp_med * MED_TAILLE_LNOM + 1, char);

    }
    else {

      grp_des_med = NULL;

    }

    ret_med = MEDfamInfo(fic_maillage->fid,
                         nom_maillage_med,
                         ifam_med + 1,
                         nom_fam_med,
                         &num_fam_med,
                         att_ide_med,
                         att_val_med,
                         att_des_med,
                         &nbr_att_med,
                         grp_des_med,
                         &nbr_grp_med);


    if (num_fam_med != 0) {

      descr_tete = NULL;

      if (nbr_att_med == 0 && nbr_grp_med == 0)
        bft_printf(_("  Family %2d is described by no attribute or group.\n"),
                   (int)(ECS_ABS(num_fam_med)));


      for (iatt_med = 0; iatt_med < nbr_att_med; iatt_med++) {

        /* Rcupration de la valeur entire du descripteur */
        /*--------------------------------------------------*/

        ide =  (ecs_int_t)att_val_med[iatt_med];

        /* Recupration du descripteur */
        /*-----------------------------*/

        strncpy(un_att_des_med,
                att_des_med + iatt_med * MED_TAILLE_DESC, MED_TAILLE_DESC);
        un_att_des_med[MED_TAILLE_DESC] = '\0';

        /* On regarde si la chane ne contient pas que des blancs */
        /* On ne garde que la partie de la chane                 */
        /*  qui n'a pas de blancs aux extrmits                  */

        deb_pos_sans_blanc = 0;
        while (*(un_att_des_med + deb_pos_sans_blanc) != '\0' &&
               *(un_att_des_med + deb_pos_sans_blanc) == ' '     )
          deb_pos_sans_blanc++;

        for (fin_pos_sans_blanc = deb_pos_sans_blanc;
             (   fin_pos_sans_blanc < MED_TAILLE_DESC - 1
               && *(un_att_des_med + fin_pos_sans_blanc) != '\0');
             fin_pos_sans_blanc++);

        if (   fin_pos_sans_blanc > deb_pos_sans_blanc
            && deb_pos_sans_blanc < MED_TAILLE_DESC) {

          /* La chane ne contient pas que des blancs */

          while (fin_pos_sans_blanc                     != 0    &&
                 *(un_att_des_med + fin_pos_sans_blanc) == ' '     )
            fin_pos_sans_blanc--;

          nbr_car = fin_pos_sans_blanc - deb_pos_sans_blanc + 1;
          BFT_MALLOC(nom, nbr_car + 1, char);
          for (icar = 0; icar < nbr_car; icar++)
            *(nom + icar) = *(un_att_des_med + deb_pos_sans_blanc + icar);
          *(nom + nbr_car) = '\0';

        }
        else {

          nom   = NULL;

        }

        descr = ecs_descr__cree(ECS_DESCR_TYP_COULEUR,
                                ide,
                                nom);

        ecs_descr_chaine__ajoute(&descr_tete,
                                 descr);

        if (nom != NULL)
          BFT_FREE(nom);

      } /* Fin : boucle sur les attributs de la famille */


      for (igrp_med = 0; igrp_med < nbr_grp_med; igrp_med++) {

        /* Rcuperation de la chane de caracteres du descripteur */

        strncpy(un_grp_des_med,
                grp_des_med + igrp_med * MED_TAILLE_LNOM, MED_TAILLE_LNOM);
        un_grp_des_med[MED_TAILLE_LNOM] = '\0';

        /* On regarde si la chane ne contient pas que des blancs */
        /* On ne garde que la partie de la chane                 */
        /*  qui n'a pas de blancs aux extrmits                  */

        deb_pos_sans_blanc = 0;
        while (*(un_grp_des_med + deb_pos_sans_blanc) != '\0' &&
               *(un_grp_des_med + deb_pos_sans_blanc) == ' '     )
          deb_pos_sans_blanc++;


        /* On verifie que la chane ne contient pas que des blancs */
        assert(deb_pos_sans_blanc < MED_TAILLE_LNOM + 1);

        fin_pos_sans_blanc = MED_TAILLE_LNOM - 1;
        while (fin_pos_sans_blanc              != 0    &&
               *(un_grp_des_med + fin_pos_sans_blanc) == ' '     )
          fin_pos_sans_blanc--;

        nbr_car = fin_pos_sans_blanc - deb_pos_sans_blanc + 1;
        BFT_MALLOC(nom, nbr_car + 1, char);
        for (icar = 0; icar < nbr_car; icar++)
          *(nom + icar) = *(un_grp_des_med + deb_pos_sans_blanc + icar);
        *(nom + nbr_car) = '\0';


        /* Pas de valeur entire associe */

        descr = ecs_descr__cree(ECS_DESCR_TYP_GROUPE,
                                ECS_DESCR_IDE_NUL,
                                nom);

        ecs_descr_chaine__ajoute(&descr_tete,
                                 descr);

        if (nom != NULL)
          BFT_FREE(nom);

      } /* Fin : boucle sur les groupes de la famille */


      /* Dtermination de l'entit concerne par la famille */

      if (num_fam_med < 0) {

        /* On accroche toutes les familles des lments */
        /*  sur les familles des cellules               */

        num_fam_ent = ECS_FAMILLE_CEL;

      }
      else {

        num_fam_ent = ECS_FAMILLE_SOM;

      }

      num_fam = (ecs_int_t)num_fam_med;

      famille = ecs_famille__cree(num_fam, descr_tete);

      liste_famille_ent[num_fam_ent][nbr_famille_ent[num_fam_ent]] = famille;
      nbr_famille_ent[num_fam_ent]++;


    } /* Fin : si la famille n'est pas la famille par dfaut de MED */


    if (nbr_att_med != 0) {

      BFT_FREE(att_ide_med);
      BFT_FREE(att_val_med);
      BFT_FREE(att_des_med);

    }
    if (nbr_grp_med != 0) {

      BFT_FREE(grp_des_med);

    }


  } /* Fin : boucle sur les familles lues */


  for (ifam_ent = ECS_FAMILLE_DEB; ifam_ent < ECS_FAMILLE_FIN; ifam_ent++) {

    for (ifam = 0; ifam < nbr_famille_ent[ifam_ent]; ifam++) {

      ecs_famille_chaine__ajoute(&vect_famille_tete[ifam_ent],
                                 liste_famille_ent[ifam_ent][ifam]);
    }

  }


  for (ifam_ent = ECS_FAMILLE_DEB; ifam_ent < ECS_FAMILLE_FIN; ifam_ent++)
    BFT_FREE(liste_famille_ent[ifam_ent]);


  return vect_famille_tete;


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un pointeur sur un tableau de type `ecs_int_t'
 *   dont les valeurs sont converties si ncessaire
 *    partir du tableau de connectivit `val_med' ayant `nbr_val' valeurs
 *   de type `med_int', avec `pas_med' valeurs MED et `pas_ecs' valeurs
 *   locales par lement.
 *
 *  Cette fonction supprime donc les rfrences inutiles (noeuds
 *   quadratiques ou connectivits supplmentaires ventuelles)
 *
 *  Le tableau en entre `val_med' est libr si le tableau renvoy est allou
 *----------------------------------------------------------------------------*/

static ecs_int_t * ecs_loc_convert_connect_med_ecs
(
       med_int    * val_med ,
 const ecs_int_t    nbr_val,
 const ecs_int_t    pas_med,
 const ecs_int_t    pas_ecs
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(val_med != NULL);
  assert(nbr_val != 0   );
  assert(pas_med != 0   );
  assert(pas_ecs != 0   );


  if ((pas_med == pas_ecs) && (sizeof(med_int) == sizeof(ecs_int_t))) {

    /* Le nombre de rfrences par lment    */
    /* MED est le mme que celui par lment  */
    /* local (lment linaire, pas de        */
    /* connectivit supplmentaire)           */

    /* Les entiers de type `med_int' et       */
    /* les entiers de type `ecs_int_t'        */
    /* sont cods sur le meme nombre d'octets */

    /* Aucune conversion n'est ncessaire     */

    return (ecs_int_t *)val_med;

  }
  else {

    ecs_int_t    ielt;
    ecs_int_t    iloc;
    ecs_int_t    ipos_ecs;
    ecs_int_t    ipos_med;
    ecs_int_t    nbr_val_ecs;
    ecs_int_t  * val_ecs;

    /* Les entiers de type `med_int' et              */
    /* les entiers de type `ecs_int_t'               */
    /* ne sont pas codes sur le mme nombre d'octets */

    /* On effectue la conversion de type pour chaque valeur */

    nbr_val_ecs = (nbr_val / pas_med) * pas_ecs;

    BFT_MALLOC(val_ecs, nbr_val_ecs, ecs_int_t);

    ipos_ecs = 0;

    for (ielt = 0; ielt < (nbr_val / pas_med); ielt++) {

      ipos_med = ielt * pas_med;

      for (iloc = 0; iloc < pas_ecs; iloc++)
        val_ecs[ipos_ecs++] = (ecs_int_t)val_med[ipos_med++];

    }

    BFT_FREE(val_med);

    return val_ecs;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un pointeur sur un tableau de type `ecs_real_t'
 *   dont les valeurs sont converties si ncessaire
 *    partir du tableau `val_med' ayant `nbr_val' valeurs de type `med_float'
 *
 *  Le tableau en entre `val_med' est libere si le tableau renvoy est allou
 *----------------------------------------------------------------------------*/

static ecs_real_t * ecs_loc_convert_real_med_ecs
(
       med_float * val_med ,
 const ecs_int_t   nbr_val
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(val_med != NULL);
  assert(nbr_val != 0   );


  if (sizeof(med_float) == sizeof(ecs_real_t)) {

    /* Les rels de type `med_float' et       */
    /* les rels de type `ecs_real_t'               */
    /* sont cods sur le mme nombre d'octets */

    /* Aucune conversion n'est ncessaire */

    return (ecs_real_t *)val_med;

  }
  else {


    ecs_int_t   ival   ;
    ecs_real_t    * val_ecs;


    /* Les rels de type `med_float' et              */
    /* les rels de type `ecs_real_t'                      */
    /* ne sont pas cods sur le mme nombre d'octets */

    /* On effectue la conversion de type pour chaque valeur */

    BFT_MALLOC(val_ecs, nbr_val, ecs_real_t);

    for (ival = 0; ival < nbr_val; ival++)
      val_ecs[ival] = (ecs_real_t)val_med[ival];

    BFT_FREE(val_med);

    return val_ecs;

  }


}


#endif /* HAVE_MED */
