/*============================================================================
 *  Dfinition de la fonction
 *   de lecture d'un fichier de maillage au format CGNS
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/


#include "ecs_config.h"

#if defined(HAVE_CGNS)


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include <bft_error.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_chaine_glob.h"
#include "ecs_elt_typ_liste.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr.h"
#include "ecs_descr_chaine.h"
#include "ecs_champ.h"
#include "ecs_champ_att.h"
#include "ecs_entmail.h"
#include "ecs_maillage.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail_pre.h"
#include "ecs_cgns_def.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_pre_cgns.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/


/*============================================================================
 *                  Dfinitions de paramtres et macros
 *============================================================================*/

#define ECS_CGNS_TAILLE_NOM    32      /* Longueur nom max (la documentation
                                          ne prcise rien, mais les exemples
                                          CGNS et le fichier ADF.h prvoient
                                          des chanes de 32 caractres) */

#define ECS_CGNS_SSELT_NBR_MAX_SOM   4  /* Nombre maximal de sommets
                                           dfinissant un sous-lment */


/* Compatibilit avec diverses versions de CGNS */

#if !defined(CG_OK)
#define CG_OK  ALL_OK
#endif


/*============================================================================
 *                  Dfinition de structures locales
 *============================================================================*/

/* Dfinition d'une base CGNS */

typedef struct {

  char                * nom_fic       ; /* Nom du fichier CGNS                */
  int                   num_fic       ; /* Identificateur de fichier CGNS     */

  int                   num_base      ; /* Identificateur de base CGNS        */

  int                   dim_espace    ; /* Dimension de l'espace associ      */
  int                   dim_entite    ; /* Dimension entits maillage         */

} ecs_loc_cgns_base_t ;


/* Dfinition de conditions aux limites */

typedef struct {
  int              ind_nom                        ; /* Numro de nom associ  */
  int              num_zone                       ; /* Num zone associe      */
  GridLocation_t   support                        ; /* Support associ        */
  PointSetType_t   ptset_type                     ; /* Type de liste          */
  int              npnts                          ; /* Nombre de points       */
  int             *pnts                           ; /* Liste de points        */
} ecs_loc_cgns_boco_t ;


/* Dfinition d'une section (pour zones non structures) */

typedef struct {

  char           nom[ECS_CGNS_TAILLE_NOM + 1] ;     /* Nom de la section      */
  ElementType_t  type                             ; /* Type lment           */
  int            num_elt_deb                      ; /* Numro premier lement */
  int            num_elt_fin                      ; /* Numro dernier lement */
  int            nbr_brd                          ; /* Nbr. lments de bord  */
  int            parent                           ; /* 0 si pas de parents,
                                                       1 sinon                */
  int           *elems                            ; /* Connect. temporaire    */
} ecs_loc_cgns_section_t ;


/* Dfinition d'une zone */

typedef struct {

  char            nom[ECS_CGNS_TAILLE_NOM + 1]  ; /* Nom de la zone          */
  ZoneType_t      type                          ; /* Type de la zone         */
  int             nbr_som                       ; /* Nombre de sommets       */
  int             nbr_cel                       ; /* Nombre de cellules      */
  int             taille[3*3]                   ; /* Indices maximaux pour
                                                     les sommets, cellules,
                                                     et cellules de bord     */
  int             nbr_sections                  ; /* Nombre de sections      */
  int             num_som_deb                   ; /* Numro premier sommet   */
  int             num_elt_deb                   ; /* Numro premier lement  */
  int             num_elt_fin                   ; /* Numro dernier lement  */
  AngleUnits_t    angle                         ; /* Units angles           */
  ecs_loc_cgns_section_t  *tab_sections         ; /* Descriptions sections   */
  ecs_bool_t      trait_renum                   ; /* Utilisation de renum ?  */
  int            *renum                         ; /* Numro CGNS -> ECS      */

} ecs_loc_cgns_zone_t ;


/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Cration d'une structure `ecs_loc_cgns_base_t' et ouverture d'un
 *  fichier CGNS en lecture
 *----------------------------------------------------------------------------*/

static ecs_loc_cgns_base_t *ecs_loc_pre_cgns__cree
(
 const char       *const nom_fichier,
       ecs_int_t         num_maillage
) ;


/*----------------------------------------------------------------------------
 *  Fermeture d'un fichier CGNS en lecture et destruction de la structure
 *  `ecs_loc_cgns_base_t' associe
 *----------------------------------------------------------------------------*/

static ecs_loc_cgns_base_t * ecs_loc_pre_cgns__detruit
(
 ecs_loc_cgns_base_t  * base
) ;


/*----------------------------------------------------------------------------
 *  Cration d'une structure `ecs_loc_cgns_base_t' et ouverture d'un
 *  fichier CGNS en lecture
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__aff_titre_cas
(
 const int          num_fic  ,
 const int          num_base ,
 const char  *const nom_rub
) ;


/*----------------------------------------------------------------------------
 *                 Lecture d'informations sur les zones
 *----------------------------------------------------------------------------*/

static ecs_loc_cgns_zone_t * ecs_loc_pre_cgns__lit_zones
(
 const ecs_loc_cgns_base_t  *const base_maillage ,
 const int                         nzones
) ;


/*----------------------------------------------------------------------------
 *                    Lecture des conditions aux limites
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__lit_boco
(
 const ecs_loc_cgns_base_t      *const base_maillage ,
 const int                             nzones        ,
       ecs_loc_cgns_zone_t      *const tab_zone      ,
       int                      *const nbr_nom_boco  ,
       int                      *const nbr_boco_tot  ,
       char                 * * *const tab_nom_boco  ,
       ecs_loc_cgns_boco_t    * *const tab_boco
) ;


/*----------------------------------------------------------------------------
 *                         Lecture des sommets
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * ecs_loc_pre_cgns__lit_som
(
 const ecs_loc_cgns_base_t  *const base_maillage ,
 const int                         nzones        ,
       ecs_loc_cgns_zone_t  *const tab_zone
) ;


/*----------------------------------------------------------------------------
 *                        Lecture des lments
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * * ecs_loc_pre_cgns__lit_ele
(
 const ecs_loc_cgns_base_t      *const base_maillage      ,
 const int                             nzones             ,
       ecs_loc_cgns_zone_t      *const tab_zone           ,
 const int                             nbr_boco_tot       ,
       ecs_loc_cgns_boco_t      *const tab_boco           ,
       ecs_int_t                *const nbr_boco_som       ,
       ecs_int_t              * *const ind_nom_boco_som   ,
       ecs_int_t              * *const nbr_sselt_boco_som ,
       ecs_int_t            * * *const tab_sselt_boco_som ,
       ecs_int_t              * *const ind_section_cel    ,
       ecs_int_t              * *const ind_section_fac    ,
       ecs_int_t              * *const ind_zone_cel       ,
       ecs_int_t              * *const ind_zone_fac
) ;


/*----------------------------------------------------------------------------
 *  Cration des groupes bass sur les C.L.
 *   et de manire optionnelle sur les zones et les sections
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__cree_groupes
(
 const ecs_loc_cgns_base_t           *const base_maillage       ,
 const int                                  nzones              ,
       ecs_loc_cgns_zone_t           *const tab_zone            ,
 const int                                  nbr_boco_tot        ,
       char                   *const *const tab_nom_boco        ,
       ecs_loc_cgns_boco_t           *const tab_boco            ,
 const ecs_int_t                            nbr_boco_som        ,
       ecs_int_t              *      *const ind_nom_boco_som    ,
       ecs_int_t              *      *const nbr_sselt_boco_som  ,
       ecs_int_t            * *      *const tab_sselt_boco_som  ,
       ecs_int_t                     *const tab_ind_section_cel ,
       ecs_int_t                     *const tab_ind_section_fac ,
       ecs_int_t                     *const tab_ind_zone_cel    ,
       ecs_int_t                     *const tab_ind_zone_fac    ,
       ecs_entmail_t          *      *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *                   Cration des groupes bass sur les C.L.
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__cree_grps_boco
(
 const ecs_loc_cgns_base_t           *const base_maillage      ,
       ecs_loc_cgns_zone_t           *const tab_zone           ,
 const int                                  nbr_boco_tot       ,
       char                   *const *const tab_nom_boco       ,
       ecs_loc_cgns_boco_t           *const tab_boco           ,
 const ecs_int_t                            nbr_boco_som       ,
       ecs_int_t              *      *const ind_nom_boco_som   ,
       ecs_int_t              *      *const nbr_sselt_boco_som ,
       ecs_int_t            * *      *const tab_sselt_boco_som ,
       size_t                        *const ent_nbr_elt        ,
       ecs_int_t                     *const ent_cpt_grp        ,
       ecs_champ_t            *      *const ent_champ_grp
) ;


/*----------------------------------------------------------------------------
 *            Cration des groupes bass sur les zones et sections
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__cree_grps_zs
(
 const ecs_loc_cgns_base_t           *const base_maillage       ,
 const int                                  nzones              ,
       ecs_loc_cgns_zone_t           *const tab_zone            ,
       ecs_int_t                     *const tab_ind_section_cel ,
       ecs_int_t                     *const tab_ind_section_fac ,
       ecs_int_t                     *const tab_ind_zone_cel    ,
       ecs_int_t                     *const tab_ind_zone_fac    ,
       size_t                        *const ent_nbr_elt         ,
       ecs_int_t                     *const ent_cpt_grp         ,
       ecs_champ_t                 * *const ent_champ_grp
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cr des entits de maillage supplmentaires pour porter
 *  des conditionrs aux limites dfinies sur les sommets.
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre__cgns__cree_ent_inf_som
(
 const ecs_loc_cgns_base_t      *const base_maillage         ,
       size_t                          nbr_elt_ent        [] ,
       ecs_size_t               *      elt_pos_som_ent    [] ,
       ecs_int_t                *      elt_val_som_ent    [] ,
 const int                             nzones                ,
       ecs_loc_cgns_zone_t      *const tab_zone              ,
 const int                             nbr_boco_tot          ,
       ecs_loc_cgns_boco_t      *const tab_boco              ,
       ecs_int_t                *const nbr_boco_som          ,
       ecs_int_t              * *const ind_nom_boco_som      ,
       ecs_int_t              * *const nbr_sselt_boco_som    ,
       ecs_int_t            * * *const tab_sselt_boco_som
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui marque les sommets correspondant  une liste.
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__marque_som
(
 const ecs_loc_cgns_base_t  *const base_maillage ,
 const int                         num_marque    ,
       ecs_loc_cgns_zone_t  *const ptr_zone      ,
 const PointSetType_t              ptset_type    ,
 const int                         npnts         ,
 const int                  *const pnts          ,
       ecs_int_t            *const indic_som
) ;


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Lecture d'un fichier au format CGNS
 *   et affectation des donnes dans la structure de maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_pre_cgns__lit_maillage
(
 const char       *const nom_fic_maillage     ,  /* --> Nom du fichier  lire */
 const ecs_int_t         num_maillage         ,  /* --> Numro du maillage    */
 const ecs_bool_t        cree_grp_cel_section ,
 const ecs_bool_t        cree_grp_cel_zone    ,
 const ecs_bool_t        cree_grp_fac_section ,
 const ecs_bool_t        cree_grp_fac_zone
)
{

  int                ind ;
  int                nbr_nom_boco ;
  int                nbr_boco_tot ;

  ecs_int_t          nbr_boco_som ;
  ecs_int_t        * ind_nom_boco_som ;
  ecs_int_t        * nbr_sselt_boco_som ;
  ecs_int_t       ** tab_sselt_boco_som ;

  char            ** tab_nom_boco ;

  ecs_int_t        * tab_ind_section_fac ;
  ecs_int_t        * tab_ind_section_cel ;
  ecs_int_t        * tab_ind_zone_cel ;
  ecs_int_t        * tab_ind_zone_fac ;

  ecs_int_t       ** ptr_tab_ind_section_fac ;
  ecs_int_t       ** ptr_tab_ind_section_cel ;
  ecs_int_t       ** ptr_tab_ind_zone_cel ;
  ecs_int_t       ** ptr_tab_ind_zone_fac ;

  ecs_int_t          ient ;

  ecs_entmail_t    * vect_entmail[ECS_ENTMAIL_FIN];
  ecs_entmail_t  * * vect_entmail_ret ;

  ecs_maillage_t   * maillage = NULL ;

  ecs_loc_cgns_base_t  * base_maillage ;
  ecs_loc_cgns_zone_t  *tab_zone ;
  ecs_loc_cgns_boco_t  *tab_boco ;

  /* Dclarations des variables pour CGNS */
  /*--------------------------------------*/

  char       nom_tmp[ECS_CGNS_TAILLE_NOM + 1] ;

  int        num_fic, num_base, cell_dim, phys_dim, nzones ;
  int        ret_cgns = 0 ;
  float      version_cgns ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  bft_printf(_("\n\n"
               "Reading mesh from file in CGNS format\n"
               "----------------------\n")) ;

  bft_printf(_("  Mesh file: %s\n\n\n"),
             nom_fic_maillage) ;


  /* Ouverture du fichier CGNS en lecture */
  /*--------------------------------------*/

  base_maillage = ecs_loc_pre_cgns__cree(nom_fic_maillage,
                                         num_maillage) ;

  num_fic  = base_maillage->num_fic ;
  num_base = base_maillage->num_base ;

  ret_cgns = cg_version(num_fic, &version_cgns) ;

  if (ret_cgns == 0)
    bft_printf(_("  CGNS version       : %3.1f\n"), version_cgns) ;


  /* Rcupration titre si possible */

  ecs_loc_pre_cgns__aff_titre_cas(num_fic,
                                  num_base,
                                  _("  Title              : ")) ;

  /* Autres informations sur la base */

  if (cg_base_read(num_fic, num_base, nom_tmp, &cell_dim, &phys_dim) != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("CGNS error:\n%s"), cg_get_error()) ;

  bft_printf(_("  Base name          : %s\n"), nom_tmp) ;
  bft_printf(_("  Physical dimension : %d\n"), phys_dim) ;
  bft_printf(_("  Cell dimension     : %d\n"), cell_dim) ;

  base_maillage->dim_entite = cell_dim ;
  base_maillage->dim_espace = phys_dim ;

  /* Nombre de zones */

  if (cg_nzones(num_fic, num_base, &nzones) != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("CGNS error:\n%s"), cg_get_error()) ;
  bft_printf(_("  Number of zones    : %d\n"), nzones) ;

  if (cell_dim < 2)
    bft_error(__FILE__, __LINE__, 0,
              _("Mesh entities are of dimension: %d\n"),
              cell_dim) ;


  /* Lecture des informations principales sur les zones */
  /*----------------------------------------------------*/

  tab_zone = ecs_loc_pre_cgns__lit_zones(base_maillage,
                                         nzones) ;

  /* Lecture des conditions aux limites */
  /*------------------------------------*/

  ecs_loc_pre_cgns__lit_boco(base_maillage,
                             nzones,
                             tab_zone,
                             &nbr_nom_boco,
                             &nbr_boco_tot,
                             &tab_nom_boco,
                             &tab_boco) ;

  /* Lecture des sommets */
  /*---------------------*/

  vect_entmail[ECS_ENTMAIL_SOM] = ecs_loc_pre_cgns__lit_som(base_maillage,
                                                            nzones,
                                                            tab_zone) ;

  /* Lecture des lments */
  /*----------------------*/

  tab_ind_section_fac = NULL ;
  tab_ind_section_cel = NULL ;
  tab_ind_zone_cel    = NULL ;
  tab_ind_zone_fac    = NULL ;

  ptr_tab_ind_section_cel
    = (cree_grp_cel_section == ECS_TRUE) ? &tab_ind_section_cel : NULL ;
  ptr_tab_ind_section_fac
    = (cree_grp_fac_section == ECS_TRUE) ? &tab_ind_section_fac : NULL ;
  ptr_tab_ind_zone_cel
    = (cree_grp_cel_zone == ECS_TRUE) ? &tab_ind_zone_cel : NULL ;
  ptr_tab_ind_zone_fac
    = (cree_grp_fac_zone == ECS_TRUE) ? &tab_ind_zone_fac : NULL ;

  vect_entmail_ret = ecs_loc_pre_cgns__lit_ele(base_maillage,
                                               nzones,
                                               tab_zone,
                                               nbr_boco_tot,
                                               tab_boco,
                                               &nbr_boco_som,
                                               &ind_nom_boco_som,
                                               &nbr_sselt_boco_som,
                                               &tab_sselt_boco_som,
                                               ptr_tab_ind_section_cel,
                                               ptr_tab_ind_section_fac,
                                               ptr_tab_ind_zone_cel,
                                               ptr_tab_ind_zone_fac) ;

  for (ient = ECS_ENTMAIL_ARE ; ient < ECS_ENTMAIL_FIN ; ient++)
    vect_entmail[ient] = vect_entmail_ret[ient] ;

  BFT_FREE(vect_entmail_ret) ;


  /* Cration des groupes */
  /*----------------------*/

  ecs_loc_pre_cgns__cree_groupes(base_maillage,
                                 nzones,
                                 tab_zone,
                                 nbr_boco_tot,
                                 tab_nom_boco,
                                 tab_boco,
                                 nbr_boco_som,
                                 &ind_nom_boco_som,
                                 &nbr_sselt_boco_som,
                                 &tab_sselt_boco_som,
                                 tab_ind_section_cel,
                                 tab_ind_section_fac,
                                 tab_ind_zone_cel,
                                 tab_ind_zone_fac,
                                 vect_entmail) ;


  /* Libration mmoire */

  BFT_FREE(tab_ind_section_cel) ;
  BFT_FREE(tab_ind_section_fac) ;
  BFT_FREE(tab_ind_zone_cel) ;
  BFT_FREE(tab_ind_zone_fac) ;

  for (ind = 0 ; ind < nbr_nom_boco ; ind++)
    BFT_FREE(tab_nom_boco[ind]) ;
  BFT_FREE(tab_nom_boco) ;

  BFT_FREE(tab_boco) ;

  for (ind = 0 ; ind < nzones ; ind++) {
    if ((tab_zone + ind)->tab_sections != NULL)
      BFT_FREE((tab_zone + ind)->tab_sections) ;
    if ((tab_zone + ind)->renum != NULL)
      BFT_FREE((tab_zone + ind)->renum) ;
  }
  BFT_FREE(tab_zone) ;


  maillage = ecs_maillage__cree_nodal((ecs_dim_t) phys_dim,
                                      vect_entmail) ;


  /* Fermeture du fichier de lecture du maillage */

  ecs_loc_pre_cgns__detruit(base_maillage) ;


  /* Renvoi de la structure de maillage */

  return maillage ;

}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Cration d'une structure `ecs_loc_cgns_base_t' et ouverture d'un
 *  fichier CGNS en lecture
 *----------------------------------------------------------------------------*/

static ecs_loc_cgns_base_t *ecs_loc_pre_cgns__cree
(
 const char       *const nom_fichier,
       ecs_int_t         num_maillage
)
{

  ecs_loc_cgns_base_t  * base ;

  char  nom_tmp[ECS_CGNS_TAILLE_NOM + 1] ;
  int   nbases, cell_dim, phys_dim ;
  int   ind ;
  int   ret ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  /* Base et fichier associs */

  BFT_MALLOC(base, 1, ecs_loc_cgns_base_t) ;

  BFT_MALLOC(base->nom_fic, strlen(nom_fichier) + 1, char) ;
  strcpy(base->nom_fic, nom_fichier) ;

  ret = cg_open(base->nom_fic, MODE_READ, &(base->num_fic)) ;

  if (ret < 0)
    bft_error(__FILE__, __LINE__, 0,
              _("CGNS: error opening file \"%s\":\n%s"),
              nom_fichier, cg_get_error());


  /* Vrification du nombre de bases (affichage nom si plusieurs) */

  if (cg_nbases(base->num_fic, &nbases) != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("CGNS error:\n%s"), cg_get_error()) ;

  if (nbases > 1) {
    bft_printf(_("\n  The file contains multiple bases:\n")) ;
    for (ind = 0 ; ind < nbases ; ind++) {
      if (   cg_base_read(base->num_fic, ind + 1, nom_tmp, &cell_dim, &phys_dim)
          != 0)
        bft_error(__FILE__, __LINE__, 0,
                  _("CGNS error:\n%s"), cg_get_error()) ;
      bft_printf(_("    Base %2d: %s\n"), ind + 1, nom_tmp) ;
    }
    if (num_maillage == 0)
      bft_printf(_("\n  No base was requested; "
                   "the first one is used\n\n")) ;
    else if (num_maillage > 0)
      bft_printf(_("\n  Base number %d was requested\n\n"), num_maillage) ;
  }
  else if (nbases == 0)
    bft_error(__FILE__, __LINE__, 0,
              _("no CGNS base in file.\n")) ;

  assert (num_maillage >= 0) ;
  if (num_maillage > nbases)
    bft_error(__FILE__, __LINE__, 0,
              _("The mesh number requested (%d) is higher than\n"
                "the number of bases defined (%d) in file\n%s.\n"),
              num_maillage, nbases, nom_fichier) ;
  else
    base->num_base = ECS_MAX(1, num_maillage) ;


  /* Fin d'initialisation et renvoi de la structure */

  base->dim_entite = 0 ;
  base->dim_espace = 0 ;

  return base ;

}


/*----------------------------------------------------------------------------
 *  Fermeture d'un fichier CGNS en lecture et destruction de la structure
 *  `ecs_loc_cgns_base_t' associe
 *----------------------------------------------------------------------------*/

static ecs_loc_cgns_base_t * ecs_loc_pre_cgns__detruit
(
 ecs_loc_cgns_base_t  * base
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(base != NULL) ;

  if (cg_close(base->num_fic) != 0)
    bft_error(__FILE__, __LINE__, 0,
              _("CGNS: error closing file \"%s\":\n%s"),
              base->nom_fic, cg_get_error());

  BFT_FREE(base->nom_fic) ;

  BFT_FREE(base) ;

  return base ;

}


/*----------------------------------------------------------------------------
 *  Affichage du titre associ  une base CGNS
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__aff_titre_cas
(
 const int          num_fic  ,
 const int          num_base ,
 const char  *const nom_rub
)
{
  char  nom_tmp[ECS_CGNS_TAILLE_NOM + 1] ;
  int   n_desc, ind_desc ;
  int   ind_cur, ind_fin, ind_deb, l_dec ;

  const int l_ligne = 72 ;

  char *text = NULL ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  if (   cg_goto(num_fic, num_base, "end") == 0
      && cg_ndescriptors(&n_desc) == 0) {

    if (n_desc > 0) {
      for (ind_desc = 0 ; ind_desc < n_desc ; ind_desc++) {

        /*
          L'appel cg_descriptor_read alloue text ;
          on devra dsallouer aprs utilisation
        */

        if (   cg_descriptor_read(ind_desc + 1, nom_tmp, &text) == 0
            && strcmp(nom_tmp, "CaseTitle") == 0) {

          bft_printf("%s", nom_rub) ;

          l_dec = strlen(nom_rub) ;
          ind_fin = strlen(text) ;
          ind_deb   = 0 ;
          ind_cur   = l_ligne - l_dec ;

          /*
            Dcoupage de l'affichage pour placer des retours  la
            ligne sur titres longs
          */

          while (ind_cur < ind_fin) {

            while (   ind_cur > ind_deb
                   && text[ind_cur] != ' '
                   && text[ind_cur] != '\n'
                   && text[ind_cur] != '\t')
              ind_cur-- ;

            if (ind_cur > ind_deb)
              text[ind_cur] = '\0' ;
            else
              break ;

            bft_printf("%s\n%*s", text + ind_deb, l_dec, " ") ;

            ind_deb = ind_cur + 1 ;
            while (   ind_deb < ind_fin
                   && text[ind_deb] == ' '
                   && text[ind_deb] == '\n'
                   && text[ind_deb] == '\t')
              ind_deb++ ;

            ind_cur = ind_deb + l_ligne - l_dec ;

          }

          if (ind_deb < ind_fin)
            bft_printf("%s\n", text + ind_deb) ;

        }
        free(text);
      }

    }

  }

}


/*----------------------------------------------------------------------------
 *                 Lecture d'informations sur les zones
 *----------------------------------------------------------------------------*/

static ecs_loc_cgns_zone_t * ecs_loc_pre_cgns__lit_zones
(
 const ecs_loc_cgns_base_t  *const base_maillage ,
 const int                         nzones
)
{

  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  char          nom_tmp [ECS_CGNS_TAILLE_NOM + 1] ;

  const char   *nom_coo_type[] = {N_("cartesian"),
                                  N_("cylindrical"),
                                  N_("spherical")} ;

  ecs_int_t     coo_type ;

  ecs_int_t     nbr_cel_loc ;
  ecs_int_t     nbr_som_loc ;
  ecs_int_t     nbr_som_tot ;

  ecs_bool_t    connect_1to1 ;
  ecs_bool_t    connect_abutt ;
  ecs_bool_t    connect_overset ;

  ecs_loc_cgns_zone_t     *ptr_zone ;
  ecs_loc_cgns_section_t  *ptr_section ;

  ecs_loc_cgns_zone_t     *tab_zone ;


  /* Dclarations des variables pour CGNS */
  /*-------------------------------------*/

  int         ind_conn ;
  int         ind_section ;
  int         ind_zone ;
  int         nconns ;
  int         num_fic ;
  int         num_base ;
  int         num_section ;
  int         num_zone ;
  int         ngrids ;

  MassUnits_t         mass ;
  LengthUnits_t       length ;
  TimeUnits_t         time ;
  TemperatureUnits_t  temperature ;
  AngleUnits_t        angle;

  DataType_t  type_coord_lu ;

  int         ret = 0 ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  num_fic  = base_maillage->num_fic ;
  num_base = base_maillage->num_base ;

  BFT_MALLOC(tab_zone, nzones, ecs_loc_cgns_zone_t) ;

  nbr_som_tot = 0 ;

  connect_1to1    = ECS_FALSE ;
  connect_abutt   = ECS_FALSE ;
  connect_overset = ECS_FALSE ;


  /* Boucle sur les zones */
  /*----------------------*/

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    ptr_zone->trait_renum = ECS_FALSE ;
    ptr_zone->num_elt_deb = INT_MAX ;
    ptr_zone->num_elt_fin = INT_MIN ;
    ptr_zone->renum = NULL ;


    /* Informations principales sur la zone */

    if (   cg_zone_read(num_fic, num_base, num_zone, ptr_zone->nom,
                        ptr_zone->taille) != CG_OK
        || cg_zone_type(num_fic, num_base, num_zone,
                        &(ptr_zone->type)) != CG_OK)
      bft_error(__FILE__, __LINE__, 0,
                _("CGNS error:\n%s"), cg_get_error()) ;

    /* Informations sur le type de coordonnes */

    coo_type = 0 ;

    ret = cg_coord_info(num_fic, num_base, num_zone, 1, &type_coord_lu,
                        nom_tmp) ;

    if (ret == CG_OK) {

      if (strcmp("CoordinateR", nom_tmp) == 0)
        coo_type = 1 ;

      if (base_maillage->dim_espace == 3) {
        ret = cg_coord_info(num_fic, num_base, num_zone, 3, &type_coord_lu,
                            nom_tmp) ;
        if (   ret == CG_OK
            && (coo_type > 0 && strcmp("CoordinatePhi", nom_tmp) == 0))
          coo_type = 2 ;
      }

    }
    ret = cg_goto(num_fic, num_base, "Zone_t", num_zone,
                  "GridCoordinates_t", 1, "end") ;

    if (ret != CG_OK)
      bft_error(__FILE__, __LINE__, 0,
                _("CGNS error:\n%s"), cg_get_error()) ;

    /* Type d'unit (information optionnelle) */

    length = LengthUnitsNull ;
    ret = cg_goto(num_fic, num_base, "Zone_t", num_zone,
                  "GridCoordinates_t", 1, "end") ;
    if (ret == CG_OK)
      ret = cg_units_read(&mass, &length, &time, &temperature, &angle) ;

    if (ret == CG_OK)
      ptr_zone->angle = angle ;

    /* Affichage et stockage des dimensions */

    nbr_som_loc = 0 ;
    nbr_cel_loc = 0 ;

    bft_printf(_("\n    Zone %d: \"%s\"; type \"%s\"\n"),
               num_zone, ptr_zone->nom, ZoneTypeName[ptr_zone->type]) ;

    if (ptr_zone->type == Structured && base_maillage->dim_entite > 1) {
      if (base_maillage->dim_entite == 2) {
        nbr_som_loc = ptr_zone->taille[0] * ptr_zone->taille[1] ;
        nbr_cel_loc = ptr_zone->taille[2] * ptr_zone->taille[3] ;

        bft_printf(_("\n      %d (%d x %d) vertices; %d (%d x %d) cells\n"),
                   nbr_som_loc, ptr_zone->taille[0], ptr_zone->taille[1],
                   nbr_cel_loc, ptr_zone->taille[2], ptr_zone->taille[3]) ;
      }
      else if (base_maillage->dim_entite == 3) {
        nbr_som_loc =   ptr_zone->taille[0] * ptr_zone->taille[1]
                      * ptr_zone->taille[2] ;
        nbr_cel_loc =   ptr_zone->taille[3] * ptr_zone->taille[4]
                      * ptr_zone->taille[5];

        bft_printf(_("\n      %d (%d x %d x %d) vertices;"
                     " %d (%d x %d x %d) cells\n"),
                   nbr_som_loc, ptr_zone->taille[0], ptr_zone->taille[1],
                   ptr_zone->taille[2], nbr_cel_loc, ptr_zone->taille[3],
                   ptr_zone->taille[4], ptr_zone->taille[5]) ;
      }
    }
    else {
      nbr_som_loc = ptr_zone->taille[0] ;
      nbr_cel_loc = ptr_zone->taille[1] ;

      bft_printf(_("\n      %d vertices; %d cells\n"),
                 nbr_som_loc, nbr_cel_loc) ;
    }

    ptr_zone->nbr_som = nbr_som_loc ;
    ptr_zone->nbr_cel = nbr_cel_loc ;

    ptr_zone->num_som_deb  = nbr_som_tot + 1 ;
    nbr_som_tot           += nbr_som_loc ;

    /* Informations complmentaires (non indispensables) */

    bft_printf(_("      (%s coordinates, \"%s\" precision, unit \"%s\")\n"),
               nom_coo_type[coo_type],
               DataTypeName[type_coord_lu], LengthUnitsName[length]) ;

    if (cg_ngrids(num_fic, num_base, num_zone, &ngrids) == CG_OK) {
      if (ngrids > 1)
        bft_printf(_("      %d time-varying coordinates "
                     "(the first are used)\n"), ngrids) ;
    }

    /* Informations sur les connectivits multizones */

    ret = cg_n1to1(num_fic, num_base, num_zone, &nconns) ;

    if (ret == CG_OK) {
      if (nconns > 0)
        connect_1to1 = ECS_TRUE ;
    }

    ret = cg_nconns(num_fic, num_base, num_zone, &nconns) ;

    if (ret == CG_OK) {

      char nom_tmp_aux [ECS_CGNS_TAILLE_NOM + 1] ;

      GridLocation_t         location ;
      GridConnectivityType_t connect_type ;
      PointSetType_t         ptset_type ;
      int                    npnts ;
      ZoneType_t             donor_zonetype ;
      PointSetType_t         donor_ptset_type ;
      DataType_t             donor_datatype ;
      int                    ndata_donor ;

      for (ind_conn = 0 ; ind_conn < nconns ; ind_conn++) {

        ret = cg_conn_info(num_fic, num_base, num_zone, ind_conn + 1,
                           nom_tmp, &location, &connect_type, &ptset_type,
                           &npnts, nom_tmp_aux, &donor_zonetype,
                           &donor_ptset_type, &donor_datatype,
                           &ndata_donor) ;

        if (ret == CG_OK) {

          switch (connect_type) {
          case Overset:
            connect_overset = ECS_TRUE ;
            break ;
          case Abutting:
            connect_abutt = ECS_TRUE ;
            break ;
          case Abutting1to1:
            connect_1to1 = ECS_TRUE ;
            break ;
          default:
            break ;
          }

        }

      }

    }

    /* Informations sur le nombre de sections (non structur) */

    ptr_zone->nbr_sections = 0 ;
    ptr_zone->tab_sections = NULL ;

    if (ptr_zone->type == Structured) {

      ptr_zone->num_elt_deb = 1 ;
      ptr_zone->num_elt_fin = ptr_zone->nbr_cel ;

    }
    else {

      ret = cg_nsections(num_fic, num_base, num_zone,
                         &(ptr_zone->nbr_sections)) ;
      if (ret == CG_OK) {

        BFT_MALLOC(ptr_zone->tab_sections, ptr_zone->nbr_sections,
                   ecs_loc_cgns_section_t) ;

        for (ind_section = 0 ;
             ind_section < ptr_zone->nbr_sections ;
             ind_section++) {

          num_section = ind_section + 1 ;
          ptr_section = ptr_zone->tab_sections + ind_section ;

          if (cg_section_read(num_fic,
                              num_base,
                              num_zone,
                              num_section,
                              ptr_section->nom,
                              &(ptr_section->type),
                              &(ptr_section->num_elt_deb),
                              &(ptr_section->num_elt_fin),
                              &(ptr_section->nbr_brd),
                              &(ptr_section->parent)) != CG_OK)
            bft_error(__FILE__, __LINE__, 0,
                      _("CGNS error:\n%s"), cg_get_error()) ;

          ptr_section->elems = NULL ;

          bft_printf(_("      Section %2d: \"%s\";\n"
                       "                   (indices %d to %d, type \"%s\")\n"),
                     num_section, ptr_section->nom,
                     ptr_section->num_elt_deb, ptr_section->num_elt_fin,
                     ElementTypeName[ptr_section->type]) ;

          ptr_zone->num_elt_deb = ECS_MIN(ptr_section->num_elt_deb,
                                          ptr_zone->num_elt_deb) ;
          ptr_zone->num_elt_fin = ECS_MAX(ptr_section->num_elt_fin,
                                          ptr_zone->num_elt_fin) ;

        }

      }

    }

  }

  bft_printf("\n") ;


  /* Informations sur les connectivits */

  if (connect_1to1 == ECS_TRUE) {
    ecs_warn() ;
    bft_printf(_("The CGNS mesh read contains multizone (\"one to one\")\n"
                 "vertex equivalences which are not automatically handled\n"
                 "by the Preprocessor.\n"
                 "-> Use an appropriate joining option\n")) ;
  }

  if (connect_abutt == ECS_TRUE) {
    ecs_warn() ;
    bft_printf(_("The CGNS mesh read contains non-conforming (\"abutting\")\n"
                 "connectivity information which is not automatically handled\n"
                 "by the Preprocessor.\n"
                 "-> Use an appropriate joining option\n")) ;
  }

  if (connect_overset == ECS_TRUE) {
    ecs_warn() ;
    bft_printf(_("The CGNS mesh read contains (\"overset\") connectivity\n"
                 "information which is not handled by the Preprocessor.\n")) ;
  }


  /* Renvoi du tableau d'information sur les zones */

  return tab_zone ;

}


/*----------------------------------------------------------------------------
 *                    Lecture des conditions aux limites
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__lit_boco
(
 const ecs_loc_cgns_base_t      *const base_maillage ,
 const int                             nzones        ,
       ecs_loc_cgns_zone_t      *const tab_zone      ,
       int                      *const nbr_nom_boco  ,
       int                      *const nbr_boco_tot  ,
       char                 * * *const tab_nom_boco  ,
       ecs_loc_cgns_boco_t    * *const tab_boco
)
{

  char         nom_tmp [ECS_CGNS_TAILLE_NOM + 1] ;
  char         nom_fam [ECS_CGNS_TAILLE_NOM + 1] ;

  ecs_int_t     *nbr_boco_loc ;

  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  ecs_loc_cgns_zone_t     *ptr_zone ;

  ecs_loc_cgns_boco_t     *tab_boco_loc ;
  char                   **nom_boco_loc ;

  /* Dclarations des variables pour CGNS */
  /*-------------------------------------*/

  int         ind_boco ;
  int         ind_boco_glob ;
  int         ind_nom ;
  int         ind_zone ;
  int         num_boco ;
  int         num_fic ;
  int         num_base ;
  int         num_zone ;

  int         nbocos ;
  int         npnts ;

  BCType_t        bocotype ;
  GridLocation_t  GridLocation ;
  PointSetType_t  ptset_type ;
  int             NormalIndex[3] ;
  int             NormalListFlag ;
  DataType_t      NormalDataType ;
  int             ndataset ;

  void        *normales ;

  int         ret = CG_OK ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  num_fic  = base_maillage->num_fic ;
  num_base = base_maillage->num_base ;

  BFT_MALLOC(nbr_boco_loc, nzones, int) ;

  *nbr_nom_boco = 0 ;
  *tab_nom_boco = NULL ;

  *tab_boco = NULL ;

  /* Comptage */

  *nbr_boco_tot = 0 ;

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    ret = cg_nbocos(num_fic, num_base, num_zone, &nbocos) ;

    if (ret == CG_OK) {
      nbr_boco_loc[ind_zone] = nbocos ;
      *nbr_boco_tot += nbocos ;
    }
    else
      bft_error(__FILE__, __LINE__, 0,
                _("CGNS error:\n%s"), cg_get_error()) ;

  }

  if (*nbr_boco_tot > 0) {

    bft_printf(_("  CGNS boundary condition information:\n")) ;

    BFT_MALLOC(tab_boco_loc, *nbr_boco_tot, ecs_loc_cgns_boco_t) ;
    BFT_MALLOC(nom_boco_loc, *nbr_boco_tot, char *) ;

    for (ind_nom = 0 ; ind_nom < *nbr_boco_tot ; ind_nom++)
      nom_boco_loc[ind_nom] = NULL ;

    *tab_boco = tab_boco_loc ;

  }


  /*------------------------------------*/
  /* Lecture des conditions aux limites */
  /*------------------------------------*/

  ind_boco_glob = 0 ;

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    if (nbr_boco_loc[ind_zone] > 0) {

      bft_printf(_("\n    Zone %d\n"), num_zone) ;

      for (ind_boco = 0 ; ind_boco < nbr_boco_loc[ind_zone] ; ind_boco++) {

        num_boco = ind_boco + 1 ;

        /* Informations principales */

        ret = cg_boco_info(num_fic, num_base, num_zone, num_boco, nom_tmp,
                           &bocotype, &ptset_type, &npnts, NormalIndex,
                           &NormalListFlag, &NormalDataType, &ndataset) ;

        if (ret != CG_OK)
          break ;

        nom_tmp [ECS_CGNS_TAILLE_NOM] = '\0' ;

        /* Informations sur support et impression */

        GridLocation = Vertex ;

        ret = cg_goto(num_fic, num_base, "Zone_t", num_zone,
                      "ZoneBC_t", 1, "BC_t", num_boco, "end") ;

        if (ret == CG_OK) {

          ret = cg_famname_read(nom_fam) ;

          if (ret == CG_OK) {
            nom_fam[ECS_CGNS_TAILLE_NOM] = '\0' ;
            strcpy(nom_tmp, nom_fam) ;
          }

          if (ptr_zone->type != Structured) {

            ret = cg_gridlocation_read(&GridLocation) ;

            if (ret != CG_OK)
              GridLocation = Vertex ;

          }

        }

        if (ptr_zone->type == Structured) {

          bft_printf(_("      BC %2d: \"%s\" (\"%s\")\n"),
                     num_boco, nom_tmp, BCTypeName[bocotype]) ;

        }
        else {

          if (ptset_type == ElementList || ptset_type == ElementRange)
            GridLocation = FaceCenter ;

          bft_printf(_("      BC %2d: \"%s\" (\"%s\" on \"%s\")\n"),
                     num_boco, nom_tmp, BCTypeName[bocotype],
                     GridLocationName[GridLocation]) ;

          if (GridLocation != Vertex)
            ptr_zone->trait_renum = ECS_TRUE ;

        }

        /* stockage */

        for (ind_nom = 0 ; nom_boco_loc[ind_nom] != NULL ; ind_nom++) {
          if (strcmp(nom_boco_loc[ind_nom], nom_tmp) == 0)
            break ;
        }
        if (nom_boco_loc[ind_nom] == NULL) {
          BFT_MALLOC(nom_boco_loc[ind_nom], strlen(nom_tmp) + 1, char) ;
          strcpy(nom_boco_loc[ind_nom], nom_tmp) ;
        }

        (tab_boco_loc[ind_boco_glob]).ind_nom     = ind_nom ;
        (tab_boco_loc[ind_boco_glob]).num_zone    = num_zone ;
        (tab_boco_loc[ind_boco_glob]).support     = GridLocation ;
        (tab_boco_loc[ind_boco_glob]).ptset_type  = ptset_type ;
        (tab_boco_loc[ind_boco_glob]).npnts       = npnts ;

        /* Lecture listes */

        if (ptset_type == PointRange || ptset_type == ElementRange)
          BFT_MALLOC((tab_boco_loc[ind_boco_glob]).pnts, npnts * 3, int) ;
        else
          BFT_MALLOC((tab_boco_loc[ind_boco_glob]).pnts, npnts, int) ;

        if (NormalListFlag > 0) {
          if (NormalDataType == RealSingle)
            BFT_MALLOC(normales, NormalListFlag, float) ;
          else
            BFT_MALLOC(normales, NormalListFlag, double) ;
        }
        else
          normales = NULL ;

        ret = cg_boco_read(num_fic, num_base, num_zone, num_boco,
                           (tab_boco_loc[ind_boco_glob]).pnts,
                           normales) ;

        if (ret != CG_OK)
          break ;

        if (normales != NULL)
          BFT_FREE(normales) ;

        /* Incrmentation compteur */

        ind_boco_glob += 1 ;

      }

      if (ret != CG_OK)
        bft_error(__FILE__, __LINE__, 0,
                  _("CGNS error:\n%s"), cg_get_error()) ;

    }

  }

  if (*nbr_boco_tot > 0) {

    for (ind_nom = 0 ;
         ind_nom < *nbr_boco_tot && nom_boco_loc[ind_nom] != NULL ;
         ind_nom++) ;

    BFT_REALLOC(nom_boco_loc, ind_nom, char *) ;

    *nbr_nom_boco = ind_nom ;
    *tab_nom_boco = nom_boco_loc ;

  }

  if (*nbr_boco_tot > 0) {

    bft_printf("\n") ;

  }

  BFT_FREE(nbr_boco_loc) ;

}


/*----------------------------------------------------------------------------
 *                         Lecture des sommets
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * ecs_loc_pre_cgns__lit_som
(
 const ecs_loc_cgns_base_t  *const base_maillage ,
 const int                         nzones        ,
       ecs_loc_cgns_zone_t  *const tab_zone
)
{

  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  ecs_int_t     coo_type ;
  ecs_int_t     ind_som ;
  ecs_int_t     nbr_som ;
  ecs_real_t  * som_val_tmp ;
  ecs_real_t  * som_val_coord ;                   /* Coordonnes des sommets  */

  ecs_real_t    cnv_angle ;
  ecs_real_t    r_tmp ;
  ecs_real_t    t_tmp ;
  ecs_real_t    p_tmp ;

  ecs_entmail_t  * entmail_som ;

  ecs_loc_cgns_zone_t  *ptr_zone ;


  /* Dclarations des variables pour CGNS */
  /*-------------------------------------*/

  int         ind_zone ;
  int         num_fic ;
  int         num_base ;
  int         num_zone ;
  int         phys_dim ;
  int         irmin[3] ;
  int         irmax[3] ;

  int        *ind_som_deb ;

  DataType_t  type_coord ;

  int         ret = 0 ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  num_fic  = base_maillage->num_fic ;
  num_base = base_maillage->num_base ;

  phys_dim = base_maillage->dim_espace ;

  if (sizeof(ecs_real_t) == sizeof(float))
    type_coord = RealSingle ;
  else if (sizeof(ecs_real_t) == sizeof(double))
    type_coord = RealDouble ;
  else
    assert (   (sizeof(ecs_real_t) == sizeof(float))
            || (sizeof(ecs_real_t) == sizeof(double))) ;

  BFT_MALLOC(ind_som_deb, nzones + 1, int) ;

  ind_som_deb[0] = 0 ;


  /* Dimensionnement pour lectures */
  /*-------------------------------*/

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    ind_som_deb[ind_zone + 1] = ind_som_deb[ind_zone] + ptr_zone->nbr_som ;

  }

  BFT_MALLOC(som_val_tmp,   ind_som_deb[nzones] * phys_dim,  ecs_real_t) ;
  BFT_MALLOC(som_val_coord, ind_som_deb[nzones] * phys_dim, ecs_real_t) ;

  nbr_som = ind_som_deb[nzones] ;


  /* Lecture effective des coordonnes et lments*/
  /*----------------------------------------------*/

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    /*
      Lecture des coordonnes (CGNS convertit automatiquement du type
      simple prcision ou double prcision au type demand si ncessaire).
    */

    /* Cas structur */

    if (ptr_zone->type == Structured) {
      irmin[0] = 1 ;
      irmin[1] = 1 ;
      irmax[0] = ptr_zone->taille[0] ;
      irmax[1] = ptr_zone->taille[1] ;
      if (phys_dim == 3) {
        irmin[2] = 1 ;
        irmax[2] = ptr_zone->taille[2] ;
      }
    }

    /* Cas non structur */

    else if (ptr_zone->type == Unstructured) {
      irmin[0] = 1 ;
      irmin[1] = 1 ;
      irmax[0] = ptr_zone->taille[0] ;
      irmax[1] = 1 ;
      irmin[2] = 1 ;
      irmax[2] = 1 ;
    }

    /* Premire coordonne (X ou R) */

    coo_type = 0 ;

    ret = cg_coord_read(num_fic, num_base, num_zone, "CoordinateX",
                        type_coord, irmin, irmax,
                        som_val_tmp + ind_som_deb[ind_zone]) ;
    if (ret != 0) {
      ret = cg_coord_read(num_fic, num_base, num_zone, "CoordinateR",
                          type_coord, irmin, irmax,
                          som_val_tmp + ind_som_deb[ind_zone] + nbr_som) ;
      if (ret == 0)
        coo_type = 1 ;
    }

    /* Seconde coordonne (Y ou Theta) */

    if (ret == 0 && coo_type == 0)
      ret = cg_coord_read(num_fic, num_base, num_zone, "CoordinateY",
                          type_coord, irmin, irmax,
                          som_val_tmp + ind_som_deb[ind_zone] + nbr_som) ;
    else if (ret == 0 && coo_type == 1)
        ret = cg_coord_read(num_fic, num_base, num_zone, "CoordinateTheta",
                          type_coord, irmin, irmax,
                          som_val_tmp + ind_som_deb[ind_zone] + nbr_som) ;

    /* Troisime coordonne (Z ou Phi) */

    if (ret == 0 && phys_dim == 3) {
      ret = cg_coord_read(num_fic, num_base, num_zone, "CoordinateZ",
                          type_coord, irmin, irmax,
                          som_val_tmp + ind_som_deb[ind_zone] + (nbr_som*2)) ;
      if (ret != 0 && coo_type == 1) {
        ret = cg_coord_read(num_fic, num_base, num_zone, "CoordinatePhi",
                          type_coord, irmin, irmax,
                          som_val_tmp + ind_som_deb[ind_zone] + nbr_som) ;
        if (ret == 0)
          coo_type = 2 ;
      }
    }

    if (ret != 0)
      bft_error(__FILE__, __LINE__, 0,
                _("CGNS error:\n%s"), cg_get_error()) ;

    /* Conversion en coordonnes Cartsiennes si ncessaire */

    if (coo_type > 0) {
      if (ptr_zone->angle == Degree)
        cnv_angle = 4 * atan(1) / 180.0 ;
      else if (ptr_zone->angle == Radian)
        cnv_angle = 1.0 ;
      else {
        cnv_angle = 1.0 ;
        ecs_warn() ;
        bft_printf(_("Cylindrical or spherical coordinates with unknown\n"
                     "or undefined angle unit "
                     "(-> radians are considered)\n")) ;
      }
    }

    if (coo_type == 1 && phys_dim > 1) {
      for (ind_som = 0 ; ind_som < nbr_som ; ind_som++) {
        r_tmp = som_val_tmp[ind_som           ] ;
        t_tmp = som_val_tmp[ind_som + nbr_som ] * cnv_angle ;
        som_val_tmp[ind_som          ] = r_tmp * cos(t_tmp) ;
        som_val_tmp[ind_som + nbr_som] = r_tmp * sin(t_tmp);
      }
    }
    else if (coo_type == 2 && phys_dim > 2) {
      for (ind_som = 0 ; ind_som < nbr_som ; ind_som++) {
        r_tmp = som_val_tmp[ind_som                ] ;
        t_tmp = som_val_tmp[ind_som +      nbr_som ] * cnv_angle ;
        p_tmp = som_val_tmp[ind_som + (2 * nbr_som)] * cnv_angle ;
        som_val_tmp[ind_som                ] = r_tmp * sin(t_tmp) * cos(p_tmp) ;
        som_val_tmp[ind_som +      nbr_som ] = r_tmp * sin(t_tmp) * sin(p_tmp) ;
        som_val_tmp[ind_som + (2 * nbr_som)] = r_tmp * cos(t_tmp);
      }
    }

  }

  /* Conversion au format local */

  if (phys_dim == 1) {
    for (ind_som = 0 ; ind_som < nbr_som ; ind_som++) {
      som_val_coord[ind_som] = som_val_tmp[ind_som] ;
    }
  }
  else if (phys_dim == 2) {
    for (ind_som = 0 ; ind_som < nbr_som ; ind_som++) {
      som_val_coord[ind_som * 2    ] = som_val_tmp[ind_som] ;
      som_val_coord[ind_som * 2 + 1] = som_val_tmp[ind_som + nbr_som ] ;
    }
  }
  else if (phys_dim == 3) {
    for (ind_som = 0 ; ind_som < nbr_som ; ind_som++) {
      som_val_coord[ind_som * 3    ] = som_val_tmp[ind_som                ] ;
      som_val_coord[ind_som * 3 + 1] = som_val_tmp[ind_som +      nbr_som ] ;
      som_val_coord[ind_som * 3 + 2] = som_val_tmp[ind_som + (2 * nbr_som)] ;
    }
  }
  else {
    BFT_FREE(som_val_coord) ;
  }

  BFT_FREE(som_val_tmp) ;


  /* Transfert des valeurs lues dans la structure d'entit de maillage */
  /*===================================================================*/

  entmail_som = ecs_entmail_pre__cree_som(ECS_DIM_3,
                                          nbr_som,
                                          som_val_coord,
                                          NULL) ;


  /* Libration des tableaux de travail */

  BFT_FREE(som_val_tmp) ;

  BFT_FREE(ind_som_deb) ;


  /* Renvoi de la structure associe aux sommets */

  return entmail_som ;

}


/*----------------------------------------------------------------------------
 *                        Lecture des lments
 *----------------------------------------------------------------------------*/

static ecs_entmail_t * * ecs_loc_pre_cgns__lit_ele
(
 const ecs_loc_cgns_base_t      *const base_maillage      ,
 const int                             nzones             ,
       ecs_loc_cgns_zone_t      *const tab_zone           ,
 const int                             nbr_boco_tot       ,
       ecs_loc_cgns_boco_t      *const tab_boco           ,
       ecs_int_t                *const nbr_boco_som       ,
       ecs_int_t              * *const ind_nom_boco_som   ,
       ecs_int_t              * *const nbr_sselt_boco_som ,
       ecs_int_t            * * *const tab_sselt_boco_som ,
       ecs_int_t              * *const ind_section_cel    ,
       ecs_int_t              * *const ind_section_fac    ,
       ecs_int_t              * *const ind_zone_cel       ,
       ecs_int_t              * *const ind_zone_fac
)
{

  ecs_int_t    ient ;
  ecs_int_t    ient_max ;

  ecs_int_t    ind_type ;
  ecs_int_t    ind_pos ;
  ecs_int_t    ind_val ;
  ecs_int_t    ind_som ;
  ecs_int_t    cpt_elt_loc ;
  ecs_int_t    cpt_elt_zone ;
  ecs_int_t    cpt_section ;
  ecs_int_t    nbr_elt_loc ;
  ecs_int_t    nbr_elt_zone ;
  ecs_int_t    nbr_som_elt ;

  ecs_int_t    ii, jj, kk ;
  ecs_int_t    ni, nj, nk ;

  ecs_int_t    num_som_deb ;
  ecs_int_t    num_som_loc ;

  ecs_elt_typ_t  ecs_typ ;

  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  size_t       cpt_elt_ent[ECS_ENTMAIL_FIN] ;
  size_t       cpt_val_ent[ECS_ENTMAIL_FIN] ;
  ecs_int_t    cpt_coul_ent[ECS_ENTMAIL_FIN] ;

  ecs_size_t  * elt_pos_som_ent    [ECS_ENTMAIL_FIN] ; /* Positions sommets    */
  ecs_int_t   * elt_val_som_ent    [ECS_ENTMAIL_FIN] ; /* Numros des sommets  */
  ecs_int_t   * elt_val_couleur_ent[ECS_ENTMAIL_FIN] ; /* Couleurs lments    */
  ecs_int_t   * val_coul_ent       [ECS_ENTMAIL_FIN] ;
  ecs_size_t  * cpt_elt_coul_ent   [ECS_ENTMAIL_FIN] ;
  ecs_int_t   * ind_zone_ent       [ECS_ENTMAIL_FIN] ;
  ecs_int_t   * ind_section_ent    [ECS_ENTMAIL_FIN] ;

  ecs_entmail_t ** vect_entmail ;

  ecs_loc_cgns_section_t  *ptr_section ;
  ecs_loc_cgns_zone_t     *ptr_zone ;


  /* Dclarations des variables pour CGNS */
  /*-------------------------------------*/

  int         cel_dim ;
  int         ind_zone ;
  int         ind_section ;
  int         num_fic ;
  int         num_base ;
  int         num_section ;
  int         num_zone ;
  int         taille_loc ;

  int        *ptr_ele ;

  int        *parentdata ;

  int         ret = 0 ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  num_fic  = base_maillage->num_fic ;
  num_base = base_maillage->num_base ;

  cel_dim = base_maillage->dim_entite ;

  if (cel_dim == 1)
    ient_max = ECS_ENTMAIL_ARE ;
  else if (cel_dim == 2)
    ient_max = ECS_ENTMAIL_FAC ;
  else
    ient_max = ECS_ENTMAIL_CEL ;

  /* Allocations des tableaux locaux */

  /* Attention au dcalage de `1' !!!         */
  /* On n'allouera pas les tableaux locaux pour */
  /* `ECS_ENTMAIL_DEB = ECS_ENTMAIL_SOM'      */

  for (ient = ECS_ENTMAIL_SOM ; ient < ECS_ENTMAIL_FIN ; ient++) {

    cpt_elt_ent        [ient] = 0    ;
    cpt_val_ent        [ient] = 0    ;
    cpt_coul_ent       [ient] = 0    ;

    elt_pos_som_ent    [ient] = NULL ;
    elt_val_som_ent    [ient] = NULL ;
    elt_val_couleur_ent[ient] = NULL ;

    val_coul_ent       [ient] = NULL ;
    cpt_elt_coul_ent   [ient] = NULL ;

    ind_zone_ent       [ient] = NULL ;
    ind_section_ent    [ient] = NULL ;

  }


  /*---------------------------*/
  /* Lecture des connectivits */
  /*---------------------------*/

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    for (ind_section = 0 ;
         ind_section < ptr_zone->nbr_sections ;
         ind_section++) {

      num_section = ind_section + 1 ;
      ptr_section = ptr_zone->tab_sections + ind_section ;

      ret = cg_ElementDataSize(num_fic, num_base, num_zone, num_section,
                               &taille_loc) ;

      /*
        Si la section est de type "MIXED", taille_loc correspond
        aux tailles "pos" + "val" ; sinon, taille_loc correspond
        aux tailles "val" uniquement. On ne sait pas si ces
        donnes correspondent aux lments de dimension maximale,
        et on les lit donc immdiatement.
      */


      if (ret == CG_OK) {

        BFT_MALLOC(ptr_section->elems, taille_loc, int) ;

        if (ptr_section->parent > 0)
          BFT_MALLOC(parentdata,
                     (  ptr_section->num_elt_fin
                      - ptr_section->num_elt_deb + 1) * 4,
                     int) ;
        else
          parentdata = NULL ;

        ret = cg_elements_read(num_fic, num_base, num_zone, num_section,
                               ptr_section->elems, parentdata) ;

        if (ret != CG_OK)
          bft_error(__FILE__, __LINE__, 0,
                    _("CGNS error:\n%s"), cg_get_error()) ;

        BFT_FREE(parentdata) ;

      }

    }

  }


  /*--------------------------------------------------------*/
  /* Dimensionnement de la connectivit nodale des lments */
  /*--------------------------------------------------------*/

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    nbr_elt_zone = 0 ;


    /* Cas d'une zone structure */

    if (ptr_zone->type == Structured) {

      nbr_elt_loc   = ptr_zone->nbr_cel ;
      nbr_elt_zone += nbr_elt_loc ;

      if (cel_dim == 1) {
        cpt_elt_ent[ECS_ENTMAIL_ARE] += nbr_elt_loc;
        cpt_val_ent[ECS_ENTMAIL_ARE] += nbr_elt_loc * 2 ;
      }
      else if (cel_dim == 2) {
        cpt_elt_ent[ECS_ENTMAIL_FAC] += nbr_elt_loc;
        cpt_val_ent[ECS_ENTMAIL_FAC] += nbr_elt_loc * 4 ;
      }
      else if (cel_dim == 3) {
        cpt_elt_ent[ECS_ENTMAIL_CEL] += nbr_elt_loc;
        cpt_val_ent[ECS_ENTMAIL_CEL] += nbr_elt_loc * 8 ;
      }

    }

    /* Cas d'une zone non structure */

    else if (ptr_zone->type == Unstructured) {

      for (ind_section = 0 ;
           ind_section < ptr_zone->nbr_sections ;
           ind_section++) {

        num_section = ind_section + 1 ;
        ptr_section = ptr_zone->tab_sections + ind_section ;

        nbr_elt_loc   = ptr_section->num_elt_fin - ptr_section->num_elt_deb + 1 ;
        nbr_elt_zone += nbr_elt_loc ;

        if (ptr_section->type != MIXED) {

          if (ptr_section->type < NGON_n) {
            ecs_typ     = ecs_cgns_elt_liste_c[  ptr_section->type
                                               - NODE].ecs_type ;
            nbr_som_elt = ecs_fic_elt_typ_liste_c[ecs_typ].nbr_som ;
          }
          else {
            ecs_typ     = ECS_ELT_TYP_FAC_POLY ;
            nbr_som_elt = ptr_section->type - NGON_n ;
          }

          ient = ecs_entmail_pre__ret_typ_geo(ecs_typ) ;

          cpt_elt_ent[ient] += nbr_elt_loc ;
          cpt_val_ent[ient] += nbr_elt_loc * nbr_som_elt ;

        }
        else { /* if (ptr_section->type == MIXED) */

          cpt_elt_loc = 0 ;
          ptr_ele = ptr_section->elems ;

          while (cpt_elt_loc < nbr_elt_loc) {

            ind_type    = *ptr_ele - NODE ;
            if (ind_type < NGON_n - NODE) {
              ecs_typ     = ecs_cgns_elt_liste_c[ind_type].ecs_type ;
              nbr_som_elt = ecs_fic_elt_typ_liste_c[ecs_typ].nbr_som ;
              ptr_ele += ecs_cgns_elt_liste_c[*ptr_ele - NODE].nbr_som + 1 ;
            }
            else {
              ecs_typ     = ECS_ELT_TYP_FAC_POLY ;
              nbr_som_elt = *ptr_ele - NGON_n ;
              ptr_ele += nbr_som_elt + 1 ;
            }

            cpt_elt_loc++ ;

            /*
              Calcul entit correspondante sans passer par
              ecs_entmail_pre__ret_typ_geo(ecs_typ) pour viter
              appel de fonction  chaque lment.
            */

            if (ecs_typ == NODE)
              ient = ECS_ENTMAIL_SOM ;
            else if (ecs_typ == ECS_ELT_TYP_ARE)
              ient = ECS_ENTMAIL_ARE ;
            else if (ecs_typ < ECS_ELT_TYP_CEL_TETRA)
              ient = ECS_ENTMAIL_FAC ;
            else if (ecs_typ < ECS_ELT_TYP_FAC_POLY)
              ient = ECS_ENTMAIL_CEL ;
            else if (ecs_typ == ECS_ELT_TYP_FAC_POLY)
              ient = ECS_ENTMAIL_FAC ;
            else if (ecs_typ == ECS_ELT_TYP_CEL_POLY)
              ient = ECS_ENTMAIL_CEL ;
            else
              ient = ECS_ENTMAIL_FIN ;

            cpt_elt_ent[ient] += 1 ;
            cpt_val_ent[ient] += nbr_som_elt ;

          }

        }  /* fin traitement selon si section de type "MIXED" ou non) */

      } /* Fin boucle sur les sections */

    } /* Fin traitement structur/non structur */

    /* Ajout pour conditions aux limites */

    if (ptr_zone->trait_renum == ECS_TRUE) {
      BFT_MALLOC(ptr_zone->renum, nbr_elt_zone, ecs_int_t) ;
    }

  } /* Fin boucle sur les zones */


  /* Allocation mmoire et remise compteurs  zro */
  /*-----------------------------------------------*/

  for (ient = ECS_ENTMAIL_SOM ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (cpt_elt_ent[ient] > 0) {

      BFT_MALLOC(elt_pos_som_ent[ient], cpt_elt_ent[ient] + 1, ecs_size_t) ;
      BFT_MALLOC(elt_val_som_ent[ient], cpt_val_ent[ient]    , ecs_int_t) ;

      if (ient == ient_max - 1) {
        if (ind_zone_fac != NULL) {
          BFT_MALLOC(ind_zone_ent[ient], cpt_elt_ent[ient], ecs_int_t) ;
          *ind_zone_fac = ind_zone_ent[ient] ;
        }
        if (ind_section_fac != NULL) {
          BFT_MALLOC(ind_section_ent[ient], cpt_elt_ent[ient], ecs_int_t) ;
          *ind_section_fac = ind_section_ent[ient] ;
        }
      }
      else if (ient == ient_max) {
        if (ind_zone_cel != NULL) {
          BFT_MALLOC(ind_zone_ent[ient], cpt_elt_ent[ient], ecs_int_t) ;
          *ind_zone_cel = ind_zone_ent[ient] ;
        }
        if (ind_section_cel != NULL) {
          BFT_MALLOC(ind_section_ent[ient], cpt_elt_ent[ient], ecs_int_t) ;
          *ind_section_cel = ind_section_ent[ient] ;
        }
      }

      cpt_elt_ent [ient] = 0 ;
      cpt_val_ent [ient] = 0 ;

      elt_pos_som_ent[ient][0] = 1 ;

    }

  }


  /*-----------------------------------------------------*/
  /* Construction de la connectivit nodale des lments */
  /*-----------------------------------------------------*/

  cpt_section = 0 ;
  num_som_deb = 1 ;

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

    num_zone = ind_zone + 1 ;
    ptr_zone = tab_zone + ind_zone ;

    cpt_elt_zone = 0 ;

    /* Cas d'une zone structure */

    if (ptr_zone->type == Structured) {

      nbr_elt_loc = ptr_zone->nbr_cel ;
      cpt_elt_loc = 0 ;

      if (cel_dim == 1) {

        ient = ECS_ENTMAIL_ARE ;
        ecs_typ = ECS_ELT_TYP_ARE ;

        nbr_som_elt = 2 ;

        ni = ptr_zone->taille[0] ;

        for (ii = 0 ; ii < ni - 1 ; ii++) {

          ind_pos = cpt_elt_ent[ient] ;
          ind_val = elt_pos_som_ent[ient][ind_pos] - 1 ;
          elt_pos_som_ent[ient][ind_pos + 1]
            =  elt_pos_som_ent[ient][ind_pos] + nbr_som_elt ;

          num_som_loc = num_som_deb + ii ;

          elt_val_som_ent[ient][ind_val++] = num_som_loc ;
          elt_val_som_ent[ient][ind_val++] = num_som_loc + 1 ;

          if (ind_zone_ent[ient] != NULL)
            ind_zone_ent[ient][cpt_elt_ent[ient]] = ind_zone ;

          cpt_elt_ent[ient]++ ;

        }

      }
      else if (cel_dim == 2) {

        ient = ECS_ENTMAIL_FAC ;
        ecs_typ = ECS_ELT_TYP_FAC_QUAD ;

        nbr_som_elt = 4 ;

        ni = ptr_zone->taille[0] ;
        nj = ptr_zone->taille[1] ;

        for (jj = 0 ; jj < nj - 1 ; jj++) {
          for (ii = 0 ; ii < ni - 1 ; ii++) {

            ind_pos = cpt_elt_ent[ient] ;
            ind_val = elt_pos_som_ent[ient][ind_pos] - 1 ;
            elt_pos_som_ent[ient][ind_pos + 1]
              =  elt_pos_som_ent[ient][ind_pos] + nbr_som_elt ;

            num_som_loc = num_som_deb + ii + jj*ni ;

            elt_val_som_ent[ient][ind_val++] = num_som_loc ;
            elt_val_som_ent[ient][ind_val++] = num_som_loc + 1 ;
            elt_val_som_ent[ient][ind_val++] = num_som_loc + 1 + ni ;
            elt_val_som_ent[ient][ind_val++] = num_som_loc + ni ;

            if (ind_zone_ent[ient] != NULL)
              ind_zone_ent[ient][cpt_elt_ent[ient]] = ind_zone ;

            cpt_elt_ent[ient]++ ;

          }
        }

      }
      else if (cel_dim == 3) {

        ient = ECS_ENTMAIL_CEL ;
        ecs_typ = ECS_ELT_TYP_CEL_HEXA ;
        nbr_som_elt = 8 ;

        ni = ptr_zone->taille[0] ;
        nj = ptr_zone->taille[1] ;
        nk = ptr_zone->taille[2] ;

        for (kk = 0 ; kk < nk - 1 ; kk++) {
          for (jj = 0 ; jj < nj - 1 ; jj++) {
            for (ii = 0 ; ii < ni - 1 ; ii++) {

              ind_pos = cpt_elt_ent[ient] ;
              ind_val = elt_pos_som_ent[ient][ind_pos] - 1 ;
              elt_pos_som_ent[ient][ind_pos + 1]
                =  elt_pos_som_ent[ient][ind_pos] + nbr_som_elt ;

              num_som_loc = num_som_deb + ii + jj*ni + kk*ni*nj ;

              elt_val_som_ent[ient][ind_val++] = num_som_loc ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + 1 ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + 1 + ni ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + ni ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + ni*nj ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + ni*nj + 1 ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + ni*nj + 1 + ni ;
              elt_val_som_ent[ient][ind_val++] = num_som_loc + ni*nj + ni ;

              if (ind_zone_ent[ient] != NULL)
                ind_zone_ent[ient][cpt_elt_ent[ient]] = ind_zone ;

              cpt_elt_ent[ient]++ ;

            }
          }
        }

      }

    }

    /* Cas d'une zone non structure */

    else if (ptr_zone->type == Unstructured) {

      for (ind_section = 0 ;
           ind_section < ptr_zone->nbr_sections ;
           ind_section++) {

        num_section = ind_section + 1 ;
        ptr_section = ptr_zone->tab_sections + ind_section ;

        nbr_elt_loc = ptr_section->num_elt_fin - ptr_section->num_elt_deb + 1 ;
        cpt_elt_loc = 0 ;

        ptr_ele = ptr_section->elems ;

        if (ptr_section->type != MIXED) {

          ind_type    = ptr_section->type - NODE ;
          if (ind_type < NGON_n - NODE) {
            ecs_typ     = ecs_cgns_elt_liste_c[ind_type].ecs_type ;
            nbr_som_elt = ecs_fic_elt_typ_liste_c[ecs_typ].nbr_som ;
          }
          else {
            ecs_typ     = ECS_ELT_TYP_FAC_POLY ;
            nbr_som_elt = ptr_section->type - NGON_n ;
          }

          ient = ecs_entmail_pre__ret_typ_geo(ecs_typ) ;

        }

        while (cpt_elt_loc < nbr_elt_loc) {

          if (ptr_section->type == MIXED) {

            ind_type    = *ptr_ele - NODE ;
            if (ind_type < NGON_n - NODE) {
              ecs_typ     = ecs_cgns_elt_liste_c[ind_type].ecs_type ;
              nbr_som_elt = ecs_fic_elt_typ_liste_c[ecs_typ].nbr_som ;
            }
            else {
              ecs_typ     = ECS_ELT_TYP_FAC_POLY ;
              nbr_som_elt = *ptr_ele - NGON_n ;
            }

            ptr_ele += 1 ;

            /*
              Calcul entit correspondante sans passer par
              ecs_entmail_pre__ret_typ_geo(ecs_typ) pour viter
              appel de fonction  chaque lment.
            */

            if (ecs_typ == NODE)
              ient = ECS_ENTMAIL_SOM ;
            else if (ecs_typ == ECS_ELT_TYP_ARE)
              ient = ECS_ENTMAIL_ARE ;
            else if (ecs_typ < ECS_ELT_TYP_CEL_TETRA)
              ient = ECS_ENTMAIL_FAC ;
            else if (ecs_typ < ECS_ELT_TYP_FAC_POLY)
              ient = ECS_ENTMAIL_CEL ;
            else if (ecs_typ == ECS_ELT_TYP_FAC_POLY)
              ient = ECS_ENTMAIL_FAC ;
            else if (ecs_typ == ECS_ELT_TYP_CEL_POLY)
              ient = ECS_ENTMAIL_CEL ;
            else
              ient = ECS_ENTMAIL_FIN ;

          }

          ind_pos = cpt_elt_ent[ient] ;
          ind_val = elt_pos_som_ent[ient][ind_pos] - 1 ;
          elt_pos_som_ent[ient][ind_pos + 1]
            =  elt_pos_som_ent[ient][ind_pos] + nbr_som_elt ;

          for (ind_som = 0 ; ind_som < nbr_som_elt ; ind_som++) {
            if (ind_type < NGON_n - NODE)
              elt_val_som_ent[ient][ind_val++]
                = *(ptr_ele
                    + ecs_cgns_elt_liste_c[ind_type].num_som[ind_som] - 1)
                    + num_som_deb - 1 ;
            else
              elt_val_som_ent[ient][ind_val++]
                = *(ptr_ele + ind_som) + num_som_deb - 1 ;
          }

          if (ind_type < NGON_n - NODE)
            ptr_ele += ecs_cgns_elt_liste_c[ind_type].nbr_som ;
          else
            ptr_ele += nbr_som_elt ;

          if (ptr_zone->trait_renum == ECS_TRUE) {
            if (ient == ient_max - 1)
              ptr_zone->renum[cpt_elt_zone++] = cpt_elt_ent[ient] + 1 ;
            else if (ient == ient_max)
              ptr_zone->renum[cpt_elt_zone++] = -(cpt_elt_ent[ient] + 1) ;
            else
              ptr_zone->renum[cpt_elt_zone++] = 0 ;
          }

          if (ind_zone_ent[ient] != NULL)
            ind_zone_ent[ient][cpt_elt_ent[ient]] = ind_zone ;

          if (ind_section_ent[ient] != NULL)
            ind_section_ent[ient][cpt_elt_ent[ient]] = cpt_section ;

          cpt_elt_loc++ ;
          cpt_elt_ent[ient]++ ;

        }

        /* On libre la mmoire */

        BFT_FREE(ptr_section->elems) ;

        cpt_section += 1 ;

      } /* Fin boucle sur les sections */

    } /* Fin traitement structur/non structur */

    num_som_deb += ptr_zone->nbr_som ;

  } /* Fin boucle sur les zones */


  /* Suppression des entits inutilises */

  for (ient = ECS_ENTMAIL_SOM ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (cpt_elt_ent[ient] > 0 && ient < ient_max - 1) {

      BFT_FREE(elt_pos_som_ent[ient]) ;
      BFT_FREE(elt_val_som_ent[ient]) ;

      cpt_elt_ent [ient] = 0 ;
      cpt_val_ent [ient] = 0 ;

    }

  }


  /* Cration des entits supplmentaires pour porter les C.L. aux sommets */
  /*-----------------------------------------------------------------------*/

  ecs_loc_pre__cgns__cree_ent_inf_som(base_maillage,
                                      cpt_elt_ent,
                                      elt_pos_som_ent,
                                      elt_val_som_ent,
                                      nzones,
                                      tab_zone,
                                      nbr_boco_tot,
                                      tab_boco,
                                      nbr_boco_som,
                                      ind_nom_boco_som,
                                      nbr_sselt_boco_som,
                                      tab_sselt_boco_som) ;


  /* Transfert des valeurs lues dans les structures d'entit de maillage */
  /*=====================================================================*/

  vect_entmail = ecs_entmail_pre__cree_elt(cpt_elt_ent,
                                           elt_pos_som_ent,
                                           elt_val_som_ent,
                                           NULL,
                                           NULL,
                                           elt_val_couleur_ent,
                                           cpt_coul_ent,
                                           val_coul_ent,
                                           cpt_elt_coul_ent) ;


  return vect_entmail ;

}


/*----------------------------------------------------------------------------
 *  Cration des groupes bass sur les C.L.
 *   et de manire optionnelle sur les zones et les sections
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__cree_groupes
(
 const ecs_loc_cgns_base_t           *const base_maillage       ,
 const int                                  nzones              ,
       ecs_loc_cgns_zone_t           *const tab_zone            ,
 const int                                  nbr_boco_tot        ,
       char                   *const *const tab_nom_boco        ,
       ecs_loc_cgns_boco_t           *const tab_boco            ,
 const ecs_int_t                            nbr_boco_som        ,
       ecs_int_t              *      *const ind_nom_boco_som    ,
       ecs_int_t              *      *const nbr_sselt_boco_som  ,
       ecs_int_t            * *      *const tab_sselt_boco_som  ,
       ecs_int_t                     *const tab_ind_section_cel ,
       ecs_int_t                     *const tab_ind_section_fac ,
       ecs_int_t                     *const tab_ind_zone_cel    ,
       ecs_int_t                     *const tab_ind_zone_fac    ,
       ecs_entmail_t          *      *const vect_entmail
)
{

  ecs_int_t      ient ;

  size_t         ent_nbr_elt[ECS_ENTMAIL_FIN] ;
  ecs_int_t      ent_cpt_grp[ECS_ENTMAIL_FIN] ;

  ecs_champ_t  * ent_champ_def[ECS_ENTMAIL_FIN] ;
  ecs_champ_t  * ent_champ_grp[ECS_ENTMAIL_FIN] ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */
  /*=================*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    ent_cpt_grp[ient] = 0 ;

    if (vect_entmail[ient] != NULL)
      ent_champ_def[ient] = ecs_entmail__ret_champ(vect_entmail[ient],
                                                   ECS_CHAMP_DEF) ;
    else
      ent_champ_def[ient] = NULL ;

    if (ent_champ_def[ient] != NULL)
      ent_nbr_elt[ient] = ecs_champ__ret_pos_nbr(ent_champ_def[ient]) - 1 ;
    else
      ent_nbr_elt[ient] = 0 ;

    ent_champ_grp[ient] = NULL ;

  }


  /* Cration des groupes */
  /*======================*/

  ecs_loc_pre_cgns__cree_grps_boco(base_maillage,
                                   tab_zone,
                                   nbr_boco_tot,
                                   tab_nom_boco,
                                   tab_boco,
                                   nbr_boco_som,
                                   ind_nom_boco_som,
                                   nbr_sselt_boco_som,
                                   tab_sselt_boco_som,
                                   ent_nbr_elt,
                                   ent_cpt_grp,
                                   ent_champ_grp) ;

  ecs_loc_pre_cgns__cree_grps_zs(base_maillage,
                                 nzones,
                                 tab_zone,
                                 tab_ind_section_cel,
                                 tab_ind_section_fac,
                                 tab_ind_zone_cel,
                                 tab_ind_zone_fac,
                                 ent_nbr_elt,
                                 ent_cpt_grp,
                                 ent_champ_grp) ;


  /* Transfert des valeurs lues dans les structures d'entit de maillage */
  /*=====================================================================*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (vect_entmail[ient] != NULL && ent_champ_grp[ient] != NULL) {

      ecs_entmail__ajoute_champ(vect_entmail[ient] ,
                                ent_champ_grp[ient] ,
                                ECS_CHAMP_ATT ) ;

    }

  }
}


/*----------------------------------------------------------------------------
 *                   Cration des groupes bass sur les C.L.
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__cree_grps_boco
(
 const ecs_loc_cgns_base_t           *const base_maillage      ,
       ecs_loc_cgns_zone_t           *const tab_zone           ,
 const int                                  nbr_boco_tot       ,
       char                   *const *const tab_nom_boco       ,
       ecs_loc_cgns_boco_t           *const tab_boco           ,
 const ecs_int_t                            nbr_boco_som       ,
       ecs_int_t              *      *const ind_nom_boco_som   ,
       ecs_int_t              *      *const nbr_sselt_boco_som ,
       ecs_int_t            * *      *const tab_sselt_boco_som ,
       size_t                        *const ent_nbr_elt        ,
       ecs_int_t                     *const ent_cpt_grp        ,
       ecs_champ_t            *      *const ent_champ_grp
)
{

  ecs_int_t    ient ;
  ecs_int_t    ient_max ;

  ecs_int_t    ind ;
  ecs_int_t    ind_boco ;
  ecs_int_t    ind_boco_sub ;
  ecs_int_t    ind_ent ;
  ecs_int_t    ind_glob ;
  ecs_int_t    ind_nom ;

  int          cel_dim ;

  ecs_int_t    nbr_sselt_boco ;
  ecs_int_t   *tab_sselt_boco ;

  ecs_loc_cgns_zone_t   *ptr_zone ;
  ecs_loc_cgns_boco_t   *ptr_boco ;

  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  char       * nom_boco ;

  ecs_bool_t   bool_aff_grp ;

  ecs_descr_t  * descr_grp ;
  size_t         ent_cpt_elt[ECS_ENTMAIL_FIN] ;
  ecs_int_t    * ent_val_grp[ECS_ENTMAIL_FIN] ;

  ecs_champ_t  * champ_grp ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  cel_dim = base_maillage->dim_entite ;

  if (cel_dim == 1)
    ient_max = ECS_ENTMAIL_ARE ;
  else if (cel_dim == 2)
    ient_max = ECS_ENTMAIL_FAC ;
  else
    ient_max = ECS_ENTMAIL_CEL ;

  ind_boco = 0 ;


  /* Allocations des tableaux locaux */

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    ent_cpt_grp[ient] = 0 ;

    ent_champ_grp[ient] = NULL ;

    ent_cpt_elt[ient] = 0 ;
    ent_val_grp[ient] = NULL ;

  }


  /*-----------------------------------------*/
  /* Gnration des groupes associs aux C.L */
  /*-----------------------------------------*/

  for (ind_boco = 0 ; ind_boco < nbr_boco_tot ; ind_boco++) {

    ptr_boco = tab_boco + ind_boco ;

    ind_nom  = ptr_boco->ind_nom ;

    if (ind_nom < 0)
      continue ;

    nom_boco = tab_nom_boco[ind_nom] ;


    /* On alloue et initialise pour le groupe  lire */

    for (ient = ient_max - 1 ; ient <= ient_max ; ient++) {

      ent_cpt_elt[ient] = 0 ;

      if (ent_nbr_elt[ient] != 0)
        BFT_MALLOC(ent_val_grp[ient], ent_nbr_elt[ient], ecs_int_t) ;

      for (ind = 0 ; ind < (ecs_int_t)(ent_nbr_elt[ient]) ; ind++)
        ent_val_grp[ient][ind] = 0 ;

    }


    /* Boucle pour le traitement des C.L. aux sommets */
    /*------------------------------------------------*/

    for (ind_boco_sub = 0 ;
         ind_boco_sub < nbr_boco_som ;
         ind_boco_sub++) {

      if ((*ind_nom_boco_som)[ind_boco_sub] != ind_nom)
        continue ;

      /* Traitement prpar dans ecs_loc_pre__cgns__cree_ent_inf_som() */

      ient = ient_max - 1 ;

      nbr_sselt_boco = (*nbr_sselt_boco_som)[ind_boco_sub] ;
      tab_sselt_boco = (*tab_sselt_boco_som)[ind_boco_sub] ;

      for (ind = 0 ; ind < nbr_sselt_boco ; ind++) {

        ind_glob = tab_sselt_boco[ind] ;

        /* Stockage des valeurs lues avant transfert dans maillage */

        ent_val_grp[ient][ind_glob] = ent_cpt_grp[ient] + 1 ;
        ent_cpt_elt[ient]++ ;

      }

      nbr_sselt_boco = 0 ;
      tab_sselt_boco = NULL ;

      (*nbr_sselt_boco_som)[ind_boco_sub] = 0;
      BFT_FREE((*tab_sselt_boco_som)[ind_boco_sub]) ;

      /* Marquage C.L. sommets comme dj utilise */

      (*ind_nom_boco_som)[ind_boco_sub] = -1 ;

    }


    /* Boucle pour le traitement des C.L. aux faces */
    /*----------------------------------------------*/

    /*
      Boucle infrieure pour parcourir toutes les C.L. de mme nom
      (utile notamment si l'on a plusieurs zones)
    */

    for (ind_boco_sub = ind_boco ;
         ind_boco_sub < nbr_boco_tot ;
         ind_boco_sub++) {

      ptr_boco = tab_boco + ind_boco_sub ;

      if (ptr_boco->ind_nom != ind_nom)
        continue ;

      /* Traitement selon le support */
      /*-----------------------------*/

      if (ptr_boco->support == FaceCenter) {

        ptr_zone = tab_zone + ptr_boco->num_zone - 1 ;

        /* Liste dfinie par numro de dbut et fin */

        if (   ptr_boco->ptset_type == PointRange
            || ptr_boco->ptset_type == ElementRange) {

          for (ind = ptr_boco->pnts[0] ; ind <= ptr_boco->pnts[1] ; ind++) {

            ind_ent = ind - ptr_zone->num_elt_deb ;

            /* Stockage des valeurs lues avant transfert dans maillage */

            if (ptr_zone->renum[ind_ent] > 0) {
              ient = ient_max - 1 ;
              ind_glob = ptr_zone->renum[ind_ent] - 1 ;
              ent_val_grp[ient][ind_glob] = ent_cpt_grp[ient] + 1 ;
              ent_cpt_elt[ient]++ ;
            }
            else if (ptr_zone->renum[ind_ent] < 0) {
              ient = ient_max ;
              ind_glob = - ptr_zone->renum[ind_ent] - 1 ;
              ent_val_grp[ient][ind_glob] = ent_cpt_grp[ient] + 1 ;
              ent_cpt_elt[ient]++ ;
            }

          }

        }

        /* Liste dfinie explicitement */

        else {

          for (ind = 0 ; ind < ptr_boco->npnts ; ind++) {

            ind_ent = ptr_boco->pnts[ind] - ptr_zone->num_elt_deb ;

            /* Stockage des valeurs lues avant transfert dans maillage */

            if (ptr_zone->renum[ind_ent] > 0) {
              ient = ient_max - 1 ;
              ind_glob = ptr_zone->renum[ind_ent] - 1 ;
              ent_val_grp[ient][ind_glob] = ent_cpt_grp[ient] + 1 ;
              ent_cpt_elt[ient]++ ;
            }
            else if (ptr_zone->renum[ind_ent] < 0) {
              ient = ient_max ;
              ind_glob = - ptr_zone->renum[ind_ent] - 1 ;
              ent_val_grp[ient][ind_glob] = ent_cpt_grp[ient] + 1 ;
              ent_cpt_elt[ient]++ ;
            }

          }

        }

        /* Marquage C.L. aprs utilisation et libration mmoire partielle */

        ptr_boco->ind_nom = - 1 ;

        BFT_FREE(ptr_boco->pnts) ;

      }

    } /* Fin de la boucle infrieure */

    /* Retour si aucune C.L. traite */

    if (ient > ient_max)
      continue ;


    /* Remplissage des entits du maillage */
    /*-------------------------------------*/

    bool_aff_grp = ECS_FALSE ;

    if (ent_cpt_elt[ient] != 0) {

      bool_aff_grp = ECS_TRUE ;

      assert(ent_cpt_elt[ient] <= (size_t)(ent_nbr_elt[ient])) ;

      ent_cpt_grp[ient]++ ;

      /* Cration du descripteur de champ correspondant au groupe lu */

      descr_grp = ecs_descr__cree(ECS_DESCR_TYP_GROUPE,
                                  ECS_DESCR_IDE_NUL,
                                  nom_boco) ;


      /* Transformation du tableau rfrencant le groupe en une table */

      champ_grp = ecs_champ__transforme_tableau(ent_nbr_elt[ient],
                                                ent_val_grp[ient],
                                                ECS_CHAMP_NOM_GROUPE,
                                                descr_grp,
                                                ECS_CHAMP_STATUT_HERITABLE) ;

      if (ent_champ_grp[ient] != NULL)
          ecs_champ_att__assemble(ent_champ_grp[ient],
                                  champ_grp,
                                  NULL) ;

      else
        ent_champ_grp[ient] = champ_grp ;

    } /* Fin si le nombre d'lments rfrencant le groupe n'est pas nul */


    /* Libration mmoire */

    for (ient = ient_max - 1 ; ient <= ient_max ; ient++) {
      if (ent_nbr_elt[ient] != 0)
        BFT_FREE(ent_val_grp[ient]) ;
    }


    /* Affichage du bilan des donnes lues pour les groupes */
    /*------------------------------------------------------*/

    if (bool_aff_grp == ECS_TRUE)
      bft_printf("  %s \"%s\"\n",
                 _("Group"), nom_boco) ;

    ecs_entmail__aff_nbr_par_ent(ent_cpt_elt,
                                 0) ;

  } /* Fin de la boucle sur les C.L. */


  /* Libration mmoire associe aux tableaux sur les sommets */

  if (nbr_boco_som > 0) {
    BFT_FREE(*ind_nom_boco_som) ;
    BFT_FREE(*nbr_sselt_boco_som) ;
    BFT_FREE(*tab_sselt_boco_som) ;
  }

  /* Libration de la mmoire associe  d'ventuelles listes non utilises */

  for (ind_boco = 0 ; ind_boco < nbr_boco_tot ; ind_boco++)
    BFT_FREE(tab_boco[ind_boco].pnts) ;

}


/*----------------------------------------------------------------------------
 *            Cration des groupes bass sur les zones et sections
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__cree_grps_zs
(
 const ecs_loc_cgns_base_t           *const base_maillage       ,
 const int                                  nzones              ,
       ecs_loc_cgns_zone_t           *const tab_zone            ,
       ecs_int_t                     *const tab_ind_section_cel ,
       ecs_int_t                     *const tab_ind_section_fac ,
       ecs_int_t                     *const tab_ind_zone_cel    ,
       ecs_int_t                     *const tab_ind_zone_fac    ,
       size_t                        *const ent_nbr_elt         ,
       ecs_int_t                     *const ent_cpt_grp         ,
       ecs_champ_t                *  *const ent_champ_grp
)
{

  ecs_int_t    ient ;
  ecs_int_t    ient_max ;

  ecs_int_t    ind ;

  int          cel_dim ;

  ecs_loc_cgns_zone_t      *ptr_zone ;
  ecs_loc_cgns_section_t   *ptr_section ;

  ecs_int_t     nbr_sections_trait ;
  ecs_int_t     nbr_zones_trait ;

  ecs_int_t    *renum_sections ;
  ecs_int_t    *renum_zones ;
  char        **ptr_noms_sections ;
  char        **ptr_noms_zones ;

  ecs_bool_t   trait_section ;
  ecs_bool_t   trait_zone ;

  ecs_int_t    cpt_section ;
  ecs_int_t    ind_section ;
  ecs_int_t    ind_pass ;
  ecs_int_t    ind_sub ;
  ecs_int_t    ind_zone ;
  ecs_int_t    ind_zs ;

  char        *nom_grp ;

  ecs_int_t   *tab_ind_zs_ent ;
  ecs_int_t   *renum_zs ;
  char       **ptr_nom_zs ;

  /* Dclarations des variables de stockage        */
  /* avant transfert dans la structure du maillage */
  /*-----------------------------------------------*/

  ecs_bool_t   bool_aff_grp ;
  ecs_bool_t   bool_termine ;

  ecs_descr_t  * descr_grp ;
  size_t         ent_cpt_elt[ECS_ENTMAIL_FIN] ;
  ecs_int_t    * ent_val_grp[ECS_ENTMAIL_FIN] ;

  ecs_champ_t  * champ_grp ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  cel_dim = base_maillage->dim_entite ;

  if (cel_dim == 1)
    ient_max = ECS_ENTMAIL_ARE ;
  else if (cel_dim == 2)
    ient_max = ECS_ENTMAIL_FAC ;
  else
    ient_max = ECS_ENTMAIL_CEL ;


  nbr_sections_trait = 0 ;
  nbr_zones_trait    = 0 ;

  renum_sections = NULL ;
  renum_zones    = NULL ;
  ptr_noms_sections = NULL ;
  ptr_noms_zones    = NULL ;

  trait_section = ECS_FALSE ;
  if (   tab_ind_section_cel != NULL
      || tab_ind_section_fac != NULL)
    trait_section = ECS_TRUE ;

  trait_zone = ECS_FALSE ;
  if (   tab_ind_zone_cel != NULL
      || tab_ind_zone_fac != NULL)
    trait_zone = ECS_TRUE ;


  if (trait_section == ECS_FALSE && trait_zone == ECS_FALSE)
    return ;

  /* Renumrotation des noms et numros de sections si ncessaire */

  if (trait_section == ECS_TRUE) {

    cpt_section = 0 ;

    for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {
      ptr_zone = tab_zone + ind_zone ;
      if (ptr_zone->type == Unstructured)
        cpt_section += ptr_zone->nbr_sections ;
    }

    BFT_MALLOC(renum_sections, cpt_section, ecs_int_t) ;
    BFT_MALLOC(ptr_noms_sections, cpt_section, char *) ;

    cpt_section = 0 ;

    for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

      ptr_zone = tab_zone + ind_zone ;

      if (ptr_zone->type == Unstructured) {

        for (ind_section = 0 ;
             ind_section < ptr_zone->nbr_sections ;
             ind_section++) {

          ptr_section = ptr_zone->tab_sections + ind_section ;

          for (ind_sub = 0 ; ind_sub < cpt_section ; ind_sub++) {
            if (strcmp(ptr_noms_sections[ind_sub], ptr_section->nom) == 0)
              break ;
          }

          nbr_sections_trait = ECS_MAX(ind_sub, nbr_sections_trait) ;

          renum_sections[cpt_section] = ind_sub ;
          ptr_noms_sections[renum_sections[cpt_section]] = ptr_section->nom ;

          cpt_section += 1 ;

        }

      }

    }

    nbr_sections_trait += 1 ;

  }


  /* Renumrotation des noms et numros de zones si ncessaire */

  if (trait_zone == ECS_TRUE) {

    BFT_MALLOC(renum_zones, nzones, ecs_int_t) ;
    BFT_MALLOC(ptr_noms_zones, nzones, char *) ;

    for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++) {

      ptr_zone = tab_zone + ind_zone ;

      for (ind_sub = 0 ; ind_sub < ind_zone ; ind_sub++) {
        if (strcmp(ptr_noms_zones[ind_sub], ptr_zone->nom) == 0)
          break ;
      }

      nbr_zones_trait = ECS_MAX(ind_sub, nbr_zones_trait) ;

      renum_zones[ind_zone] = ind_sub ;
      ptr_noms_zones[renum_zones[ind_zone]] = ptr_zone->nom ;

    }

    nbr_zones_trait += 1 ;

  }


  /* Initialisations des tableaux locaux */

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    ent_cpt_elt[ient] = 0 ;
    ent_val_grp[ient] = NULL ;

  }


  /*-------------------------------------------------------*/
  /* Gnration des groupes associs aux zones et sections */
  /*-------------------------------------------------------*/

  for (ind_pass = 0 ; ind_pass < 4 ; ind_pass++) {

    switch(ind_pass) {

    case 0:
      tab_ind_zs_ent = tab_ind_section_fac ;
      ient = ient_max - 1 ;
      renum_zs = renum_sections ;
      ptr_nom_zs = ptr_noms_sections ;

      break ;

    case 1:
      tab_ind_zs_ent = tab_ind_zone_fac ;
      ient = ient_max - 1 ;
      renum_zs = renum_zones ;
      ptr_nom_zs = ptr_noms_zones ;

      break ;

    case 2:
      tab_ind_zs_ent = tab_ind_section_cel ;
      ient = ient_max ;
      renum_zs = renum_sections ;
      ptr_nom_zs = ptr_noms_sections ;

      break ;

    case 3:
      tab_ind_zs_ent = tab_ind_zone_cel ;
      ient = ient_max ;
      renum_zs = renum_zones ;
      ptr_nom_zs = ptr_noms_zones ;

      break ;

    default:
      assert(0) ;

    }

    if (tab_ind_zs_ent != NULL && ient > ECS_ENTMAIL_SOM) {

      ind_zs = 0 ;

      do { /* Boucle tant que sur les groupes  crer */

        bool_termine = ECS_TRUE ;

        /* On alloue et initialise pour le groupe  lire */

        ent_cpt_elt[ient] = 0 ;

        if (ent_nbr_elt[ient] != 0)
          BFT_MALLOC(ent_val_grp[ient], ent_nbr_elt[ient], ecs_int_t) ;

        for (ind = 0 ; ind < (ecs_int_t)(ent_nbr_elt[ient]) ; ind++)
          ent_val_grp[ient][ind] = 0 ;

        for (ind = 0 ; ind < (ecs_int_t)(ent_nbr_elt[ient]) ; ind++) {

          if (renum_zs[tab_ind_zs_ent[ind]] == ind_zs) {

            ent_val_grp[ient][ind] = ent_cpt_grp[ient] + 1 ;
            ent_cpt_elt[ient]++ ;
            nom_grp = ptr_nom_zs[ind_zs] ;

          }
          else if (renum_zs[tab_ind_zs_ent[ind]] > ind_zs)

            bool_termine = ECS_FALSE ;

        }

        ind_zs += 1 ;

        /* Remplissage des entits du maillage */
        /*-------------------------------------*/

        bool_aff_grp = ECS_FALSE ;

        if (ent_cpt_elt[ient] != 0) {

          bool_aff_grp = ECS_TRUE ;

          assert(ent_cpt_elt[ient] <= (size_t)(ent_nbr_elt[ient])) ;

          ent_cpt_grp[ient]++ ;

          /* Cration du descripteur de champ correspondant au groupe lu */

          descr_grp = ecs_descr__cree(ECS_DESCR_TYP_GROUPE,
                                      ECS_DESCR_IDE_NUL,
                                      nom_grp) ;


          /* Transformation du tableau rfrencant le groupe en une table */

          champ_grp = ecs_champ__transforme_tableau(ent_nbr_elt[ient],
                                                    ent_val_grp[ient],
                                                    ECS_CHAMP_NOM_GROUPE,
                                                    descr_grp,
                                                    ECS_CHAMP_STATUT_HERITABLE) ;

          if (ent_champ_grp[ient] != NULL) {
            ecs_champ_att__assemble(ent_champ_grp[ient],
                                    champ_grp,
                                    NULL) ;
          }

          else
            ent_champ_grp[ient] = champ_grp ;

        } /* Fin si le nombre d'lments rfrencant le groupe n'est pas nul */


        /* Libration mmoire */

        if (ent_nbr_elt[ient] != 0)
          BFT_FREE(ent_val_grp[ient]) ;


        /* Affichage du bilan des donnes lues pour les groupes */
        /*------------------------------------------------------*/

        if (bool_aff_grp == ECS_TRUE)
          bft_printf("  %s \"%s\"\n",
                     _("Group"), nom_grp) ;

        ecs_entmail__aff_nbr_par_ent(ent_cpt_elt,
                                     0) ;

        ent_cpt_elt[ient] = 0 ;


      } while (bool_termine == ECS_FALSE) ;

    } /* Fin test si combinaison entit/section ou zones  traiter */

  } /* Fin boucle sur les combinaisons */


  /* Libration mmoire*/

  BFT_FREE(renum_sections) ;
  BFT_FREE(renum_zones) ;
  BFT_FREE(ptr_noms_sections) ;
  BFT_FREE(ptr_noms_zones) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui cr des entits de maillage supplmentaires pour porter
 *  des conditions aux limites dfinies sur les sommets.
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre__cgns__cree_ent_inf_som
(
 const ecs_loc_cgns_base_t      *const base_maillage         ,
       size_t                          nbr_elt_ent        [] ,
       ecs_size_t               *      elt_pos_som_ent    [] ,
       ecs_int_t                *      elt_val_som_ent    [] ,
 const int                             nzones                ,
       ecs_loc_cgns_zone_t      *const tab_zone              ,
 const int                             nbr_boco_tot          ,
       ecs_loc_cgns_boco_t      *const tab_boco              ,
       ecs_int_t                *const nbr_boco_som          ,
       ecs_int_t              * *const ind_nom_boco_som      ,
       ecs_int_t              * *const nbr_sselt_boco_som    ,
       ecs_int_t            * * *const tab_sselt_boco_som
)
{

  size_t       cpt_sselt ;
  size_t       cpt_sselt_boco ;
  size_t       cpt_val_sselt ;
  size_t       nbr_sselt_ini ;
  size_t       nbr_val_sselt_ini ;

  int          ient_max ;

  int          ind_boco ;
  int          ind_boco_sub ;
  ecs_int_t    cpt_boco_som ;
  size_t       ind_elt ;
  size_t       ind_pos_elt ;
  size_t       ind_pos_sselt ;
  size_t       ind_som ;
  size_t       ind_sselt ;

  size_t       nbr_elt ;
  size_t       nbr_som ;
  size_t       nbr_som_elt ;
  int          num_boco ;
  ecs_int_t    num_def ;

  ecs_int_t    typ_elt ;

  ecs_bool_t   boco_som ;
  ecs_bool_t   bool_cree ;

  int          cel_dim ;
  int          ind_zone ;

  ecs_int_t   *indic_som ;
  ecs_int_t   *indic_sselt ;
  ecs_int_t   *liste_sselt ;
  ecs_int_t   *renum_sselt ;

  ecs_size_t  *pos_som_elt ;
  ecs_int_t   *val_som_elt ;

  ecs_size_t  *pos_som_sselt ;
  ecs_int_t   *val_som_sselt ;

  ecs_loc_cgns_zone_t   *ptr_zone ;
  ecs_loc_cgns_boco_t   *ptr_boco ;

  const ecs_elt_typ_t  *typ_geo_base ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructionsxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  cel_dim = base_maillage->dim_entite ;

  if (cel_dim == 1)
    ient_max = ECS_ENTMAIL_ARE ;
  else if (cel_dim == 2)
    ient_max = ECS_ENTMAIL_FAC ;
  else
    ient_max = ECS_ENTMAIL_CEL ;

  if (ient_max < ECS_ENTMAIL_FAC)
    return ;

  ind_boco = 0 ;

  cpt_boco_som = 0 ;

  *nbr_boco_som = 0 ;
  *ind_nom_boco_som = NULL ;
  *nbr_sselt_boco_som = NULL ;
  *tab_sselt_boco_som = NULL ;


  /* Dtermination de base du type d'lment "classique" */

  assert(cel_dim >=1 && cel_dim <= 3) ;

  typ_geo_base = ecs_glob_typ_elt[cel_dim - 1] ;


  /* On vrifie que l'on a quelque chose  faire */
  /*---------------------------------------------*/

  boco_som = ECS_FALSE ;

  for (ind_boco = 0 ; ind_boco < nbr_boco_tot ; ind_boco++) {

    ptr_boco = tab_boco + ind_boco ;

    if (ptr_boco->support == Vertex)
      boco_som = ECS_TRUE ;

  }

  if (boco_som == ECS_FALSE)
    return ;


  /* Initialisation du marquage des sommets */
  /*----------------------------------------*/

  nbr_som = 0 ;

  for (ind_zone = 0 ; ind_zone < nzones ; ind_zone++)
    nbr_som += (tab_zone + ind_zone)->nbr_som ;

  BFT_MALLOC(indic_som, nbr_som, ecs_int_t) ;

  for (ind_som = 0 ; ind_som < nbr_som ; ind_som++)
    indic_som[ind_som] = 0 ;

  for (ind_boco = 0 ; ind_boco < nbr_boco_tot ; ind_boco++) {

    ptr_boco = tab_boco + ind_boco ;

    num_boco = ptr_boco->ind_nom + 1 ;

    if (ptr_boco->support == Vertex) {

      ptr_zone = tab_zone + ptr_boco->num_zone - 1 ;

      ecs_loc_pre_cgns__marque_som(base_maillage,
                                   num_boco,
                                   ptr_zone,
                                   ptr_boco->ptset_type,
                                   ptr_boco->npnts,
                                   ptr_boco->pnts,
                                   indic_som) ;

    }

  }


  /* Comptage pour la cration des entits infrieures */
  /*---------------------------------------------------*/

  /*
    On cr des entits infrieures lorsque tous les sommets
    d'une entit infrieure appartiennent  une C.L.
    Ceci cre des faces en trop (i.e. toutes les faces intrieures)
    dans le cas d'un maillage 2D extrud avec une seule couche
    d'lments, mais vite d'effectuer une boucle sur les lments
    par C.L. et par zone. On compactera les entits par la suite
    (avant de passer en connectivit descendante)
  */

  nbr_elt     = nbr_elt_ent        [ient_max] ;
  pos_som_elt = elt_pos_som_ent    [ient_max] ;
  val_som_elt = elt_val_som_ent    [ient_max] ;


  /* Boucle sur lments  */

  cpt_sselt  = 0 ;
  cpt_val_sselt  = 0 ;

  for (ind_elt = 0 ; ind_elt < nbr_elt ; ind_elt++) {

    nbr_som_elt = pos_som_elt[ind_elt + 1] - pos_som_elt[ind_elt] ;

    if (nbr_som_elt < 9)
      typ_elt = typ_geo_base[nbr_som_elt] ;
    else
      typ_elt = ECS_ELT_TYP_FAC_POLY ;

    ind_pos_elt = pos_som_elt[ind_elt] - 1 ;

    if (typ_elt != ECS_ELT_TYP_FAC_POLY) {

      /* Boucle sur les sous-lments dfinissant l'lment */

      for (ind_sselt = 0 ;
           ind_sselt < (size_t)(ecs_fic_elt_typ_liste_c[typ_elt].nbr_sous_elt);
           ind_sselt++) {

        /* Boucle sur les sommets dfinissant le sous-lment */

        ind_som = 0 ;

        bool_cree = ECS_TRUE ;

        while(   ind_som < ECS_CGNS_SSELT_NBR_MAX_SOM
              && bool_cree == ECS_TRUE
              && (num_def = ecs_fic_elt_typ_liste_c
                              [typ_elt].sous_elt[ind_sselt].som[ind_som]) != 0) {

          if (indic_som[val_som_elt[ind_pos_elt + num_def - 1] - 1] == 0)
            bool_cree = ECS_FALSE ;

          ind_som++ ;

        } /* Fin de la boucle sur les sommets dfinissant le sous-lment */

        if (bool_cree == ECS_TRUE) {
          cpt_sselt     += 1 ;
          cpt_val_sselt += ind_som ;
        }

      }  /* Fin de la boucle sur les sous-lments d'un lment */

    }
    else if (typ_elt == ECS_ELT_TYP_FAC_POLY) {

      bool_cree = ECS_TRUE ;

      while(   ind_pos_elt + ind_som < pos_som_elt[ind_elt + 1] - 1
            && bool_cree == ECS_TRUE) {

        if (indic_som[val_som_elt[ind_pos_elt + ind_som] - 1] == 0)
          bool_cree = ECS_FALSE ;

        ind_som++ ;

      } /* Fin de la boucle sur les sommets dfinissant le sous-lment */

      if (bool_cree == ECS_TRUE) {
        cpt_sselt     += 1 ;
        cpt_val_sselt += ind_som ;
      }

    } /* Fin du cas spcifique  un polygone */

  } /* Fin de la boucle sur les lments */


  if (cpt_sselt == 0) {
    BFT_FREE(indic_som) ;
    return ;
  }

  /* Construction des entits infrieures */
  /*---------------------------------------*/

  nbr_sselt_ini = nbr_elt_ent[ient_max - 1] ;
  if (nbr_sselt_ini > 0)
    nbr_val_sselt_ini = elt_pos_som_ent[ient_max - 1][nbr_sselt_ini] - 1 ;
  else
    nbr_val_sselt_ini = 0;

  BFT_REALLOC(elt_pos_som_ent[ient_max - 1], nbr_sselt_ini + cpt_sselt + 1,
              ecs_size_t) ;
  BFT_REALLOC(elt_val_som_ent[ient_max - 1], nbr_val_sselt_ini + cpt_val_sselt,
              ecs_int_t) ;

  nbr_elt_ent[ient_max - 1] += cpt_sselt ;

  pos_som_sselt = elt_pos_som_ent[ient_max - 1] ;
  val_som_sselt = elt_val_som_ent[ient_max - 1] ;

  if (nbr_sselt_ini == 0)
    pos_som_sselt[0] = 1 ;

  /* Boucle sur les lments  */

  cpt_sselt  = nbr_sselt_ini ;
  cpt_val_sselt  = nbr_val_sselt_ini ;

  for (ind_elt = 0 ; ind_elt < nbr_elt ; ind_elt++) {

    nbr_som_elt = pos_som_elt[ind_elt + 1] - pos_som_elt[ind_elt] ;

    if (nbr_som_elt < 9)
      typ_elt = typ_geo_base[nbr_som_elt] ;
    else
      typ_elt = ECS_ELT_TYP_FAC_POLY ;

    ind_pos_elt = pos_som_elt[ind_elt] - 1 ;

    if (typ_elt != ECS_ELT_TYP_FAC_POLY) {

      /* Boucle sur les sous-lments dfinissant l'lment */

      for (ind_sselt = 0 ;
           ind_sselt < (size_t)(ecs_fic_elt_typ_liste_c[typ_elt].nbr_sous_elt);
           ind_sselt++) {

        /* Boucle sur les sommets dfinissant le sous-lment */

        ind_som = 0 ;

        bool_cree = ECS_TRUE ;

        while(   ind_som < ECS_CGNS_SSELT_NBR_MAX_SOM
              && bool_cree == ECS_TRUE
              && (num_def = ecs_fic_elt_typ_liste_c
                              [typ_elt].sous_elt[ind_sselt].som[ind_som]) != 0) {

          if (indic_som[val_som_elt[ind_pos_elt + num_def - 1] - 1] == 0)
            bool_cree = ECS_FALSE ;

          ind_som++ ;

        }


        /* Si l'on cre un lment */

        if (bool_cree == ECS_TRUE) {

          ind_som = 0 ;

          while(   ind_som < ECS_CGNS_SSELT_NBR_MAX_SOM
                && bool_cree == ECS_TRUE
                && (num_def
                      = ecs_fic_elt_typ_liste_c
                          [typ_elt].sous_elt[ind_sselt].som[ind_som]) != 0) {

            /* Dfinition du sous-lment en fonction des sommets */

            val_som_sselt[cpt_val_sselt++]
              = val_som_elt[ind_pos_elt + num_def - 1] ;

            ind_som++ ;

          }

          /* Position du sous-elt dans sa dfinition en fonction des sommets */

          pos_som_sselt[cpt_sselt + 1] = pos_som_sselt[cpt_sselt] + ind_som ;

          cpt_sselt     += 1 ;

        } /* Fin cration lment */

      } /* Fin de la boucle sur les sous-lments d'un lment */

    }
    else if (typ_elt == ECS_ELT_TYP_FAC_POLY) {

      while(   ind_pos_elt + ind_som < pos_som_elt[ind_elt + 1] - 1
            && bool_cree == ECS_TRUE) {

        if (indic_som[val_som_elt[ind_pos_elt + ind_som] - 1] == 0)
          bool_cree = ECS_FALSE ;

        ind_som++ ;

      } /* Fin de la boucle sur les sommets dfinissant le sous-lment */

      if (bool_cree == ECS_TRUE) {
        cpt_sselt     += 1 ;
        cpt_val_sselt += ind_som ;
      }

      /* Si l'on cre un lment */

      if (bool_cree == ECS_TRUE) {

        ind_som = 0 ;

        while(   ind_pos_elt + ind_som < pos_som_elt[ind_elt + 1] - 1
                 && bool_cree == ECS_TRUE) {

          /* Dfinition du sous-lment en fonction des sommets */

          val_som_sselt[cpt_val_sselt++] = val_som_elt[ind_pos_elt + ind_som] ;

          ind_som++ ;

        }

        /* Position du sous-elt dans sa dfinition en fonction des sommets */

        pos_som_sselt[cpt_sselt + 1] = pos_som_sselt[cpt_sselt] + ind_som ;

        cpt_sselt     += 1 ;

      } /* Fin cration lment */

    } /* Fin du cas spcifique  un polygone */


  } /* Fin de la boucle sur les lments */


  /* Cration listes d'lments infrieurs et prparation compactage */
  /*-----------------------------------------------------------------*/

  BFT_MALLOC(indic_sselt, nbr_elt_ent[ient_max - 1], ecs_int_t) ;
  BFT_MALLOC(liste_sselt, nbr_elt_ent[ient_max - 1], ecs_int_t) ;


  for (ind_sselt = 0 ; ind_sselt < nbr_elt_ent[ient_max - 1] ; ind_sselt++)
    indic_sselt[ind_sselt] = 0 ;


  cpt_sselt = nbr_sselt_ini ;

  for (ind_boco = 0 ; ind_boco < nbr_boco_tot ; ind_boco++) {

    ptr_boco = tab_boco + ind_boco ;

    if (ptr_boco->ind_nom > -1 && ptr_boco->support == Vertex) {

      num_boco = ptr_boco->ind_nom + 1 ;

      BFT_REALLOC(*ind_nom_boco_som,   cpt_boco_som + 1, ecs_int_t) ;
      BFT_REALLOC(*nbr_sselt_boco_som, cpt_boco_som + 1, ecs_int_t) ;
      BFT_REALLOC(*tab_sselt_boco_som, cpt_boco_som + 1, ecs_int_t *) ;

      (*ind_nom_boco_som)[cpt_boco_som] = ptr_boco->ind_nom ;
      (*nbr_sselt_boco_som)[cpt_boco_som] = 0 ;

      cpt_sselt_boco = 0 ;

      /*
        Boucle infrieure pour parcourir toutes les C.L. de mme nom
        (utile notamment si l'on a plusieurs zones)
      */

      for (ind_boco_sub = ind_boco ;
           ind_boco_sub < nbr_boco_tot ;
           ind_boco_sub++) {

        ptr_boco = tab_boco + ind_boco_sub ;

        if (ptr_boco->ind_nom + 1 != num_boco)
          continue ;

        if (ptr_boco->support == Vertex) {

          ptr_zone = tab_zone + ptr_boco->num_zone - 1 ;

          ecs_loc_pre_cgns__marque_som(base_maillage,
                                       num_boco,
                                       ptr_zone,
                                       ptr_boco->ptset_type,
                                       ptr_boco->npnts,
                                       ptr_boco->pnts,
                                       indic_som) ;

          /* Marquage C.L. aprs utilisation et libration mmoire partielle */

          if (ind_boco_sub > ind_boco)
            ptr_boco->ind_nom = - 1 ;

          ptr_boco->ptset_type = PointSetTypeNull ;
          ptr_boco->npnts = 0 ;
          BFT_FREE(ptr_boco->pnts) ;

        }

      } /* Fin de la boucle infrieure */

      /* On marque les lments infrieurs effectivement utiliss */

      for (ind_sselt = nbr_sselt_ini ;
           ind_sselt < nbr_elt_ent[ient_max - 1] ;
           ind_sselt++) {

        bool_cree = ECS_TRUE ;
        for (ind_pos_sselt = pos_som_sselt[ind_sselt    ] - 1 ;
             ind_pos_sselt < pos_som_sselt[ind_sselt + 1] - 1 ;
             ind_pos_sselt++) {
          if (indic_som[val_som_sselt[ind_pos_sselt] - 1] != num_boco)
            bool_cree = ECS_FALSE ;
        }

        if (bool_cree == ECS_TRUE) {

          if (indic_sselt[ind_sselt] == 0) {
            indic_sselt[ind_sselt] = 1 ;
            cpt_sselt++ ;
          }

          liste_sselt[cpt_sselt_boco] = ind_sselt ;
          cpt_sselt_boco++ ;

        }

      }

      if (cpt_sselt_boco > 0) {

        (*nbr_sselt_boco_som)[cpt_boco_som] = cpt_sselt_boco ;

        BFT_MALLOC((*tab_sselt_boco_som)[cpt_boco_som],
                   cpt_sselt_boco, ecs_int_t) ;

        for (ind_sselt = 0 ; ind_sselt < cpt_sselt_boco ; ind_sselt++)
          (*tab_sselt_boco_som)[cpt_boco_som][ind_sselt]
            = liste_sselt[ind_sselt] ;

        cpt_boco_som++ ;

      }

    }

  }

  *nbr_boco_som = cpt_boco_som ;

  BFT_FREE(indic_som) ;
  BFT_FREE(liste_sselt) ;


  /* Compactage des dfinitions des lments infrieurs */
  /*----------------------------------------------------*/

  cpt_sselt      = nbr_sselt_ini ;
  cpt_val_sselt  = nbr_val_sselt_ini ;

  BFT_MALLOC(renum_sselt, nbr_elt_ent[ient_max - 1], ecs_int_t) ;

  for (ind_sselt = nbr_sselt_ini ;
       ind_sselt < nbr_elt_ent[ient_max - 1] ;
       ind_sselt++) {

    if (indic_sselt[ind_sselt] != 0) {

      renum_sselt[ind_sselt] = cpt_sselt ;

      pos_som_sselt[cpt_sselt] = cpt_val_sselt + 1 ;

      for (ind_pos_sselt = pos_som_sselt[ind_sselt    ] - 1 ;
           ind_pos_sselt < pos_som_sselt[ind_sselt + 1] - 1 ;
           ind_pos_sselt++) {
        val_som_sselt[cpt_val_sselt++]
          = val_som_sselt[ind_pos_sselt] ;
      }

      cpt_sselt += 1 ;

    }
    else

      renum_sselt[ind_sselt] = -1 ;

  }

  pos_som_sselt[cpt_sselt] = cpt_val_sselt + 1 ;

  /* Renumrotation des listes */

  for (ind_boco = 0 ; ind_boco < *nbr_boco_som ; ind_boco++) {

    for (ind_sselt = 0 ;
         ind_sselt < (size_t)((*nbr_sselt_boco_som)[ind_boco]) ;
         ind_sselt++)

      (*tab_sselt_boco_som)[ind_boco][ind_sselt]
        = renum_sselt[(*tab_sselt_boco_som)[ind_boco][ind_sselt]] ;

  }

  BFT_FREE(indic_sselt) ;
  BFT_FREE(renum_sselt) ;

  BFT_REALLOC(elt_pos_som_ent[ient_max - 1], cpt_sselt + 1, ecs_size_t) ;
  BFT_REALLOC(elt_val_som_ent[ient_max - 1], cpt_val_sselt, ecs_int_t) ;

  nbr_elt_ent[ient_max - 1] = cpt_sselt ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui marque les sommets correspondant  une liste.
 *----------------------------------------------------------------------------*/

static void ecs_loc_pre_cgns__marque_som
(
 const ecs_loc_cgns_base_t  *const base_maillage ,
 const int                         num_marque    ,
       ecs_loc_cgns_zone_t  *const ptr_zone      ,
 const PointSetType_t              ptset_type    ,
 const int                         npnts         ,
 const int                  *const pnts          ,
       ecs_int_t            *const indic_som
)
{

  ecs_int_t    ient_max ;

  ecs_int_t    ind_som ;
  ecs_int_t    ind_som_deb ;

  ecs_int_t    ii, jj, kk ;
  ecs_int_t    ni, nj ;

  int          cel_dim ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Initialisations */

  cel_dim = base_maillage->dim_entite ;

  if (cel_dim == 1)
    ient_max = ECS_ENTMAIL_ARE ;
  else if (cel_dim == 2)
    ient_max = ECS_ENTMAIL_FAC ;
  else
    ient_max = ECS_ENTMAIL_CEL ;


  /* Marquage des sommets */
  /*----------------------*/

  ind_som_deb = ptr_zone->num_som_deb - 1 ;


  /* Points dfinis par leur tendue */

  if (ptset_type == PointRange) {

    /* Cas structur */

    if (ptr_zone->type == Structured) {

      if (ient_max == ECS_ENTMAIL_ARE) {

        for (ii = pnts[0] - 1 ; ii < pnts[3] ; ii++)
          indic_som[ind_som_deb + ii] = num_marque ;

      }
      else if (ient_max == ECS_ENTMAIL_FAC) {

        ni = ptr_zone->taille[0] ;

        for (jj = pnts[1] - 1 ; jj < pnts[4] ; jj++) {
          for (ii = pnts[0] - 1 ; ii < pnts[3] ; ii++) {
            ind_som = ii + jj*ni ;
            indic_som[ind_som_deb + ind_som] = num_marque ;
          }
        }

      }
      else if (ient_max == ECS_ENTMAIL_CEL) {

        ni = ptr_zone->taille[0] ;
        nj = ptr_zone->taille[1] ;

        for (kk = pnts[2] - 1 ; kk < pnts[5] ; kk++) {
          for (jj = pnts[1] - 1 ; jj < pnts[4] ; jj++) {
            for (ii = pnts[0] - 1 ; ii < pnts[3] ; ii++) {
              ind_som = ii + jj*ni + kk*ni*nj ;
              indic_som[ind_som_deb + ind_som] = num_marque ;
            }
          }
        }
      }

    }

    /* Cas non structur */

    else {

      for (ind_som = pnts[0] ; ind_som < pnts[1] ; ind_som++)
        indic_som[ind_som_deb + ind_som] = num_marque ;

    }

  }

  /* Points dfinis par une liste */

  else {

    for (ind_som = 0 ; ind_som < npnts ; ind_som++)
      indic_som[ind_som_deb + pnts[ind_som]] = num_marque ;

  }

}


#endif /* HAVE_CGNS */
