/*============================================================================
 *  Dfinitions des fonctions de base
 *  associes  la structure `ecs_maillage_t' dcrivant un maillage.
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

#include "ecs_config.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>

#include <bft_error.h>
#include <bft_file.h>
#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' systme
 *----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_fic.h"
#include "ecs_tab.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_connect.h"
#include "ecs_entmail.h"
#include "ecs_entmail_pcp.h"
#include "ecs_entmail_post.h"
#include "ecs_famille_chaine.h"
#include "ecs_param_perio_glob.h"
#include "ecs_param_rc_glob.h"
#include "ecs_post.h"
#include "ecs_select_fac_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_post.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associ au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privs   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_priv.h"



/*============================================================================
 *                       Prototypes de fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *
 *  Simplification d'un maillage extrait correspondant  un cas de
 *  coupe de faces : on ne conserve du maillage que l'entit des
 *  faces et de cette entit que les champs filiation et type,
 *  suffisants pour raliser le post-traitement des variables
 *  centres sur les faces.
 *
 *----------------------------------------------------------------------------*/

static void ecs_loc_maillage__epure_coupe
(
       ecs_maillage_t    *const maillage_coupe ,
 const ecs_int_t                nbr_dump       ,
 const char              *const nom_fic_dump
) ;


/*----------------------------------------------------------------------------
 *
 *    Fonction d'affichage des infos d'une priodicit
 *
 *----------------------------------------------------------------------------*/

static void ecs_loc_maillage__print_perio
(
 const ecs_param_perio_t   param_perio
) ;


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/


/*----------------------------------------------------------------------------
 *   Fonction qui cr une structure dfinissant un maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__cree
(                                      /* <-- Pointeur structure maillage     */
 ecs_dim_t         dim_e        ,      /* --> Dimension spatiale              */
 ecs_entmail_t  ** vect_entmail ,      /* --> Tableau des entits de maillage */
 ecs_connect_t  ** vect_connect        /* --> Tableau des connectivits       */
)
{
  ecs_maillage_t * maillage ;

  ecs_int_t      iconnect   ; /* Indice de boucle sur les connectivits       */
  ecs_int_t      ient       ; /* Indice de boucle sur les entits de maillage */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Structure globale du maillage */
  /*-------------------------------*/

  /* Allocation de la structure globale du maillage */

  BFT_MALLOC(maillage, 1, ecs_maillage_t);

  /* Initialisation de la dimension d'espace */

  maillage->dim_e = dim_e ;


  /* Connectivits */
  /*---------------*/

  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    maillage->connect[iconnect] = vect_connect[iconnect] ;

  }


  /* Entits de maillage */
  /*---------------------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    maillage->entmail[ient] = vect_entmail[ient] ;

  }


  /* Familles */
  /*----------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    maillage->famille[ient] = NULL ;

  }


  return maillage ;


}


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_maillage_t' donne en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__detruit
(
 ecs_maillage_t * this_maillage
)
{

  ecs_int_t      iconnect    ; /* Indice de boucle sur les connectivits      */
  ecs_int_t      ient        ; /* Indice de boucle sur les entits de maillage*/


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Structures des connectivits */
  /*==============================*/

  /* Libration du contenu des connectivits */
  /*-----------------------------------------*/

  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    if (this_maillage->connect[iconnect] != NULL) {

      this_maillage->connect[iconnect]
        = ecs_connect__detruit(this_maillage->connect[iconnect]) ;

    }

  }


  /* Structures des entits de maillage */
  /*====================================*/

  /* Libration du contenu des entits */
  /*-----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      this_maillage->entmail[ient]
        = ecs_entmail__detruit(this_maillage->entmail[ient]) ;

    }

  }


  /* Liste chane des familles */
  /*============================*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (this_maillage->famille[ient] != NULL) {

      ecs_famille_chaine__detruit(&this_maillage->famille[ient]) ;

    }

  }

  /* Structure globale du maillage */
  /*==============================*/

  BFT_FREE(this_maillage);


  return this_maillage ;


}


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_maillage_t' donne
 *   dans le fichier de nom donn
 *   prcd par le `titre'
 *----------------------------------------------------------------------------*/

void ecs_maillage__imprime
(
 const ecs_maillage_t  *const this_maillage    ,
 const char            *const nom_fichier_dump ,
 const ecs_int_t              nbr_imp          ,
 const char            *const titre
)
{

  const char *nom_connect_c[ECS_CONNECT_FIN] = {
    "CONNECT_ARE_SOM"     ,
    "CONNECT_FAC_SOM"     ,
    "CONNECT_FAC_ARE"     ,
    "CONNECT_CEL_SOM"     ,
    "CONNECT_CEL_FAC"
  } ;


  const char *nom_ent_c[ECS_ENTMAIL_FIN] = {
    "ENTMAIL_SOM" ,
    "ENTMAIL_ARE" ,
    "ENTMAIL_FAC" ,
    "ENTMAIL_CEL"
  } ;


  const char *nom_fam_c[ECS_ENTMAIL_FIN] = {
    "FAMILLE_SOM" ,
    "FAMILLE_ARE" ,
    "FAMILLE_FAC" ,
    "FAMILLE_CEL"
  } ;


  bft_file_t  *fic_imp       ; /* Descripteur du fichier d'impression         */

  ecs_int_t    imp_col       ;

  ecs_int_t    iconnect      ; /* Indice de boucle sur les connectivits      */
  ecs_int_t    ient          ; /* Indice de boucle sur les entits de maillage*/


#define ECS_FCT_IMP_MAILLAGE_DIM_SP        "dim_e"
#define ECS_FCT_IMP_MAILLAGE_FAMILLE       "famille"


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_maillage != NULL) ;


  imp_col = 0 ;


  /* Ouverture du fichier d'impression */
  /*===================================*/

  fic_imp = bft_file_open(nom_fichier_dump,
                          BFT_FILE_MODE_APPEND, BFT_FILE_TYPE_TEXT) ;


  /* Message sur la sortie standard */
  /*================================*/

  bft_printf("\n\n%s %s\n%s\n\n",
             _("Dump mesh structure to file"),
             nom_fichier_dump, titre) ;


  /* criture du titre */
  /*===================*/

  bft_file_printf(fic_imp, "\n\n%s\n\n", titre ) ;


  /* Impression des membres scalaires de la structure */
  /*==================================================*/

  ecs_fic__imprime_val(fic_imp, imp_col, ECS_FCT_IMP_MAILLAGE_DIM_SP,
                       ECS_TYPE_ecs_int_t, &this_maillage->dim_e) ;


  /* Impression des connectivits */
  /*==============================*/

  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    /* Impression du pointeur sur une connectivit */

    ecs_fic__imprime_ptr(fic_imp, imp_col,
                         nom_connect_c[iconnect],
                         (void *)this_maillage->connect[iconnect]) ;


    if (this_maillage->connect[iconnect] != NULL) {

      /* Impression du contenu d'une connectivit */

      ecs_connect__imprime(this_maillage->connect[iconnect],
                           imp_col,
                           fic_imp) ;

    }

  }


  /* Impressions des entits de maillage */
  /*=====================================*/


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    /* Impression du pointeur sur une entit principale */

    ecs_fic__imprime_ptr(fic_imp, imp_col,
                         nom_ent_c[ient],
                         (void *)this_maillage->entmail[ient]) ;


    if (this_maillage->entmail[ient] != NULL) {

      /* Impression du contenu d'une entit principale */

      ecs_entmail__imprime(this_maillage->entmail[ient],
                           imp_col + 1,
                           nbr_imp,
                           fic_imp) ;

    }

  }


  /* Impression des familles */
  /*=========================*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    /* Impression du pointeur sur une entit principale */

    ecs_fic__imprime_ptr(fic_imp, imp_col,
                         nom_fam_c[ient],
                         (void *)this_maillage->famille[ient]) ;


    if (this_maillage->famille[ient] != NULL) {

      ecs_famille_chaine__imprime(this_maillage->famille[ient],
                                  imp_col + 1,
                                  fic_imp) ;

    }

  }


  /* Fermeture du fichier d'impression */
  /*===================================*/

  bft_file_free(fic_imp) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le type d'entit de plus grande dimension
 *   contenue dans une structure `ecs_maillage_t'
 *----------------------------------------------------------------------------*/


ECS_ENTMAIL_E ecs_maillage__ret_entmail_max
(
 const ecs_maillage_t * const this_maillage
)
{
  ECS_ENTMAIL_E entmail_max = ECS_ENTMAIL_DEB ;
  ecs_int_t ient ;

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {
    if (this_maillage->entmail[ient] != NULL) {
      if (ecs_entmail__ret_nbr_ele(this_maillage->entmail[ient]) > 0)
        entmail_max = (ECS_ENTMAIL_E) ient ;
    }
  }

  return entmail_max ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_maillage_t'.
 *----------------------------------------------------------------------------*/

float ecs_maillage__ret_taille
(
 const ecs_maillage_t *const this_maillage
)
{

  float        taille        ;

  ecs_int_t    iconnect      ; /* Indice de boucle sur les connectivits      */
  ecs_int_t    ient          ; /* Indice de boucle sur les entits de maillage*/


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(this_maillage != NULL) ;


  taille = sizeof(*this_maillage) ;


  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    if (this_maillage->connect[iconnect] != NULL) {

      taille += ecs_connect__ret_taille(this_maillage->connect[iconnect]) ;

    }

  }


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      taille += ecs_entmail__ret_taille(this_maillage->entmail[ient]) ;

    }

  }


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (this_maillage->famille[ient] != NULL) {

      taille += ecs_famille_chaine__ret_taille(this_maillage->famille[ient]) ;

    }

  }


  return taille ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui cr une structure maillage en connectivit nodale
 *   (les connectivits tant lies par les tiquettes)
 *    partir du vecteur des entits de maillage
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__cree_nodal
(
 ecs_dim_t              dim_e        ,
 ecs_entmail_t * *const vect_entmail
)
{

  ecs_connect_t  * * vect_connect ;  /* Tableau des connectivits             */

  ecs_maillage_t *   maillage     ;  /* Structure de maillage                 */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Cration des connectivits en fonction des entits de maillages lues */
  /*======================================================================*/

  vect_connect = ecs_connect__cree_nodal(vect_entmail,
                                         ECS_CONNECT_NAT_LABEL) ;


  /* Cration de la structure du maillage */
  /*======================================*/

  maillage = ecs_maillage__cree(dim_e,
                                vect_entmail,
                                vect_connect ) ;


  BFT_FREE(vect_connect) ;


  /* Transformation des rfrences  des tiquettes d'entit de maillage */
  /*  en rfrences par rapport aux indices de l'entit de maillage      */
  /*=====================================================================*/

  ecs_maillage__cree_indice(maillage) ;


  /* Suppression des sommets ne participant pas  la connectivit */
  /* et fusion des lments surfaciques confondus ventuels       */
  /*==============================================================*/

  ecs_maillage__nettoie_nodal(maillage) ;


  return maillage ;

}


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus et suppression des lments dgnrs
 *----------------------------------------------------------------------------*/

void ecs_maillage__nettoie_descend
(
 ecs_maillage_t  *this_maillage
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Suppression des sommets inutiles */

  ecs_entmail_pcp__nettoie_descend(this_maillage->entmail) ;


}


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas  la connectivit
 *   et fusion des lments surfaciques confondus ventuels
 *----------------------------------------------------------------------------*/

void ecs_maillage__nettoie_nodal
(
 ecs_maillage_t  *this_maillage
)
{

  ecs_entmail_t * liste_entmail_connect[1] ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Suppression des sommets inutiles */

  ecs_entmail_pcp__nettoie_nodal(this_maillage->entmail) ;


  /*
    Fusion d'lments surfaciques confondus ventuels (issus par exemple,
    de l'utilisation conjointe de couleurs de faces et faces de bord
    sous Simail, double `surface coating' sous I-DEAS, ...)
  */

  liste_entmail_connect[0] = NULL ;

  if (this_maillage->entmail[ECS_ENTMAIL_CEL] != NULL) {

    if (this_maillage->entmail[ECS_ENTMAIL_FAC] != NULL)
      ecs_entmail_pcp__fusionne(this_maillage->entmail[ECS_ENTMAIL_FAC],
                                liste_entmail_connect) ;

  }
  else if (this_maillage->entmail[ECS_ENTMAIL_FAC] != NULL) {

    if (this_maillage->entmail[ECS_ENTMAIL_ARE] != NULL)
      ecs_entmail_pcp__fusionne(this_maillage->entmail[ECS_ENTMAIL_ARE],
                                liste_entmail_connect) ;

  }

}


/*----------------------------------------------------------------------------
 *  Correction si ncessaire de l'orientation des lments en
 *   connectivit nodale.
 *----------------------------------------------------------------------------*/

void ecs_maillage__orient_nodal
(
       ecs_maillage_t  *const this_maillage ,  /*  -> Dfinition du maillage  */
       ecs_tab_int_t   *const liste_cel_err ,  /* <-  Liste cellules avec
                                                *     erreur (optionnelle)    */
       ecs_tab_int_t   *const liste_cel_cor ,  /* <-  Liste cellules cor-
                                                *     riges (optionnelle)    */
 const ecs_bool_t             correc_orient    /*  -> Correction ou non       */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  ecs_entmail_pcp__orient_nodal(this_maillage->entmail,
                                liste_cel_err,
                                liste_cel_cor,
                                correc_orient) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui assigne la tte de la liste chane des familles donne
 *    la structure de maillage donne
 *----------------------------------------------------------------------------*/

void ecs_maillage__definit_famille
(
 ecs_maillage_t   *const maillage     ,
 ecs_famille_t  * *const vect_famille
)
{

  ecs_int_t ifam_ent ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  for (ifam_ent = ECS_FAMILLE_DEB ; ifam_ent < ECS_FAMILLE_FIN ; ifam_ent++) {

    maillage->famille[ifam_ent] = vect_famille[ifam_ent] ;

  }


}


/*----------------------------------------------------------------------------
 *  Fonction ralisant, pour tout le maillage,
 *   la transformation des rfrences  des tiquettes d'entit de maillage
 *   en rfrences par rapport aux indices de l'entit de maillage
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_indice
(
 ecs_maillage_t  *this_maillage
)
{

  ecs_int_t    iconnect ;  /* Indice de boucle sur les connectivits          */
  ecs_int_t    ient     ;  /* Indice de boucle sur les entits                */


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  for (iconnect = ECS_CONNECT_DEB ;
       iconnect < ECS_CONNECT_FIN ;
       iconnect++              ) {

    if (this_maillage->connect[iconnect] != NULL) {

      /* Appel  la fonction ralisant la transformation sur une connectivit */

      ecs_connect__cree_indice(this_maillage->connect[iconnect]) ;

    }

  } /* Fin : boucle sur les connectivits */


  for (ient = ECS_ENTMAIL_SOM ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      ecs_entmail_pcp__detruit_att_nom(this_maillage->entmail[ient],
                                       ECS_CHAMP_NOM_LABEL) ;

    }

  }

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant,  partir d'une connectivit de maillage donne,
 *   la connectivit descendante du maillage
 *----------------------------------------------------------------------------*/

void ecs_maillage__connect_descend
(
 ecs_maillage_t * maillage   /* <-> Maillage pour la connectivit descendante */
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  if (maillage->connect[ECS_CONNECT_CEL_SOM] != NULL) {

    /* On est en ECS_DIM_3 */

    /* ECS_CONNECT_CEL_SOM ==> ECS_CONNECT_CEL_FAC */

    maillage->connect[ECS_CONNECT_CEL_FAC] =
      ecs_connect__cree_sous_connect(maillage->connect[ECS_CONNECT_CEL_SOM],
                                     &maillage->connect[ECS_CONNECT_FAC_SOM],
                                     &maillage->entmail[ECS_ENTMAIL_FAC],
                                     ECS_CONNECT_CEL_SOM) ;

    maillage->connect[ECS_CONNECT_CEL_SOM]
      = ecs_connect__detruit(maillage->connect[ECS_CONNECT_CEL_SOM]) ;

  }


  /* ECS_CONNECT_FAC_SOM ==> ECS_CONNECT_FAC_ARE */

  if (maillage->connect[ECS_CONNECT_FAC_SOM] == NULL)
    bft_error(__FILE__, __LINE__, 0,
              _("The mesh does not seem to contain a nodal connectivity.")) ;

  maillage->connect[ECS_CONNECT_FAC_ARE] =
   ecs_connect__cree_sous_connect(maillage->connect[ECS_CONNECT_FAC_SOM],
                                  &maillage->connect[ECS_CONNECT_ARE_SOM],
                                  &maillage->entmail[ECS_ENTMAIL_ARE],
                                  ECS_CONNECT_FAC_SOM) ;


  maillage->connect[ECS_CONNECT_FAC_SOM]
    = ecs_connect__detruit(maillage->connect[ECS_CONNECT_FAC_SOM]) ;

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant le tri des lments suivant leur type gomtrique
 *  La fonction affiche le nombre d'lments par type gomtrique
 *----------------------------------------------------------------------------*/

void ecs_maillage__trie_typ_geo
(
 ecs_maillage_t * maillage     /* <-> Maillage contenant les lments  trier */
)
{

  ecs_int_t      ient ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  bft_printf("\n") ;


  for (ient = ECS_ENTMAIL_FAC ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (maillage->entmail[ient] != NULL) {

      ecs_entmail_pcp__trie_typ_geo(maillage->entmail[ient],
                                    ient) ;

    }

  }

  bft_printf("\n") ;

}


/*----------------------------------------------------------------------------
 *  Fonction ralisant le dcoupage des faces polygonales en triangles
 *
 *  Le maillage doit tre en connectivit nodale (faces -> sommets)
 *----------------------------------------------------------------------------*/

void ecs_maillage__dec_poly_tria
(
 ecs_maillage_t  *const maillage
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;

  ecs_entmail_pcp__dec_poly_tria(maillage->entmail) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui dfinit un nouveau maillage
 *   par extraction d'une partie du maillage donn
 *
 *  La partie extraite correspond aux lments :
 *  -       ayant        une des couleurs donnes dans la liste des couleurs
 *  - et/ou appartenant  un des groupes  donns  dans la liste des groupes
 *
 *  On peut aussi choisir d'inverser la slection
 *
 *  Les lments  extraire doivent tre tous de mme dimension :
 *  cellules ou faces ou artes ou sommets
 *
 *  On construit automatiquement une filiation.
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__extrait
(
       ecs_maillage_t       *const maillage         ,
       ECS_ENTMAIL_E               entmail_sel_e    ,
 const ecs_tab_int_t        *const liste_filtre     ,
 const ecs_tab_int_t               liste_couleur    ,
 const ecs_tab_char_t              liste_groupe     ,
 const ecs_bool_t                  inv_selection    ,
 const ecs_bool_t                  herite_attributs
)
{

  ecs_tab_bool_t *     bool_elt_select   ;

  ecs_int_t            iconnect          ;
  ecs_int_t            ient              ;

  ecs_connect_t  *     vect_connect[ECS_CONNECT_FIN] ; /* Table connectivits */
  ecs_entmail_t  * *   vect_entmail      ;

  ecs_maillage_t *     this_maillage_new ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  if (liste_couleur.nbr > 0 || liste_groupe.nbr > 0)

    bft_printf(_("  Extraction of elements selected\n"
                 "   by their color or group membership\n"
                 "  -----------------------------------\n\n")) ;


  /*--------------------------------------------------------------------------*/
  /* 1re tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Construction de la liste des lments du maillage d'origine  extraire   */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/


  bool_elt_select = ecs_entmail_pcp__selectionne(maillage->entmail,
                                                 entmail_sel_e,
                                                 liste_filtre,
                                                 liste_couleur,
                                                 liste_groupe,
                                                 inv_selection) ;


  /*--------------------------------------------------------------------------*/
  /* 2nde tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Extraction des lments slectionns du maillage d'origine               */
  /*  pour l'ensemble des entits de maillage                                 */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/


  vect_entmail = ecs_entmail_pcp__extrait(maillage->entmail,
                                          entmail_sel_e,
                                          bool_elt_select,
                                          herite_attributs) ;


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (bool_elt_select[ient].val != NULL)
      BFT_FREE(bool_elt_select[ient].val) ;

  }

  BFT_FREE(bool_elt_select) ;



  /*--------------------------------------------------------------------------*/
  /* 3me tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Cration des connectivits en fonction des entits de maillage extraites */
  /*  et cration de la structure "maillage"                                  */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/


  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++)
    vect_connect[iconnect] = NULL ;

  if (vect_entmail[ECS_ENTMAIL_CEL] != NULL) {

    vect_connect[ECS_CONNECT_CEL_FAC]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_CEL] ,
                          vect_entmail[ECS_ENTMAIL_FAC] ,
                          ECS_CONNECT_NAT_INDEX  ) ;
  }

  if (vect_entmail[ECS_ENTMAIL_FAC] != NULL) {

    vect_connect[ECS_CONNECT_FAC_ARE]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_FAC] ,
                          vect_entmail[ECS_ENTMAIL_ARE] ,
                          ECS_CONNECT_NAT_INDEX  ) ;

  }

  if (vect_entmail[ECS_ENTMAIL_ARE] != NULL) {

    vect_connect[ECS_CONNECT_ARE_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_ARE] ,
                          vect_entmail[ECS_ENTMAIL_SOM] ,
                          ECS_CONNECT_NAT_INDEX  ) ;
  }


  this_maillage_new = ecs_maillage__cree(maillage->dim_e,
                                         vect_entmail,
                                         vect_connect ) ;


  BFT_FREE(vect_entmail) ;


  /*--------------------------------------------------------------------------*/
  /* 4me tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Recopie des descripteurs de familles                                     */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/

  if (herite_attributs == ECS_TRUE) {

    for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

      if (   this_maillage_new->entmail[ient] != NULL
          && maillage->famille[ient] != NULL) {

        this_maillage_new->famille[ient]
          = ecs_famille_chaine__copie(maillage->famille[ient]) ;
      }

    }

  }


  return this_maillage_new ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui dfinit un nouveau maillage nodal
 *   par extraction d'une partie du maillage nodal donn
 *
 *  La partie extraite correspond aux lments :
 *  -       ayant        une des couleurs donnes dans la liste des couleurs
 *  - et/ou appartenant  un des groupes  donns  dans la liste des groupes
 *
 *  On peut aussi choisir d'inverser la slection
 *
 *  Les lments  extraire doivent tre tous de mme dimension :
 *  cellules ou faces ou artes ou sommets
 *----------------------------------------------------------------------------*/

ecs_maillage_t * ecs_maillage__extrait_nodal
(
       ecs_maillage_t       *const maillage      ,
       ECS_ENTMAIL_E               entmail_sel_e ,
 const ecs_tab_int_t        *const liste_filtre  ,
 const ecs_tab_int_t               liste_couleur ,
 const ecs_tab_char_t              liste_groupe  ,
 const ecs_bool_t                  inv_selection
)
{

  ecs_tab_bool_t *     bool_elt_select   ;

  ecs_int_t            iconnect          ;
  ecs_int_t            ient              ;

  ecs_connect_t  *     vect_connect[ECS_CONNECT_FIN] ; /* Table connectivits */
  ecs_entmail_t  * *   vect_entmail      ;

  ecs_maillage_t *     this_maillage_new ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  if (liste_couleur.nbr > 0 || liste_groupe.nbr > 0)

    bft_printf(_("  Extraction of elements selected\n"
                 "   by their color or group membership\n"
                 "  -----------------------------------\n\n")) ;


  /*--------------------------------------------------------------------------*/
  /* Construction de la liste des lments du maillage d'origine  extraire   */
  /*--------------------------------------------------------------------------*/

  bool_elt_select = ecs_entmail_pcp__selectionne(maillage->entmail,
                                                 entmail_sel_e,
                                                 liste_filtre,
                                                 liste_couleur,
                                                 liste_groupe,
                                                 inv_selection) ;


  /*--------------------------------------------------------------------------*/
  /* Extraction des lments slectionns du maillage d'origine               */
  /*  et des sommets associs                                                 */
  /*--------------------------------------------------------------------------*/

  vect_entmail = ecs_entmail_pcp__extrait_nodal(maillage->entmail,
                                                entmail_sel_e,
                                                bool_elt_select) ;


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (bool_elt_select[ient].val != NULL)
      BFT_FREE(bool_elt_select[ient].val) ;

  }

  BFT_FREE(bool_elt_select) ;


  /*--------------------------------------------------------------------------*/
  /* Cration des connectivits en fonction des entits de maillage extraites */
  /*  et cration de la structure "maillage"                                  */
  /*--------------------------------------------------------------------------*/

  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++)
    vect_connect[iconnect] = NULL ;

  if (vect_entmail[ECS_ENTMAIL_CEL] != NULL) {

    vect_connect[ECS_CONNECT_CEL_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_CEL] ,
                          vect_entmail[ECS_ENTMAIL_SOM] ,
                          ECS_CONNECT_NAT_INDEX) ;
  }

  if (vect_entmail[ECS_ENTMAIL_FAC] != NULL) {

    vect_connect[ECS_CONNECT_FAC_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_FAC] ,
                          vect_entmail[ECS_ENTMAIL_SOM] ,
                          ECS_CONNECT_NAT_INDEX) ;

  }

  if (vect_entmail[ECS_ENTMAIL_ARE] != NULL) {

    vect_connect[ECS_CONNECT_ARE_SOM]
      = ecs_connect__cree(vect_entmail[ECS_ENTMAIL_ARE] ,
                          vect_entmail[ECS_ENTMAIL_SOM] ,
                          ECS_CONNECT_NAT_INDEX) ;
  }


  this_maillage_new = ecs_maillage__cree(maillage->dim_e,
                                         vect_entmail,
                                         vect_connect ) ;

  BFT_FREE(vect_entmail) ;

  return this_maillage_new ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie une liste d'entits portant des familles
 *   non vides dans un maillage donn
 *
 *  La partie slectionne correspond aux lments :
 *  -       ayant        une des couleurs donnes dans la liste des couleurs
 *  - et/ou appartenant  un des groupes  donns  dans la liste des groupes
 *
 *  La slection peut tre inverse (en restant dans les entits de
 *  famille non vides)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t  ecs_maillage__selectionne_fam
(
       ecs_maillage_t       *const maillage      ,
       ECS_ENTMAIL_E               entmail_sel_e ,
 const ecs_tab_int_t        *const liste_filtre  ,
 const ecs_tab_int_t               liste_couleur ,
 const ecs_tab_char_t              liste_groupe  ,
 const ecs_bool_t                  inv_selection
)
{

  ecs_tab_bool_t       bool_fam_select ;

  size_t               ind ;

  ecs_tab_int_t        liste_elt_select ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  if (liste_couleur.nbr > 0 || liste_groupe.nbr > 0)

    bft_printf(_("  Selection of elements\n"
                 "   by their color or group membership\n"
                 "  -----------------------------------\n\n")) ;


  liste_elt_select.nbr = 0 ;
  liste_elt_select.val = NULL ;


  /* Construction de la liste des familles  slectionner */
  /*------------------------------------------------------*/

  if (maillage->famille[entmail_sel_e] == NULL)
    return liste_elt_select ;


  bool_fam_select
    = ecs_famille_chaine__indic_fam_att(maillage->famille[entmail_sel_e],
                                        liste_couleur,
                                        liste_groupe) ;

  if (bool_fam_select.nbr < 2) {
    BFT_FREE(bool_fam_select.val) ;
    return liste_elt_select ;
  }

  if (liste_couleur.nbr == 0 && liste_groupe.nbr == 0) {
    for (ind = 1 ; ind < bool_fam_select.nbr ; ind++)
      bool_fam_select.val[ind] = ECS_TRUE ;
  }

  bool_fam_select.val[0] = ECS_FALSE ;

  if (inv_selection == ECS_TRUE) {

    for (ind = 1 ; ind < bool_fam_select.nbr ; ind++) {
      bool_fam_select.val[ind]
        = (bool_fam_select.val[ind] == ECS_TRUE) ? ECS_FALSE : ECS_TRUE ;
    }

  }


  /* Construction de la liste des lments du maillage  slectionner */
  /*------------------------------------------------------------------*/

  liste_elt_select = ecs_entmail_pcp__liste_ent_fam(maillage->entmail,
                                                    entmail_sel_e,
                                                    liste_filtre,
                                                    &bool_fam_select) ;


  bft_printf(_("  Number of selected elements: %d\n"),
             liste_elt_select.nbr) ;


  /* Libration mmoire et retour */
  /*------------------------------*/

  bool_fam_select.nbr = 0 ;
  BFT_FREE(bool_fam_select.val) ;

  return liste_elt_select ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui concatne dans un maillage rcepteur donn,
 *   un maillage  concatner donn
 *----------------------------------------------------------------------------*/

void ecs_maillage__concatene
(
 ecs_maillage_t *const maillage_recept ,
 ecs_maillage_t *const maillage_concat
)
{

  ecs_int_t             iconnect      ;
  ecs_int_t             ient          ;
  ecs_int_t             ient_connect  ;
  ecs_int_t             iliste        ;

  ECS_CONNECT_TYP_E     connect_typ_e ;
  ECS_CONNECT_TYP_E     connect_typ_concat_e ;

  ecs_entmail_t       * liste_entmail_connect[2 * (ECS_ENTMAIL_FIN - 1)] ;
  ecs_connect_t     * * vect_connect ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage_recept != NULL) ;
  assert(maillage_concat != NULL) ;


  connect_typ_e = ecs_connect__ret_connect_typ(maillage_recept->connect,
                                               maillage_recept->entmail ) ;

  connect_typ_concat_e
    = ecs_connect__ret_connect_typ(maillage_concat->connect,
                                   maillage_concat->entmail ) ;

  assert(connect_typ_e        == connect_typ_concat_e ||
         connect_typ_e        == ECS_CONNECT_TYP_NUL      ||
         connect_typ_concat_e == ECS_CONNECT_TYP_NUL        ) ;


  /*--------------------------------------------------------------------------*/
  /* 1re tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Concatnation et fusion des lments provenant des diffrentes           */
  /*  entits de maillage extraites                                           */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {


    if (maillage_concat->entmail[ient] != NULL) {


      if (maillage_recept->entmail[ient] != NULL) {

        /* - en connectivit descendante,                                   */
        /*      l'entit de maillage `ient' ne participe qu'a la dfinition */
        /*   de l'entit de maillage `ient + 1'                             */
        /* - en connectivit nodale,                                        */
        /*   seule l'entit de maillage sur les sommets participe           */
        /*     la dfinition de toutes les autres entits de maillage      */


        for (ient_connect = ECS_ENTMAIL_DEB ;
             ient_connect < 2 * (ECS_ENTMAIL_FIN - 1) ; ient_connect++)

          liste_entmail_connect[ient_connect] = NULL ;


        iliste = 0 ;

        if ((ECS_ENTMAIL_E)ient != ECS_ENTMAIL_CEL) {

          switch(connect_typ_e) {

          case ECS_CONNECT_TYP_NODALE:

            if ((ECS_ENTMAIL_E)ient == ECS_ENTMAIL_SOM) {

              for (ient_connect = ECS_ENTMAIL_ARE ;
                   ient_connect < ECS_ENTMAIL_FIN ; ient_connect++ ) {

                if (maillage_concat->entmail[ient_connect] != NULL) {

                  liste_entmail_connect[iliste++]
                    = maillage_concat->entmail[ient_connect] ;

                }

              }

            }

            break ;

          case ECS_CONNECT_TYP_DESCENDANTE:
          case ECS_CONNECT_TYP_NUL:

            if (maillage_concat->entmail[ient + 1] != NULL) {

              liste_entmail_connect[iliste++]
                =  maillage_concat->entmail[ient + 1] ;

            }

            break ;

          }

        } /* Fin : si l'entit de maillage ne concerne pas les cellules */


        ecs_entmail_pcp__concatene(maillage_recept->entmail[ient],
                                   maillage_concat->entmail[ient],
                                   liste_entmail_connect) ;

        maillage_concat->entmail[ient]
          = ecs_entmail__detruit(maillage_concat->entmail[ient]) ;

      }
      else {

        maillage_recept->entmail[ient]
          = maillage_concat->entmail[ient] ;

        maillage_concat->entmail[ient] = NULL ;

      }


    }  /* else : rien  faire */


  }  /* Fin : boucle sur `ient' */


  ecs_maillage__detruit(maillage_concat) ;


  /*--------------------------------------------------------------------------*/
  /* 2nde tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Cration des connectivits en fonction des entits de maillage extraites */
  /*  et cration de la structure "maillage"                                  */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/

  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    if (maillage_recept->connect[iconnect] != NULL)
      ecs_connect__detruit(maillage_recept->connect[iconnect]) ;

  }


  switch(connect_typ_e) {

  case ECS_CONNECT_TYP_NODALE:

    vect_connect = ecs_connect__cree_nodal(maillage_recept->entmail,
                                           ECS_CONNECT_NAT_INDEX) ;

    break ;

  case ECS_CONNECT_TYP_DESCENDANTE:
  case ECS_CONNECT_TYP_NUL:

    vect_connect = ecs_connect__cree_descendant(maillage_recept->entmail) ;

    break ;

  }


  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    maillage_recept->connect[iconnect] = vect_connect[iconnect] ;

  }


  BFT_FREE(vect_connect) ;


}


/*----------------------------------------------------------------------------
 *  Cration et initialisation d'un maillage extrait correspondant  un cas de
 *  coupe de faces : on extrait un maillage correspondant  la zone
 *  slectionne par liste, puis on dcoupe ventuellement les faces de plus
 *  de 4 artes de ce maillage en triangles ; finalement, on l'ordonne
 *  selon le numro de type de face.
 *
 *  On remplit un tableau de pointeurs sur les structures maillage associes
 *   chaque niveau de dcoupage de la coupe.
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_coupe
(
 const char              *const   nom_coupe                            ,
       ecs_maillage_t    *const   maillage                             ,
 const ecs_tab_int_t              liste_fac                            ,
       ecs_maillage_t    *        maillage_coupe[ECS_POST_FAC_MAX_FIN] ,
 const ecs_int_t                  nbr_dump                             ,
 const char              *const   nom_fic_dump                         ,
 const ecs_bool_t                 herite_attributs                     ,
       ecs_post_type_t            type_post                            ,
       ecs_post_t        *const   cas_post
)
{

  ecs_maillage_t  * maillage_extrait ;

  char            * nom_coupe_loc ;

  ecs_int_t         ind_decoup ;

  size_t            nbr_fac_poly ;

  ecs_bool_t        bool_post[ECS_POST_FAC_MAX_FIN] ;

  ecs_tab_int_t     liste_couleur ;
  ecs_tab_char_t    liste_groupe  ;

  ecs_post_t        cas_post_tmp ;

  static char nom_aretes[] = N_(" edges") ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(maillage != NULL) ;

  /* Initialisations */
  /*-----------------*/

  liste_couleur.nbr = 0 ;
  liste_couleur.val = NULL ;

  liste_groupe.nbr = 0 ;
  liste_groupe.val = NULL ;

  for (ind_decoup = 0 ; ind_decoup < ECS_POST_FAC_MAX_FIN ; ind_decoup++) {
    maillage_coupe[ind_decoup] = NULL ;
    bool_post[ind_decoup] = ECS_FALSE ;
  }

  nbr_fac_poly = 0 ;


  /*
    On dtermine si l'on utilise un maillage de post-traitement
    autorisant les faces de plus de 4 artes, un maillage
    autorisant uniquement les triangles et quadrangles, ou bien
    les deux (si plusieurs formats de sortie simultans).
  */

  if (cas_post != NULL) {

    if (cas_post->post_ens == ECS_TRUE) {
      if (cas_post->opt_ens.dec_poly == ECS_FALSE)
        bool_post[ECS_POST_FAC_MAX_POLY] = ECS_TRUE ;
      else
        bool_post[ECS_POST_FAC_MAX_QUAD] = ECS_TRUE ;
    }

#if defined(HAVE_CGNS)
    if (cas_post->post_cgns == ECS_TRUE) {
      if (cas_post->opt_cgns.dec_poly == ECS_FALSE)
        bool_post[ECS_POST_FAC_MAX_POLY] = ECS_TRUE ;
      else
        bool_post[ECS_POST_FAC_MAX_QUAD] = ECS_TRUE ;
    }
#endif

#if defined(HAVE_MED)
    if (cas_post->post_med == ECS_TRUE) {
      if (cas_post->opt_med.dec_poly == ECS_FALSE)
        bool_post[ECS_POST_FAC_MAX_POLY] = ECS_TRUE ;
      else
        bool_post[ECS_POST_FAC_MAX_QUAD] = ECS_TRUE ;
    }
#endif

  }

  /* Si aucun post-traitement, rien de plus  faire */

  else

    return ;


  /* Extraction des faces slectionnes */
  /*------------------------------------*/

  maillage_extrait = ecs_maillage__extrait(maillage,
                                           ECS_ENTMAIL_FAC,
                                           &liste_fac,
                                           liste_couleur,
                                           liste_groupe,
                                           ECS_FALSE,
                                           herite_attributs) ;

  maillage_extrait->dim_e = ECS_DIM_3 ;


  if (nbr_dump > 0)
    ecs_maillage__imprime(maillage_extrait,
                          nom_fic_dump,
                          nbr_dump,
                          nom_coupe) ;


  /* Sera-il ncessaire de dcouper les faces de plus de 4 artes ? */
  /*----------------------------------------------------------------*/

  if (bool_post[ECS_POST_FAC_MAX_QUAD] == ECS_TRUE)
    nbr_fac_poly = ecs_entmail_pcp__nbr_fac_poly
                     (maillage_extrait->entmail[ECS_ENTMAIL_FAC]) ;

  if (   bool_post[ECS_POST_FAC_MAX_QUAD] == ECS_TRUE
      && nbr_fac_poly == 0) {
    bool_post[ECS_POST_FAC_MAX_POLY] = ECS_TRUE ;
    bool_post[ECS_POST_FAC_MAX_QUAD] = ECS_FALSE ;
  }


  /* Cas du maillage non dcoup */
  /*-----------------------------*/

  if (bool_post[ECS_POST_FAC_MAX_POLY] == ECS_TRUE) {

    /*
      Cas local pour post-traitement, selon si dcoupage de polygones
      activ ou non ncessaire.
    */

    cas_post_tmp = *cas_post ;

    if (nbr_fac_poly > 0) {

      if (   cas_post_tmp.post_ens == ECS_TRUE
          && cas_post_tmp.opt_ens.dec_poly == ECS_TRUE) {
        cas_post_tmp.post_ens = ECS_FALSE ;
        cas_post_tmp.cas_ens  = NULL ;
      }

#if defined(HAVE_CGNS)
      if (   cas_post_tmp.post_cgns == ECS_TRUE
          && cas_post_tmp.opt_cgns.dec_poly == ECS_TRUE) {
        cas_post_tmp.post_cgns = ECS_FALSE ;
        cas_post_tmp.cas_cgns  = NULL ;
      }
#endif

#if defined(HAVE_MED)
      if (   cas_post_tmp.post_med == ECS_TRUE
          && cas_post_tmp.opt_med.dec_poly == ECS_TRUE) {
        cas_post_tmp.post_med = ECS_FALSE ;
        cas_post_tmp.cas_med  = NULL ;
      }
#endif

    }


    /* Passage des faces en connectivit nodale */

    ecs_maillage__connect_fac_som(maillage_extrait) ;


    /* Tri selon le type des faces */

    bft_printf(_("\nCreating mesh for output: %s"
                 "\n-------------------------\n\n"),
               nom_coupe) ;

    ecs_entmail_pcp__trie_typ_geo(maillage_extrait->entmail[ECS_ENTMAIL_FAC],
                                  ECS_ENTMAIL_FAC) ;


    if (nbr_dump > 0)
      ecs_maillage__imprime(maillage_extrait,
                            nom_fic_dump,
                            nbr_dump,
                            "Extracted nodal mesh") ;


    /* Sortie de la gomtrie pour post-traitement */

    ecs_maillage_post__ecr(nom_coupe,
                           maillage_extrait,
                           ECS_ENTMAIL_FAC,
                           type_post,
                           &cas_post_tmp) ;

    if (cas_post->post_ens == ECS_TRUE && cas_post_tmp.post_ens == ECS_TRUE)
      cas_post->cas_ens = cas_post_tmp.cas_ens ;
#if defined(HAVE_CGNS)
    if (cas_post->post_cgns == ECS_TRUE && cas_post_tmp.post_cgns == ECS_TRUE)
      cas_post->cas_cgns = cas_post_tmp.cas_cgns ;
#endif
#if defined(HAVE_MED)
    if (cas_post->post_med == ECS_TRUE && cas_post_tmp.post_med == ECS_TRUE)
      cas_post->cas_med = cas_post_tmp.cas_med ;
#endif

    /*
      Simplification du maillage extrait si seuls post-traitements
       raliser par la suite (gain mmoire)
    */

    ecs_loc_maillage__epure_coupe(maillage_extrait,
                                  nbr_dump,
                                  nom_fic_dump) ;


    /* Valeur de retour (et mise  NULL de maillage_extrait) */

    maillage_coupe[ECS_POST_FAC_MAX_POLY] = maillage_extrait ;

    maillage_extrait = NULL ;

  }
  else

    maillage_coupe[ECS_POST_FAC_MAX_POLY] = NULL ;



  /* Dcoupage en triangles des faces de plus de quatre artes */
  /*-----------------------------------------------------------*/

  if (bool_post[ECS_POST_FAC_MAX_QUAD] == ECS_TRUE) {

    /* Cas local pour post-traitement : sorties si dcoupage ncessaire. */

    cas_post_tmp = *cas_post ;

    assert (nbr_fac_poly > 0) ;

    if (   cas_post_tmp.post_ens == ECS_TRUE
        && cas_post_tmp.opt_ens.dec_poly == ECS_FALSE) {
      cas_post_tmp.post_ens = ECS_FALSE ;
      cas_post_tmp.cas_ens = NULL ;
    }

#if defined(HAVE_CGNS)
    if (   cas_post_tmp.post_cgns == ECS_TRUE
        && cas_post_tmp.opt_cgns.dec_poly == ECS_FALSE) {
      cas_post_tmp.post_cgns = ECS_FALSE ;
      cas_post_tmp.cas_cgns = NULL ;
    }
#endif

#if defined(HAVE_MED)
    if (   cas_post_tmp.post_med == ECS_TRUE
        && cas_post_tmp.opt_med.dec_poly == ECS_FALSE) {
      cas_post_tmp.post_med = ECS_FALSE ;
      cas_post_tmp.cas_med = NULL ;
    }
#endif


    /*
      Si le maillage non dcoup a dj t utilis (dans quel cas
      le pointeur a t remis  NULL), on extrait une seconde copie
    */

    if (maillage_extrait == NULL)

      maillage_extrait = ecs_maillage__extrait(maillage,
                                               ECS_ENTMAIL_FAC,
                                               &liste_fac,
                                               liste_couleur,
                                               liste_groupe,
                                               ECS_FALSE,
                                               herite_attributs) ;

    /* Sortie des artes pour post-traitement */

    BFT_MALLOC(nom_coupe_loc, strlen(nom_coupe) + strlen(_(nom_aretes)) + 1,
               char) ;

    strcpy(nom_coupe_loc, nom_coupe) ;
    strcat(nom_coupe_loc, _(nom_aretes)) ;

    ecs_maillage_post__ecr(nom_coupe_loc,
                           maillage_extrait,
                           ECS_ENTMAIL_ARE,
                           type_post,
                           &cas_post_tmp) ;

    if (cas_post->post_ens == ECS_TRUE && cas_post_tmp.post_ens == ECS_TRUE)
      cas_post->cas_ens = cas_post_tmp.cas_ens ;
#if defined(HAVE_CGNS)
    if (cas_post->post_cgns == ECS_TRUE && cas_post_tmp.post_cgns == ECS_TRUE)
      cas_post->cas_cgns = cas_post_tmp.cas_cgns ;
#endif
#if defined(HAVE_MED)
    if (cas_post->post_med == ECS_TRUE && cas_post_tmp.post_med == ECS_TRUE)
      cas_post->cas_med = cas_post_tmp.cas_med ;
#endif

    BFT_FREE(nom_coupe_loc) ;

    /* Passage des faces en connectivit nodale */

    ecs_maillage__connect_fac_som(maillage_extrait) ;

    /* Dcoupage de faces polygonales en triangles */

    ecs_maillage__dec_poly_tria(maillage_extrait) ;


    bft_printf(_("\nCreating subdivided mesh for output: %s"
                 "\n------------------------------------\n\n"),
               nom_coupe) ;

    /* Tri selon le type des faces */

    ecs_entmail_pcp__trie_typ_geo(maillage_extrait->entmail[ECS_ENTMAIL_FAC],
                                  ECS_ENTMAIL_FAC) ;


    if (nbr_dump > 0)
      ecs_maillage__imprime(maillage_extrait,
                            nom_fic_dump,
                            nbr_dump,
                            "Subdivided extracted nodal mesh") ;


    /* Sortie de la gomtrie pour post-traitement */

    ecs_maillage_post__ecr(nom_coupe,
                           maillage_extrait,
                           ECS_ENTMAIL_FAC,
                           type_post,
                           &cas_post_tmp) ;

    /*
      Simplification du maillage extrait si seuls post-traitements
       raliser par la suite (gain mmoire)
    */

    ecs_loc_maillage__epure_coupe(maillage_extrait,
                                  nbr_dump,
                                  nom_fic_dump) ;


    /* Valeur de retour (et mise  NULL de maillage_extrait) */

    maillage_coupe[ECS_POST_FAC_MAX_QUAD] = maillage_extrait ;

    maillage_extrait = NULL ;

  }

}


/*----------------------------------------------------------------------------
 *  Destruction des structures correspondant  une coupe
 *----------------------------------------------------------------------------*/

void ecs_maillage__detruit_coupe
(
       ecs_maillage_t    *const maillage                               ,
       ecs_maillage_t    *      maillage_extrait[ECS_POST_FAC_MAX_FIN]
)
{

  ecs_int_t         ind_decoup ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(maillage         != NULL) ;
  assert(maillage_extrait != NULL) ;


  /* Destruction des maillage extraits (dcoups ou non) */

  for (ind_decoup = 0 ; ind_decoup < ECS_POST_FAC_MAX_FIN ; ind_decoup++) {

    if (maillage_extrait[ind_decoup] != NULL)
      maillage_extrait[ind_decoup]
        = ecs_maillage__detruit(maillage_extrait[ind_decoup]) ;

  }

}


/*----------------------------------------------------------------------------
 *  Fonction qui transforme la connectivit descendante de la partie
 *  surfacique d'un maillage en connectivit nodale
 *
 *  La connectivit "aretes -> sommets" est supprime, ainsi que
 *  l'entit de maillage "artes".
 *----------------------------------------------------------------------------*/

void ecs_maillage__connect_fac_som
(
 ecs_maillage_t  *const maillage
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;
  assert(maillage->connect != NULL) ;

  /* On vrifie qu'on a bien un maillage  */
  /*  en connectivit descendante         */

  assert(maillage->connect[ECS_CONNECT_FAC_ARE] != NULL) ;
  assert(maillage->connect[ECS_CONNECT_ARE_SOM] != NULL) ;

  assert(maillage->entmail[ECS_ENTMAIL_FAC] != NULL) ;
  assert(maillage->entmail[ECS_ENTMAIL_ARE] != NULL) ;
  assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL) ;



  /*--------------------------------------------------------------------------*/
  /* 1re tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Transformation de la dfinition des faces  en artes                     */
  /*                en    dfinition des faces  en sommets                    */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/


  ecs_entmail_pcp__def_fac_som(maillage->entmail[ECS_ENTMAIL_FAC],
                               maillage->entmail[ECS_ENTMAIL_ARE] ) ;



  /*--------------------------------------------------------------------------*/
  /* 2nde tape                                                               */
  /*--------------------------------------------------------------------------*/
  /*                                                                          */
  /* Modification des connectivits pour le passage en nodal                  */
  /*                                                                          */
  /*--------------------------------------------------------------------------*/

  assert(maillage->connect[ECS_CONNECT_FAC_ARE] != NULL) ;

  maillage->connect[ECS_CONNECT_FAC_SOM] =
   ecs_connect__cree(maillage->entmail[ECS_ENTMAIL_FAC],
                     maillage->entmail[ECS_ENTMAIL_SOM],
                     ECS_CONNECT_NAT_INDEX);

  maillage->connect[ECS_CONNECT_FAC_ARE]
    = ecs_connect__detruit(maillage->connect[ECS_CONNECT_FAC_ARE]) ;

  maillage->connect[ECS_CONNECT_ARE_SOM]
    = ecs_connect__detruit(maillage->connect[ECS_CONNECT_ARE_SOM]) ;

  maillage->entmail[ECS_ENTMAIL_ARE]
    = ecs_entmail__detruit(maillage->entmail[ECS_ENTMAIL_ARE]) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui cre la liste des faces intrieures  un maillage.
 *  On ne compte pas ici les faces priodiques parmi les faces intrieures,
 *  cette fonction tant destine  filtrer le post-triatment, et les
 *  faces priodiques tant dj affichables par ailleurs.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_maillage__liste_fac_int
(
 ecs_maillage_t  *const maillage
)
{
  return ecs_entmail_pcp__liste_fac_int(maillage->entmail) ;
}


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attaches  une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_maillage__liste_cel_fac
(
       ecs_maillage_t  *const maillage  ,
 const ecs_tab_int_t          liste_fac
)
{

  return ecs_entmail_pcp__liste_cel_fac(maillage->entmail,
                                        liste_fac) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnes min et max du domaine
 *----------------------------------------------------------------------------*/

void ecs_maillage__calc_coo_ext
(
 ecs_maillage_t *const maillage
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  if (maillage->dim_e == ECS_DIM_3) {


    assert(maillage->entmail != NULL) ;

    assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL) ;


    ecs_entmail_pcp__calc_coo_ext(maillage->entmail[ECS_ENTMAIL_SOM]);

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnes du maillage
 *----------------------------------------------------------------------------*/

void ecs_maillage__transf_coo
(
 ecs_maillage_t  *const maillage,
 const double           matrice[3][4]
)
{

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  if (maillage->dim_e == ECS_DIM_3) {


    assert(maillage->entmail != NULL) ;

    assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL) ;


    ecs_entmail_pcp__transf_coo(maillage->entmail[ECS_ENTMAIL_SOM],
                                matrice);

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui recolle les faces non conformes
 *
 *  Les listes des faces nouvelles ou modifies sont construites (et alloues)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entre ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le dcoupage en sous-faces a chou
 *
 *  On renvoie 1 si l'on a appel effectivement le recollement, 0 sinon
 *  (i.e. si la slection de faces  recoller est vide).
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_maillage__recolle
(
       ecs_maillage_t    *const maillage      ,
 const ecs_select_fac_t         select_fac_rc ,
       ecs_tab_int_t     *const liste_fac_new ,
       ecs_tab_int_t     *const liste_fac_mod ,
       ecs_tab_int_t     *const liste_fac_err ,
 const ecs_param_rc_t           param_rc
)
{

  size_t             nbr_fac ;
  ecs_int_t          ient ;
  ecs_tab_bool_t  *  bool_elt_select ;

  ecs_tab_int_t     liste_fac_isolee ;
  ecs_tab_int_t     liste_fac_de_bord ;

  ecs_int_t         val_ret ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;


  /* Message d'information pour l'utilisateur */
  /*------------------------------------------*/

  if (param_rc.param_perio == NULL)

    bft_printf("\n\n%s", _("Joining of non-conforming faces\n"
                           "-------------------------------\n")) ;
  else {

    bft_printf(_("\n\nProcessing periodicity %1d\n"
                 "----------------------------\n"),
               (*(param_rc.param_perio)).num_perio) ;

    ecs_loc_maillage__print_perio(*param_rc.param_perio) ;

  }


  /* Construction de la liste des faces du maillage d'origine */
  /*  concernes par le recollement                           */
  /*----------------------------------------------------------*/

  ecs_entmail_pcp__compte_typ_fac(maillage->entmail,
                                  NULL,
                                  NULL,
                                  &liste_fac_de_bord,
                                  &liste_fac_isolee) ;

  nbr_fac = ecs_entmail__ret_nbr_ele(maillage->entmail[ECS_ENTMAIL_FAC]) ;

  /* Si l'on a des faces "libres", on veut les traiter aussi, sauf
     dans le cas de la priodicit */

  if (param_rc.param_perio != NULL) {
    liste_fac_isolee.nbr = 0 ;
    BFT_FREE(liste_fac_isolee.val) ;
  }

  if (liste_fac_de_bord.nbr > 0 && liste_fac_isolee.nbr > 0) {

    size_t ind_isol, ind_bord, ind_filtre ;
    ecs_int_t  *liste_filtre ;

    ind_bord = liste_fac_de_bord.nbr ;

    BFT_MALLOC(liste_filtre,
               liste_fac_de_bord.nbr + liste_fac_isolee.nbr,
               ecs_int_t) ;

    ind_bord = 0 ;
    ind_isol = 0 ;
    ind_filtre = 0 ;

    while (   ind_bord < liste_fac_de_bord.nbr
           && ind_isol < liste_fac_isolee.nbr) {

      if (  liste_fac_de_bord.val[ind_bord]
          < liste_fac_isolee.val[ind_isol])
        liste_filtre[ind_filtre++] = liste_fac_de_bord.val[ind_bord++] ;
      else
        liste_filtre[ind_filtre++] = liste_fac_isolee.val[ind_isol++] ;

    }

    while (ind_bord < liste_fac_de_bord.nbr)
      liste_filtre[ind_filtre++] = liste_fac_de_bord.val[ind_bord++] ;

    while (ind_isol < liste_fac_isolee.nbr)
      liste_filtre[ind_filtre++] = liste_fac_isolee.val[ind_isol++] ;

    assert(ind_filtre == liste_fac_de_bord.nbr + liste_fac_isolee.nbr) ;

    BFT_FREE(liste_fac_de_bord.val) ;
    BFT_FREE(liste_fac_isolee.val) ;

    liste_fac_de_bord.val = liste_filtre ;

    liste_fac_de_bord.nbr += liste_fac_isolee.nbr ;
    liste_fac_isolee.nbr = 0 ;

  }

  else if (liste_fac_de_bord.nbr == 0 && liste_fac_isolee.nbr > 0) {

    liste_fac_de_bord.nbr = liste_fac_isolee.nbr ;
    liste_fac_de_bord.val = liste_fac_isolee.val ;

    liste_fac_isolee.nbr = 0 ;
    liste_fac_isolee.val = NULL ;

  }

  bool_elt_select
    = ecs_entmail_pcp__selectionne(maillage->entmail,
                                   ECS_ENTMAIL_FAC,
                                   &liste_fac_de_bord,
                                   select_fac_rc.liste_couleur_fac,
                                   select_fac_rc.liste_groupe_fac,
                                   select_fac_rc.inv_selection) ;

  BFT_FREE(liste_fac_de_bord.val) ;

  /* Recollement des faces concernes */
  /*----------------------------------*/

  val_ret = ecs_entmail_pcp__recolle(maillage->entmail,
                                     NULL,
                                     &bool_elt_select[ECS_ENTMAIL_FAC],
                                     liste_fac_new,
                                     liste_fac_mod,
                                     liste_fac_err,
                                     param_rc) ;


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (bool_elt_select[ient].val != NULL)
      BFT_FREE(bool_elt_select[ient].val) ;

  }

  BFT_FREE(bool_elt_select) ;


  return val_ret ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui recolle des faces non conformes dont la visibilit
 *  est connue (par exemple une filiation) ;
 *
 *  Les listes des faces nouvelles ou modifies sont construites (et alloues)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entre ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le dcoupage en sous-faces a chou
 *
 *  On renvoie 1 si l'on a effectivement appel le recollement, 0 sinon
 *  (i.e. si le champ de type "visibilit" est introuvable).
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_maillage__recolle_vis
(
 ecs_maillage_t    *const maillage      ,
 ecs_tab_int_t     *const liste_fac_new ,
 ecs_tab_int_t     *const liste_fac_mod ,
 ecs_tab_int_t     *const liste_fac_err
)
{

  ecs_param_rc_t     param_rc ;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  param_rc.num_rc        = -1 ;
  param_rc.fraction_dist = 0.1 ;
  param_rc.epsilon_plan  = 0.8 ;
  param_rc.param_perio   = NULL ;
  param_rc.semi_conforme = ECS_FALSE ;


  assert(maillage != NULL) ;

  /* Recollement des faces concernes */
  /*----------------------------------*/

  return ecs_entmail_pcp__recolle(maillage->entmail,
                                  ECS_CHAMP_NOM_CONNECT,
                                  NULL,
                                  liste_fac_new,
                                  liste_fac_mod,
                                  liste_fac_err,
                                  param_rc) ;

}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_famille
(
 ecs_maillage_t *const maillage  ,
 ECS_ENTMAIL_E         ent_deb_e ,
 ECS_ENTMAIL_E         ent_fin_e
)
{

  ecs_int_t       ient ;
  ecs_int_t       ifam_ent ;

  ecs_entmail_t *   vect_entmail[ECS_ENTMAIL_FIN] ;
  ecs_famille_t * * vect_famille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(ent_fin_e <  ECS_ENTMAIL_FIN) ;
  assert(ent_deb_e <= ent_fin_e  ) ;


  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++)
    vect_entmail[ient] = NULL ;


  for (ient = ent_deb_e ; ient < (ecs_int_t)ent_fin_e + 1 ; ient++) {

    if (maillage->entmail[ient] != NULL)
      vect_entmail[ient] = maillage->entmail[ient] ;

  }


  vect_famille = ecs_entmail_pcp__cree_famille(vect_entmail) ;


  for (ifam_ent = ent_deb_e ; ifam_ent < (ecs_int_t)ent_fin_e + 1 ; ifam_ent++)
    maillage->famille[ifam_ent] = vect_famille[ifam_ent] ;

  BFT_FREE(vect_famille) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui dtruit les familles
 *----------------------------------------------------------------------------*/

void ecs_maillage__detruit_famille
(
 ecs_maillage_t *const maillage
)
{

  ecs_int_t       ient ;
  ecs_int_t       ifam_ent ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Destruction des champs "famille" */
  /*----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (maillage->entmail[ient] != NULL ) {

      ecs_entmail_pcp__detruit_att_nom(maillage->entmail[ient],
                                       ECS_CHAMP_NOM_FAMILLE) ;

    }

  }


  /* Destruction des familles */
  /*--------------------------*/

  for (ifam_ent  = (ecs_int_t)ECS_FAMILLE_FIN - 1 ;
       ifam_ent >= (ecs_int_t)ECS_FAMILLE_DEB     ; ifam_ent--) {

    if (maillage->famille[ifam_ent] != NULL) {

      ecs_famille_chaine__detruit(&maillage->famille[ifam_ent]) ;

    }

  }


}


/*----------------------------------------------------------------------------
 *  Fonction qui construit les attributs "groupe" et "couleur"
 *    partir des familles
 *----------------------------------------------------------------------------*/

void ecs_maillage__cree_attribut
(
 ecs_maillage_t *const maillage  ,
 ECS_ENTMAIL_E         ent_deb_e ,
 ECS_ENTMAIL_E         ent_fin_e
)
{

  ecs_int_t        ient ;
  ecs_int_t        ifam_ent ;

  ecs_famille_t    * famille ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(ent_fin_e <  ECS_ENTMAIL_FIN) ;
  assert(ent_deb_e <= ent_fin_e  ) ;


  /* Concatnation des familles des diffrentes entits */

  famille = NULL ;

  for (ifam_ent =  (ecs_int_t)ent_fin_e ;
       ifam_ent >= (ecs_int_t)ent_deb_e ; ifam_ent--) {

    if (maillage->famille[ifam_ent] != NULL) {

      ecs_famille_chaine__ajoute(&famille,
                                 maillage->famille[ifam_ent]) ;

      maillage->famille[ifam_ent] = NULL ;


    }

  }


  if (famille != NULL) {

    for (ient =  (ecs_int_t)ent_fin_e ;
         ient >= (ecs_int_t)ent_deb_e ; ient--) {

      if (maillage->entmail[ient] != NULL ) {

        ecs_entmail_pcp__cree_attribut(maillage->entmail[ient],
                                       famille) ;

      }

    }

  }


  ecs_famille_chaine__detruit(&famille) ;


}


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les attributs "groupe" et "couleur"
 *----------------------------------------------------------------------------*/

void ecs_maillage__supprime_attributs
(
 ecs_maillage_t * this_maillage
)
{

  ecs_int_t  ient ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  /* Structures des entits de maillage */
  /*====================================*/

  /* Libration du contenu des entits */
  /*-----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (this_maillage->entmail[ient] != NULL) {

      ecs_entmail_pcp__suppr_attribut(this_maillage->entmail[ient]) ;

    }

  }

}


/*----------------------------------------------------------------------------
 *  Vrification d'un maillage
 *----------------------------------------------------------------------------*/

ecs_bool_t ecs_maillage__verif
(
       ecs_maillage_t      * const maillage     ,
 const ecs_int_t                   nbr_dump     ,
 const char                * const nom_fic_dump ,
       ecs_post_t          * const cas_post
)
{

  ecs_int_t         ind_decoup ;

  ecs_tab_int_t     liste_fac_erreur ;
  ecs_tab_int_t     liste_fac_de_bord ;
  ecs_tab_int_t     liste_fac_isolee ;

  ecs_bool_t        bool_coherent ;

  ecs_maillage_t   *maillage_coupe_bord[ECS_POST_FAC_MAX_FIN] ;
  ecs_maillage_t   *maillage_coupe_int[ECS_POST_FAC_MAX_FIN] ;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage != NULL) ;

  bool_coherent = ECS_TRUE ;

  if (maillage->dim_e != ECS_DIM_3)
    return bool_coherent ;


  assert(maillage->connect != NULL) ;

  /* On vrifie qu'on est bien en connectivit descendante */

  assert(maillage->connect[ECS_CONNECT_CEL_FAC] != NULL) ;
  assert(maillage->connect[ECS_CONNECT_FAC_ARE] != NULL) ;
  assert(maillage->connect[ECS_CONNECT_ARE_SOM] != NULL) ;

  assert(maillage->entmail != NULL) ;

  assert(maillage->entmail[ECS_ENTMAIL_CEL] != NULL) ;
  assert(maillage->entmail[ECS_ENTMAIL_FAC] != NULL) ;
  assert(maillage->entmail[ECS_ENTMAIL_ARE] != NULL) ;
  assert(maillage->entmail[ECS_ENTMAIL_SOM] != NULL) ;


  /* Initialisation */

  for (ind_decoup = 0 ; ind_decoup < ECS_POST_FAC_MAX_FIN ; ind_decoup++) {

    maillage_coupe_bord[ind_decoup] = NULL ;
    maillage_coupe_int [ind_decoup] = NULL ;

  }


  /* Cration des coupes (pour post-traitement) */

  if (cas_post != NULL) {

    ecs_entmail_pcp__compte_typ_fac(maillage->entmail,
                                    &liste_fac_erreur,
                                    NULL,
                                    &liste_fac_de_bord,
                                    &liste_fac_isolee) ;


    ecs_maillage__cree_coupe(_(ECS_MAILLAGE_NOM_BORD),
                             maillage,
                             liste_fac_de_bord,
                             maillage_coupe_bord,
                             nbr_dump,
                             nom_fic_dump,
                             ECS_TRUE,
                             ECS_POST_TYPE_BORD,
                             cas_post) ;

    BFT_FREE(liste_fac_de_bord.val) ;
    liste_fac_de_bord.nbr = 0 ;


    /* Affichage des faces isoles ou de connectivit excessive */

    if (liste_fac_erreur.nbr > 0) {
      ecs_maillage_post__ecr_fac_liste(_(ECS_MAILLAGE_NOM_FAC_ERR_CONNECT),
                                       maillage,
                                       liste_fac_erreur,
                                       ECS_FALSE,
                                       ECS_POST_TYPE_ERREUR,
                                       cas_post) ;
      BFT_FREE(liste_fac_erreur.val) ;
      liste_fac_erreur.nbr = 0 ;
    }

    if (liste_fac_isolee.nbr > 0) {
      ecs_maillage_post__ecr_fac_liste(_(ECS_MAILLAGE_NOM_FAC_ISOLEE),
                                       maillage,
                                       liste_fac_isolee,
                                       ECS_FALSE,
                                       ECS_POST_TYPE_INFO,
                                       cas_post) ;
      BFT_FREE(liste_fac_isolee.val) ;
      liste_fac_isolee.nbr = 0 ;
    }

  }

  /* Vrification */

  bool_coherent = ecs_entmail_pcp__verif(maillage->entmail,
                                         cas_post) ;


  /* Destruction des coupes */

  if (cas_post != NULL) {

    ecs_maillage__detruit_coupe(maillage,
                                maillage_coupe_bord) ;

  }


  return bool_coherent ;
}


/*============================================================================
 *                              Fonctions prives
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Simplification d'un maillage extrait correspondant  un cas de
 *  coupe de faces : on ne conserve du maillage que l'entit des
 *  faces et de cette entit que les champs filiation et type,
 *  suffisants pour raliser le post-traitement des variables
 *  centres sur les faces.
 *----------------------------------------------------------------------------*/

static void ecs_loc_maillage__epure_coupe
(
       ecs_maillage_t    *const maillage_coupe ,
 const ecs_int_t                nbr_dump       ,
 const char              *const nom_fic_dump
)
{

  ecs_int_t      iconnect    ; /* Indice de boucle sur les connectivits      */
  ecs_int_t      ient        ; /* Indice de boucle sur les entits de maillage*/


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(maillage_coupe != NULL) ;


  /* Structures des connectivits */
  /*==============================*/

  /* Libration du contenu des connectivits */
  /*-----------------------------------------*/

  for (iconnect = ECS_CONNECT_DEB ; iconnect < ECS_CONNECT_FIN ; iconnect++) {

    if (maillage_coupe->connect[iconnect] != NULL) {

      maillage_coupe->connect[iconnect]
        = ecs_connect__detruit(maillage_coupe->connect[iconnect]) ;

    }

  }


  /* Libration du contenu des entits */
  /*-----------------------------------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (ient != ECS_ENTMAIL_FAC && maillage_coupe->entmail[ient] != NULL)

      maillage_coupe->entmail[ient]
        = ecs_entmail__detruit(maillage_coupe->entmail[ient]) ;

  }

  ecs_entmail_pcp__epure_coupe(maillage_coupe->entmail[ECS_ENTMAIL_FAC]) ;

  /* Liste chane des familles */
  /*----------------------------*/

  for (ient = ECS_ENTMAIL_DEB ; ient < ECS_ENTMAIL_FIN ; ient++) {

    if (maillage_coupe->famille[ient] != NULL)

      ecs_famille_chaine__detruit(&maillage_coupe->famille[ient]) ;

  }


  if (nbr_dump > 0)
    ecs_maillage__imprime(maillage_coupe,
                          nom_fic_dump,
                          nbr_dump,
                          "Reduced clip mesh") ;

}


/*----------------------------------------------------------------------------
 *    Fonction d'affichage des infos d'une priodicit
 *----------------------------------------------------------------------------*/

static void ecs_loc_maillage__print_perio
(
 const ecs_param_perio_t   param_perio
)
{


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  if (param_perio.type_perio == ECS_PERIO_TYPE_TRANS)
    bft_printf(_("\n  Periodicity type: pure translation\n")) ;
  else if (param_perio.type_perio == ECS_PERIO_TYPE_ROTA)
    bft_printf(_("\n  Periodicity type: translation + rotation\n")) ;
  else
    assert(   param_perio.type_perio == ECS_PERIO_TYPE_TRANS
           || param_perio.type_perio == ECS_PERIO_TYPE_NUL) ;


  if (   param_perio.type_perio == ECS_PERIO_TYPE_TRANS
      || param_perio.type_perio == ECS_PERIO_TYPE_ROTA)

    bft_printf(_("\n    Translation parameters\n"
                 "      - translation vector:     % 10.5e % 10.5e % 10.5e\n\n"),
               param_perio.translation[0],
               param_perio.translation[1],
               param_perio.translation[2]) ;

  if (   param_perio.type_perio == ECS_PERIO_TYPE_ROTA
      && ECS_ABS(param_perio.angle) < 1.e-6)

    bft_printf(_("\n    Rotation parameters\n"
                 "      - invariant point:        % 10.5e % 10.5e % 10.5e\n"
                 "      - rotation matrix:        % 10.5e % 10.5e % 10.5e\n"
                 "                                % 10.5e % 10.5e % 10.5e\n"
                 "                                % 10.5e % 10.5e % 10.5e\n\n"),
               param_perio.point_inv[0],
               param_perio.point_inv[1],
               param_perio.point_inv[2],
               param_perio.matrice[0][0], param_perio.matrice[0][1],
               param_perio.matrice[0][2],
               param_perio.matrice[1][0], param_perio.matrice[1][1],
               param_perio.matrice[1][2],
               param_perio.matrice[2][0], param_perio.matrice[2][1],
               param_perio.matrice[2][2]) ;

  else if (   param_perio.type_perio == ECS_PERIO_TYPE_ROTA
           && ECS_ABS(param_perio.angle) > 1.e-6)

    bft_printf(_("    Rotation parameters\n"
                 "      - invariant point:        % 10.5e % 10.5e % 10.5e\n"
                 "      - direction vector:       % 10.5e % 10.5e % 10.5e\n"
                 "      - angle (in degrees):     % 10.5e\n\n"),
               param_perio.point_inv[0],
               param_perio.point_inv[1],
               param_perio.point_inv[2],
               param_perio.direction[0],
               param_perio.direction[1],
               param_perio.direction[2],
               param_perio.angle) ;


}

