#ifndef _ECS_VEC_INT_H_
#define _ECS_VEC_INT_H_

/*============================================================================
 *  Prototypes des fonctions
 *   associes  la structure `ecs_vec_int_t' dcrivant un vecteur index entier
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int_publ.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction allouant et initialisant une structure `ecs_vec_int_t'
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__cree
(                           /* <-- Pointeur sur une structure `ecs_vec_int_t' */
 void
) ;


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_vec_int_t' donne en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__detruit
(
 ecs_vec_int_t * this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui alloue une structure `ecs_vec_int_t' ainsi que ses membres
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__alloue
(                       /* <-- Renvoie la rfrence  la structure cre      */
 size_t      pos_nbr , /* --> Nombre de positions pour  tableau des positions */
 size_t      val_nbr   /* --> Nombre de valeurs   pour  tableau des valeurs   */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui redimensionne les membres d'une structure `ecs_vec_int_t'
 *----------------------------------------------------------------------------*/

void ecs_vec_int__redimensionne
(
 ecs_vec_int_t  *const this_vec_int ,
 size_t                pos_nbr , /* --> Nouvelle dim. du tableau des pos.     */
 size_t                val_nbr   /* --> Nouvelle dim. du tableau des val. x   */
) ;


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_vec_int_t'
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__libere
(                               /* <-- Renvoie un pointeur NULL               */
 ecs_vec_int_t  *this_vec_int   /* <-> Structure `ecs_vec_int_t'  liberer    */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cre un vecteur `ecs_vec_int_t'
 *   et qui l'initialise avec les rfrences donnes
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__initialise
(
 size_t      pos_nbr ,
 ecs_size_t *pos_tab ,
 ecs_int_t  *val_tab
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cr une structure `ecs_vec_int_t'
 *    partir d'un tableau bi-dimensionnel `tab_elt' contenant
 *   pour chaque lment un ensemble de valeurs
 *  Si un lment `ielt' n'a pas de valeur associe, `tab_elt[ielt].nbr = 0'
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__transforme_bi_tab
(
 const ecs_tab_int_t *const tab_elt ,
       size_t               nbr_elt ,
       size_t               nbr_val
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_int__ret_pos_nbr
(
 const ecs_vec_int_t *const this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de valeurs associes au tableau des positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_int__ret_val_nbr
(
 const ecs_vec_int_t *const this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant la rfrence du tableau des positions
 *----------------------------------------------------------------------------*/

ecs_size_t * ecs_vec_int__ret_pos_tab
(
 ecs_vec_int_t *this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant la rfrence du tableau des valeurs
 *----------------------------------------------------------------------------*/

ecs_int_t * ecs_vec_int__ret_val_tab
(
 ecs_vec_int_t *this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le tableau des valeurs du vecteur index donn,
 *   dimensionn au nombre de valeurs distinctes du vecteur index,
 *   (valeurs qui correspondent  des rfrences)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int__ret_reference
(
 ecs_vec_int_t     *const this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui incrmente les valeurs d'un vecteur index donn
 *   d'une constante donne
 *----------------------------------------------------------------------------*/

void ecs_vec_int__incremente_val
(
       ecs_vec_int_t *const this_vec_int ,
 const ecs_int_t            increment
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui incrmente les valeurs d'un vecteur index donn
 *   d'une constante donne
 *----------------------------------------------------------------------------*/

void ecs_vec_int__incremente_val_sgn
(
       ecs_vec_int_t *const this_vec_int ,
 const ecs_int_t            increment
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un vecteur index
 *   en appliquant directement le vecteur de transformation donn
 *   sur ses valeurs
 *
 *  Le nombre de valeurs transformes peut tre infrieur
 *  au nombre de valeurs avant transformation
 *----------------------------------------------------------------------------*/

void ecs_vec_int__transforme_val
(
       ecs_vec_int_t  *const this_vec_int ,
       size_t                nbr_elt_new  ,
 const ecs_tab_int_t         vect_transf
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un vecteur index
 *   en appliquant directement le vecteur de transformation donn
 *   sur ses positions
 *
 *  Le nombre de valeurs transformes doit tre gal
 *  au nombre de valeurs avant transformation
 *----------------------------------------------------------------------------*/

void ecs_vec_int__transforme_pos
(
       ecs_vec_int_t  *const this_vec_int ,
       size_t                nbr_elt_ref  ,
 const ecs_tab_int_t         vect_transf
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un vecteur index
 *   en appliquant directement le vecteur de transformation donn
 *   sur les valeurs associes  ses lments
 *----------------------------------------------------------------------------*/

void ecs_vec_int__renumerote
(
       ecs_vec_int_t  *const this_vec_int ,
 const ecs_tab_int_t         vect_transf
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant la transformation d'un vecteur index
 *   en appliquant directement le vecteur de transformation donn
 *   sur les valeurs associes  ses lments
 *----------------------------------------------------------------------------*/

void ecs_vec_int__renumerote_et_sgn
(
       ecs_vec_int_t  *const this_vec_int ,
 const ecs_tab_int_t         vect_transf  ,
 const ecs_tab_int_t         signe_elt
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les rfrences  des lments
 *  en des rfrences  d'autres lments lis aux premiers
 *  par un vecteur index
 *----------------------------------------------------------------------------*/

void ecs_vec_int__remplace_ref
(
       ecs_vec_int_t  *const vec_rep ,
 const ecs_vec_int_t  *const vec_def
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les rfrences  des lments
 *  en des rfrences  d'autres lments lis aux premiers
 *  par un vecteur index ; les rfrences  0 sont ignores
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__ret_remplace_ref
(
       ecs_vec_int_t  *const vec_rep ,
 const ecs_vec_int_t  *const vec_def
) ;


/*----------------------------------------------------------------------------
 *  Fonction ralisant le transfert des valeurs d'un champ associes
 *   aux lments, sur les sous-lments
 *   participant  la dfinition des lments
 *   dans une connectivit "lment -> sous-lment"
 *
 *  Un mme sous-lment peut appartenir  plusieurs lments
 *  Un lment est suppos n'avoir qu'une seule valeur
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__herite
(                                                /* <-- Valeurs transfres   */
 const ecs_vec_int_t  *const vec_val_elt       , /* --> Valeurs  transfrer  */
 const ecs_vec_int_t  *const vec_elt_def_sselt , /* --> Dfinition lments   */
       size_t                nbr_sselt
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui dtermine un nouveau vecteur  partir d'un vecteur donn
 *   en extrayant de ce dernier les lments slectionns
 *   par le tableau de boolens
 *
 *  Cette fonction renvoie le vecteur qui dfinit les anciens lments
 * du vecteur de rfrence en fonction des nouveaux lments du vecteur renvoy
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__extrait
(
       ecs_vec_int_t   *const vec_new         ,
 const ecs_vec_int_t   *const vec_ref         ,
 const ecs_tab_bool_t         bool_elt_select
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit la  table de connectivit inverse
 *  "sous-lments -> lments"
 *
 *  Si dans la connectivit "lments -> sous-lments",
 *  le sous-lment est rfrenc ngativement dans la dfinition de l'lment,
 *  alors dans la connectivit inverse,
 *  l'lment est rfrenc ngativement dans la dfinition du sous-lment
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__inverse
(
 const ecs_vec_int_t  *const vec_def_elt ,
       size_t                nbr_sselt
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui ralise l'ordination des valeurs d'un vecteur index
 *     en triant les valeurs pour chaque ensemble de valeurs entre 2 positions
 *  et en triant chaque ensemble de valeurs dans l'ordre lexicographique
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int__trie_val
(
 ecs_vec_int_t * *const this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui compare les ensembles de valeurs comprises entre 2 positions
 *   et renvoie un tableau donnant pour chaque ensemble
 *      le numro du 1er ensemble de chaque groupe d'ensembles identiques
 *
 *  Si entre 2 positions il n'y a pas de valeurs, le numro est `-1'
 *
 *  Cette fonction prend tout son intrt lorsque
 *   - les valeurs ont t tries dans chaque ensemble de valeurs
 *   - les ensembles de valeurs ont t tries dans l'ordre lexicographique
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int__compare_val_pos
(
 const ecs_vec_int_t  *const this_vec_int
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui prolonge le tableau des valeurs d'une structure
 *  ecs_vec_int_t de manire  ce qu' chaque position corresponde au
 *  moins une valeur.
 *
 *  Si la structure ecs_vec_int_t n'existe pas, on en cre une.
 *
 *  Cette fonction est utile notamment lors de la construction des familles,
 *  afin que chaque lment appartienne bien explicitement  une famille
 *  donne, et non pas seulement implicitement  la famille 0 s'il ne porte
 *  aucune couleur ou groupe.
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__prolonge_val_pos
(
 ecs_vec_int_t *const this_vec_int ,
 ecs_int_t            val_defaut   ,
 size_t               nbr_defaut
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un vecteur index donnant pour chaque lment
 *   le numro de famille  laquelle il appartient
 *    partir -    du vecteur index donnant pour chaque lment
 *               les numros d'attribut le caractrisant
 *            - et du tableau donnant pour chaque lment :
 *                 - le 1er lment ayant la mme liste d'attributs
 *                 - ou `-1' s'il n'a pas d'attribut
 *
 *  La fonction dtermine aussi :
 *   - les dfinitions des familles en fonction des numros de descripteur
 *   - les nombres de descripteurs composant chaque famille
 *   - le  nombre  de familles
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__attribue_fam
(
 const ecs_vec_int_t      *const vec_int_att_unifie ,
 const ecs_tab_int_t             tab_idem           ,
       ecs_int_t      * * *      def_fam_descr      ,
       ecs_int_t        * *      nbr_descr_fam      ,
 const ecs_int_t                 num_fam_deb        ,
       ecs_int_t          *const nbr_fam
) ;


#endif /* _ECS_VEC_INT_H_ */
