#ifndef _ECS_CHAMP_H_
#define _ECS_CHAMP_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associes  la structure `ecs_champ_t' dcrivant un champ
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilit
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <bft_file.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_descr_publ.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ_publ.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction qui cre une structure `ecs_champ_t'
 *
 *  La structure devient propritaire des tableaux tab_pos et tab_val
 *   fournis en argument.
 *
 *   nbr_elt   : Nombre d'lments  remplir
 *   pas_pos   : Pas des positions  si REGLE
 *   tab_pos   : Positions du champ si non REGLE
 *   tab_val   : Valeurs du champ
 *   typ_val_e : Type des valeurs du champ
 *   nom       : Nom du champ
 *   descr     : Pointeur sur le descripteur
 *   statut_e  : Statut dans une transformation
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__cree
(
       size_t                      nbr_elt,
       size_t                      pas_pos,
       ecs_size_t           *const tab_pos,
       void                 *const tab_val,
       ecs_type_t                  typ_val_e,
 const char                 *const nom,
       ecs_descr_t          *const descr,
       ECS_CHAMP_STATUT_E          statut_e
) ;


/*----------------------------------------------------------------------------
 *  Fonction librant une structure `ecs_champ_t' donne en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__detruit
(
 ecs_champ_t * this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction imprimant le contenu d'une structure `ecs_champ_t' donne
 *   sur le flux dcrit par la structure `bft_file_t'
 *----------------------------------------------------------------------------*/

void ecs_champ__imprime
(
 const ecs_champ_t  *const this_champ, /* --> Structure  imprimer           */
       size_t              imp_col,
       size_t              nbr_imp,
       bft_file_t   *const fic_imp      /* --> Descripteur fic. d'impression  */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie la taille en octets d'une structure `ecs_champ_t'
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_taille
(
 const ecs_champ_t *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un champ entirement rallou
 *   dont le contenu est copi  partir du champ donn
 *
 *  Le membre donnant le lien sur un champ suivant `l_champ_sui'
 *   n'est pas copi et est mis  `NULL'
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__copie
(
 ecs_champ_t *const champ_init
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui cr une structure `ecs_champ_t'
 *    partir d'un tableau `tab_elt' contenant les valeurs du champ.
 *
 *  Les valeurs du champ doivent tre des valeurs entires.
 *  Si un lment n'a pas de valeur associe, la valeur correspondante
 *   dans `tab_elt' est `0'
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__transforme_tableau
(                                              /* <-- Pointeur cr           */
       size_t                      nbr_elt,    /* --> Nbr.lments  remplir  */
 const ecs_int_t            *const tab_elt,    /* --> Valeurs du champ        */
 const char                 *const nom,        /* --> Nom du champ            */
       ecs_descr_t          *const descr,      /* --> Ptr. sur descripteur    */
       ECS_CHAMP_STATUT_E          statut_e    /* --> Statut pour transform.  */
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de positions d'un champ donn
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_pos_nbr
(
 const ecs_champ_t *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction renvoyant le nombre de valeurs associes  un champ donn
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_val_nbr
(
 const ecs_champ_t  *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de descripteurs d'un champ donn
 *----------------------------------------------------------------------------*/

size_t ecs_champ__ret_descr_nbr
(
 const ecs_champ_t *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction retournant le type des valeurs d'un champ donn
 *----------------------------------------------------------------------------*/

ecs_type_t ecs_champ__ret_val_typ
(
 const ecs_champ_t *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction renvoyant un pointeur sur le tableau des positions d'une
 *   structure `ecs_champ_t' donne.
 *
 *  Si les positions correspondent  une REGLE, le tableau est allou et
 *   construit. S'il est dj dfini, on renvoie simplement son pointeur.
 *   Pour librer le tableau lorsqu'il correspond  une REGLE (sans le
 *   librer si ce n'est pas le cas), on appele ecs_champ__libere_pos_tab().
 *----------------------------------------------------------------------------*/

ecs_size_t * ecs_champ__ret_pos_tab
(
 ecs_champ_t  *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction librant un pointeur sur le tableau des positions d'une
 *   structure `ecs_champ_t' donne.
 *
 *  Si les positions correspondent  une REGLE, le tableau est libr.
 *   Sinon, il est conserv par la structure ecs_champ_t.
 *----------------------------------------------------------------------------*/

void ecs_champ__libere_pos_tab
(
 const ecs_champ_t  *const this_champ,
       ecs_size_t   *      pos_tab
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui concatne dans un champ rcepteur donn,
 *   un champ  concatner donn
 *
 *  Les 2 champs  concatner sont supposs avoir le mme nom
 *
 *  La concatnation de 2 champs consiste  concatner :
 *  - les tables des positions des 2 champs ;
 *  - les tables des valeurs   des 2 champs ;
 *  - les listes chanes des descripteurs des 2 champs
 *    (ncessaire uniquement pour des champs de type "attribut")
 *
 *  Les autres membres de la structure du champ rcepteur ne sont pas modifis
 *----------------------------------------------------------------------------*/

void ecs_champ__concatene
(
 ecs_champ_t *const champ_recept,
 ecs_champ_t *const champ_concat
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui prolonge un champ rcepteur donn
 *
 *  Il s'agit en fait de concatner le champ avec un champ vide. Seule la
 *  table des positions est modifie. Les autres membres de la structure du
 *  champ rcepteur ne sont pas modifis.
 *----------------------------------------------------------------------------*/

void ecs_champ__prolonge
(
 ecs_champ_t *const champ_recept,
 size_t             nbr_elt_prec,
 size_t             nbr_elt_suiv
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui convertit, si possible,
 *   le tableau des positions d'un champ en REGLE
 *----------------------------------------------------------------------------*/

void ecs_champ__pos_en_regle
(
 ecs_champ_t  *this_champ    /* --> Structure contenant le champ             */
) ;


#endif /* _ECS_CHAMP_H_ */
