/**********************************************************************************/
/* $Id: Crc.c,v 1.2 2003/06/03 22:51:34 sleeper Exp $    		  */
/*										  */
/* Copyright (c) 2001, Analog Devices Inc., All Rights Reserved			  */
/*										  */
/* Crc.c									  */
/*										  */
/* Responsible for calculation of 32-bit CRC's for CPCS PDU CRC field		  */
/*										  */
/* Taken (with a few modificaitons) from ADI's NDIS Crc code.			  */
/*										  */
/* This file is part of the "ADI USB ADSL Driver for Linux".			  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is free software; you can redistribute it 	  */
/* and/or modify it under the terms of the GNU General Public License as 	  */
/* published by the Free Software Foundation; either version 2 of the License, 	  */
/* or (at your option) any later version.					  */
/*										  */
/* "ADI USB ADSL Driver for Linux" is distributed in the hope that it will be     */
/* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of	  */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		  */
/* GNU General Public License for more details.					  */
/*										  */
/* You should have received a copy of the GNU General Public License		  */
/* along with "ADI USB ADSL Driver for Linux"; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA      */
/**********************************************************************************/

#include "Adiutil.h"
#include "Crc.h"
#include "debug.h"

static UInt32 CrcTable[256];

/***************************************************************************/
/* Package to compute 32-bit CRC one byte at a time using		   */
/* the high-bit first (Big-Endian) bit ordering convention		   */
/*									   */
/* Synopsis:								   */
/*  GenerateCrcTable() -- generates a 256-word table containing all CRC	   */
/*                     remainders for every possible 8-bit byte.  It	   */
/*                     must be executed (once) before any CRC updates.	   */
/*									   */
/*  It is assumed that an unsigned long is at least 32 bits wide and	   */
/*  that the predefined type char occupies one 8-bit byte of storage.      */
/*									   */
/*  The generator polynomial used for this version of the package is       */
/*  x^32+x^26+x^23+x^22+x^16+x^12+x^11+x^10+x^8+x^7+x^5+x^4+x^2+x^1+x^0    */
/*  as specified in the Autodin/Ethernet/ADCCP protocol standards.         */
/*  Other degree 32 polynomials may be substituted by re-defining the      */
/*  symbol POLYNOMIAL below.  Lower degree polynomials must first be       */
/*  multiplied by an appropriate power of x.  The representation usedm     */
/*  is that the coefficient of x^0 is stored in the LSB of the 32-bit      */
/*  word and the coefficient of x^31 is stored in the most significant     */
/*  bit.  The CRC is to be appended to the data most significant byte      */
/*  first.  For those protocols in which bytes are transmitted MSB         */
/*  first and in the same order as they are encountered in the block       */
/*  this convention results in the CRC remainder being transmitted with    */
/*  the coefficient of x^31 first and with that of x^0 last (just as       */
/*  would be done by a hardware shift register mechanization).		   */
/*									   */
/*  The table lookup technique was adapted from the algorithm described    */
/*  by Avram Perez, Byte-wise CRC Calculations, IEEE Micro 3, 40 (1983).   */
/***************************************************************************/
#define POLYNOMIAL 0x04c11db7L


/***************************************************************************/
/*  NAME: CrcGenerateTable						   */
/*            								   */
/*  ABSTRACT: 								   */
/*		Generates a table of CRC remainders for all possible bytes */
/*								           */
/*  RETURNS: 								   */
/*		void							   */
/*              							   */
/*  ASSUMPTIONS: 							   */
/*		Must be called prior to calling CrcCalculate   		   */
/***************************************************************************/
void CrcGenerateTable(void)
{ 
   UInt32 i;  
   UInt8 j;
   UInt32 CrcAccum;
   
   for( i = 0; i < 256; i++ )
   { 
      CrcAccum = ( (UInt32) i << 24 );
      for ( j = 0;  j < 8;  j++ )
      { 
          if ( CrcAccum & 0x80000000L )
          {
             CrcAccum = ( CrcAccum << 1 ) ^ POLYNOMIAL;
          }
          else
         {
             CrcAccum = ( CrcAccum << 1 ); 
         }
      }
      CrcTable[i] = CrcAccum; 
   } 
   return; 
}

/****************************************************************************/
/*  NAME: CrcCalculate							    */
/*            								    */
/*  ABSTRACT: 								    */
/*		Computes the CRC of 'count' bytes in the given cell payload,*/
/*      using the initialized or previous partial CRC value		    */
/*									    */
/*  RETURNS: 								    */
/*		the new partial CRC					    */
/*              							    */
/*  ASSUMPTIONS: 							    */
/*		Call CrcGenerateTable prior to calling this routine	    */
/****************************************************************************/
UInt32 CrcCalculate(UInt8 *Cell, UInt32 Length, UInt32 Crc)
{ 
   UInt32 i, j;
   UInt32 NewCrc;

   adi_enters (DBG_CRC);
   adi_dbg (DBG_CRC,"Cell=%p, len=%x, crc=%x\n",
            Cell, Length, Crc);

   NewCrc = Crc;

   for ( j = 0;  j < Length;  j++ )
   {
       i = ( (SInt32) ( NewCrc >> 24) ^ Cell[j]) & 0xff;
       NewCrc = ( NewCrc << 8 ) ^ CrcTable[i]; 
   }
   
   adi_dbg (DBG_CRC,"newcrc=0x%x\n",NewCrc);
   
   adi_leaves (DBG_CRC);
   
   return( NewCrc); 
}

/***********************************************************************************
$Log: Crc.c,v $
Revision 1.2  2003/06/03 22:51:34  sleeper
Changed ZAPS to adi_dbg/err macros

Revision 1.1.1.1  2003/02/10 23:29:49  sleeper
Imported sources

Revision 1.30  2002/05/24 21:59:30  Anoosh Naderi
Clean upthe code.

Revision 1.2  2002/01/14 21:59:30  chris.edgington
Added GPL header.

Revision 1.1  2002/01/02 21:56:31  chris.edgington
First version - from MacOS9 project (with Linux mods to get to compile).


------------------------------------------------------------------------------
 Log from MacOS9 project
------------------------------------------------------------------------------

Revision 1.6  2001/12/05 22:09:55  chris.edgington
Added Analog Devices copyright notice.

Revision 1.5  2001/10/05 15:57:05  chris.edgington
Added debug messages.

Revision 1.4  2001/09/10 18:19:33  chris.edgington
Changed UInt16 parameter type to UInt32.

Revision 1.3  2001/08/31 19:33:42  chris.edgington
Changed Int32 to SInt32 to fix compile error.

Revision 1.2  2001/08/30 20:27:27  chris.edgington
Changed type identifiers to Macintosh style.

Revision 1.1  2001/08/30 20:08:52  chris.edgington
Initial version - copied directly from ADI NDIS code.
************************************************************************/
