/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

/* Drivel's MT interface is modeled off of Sherzod Ruzmetov's 
 * Net::MovableType PERL class. */

#define _XOPEN_SOURCE /* glibc2 needs this */

#include <sys/types.h>
#include <config.h>

#include <time.h>
#include <string.h>
#include <libxml/parser.h>

#include "drivel.h"
#include "drivel_request.h"
#include "journal.h"
#include "network.h"
#include "xmlrpc.h"
#include "blog_mt.h"

/* to 'login' to an mt blog, we figure out whether the url supplied by the
 * user is for and rsd file on the server, the local machine, or the xml-rpc
 * gateway.  we then parse the result and extract the blog id. */

DrivelRequest*
blog_mt_build_login_request (const gchar *url, const gchar *username, 
		const gchar *password)
{
	DrivelRequest *dr;
	gchar *packet;
	
	g_return_val_if_fail (url, NULL);
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	
	debug ("blog_mt_build_login_request()\n");
	
	packet = xmlrpc_build_packet ("blogger.getUsersBlogs", 
			XMLRPC_TYPE_STRING, DRIVEL_APPKEY,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			-1);
	dr = drivel_request_new_with_items (REQUEST_TYPE_LOGIN,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT, 
			url,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

void
blog_mt_parse_login_request (DrivelClient *dc, DrivelRequest *dr)
{
	DrivelJournal *dj;
	GtkTreeIter iter;
	const gchar *mesg, *uri;
	gboolean finished;
	gint count;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_login_request()\n");
	
	/* Clear out categories from other users */
	gtk_list_store_clear (dc->category_store);
	gtk_list_store_append (dc->category_store, &iter);
	gtk_list_store_set (dc->category_store, &iter, 
			STORE_CATEGORY_NAME, _("None"),
			STORE_CATEGORY_ID, "none",
			STORE_CATEGORY_BLOG, "",
			-1);
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	add_account_to_list (dc);
	
	uri = drivel_request_get_uri (dr);
	
	/* build the list of journals */
	for (count = 0, finished = FALSE; !finished; count++)
	{
		gchar *key;
		
		key = g_strdup_printf ("blogname%d", count);
		mesg = drivel_request_value_lookup (dr, key);
		g_free (key);
		if (mesg)
		{
			gchar *name;
			const gchar *url, *id;
			
			name = g_strdup (mesg);
			key = g_strdup_printf ("blogid%d", count);
			id = drivel_request_value_lookup (dr, key);
			g_free (key);
			key = g_strdup_printf ("url%d", count);
			url = drivel_request_value_lookup (dr, key);
			g_free (key);
			if (id && url)
			{
				DrivelRequest *dr;
				
				dj = drivel_journal_new ();
				dj->name = name;
				dj->uri_view = g_strdup (url);
				dj->id = g_strdup (id);
				dj->type = JOURNAL_TYPE_USER;
				dc->journal_list = g_slist_prepend (dc->journal_list, dj);
				
				/* get the categories for this journal */
				dr = blog_mt_build_getcategories_request (dc->user->username, 
						dc->user->password, uri, dj->name, dj->id);
				net_enqueue_request (dc, dr);
				/* get the recent entries for this journal */
				dr = blog_mt_build_getevents_request (dc->user->username,
						dc->user->password, uri, dj->id, FALSE);
				net_enqueue_request (dc, dr);
			}
			else
				g_warning ("blog_mt_parse_login_request: could not read blogid");
		}
		else
			finished = TRUE;
	}
	
	dc->journal_list = g_slist_sort (dc->journal_list, (GCompareFunc)sort_journals);
	dc->journals = count - 1;
	
	/* Build the journal window */
	gtk_widget_hide (dc->login_window);
	journal_window_build (dc);
	
	return;
}

/* Build a DrivelRequest with all of the content needed by metaWeblog.newPost() */

DrivelRequest*
blog_mt_build_postevent_request (const gchar *username, const gchar *password,
		const gchar *uri, const gchar *blogid, gboolean publish, 
		const gchar *title, const gchar *content)
{
	DrivelRequest *dr;
	gchar *packet;
	xmlNodePtr node, node_title, node_description;
	
	debug ("blog_mt_build_postevent_request()\n");
	
	node_title = xmlrpc_build_value_string (title);
	node_description = xmlrpc_build_value_string (content);
	node = xmlrpc_build_value_struct ();
	xmlrpc_struct_add (node, "title", node_title);
	xmlrpc_struct_add (node, "description", node_description);
	
	packet = xmlrpc_build_packet ("metaWeblog.newPost", 
			XMLRPC_TYPE_STRING, blogid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_NODE, node,
			XMLRPC_TYPE_BOOL, publish,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_POSTEVENT,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT,
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

/* Parse the server's response to our post request */

void
blog_mt_parse_postevent_request (DrivelClient *dc, DrivelRequest *dr)
{
	DrivelRequest *dr2;
	const gchar *mesg;
	GtkTreeModel *model;
	GtkTreeIter iter;
	const gchar *postid, *uri;
	gchar *category = NULL;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_postevent_request()\n");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	net_ping_technorati (dc);
	
	/* First check for a "postid" value (set in the editevent method) since
	   metaWeblog.editPost returns a boolean, not the postid */
	postid = drivel_request_item_lookup (dr, "postid");
	if (!postid)
		postid = drivel_request_value_lookup (dr, "response");
	
	model = gtk_combo_box_get_model (GTK_COMBO_BOX (dc->journal_category));
	gtk_combo_box_get_active_iter (GTK_COMBO_BOX (dc->journal_category), &iter);
	gtk_tree_model_get (model, &iter, 1, &category, -1);
	/* If a category was selected, set it on the server */
	uri = drivel_request_get_uri (dr);
	dr2 = blog_mt_build_setpostcategories_request (dc->user->username,
			dc->user->password, uri, postid, category);
	net_enqueue_request (dc, dr2);
	dr2 = blog_mt_build_publish_request (dc->user->username,
			dc->user->password, uri, postid);
	net_enqueue_request (dc, dr2);
	
	if (category)
		g_free (category);
	
	return;
}

DrivelRequest *
blog_mt_build_editevent_request (const gchar *username, const gchar *password,
		const gchar *uri, const gchar *postid, gboolean publish,
		const gchar *title, const gchar *content)
{
	DrivelRequest *dr;
	gchar *packet;
	xmlNodePtr node, node_title, node_description;
	
	debug ("blog_mt_build_editevent_request()\n");
	
	node_title = xmlrpc_build_value_string (title);
	node_description = xmlrpc_build_value_string (content);
	node = xmlrpc_build_value_struct ();
	xmlrpc_struct_add (node, "title", node_title);
	xmlrpc_struct_add (node, "description", node_description);
	
	packet = xmlrpc_build_packet ("metaWeblog.editPost", 
			XMLRPC_TYPE_STRING, postid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_NODE, node,
			XMLRPC_TYPE_BOOL, publish,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_POSTEVENT,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT,
			uri,
			g_strdup ("xml"), packet,
			g_strdup ("postid"), g_strdup (postid),
			NULL);
	
	return dr;
}

/* Get the list of categories from the server */

DrivelRequest*
blog_mt_build_getcategories_request (const gchar *username, 
		const gchar *password, const gchar *uri, const gchar *journal_name,
		const gchar *blogid)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_mt_build_category_request()\n");
	
	packet = xmlrpc_build_packet ("mt.getCategoryList", 
			XMLRPC_TYPE_STRING, blogid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			-1);

	dr = drivel_request_new_with_items (REQUEST_TYPE_GETCATEGORIES,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT, 
			uri,
			g_strdup ("xml"), packet,
			g_strdup ("journal_name"), g_strdup (journal_name),
			NULL);
	
	return dr;
}

/* Parse the server's list of categories */

void
blog_mt_parse_getcategories_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gint i;
	GtkTreeIter iter;
	gboolean finished = FALSE;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_category_request()\n");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	/* Store the categories in a linked-list */
	for (i = 0; !finished; i++)
	{
		gchar *key1, *key2;
		const gchar *mesg1, *mesg2, *journal_name;
		
		key1 = g_strdup_printf ("categoryname%d", i);
		key2 = g_strdup_printf ("categoryid%d", i);
		
		mesg1 = drivel_request_value_lookup (dr, key1);
		mesg2 = drivel_request_value_lookup (dr, key2);
		journal_name = drivel_request_item_lookup (dr, "journal_name");
		
		if (mesg1 && mesg2 && journal_name)
		{
			gtk_list_store_append (dc->category_store, &iter);
			gtk_list_store_set (dc->category_store, &iter, 
					STORE_CATEGORY_NAME, mesg1,
					STORE_CATEGORY_ID, mesg2,
					STORE_CATEGORY_BLOG, journal_name,
					-1);
		}
		else
			finished = TRUE;
		
		g_free (key1);
		g_free (key2);
	}
	
	gtk_tree_model_filter_refilter (
			GTK_TREE_MODEL_FILTER (dc->category_store_filtered));
	gtk_combo_box_set_active (GTK_COMBO_BOX (dc->journal_category), 0);
	
	return;
}

/* Set the categories for a specific post */

DrivelRequest*
blog_mt_build_setpostcategories_request (const gchar *username, 
		const gchar *password, const gchar *uri, const gchar *postid, 
		const gchar *category)
{
	DrivelRequest *dr;
	xmlNodePtr node, node_struct, node_category, node_primary;
	gchar *packet;
	
	debug ("blog_mt_build_setpostcategories_request()\n");
	
	node = xmlrpc_build_value_array ();
	if (category && strcmp (category, "none"))
	{
		node_struct = xmlrpc_build_value_struct ();
		node_primary = xmlrpc_build_value_bool (TRUE);
		node_category = xmlrpc_build_value_string (category);
		xmlrpc_struct_add (node_struct, "isPrimary", node_primary);
		xmlrpc_struct_add (node_struct, "categoryId", node_category);
		xmlrpc_array_add (node, node_struct);
	}
	
	packet = xmlrpc_build_packet ("mt.setPostCategories", 
			XMLRPC_TYPE_STRING, postid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_NODE, node,
			-1);

	dr = drivel_request_new_with_items (REQUEST_TYPE_SETPOSTCATEGORIES,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

/* Parse the server's response to our setpostcategories request */

void
blog_mt_parse_setpostcategories_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_setpostcategories_request()\n");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	return;
}

DrivelRequest*
blog_mt_build_publish_request (const gchar *username, const gchar *password,
		const gchar *uri, const gchar *postid)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_mt_build_publish_request()\n");
	
	packet = xmlrpc_build_packet ("mt.publishPost", 
			XMLRPC_TYPE_STRING, postid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			-1);

	dr = drivel_request_new_with_items (REQUEST_TYPE_PUBLISH,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

void
blog_mt_parse_publish_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_publish_request()\n");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	journal_finished_post (dc);
	
	return;
}

DrivelRequest*
blog_mt_build_getevents_request (const gchar *username, const gchar *password,
		const gchar *uri, const gchar *blogid, gboolean last_entry)
{
	DrivelRequest *dr;
	gchar *packet;
	gint howmany;
	
	debug ("blog_mt_build_getevents_request()\n");
	
	if (last_entry)
		howmany = 1;
	else
		howmany = DRIVEL_N_RECENT_POSTS;

	packet = xmlrpc_build_packet ("metaWeblog.getRecentPosts", 
			XMLRPC_TYPE_STRING, blogid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			XMLRPC_TYPE_INT, howmany,
			-1);

	dr = drivel_request_new_with_items (REQUEST_TYPE_GETEVENTS,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT, 
			uri,
			g_strdup ("xml"), packet,
			g_strdup ("blogid"), blogid, /* needed when parsing the reply */
			NULL);

	if (last_entry)
	{
		drivel_request_add_items (dr, 
				g_strdup ("last_entry"), g_strdup ("true"),
				NULL);
	}

	return dr;
}

void
blog_mt_parse_getevents_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	gint i;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_getevents_request");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	if (drivel_request_item_lookup (dr, "last_entry"))
	{
		const gchar *postid, *content, *title;
		
		postid = drivel_request_value_lookup (dr, "postid0");
		content = drivel_request_value_lookup (dr, "description0");
		title = drivel_request_value_lookup (dr, "title0");
		
		if (postid && content)
		{
			DrivelRequest *dr;
			
			journal_edit_entry (dc, postid, content, NULL, NULL, title, NULL,
					NULL, NULL, NULL, NULL, NULL, NULL, NULL);
			
			dr = blog_mt_build_getpostcategories_request (dc->user->username,
					dc->user->password, dc->user->server, postid);
			net_enqueue_request (dc, dr);
		}
	}
	else
	{
		for (i = 0; i < DRIVEL_N_RECENT_POSTS; i++)
		{
			const gchar *postid, *content, *userid, *date, *title;
			gchar *key;
			
			key = g_strdup_printf ("postid%d", i);
			postid = drivel_request_value_lookup (dr, key);
			g_free (key);
			key = g_strdup_printf ("description%d", i);
			content = drivel_request_value_lookup (dr, key);
			g_free (key);
			userid = drivel_request_item_lookup (dr, "blogid");
			key = g_strdup_printf ("datecreated%d", i);
			date = drivel_request_value_lookup (dr, key);
			g_free (key);
			key = g_strdup_printf ("title%d", i);
			title = drivel_request_value_lookup (dr, key);
			g_free (key);
			
			if (postid && content && userid)
			{
				DrivelJournalEntry *entry = journal_entry_new ();
				
				entry->postid = g_strdup (postid);
				entry->content = g_strdup (content);
				entry->userid = g_strdup (userid);
				if (title)
					entry->subject = g_strdup (title);
				strptime (date, "%Y%m%dT%H:%M:%S", &entry->date_posted);
				g_ptr_array_add (dc->recent_entries, entry);
			}
		}
		
		journal_refresh_recent_entries (dc);
	}
	
	return;
}

DrivelRequest*
blog_mt_build_getpostcategories_request (const gchar *username,
		const gchar *password, const gchar *uri, const gchar *postid)
{
	DrivelRequest *dr;
	gchar *packet;
	
	debug ("blog_mt_build_getpostcategories_request()\n");
	
	packet = xmlrpc_build_packet ("mt.getPostCategories", 
			XMLRPC_TYPE_STRING, postid,
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_GETPOSTCATEGORIES,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_MT, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	return dr;
}

void
blog_mt_parse_getpostcategories_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	GtkTreeIter iter;
	gboolean valid;
	gint index;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	debug ("blog_mt_parse_getpostcategories_request");
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	mesg = drivel_request_value_lookup (dr, "categoryid0");
	if (!mesg)
	{
		gtk_combo_box_set_active (GTK_COMBO_BOX (dc->journal_category), 0);
	}
	else
	{
		index = 0;
		valid = gtk_tree_model_get_iter_first (
				GTK_TREE_MODEL (dc->category_store), &iter);
		while (valid)
		{
			gchar *id;
			
			gtk_tree_model_get (GTK_TREE_MODEL (dc->category_store), &iter,
					STORE_CATEGORY_ID, &id,
					-1);
			if (id && !strcmp (id, mesg))
				valid = FALSE;
			else
			{
				index++;
				valid = gtk_tree_model_iter_next (
						GTK_TREE_MODEL (dc->category_store), &iter);
			}
			
			g_free (id);
		}
		
		gtk_combo_box_set_active (GTK_COMBO_BOX (dc->journal_category), index);
	}
	
	return;
}
