/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2005 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <config.h>

#include <string.h>

#include "drivel.h"
#include "drivel_request.h"
#include "network.h"
#include "journal.h"
#include "xmlrpc.h"
#include "blog_advogato.h"

/* Convert line-breaks to HTML breaks (<br>) */
static gchar*
convert_linebreaks_to_html (const gchar *content)
{
	gchar **split, *converted;
	
	g_return_val_if_fail (content, NULL);
	
	split = g_strsplit (content, "\n", -1);
	if (split)
	{
		converted = g_strjoinv ("<br />", split);
		g_strfreev (split);
	}
	else
		converted = g_strdup (content);
	
	
	return converted;
}

/* Convert HTML breaks to line-breaks */
static gchar*
convert_linebreaks_from_html (const gchar *content)
{
	gchar **split, *converted, *temp;
	gchar *breaks[] = { "<br>", "<BR>", "<br />", "<BR />" };
	gint i;
	
	g_return_val_if_fail (content, NULL);
	
	temp = g_strdup (content);
	
	for (i = 0; i < 4; i++)
	{
		split = g_strsplit (temp, breaks[i], -1);
		if (split)
		{
			converted = g_strjoinv ("\n", split);
			g_strfreev (split);
		}
		else
			converted = g_strdup (content);
		
		g_free (temp);
		temp = converted;
	}
	
	return converted;
}

/* Get the number of journal entries */
static DrivelRequest*
build_getevents_len_request (const gchar *username, const gchar *uri)
{
	DrivelRequest *dr;
	gchar *packet;
	
	packet = xmlrpc_build_packet ("diary.len", 
			XMLRPC_TYPE_STRING, username,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_GETEVENTS,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_ADVOGATO, 
			uri,
			g_strdup ("xml"), packet,
			g_strdup ("type"), g_strdup ("len"),
			g_strdup ("username"), g_strdup (username),
			g_strdup ("uri"), g_strdup (uri),
			NULL);
	
	return dr;
}

/* Get the most recent journal entry */
static DrivelRequest*
build_getevents_get_request (const gchar *username, const gchar *uri, 
		gint itemid)
{
	DrivelRequest *dr;
	gchar *packet;
	
	packet = xmlrpc_build_packet ("diary.get", 
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_INT, itemid,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_GETEVENTS,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_ADVOGATO, 
			uri,
			g_strdup ("xml"), packet,
			g_strdup ("type"), g_strdup ("get"),
			g_strdup ("itemid"), g_strdup_printf ("%d", itemid),
			NULL);
	
	return dr;
}

DrivelRequest*
blog_advogato_build_login_request (const gchar *username, const gchar *password,
		const gchar *uri)
{
	DrivelRequest *dr;
	gchar *packet;
	
	g_return_val_if_fail (username, NULL);
	g_return_val_if_fail (password, NULL);
	
	packet = xmlrpc_build_packet ("authenticate", 
			XMLRPC_TYPE_STRING, username,
			XMLRPC_TYPE_STRING, password,
			-1);
	dr = drivel_request_new_with_items (REQUEST_TYPE_LOGIN,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_ADVOGATO,
			uri,
			g_strdup ("xml"), packet,
			NULL);

	return dr;
}

void
blog_advogato_parse_login_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	DrivelJournal *dj;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		g_print ("mesg: %s\n", mesg);
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	mesg = drivel_request_value_lookup (dr, "response");
	if (mesg)
	{
		add_account_to_list (dc);
		dc->user->cookie = g_strdup (mesg);
	}
	else
	{
		display_error_dialog (dc, _("Server error"), NULL);
		return;
	}
	
	/* Advogato only supports a single journal per username */
	dj = drivel_journal_new ();
	dj->name = g_strdup (dc->user->username);
	dj->description = g_strdup (dc->user->username);
	dj->type = JOURNAL_TYPE_USER;
	dc->journal_list = g_slist_prepend (dc->journal_list, dj);
	dc->journals = 1;
	
	gtk_widget_hide (dc->login_window);
	journal_window_build (dc);
	
	return;
}

DrivelRequest*
blog_advogato_build_postevent_request (const gchar *cookie, const gchar *uri, 
		gint index, const gchar *title, const gchar *content)
{
	DrivelRequest *dr;
	gchar *packet, *linebreak_content, *full_entry;
	
	g_return_val_if_fail (cookie, NULL);
	
	linebreak_content = convert_linebreaks_to_html (content);
	if (title)
		full_entry = g_strdup_printf ("%s\n<p>%s</p>", title, linebreak_content);
	else
		full_entry = g_strdup (linebreak_content);
	
	packet = xmlrpc_build_packet ("diary.set", 
			XMLRPC_TYPE_STRING, cookie,
			XMLRPC_TYPE_INT, index,
			XMLRPC_TYPE_STRING, full_entry,
			-1);
	
	dr = drivel_request_new_with_items (REQUEST_TYPE_POSTEVENT,
			REQUEST_PROTOCOL_XMLRPC, 
			BLOG_API_ADVOGATO, 
			uri,
			g_strdup ("xml"), packet,
			NULL);
	
	g_free (linebreak_content);
	g_free (full_entry);
	
	return dr;
}

void
blog_advogato_parse_postevent_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	net_ping_technorati (dc);
	journal_finished_post (dc);
	
	return;
}

/* Get the most recent entry */
DrivelRequest*
blog_advogato_build_getevents_request (const gchar *username, const gchar *uri)
{
	DrivelRequest *dr;
	
	g_return_val_if_fail (username, NULL);
	
	dr = build_getevents_len_request (username, uri);
	
	return dr;
}

void
blog_advogato_parse_getevents_request (DrivelClient *dc, DrivelRequest *dr)
{
	const gchar *mesg, *response;
	
	g_return_if_fail (dc);
	g_return_if_fail (dr);
	
	mesg = drivel_request_value_lookup (dr, "success");
	if (!mesg || !strcmp (mesg, "fault"))
	{
		mesg = drivel_request_value_lookup (dr, "faultstring0");
		display_error_dialog (dc, _("Server error"), mesg);
		return;
	}
	
	response = drivel_request_value_lookup (dr, "response");
	
	mesg = drivel_request_item_lookup (dr, "type");
	if (!strcmp (mesg, "len"))
	{
		DrivelRequest *dr2;
		const gchar *username, *uri;
		gint entries;
		
		username = drivel_request_item_lookup (dr, "username");
		uri = drivel_request_item_lookup (dr, "uri");
		entries = (gint) g_strtod (response, NULL);
		
		dr2 = build_getevents_get_request (username, uri, entries - 1);
		net_enqueue_request (dc, dr2);
	}
	else
	{
		const gchar *itemid;
		gchar *entry;
		
		itemid = drivel_request_item_lookup (dr, "itemid");
		entry = convert_linebreaks_from_html (response);
		
		journal_edit_entry (dc, itemid, entry, NULL, NULL, NULL, NULL, NULL, 
				NULL, NULL, NULL, NULL, NULL, NULL);
		
		g_free (entry);
	}
	
	return;
}
