/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2003 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <gnome.h>
#include <curl/curl.h>

#include "tray.h"
#include "eggtrayicon.h"
#include "network.h"
#include "journal.h"
#include "drivel.h"

#define CHECKFRIENDS_INTERVAL	30

EggTrayIcon *tray_icon = NULL;
GtkWidget *icon = NULL;
GtkWidget *menu = NULL;

gboolean
checkfriends (gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	gchar *post_data, *username, *lastupdate;
	
	if (dc->checking_friends || dc->friends_update)
		return TRUE;
	else
		dc->checking_friends = TRUE;
	
	if (dc->time_since_checkfriends < 1)
	{
		username = lj_format_string (dc->username);
		lastupdate = lj_format_string (dc->lastupdate);
		
		post_data = g_strdup_printf (
				"mode=checkfriends&ver=1&user=%s&hpassword=%s&lastupdate=%s",
				username, dc->password, lastupdate);
		
		lj_send_request (dc, post_data, REQUEST_CHECKFRIENDS);

		curl_free (username);
		curl_free (lastupdate);		
	}
	else
	{
		dc->time_since_checkfriends -= CHECKFRIENDS_INTERVAL;
		dc->checking_friends = FALSE;
	}
	
	return TRUE;
}

void
menu_show_cb (GtkMenuItem *item, DrivelClient *dc)
{
	tray_show_entry (dc);
	
	return;
}

void
menu_hide_cb (GtkMenuItem *item, DrivelClient *dc)
{
	tray_hide_entry (dc);
	
	return;
}

void
menu_quit_cb (GtkMenuItem *item, DrivelClient *dc)
{
	gtk_widget_destroy (dc->current_window);
	
	gtk_main_quit ();
	
	return;
}

void
view_friends_cb (GtkMenuItem *item, DrivelClient *dc)
{
	gchar *url;
	
	url = g_strdup_printf ("http://www.livejournal.com/~%s/friends/", 
			dc->username);
	
	gnome_url_show (url, NULL);
	
	dc->friends_update = FALSE;
	g_free (dc->lastupdate);
	dc->lastupdate = g_strdup ("");
	
	tray_icon_update (TRAY_NORMAL);
	
	return;
}
	
void
create_menu (DrivelClient *dc)
{
	GtkWidget *menuitem;
	
	menu = gtk_menu_new ();
	
	menuitem = gtk_menu_item_new_with_label ("View Friends Page");
	g_signal_connect (G_OBJECT (menuitem), "activate",
		G_CALLBACK (view_friends_cb), dc);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label ("Show Entry");
	g_signal_connect (G_OBJECT (menuitem), "activate",
		G_CALLBACK (menu_show_cb), dc);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label ("Hide Entry");
	g_signal_connect (G_OBJECT (menuitem), "activate",
		G_CALLBACK (menu_hide_cb), dc);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	menuitem = gtk_menu_item_new_with_label ("Quit");
	g_signal_connect (G_OBJECT (menuitem), "activate",
		G_CALLBACK (menu_quit_cb), dc);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	
	gtk_widget_show_all (menu);
	
	return;
}

void
viewed_friends (DrivelClient *dc)
{
	dc->friends_update = FALSE;
	g_free (dc->lastupdate);
	dc->lastupdate = g_strdup ("");
	
	tray_icon_update (TRAY_NORMAL);
	
	return;
}

void
clicked (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	DrivelClient *dc = (DrivelClient *) data;
	
	if (event->type != GDK_BUTTON_PRESS)
		return;
	
	switch (event->button) {
		case 1:
		{
			viewed_friends (dc);
			
			break;
		}
		case 2:
		{
			
			break;
		}
		case 3:
		{
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, dc, 
				event->button, event->time);
			
			break;
		}
	}
	
	return;
}

void
tray_hide_entry (DrivelClient *dc)
{
	journal_window_state_save (dc);
	
	gtk_widget_hide (dc->journal_window);
	gtk_widget_hide (dc->current_window);
	
	return;
}

void
tray_show_entry (DrivelClient *dc)
{
	GdkWindowState window_state;
	
	window_state = gdk_window_get_state (dc->current_window->window);
	if (window_state & GDK_WINDOW_STATE_ICONIFIED)
		gtk_window_deiconify (GTK_WINDOW (dc->current_window));
	else
	{
		gtk_widget_show (dc->current_window);
		gtk_widget_show (dc->journal_window);
	}
	
	journal_window_state_restore (dc);
	
	return;
}

void
tray_icon_update (TrayStatus status)
{
	GdkPixbuf *pixbuf;
	
	switch (status)
	{
		case TRAY_NORMAL:
		{
			pixbuf = gtk_image_get_pixbuf (GTK_IMAGE (icon));
			if (pixbuf)
				gdk_pixbuf_unref (pixbuf);
			
			pixbuf = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
				G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S 
				"livejournal_tray_normal.png", NULL);
			
			gtk_image_set_from_pixbuf (GTK_IMAGE (icon), pixbuf);
			
			break;
		}
		case TRAY_UPDATED_FRIENDS:
		{
			pixbuf = gtk_image_get_pixbuf (GTK_IMAGE (icon));
			if (pixbuf)
				gdk_pixbuf_unref (pixbuf);
			
			pixbuf = gdk_pixbuf_new_from_file (DATADIR G_DIR_SEPARATOR_S "pixmaps" 
				G_DIR_SEPARATOR_S "drivel" G_DIR_SEPARATOR_S 
				"livejournal_tray_updated.png", NULL);
			
			gtk_image_set_from_pixbuf (GTK_IMAGE (icon), pixbuf);
			
			break;
		}
	}
	
	return;
}
 
void
tray_create (DrivelClient *dc)
{
	GtkWidget *box;
	
	if (tray_icon)
		return;
	
	if (!menu)
		create_menu (dc);
	
	tray_icon = egg_tray_icon_new ("Drivel");
	box = gtk_event_box_new ();
	icon = gtk_image_new ();
	 
	gtk_container_add (GTK_CONTAINER (box), icon);
	gtk_container_add (GTK_CONTAINER (tray_icon), box);
	gtk_widget_show_all (GTK_WIDGET (tray_icon));
	
	g_signal_connect (G_OBJECT (box), "button-press-event", 
		G_CALLBACK (clicked), dc);
	
	tray_icon_update (TRAY_NORMAL);
	
	dc->id.loop_checkfriends_id = g_timeout_add (1000 * CHECKFRIENDS_INTERVAL,
			checkfriends, dc);
	
	return;
}

void
tray_destroy (DrivelClient *dc)
{
	GdkPixbuf *pixbuf;
	
	if (!tray_icon)
		return;
	
	g_source_remove (dc->id.loop_checkfriends_id);
	
	pixbuf = gtk_image_get_pixbuf (GTK_IMAGE (icon));
	if (pixbuf)
		gdk_pixbuf_unref (pixbuf);
	
	gtk_widget_destroy (icon);
	icon = NULL;
	
	gtk_widget_destroy (GTK_WIDGET (tray_icon));
	tray_icon = NULL;
	
	return;
}
