/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2002 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 */

#include <gnome.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include <string.h>
#include <curl/curl.h>

#include "network.h"
#include "about.h"
#include "journal.h"
#include "md5.h"
#include "login.h"

void
about_cb(GtkWidget *widget, gpointer data)
{
	about_show (data);
	
	return;
}

gboolean
get_sensitivity (GtkWidget *entry)
{
	gchar *text;
	gint len;
	gboolean retval;
	
	text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	len = strlen (text);
	
	if (len > 0)
		retval = TRUE;
	else
		retval = FALSE;
	
	g_free (text);
	
	return retval;
}

void
set_user_login_prefs (DrivelClient *dc, const gchar *username)
{
	gchar *user_password, *proxy_url;
	gboolean user_save_password, user_autologin;

	user_password = gconf_client_get_string (dc->client, dc->gconf->password, NULL);
	user_save_password = gconf_client_get_bool (dc->client, dc->gconf->save_password, NULL);
	user_autologin = gconf_client_get_bool (dc->client, dc->gconf->autologin, NULL);
	
	if (!user_password)
		user_password = g_strdup ("");
	
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (dc->login_name)->entry), username);
	gtk_entry_set_text (GTK_ENTRY (dc->login_password), user_password);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->save_password), user_save_password);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), user_autologin);

	g_free (dc->server);
	dc->server = gconf_client_get_string (dc->client, dc->gconf->default_server, NULL);
	if (!dc->server)
		dc->server = g_strdup ("www.livejournal.com");			
	
	g_free (dc->proxy_url);
	dc->proxy = gconf_client_get_bool (dc->client, dc->gconf->proxy, NULL);
	proxy_url = gconf_client_get_string (dc->client, dc->gconf->proxy_url, NULL);
	if (proxy_url)
		dc->proxy_url = g_strdup (proxy_url);
	else
		dc->proxy_url = NULL;
	dc->proxy_port = gconf_client_get_int (dc->client, dc->gconf->proxy_port, NULL);
	
	g_free (user_password);
	
	return;
}

void
user_list_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
	GConfValue *value;
	GSList *list, *user_list = NULL;
	DrivelClient *dc = (DrivelClient *) data;
	
	value = gconf_entry_get_value (entry);
	
	for (list = gconf_value_get_list (value); list; list = list->next)
		user_list = g_slist_append (user_list, g_strdup (gconf_value_get_string (list->data)));

	user_list = g_slist_sort (user_list, string_compare);
		
	gtk_combo_set_popdown_strings (GTK_COMBO (dc->login_name), (GList *) user_list);

	return;
}

gboolean
valid_username (const gchar *username, GSList *user_list)
{
	gboolean retval = FALSE;
	
	if (g_slist_find_custom (user_list, username, string_compare))
		retval = TRUE;

	return retval;
}

gboolean
login_focus_out_cb (GtkWidget *entry, GdkEventFocus *event, gpointer data)
{
	gchar *username;
	DrivelClient *dc = (DrivelClient *) data;
	
	username = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

	if (!validate_username (username))
	{		
		g_free (username);
			
		return FALSE;
	}

	if (username && strcmp (username, ""))
		drivel_gconf_data_fill (dc->gconf, username, dc->client, &dc->id, dc);
	
	g_free (dc->username);
	dc->username = username;
	
	return FALSE;
}

void
login_changed_cb (GtkWidget *entry, gpointer data)
{
	gboolean state;
	gchar *username;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = get_sensitivity (entry);

	if (state)
	{
		gtk_widget_set_sensitive (dc->login_password, TRUE);
		gtk_widget_set_sensitive (dc->sign_up_button, FALSE);
		gtk_widget_set_sensitive (dc->sign_up_menu, FALSE);
		
		username = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
		
		if (!validate_username (username))
		{
			display_error_dialog (dc, _("Username contains invalid characters."));
		
			g_free (username);
			
			return;
		}
		
		if (username && valid_username (username,
				gconf_client_get_list (dc->client, "/apps/drivel/global_settings/user_list", GCONF_VALUE_STRING, NULL)))
		{
			drivel_gconf_data_fill (dc->gconf, username, dc->client, &dc->id, dc);
			
			gconf_client_set_string (dc->client, "/apps/drivel/global_settings/current_user", username, NULL);
			
			set_user_login_prefs (dc, username);
		}
		
		g_free (dc->username);
		dc->username = username;
	}
	else
	{
		g_free (dc->username);
		dc->username = NULL;

		gtk_editable_delete_text (GTK_EDITABLE (dc->login_password), 0, -1);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->save_password), FALSE);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), FALSE);
		gtk_widget_set_sensitive (dc->login_password, FALSE);
		gtk_widget_set_sensitive (dc->save_password, FALSE);
		gtk_widget_set_sensitive (dc->autologin, FALSE);
		gtk_widget_set_sensitive (dc->login_button, FALSE);
		gtk_widget_set_sensitive (dc->login_menu, FALSE);
		gtk_widget_set_sensitive (dc->sign_up_button, TRUE);
		gtk_widget_set_sensitive (dc->sign_up_menu, TRUE);
	}

	return;
}

void
password_changed_cb (GtkWidget *entry, gpointer data)
{
	gboolean state;
	gchar *password;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = get_sensitivity (entry);
	
	if (state)
	{
		gtk_widget_set_sensitive (dc->save_password, TRUE);
		gtk_widget_set_sensitive (dc->login_button, TRUE);
		gtk_widget_set_sensitive (dc->login_menu, TRUE);
		
		if (dc->username && 
				gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dc->save_password)))
		{
			password = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

			gconf_client_set_string (dc->client, dc->gconf->password, password, NULL);
			g_free (password);
		}
	}
	else
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->save_password), FALSE);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), FALSE);
		gtk_widget_set_sensitive (dc->save_password, FALSE);
		gtk_widget_set_sensitive (dc->autologin, FALSE);
		gtk_widget_set_sensitive (dc->login_button, FALSE);
		gtk_widget_set_sensitive (dc->login_menu, FALSE);
	}
		
	return;
}

void
save_password_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	gchar *password;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));

	if (dc->username && dc->gconf->user)
		gconf_client_set_bool (dc->client, dc->gconf->save_password, state, NULL);
		
	if (state)
	{
		gtk_widget_set_sensitive (dc->autologin, TRUE);
		
		if (dc->username && dc->gconf->user)
		{
			password = gtk_editable_get_chars (GTK_EDITABLE (dc->login_password), 0, -1);

			gconf_client_set_string (dc->client, dc->gconf->password, password, NULL);
			g_free (password);
		}
	}
	else
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dc->autologin), FALSE);
		gtk_widget_set_sensitive (dc->autologin, FALSE);
		
		if (dc->username && dc->gconf->user)
			gconf_client_set_string (dc->client, dc->gconf->password, "", NULL);
	}
	
	return;
}

void
autologin_cb (GtkWidget *button, gpointer data)
{
	gboolean state;
	DrivelClient *dc = (DrivelClient *) data;
	
	state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	
	if (dc->username && dc->gconf->user)
		gconf_client_set_bool (dc->client, dc->gconf->autologin, state, NULL);
	
	return;
}

void
login_button_cb (GtkWidget *widget, gpointer data)
{
	gchar *login, *password, *login_data;
	gchar *version;
	gchar *values [2];
	gint i;
	guchar digest [16], md5_password [34];
	guint len;
	MD5_CTX context;
	DrivelClient *dc = (DrivelClient *)data;

//	journal_enable (dc, FALSE);
	
	login = gtk_editable_get_chars (GTK_EDITABLE (GTK_COMBO(dc->login_name)->entry), 0, -1);
	password = gtk_editable_get_chars (GTK_EDITABLE (dc->login_password), 0, -1);
	version = g_strdup_printf ("GNOME-Drivel/%s", VERSION);
	
	values [0] = lj_format_string (login);
	values [1] = lj_format_string (version);
	len = strlen (password);
	
	/* encrypt the password */
	MD5Init (&context);
	MD5Update (&context, password, len);
	MD5Final (digest, &context);
	
	for (i = 0; i < 16; i++)
		g_snprintf (md5_password + i * 2, 3, "%02X", digest [i]);
	md5_password [32] = '\0';
	md5_password [33] = '\0';
	
	login_data = g_strdup_printf (
				"mode=login&ver=1&user=%s&hpassword=%s&"
				"clientversion=%s&"
				"getpickws=1&getmenus=1&getmoods=%d",
				values [0], md5_password, values [1],
				dc->moods);
	
	g_free (dc->username);
	g_free (dc->password);
	dc->username = login;
	dc->password = g_strdup (md5_password);
	
	for (i = 0; i < 2; i++)
		curl_free (values [i]);
	
//	dc->login_send_data = login_data;
	
	lj_http_reslv_name (dc);
	lj_send_request (dc, login_data, REQUEST_LOGIN);
		
	return;
}

void
sign_up_cb (GtkWidget *menu, gpointer data)
{
	gnome_url_show ("http://www.livejournal.com/create.bml", NULL);
	
	return;
}

void
what_is_lj_cb (GtkWidget *menu, gpointer data)
{
	gnome_url_show ("http://www.livejournal.com/", NULL);
	
	return;
}

void
login_window_build (DrivelClient *dc)
{	
	GtkWidget *vbox, *hbox;
	GtkSizeGroup *size_group;
	GtkWidget *login_window;
	GtkWidget *login;
	GtkWidget *password;
	GtkWidget *button;
	GtkWidget *label;
	GtkWidget *save_password;
	GtkWidget *autologin;
	GtkWidget *image;
	GtkWidget *frame;
	GdkPixbuf *pixbuf;
	GSList *user_list;
	GList *icon_list = NULL;
	gchar *user_login;
	
	GnomeUIInfo file_menu[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Sign _Up..."), N_("Sign up for a new LiveJournal"), sign_up_cb, "drivel-sign-up"),
		GNOMEUIINFO_ITEM_STOCK (N_("_Login"), N_("Login to  a livejournal server"), login_button_cb, "drivel-login"),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_EXIT_ITEM (exit_cb, dc),
		GNOMEUIINFO_END
	};

	GnomeUIInfo edit_menu[] =
	{
		GNOMEUIINFO_MENU_PREFERENCES_ITEM (edit_preferences_cb, dc),
		GNOMEUIINFO_END
	};
	
	GnomeUIInfo help_menu[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("What's a _LiveJournal?"), N_("Goto the LiveJournal homepage"), what_is_lj_cb, "drivel-what-is-lj"),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_ABOUT_ITEM (about_cb, dc),
		GNOMEUIINFO_END
	};

	GnomeUIInfo main_menu[] =
	{
		GNOMEUIINFO_MENU_FILE_TREE (file_menu),
		GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
		GNOMEUIINFO_MENU_HELP_TREE (help_menu),
		GNOMEUIINFO_END
	};
	
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S 
		"livejournal.png", NULL);
	icon_list = g_list_prepend (icon_list, pixbuf);
	gtk_window_set_default_icon_list (icon_list);
	
	login_window = gnome_app_new (PACKAGE, _("Drivel Login"));
	gtk_widget_realize (login_window);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
	
	/* image */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 0);
	
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, FALSE, 0);
	
	pixbuf = gdk_pixbuf_new_from_file (
		DATADIR G_DIR_SEPARATOR_S "pixmaps" G_DIR_SEPARATOR_S "drivel" 
		G_DIR_SEPARATOR_S "drivel_splash.png", NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	gtk_container_add (GTK_CONTAINER (frame), image);
	
	/* align the labels using a size group */
	size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	/* username */
	hbox = gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Username:"));
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	login = gtk_combo_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), GTK_COMBO (login)->entry);
	gtk_entry_set_activates_default (GTK_ENTRY (GTK_COMBO (login)->entry), TRUE);
	gtk_combo_disable_activate (GTK_COMBO (login));
	gtk_widget_set_size_request (login, 40, -1);
	gtk_box_pack_start (GTK_BOX (hbox), login, TRUE, TRUE, 3);
	
	/* password */
	hbox = gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 3);
	
	label = gtk_label_new_with_mnemonic (_("_Password:"));
	gtk_size_group_add_widget (size_group, label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 3);
	
	password = gtk_entry_new ();
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), password);
	gtk_entry_set_activates_default (GTK_ENTRY (password), TRUE);
	gtk_widget_set_sensitive (password, FALSE);
	gtk_entry_set_visibility (GTK_ENTRY (password), FALSE);
	gtk_widget_set_size_request (password, 40, -1);
	gtk_box_pack_start (GTK_BOX (hbox), password, TRUE, TRUE, 3);
	
	/* save password / auto-login */
	hbox = gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 3);
	
	save_password = gtk_check_button_new_with_mnemonic (_("_Save password"));
	gtk_widget_set_sensitive (save_password, FALSE);
	gtk_box_pack_start (GTK_BOX (hbox), save_password, TRUE, FALSE, 2);
	
	autologin = gtk_check_button_new_with_mnemonic (_("_Auto-login"));
	gtk_widget_set_sensitive (autologin, FALSE);
	gtk_box_pack_start (GTK_BOX (hbox), autologin, TRUE, FALSE, 2);
	
	/* login button */
	hbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox), GTK_BUTTONBOX_EDGE);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 3);
	
	button = gtk_button_new_from_stock ("drivel-sign-up");
	dc->sign_up_button = button;
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	
	button = gtk_button_new_from_stock ("drivel-login");
	gtk_widget_set_sensitive (button, FALSE);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	dc->login_button = button;
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	
	gnome_app_set_contents (GNOME_APP (login_window), vbox);
	
	dc->login_window = login_window;
	dc->login_name = login;
	dc->login_password = password;
	dc->save_password = save_password;
	dc->autologin = autologin;

	drivel_push_current_window (dc, dc->login_window);
	
	gnome_app_create_menus_with_data (GNOME_APP(login_window), main_menu, dc);
	
	gtk_widget_show_all (login_window);
	
	dc->sign_up_menu = file_menu [0].widget;
	dc->login_menu = file_menu [1].widget;
	gtk_widget_set_sensitive (dc->login_menu, FALSE);
	
	/* do this before connecting the changed signal */
	user_list = gconf_client_get_list (dc->client, "/apps/drivel/global_settings/user_list",
			GCONF_VALUE_STRING, NULL);
	gconf_client_notify_add (dc->client, "/apps/drivel/global_settings/user_list", user_list_changed_cb,
			dc, NULL, NULL);
	
	if (user_list)
	{
		user_list = g_slist_sort (user_list, string_compare);
		gtk_combo_set_popdown_strings (GTK_COMBO (login), (GList *) user_list);
	}
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (login)->entry), "");
	
	/* connect the signals */
	
	g_signal_connect (G_OBJECT (GTK_COMBO (login)->entry), "focus_out_event",
			G_CALLBACK (login_focus_out_cb), dc);
	g_signal_connect (G_OBJECT (GTK_COMBO (login)->entry), "changed", 
			G_CALLBACK (login_changed_cb), dc);
	g_signal_connect (G_OBJECT (password), "changed",
			G_CALLBACK (password_changed_cb), dc);
	g_signal_connect (G_OBJECT (save_password), "toggled",
			G_CALLBACK (save_password_cb), dc);
	g_signal_connect (G_OBJECT (autologin), "toggled",
			G_CALLBACK (autologin_cb), dc);
	g_signal_connect (G_OBJECT (dc->sign_up_button), "clicked",
			G_CALLBACK (sign_up_cb), NULL);
	g_signal_connect (G_OBJECT (dc->login_button), "clicked",
			G_CALLBACK (login_button_cb), dc);
	g_signal_connect (G_OBJECT (login_window), "delete_event",
			G_CALLBACK (delete_event_cb), dc);

	/* setup the user's preferences */
	user_login = gconf_client_get_string (dc->client, "/apps/drivel/global_settings/current_user", NULL);	
	
	if (user_login && user_login [0] != '\0')
	{
		drivel_gconf_data_fill (dc->gconf, user_login, dc->client, &dc->id, dc);
		
		set_user_login_prefs (dc, user_login);		
	}
	
	g_free (user_login);
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (autologin)))
		login_button_cb (button, dc);
	
	gtk_editable_select_region (GTK_EDITABLE (GTK_COMBO (dc->login_name)->entry), 0, -1);
	gtk_widget_grab_default (dc->login_button);
	
	return;
}
